/**
 * @file         : SoundControllerPrj.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : SoundControllerPrj is project extension of HMIBase SoundController
 * @copyright    : (c) 2017-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "hall_std_if.h"
#include "hmi_trace_if.h"
#include "SoundControllerPrj.h"
#include "ResourceControllerPrj.h"
#include "AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h"
#include "AppHmi_MasterStateMachine.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/SoundControllerPrj.cpp.trc.h"
#endif


using namespace ::hmibase::apphmi_master::audio;


SoundControllerPrj::SoundControllerPrj()
{
   _systemMuteStatusInfo.clear();
}


SoundControllerPrj::~SoundControllerPrj()
{
   _systemMuteStatusInfo.clear();
}


bool SoundControllerPrj::onVolumeValueChange(int sinkId, int volume)
{
   ETG_TRACE_USR4(("SoundControllerPrj: onVolumeValueChange: SinkId = %d : Volume = %d", sinkId, volume));
   bool ret = SoundController::onVolumeValueChange(sinkId, volume);
   ResourceControllerPrj* imp = AudioControllerObjectManager::getInstance().getResourceController<ResourceControllerPrj>().get();
   int SinkIndex = imp->GetIndexOnSinkID(sinkId);
   if ((ret) && (NULL != imp))
   {
      for (::std::vector< IVolumeChangeListenerCB* >::const_iterator itr = _volumeChangeListeners.begin(); (itr != _volumeChangeListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onVolumeChangeUpdate(SinkIndex, volume);
         }
      }
   }
   return ret;
}


void SoundControllerPrj::onSinkMuteStateChange(int nSinkId, int nMuteState)
{
   ETG_TRACE_USR4(("SoundControllerPrj: onSinkMuteStateChange: SinkId = %d : MuteState = %d", nSinkId, nMuteState));
   SoundController::onSinkMuteStateChange(nSinkId, nMuteState);
   ResourceControllerPrj* imp = AudioControllerObjectManager::getInstance().getResourceController<ResourceControllerPrj>().get();
   int SinkIndex = imp->GetIndexOnSinkID(nSinkId);
   if (NULL != imp)
   {
      for (::std::vector< IMuteStateListenerCB* >::const_iterator itr = _muteStateListeners.begin(); (itr != _muteStateListeners.end()); ++itr)
      {
         if (NULL != (*itr))
         {
            (*itr)->onMuteStateUpdate(SinkIndex, nMuteState);
         }
      }
   }
}


void SoundControllerPrj::onSoundPropertyListUpdate(int nSinkId)
{
   ETG_TRACE_USR4(("SoundControllerPrj: onSoundPropertyListUpdate:  nSinkId = %d", nSinkId));
   SoundController::onSoundPropertyListUpdate(nSinkId);
   for (::std::vector< ISoundPropertyListenerCB* >::const_iterator itr = _soundPropertyListeners.begin(); (itr != _soundPropertyListeners.end()); ++itr)
   {
      if (NULL != (*itr))
      {
         (*itr)->onSoundPropertyUpdate(nSinkId);
      }
   }
}


bool SoundControllerPrj::getSystemMuteStatusForRegionId(const uint32 regionId) const
{
   bool muteStatus = false;
   ::std::map<uint32, bool>::const_iterator itr1 = _systemMuteStatusInfo.find(regionId);
   if (itr1 != _systemMuteStatusInfo.end())
   {
      muteStatus = itr1->second;
   }
   return muteStatus;
}


void SoundControllerPrj::updateSetSystemMuteReq(bool muteStatus, uint32 regionId)
{
   if (((_systemMuteStatusInfo.count(regionId) == 0) || (_systemMuteStatusInfo[regionId] != muteStatus)) && (fnIsRegionSupported(regionId)))
   {
      ETG_TRACE_USR4(("SoundControllerPrj: updateSetSystemMuteReq: muteStatus = %d  nSinkId = %d", muteStatus, fnGetAudioSinkIdForRegion(regionId)));
      _systemMuteStatusInfo[regionId] = muteStatus;
      SoundController::setSystemMute(muteStatus, fnGetAudioSinkIdForRegion(regionId));
   }
}


void SoundControllerPrj::onUpdateSinkChange(const uint32 sinkIndex)
{
   ETG_TRACE_USR4(("SoundControllerPrj: onUpdateSinkChange:  sinkIndex = %d", sinkIndex));
   ResourceControllerPrj* imp = AudioControllerObjectManager::getInstance().getResourceController<ResourceControllerPrj>().get();
   if ((NULL != imp) && (imp->isSinkAdded(sinkIndex)))
   {
      for (::std::map< uint32, bool >::const_iterator itr = _systemMuteStatusInfo.begin(); (itr != _systemMuteStatusInfo.end()); ++itr)
      {
         if (fnGetAudioSinkIdForRegion(itr->first) == sinkIndex)
         {
            SoundController::setSystemMute(itr->second, sinkIndex);
            break;
         }
      }
   }
}
