/**************************************************************************************
* @file         : VehicleClientHandler.cpp
* @author       : ECG5- shanmugapriya Murugan
* @addtogroup   :
* @brief        : Class contains necessary business logic required to handle source selection.
* @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "VehicleClientHandler.h"
#include "ProjectPluginMsgs.h"
#include "PluginConstants.h"
#include "CameraMain.h"
#include "CameraListHandler.h"
#include "CameraConfig/CameraConfig.h"
#include "Common/ApplicationSwitchClientComponent/ApplicationSwitchClientComponent.h"
#include "CameraSourceChangeHandler/CameraSourceChangeHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::VehicleClientHandler::
#include "trcGenProj/Header/VehicleClientHandler.cpp.trc.h"
#endif


using namespace ::vehicle_main_fi_types;
using namespace ::VEHICLE_MAIN_FI;

namespace App {
namespace Core {

#define RVC_TIMEOUT_PROCESS_TIME 3000

#define VEHICLEMAINFIPORT "vehicleMainFiPort"
#define ACTIVATE_CAM 1
#define DEACTIVATE_CAM 0
VehicleClientHandler* VehicleClientHandler::m_poVehicleClientHandler = NULL;

/******************************************************************************
*NAME        : VehicleClientHandler
*SYSFL       : NA
*Description : Constructor class for VehicleClientHandler
******************************************************************************/
VehicleClientHandler::VehicleClientHandler():
   m_vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy(VEHICLEMAINFIPORT, *this)),
   m_poRvcFiProxy(::rvc_cca_fi::Rvc_cca_fiProxy::createProxy("rvcFiPort", *this)),
   _rvcTimer(*this, RVC_TIMEOUT_PROCESS_TIME)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("VehicleClientHandler::VehicleClientHandler"));
   m_CANupdate = false;
   m_poCameraDataBindingHandler = CameraDataBindingHandler::poGetInstance();

   currentCamTextID = -1; //No CAN Text ID
   previousCamTextID = -1; //No CAN Text ID

   //digitalInputFlagInRvcMode = false;
   // NanoMsgClientHandler::poGetInstance()->uPdateDigitalInputFlagStatus(false);

   if (m_vehicleProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, m_vehicleProxy->getPortName());
      ETG_TRACE_USR4(("VehicleClientHandler::VehicleClientHandler, creating proxy"));
   }
   if (m_poRvcFiProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, m_poRvcFiProxy->getPortName());
      ETG_TRACE_USR4(("RearViewCamera Constructor after creating proxy"));
   }
   readLanguageOnStartUp();
   m_parkBrakeStatus = PARKING_BRAKE_SNA;
   m_speedValue = DEFAULT_SPEED_VALUE_SNA;
   m_camTextStatusValue = false;

   stCamCANActivation default_init;

   for (int i = 0; i < 10; i++)
   {
      default_init.cameraName = CAMOFF;
      default_init.actStatus = 0;
      default_init.camSrcName = "NO_SRC";
      vecStCANCameraOnStartup.push_back(default_init);
      vecStCANCameraAfterStartup.push_back(default_init);
   }	// default_init to initialize
}


/******************************************************************************
*NAME        : ~VehicleClientHandler
*SYSFL       : NA
*Description : Desctructor Class for VehicleClientHandler
******************************************************************************/
VehicleClientHandler::~VehicleClientHandler()
{
   currentCamTextID = -1; //No CAN Text ID
   previousCamTextID = -1; //No CAN Text ID
   ETG_TRACE_USR4(("VehicleClientHandler::~VehicleClientHandler"));
   if (m_poVehicleClientHandler != NULL)
   {
      delete m_poVehicleClientHandler;
      m_poVehicleClientHandler = NULL;
   }
   if (m_poCameraDataBindingHandler != NULL)
   {
      delete m_poCameraDataBindingHandler;
      m_poCameraDataBindingHandler = NULL;
   }
   vecStCANCameraOnStartup.clear();
   vecStCANCameraAfterStartup.clear();
   ETG_I_UNREGISTER_FILE();
}


/******************************************************************************
*NAME        : onAvailable
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onAvailable()"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);

   if (m_vehicleProxy && (proxy == m_vehicleProxy))
   {
      ETG_TRACE_USR4(("VehicleClientHandler::registerProperties(): valid proxy"));
      if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE))
      {
         m_vehicleProxy->sendActCam1UpReg(*this);
         m_vehicleProxy->sendActCam2UpReg(*this);
         m_vehicleProxy->sendActCam3UpReg(*this);
         m_vehicleProxy->sendActCam4UpReg(*this);
         m_vehicleProxy->sendActCam5UpReg(*this);
         m_vehicleProxy->sendActCam6UpReg(*this);
         m_vehicleProxy->sendActCam7UpReg(*this);
         m_vehicleProxy->sendActCam8UpReg(*this);
         m_vehicleProxy->sendActCam9UpReg(*this);
         m_vehicleProxy->sendActCam10UpReg(*this);
      }
      m_vehicleProxy->sendCamTxtUpReg(*this);
      m_vehicleProxy->sendRearCamReqUpReg(*this);
      //response
      m_vehicleProxy->sendCamSplitQuadUpReg(*this);
      m_vehicleProxy->sendLanguageUpReg(*this);
      m_vehicleProxy->sendReverseGearUpReg(*this);
      m_vehicleProxy->sendSpeedUpReg(*this);
      m_vehicleProxy->sendParkingBreakSwitchUpReg(*this);
   }
   if (m_poRvcFiProxy != NULL)
   {
      m_poRvcFiProxy->sendReverseSignalStatusUpReg(*this);
   }
}


/******************************************************************************
*NAME        : onUnavailable
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onUnavailable()"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);

   if (m_vehicleProxy && (proxy == m_vehicleProxy))
   {
      m_vehicleProxy->sendCamTxtRelUpRegAll();
      m_vehicleProxy->sendRearCamReqRelUpRegAll();
      if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE))
      {
         m_vehicleProxy->sendActCam1RelUpRegAll();
         m_vehicleProxy->sendActCam2RelUpRegAll();
         m_vehicleProxy->sendActCam3RelUpRegAll();
         m_vehicleProxy->sendActCam4RelUpRegAll();
         m_vehicleProxy->sendActCam5RelUpRegAll();
         m_vehicleProxy->sendActCam6RelUpRegAll();
         m_vehicleProxy->sendActCam7RelUpRegAll();
         m_vehicleProxy->sendActCam8RelUpRegAll();
         m_vehicleProxy->sendActCam9RelUpRegAll();
         m_vehicleProxy->sendActCam10RelUpRegAll();
      }
      m_vehicleProxy->sendLanguageRelUpRegAll();
      m_vehicleProxy->sendReverseGearRelUpRegAll();
      m_vehicleProxy->sendSpeedRelUpRegAll();
      m_vehicleProxy->sendParkingBreakSwitchRelUpRegAll();
   }
   if (m_poRvcFiProxy != NULL)
   {
      m_poRvcFiProxy->sendReverseSignalStatusRelUpRegAll();
   }
}


/******************************************************************************
*NAME        : registerProperties
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::registerProperties()"));
   if (m_vehicleProxy && (proxy == m_vehicleProxy))
   {
      ETG_TRACE_USR4(("VehicleClientHandler register property is called"));
      m_vehicleProxy->sendLanguageUpReg(*this);
      m_vehicleProxy->sendSpeedUpReg(*this);
      m_vehicleProxy->sendParkingBreakSwitchUpReg(*this);
   }
}


/******************************************************************************
*NAME        : deregisterProperties
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::deregisterProperties()"));
   if (m_vehicleProxy && (proxy == m_vehicleProxy))
   {
      m_vehicleProxy->sendLanguageRelUpRegAll();
      m_vehicleProxy->sendSpeedRelUpRegAll();
      m_vehicleProxy->sendParkingBreakSwitchRelUpRegAll();
   }
}


/******************************************************************************
*NAME        : HandleCamRequest
*SYSFL       : NA
*Description : Handling of Camera streaming request received by CAN.
******************************************************************************/
void VehicleClientHandler::HandleCamRequest(uint8 camValue, enCameraCAN currentCam, std::string cameraSrcName)
{
   uint8 m_isBlackScreenEnabledValue = KDSHandler::poGetInstance()->u8GetCameraBlackScreenFeatureValue();

   ETG_TRACE_USR4(("void VehicleClientHandler::HandleCamRequest m_isBlackScreenEnabledValue, camValue, (enCameraCAN)currentCam %d %d %d", m_isBlackScreenEnabledValue, camValue, (enCameraCAN)currentCam));
   stCamCANActivation actCAM;
   actCAM.cameraName = (enCameraCAN)currentCam;
   actCAM.actStatus = camValue;
   actCAM.camSrcName = cameraSrcName;
   int index = CameraListHandler::getInstance()->bCheckRequestedCamaraExists(cameraSrcName);
   if (index != -1)
   {
      vecStCANCameraAfterStartup.at(index) = actCAM;
   }
   if ((ACTIVATE_CAM == camValue) && (sm_gCMPStatus == 1) && (sm_gRVCStatus == 0))  // check for CMP status, RVC is inactive.
   {
      vector<int> camTextId;
      camTextId = CameraListHandler::getInstance()->getAllCameraTextIdOnSelection();
      vector<int> camAvailableTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();	  // get the camera list w.r.t xml or camtext signal

      if ((camTextId.size() > 0) && ((uint32)currentCam < camTextId.size()))         // check size, and also check range
      {
         int index = CameraListHandler::getInstance()->bCheckRequestedCamaraExists(cameraSrcName);
         if ((camTextId.at(currentCam) == -1) || (index == -1) || (camAvailableTextId.at(index) == 0xFF))
         {
            POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupNotConfigMsg)()));
            return;   // If no camera connected, then exit.
         }

         bool speedAndParkStatus = getSpeedAndParkingStatus();
         int _speedStatus = 0;
         // vector<int> camAvailableTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();
         if (camAvailableTextId.size() > 0)
         {
            CameraConfig::getInstance()-> getSpeedStatus(camAvailableTextId.at(index), _speedStatus);
            bool bSpeedAndParkBrakeStatusValidity = bGetSpeedAndParkBrakeValidityStatus();
            //Added validity check for speed and parking brake status. Bug ID 1267971
            if (bSpeedAndParkBrakeStatusValidity == SPEED_AND_PARKBRAKE_STATUS_VALID)
            {
               if (speedAndParkStatus == ALLOW_CAMERA)   // no restrictions. Allow all cameras. ie, speed is 0
               {
                  ETG_TRACE_COMP(("void VehicleClientHandler::HandleCamRequest RequestCameraBlackScreen ALLOW_CAMERA"));
                  vPerformActCamActivation(index);
               }
               if (speedAndParkStatus == ALLOW_PARTIAL_OR_BLOCK_CAMERA) 	// restrictions on few cameras. ie, speed is > 0
               {
                  // None, cp, cb. cpcb
                  if ((_speedStatus == COCKPIT_ONLY) || (_speedStatus == COCKPIT_CABIN))   // ignore speed.
                  {
                     vecStCANCameraAfterStartup.at(index) = actCAM;
                     ETG_TRACE_COMP(("void VehicleClientHandler::HandleCamRequest RequestCameraBlackScreen  ALLOW_PARTIAL_OR_BLOCK_CAMERA"));
                     vPerformActCamActivation(index);
                  }
                  else
                  {
                     POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
                  }
               }
            }
            else
            {
               POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
            }
            //RVC CRQ -> Check if the first camera is configured for RVC to display warning message.
            if ((currentCam == CAM1) && (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable()))
            {
               m_poCameraDataBindingHandler->vShowCameraWarningText(true);
            }
            else
            {
               m_poCameraDataBindingHandler->vShowCameraWarningText(false);
            }
         }
      }
   }
   else if ((DEACTIVATE_CAM == camValue) && (sm_gCMPStatus == 1) && (sm_gRVCStatus == 0))   // check for CMP status, RVC is inactive.
   {
      // ded6kor commented here since its filled above. Testing.
      // int index1 = CameraListHandler::getInstance()->bCheckRequestedCamaraExists(cameraSrcName);
      // if (index1 != -1)
      // {
      // vecStCANCameraAfterStartup.at(index1) = actCAM;
      // }
      //todo:: ded6kor
      //CRQ 1366236 Enable Camera Power Management at EvoBus

      ETG_TRACE_USR3(("HandleCamRequest:COCKPIT DEACTIVATE "));
      int value = 0;
      bool streamStatus = false;
      std::string strCamSrc = "";

      if (sm_gRVCStatus == 0)
      {
         CameraMain::poGetInstance()->vRequestStopCameraStream();
      }
      vsendDispCamCls_Rq(1);// 1 is send on deactivation of camera

      //CRQ 1366236 Enable Camera Power Management at EvoBus
      int index = CameraListHandler::getInstance()->bCheckRequestedCamaraExists(cameraSrcName);
      if (index != -1) // this is duplicate code. to be improved.
      {
         CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(index, value, strCamSrc, streamStatus);
         ETG_TRACE_USR3(("HandleCamRequest:COCKPIT DEACTIVATE sm_gCurrentRegion %d index %d, value %d ,streamStatus %d  strCamSrc %s", sm_gCurrentRegion, index, value, streamStatus, strCamSrc.c_str()));
         vUpdatetoCANfromHMI(strCamSrc, DEACTIVATE_CAM);
      }
      SetCANUpdate(false);

      //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
      // priority order 1st RVC 2nd MIC 3rd Camera/CanCamera

      // Handles CAN ACTIVATION AND DEACTIVATION for Full and Normal Scene
      if (sm_gCurrentRegion == REGION_C)
      {
         bool currentViewMode = CameraMain::poGetInstance()->GetToggleValue();
         ETG_TRACE_USR3(("HandleCamRequest:COCKPIT DEACTIVATE sm_gCurrentRegion %d currentViewMode %d, ", sm_gCurrentRegion, currentViewMode));
         if (currentViewMode == true) //enActivityIDs__eActivityID_CAMERA_STREAM
         {
            ETG_TRACE_USR3(("HandleCamRequest:COCKPIT DEACTIVATE full can scene"));
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
         else //  enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW
         {
            ETG_TRACE_USR3(("HandleCamRequest:COCKPIT DEACTIVATE normal can  scene"));
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
      }
      else if (sm_gCurrentRegion == REGION_A)
      {
         ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
      }
   }
   else
   {
      ETG_TRACE_USR4(("VehicleClientHandler::HandleCamRequest Invalid value"));
   }
}


void VehicleClientHandler::vPerformActCamActivation(int index)
{
   int value = 0;
   std::string strCamSrc = "";
   bool streamStatus = false;
   //CRQ 1366236 Enable Camera Power Management at EvoBus
   CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(index, value, strCamSrc, streamStatus);
   ETG_TRACE_USR3(("CameraListHandler::vPerformActCamActivation:ACTIVATE_CAM COCKPIT:index, value, ,streamStatus strCamSrc.c_str()=%d %d  %d %s", index, value, streamStatus, strCamSrc.c_str()));
   vUpdatetoCANfromHMI(strCamSrc, ACTIVATE_CAM);
   CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, value, strCamSrc);
   vContextSwitchRequestToCameraScreenViaCANTrigger();//Context switch to 258(Black screen of normal camera screen) [or] 251 (directly to normal camera screen)
   CameraMain::poGetInstance()->vRequestStartCameraStream();
   CameraListHandler::getInstance()->setSelectedCockpitCameraIndex(index);
   vsendDispCamCls_Rq(0);
   SetCANUpdate(true);
}


// vContextSwitchRequestToCameraScreenViaCANTrigger--> Context switch to 258(Black screen of normal camera screen) [or] 251 (directly to normal camera screen)
void VehicleClientHandler::vContextSwitchRequestToCameraScreenViaCANTrigger()
{
   uint8 m_isBlackScreenEnabledValue = KDSHandler::poGetInstance()->u8GetCameraBlackScreenFeatureValue();
   ETG_TRACE_USR3(("void CameraListHandler::SwitchToCameraScreenViaCANtRIGGER() m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));
   if (m_isBlackScreenEnabledValue == true)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)258);
   }
   else
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)251);
   }
}


/******************************************************************************
*NAME        : onActCam1Error
*SYSFL       : 5585
******************************************************************************/
void VehicleClientHandler::onActCam1Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam1Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam1Error"));
}


/******************************************************************************
*NAME        : onActCam1Status
*SYSFL       : 5585
*Description : Handling of CAM1 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam1Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam1Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam1Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable())
   {
      //do nothing, since ActCam1Status is same as RVC signal.
      ETG_TRACE_USR4(("onActCam1Status do nothing, since RVC signal takes care of it"));
   }
   else
   {
      if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //&& (CameraMain::poGetInstance()->getRegionValue() == REGION_C))//1507548     - Camerea activation is applicable to both region
      {
         uint8 camVal = status->getActCam1Value();
         ETG_TRACE_USR4(("VehicleClientHandler::onActCam1Status(cam1Value:%d)", camVal));//active =1;deactivate =0 ;error = 2;sna=3;

         // Process other CAN requests only if RVC is not activated. sm_gRVCStatus = 0 means RVC is not active. 1 means active.
         if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
         {
            HandleCamRequest(camVal, CAM1, "CMP1_CAM1");
         }
         if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
         {
            stCamCANActivation actCAM1;
            actCAM1.cameraName = CAM1;
            actCAM1.actStatus = camVal;
            actCAM1.camSrcName = "CMP1_CAM1";
            //vecStCANCameraOnStartup.push_back(actCAM1);
            vecStCANCameraOnStartup.at(0) = actCAM1;
            ETG_TRACE_USR4(("VehicleClientHandler::CAM1 added to vector"));
         }
      }
   }
}


/******************************************************************************
*NAME        : onActCam2Error
*SYSFL       : 5586
******************************************************************************/
void VehicleClientHandler::onActCam2Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam2Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam2Error"));
}


/******************************************************************************
*NAME        : onActCam2Status
*SYSFL       : 5586
*Description : Handling of CAM2 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam2Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam2Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam2Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam2Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam2Status(cam2Value:%d)", camVal));

      // Process other CAN requests only if RVC is not activated. sm_gRVCStatus = 0 means RVC is not active. 1 means active.
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM2, "CMP1_CAM2");
      }

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM2;
         actCAM2.cameraName = CAM2;
         actCAM2.actStatus = camVal;
         actCAM2.camSrcName = "CMP1_CAM2";
         //vecStCANCameraOnStartup.push_back(actCAM2);
         vecStCANCameraOnStartup.at(1) = actCAM2;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM2 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam3Error
*SYSFL       : 5587
******************************************************************************/
void VehicleClientHandler::onActCam3Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam3Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam3Error"));
}


/******************************************************************************
*NAME        : onActCam3Status
*SYSFL       : 5587
*Description : Handling of CAM3 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam3Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam3Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam3Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam3Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam3Status(cam3Value:%d)", camVal));

      // Process other CAN requests only if RVC is not activated. sm_gRVCStatus = 0 means RVC is not active. 1 means active.
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM3, "CMP1_CAM3");
      }
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM3;
         actCAM3.cameraName = CAM3;
         actCAM3.actStatus = camVal;
         actCAM3.camSrcName = "CMP1_CAM3";

         //vecStCANCameraOnStartup.push_back(actCAM3);
         vecStCANCameraOnStartup.at(2) = actCAM3;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM3 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam4Error
*SYSFL       : 5588
******************************************************************************/
void VehicleClientHandler::onActCam4Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam4Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam4Error"));
}


/******************************************************************************
*NAME        : onActCam4Status
*SYSFL       : 5588
*Description : Handling of CAM4 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam4Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam4Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam4Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam4Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam4Status(cam4Value:%d)", camVal));

      std::string cameraSrcName;
      //IF NO OF CAMPORTS IS > 1, THEN
      int numberOfCMPs = 0;
      tclAvRoutingParser* _avRoutingParser = tclAvRoutingParser::pGetInstance();
      if (_avRoutingParser != NULL)
      {
         bool m_bParseState = _avRoutingParser->bParseXml();
         numberOfCMPs = _avRoutingParser->vectorGetEntityID("CMP").size();
      }
      (numberOfCMPs > 1) ? cameraSrcName = "CMP2_CAM1" : cameraSrcName = "CMP1_CAM4";

      // Process other CAN requests only if RVC is not activated. sm_gRVCStatus = 0 means RVC is not active. 1 means active.
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM4, cameraSrcName);
      }

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM4;
         actCAM4.cameraName = CAM4;
         actCAM4.actStatus = camVal;
         actCAM4.camSrcName = cameraSrcName;

         //vecStCANCameraOnStartup.push_back(actCAM4);
         vecStCANCameraOnStartup.at(3) = actCAM4;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM4 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam5Error
*SYSFL       : 5589
******************************************************************************/
void VehicleClientHandler::onActCam5Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam5Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam5Error"));
}


/******************************************************************************
*NAME        : onActCam5Status
*SYSFL       : 5589
*Description : Handling of CAM5 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam5Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam5Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam5Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam5Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam5Status(cam5Value:%d)", camVal));

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM5, "CMP2_CAM2");
      }
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM5;
         actCAM5.cameraName = CAM5;
         actCAM5.actStatus = camVal;
         actCAM5.camSrcName = "CMP2_CAM2";

         //vecStCANCameraOnStartup.push_back(actCAM5);
         vecStCANCameraOnStartup.at(4) = actCAM5;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM5 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam6Error
*SYSFL       : 5590
******************************************************************************/
void VehicleClientHandler::onActCam6Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam6Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam6Error"));
}


/******************************************************************************
*NAME        : onActCam6Status
*SYSFL       : 5590
*Description : Handling of CAM6 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam6Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam6Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam6Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam6Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam6Status(cam6Value:%d)", camVal));

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM6, "CMP2_CAM3");
      }

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM6;
         actCAM6.cameraName = CAM6;
         actCAM6.actStatus = camVal;
         actCAM6.camSrcName = "CMP2_CAM3";

         //vecStCANCameraOnStartup.push_back(actCAM6);
         vecStCANCameraOnStartup.at(5) = actCAM6;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM6 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam7Error
*SYSFL       : 5591
******************************************************************************/
void VehicleClientHandler::onActCam7Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam7Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam7Error"));
}


/******************************************************************************
*NAME        : onActCam7Status
*SYSFL       : 5591
*Description : Handling of CAM7 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam7Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam7Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam7Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE))//1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam7Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam7Status(cam7Value:%d)", camVal));

      int numberOfCMPs = 0;
      std::string cameraSrcName;
      tclAvRoutingParser* _avRoutingParser = tclAvRoutingParser::pGetInstance();
      if (_avRoutingParser != NULL)
      {
         bool m_bParseState = _avRoutingParser->bParseXml();
         numberOfCMPs = _avRoutingParser->vectorGetEntityID("CMP").size();
      }
      (numberOfCMPs > 2) ? cameraSrcName = "CMP3_CAM1" : cameraSrcName = "CMP2_CAM4";
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM7, cameraSrcName);
      }
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM7;
         actCAM7.cameraName = CAM7;
         actCAM7.actStatus = camVal;
         actCAM7.camSrcName = cameraSrcName;

         //vecStCANCameraOnStartup.push_back(actCAM7);
         vecStCANCameraOnStartup.at(6) = actCAM7;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM7 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam8Error
*SYSFL       : 5591
******************************************************************************/
void VehicleClientHandler::onActCam8Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam8Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam8Error"));
}


/******************************************************************************
*NAME        : onActCam8Status
*SYSFL       : 5591
*Description : Handling of CAM8 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam8Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam8Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam8Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE))//1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam8Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam8Status(cam8Value:%d)", camVal));

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM8, "CMP3_CAM2");
      }

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM8;
         actCAM8.cameraName = CAM8;
         actCAM8.actStatus = camVal;
         actCAM8.camSrcName = "CMP3_CAM2";

         //vecStCANCameraOnStartup.push_back(actCAM8);
         vecStCANCameraOnStartup.at(7) = actCAM8;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM8 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam9Error
*SYSFL       : 5591
******************************************************************************/
void VehicleClientHandler::onActCam9Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam9Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam9Error"));
}


/******************************************************************************
*NAME        : onActCam9Status
*SYSFL       : 5591
*Description : Handling of CAM9 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam9Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam9Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam9Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE))//1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam9Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam9Status(cam9Value:%d)", camVal));

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM9, "CMP3_CAM3");
      }
      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM9;
         actCAM9.cameraName = CAM9;
         actCAM9.actStatus = camVal;
         actCAM9.camSrcName = "CMP3_CAM3";

         //vecStCANCameraOnStartup.push_back(actCAM9);
         vecStCANCameraOnStartup.at(8) = actCAM9;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM9 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onActCam10Error
*SYSFL       : 5591
******************************************************************************/
void VehicleClientHandler::onActCam10Error(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam10Error >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam10Error"));
}


/******************************************************************************
*NAME        : onActCam10Status
*SYSFL       : 5591
*Description : Handling of CAM10 request received by CAN.
******************************************************************************/
void VehicleClientHandler::onActCam10Status(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ActCam10Status >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onActCam10Status  Region ==>%d", CameraMain::poGetInstance()->getRegionValue()));
   if ((m_camActivationValue == CAN_NANO_ACTIVE) || (m_camActivationValue == CAN_ACTIVE)) //1507548- Camerea activation is applicable to both region
   {
      uint8 camVal = status->getActCam10Value();
      ETG_TRACE_USR4(("VehicleClientHandler::onActCam10Status(cam10Value:%d)", camVal));

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 1) && (_rvcTimer.isActive() == false))
      {
         HandleCamRequest(camVal, CAM10, "CMP3_CAM4");
      }

      if (/*(sm_gRVCStatus == 0) &&  */(sm_gCMPStatus == 0))  //&& (camVal == 1))
      {
         stCamCANActivation actCAM10;
         actCAM10.cameraName = CAM10;
         actCAM10.actStatus = camVal;
         actCAM10.camSrcName = "CMP3_CAM4";

         //vecStCANCameraOnStartup.push_back(actCAM10);
         vecStCANCameraOnStartup.at(9) = actCAM10;
         ETG_TRACE_USR4(("VehicleClientHandler::CAM10 added to vector"));
      }
   }
}


/******************************************************************************
*NAME        : onCamSplitQuadError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCamSplitQuadError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< CamSplitQuadError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCamSplitQuadError"));
   //To DO
}


/******************************************************************************
*NAME        : onCamSplitQuadStatus
*SYSFL       : NA
*Description : Handling of SPLIT/QUAD CAM request received by CAN.
******************************************************************************/
void VehicleClientHandler::onCamSplitQuadStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< CamSplitQuadStatus >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCamSplitQuadStatus"));
   //To Do
}


/******************************************************************************
*NAME        : onCam1ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler:: onCam1ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam1ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam1ManActError"));
}


/******************************************************************************
*NAME        : onCam1ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler:: onCam1ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam1ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam1ManActResult"));
}


/******************************************************************************
*NAME        : onCam2ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam2ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam2ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam2ManActError"));
}


/******************************************************************************
*NAME        : onCam2ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam2ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam2ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam2ManActResult"));
}


/******************************************************************************
*NAME        : onCam3ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam3ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam3ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam3ManActError"));
}


/******************************************************************************
*NAME        : onCam3ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam3ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam3ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam3ManActResult"));
}


/******************************************************************************
*NAME        : onCam4ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam4ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam4ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam4ManActError"));
}


/******************************************************************************
*NAME        : onCam4ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam4ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam4ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam4ManActResult"));
}


/******************************************************************************
*NAME        : onCam5ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam5ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam5ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam5ManActError"));
}


/******************************************************************************
*NAME        : onCam5ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam5ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam5ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam5ManActResult"));
}


/******************************************************************************
*NAME        : onCam6ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam6ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam6ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam6ManActError"));
}


/******************************************************************************
*NAME        : onCam6ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam6ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam6ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam6ManActResult"));
}


/******************************************************************************
*NAME        : onCam7ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam7ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam7ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam7ManActError"));
}


/******************************************************************************
*NAME        : onCam7ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam7ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam7ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam7ManActResult"));
}


/******************************************************************************
*NAME        : onCam8ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam8ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam8ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam8ManActError"));
}


/******************************************************************************
*NAME        : onCam8ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam8ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam8ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam8ManActResult"));
}


/******************************************************************************
*NAME        : onCam9ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam9ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam9ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam9ManActError"));
}


/******************************************************************************
*NAME        : onCam9ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam9ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam9ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam9ManActResult"));
}


/******************************************************************************
*NAME        : onCam10ManActError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam10ManActError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam10ManActError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCam10ManActError"));
}


/******************************************************************************
*NAME        : onCam10ManActResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onCam10ManActResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< Cam10ManActResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onCam10ManActResult"));
}


/******************************************************************************
*NAME        : onCamTxtValueError

******************************************************************************/
void VehicleClientHandler::onCamTxtError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< CamTxtError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueError"));
   m_camTextStatusValue = false;
}


/******************************************************************************
*NAME        : onSetLanguageResult
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onSetLanguageResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< SetLanguageResult >& result)
{
   ETG_TRACE_USR4(("onSetLanguageResult received :  %d", result->getResult()));
}


/******************************************************************************
*NAME        : onSetLanguageError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onSetLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SetLanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onSetLanguageError"));
}


/******************************************************************************
*NAME        : onLanguageError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageError >& /*error*/)
{
   ETG_TRACE_USR4(("onLanguageError"));
}


/******************************************************************************
*NAME        : onLanguageSyncSourceStatus
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onLanguageSyncSourceStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageSyncSourceStatus >& /*status*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onLanguageSyncSourceStatus received"));
}


/******************************************************************************
*NAME        : onLanguageSyncSourceError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onLanguageSyncSourceError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageSyncSourceError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onLanguageSyncSourceError"));
}


/******************************************************************************
*NAME        : onCamTxtValueStatus
*Description : Method to handle Camera text Status received by CAN
******************************************************************************/
void VehicleClientHandler::onCamTxtStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr<CamTxtStatus >& status)
{
   uint64 value = status->getCamTxtValue();
   ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus(value :%u) currentCamTextID :%u previousCamTextID:%u", value, currentCamTextID, previousCamTextID));

   currentCamTextID = value;
   // Bug 1318286: When Cam_Txt_ID is set to SNA, the last camera name is displayed in Camera list
   if (value != -1)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus  IF"));
      sendCamTxtIdUpdateToHMI(value);
      previousCamTextID = currentCamTextID;
   }
   // Bug 1318286: When Cam_Txt_ID is set to SNA, the last camera name is displayed in Camera list
   else if ((currentCamTextID == -1) & (previousCamTextID != -1))
   {
      ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus  ELSE IF"));
      sendCamTxtIdUpdateToHMI(value);
      previousCamTextID = currentCamTextID;
   }
   else
   {
      ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus(Read from config)"));
      m_camTextStatusValue = false;
   }
}


void VehicleClientHandler::sendCamTxtIdUpdateToHMI(uint64 CANtextIdvalue)
{
   m_camTextStatusValue = true;
   uint32 camnamekeyArray[8];

   uint32 camValueArray[8];
   unsigned long camValArr[8];
   int key;
   uint32 camvalue;
   ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus(size of long :%d)", sizeof(long)));
   for (int i = 0; i < 8; ++i)
   {
      camnamekeyArray[i] = CANtextIdvalue >> (8 * i) & 0xFF;
      key = CANtextIdvalue >> (8 * i) & 0xFF;
      ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus(CANtextIdvalue :%02X)", camnamekeyArray[i]));
      CameraConfig::getInstance()->getCameraValues(key, camvalue);
      camValueArray[i] =  camvalue;
   }
   CameraListHandler::getInstance()->updateHMIAvailableCamerasFromCAN(camnamekeyArray, m_camTextStatusValue);
}


/******************************************************************************
*NAME        : onCamTxtValueStatus
*Description : Method to handle Camera text Status received by TTFIs.
Note : To be deleted.
******************************************************************************/
void VehicleClientHandler::Testcode(int c1, int c2, int c3, int c4, int c5, int c6, int c7, int c8)
{
   uint32  camnamekeyArray[8];
   camnamekeyArray[0] = c1;
   camnamekeyArray[1] = c2;
   camnamekeyArray[2] = c3;
   camnamekeyArray[3] = c4;
   camnamekeyArray[4] = c5;
   camnamekeyArray[5] = c6;
   camnamekeyArray[6] = c7;
   camnamekeyArray[7] = c8;

   for (int i = 0; i < 8; ++i)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::onCamTxtValueStatus(value :%02X)", camnamekeyArray[i]));
   }
   CameraListHandler::getInstance()->updateHMIAvailableCamerasFromCAN_TTFIs(camnamekeyArray, true);
}


/******************************************************************************
*NAME        : onRearCamReqError
*SYSFL       : NA
******************************************************************************/
void VehicleClientHandler::onRearCamReqError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< RearCamReqError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onRearCamReqError"));
}


/******************************************************************************
*NAME        : onRearCamReqStatus
*Description : Function to handle the request of RearCam
******************************************************************************/
void VehicleClientHandler::onRearCamReqStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< RearCamReqStatus >& status)
{
   uint8 camVal = status->getRearCamReqValue();
   ETG_TRACE_USR4(("VehicleClientHandler::onRearCamReqStatus(RearCamReqValue:%d)", camVal));
   if (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable())
   {
      if ((ACTIVATE_CAM == camVal) && (sm_gRVCStatus == 0) && (sm_gCMPStatus == 1))
      {
         ETG_TRACE_USR4(("VehicleClientHandler::onRearCamReqStatus:ACTIVATE"));
         vPerformActCamActivation(0);
      }
      else if (DEACTIVATE_CAM == camVal)
      {
         if (sm_gRVCStatus == 0)
         {
            CameraMain::poGetInstance()->vRequestStopCameraStream();
         }
         vsendDispCamCls_Rq(1);

         //De-activate RVC CRQ 1366236 Enable Camera Power Management at EvoBus
         ETG_TRACE_USR4(("VehicleClientHandler::onRearCamReqStatus COCKPIT RVC Deactivation"));
         vUpdatetoCANfromHMI("CMP1_CAM1", DEACTIVATE);
         SetCANUpdate(false);
         ETG_TRACE_USR4(("VehicleClientHandler::onRearCamReqStatus:DEACTIVATE"));
         if (sm_gCurrentRegion == REGION_C)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_HOME);
         }
         else if (sm_gCurrentRegion == REGION_A)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CABINA_HOME);
         }
      }
   }
}


/******************************************************************************
*NAME        : vUpdatetoCANfromHMI
*Description : Method to update CAN the active Camera name in the HMI.
******************************************************************************/
void VehicleClientHandler::vUpdatetoCANfromHMI(std::string strCamSrc, uint8 status)
{
   uint8 numberOfCMPs = 0;
   tclAvRoutingParser* _avRoutingParser = tclAvRoutingParser::pGetInstance();
   ETG_TRACE_USR4(("VehicleClientHandler::vUpdatetoCANfromHMI status=%d strCamSrc=%s", status, strCamSrc.c_str()));
   if (_avRoutingParser != NULL)
   {
      numberOfCMPs = _avRoutingParser->vectorGetEntityID("CMP").size();
      ETG_TRACE_USR4(("VehicleClientHandler::vUpdatetoCANfromHMI numberOfCMPs ==>%d", numberOfCMPs));
   }
   int canIndex = CameraConfig::getInstance()->getCANIndexFromCamSrcName(strCamSrc);
   switch (canIndex)
   {
      case 1: //CMP1_CAM1:
      {
         Cam1ManAct_Deact("CMP1_CAM1", status);
      }
      break;
      case 2://CMP1_CAM2:
      {
         Cam2ManAct_Deact("CMP1_CAM2", status);
      }
      break;
      case 3: //CMP1_CAM3:
      {
         Cam3ManAct_Deact("CMP1_CAM3", status);
      }
      break;
      case 4: //CMP1_CAM4: CMP2_CAM1
      {
         if (numberOfCMPs == 1)
         {
            Cam4ManAct_Deact("CMP1_CAM4", status);
         }
         else
         {
            Cam4ManAct_Deact("CMP2_CAM1", status);
         }
      }// end of case 4
      break;
      case 5: // CMP2_CAM2
      {
         Cam5ManAct_Deact("CMP2_CAM2", status);
      }
      break;
      case 6: //CMP2_CAM3
      {
         Cam6ManAct_Deact("CMP2_CAM3", status);
      }
      break;
      case 7: //CMP2_CAM4, CMP3_CAM1
      {
         if (numberOfCMPs == 3)
         {
            Cam7ManAct_Deact("CMP3_CAM1", status);
         }
         else
         {
            Cam7ManAct_Deact("CMP2_CAM4", status);
         }
      }
      break;
      case 8: //CMP3_CAM2
      {
         Cam8ManAct_Deact("CMP3_CAM2", status);
      }
      break;
      case 9: //CMP3_CAM3
      {
         Cam9ManAct_Deact("CMP3_CAM3", status);
      }
      break;
      case 10: //CMP3_CAM4
      {
         Cam10ManAct_Deact("CMP3_CAM4", status);
      }
      break;

      default:
         break;
   }//end of switch
}


/************************************************************************
*FUNCTION: 		 Cam1ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam1ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam1ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam1ManActStart(*this, actDeactStatus);
   }
   else
   {
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam1ManActStart(*this, actDeactStatus);
      }
   }
}


void VehicleClientHandler::sendActivateCam1ManActStartRequest()
{
   ETG_TRACE_USR4(("VehicleClientHandler::sendActivateCam1ManActStartRequest is ACTIVATED"));
   m_vehicleProxy->sendCam1ManActStart(*this, 1);
}


/************************************************************************
*FUNCTION: 		 Cam2ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/

void VehicleClientHandler::Cam2ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam2ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam2ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam2ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam3ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam3ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam3ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam3ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam3ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam4ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam4ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam4ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam4ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam4ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam5ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam5ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam5ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam5ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam5ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam6ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam6ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam6ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam6ManActStart(*this, actDeactStatus);
   }
   else
   {
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam6ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam7ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam7ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam7ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam7ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam7ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam8ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam8ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam8ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam8ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam8ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam9ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam9ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam9ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam9ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam9ManActStart(*this, actDeactStatus);
      }
   }
}


/************************************************************************
*FUNCTION: 		 Cam10ManAct_Deact
*DESCRIPTION: 	 Function to Invoke the activation and Deactivation for ManAct0

*PARAMETER:		 std::string CamSrcString,uint8 actDeactStatus
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void VehicleClientHandler::Cam10ManAct_Deact(std::string CamSrcString, uint8 actDeactStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::Cam10ManAct_Deact %d %s", actDeactStatus, CamSrcString.c_str()));
   if (actDeactStatus == 1)
   {
      m_vehicleProxy->sendCam10ManActStart(*this, actDeactStatus);
   }
   else
   {
      //check if the src is active in either region
      if (FALSE == bCheckSrcActiveInEitherRegion(CamSrcString))
      {
         m_vehicleProxy->sendCam10ManActStart(*this, actDeactStatus);
      }
   }
}


void VehicleClientHandler::vDeactivateAllCameraManActCabin()
{
   ETG_TRACE_USR4(("void VehicleClientHandler::vDeactivateAllCameraManActCabin sm_gRVCStatus sm_gCMPStatus =%d %d", sm_gRVCStatus, sm_gCMPStatus));

   m_vehicleProxy->sendCam1ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam2ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam3ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam4ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam5ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam6ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam7ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam8ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam9ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam10ManActStart(*this, DEACTIVATE);
}


void VehicleClientHandler::vDeactivateSecondaryNdTeritoryCameraManActInCabin()
{
   ETG_TRACE_USR4(("void VehicleClientHandler::vDeactivateSecondaryNdTeritoryCameraManActInCabin is DEACTIVATED"));
   m_vehicleProxy->sendCam4ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam5ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam6ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam7ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam8ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam9ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam10ManActStart(*this, DEACTIVATE);
}


void VehicleClientHandler::vDeactivateTeritoryCameraManActInCabin()
{
   ETG_TRACE_USR4(("void VehicleClientHandler::vDeactivateTeritoryCameraManActInCabin is DEACTIVATED"));
   m_vehicleProxy->sendCam7ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam8ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam9ManActStart(*this, DEACTIVATE);
   m_vehicleProxy->sendCam10ManActStart(*this, DEACTIVATE);
}


std::string VehicleClientHandler::regioncheck(int regionId)
{
   std::string regionName;
   if (regionId == 2)
   {
      ETG_TRACE_USR4(("std::string VehicleClientHandler::regioncheck Cockpit"));
      regionName = "Current Region=Cockpit";
   }
   else
   {
      ETG_TRACE_USR4(("std::string VehicleClientHandler::regioncheck Cabin"));
      regionName = "Current Region=Cabin";
   }

   return  regionName;
}


/************************************************************************
*FUNCTION: 		 bCheckSrcActiveInEitherRegion
*DESCRIPTION: 	 Function Compares  SourceName"validateSrcName" is Active or Not Active for Both Cockpit and Cabin Region
                 Return False on Inactive and True on Active
*PARAMETER:		 std::string validateSrcName -- Current Valide Source Name
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/

bool VehicleClientHandler::bCheckSrcActiveInEitherRegion(std::string validateSrcName)
{
   bool requestCameraStatus = false;
   ETG_TRACE_USR4(("bool VehicleClientHandler::bCheckSrcActiveInEitherRegion validateSrcName=%s", validateSrcName.c_str()));
   //check for cockpit
   int intCockpitListPosition =  CameraListHandler::getInstance()->bCheckRequestedCamaraExists(validateSrcName);// get the requested camera in cockpit exist or Not
   int intCockpitActiveCamera =  CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();// get the index of active camera in cockpit.
   uint16 currentActivityId = CameraMain::poGetInstance()->vGetActivityID();
   ETG_TRACE_USR4(("bCheckSrcActiveInEitherRegion CpLP %d CAC %d ActID %d", intCockpitListPosition, intCockpitActiveCamera, currentActivityId));
   if ((intCockpitListPosition != -1) && (intCockpitActiveCamera != -1))
   {
      if ((intCockpitListPosition == intCockpitActiveCamera) && (251 == currentActivityId)) //camera is active in cockpit
      {
         // The camera is active and streaming in cockpit region
         ETG_TRACE_USR4(("bCheckSrcActiveInEitherRegion Cockpit camera active"));
         requestCameraStatus = true;
      }
   }

   //check in cabin
   int intCabinListPosition = CameraListHandler::getInstance()->bCheckRequestedCabinCameraExists(validateSrcName);// get the requested camera in cabin exist or Not
   int intCabinActiveCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();// get the index of active camera in cabin .
   uint32 cameraSrcInCabin = CameraSourceChangeHandler::getInstance()->getCameraSourceInCabin();
   bool cabinTileCameraToggleStatus = CameraMain::poGetInstance()->getCabinTileCameraToggleStatus();
   ETG_TRACE_USR4(("bCheckSrcActiveInEitherRegion CbLP %d CAC %d srcID %d toggle status %d", intCabinListPosition, intCabinActiveCamera, cameraSrcInCabin, cabinTileCameraToggleStatus));
   if ((intCabinListPosition != -1) && (intCabinActiveCamera != -1))
   {
      if ((intCabinListPosition == intCabinActiveCamera) && (cameraSrcInCabin != SRC_INVALID)) //camera is active in cabin
      {
         if (cabinTileCameraToggleStatus != false)
         {
            // The camera is active and streaming in cabin region
            ETG_TRACE_USR4(("bCheckSrcActiveInEitherRegion Cabin camera active"));
            requestCameraStatus = true;
         }
      }
   }
   ETG_TRACE_USR4(("bCheckSrcActiveInEitherRegion return status %d", requestCameraStatus));
   return requestCameraStatus;
}


void VehicleClientHandler::onDispCamClsError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< DispCamClsError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onDispCamClsError"));
}


void VehicleClientHandler::onDispCamClsResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< DispCamClsResult >& result)
{
   uint8 camResult = result->getResult();
   ETG_TRACE_USR4(("VehicleClientHandler::onDispCamClsStatus camResult==>%d", camResult));
}


/******************************************************************************
*NAME        : vsendDispCamCls_Rq
*Description : Method to update CAN if the streaming is in foreground or background.
******************************************************************************/
void VehicleClientHandler::vsendDispCamCls_Rq(int Value)
{
   ETG_TRACE_USR4(("VehicleClientHandler:: vsendDispCamCls_Rq Value==:%d", Value));
   if (m_vehicleProxy != NULL)
   {
      m_vehicleProxy->sendDispCamClsStart(*this, Value);
   }
   else
   {
      ETG_TRACE_USR4(("VehicleClientHandler:: NULL POINTER"));
   }
}


void VehicleClientHandler::onReverseSignalStatusError(const ::boost::shared_ptr< Rvc_cca_fiProxy >& /*proxy*/, const ::boost::shared_ptr< ReverseSignalStatusError >& /*error*/)
{
   ETG_TRACE_USR4(("VehicleClientHandler:: onReverseSignalStatusError"));
}


/******************************************************************************
*NAME        : onReverseSignalStatusStatus
*Description : Method to handle request for Reverse Gear.
******************************************************************************/
void VehicleClientHandler::onReverseSignalStatusStatus(const ::boost::shared_ptr< Rvc_cca_fiProxy >& /*proxy*/, const ::boost::shared_ptr< ReverseSignalStatusStatus >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler:: onReverseSignalStatusStatus Reverse sm_gCMPStatus %d reverse gear status %u", sm_gCMPStatus, status->getE8StatusReverseSignal()));
   //@ded6kor: added the bCheckRearCameraAvailable as double check from HMI. Fix for bug 1460442
   //@Somanat: bug id 941710 for the below change
   if ((status->getE8StatusReverseSignal() == 2) && (sm_gCMPStatus == 1) && (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable()))
   {
      m_vehicleProxy->sendCam1ManActStart(*this, 1);
      sm_gRVCStatus = 1;
      int value = 1; // always camport 1
      std::string strCamSrc = "CMP1_CAM1"; // 	always first camport
      CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, value, strCamSrc);
   }
   else
   {
      sm_gRVCStatus = 0;
      m_vehicleProxy->sendCam1ManActStart(*this, 0);

      int m_camActivationValue = KDSHandler::poGetInstance()->u8GetCameraActivationValue(); // doubt kds has data

      ETG_TRACE_USR4(("VehicleClientHandler:: onReverseSignalStatusStatus m_camActivationValue==>%d ", m_camActivationValue))
      if (m_camActivationValue == CAN_CAMERA)
      {
         verifyCANCameraActive();
      }
      else if (m_camActivationValue == DIG_PIN_CAMERA)
      {
         NanoMsgClientHandler::poGetInstance()->checkNanoMsgDigPinIsActive();
      }
      else // UNCONFIGURED
      {
         bool camActive = false;
         camActive = checkAnyCANCameraActive();
         ETG_TRACE_USR4(("VehicleClientHandler:: onReverseSignalStatusStatus camActive==>%d ", camActive))
         if (camActive == true)
         {
            verifyCANCameraActive();
         }
         else
         {
            NanoMsgClientHandler::poGetInstance()->checkNanoMsgDigPinIsActive();
         }
      }

      //@pga5cob: fix for 1828166 -->During  ign Cycle if  REVERSE Signal Status is Enabled then ManAct1 Power is Enabled
      if (status->getE8StatusReverseSignal() == 2)
      {
         ETG_TRACE_USR4(("VehicleClientHandler:: onReverseSignalStatusStatus Powered ManAct1 "));
         m_vehicleProxy->sendCam1ManActStart(*this, 1);
         sm_gRVCStatus = 1;

         //Bug 2008221: CMC HMI not able to shown RVC/ DOOR Camera.
         // On Useroff , If RVC is enable during other camera request, HMI camera processing the cmp1_cam1 to the plugin .
         int value = 1;
         std::string strCamSrc = "CMP1_CAM1";
         CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, value, strCamSrc);
      }
   }
}


void VehicleClientHandler::verifyCANCameraActive()
{
   bool camActive = false;
   camActive = checkAnyCANCameraActive();

   if (camActive == true)
   {
      ETG_TRACE_USR4(("void VehicleClientHandler::verifyCANCameraActive()Some other camera is active, start timer"));
      _rvcTimer.start();
   }
   else	//Bug 1531089 and 1530041: Blue Door CAM HMI
   {
      int activityID = CameraMain::poGetInstance()->vGetActivityID();
      if ((activityID == enActivityIDs__eActivityID_CAMERA_STREAM) && (_rvcTimer.isActive() == false))
      {
         ETG_TRACE_USR4(("void VehicleClientHandler::verifyCANCameraActive() NO Camera on RVC Exit activityID==>%d ", activityID));
         if (sm_gCurrentRegion == REGION_A)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
         else if (sm_gCurrentRegion == REGION_C)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
         }
         else
         {
            //Do Nothing
         }
      }
   }
}


void VehicleClientHandler::onReverseGearError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ReverseGearError >& error)
{
   ETG_TRACE_USR4(("VehicleClientHandler:: onReverseGearError Reverse"));
}


void VehicleClientHandler::onReverseGearStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ReverseGearStatus >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler:: onReverseGearStatus Reverse state %d", status->getState()));
   //@ded6kor: added the bCheckRearCameraAvailable as double check from HMI. Fix for bug 1460442

   //Below is redundant(onReverseSignalStatusStatus)  and blocked due to Bug 1828166: Cam-1 status shown as not active after IGN OFF ON when RVC is streaming
   /*
   if ((status->getState() == 1) && (sm_gCMPStatus == 1) && (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable()))
   {
      m_vehicleProxy->sendCam1ManActStart(*this, 1);
      // m_vehicleProxy->sendCam2ManActStart(*this, 0);
      // m_vehicleProxy->sendCam3ManActStart(*this, 0);
      // m_vehicleProxy->sendCam4ManActStart(*this, 0);
      // m_vehicleProxy->sendCam5ManActStart(*this, 0);
      // m_vehicleProxy->sendCam6ManActStart(*this, 0);
      // m_vehicleProxy->sendCam7ManActStart(*this, 0);
      // m_vehicleProxy->sendCam8ManActStart(*this, 0);
      // m_vehicleProxy->sendCam9ManActStart(*this, 0);
      // m_vehicleProxy->sendCam10ManActStart(*this, 0);
      sm_gRVCStatus = 1;
      ETG_TRACE_USR4(("VehicleClientHandler:: onReverseGearStatus--> inside if"));
   }
   else
   {
      ETG_TRACE_USR4(("VehicleClientHandler:: onReverseGearStatus--> else"));
      m_vehicleProxy->sendCam1ManActStart(*this, 0);
      // m_vehicleProxy->sendCam2ManActStart(*this, 0);
      // m_vehicleProxy->sendCam3ManActStart(*this, 0);
      // m_vehicleProxy->sendCam4ManActStart(*this, 0);
      // m_vehicleProxy->sendCam5ManActStart(*this, 0);
      // m_vehicleProxy->sendCam6ManActStart(*this, 0);
      // m_vehicleProxy->sendCam7ManActStart(*this, 0);
      // m_vehicleProxy->sendCam8ManActStart(*this, 0);
      // m_vehicleProxy->sendCam9ManActStart(*this, 0);
      // m_vehicleProxy->sendCam10ManActStart(*this, 0);
      sm_gRVCStatus = 0;
   } */
}


void VehicleClientHandler::onSpeedStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SpeedStatus >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onSpeedStatus speedvalue  = %d", status->getSpeedValue()));
   m_speedValue = status->getSpeedValue();
   m_speedValue = m_speedValue / 100;
   int16 m_speedValue_int16 = 0; //CID_R2 16723319
   m_speedValue_int16 = (int16)m_speedValue;
   updateSpeedAndParkingStatusValue(m_speedValue, m_parkBrakeStatus);
   int activityID = CameraMain::poGetInstance()->vGetActivityID();

   if (proxy == m_vehicleProxy)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::onSpeedStatus m_speedValue = %d m_speedValue_int16 = %d activityID=%d,m_parkBrakeStatus=%d", m_speedValue, m_speedValue_int16, activityID, m_parkBrakeStatus));
      //special handling for partking brake status SNA, or OFF when any of the streaming screen is active
      if ((activityID == enActivityIDs__eActivityID_CAMERA_STREAM) ||
            (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW) ||
            (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT) || (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT_FULLSCREEN))
      {
         if ((m_speedValue_int16 < ZERO_SPEED) || (m_speedValue_int16 > MAX_SPEED))
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
            POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
         }
         else
         {
            int _speedStatus = 0;
            int selectedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
            (selectedIndex == -1) ? selectedIndex = 0 : selectedIndex = selectedIndex;
            vector<int> camAvailableTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();
            if (camAvailableTextId.size() > 0)
            {
               CameraConfig::getInstance()-> getSpeedStatus(camAvailableTextId.at(selectedIndex), _speedStatus);
               if (_speedStatus == NONE)
               {
                  ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
                  POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR1(("VehicleClientHandler::onSpeedStatus  service is not available"));
   }
}


void VehicleClientHandler::onSpeedError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SpeedError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleClientHandler::onSpeedError"));
}


/******************************************************************************
*NAME        : onLanguageStatus
*Description : Setting the culture message on Language change.
******************************************************************************/
void VehicleClientHandler::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageStatus >& status)
{
   uint8 hmiindex = 0;
   std::vector<T_Language_SourceTable> langTable = status->getLangTable();
   uint8 langIndex = status->getLanguage();
   ETG_TRACE_USR4(("onLanguageStatus langIndex=%d", langIndex));
   std::string textid = getLanguageId(langIndex, hmiindex);

   COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(textid.c_str()))->Post();
   m_poCameraDataBindingHandler ->OnLanguageChange(hmiindex);
}


/******************************************************************************
*NAME        : getLanguageId
*Description : Function to get the Language ID
******************************************************************************/
std::string VehicleClientHandler::getLanguageId(int index, uint8& hmiImageIndex)
{
   std::string langId = "";
   for (int i = 0; i < (sizeof(LanguageMappingTable) / sizeof(LanguageMappingTable[0])); i++)
   {
      if (LanguageMappingTable[i].LangIndex == index)
      {
         ETG_TRACE_USR4(("onLanguageStatus Language string=%s", LanguageMappingTable[i].LangString.c_str()));
         langId = LanguageMappingTable[i].LangString.c_str();
         hmiImageIndex = LanguageMappingTable[i].hmiindex;
      }
   }
   return langId;
}


/******************************************************************************
*NAME        : readLanguageOnStartUp
*Description : Function to read the last active Language and set the Culture message at the startup.
******************************************************************************/
void VehicleClientHandler::readLanguageOnStartUp()
{
   int languageIndex = 0; //CID_R1 2584948
   uint8 hmiindex = 0;    //CID_R1 2584948
   ETG_TRACE_USR4(("VehicleClientHandler::setLanguageOnSystemLoad requested :%d", languageIndex));
   std::string textid = getLanguageId(languageIndex, hmiindex);

   COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(textid.c_str()))->Post();
   m_poCameraDataBindingHandler ->OnLanguageChange(hmiindex);
}


/******************************************************************************
*NAME        : SetCANUpdate
*Description : Function to set the value m_CANupdate= true when CAN requests for streaming.
******************************************************************************/
void VehicleClientHandler::SetCANUpdate(bool toggle)
{
   m_CANupdate = toggle;
}


/******************************************************************************
*NAME        : onLanguageStatus
*Description : Function to get the CAN status.
******************************************************************************/
bool VehicleClientHandler::GetCANUpdate()
{
   return m_CANupdate;
}


void VehicleClientHandler::onParkingBreakSwitchError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ParkingBreakSwitchError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleClientHandler::onParkingBreakSwitchError"));
}


void VehicleClientHandler::onParkingBreakSwitchStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ParkingBreakSwitchStatus >& status)
{
   ETG_TRACE_USR4(("VehicleClientHandler::onParkingBreakSwitchStatus ParkingBreakSwitch  = %d", status->getParkingBreakSwitchValue()));

   if (proxy == m_vehicleProxy)
   {
      m_parkBrakeStatus = status->getParkingBreakSwitchValue();
      updateSpeedAndParkingStatusValue(m_speedValue, m_parkBrakeStatus);
      int activityID = CameraMain::poGetInstance()->vGetActivityID();
      //special handling for partking brake status SNA, or OFF when any of the streaming screen is active
      if ((activityID == enActivityIDs__eActivityID_CAMERA_STREAM) ||
            (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW) ||
            (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT) || (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT_FULLSCREEN))
      {
         if (m_parkBrakeStatus == PARKING_BRAKE_SNA)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
            POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
         }
         if (m_parkBrakeStatus == PARKING_BRAKE_DISENG)
         {
            int _speedStatus = 0;
            int selectedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
            (selectedIndex == -1) ? selectedIndex = 0 : selectedIndex = selectedIndex;
            vector<int> camAvailableTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();
            if (camAvailableTextId.size() > 0)
            {
               CameraConfig::getInstance()-> getSpeedStatus(camAvailableTextId.at(selectedIndex), _speedStatus);
               if (_speedStatus == NONE)
               {
                  ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
                  POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
               }
            }
         }
      }
   }
   else
   {
      ETG_TRACE_USR1(("VehicleClientHandler::onParkingBreakSwitchStatus  service is not available"));
   }
}


void VehicleClientHandler::updateSpeedAndParkingStatusValue(uint16 speedValue, uint16 parkingBrakeStatus)
{
   ETG_TRACE_USR4(("VehicleClientHandler::updateSpeedAndParkingStatusValue() speedValue is :%d parking status is %d", (speedValue / 100), parkingBrakeStatus));
   if ((speedValue == ZERO_SPEED) && (parkingBrakeStatus == PARKING_BRAKE_ENG))
   {
      m_SpeedLockActive = false;   // Allow all cameras when Vehicle speed is 0 and parking brake engaged.
   }
   else
   {
      m_SpeedLockActive = true;
   }
}


bool VehicleClientHandler::getSpeedAndParkingStatus()
{
   return m_SpeedLockActive;
}


void VehicleClientHandler::checkAnyCameraActiveOnStartup()
{
   //sort the vector based on cameraName
   //check in the vector, which camera status is active
   //Request Camera activation based on active
   ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCameraActiveOnStartup"))
   vector<stCamCANActivation>::iterator itr = vecStCANCameraOnStartup.begin();
   int i = 0;
   int value = 0;
   for (itr = vecStCANCameraOnStartup.begin(); itr != vecStCANCameraOnStartup.end(); ++itr, i++)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCameraActiveOnStartup Loop %d status %d", i, vecStCANCameraOnStartup.at(i).actStatus))
      if (1 == vecStCANCameraOnStartup.at(i).actStatus)
      {
         ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCameraActiveOnStartup found camera activation"))
         HandleCamRequest(vecStCANCameraOnStartup.at(i).actStatus, vecStCANCameraOnStartup.at(i).cameraName, vecStCANCameraOnStartup.at(i).camSrcName);
         break;
      }
   }
   //Clear the contents once the activation is done.
   //vecStCANCameraOnStartup.clear();
}


bool VehicleClientHandler::checkAnyCANCameraActive()
{
   //sort the vector based on cameraName
   //check in the vector, which camera status is active
   //Request Camera activation based on active
   ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCANCameraActive"))
   vector<stCamCANActivation>::iterator itr = vecStCANCameraAfterStartup.begin();
   int i = 0;
   int value = 0;
   bool camActiveValue = false;
   for (itr = vecStCANCameraAfterStartup.begin(); itr != vecStCANCameraAfterStartup.end(); ++itr, i++)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCANCameraActive Loop %d status %d", i, vecStCANCameraAfterStartup.at(i).actStatus))
      if (1 == vecStCANCameraAfterStartup.at(i).actStatus)
      {
         ETG_TRACE_USR4(("VehicleClientHandler::checkAnyCANCameraActive found camera activation"))
         camActiveValue = true;
         //HandleCamRequest(vecStCANCameraAfterStartup.at(i).actStatus, vecStCANCameraAfterStartup.at(i).cameraName, vecStCANCameraAfterStartup.at(i).camSrcName);
         break;
      }
   }

   return camActiveValue;
}


/**
 * Description     : Timer expiry method on timeout
 *
 * @param[in]      : timer, data
 * @param[in]      : asf::core::Timer, boost::shared_ptr<asf::core::TimerPayload>
 * @return         : void
 */
void VehicleClientHandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   //If expired, check for the reason and fire SM message
   ETG_TRACE_USR4(("VehicleClientHandler::The RVC timer is Expiring"))
   if ((timer.getAct()  == _rvcTimer.getAct()) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
   {
      ETG_TRACE_USR4(("VehicleClientHandler::The RVC timer is expired"))
      bool camActive = false;
      camActive = checkAnyCANCameraActive();

      if (camActive == true)
      {
         vRequestForCameraActivationOnTimeout();
      }

      int activityID = CameraMain::poGetInstance()->vGetActivityID();
      ETG_TRACE_USR4(("VehicleClientHandler::Check activity ID %d", activityID));
      //After RVC disengaged,check for any CAN activated camera, If no camera active, and screen is
      //Camera Streaming, then switch to home screen.
      //Ex: RVC engaged, ActCam2 activate, ActCam2 deactivate, RVC disengage.
      if ((camActive == false) && (activityID == enActivityIDs__eActivityID_CAMERA_STREAM))
      {
         if (sm_gCurrentRegion == REGION_C)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_HOME);
         }
         else if (sm_gCurrentRegion == REGION_A)
         {
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CABINA_HOME);
         }
      }
   }
}


void VehicleClientHandler::vRequestForCameraActivationOnTimeout()
{
   //sort the vector based on cameraName
   //check in the vector, which camera status is active
   //Request Camera activation based on active
   ETG_TRACE_USR4(("VehicleClientHandler::vRequestForCameraActivationOnTimeout"))
   vector<stCamCANActivation>::iterator itr = vecStCANCameraAfterStartup.begin();
   int i = 0;
   int value = 0;
   for (itr = vecStCANCameraAfterStartup.begin(); itr != vecStCANCameraAfterStartup.end(); ++itr, i++)
   {
      ETG_TRACE_USR4(("VehicleClientHandler::vRequestForCameraActivationOnTimeout Loop %d status %d", i, vecStCANCameraAfterStartup.at(i).actStatus))
      if (1 == vecStCANCameraAfterStartup.at(i).actStatus)
      {
         ETG_TRACE_USR4(("VehicleClientHandler::vRequestForCameraActivationOnTimeout found camera activation"))
         HandleCamRequest(vecStCANCameraAfterStartup.at(i).actStatus, vecStCANCameraAfterStartup.at(i).cameraName, vecStCANCameraAfterStartup.at(i).camSrcName);
         break;
      }
   }
   //Clear the contents once the activation is done.
   //vecStCANCameraAfterStartup.clear();
}


//validite functions for speed value.
//speedStatus;   // SHOW WHILE DRIVING
//regionStatus;  // HMI BUTTON AVAILABLE IN REGION

//@ded6kor: 10/08/2023 Ignore the speed and brake validity. Return always TRUE.
//This is needed to ignore without CAN signals. Related Bug 1788290
bool VehicleClientHandler::bGetSpeedAndParkBrakeValidityStatus()
{
   //initialize default value as valid ( TRUE )
   ETG_TRACE_USR4(("VehicleClientHandler::bGetSpeedAndParkBrakeValidityStatus %d %d", m_speedValue, m_parkBrakeStatus));
   bool validityFlag = TRUE;
   // if ((m_speedValue < ZERO_SPEED) || (m_speedValue > MAX_SPEED)) //|| (m_parkBrakeStatus == PARKING_BRAKE_SNA))
   // {
   // validityFlag = FALSE; // Speed value is invalid ( FALSE)
   // }
   return validityFlag;
}


ETG_I_CMD_DEFINE((TraceCmd_onCameraTextStatus, "TextStatus %d %d %d %d %d %d %d %d", int, int, int, int, int, int, int, int))
void VehicleClientHandler::TraceCmd_onCameraTextStatus(int c1, int c2, int c3, int c4, int c5, int c6, int c7, int c8)
{
   ETG_TRACE_USR4(("VehicleClientHandler::TraceCmd_onCameraTextStatus"));
   VehicleClientHandler::getInstance()->Testcode(c1, c2, c3, c4, c5, c6, c7, c8);
}


ETG_I_CMD_DEFINE((TraceCmd_onCameraactcamStatus, "actcamStatus %d %d %s", int, enCameraCAN, ETG_I_STRING))
void VehicleClientHandler::TraceCmd_onCameraactcamStatus(int camValues, enCameraCAN camtype, std::string camSrcName)
{
   ETG_TRACE_USR4(("VehicleClientHandler::TraceCmd_onCameraTextStatus %d %d %s", camValues, camtype, camSrcName));
   VehicleClientHandler::getInstance()->HandleCamRequest(camValues, camtype, camSrcName);
}


}//core
}//App
