/**************************************************************************************
* @file         : CameraSourceChangeHandler.cpp
* @author       : ECG5- shanmugapriya Murugan
* @addtogroup   :
* @brief        : Class contains necessary business logic required to handle source selection.
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "../CameraSourceChangeHandler/CameraSourceChangeHandler.h"
#include "hall_std_if.h"
#include "CameraConfig/CameraConfig.h"
//workaround
#include "PluginConstants.h"
#include "ProjectPluginMsgs.h"
#include "CameraDefines.h"
#include "CameraMain.h"
#include "spm_core_fi_typesConst.h"
#include "CameraAvdeccClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::CameraSourceChangeHandler::
#include "trcGenProj/Header/CameraSourceChangeHandler.cpp.trc.h"
#endif

#define PORTNAME "applicationSwitchPort"
#define AUDIOSOURCEPORT "SxmAudioSourceChangePort"
#define SET_BIT( Data, BitNum) (Data) |= (1<<(BitNum));
#define CLR_BIT( Data, BitNum) (Data) &= (~(1<<(BitNum)));
#define VARIANT_EVOBUS  4
#define VARIANT_GENERIC 3
namespace App {
namespace Core {
//DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/CameraSourceChangeHandler", CameraSourceChangeHandler, Info);
#define DEFAULT_SUBSRC_ID -1
#define STATE_UNKNOWN 0


/******************************************************************************
*NAME        : CameraSourceChangeHandler
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
CameraSourceChangeHandler::CameraSourceChangeHandler():
   m_audioSourceChangeProxy(AudioSourceChangeProxy::createProxy(AUDIOSOURCEPORT, *this)),
   _hmiInfoServiceProxy(HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this)),
   disconnect_src(false),
   bListRequested(false),
   bPrimaryDetected(false),
   bCMP1Connected(false),
   bCMP2Connected(false),
   bCMP3Connected(false)
{
   ETG_I_REGISTER_FILE();
   _regionsMapOutStatusInfo_RegId = 0;
   _spmState = false;
   m_poCameraDataBindingHandler = CameraDataBindingHandler::poGetInstance();
   if (m_audioSourceChangeProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, AUDIOSOURCEPORT);
   }
   if (_hmiInfoServiceProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, "hmiinfoservicePort");
   }
   m_cameraCabinRestoreSrcID = SRC_INVALID;
   m_primaryCMPSystemStatus = 0;
   m_primaryCMPAvdeccSystemStatus = 0;
   m_primaryCMPNanoMsgStatus = 0;
   m_secondCMPSystemStatus = 0;
   m_thirdCMPSystemStatus = 0;
   m_CMASystemStatus = 0;
   cmpConnDisconnectStatus = 0;
   isPnmActivated = false;
   currentMic1Status = false;
   //initialize vector to default values before filling.
   for (int index = 0; index < stIPCamConnectionStatus.size(); index++)
   {
      stIPCamConnectionStatus[index].pos = -1;
      stIPCamConnectionStatus[index].srcNam = "";
      stIPCamConnectionStatus[index].camtextid = -1;
      stIPCamConnectionStatus[index].status = false;
   }
   m_A1GlassCount = 0;  // CID_R2 5936021.
   m_A2GlassCount = 0;  // CID_R2 5936021.
   fillIPCameraSrc();
   m_prevVideoMixIndex = 0;
   setMappingTextIDforCabinA2();
   _variantInfo = KDSHandler::poGetInstance()->getVariantInfoFromKDS();
   ETG_TRACE_USR4(("CameraSourceChangeHandler::CameraSourceChangeHandler _variantInfo==%d", _variantInfo));
}


/******************************************************************************
*NAME        : getInstance
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
CameraSourceChangeHandler* CameraSourceChangeHandler::getInstance()
{
   static CameraSourceChangeHandler CameraSourceChangeHandler;

   return &CameraSourceChangeHandler;
}


/******************************************************************************
*NAME        : ~CameraSourceChangeHandler
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
CameraSourceChangeHandler::~CameraSourceChangeHandler()
{
   if (m_poCameraDataBindingHandler != NULL)
   {
      delete m_poCameraDataBindingHandler;
      m_poCameraDataBindingHandler = NULL;
   }
   ETG_I_UNREGISTER_FILE();
}


/******************************************************************************
*NAME        : onAvailable
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onAvailable()"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);

   if (m_audioSourceChangeProxy == proxy)
   {
      m_audioSourceChangeProxy->sendActiveSourceRegister(*this);
      m_audioSourceChangeProxy->sendActiveSourceListRegister(*this);

      m_audioSourceChangeProxy->sendSourceListChangedRegister(*this);
      m_audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_OTHERS); // group id for camera
      ETG_TRACE_USR4(("CameraSourceChangeHandler::onAvailable(): valid proxy"));
   }
   if (_hmiInfoServiceProxy == proxy)
   {
      _hmiInfoServiceProxy->sendCabinsSystemsInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsSystemsInfoGet(*this);
      _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoGet(*this);
      _hmiInfoServiceProxy->sendCabinsVideoMixInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsVideoMixInfoGet(*this);
   }
}


/******************************************************************************
*NAME        : onUnavailable
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onUnavailable()"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);

   if (m_audioSourceChangeProxy == proxy)
   {
      m_audioSourceChangeProxy->sendDeregisterAll();
   }
   if (_hmiInfoServiceProxy == proxy)
   {
      _hmiInfoServiceProxy->sendDeregisterAll();
   }
}


/******************************************************************************
*NAME        : registerProperties
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::registerProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::registerProperties()"));
   if (_hmiInfoServiceProxy == proxy)
   {
      _regionsMapOutStatusInfo_RegId = _hmiInfoServiceProxy->sendRegionsMapOutStatusInfoRegister(*this);//Registering RegionMapout ---26.05.2020
      _hmiInfoServiceProxy->sendCabinsSystemsInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsSystemsInfoGet(*this);
      _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoGet(*this);
      _hmiInfoServiceProxy->sendCabinsVideoMixInfoRegister(*this);
      _hmiInfoServiceProxy->sendCabinsVideoMixInfoGet(*this);
      ETG_TRACE_USR4(("CameraSourceChangeHandler registerProperties _hmiInfoServiceProxy "));
   }
}


/******************************************************************************
*NAME        : deregisterProperties
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::deregisterProperties(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::deregisterProperties()"));
   _hmiInfoServiceProxy->sendCabinsSystemsInfoDeregisterAll();
   _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoDeregisterAll();
   _hmiInfoServiceProxy->sendCabinsSourceRestorationInfoDeregisterAll();
   _hmiInfoServiceProxy->sendCabinsVideoMixInfoDeregisterAll();
}


/******************************************************************************
*NAME        : onActivateSourceError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActivateSourceError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onActivateSourceError()"));
}


/******************************************************************************
*NAME        : onActivateSourceResponse
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActivateSourceResponse >& response)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onActivateSourceResponse()"));
}


/******************************************************************************
*NAME        : onDeactivateSourceError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onDeactivateSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< DeactivateSourceError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onDeactivateSourceError()"));
}


/******************************************************************************
*NAME        : onDeactivateSourceResponse
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onDeactivateSourceResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< DeactivateSourceResponse >& response)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onDeactivateSourceResponse()"));
}


/******************************************************************************
*NAME        : onActiveSourceError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActiveSourceError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActiveSourceError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onActiveSourceError()"));
}


/******************************************************************************
*NAME        : onActiveSourceUpdate
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActiveSourceUpdate(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActiveSourceUpdate >& update)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceUpdate"));
   sourceData source = update->getActiveSource();
   int32 activeSource = source.getSrcId();
   uint8 m_u8sinkId = source.getSinkId();
   ETG_TRACE_USR4(("onActiveSourceUpdate sourceID: %d sinkID: %u", activeSource, m_u8sinkId));
   uint32 connectionState = source.getConnectionState();;
   //temporary workaround to call media play, need to be removed
   if ((activeSource >= SRC_CAMERAPORT1_CAM1) && (activeSource <= SRC_CAMERAPORT1_SPLITSCREEN))
   {
      boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
      if (NULL != pluginData.get())
      {
         uint16 playAction = 0;
         pluginData->addEventData(playAction);
         ETG_TRACE_USR4(("CameraSourceChangeHandler:: post courier message()"));
         POST_MSG((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_CAMERA, CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION, pluginData)));
      }
   }
   // if ((activeSource == SRC_CAMERAPORT1_USB_VIDEO) || (activeSource == SRC_CAMERAPORT1_HDMI) || (activeSource == SRC_CMR1_USB1_VIDEO))
   // {
   // if (connectionState == MEDIA_STATE_CONNECTED)
   // {
   // CameraMain::poGetInstance() -> setToggleValue(false);
   // m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(0);
   // CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight remove
   // }
   // } //added CMR source to the check, fix for bug id 1459473
}


/******************************************************************************
*NAME        : onActiveSourceListError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActiveSourceListError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActiveSourceListError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListError"));
}


/******************************************************************************
*NAME        : onActiveSourceListUpdate
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onActiveSourceListUpdate(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< ActiveSourceListUpdate >& update)
{
   if (update != NULL)
   {
      int32 sourceID, subSourceID;
      uint16 sinkId;
      uint32 connectionState;
      ::std::vector< sourceData > activeSources = update->getActiveSourceList();
      bool isMic1Active = false;
      static int  prevActivesource = 0;
      ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListUpdate Size of the vector list received = %d", activeSources.size()));
      bool isCameraSrcInList = false;
      for (uint8 i = 0; i < activeSources.size(); i++)
      {
         sourceID = activeSources[i].getSrcId();
         subSourceID = activeSources[i].getSubSrcId();
         sinkId = activeSources[i].getSinkId();
         connectionState = activeSources[i].getConnectionState();

         ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListUpdate isMic1Active= %d prevActivesource= %d sourceID = %d,subSourceID = %d,sinkID:%u,connectionState:%u", isMic1Active, prevActivesource, sourceID, subSourceID, sinkId, connectionState));

         // This Snippit helps to know how long the mic is active or not
         //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
         if (sinkId == CABIN_A_SINK_ID)
         {
            if ((sourceID == SRC_MIC1_ANNOUNCEMENT) && (connectionState == MEDIA_STATE_CONNECTED))
            {
               prevActivesource = SRC_MIC1_ANNOUNCEMENT;
               isMic1Active = true;
               currentMic1Status = isMic1Active;
               ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListUpdate isMic1Active= %d prevActivesource= %d currentMic1Status==>%d", isMic1Active, prevActivesource, currentMic1Status));
            }
         }

         if (sinkId == CABIN_A_MEDIAGLASS1_SINK_ID)
         {
            if (connectionState == STATE_CONNECTED)
            {
               m_cameraCabinRestoreSrcID = SRC_INVALID;  //This variable is used only in case of CabinRestoration.
            }
            if (((SRC_CAMERAPORT1_CAM1 <= sourceID) && (sourceID <= SRC_CAMERAPORT1_SPLITSCREEN)) ||
                  ((SRC_CAMERAPORT2_CAM1 <= sourceID) && (sourceID <= SRC_CAMERAPORT2_SPLITSCREEN)) ||
                  ((SRC_CAMERAPORT3_CAM1 <= sourceID) && (sourceID <= SRC_CAMERAPORT3_SPLITSCREEN)) ||
                  ((SRC_CAMERAPORT4_CAM1 <= sourceID) && (sourceID <= SRC_CAMERAPORT4_SPLITSCREEN)))
            {
               if (connectionState == STATE_CONNECTED)
               {
                  isCameraSrcInList = true;
                  m_cameraSrcConnectionStatus = STATE_CONNECTED;
                  setCameraSourceInCabin(sourceID, m_cameraSrcConnectionStatus);
                  ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListUpdate connectionState is STATE_CONNECTED"));
                  int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
                  m_poCameraDataBindingHandler -> updateCameraNameInTile(CameraListHandler::getInstance()->getActiveCameraNameCabin(activeCamera));
                  m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(1);
                  CameraListHandler::getInstance()->setNoHighlightInList(false);  // cabin list highlight
                  CameraMain::poGetInstance() -> setToggleValue(true);   //enable the toggle button in the cabin hs.

                  int value = 0;
                  bool streamStatus = false;
                  std::string strCamSrc = "";
                  CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(activeCamera, value, strCamSrc, streamStatus);
                  VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, ACTIVATE);
               }
            }
            else
            {
               //setCameraSourceInCabin(SRC_INVALID, STATE_UNKNOWN ); commented for now.
            }
         }
      }

      // This Snippit helps to know how long the mic is active or not
      //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
      if ((isMic1Active  == false) && (prevActivesource == SRC_MIC1_ANNOUNCEMENT))
      {
         prevActivesource = 0; //Set To Default Value
         currentMic1Status = isMic1Active;
         ETG_TRACE_USR4(("CameraSourceChangeHandler::onActiveSourceListUpdate isMic1Active==>%d prevActivesource==>%d currentMic1Status==>%d", isMic1Active, prevActivesource, currentMic1Status));
      }

      if (isCameraSrcInList == false)
      {
         ETG_TRACE_USR4(("onActiveSourceListUpdate:: No cameras in the list"));
         vDeactivateManActForSelectedSourceInCabin();
         setCameraSourceInCabin(SRC_INVALID, STATE_UNKNOWN);
         //@ded6kor: Other sources like CMP USB, HDM, CMR USB connected, camera cabin tile  shall be updated. fix for bug 1459473
         //@ded6kor: Fix for update of camera status message in the cabin home screen. the listSize check is added to ensure
         //camera tile information is not overwritten.
         CameraMain::poGetInstance() -> setToggleValue(false);
         vector<int> camTextId = CameraListHandler::getInstance()->getAvailableCabinCamTxtIdOnSelection();
         if (camTextId.size() > 0)
         {
            m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(0);
         }
         CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight remove
      }
   }
}


/******************************************************************************
*NAME        : getCurrentMic1Status
* Description: Get the Current MIC Status
* Return Value: Bool
* 1 - ACTIVATED
* 0- DEACTIVATED
* Paramater : Void
  //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
******************************************************************************/
bool CameraSourceChangeHandler::getCurrentMic1Status()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: getCurrentMic1Status ==>%d", currentMic1Status));
   return currentMic1Status;
}


void CameraSourceChangeHandler::vDeactivateManActForSelectedSourceInCabin()
{
   int currentCanvalue4Cabin = 0;
   std::string currentStrCamSrc4Cabin = "";
   bool currentStreamStatus4Cabin = false;
   int currentCabinActiveCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();// get the Position of active camera in cabin.
   ETG_TRACE_USR4(("vDeactivateManActForSelectedSourceInCabin %d", currentCabinActiveCamera));
   if (currentCabinActiveCamera != -1)
   {
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(currentCabinActiveCamera, currentCanvalue4Cabin, currentStrCamSrc4Cabin, currentStreamStatus4Cabin);
      uint8 sourceID = CameraConfig::getInstance()->getCabinSrcIdForCam(currentStrCamSrc4Cabin);
      std::string sourceIdName = CameraConfig::getInstance()->getCabinSrcNameForCamId(sourceID);
      ETG_TRACE_USR4(("vDeactivateManActForSelectedSourceInCabin sourceID %d", sourceID));
      ETG_TRACE_USR4(("currCabinActivCam %d currStrCamSrc4Cabin %s", currentCabinActiveCamera, currentStrCamSrc4Cabin.c_str()));
      //manact is enabled corresponding to camsrc for efficient power utilization
      VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(currentStrCamSrc4Cabin, DEACTIVATE);
   }
}


/******************************************************************************
*NAME        : onGetSourceListError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onGetSourceListError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< GetSourceListError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onGetSourceListError()"));
}


/******************************************************************************
*NAME        : onGetSourceListResponse
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onGetSourceListResponse(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< GetSourceListResponse >& response)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onGetSourceListResponse()"));

   if (m_audioSourceChangeProxy == proxy)
   {
      const ::std::vector< sourceDetails >& SourceList = response->getSources();
      ::std::vector< sourceDetails >::const_iterator it = SourceList.begin();

      ETG_TRACE_USR4(("CameraSourceChangeHandler::onGetSourceListResponse Sourcelistsize:%d", SourceList.size()));

      while (it != SourceList.end())
      {
         if (it->getSrcId() >= SRC_CAMERAPORT1_CAM1)
         {
            ETG_TRACE_USR4(("onGetSourceListResponse :SrcId = %d ",  it->getSrcId()));
            ETG_TRACE_USR4(("onGetSourceListResponse :SubSrcId() = %d", it->getSubSrcId()));
            ETG_TRACE_USR4(("onGetSourceListResponse :getAvailability() = %d", it->getAvailability()));
            ETG_TRACE_USR4(("onGetSourceListResponse :getAvailabilityReason() = %d", it->getAvailabilityReason()));
         }
         ++it;
      }
   }
}


/******************************************************************************
*NAME        : onSourceListChangedError
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onSourceListChangedError(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< SourceListChangedError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onSourceListChangedError()"));
}


/******************************************************************************
*NAME        : onSourceListChangedSignal
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::onSourceListChangedSignal(const ::boost::shared_ptr< AudioSourceChangeProxy >& proxy, const ::boost::shared_ptr< SourceListChangedSignal >& signal)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onSourceListChangedSignal()"));
   if (m_audioSourceChangeProxy && signal->getGroupId() == GROUP_OTHERS)
   {
      ETG_TRACE_USR4(("sendGetSourceListRequest for camera"));
      //Request for the changed source list
      m_audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_OTHERS);
   }
}


/******************************************************************************
*NAME        : requestSourceActivation
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::requestSourceActivation(_Src SrcID, uint16 sinkId)
{
   sourceData srcData;
   srcData.setSrcId(SrcID);
   srcData.setSubSrcId(DEFAULT_SUBSRC_ID);
   _Sink currSinkId;

   if (sinkId == REG_A)
   {
      currSinkId = CABIN_A_MEDIAGLASS1_SINK_ID;
   }
   else if (sinkId == REG_B)
   {
      currSinkId = CABIN_B_MEDIAGLASS1_SINK_ID;
   }
   else
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::requestSourceActivation Invalid Region"));
      return;
   }
   ETG_TRACE_USR4(("CameraSourceChangeHandler::requestSourceActivation SrcID %d sinkId=%d currSinkId=%d", SrcID, sinkId, currSinkId));
   srcData.setSinkId(currSinkId);
   srcData.setConnectionState(STATE_UNKNOWN);

   ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendActivateSourceRequest(*this, srcData, true)));
}


/******************************************************************************
*NAME        : requestSourceDeActivation
*SYSFL       : SYSFL-5422, SYSFL-5423, SYSFL-5424, SYSFL-5425
******************************************************************************/
void CameraSourceChangeHandler::requestSourceDeActivation(_Src SrcID, uint16 sinkId)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::requestSourceDeActivation SrcID %d", SrcID));
   sourceData srcData;
   srcData.setSrcId(SrcID);
   srcData.setSubSrcId(DEFAULT_SUBSRC_ID);
   _Sink currSinkId;

   if (sinkId == REG_A)
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler:: ActiveRegion: REG_A"));
      currSinkId = CABIN_A_MEDIAGLASS1_SINK_ID;
   }
   else if (sinkId == REG_B)
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler:: ActiveRegion: REG_B"));
      currSinkId = CABIN_B_MEDIAGLASS1_SINK_ID;
   }
   else
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::Invalid Region"));
      return;
   }
   srcData.setSinkId(currSinkId);
   srcData.setConnectionState(STATE_UNKNOWN);

   ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendDeactivateSourceRequest(*this, srcData)));
}


void CameraSourceChangeHandler::onRegionsMapOutStatusInfoError(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE:: HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::RegionsMapOutStatusInfoError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onRegionsMapOutStatusInfoError() \n"));

   if ((proxy == _hmiInfoServiceProxy) && (error->getAct() == _regionsMapOutStatusInfo_RegId)) //for safety
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::onRegionsMapOutStatusInfoError Error Message in CondCheck."));
   }
}


void CameraSourceChangeHandler::onRegionsMapOutStatusInfoUpdate(const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<HMI_INFO_SERVICE_CHANGE::RegionsMapOutStatusInfoUpdate >& update)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler:: onRegionsMapOutStatusInfoUpdate() \n"));

   //RegionMapOutStatusInfo Iteration to know the RegionID
   ::std::vector< RegionMapOutStatusInfo> regionMapOutStatusInfoArrayData = update->getRegionsMapOutStatusInfo();
   ::std::vector<RegionMapOutStatusInfo>::const_iterator regionMapOutStatusInfo_iter = regionMapOutStatusInfoArrayData.begin();

   if ((proxy == _hmiInfoServiceProxy) && (update->getAct() == _regionsMapOutStatusInfo_RegId)) //for safety
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler:: onRegionsMapOutStatusInfoUpdate IFF LOOP \n"));

      while (regionMapOutStatusInfo_iter != regionMapOutStatusInfoArrayData.end())
      {
         int32  regionIdStatus = regionMapOutStatusInfo_iter->getRegionId();
         ETG_TRACE_USR4(("CameraSourceChangeHandler:: onRegionsMapOutStatusInfoUpdate regionIdStatus regionMapOutStatusInfo_iter =%d   \n", regionIdStatus));

         //MapOutStatusInfo Iteration to get glass id and Connection status getRegionsMapOutStatusInfo
         ::std::vector< MapOutStatusInfo> glassMapOutStatusInfoArrayData = regionMapOutStatusInfo_iter->getMapOutStatusInfo();
         ::std::vector<MapOutStatusInfo>::const_iterator glassMapOutStatus_iter = glassMapOutStatusInfoArrayData.begin();

         while (glassMapOutStatus_iter != glassMapOutStatusInfoArrayData.end())
         {
            int32   glassIdStatus = glassMapOutStatus_iter->getGlassId();
            int32   activeGlassStatus = glassMapOutStatus_iter->getActiveStatus();
            ETG_TRACE_USR4(("CameraSourceChangeHandler:: onRegionsMapOutStatusInfoUpdate glassIdStatus activeGlassStatus=%d %d\n", glassIdStatus, activeGlassStatus));

            if (activeGlassStatus != 0)
            {
               //CRQ 1366236 Handle(Disable) Camera Power Management at EvoBus in the Cabin Region during MAP OUT Enable
               // When the Mapout is Enabled ,  --> DEACTIVATE the power supply(can manact) Corresponding to the Selected Cabin Analog Camera
               vDeactivateManActForSelectedSourceInCabin();
               CameraMain::poGetInstance() -> setToggleValue(false);
               //m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(0); //  ded6kor commented for testing.
               CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight
            }
            else
            {}
            glassMapOutStatus_iter++;
         }
         regionMapOutStatusInfo_iter++;
      }
   }
}


void CameraSourceChangeHandler::onCabinsSystemsInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsSystemsInfoError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSystemsInfoError"));
}


void CameraSourceChangeHandler::onCabinsSystemsInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsSystemsInfoUpdate >& update)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSystemsInfoUpdate"));
   // int sourceId;
   if (update != NULL)
   {
      ::std::vector< SystemInfo > cabininfovector = update->getCabinsSystemsInfo();

      if (!cabininfovector.empty())
      {
         ::std::vector< SystemInfo >::const_iterator systemInfo_iter = cabininfovector.begin();

         if (proxy == _hmiInfoServiceProxy)
         {
            m_A1GlassCount = 0;  // reset always.
            m_A2GlassCount = 0;  // reset always.
            while (systemInfo_iter != cabininfovector.end())
            {
               uint32 systemId = systemInfo_iter->getSystemId();
               uint32 systemStatus = systemInfo_iter->getSystemStatus();
               uint32 systemCategory = systemInfo_iter->getSystemCategory();
               uint8 systemPosition = systemInfo_iter->getSystemPosition();
               uint8 systemGroup = systemInfo_iter->getSystemGroup();
               uint32 systemAvdeccStatus = systemInfo_iter->getAvdeccStatus();
               uint32 systemNanoMsgStatus = systemInfo_iter->getNanoMsgStatus();

               uint32 sourceId = SRC_INVALID;
               sourceId =  getCameraSourceInCabin();
               ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSystemsInfoUpdate systemId %d, systemPosition %d, systemGroup %d, systemStatus %d, systemCategory %d, source ID %d", systemId, systemPosition, systemGroup, systemStatus, systemCategory, sourceId));

               switch (systemCategory)
               {
                  case SYSTEM_CATEGORY_CMP:
                     vEvaluateCMP(systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus);
                     break;
                  case SYSTEM_CATEGORY_CMG:
                     vEvaluateCMG(systemGroup, systemStatus, systemAvdeccStatus, systemNanoMsgStatus);
                     break;
                  case SYSTEM_CATEGORY_CMA:
                     vEvaluateCMA(systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus);
                     break;
                  default:
                     break;
               }
               systemInfo_iter++;
            }

            //when count of A1 glass is 0, then all the glass is disconnected.
            if (0 == m_A1GlassCount)
            {
               ETG_TRACE_USR4(("CameraSourceChangeHandler::Disconnect actions on all glasses."));
               vA1GlassDisconnectionActions();

               //CRQ 1366236 Handle(DEACTIVATE) Camera Power Management at EvoBus in the Cabin Region during CMG-A1 Error
               //When All A2-Glasses are DisConnected --> DEACTIVATE the manact related to the Selected Cabin Analog Camera
               vDeactivateManActForSelectedSourceInCabin();
            }
            //when count of A2 glass is 0, then all the glass is disconnected.
            if (0 == m_A2GlassCount)
            {
               ETG_TRACE_USR4(("CameraSourceChangeHandler::Send Disconnect for A2 glass source."));
               //sendCabinVideoMixRequest(REG_B, CABIN_VIDEO_MIX_CHANGE_REASON_CMG_ERROR, CABIN_VIDEO_MIX_INDEX_NONE);
               //vA2GlassDisconnectionActions();
               //Hide the button, when the Glass A2 count is 0
               m_poCameraDataBindingHandler->vMixSourceAvailability(false);
            }
         }
      }
   }
}


// provide the  PNM status of CMA
bool CameraSourceChangeHandler::getCMApnmStatus()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::getCMApnmStatus isPnmActivated ==>%d", isPnmActivated));
   return isPnmActivated;
}


void CameraSourceChangeHandler::vEvaluateCMA(uint8 systemPosition, uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA systemPosition= %d, systemStatus= %d, systemAvdeccStatus= %d, systemNanoMsgStatus= %d", systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus));
   bool CMPStatus = getCMPStatus();
   bool CMGStatus = getCMGStatus();
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA CMP status %d CMG status = %d", CMPStatus, CMGStatus));

   if (m_CMASystemStatus != systemStatus)
   {
      m_CMASystemStatus = systemStatus;
      isPnmActivated = false; //Bug 1290363: Swivel able to open when cabin camera source is activated during PNM OFF
      if ((SYSTEM_STATUS_UNAVAILABLE  == systemStatus)
            || (SYSTEM_STATUS_DISCONNECTED == systemStatus)
            || (SYSTEM_STATUS_HMI_OFF_DISCONNECTED  == systemStatus)
            || (SYSTEM_STATUS_INACTIVE_DISCONNECTED  == systemStatus)
            || (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus)
            || (SYSTEM_STATUS_COCKPIT_DISCONNECTED  == systemStatus)
         )
      {
         // send source stop when Camera is one of the active source.
         if (m_cameraSrcID != SRC_INVALID)
         {
            CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
            m_cameraSrcID = SRC_INVALID;
         }
         ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA setSelectedCameraIndex is SETTING UP"));
         CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
         m_poCameraDataBindingHandler -> vCamAvailability(false);
         m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
         setCMAAvailable(false);
         m_cameraCabinRestoreSrcID = SRC_INVALID;

         //CRQ 1366236 Handle(DEACTIVATE) Camera Power Management at EvoBus in the Cabin Region during CMA Error
         // When the Amplifier is Disconnected,  --> DEACTIVATE the All the power supply(can manact)
         VehicleClientHandler::getInstance()->vDeactivateAllCameraManActCabin();
      }

      if ((SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus)
            || (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED  == systemStatus))
      {
         // send source stop when Camera is one of the active source.
         if (m_cameraSrcID != SRC_INVALID)
         {
            CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
            m_cameraSrcID = SRC_INVALID;
         }
         ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_PNM_DISCONNECTED OR SYSTEM_STATUS_HOURLOGIC_DISCONNECTED setSelectedCameraIndex is SETTING UP"));

         CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
         m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
         if (SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus)
         {
            //Bug 1290363: Swivel able to open when cabin camera source is activated during PNM OFF
            isPnmActivated = true;
            ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_PNM_DISCONNECTED"));
            m_poCameraDataBindingHandler -> vCamAvailability(true);
         }
         if (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED == systemStatus)
         {
            ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_HOURLOGIC_DISCONNECTED MODE"));

            if (_variantInfo == VARIANT_EVOBUS)
            {
               ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_HOURLOGIC_DISCONNECTED==> VARIANT_EVOBUS COCKPIT_TILE_DISENABLED"));
               m_poCameraDataBindingHandler -> vCamAvailability(false); // Disable Camera Tile in Cockpit
               vSwitchToCockpitHomeScreen(); // if screen is camera and System is in "On Hour Logic" --> Switch to Cockpit HomeScreen
            }
            if (_variantInfo ==  VARIANT_GENERIC)
            {
               if ((systemAvdeccStatus == TRUE) && (systemNanoMsgStatus == TRUE))
               {
                  // Note : Same is applicable for Cabin Camera Tile
                  ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_HOURLOGIC_DISCONNECTED==> VARIANT_GENERIC COCKPIT_TILE_ENABLED"));
                  m_poCameraDataBindingHandler -> vCamAvailability(true); // Disable Camera Tile in Cockpit
               }
               else
               {
                  ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA SYSTEM_STATUS_HOURLOGIC_DISCONNECTED==> VARIANT_GENERIC COCKPIT_TILE_DISENABLED"));
                  m_poCameraDataBindingHandler -> vCamAvailability(false); // Disable Camera Tile in Cockpit
                  vSwitchToCockpitHomeScreen(); // if screen is camera and System is in "On Hour Logic" --> Switch to Cockpit HomeScreen
               }
            }
         }
         setCMAAvailable(true);
         m_cameraCabinRestoreSrcID = SRC_INVALID;

         //CRQ 1366236 Handle(DEACTIVATE) Camera Power Management at EvoBus in the Cabin Region during CMA Error
         // When the Amplifier is Disconnected,  --> DEACTIVATE the All the power supply(can manact)
         vDeactivateManActForSelectedSourceInCabin();
      }

      if (SYSTEM_STATUS_CONNECTED  == systemStatus)
      {
         if ((CMPStatus == true) && (CMGStatus == true))
         {
            m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
            m_poCameraDataBindingHandler -> vCamAvailability(true);
            //vRequestCameraList();
         }

         setCMAAvailable(true);
      }
   }
   else
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMA no change in status"));
   }
}


// void CameraSourceChangeHandler::vEvaluateCMP(uint8 systemPosition, uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus)
// {
// ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMP systemPosition= %d, systemStatus= %d, systemAvdeccStatus= %d, systemNanoMsgStatus= %d", systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus));
// bool CMAStatus = getCMAStatus();
// bool CMGStatus = getCMGStatus();

// ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMP CMA status %d CMG status = %d", CMAStatus, CMGStatus));

// if (((SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus) || (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED  == systemStatus))
// && (CMAStatus == TRUE)
// && (systemAvdeccStatus == TRUE)
// && (systemNanoMsgStatus == TRUE))
// {
// CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
// m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
// m_poCameraDataBindingHandler -> vCamAvailability(true);
// bCMP1Connected = true;
// vRequestCameraList();
// }

// if (SYSTEM_STATUS_CONNECTED  == systemStatus)
// {
// switch (systemPosition)
// {
// case EN_POSITION_PRIMARY :
// bCMP1Connected = true;
// setCMPAvailable(true);    /// set this variable only if primary is available.
// vRequestCameraList();
// m_poCameraDataBindingHandler -> vCamAvailability(true);
// if ((CMAStatus == true) && (CMGStatus == true))
// {
// m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
// }
// else
// {
// m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
// }
// break;

// case EN_POSITION_SECONDARY :
// case EN_POSITION_TERTIARY :
// //(1) update camera list .. request the list only if primary is available.
// if (bCMP1Connected == true)
// {
// vRequestCameraList();
// }
// break;
// // case EN_POSITION_TERTIARY :
// // //(1) update camera list
// // vRequestCameraList();
// // break;

// default :
// break;
// }
// //setCMPAvailable(true);
// }

// if (SYSTEM_STATUS_DISCONNECTED  == systemStatus)
// {
// int sourceid = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
// int sourceidCabin = CameraListHandler::getInstance()->getSelectedCameraIndex();
// int value = 0;
// int valueCabin = 0;
// std::string strCamSrc = "";
// std::string strCamSrcCabin = "";
// bool streamStatus = false;
// bool streamStatusCabin = false;
// CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(sourceid, value, strCamSrc, streamStatus);
// CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(sourceidCabin, valueCabin, strCamSrcCabin, streamStatusCabin);
// switch (systemPosition)
// {
// case EN_POSITION_PRIMARY :
// //(1) update camera list
// //vRequestCameraList();   // Commenting this line as list request is not required on disconnection.
// //(2) disable cockpit camera tile
// //CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
// CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight
// m_poCameraDataBindingHandler -> vCamAvailability(false);
// //(3) disable cabin camera tile
// m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
// CameraMain::poGetInstance() -> vRequestStopCameraStream();
// CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
// CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
// vSwitchToHomeScreen();
// bCMP1Connected = false;
// setCMPAvailable(false);    /// set this variable only if primary is not available..
// break;

// case EN_POSITION_SECONDARY :
// case EN_POSITION_TERTIARY :
// if (bCMP1Connected == true)
// {
// vRequestCameraList();
// }
// //vRequestCameraList();
// if ((systemPosition == value) && (streamStatus == true))
// {
// CameraMain::poGetInstance() -> vRequestStopCameraStream();
// vSwitchToHomeScreen();
// //m_poCameraDataBindingHandler -> vCamAvailability(false);   // Shpuld remove since primary will be available
// }
// if ((systemPosition == valueCabin) && (m_cameraSrcID != SRC_INVALID))
// {
// CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
// vSwitchToHomeScreen();
// //m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false,false);
// }
// break;

// default :
// break;
// }
// //setCMPAvailable(false);
// }

// //(3)vUpdateCMPStatus(available / not available)
// }


void CameraSourceChangeHandler::vEvaluateCMG(uint8 systemGroup, uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus)
{
   int sourceidCabin = CameraListHandler::getInstance()->getSelectedCameraIndex();
   int valueCabin = 0;
   std::string strCamSrcCabin = "";
   bool streamStatusCabin = false;
   CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(sourceidCabin, valueCabin, strCamSrcCabin, streamStatusCabin);
   bool CMPStatus = getCMPStatus();
   bool CMAStatus = getCMAStatus();

   ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMG CMP status %d CMA status = %d SRC_ID %d", CMPStatus, CMAStatus, m_cameraSrcID));

   ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMG CMG s/m status %d", systemStatus));
   if ((SYSTEM_STATUS_UNAVAILABLE  == systemStatus)
         || (SYSTEM_STATUS_DISCONNECTED == systemStatus)
         || (SYSTEM_STATUS_HMI_OFF_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_INACTIVE_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus)
         || (SYSTEM_STATUS_COCKPIT_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus)
         || (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_NONE))
   {
#if 0

      ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMG SYSTEM_STATUS_PNM_DISCONNECTED OR SYSTEM_STATUS_HOURLOGIC_DISCONNECTED"));
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      m_poCameraDataBindingHandler -> vCamAvailability(false);			// Disable Camera Tile in Cockpit
      // Note : Same is applicable for Cabin Camera Tile

      vSwitchToHomeScreen();
      //  if (streamStatusCabin == true)
      if (m_cameraSrcID != SRC_INVALID)
      {
         CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
         CameraListHandler::getInstance()->setSelectedCameraIndex(-1);   // any glass disconnected, while streaming the index shall be set to -1
      }
      setCMGAvailable(false);
#endif
   }
   else
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMG CMG s/m status else part %d", systemStatus));
      if (SYSTEM_STATUS_CONNECTED  == systemStatus)
      {
         if (systemGroup == SYSTEM_GROUP_CMG_A)
         {
            ++m_A1GlassCount;
         }
         else if (systemGroup == SYSTEM_GROUP_CMG_A2)
         {
            ++m_A2GlassCount;
         }
         else
         {
         }
         if ((CMPStatus == true) && (CMAStatus == true))
         {
            m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
         }
         setCMGAvailable(true);
      }
   }

   // if (SYSTEM_STATUS_CONNECTED  == systemStatus)
   // {
   // ++m_A1GlassCount;
   // if ((CMPStatus == true) && (CMAStatus == true))
   // {
   // m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
   // //m_poCameraDataBindingHandler -> vCamAvailability(true);
   // //vRequestCameraList();
   // }
   // setCMGAvailable(true);
   // }
}


void CameraSourceChangeHandler::onCabinsSourceRestorationInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoError >& error)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoError"));
}


void CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy,
      const ::boost::shared_ptr< CabinsSourceRestorationInfoUpdate >& update)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate"));

   ::std::vector< CabinSourceRestorationInfo >restorevector = update->getCabinsSourceRestorationInfo();

   if (!restorevector.empty())
   {
      ::std::vector< CabinSourceRestorationInfo >::const_iterator cabinsSourceRestore_iter = restorevector.begin();

      if (proxy == _hmiInfoServiceProxy)
      {
         while (cabinsSourceRestore_iter != restorevector.end())
         {
            int cabinId = cabinsSourceRestore_iter->getCabinId();
            // int status = cabinsSourceRestore_iter->getStatus();

            //ETG_TRACE_USR4(("SrcChangeHandler::onCabinsSourceRestorationInfoUpdate : %d, cabinId %d ", status, cabinId));

            ::std::vector< SourceInfoHolder> SourcesInfoArrayData = cabinsSourceRestore_iter->getLastActiveSourcesInfo();
            ::std::vector<SourceInfoHolder>::const_iterator sourcesInfo_iter = SourcesInfoArrayData.begin();

            while (sourcesInfo_iter != SourcesInfoArrayData.end())
            {
               int lastActiveSource = sourcesInfo_iter->getSourceId();
               int16 sinkId = sourcesInfo_iter->getSinkId();
               uint32 status = sourcesInfo_iter->getStatus();
               ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate cabinId=%d lastActiveSource = %d sinkID %d status %d", cabinId, lastActiveSource, sinkId, status));

               if (((SRC_CAMERAPORT1_CAM1 <= lastActiveSource) && (lastActiveSource <= SRC_CAMERAPORT1_SPLITSCREEN))
                     || ((SRC_CAMERAPORT2_CAM1 <= lastActiveSource) && (lastActiveSource <= SRC_CAMERAPORT2_SPLITSCREEN))
                     || ((SRC_CAMERAPORT3_CAM1 <= lastActiveSource) && (lastActiveSource <= SRC_CAMERAPORT3_SPLITSCREEN))
                     || ((SRC_CAMERAPORT4_CAM1 <= lastActiveSource) && (lastActiveSource <= SRC_CAMERAPORT4_SPLITSCREEN)))
               {
                  ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate Restoration Entered"));

                  if ((status == CABIN_RESTORE_STATUS_EVALUATING) && (sinkId == CABIN_A_MEDIAGLASS1_SINK_ID))
                  {
                     ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate Restoration Evaluating"));
                     m_poCameraDataBindingHandler ->showRestorationText();
                  }

                  if (status == CABIN_RESTORE_STATUS_RESTORE)
                  {
                     ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate Requested from restore function "));
                     if (sinkId == CABIN_A_MEDIAGLASS1_SINK_ID)
                     {
                        ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate Requested CABIN_A_MEDIAGLASS1_SINK_ID A1"));
                        vRequestCameraSourceActivation(lastActiveSource);
                        m_cameraCabinRestoreSrcID = lastActiveSource;
                     }
                     if (sinkId == CABIN_A_MEDIAGLASS2_SINK_ID)
                     {
                        ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsSourceRestorationInfoUpdate Requested CABIN_A_MEDIAGLASS2_SINK_ID A2"));
                        requestSourceActDeactOnCabinA2Sink((_Src)lastActiveSource, REQ_ACTIVATE);
                     }
                  }
               }
               sourcesInfo_iter++;
            }
            cabinsSourceRestore_iter++;
         }
      }
   }
}


bool CameraSourceChangeHandler::getSpmHourLogicState()
{
   // TRUE if system status is in HL or PNM mode.
   // FALSE  if system is connected. This is default initialization.
   return _spmState;
}


uint32 CameraSourceChangeHandler::getCameraSourceInCabin()
{
   return m_cameraSrcID;
}


void CameraSourceChangeHandler::setCameraSourceInCabin(uint32 sourceID, int status)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::setCameraSourceInCabin srcid = %d status = %d", sourceID, status));
   m_cameraSrcID = sourceID;
   m_cameraSrcConnectionStatus = status;
}


void CameraSourceChangeHandler::vRequestCameraSourceActivation(int16 srcid)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vRequestCameraSourceActivation srcid = %d", srcid));

   //fetch cameraSrc name using the ID.
   std::string camCabinSrcName =  CameraConfig::getInstance()->getCabinSrcNameForCamId(srcid);
   //fetch the list index to get the active camera.
   int activeCamera = CameraListHandler::getInstance()->bCheckRequestedCabinCameraExists(camCabinSrcName);

   if (activeCamera != -1)
   {
      ETG_TRACE_USR4(("CameraListHandler::vRequestCameraSourceActivation setSelectedCameraIndex is SETTING UP activeCamera==>%d, sourceID==>%d camCabinSrcName==>%s", activeCamera, srcid, camCabinSrcName));
      CameraListHandler::getInstance()->setSelectedCameraIndex(activeCamera);
      CameraListHandler::getInstance()->setNoHighlightInList(false);  // cabin list highlight
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
      //m_poCameraDataBindingHandler -> updateCameraNameInTile(CameraListHandler::getInstance()->getActiveCameraNameCabin(activeCamera));

      int value = 0;
      bool streamStatusCabin = false;
      std::string strCamSrc = "";
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(activeCamera, value, strCamSrc, streamStatusCabin);
      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         //set the split cam bits and splitcam contril id.
         CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, SPLTSCRNCAMS, value, strCamSrc);
         //depending on the position, change the cockpitSplitview1 to CMP1_SPLITSCREEN ( and so on )
         if (value == 1)
         {
            strCamSrc = "CMP1_SPLITSCREEN";
         }
         if (value == 2)
         {
            strCamSrc = "CMP2_SPLITSCREEN";
         }
         if (value == 3)
         {
            strCamSrc = "CMP3_SPLITSCREEN";
         }
      }

      CameraMain::poGetInstance()->setToggleValue(true);   // added for bugfix on startup toggle.
      CameraListHandler::getInstance()->setprevCabinSelectedIndex(activeCamera);
      // send request to Audio
      sourceData srcData;
      srcData.setSrcId(srcid);
      srcData.setSubSrcId(DEFAULT_SUBSRC_ID);
      srcData.setConnectionState(STATE_UNKNOWN);
      srcData.setSinkId(CABIN_A_MEDIAGLASS1_SINK_ID);
      ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendActivateSourceRequest(*this, srcData, true)));
   }
   else
   {
      ETG_TRACE_USR4(("CameraListHandler::vRequestCameraSourceActivation sourceID not part of list"));
   }

// send request to Audio
   // sourceData srcData;
   // srcData.setSrcId(srcid);
   // srcData.setSubSrcId(DEFAULT_SUBSRC_ID);
   // srcData.setConnectionState(STATE_UNKNOWN);
   // srcData.setSinkId(CABIN_A_MEDIAGLASS1_SINK_ID);
   // ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendActivateSourceRequest(*this, srcData, true)));
}


void CameraSourceChangeHandler::vRequestCameraList()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vRequestCameraList cmpConnDisconnectStatus %d", cmpConnDisconnectStatus));
   //CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CAMLIST_STARTUP, 0, "CAMERA");

   int value = 0;
   ETG_TRACE_USR4(("CameraSourceChangeHandler::requestCameraListDataFromHMI %d", cmpConnDisconnectStatus));
   switch (cmpConnDisconnectStatus)
   {
      case 1:   // 001
      case 5:   // 101
         value = 1;
         break;
      case 3:
         value = 2;
         break;
      case 7:
         value = 3;
         break;
      default:
         value = 0; //clear list
         break;
   }
//	CameraAvdeccClientHandler::getInstance()->sendCameraListDataToHMI(value);
   CameraAvdeccClientHandler::getInstance()->requestCameraListDataFromHMI(value);
}


void CameraSourceChangeHandler::vSwitchToCockpitHomeScreen()
{
   int activityID = CameraMain::poGetInstance()->vGetActivityID();
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vSwitchToCockpitHomeScreen activity ID %d", activityID));
   //When camera is streaming in cockpit
   if ((activityID == 250) || (activityID == 251) || (activityID == 255))
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_HOME);
   }
}


void CameraSourceChangeHandler::vSwitchToCabinHomeScreen()
{
   int activityID = CameraMain::poGetInstance()->vGetActivityID();
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vSwitchToCabinHomeScreen activity ID %d", activityID));
   //When camera is streaming in cabin
   if (activityID == 252)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CABINA_HOME);
   }
}


bool CameraSourceChangeHandler::getPrimaryCMPConnectedStatus()
{
   return bCMP1Connected;
}


void CameraSourceChangeHandler::setCMPAvailable(bool flag)
{
   _CMPAvailable = flag; //CID_R1 11378981
}


void CameraSourceChangeHandler::setCMAAvailable(bool flag)
{
   _CMAAvailable = flag; //CID_R1 11378976
}


void CameraSourceChangeHandler::setCMGAvailable(bool flag)
{
   _CMGAvailable = flag; //CID_R1 11415697
}


bool CameraSourceChangeHandler::getCMPStatus()
{
   return _CMPAvailable;
}


bool CameraSourceChangeHandler::getCMAStatus()
{
   return _CMAAvailable;
}


bool CameraSourceChangeHandler::getCMGStatus()
{
   return _CMGAvailable;
}


void CameraSourceChangeHandler::vCheckIfSourceToBeRestored()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vCheckIfSourceToBeRestored source id %d & status %d", m_cameraCabinRestoreSrcID, m_cameraSrcConnectionStatus));
   if (((SRC_CAMERAPORT1_CAM1 <= m_cameraCabinRestoreSrcID) && (m_cameraCabinRestoreSrcID <= SRC_CAMERAPORT1_SPLITSCREEN))
         || ((SRC_CAMERAPORT2_CAM1 <= m_cameraCabinRestoreSrcID) && (m_cameraCabinRestoreSrcID <= SRC_CAMERAPORT2_SPLITSCREEN))
         || ((SRC_CAMERAPORT3_CAM1 <= m_cameraCabinRestoreSrcID) && (m_cameraCabinRestoreSrcID <= SRC_CAMERAPORT3_SPLITSCREEN))
         || ((SRC_CAMERAPORT4_CAM1 <= m_cameraCabinRestoreSrcID) && (m_cameraCabinRestoreSrcID <= SRC_CAMERAPORT4_SPLITSCREEN)))

   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler:: REQUESTING SOURCE RESTORATION"));
      vRequestCameraSourceActivation(m_cameraCabinRestoreSrcID);
   }

   ETG_TRACE_USR4(("CameraSourceChangeHandler::vCheckIfSourceToBeRestored A G P %d %d %d", getCMAStatus(), getCMGStatus(), getCMPStatus()));
   if ((TRUE == getCMAStatus()) && (TRUE == getCMGStatus()) && (TRUE == getCMPStatus()))
   {
      //enable the tile.
      m_poCameraDataBindingHandler -> vCamAvailability(true);
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vCheckIfSourceToBeRestored Invoke checkDigPinActiveOnSubSystemAvailable To Check Digital Pin  Activated "));
      NanoMsgClientHandler::poGetInstance()->checkDigPinActiveOnSubSystemAvailable();
   }
}


void CameraSourceChangeHandler::vA1GlassDisconnectionActions()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vA1GlassDisconnectionActions"));
   m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
   vSwitchToCabinHomeScreen();   //vSwitchToHomeScreen();
   //  if (streamStatusCabin == true)
   if (m_cameraSrcID != SRC_INVALID)
   {
      CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vA1GlassDisconnectionActions() setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(-1);   // any glass disconnected, while streaming the index shall be set to -1
   }
   setCMGAvailable(false);
}


void CameraSourceChangeHandler::vA2GlassDisconnectionActions()    // to recheck this method.
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vA2GlassDisconnectionActions"));
   if (m_cameraSrcID != SRC_INVALID)
   {
//     requestSourceActDeactOnCabinA2Sink(_Src SrcID, int request);
   }
//  setCMGAvailable(false);
}


void CameraSourceChangeHandler::vEvaluateCMP(uint8 systemPosition, uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMP systemPosition= %d, systemStatus= %d, systemAvdeccStatus= %d, systemNanoMsgStatus= %d", systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus));

   if (1 == systemPosition)
   {
      if (m_primaryCMPSystemStatus != systemStatus)
      {
         m_primaryCMPAvdeccSystemStatus = systemAvdeccStatus;
         m_primaryCMPNanoMsgStatus = systemNanoMsgStatus;
         m_primaryCMPSystemStatus = systemStatus;
         performPrimaryCMPActions(systemStatus, systemAvdeccStatus, systemNanoMsgStatus, systemPosition);
      }
      //Bug 1629374: LCM: Cockpit camera did not grey out after CMP power source disconnected
      if (systemStatus == SYSTEM_STATUS_PNM_DISCONNECTED) //||(systemStatus==SYSTEM_STATUS_HOURLOGIC_DISCONNECTED)
      {
         m_primaryCMPSystemStatus = systemStatus;
         ETG_TRACE_USR4(("CameraSourceChangeHandler::vEvaluateCMP is SYSTEM_STATUS_PNM_DISCONNECTED m_primaryCMPAvdeccSystemStatus %d m_primaryCMPNanoMsgStatus= %d", m_primaryCMPAvdeccSystemStatus, m_primaryCMPNanoMsgStatus));

         if ((m_primaryCMPAvdeccSystemStatus != systemAvdeccStatus) || (m_primaryCMPNanoMsgStatus != systemNanoMsgStatus))
         {
            m_primaryCMPAvdeccSystemStatus = systemAvdeccStatus;
            m_primaryCMPNanoMsgStatus = systemNanoMsgStatus;
            performPrimaryCMPActions(systemStatus, systemAvdeccStatus, systemNanoMsgStatus, systemPosition);
         }
      }
   }

   if (2 == systemPosition)
   {
      if (m_secondCMPSystemStatus != systemStatus)
      {
         m_secondCMPSystemStatus = systemStatus;
         performSecondCMPActions(systemStatus, systemAvdeccStatus, systemNanoMsgStatus, systemPosition);
      }
   }

   if (3 == systemPosition)
   {
      if (m_thirdCMPSystemStatus != systemStatus)
      {
         m_thirdCMPSystemStatus = systemStatus;
         performSecondCMPActions(systemStatus, systemAvdeccStatus, systemNanoMsgStatus, systemPosition);
         //performThirdCMPActions(systemStatus);
      }
   }
}


void CameraSourceChangeHandler::performPrimaryCMPActions(uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus, uint8 systemPosition)
{
   bool CMAStatus = getCMAStatus();
   bool CMGStatus = getCMGStatus();
   ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions CMA %d CMG %d s/m status %d", CMAStatus, CMGStatus, systemStatus));
   ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions SP= %d, SS= %d, AvS= %d, NanoMsgStatus= %d", systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus));

   // Disable Camera Tile in Cockpit  // Note : Same is applicable for Cabin Camera Tile
   if (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED == systemStatus)
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions  SYSTEM_STATUS_HOURLOGIC_DISCONNECTED MODE"));
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      VehicleClientHandler::getInstance()->vDeactivateAllCameraManActCabin();
      CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
      if (_variantInfo == VARIANT_EVOBUS)
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions  SYSTEM_STATUS_HOURLOGIC_DISCONNECTED MODE==>VARIANT_EVOBUS,COCKPIT_TILE_DISENABLED"));
         m_poCameraDataBindingHandler -> vCamAvailability(false);
         CameraMain::poGetInstance() -> vRequestStopCameraStream();
         bCMP1Connected = false;
         CLR_BIT(cmpConnDisconnectStatus, 0)
         _spmState = true;
         vSwitchToCockpitHomeScreen(); // if screen is camera and System is in "On Hour Logic" --> Switch to Cockpit HomeScreen
      }
      if (_variantInfo == VARIANT_GENERIC)
      {
         if ((systemAvdeccStatus == TRUE) && (systemNanoMsgStatus == TRUE))
         {
            ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions  SYSTEM_STATUS_HOURLOGIC_DISCONNECTED MODE==>VARIANT_GENERIC,COCKPIT_TILE_ENABLED"));
            m_poCameraDataBindingHandler -> vCamAvailability(true);
            bCMP1Connected = true;
            SET_BIT(cmpConnDisconnectStatus, 0);
            vRequestCameraList();
            ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions Request camera list on pnm - hl"));
         }
         else
         {
            ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions  SYSTEM_STATUS_HOURLOGIC_DISCONNECTED MODE==>VARIANT_GENERIC,COCKPIT_TILE_DISENABLED"));
            m_poCameraDataBindingHandler -> vCamAvailability(false);
            CameraMain::poGetInstance() -> vRequestStopCameraStream();
            bCMP1Connected = false;
            CLR_BIT(cmpConnDisconnectStatus, 0)
            _spmState = true;
            vSwitchToCockpitHomeScreen(); // if screen is camera and System is in "On Hour Logic" --> Switch to Cockpit HomeScreen
         }
      }
   }
   if (SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus)
   {
      CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions  SYSTEM_STATUS_PNM_DISCONNECTED"));
      //Bug 1629374: LCM: Cockpit camera did not grey out after CMP power source disconnected
      if ((systemAvdeccStatus == SYSTEM_STATUS_CONNECTED) && (systemNanoMsgStatus == SYSTEM_STATUS_CONNECTED))
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions power status ok"));
         m_poCameraDataBindingHandler -> vCamAvailability(true);
         bCMP1Connected = true;
         SET_BIT(cmpConnDisconnectStatus, 0);
         vRequestCameraList();
      }
      else
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions power status not ok. hence disabling cockpit "));
         m_poCameraDataBindingHandler -> vCamAvailability(false);
         bCMP1Connected = false;
         CLR_BIT(cmpConnDisconnectStatus, 0);
         VehicleClientHandler::getInstance()->vDeactivateAllCameraManActCabin();
      }
   }

   if (SYSTEM_STATUS_CONNECTED  == systemStatus)
   {
      SET_BIT(cmpConnDisconnectStatus, 0);
      bCMP1Connected = true;
      setCMPAvailable(true);    /// set this variable only if primary is available.
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions Request camera list on connected cmpConnDisconnectStatus %d sm_gRVCStatus %d sm_gCMPStatus %d ", cmpConnDisconnectStatus, sm_gRVCStatus, sm_gCMPStatus));
      vRequestCameraList();
      m_poCameraDataBindingHandler -> vCamAvailability(true);

      // added by pga5cob-> For Bug 1828166 --> This will ensure CMA and CMP Is Been Connected
      //                                    --> once RVC is Enabled and available then manact1 will be activated
      if ((sm_gRVCStatus == 1) && (true == CameraListHandler::getInstance()->bCheckRearCameraAvailable()))
      {
         VehicleClientHandler::getInstance()->sendActivateCam1ManActStartRequest();
      }

      if ((CMAStatus == true) && (CMGStatus == true))
      {
         m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
      }
      else
      {
         m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      }
      _spmState = false;
      //videoMix changes
      stIPCamConnectionStatus[0].status = true;
      vSendIPCameraSourceStatus();
   }

   if (SYSTEM_STATUS_DISCONNECTED  == systemStatus)
   {
      CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight remove
      m_poCameraDataBindingHandler -> vCamAvailability(false);
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      CameraMain::poGetInstance() -> vRequestStopCameraStream();
      CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions SYSTEM_STATUS_DISCONNECTED setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
      bCMP1Connected = false;
      setCMPAvailable(false);    /// set this variable only if primary is not available..
      vSwitchToCockpitHomeScreen();
      vSwitchToCabinHomeScreen();
      CLR_BIT(cmpConnDisconnectStatus, 0)
      VehicleClientHandler::getInstance()->vDeactivateAllCameraManActCabin();
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performPrimaryCMPActions Disconnected %d", cmpConnDisconnectStatus));
      //videoMix changes
      //when primary is disconnected, we should also send disconnection for second & third CMP ip cam if available.
      stIPCamConnectionStatus[0].status = false;
      vSendIPCameraSourceStatus();
   }

   if ((SYSTEM_STATUS_UNAVAILABLE  == systemStatus)
         || (SYSTEM_STATUS_HMI_OFF_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_INACTIVE_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus)
         || (SYSTEM_STATUS_COCKPIT_DISCONNECTED  == systemStatus)
      )
   {
      m_poCameraDataBindingHandler -> vCamAvailability(false);
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(false, false);
      bCMP1Connected = false;
      setCMPAvailable(false);    /// set this variable only if primary is not available..
      CLR_BIT(cmpConnDisconnectStatus, 0)
      VehicleClientHandler::getInstance()->vDeactivateAllCameraManActCabin();
      //videoMix changes
      vSendIPCameraSourceStatus();
   }
}


void CameraSourceChangeHandler::performSecondCMPActions(uint32 systemStatus, uint32 systemAvdeccStatus, uint32 systemNanoMsgStatus, uint8 systemPosition)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions  systemStatus %d %d", systemStatus, m_cameraSrcID));
   ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions SP= %d, SS= %d, AvS= %d, NanoMsgStatus= %d", systemPosition, systemStatus, systemAvdeccStatus, systemNanoMsgStatus));
   if (SYSTEM_STATUS_CONNECTED  == systemStatus)
   {
      if (systemPosition == 2)
      {
         SET_BIT(cmpConnDisconnectStatus, 1)
      }
      if (systemPosition == 3)
      {
         SET_BIT(cmpConnDisconnectStatus, 2)
      }
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Connected %d", cmpConnDisconnectStatus));
      if (bCMP1Connected == true)
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Request camera list on connect 2nd or 3rd"));
         vRequestCameraList();
      }
      //videoMix changes
      stIPCamConnectionStatus[systemPosition - 1].status = true;
      vSendIPCameraSourceStatus();
   }

   if (SYSTEM_STATUS_DISCONNECTED  == systemStatus)
   {
      vResetListIndexesToDefault();
      //Check if some camera is streaming in cabin
      if (m_cameraSrcID != SRC_INVALID)
      {
         CameraMain::poGetInstance() -> vRequestStopCabinCameraStream();
         vSwitchToCockpitHomeScreen();
         vSwitchToCabinHomeScreen(); //vSwitchToHomeScreen();
         //homescreen tile should show camera streaming stopped in this condition
      }
      //Check if some camera is streaming in cockpit
      int activityID = CameraMain::poGetInstance()->vGetActivityID();
      int cptListIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
      std::string strCamSrc = "";
      int value = 0;
      bool streamStatus = false;

      ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions activity ID %d", activityID));
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(cptListIndex, value, strCamSrc, streamStatus);

      ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions strCamSRC %s", strCamSrc.c_str()));

      if (activityID == enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION)
      {
         vSwitchToCockpitHomeScreen(); //vSwitchToCabinHomeScreen();vSwitchToHomeScreen();
      }

      if (((0 == strcmp(strCamSrc.c_str(), "CMP2_CAM1")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM2")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM3")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM4")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM1")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM2")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM3")) ||
            (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM4")))
            &&
            ((activityID == enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION) ||
             (activityID == enActivityIDs__eActivityID_CAMERA_STREAM) ||
             (activityID == enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW)))
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions. switch to screen "));
         CameraMain::poGetInstance() -> vRequestStopCameraStream();
         vSwitchToCockpitHomeScreen();
         vSwitchToCabinHomeScreen();//vSwitchToHomeScreen();
         //homescreen tile should show camera streaming stopped in this condition.
      }

      // clear the bits for second CMP or third cmp.
      if (systemPosition == 2)
      {
         CLR_BIT(cmpConnDisconnectStatus, 1)
         VehicleClientHandler::getInstance()->vDeactivateSecondaryNdTeritoryCameraManActInCabin();
      }
      if (systemPosition == 3)
      {
         CLR_BIT(cmpConnDisconnectStatus, 2)
         VehicleClientHandler::getInstance()->vDeactivateTeritoryCameraManActInCabin();
      }

      ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Disconnected %d", cmpConnDisconnectStatus));
      if (bCMP1Connected == true)
      {
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Request camera list on disconnect 2nd or 3rd"));
         vRequestCameraList();
      }
      //videoMix changes
      stIPCamConnectionStatus[systemPosition - 1].status = false;
      vSendIPCameraSourceStatus();
//	  vResetListIndexesToDefault();
   }

   if ((SYSTEM_STATUS_PNM_DISCONNECTED == systemStatus) || (SYSTEM_STATUS_HOURLOGIC_DISCONNECTED  == systemStatus))
   {
      if ((systemAvdeccStatus == TRUE) && (systemNanoMsgStatus == TRUE))
      {
         if (systemPosition == 2)
         {
            SET_BIT(cmpConnDisconnectStatus, 1)
         }
         if (systemPosition == 3)
         {
            SET_BIT(cmpConnDisconnectStatus, 2)
         }
         if (bCMP1Connected == true)
         {
            ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Request camera list on pnm error for  2nd or 3rd cmp"));
            vRequestCameraList();
         }
         ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Request camera list on pnm - hl on second or third CMP"));
      }
      else
      {
         if (systemPosition == 2)
         {
            CLR_BIT(cmpConnDisconnectStatus, 1)
            VehicleClientHandler::getInstance()->vDeactivateSecondaryNdTeritoryCameraManActInCabin();
         }
         if (systemPosition == 3)
         {
            CLR_BIT(cmpConnDisconnectStatus, 2)
            VehicleClientHandler::getInstance()->vDeactivateTeritoryCameraManActInCabin();
         }
      }
   }

   if ((SYSTEM_STATUS_UNAVAILABLE  == systemStatus)
         || (SYSTEM_STATUS_HMI_OFF_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_INACTIVE_DISCONNECTED  == systemStatus)
         || (SYSTEM_STATUS_USER_OFF_DISCONNECTED == systemStatus)
         || (SYSTEM_STATUS_COCKPIT_DISCONNECTED  == systemStatus)
      )
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::performSecondCMPActions Clear bits for secondary or third CMP on error cases."));
      if (systemPosition == 2)
      {
         CLR_BIT(cmpConnDisconnectStatus, 1)
         VehicleClientHandler::getInstance()->vDeactivateSecondaryNdTeritoryCameraManActInCabin();
      }
      if (systemPosition == 3)
      {
         CLR_BIT(cmpConnDisconnectStatus, 2)
         VehicleClientHandler::getInstance()->vDeactivateTeritoryCameraManActInCabin();
      }
   }
}


void CameraSourceChangeHandler::onSetCabinSourceConnectionStatusError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< SetCabinSourceConnectionStatusError >& error)
{
}


void CameraSourceChangeHandler::onSetCabinSourceConnectionStatusResponse(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< SetCabinSourceConnectionStatusResponse >& update)
{
}


void CameraSourceChangeHandler::fillIPCameraSrc()
{
   ETG_TRACE_COMP(("CameraSourceChangeHandler::fillIPCameraSrc"));
   tVecCamData tc;
   tclAvRoutingParser* _avRoutingParser = tclAvRoutingParser::pGetInstance();
   if (_avRoutingParser != NULL)
   {
      bool m_bParseState = _avRoutingParser->bParseXml();
      if (m_bParseState == true)
      {
         tc =   _avRoutingParser->vecGetIPCamSrcData();
         ETG_TRACE_USR3(("CameraSourceChangeHandler : tVevdata  %d", tc.size()));
      }
   }

   ETG_TRACE_COMP(("CameraSourceChangeHandler::fillIPCameraSrc"));
   bool status = false;
   for (int index = 0; index < tc.size(); index++)
   {
      stIPCamConnectionStatus[index].pos = tc[index].pos;
      stIPCamConnectionStatus[index].srcNam = tc[index].srcNam;
      stIPCamConnectionStatus[index].camtextid = tc[index].camtextid;
      stIPCamConnectionStatus[index].status = false;
   }

   m_IPCamCount = 0;
   std::vector < CabinSourceConnectionStatusInfo > cabinSourceConnectionStatusInfo;
   for (int index = 0; index < stIPCamConnectionStatus.size(); index++)
   {
      ETG_TRACE_COMP(("CameraSourceChangeHandler::fillIPCameraSrc Default first time signal --> loop "));
      if (stIPCamConnectionStatus[index].camtextid != 65535)
      {
         ETG_TRACE_COMP(("CameraSourceChangeHandler::fillIPCameraSrc --> valid text "));
         std::string strCam = "";

         uint32 camvalue = 0;
         CameraConfig::getInstance()-> getCameraValues(stIPCamConnectionStatus[index].camtextid, camvalue);
         _Src sourceId = CameraConfig::getInstance()->getCabinSrcIdForCam(stIPCamConnectionStatus[index].srcNam.c_str());
         bool status = stIPCamConnectionStatus[index].status;

         cabinSourceConnectionStatusInfo.push_back(CabinSourceConnectionStatusInfo(sourceId, status, strCam, camvalue));
         ++m_IPCamCount;
      }
      ETG_TRACE_COMP(("CameraSourceChangeHandler::fillIPCameraSrc <-- loop "));
   }

   _hmiInfoServiceProxy->sendSetCabinSourceConnectionStatusRequest(*this, cabinSourceConnectionStatusInfo);
}


void CameraSourceChangeHandler::vSendIPCameraSourceStatus()
{
   std::vector < CabinSourceConnectionStatusInfo > cabinSourceConnectionStatusInfo;
   for (int index = 0; index < stIPCamConnectionStatus.size(); index++)
   {
      ETG_TRACE_COMP(("CameraSourceChangeHandler::vSendIPCameraSourceStatus --> loop "));
      if (stIPCamConnectionStatus[index].camtextid != 65535)
      {
         ETG_TRACE_COMP(("CameraSourceChangeHandler::vSendIPCameraSourceStatus --> valid text "));
         std::string strCam = "";
         uint32 camvalue = 0;
         CameraConfig::getInstance()-> getCameraValues(stIPCamConnectionStatus[index].camtextid, camvalue);
         _Src sourceId = CameraConfig::getInstance()-> getCabinSrcIdForCam(stIPCamConnectionStatus[index].srcNam.c_str());
         bool status = stIPCamConnectionStatus[index].status;
         ETG_TRACE_COMP(("CameraSourceChangeHandler::vSendIPCameraSourceStatus %d, %d, %u", status, sourceId, camvalue));
         cabinSourceConnectionStatusInfo.push_back(CabinSourceConnectionStatusInfo(sourceId, status, strCam, camvalue));
      }
      ETG_TRACE_COMP(("CameraSourceChangeHandler::vSendIPCameraSourceStatus <-- loop "));
   }
   _hmiInfoServiceProxy->sendSetCabinSourceConnectionStatusRequest(*this, cabinSourceConnectionStatusInfo);
}


void CameraSourceChangeHandler::onCabinsVideoMixInfoError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsVideoMixInfoError >& error)
{
}


void CameraSourceChangeHandler::onCabinsVideoMixInfoUpdate(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< CabinsVideoMixInfoUpdate >& update)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsVideoMixInfoUpdate"));
   ::std::vector<CabinVideoMixInfo>cabinInfovector = update->getCabinsVideoMixInfo();
   std::string strCamSrc = "";
   _Src srcIdIPCam = SRC_INVALID;
   if (!cabinInfovector.empty())
   {
      ::std::vector<CabinVideoMixInfo>::const_iterator cabinInfo_iter = cabinInfovector.begin();
      if (proxy == _hmiInfoServiceProxy)
      {
         while (cabinInfo_iter != cabinInfovector.end())
         {
            uint32 cabinId = cabinInfo_iter->getCabinId();
            uint8 videoMixIndex = cabinInfo_iter->getVideoMixIndex();
            uint8 reason = cabinInfo_iter->getReason();

            if (CABIN_VIDEO_MIX_CHANGE_REASON_USER == reason)
            {
               m_prevVideoMixIndex = videoMixIndex;
            }

            ETG_TRACE_USR4(("CameraSourceChangeHandler::onCabinsVideoMixInfoUpdate cabinId %d , videoMixIndex(N/M/C1/C2/C3) %d , reason(NO/US/RS/EA/EG/EC) %d, m_prevVideoMixIndex  %d", cabinId, videoMixIndex, reason, m_prevVideoMixIndex));

            switch (videoMixIndex)
            {
               case CABIN_VIDEO_MIX_INDEX_NONE:
                  m_poCameraDataBindingHandler -> vMixSourceText(0, videoMixIndex);
                  CameraMain::poGetInstance() -> setMixToggleFlag(false);
                  vSendSrcDeactivationOnPrevMixIndex(m_prevVideoMixIndex, reason);
                  break;

               case CABIN_VIDEO_MIX_INDEX_MAP_OUT:
                  m_poCameraDataBindingHandler -> vMixSourceText(0, videoMixIndex);
                  break;

               case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1:
                  CameraMain::poGetInstance() -> setMixToggleFlag(true);
                  strCamSrc = "CMP1_IPCAM";
                  srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
                  requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_ACTIVATE);
                  break;

               case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_2:
                  strCamSrc = "CMP2_IPCAM";
                  srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
                  requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_ACTIVATE);
                  break;

               case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_3:
                  strCamSrc = "CMP3_IPCAM";
                  srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
                  requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_ACTIVATE);
                  break;

               default:
                  break;
            }

            if ((CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1 == videoMixIndex) || (CABIN_VIDEO_MIX_INDEX_IP_CAMERA_2 == videoMixIndex) || (CABIN_VIDEO_MIX_INDEX_IP_CAMERA_3 == videoMixIndex))
            {
               map<int, uint32>::iterator itr = cameraNamefromCabinA2.begin();
               uint32 camvalue = 0;
               for (; itr != cameraNamefromCabinA2.end(); ++itr)
               {
                  if (itr->first == videoMixIndex)
                  {
                     int textId = itr-> second;
                     CameraConfig::getInstance()-> getCameraValues(textId, camvalue);
                     m_poCameraDataBindingHandler -> vMixSourceText(camvalue, videoMixIndex);
                  }
               }
            }
            cabinInfo_iter++;
         }
      }
   }
}


void CameraSourceChangeHandler::setMappingTextIDforCabinA2()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::setMappingTextIDforCabinA2"));
   cameraNamefromCabinA2.insert(pair<int, uint32>(2, stIPCamConnectionStatus[0].camtextid));
   cameraNamefromCabinA2.insert(pair<int, uint32>(3, stIPCamConnectionStatus[1].camtextid));
   cameraNamefromCabinA2.insert(pair<int, uint32>(4, stIPCamConnectionStatus[2].camtextid));
}


int CameraSourceChangeHandler::getIPCamCount()
{
   return m_IPCamCount;
}


/*Function to Update Cabin Video Mix Request Updateto Master*/
void CameraSourceChangeHandler::sendCabinVideoMixRequest(uint32 region, uint8 reason, uint8 index)
{
   //CabinVideoMixInfo cvmi;
   // cvmi.setCabinId(region);
   // cvmi.setReason(reason);
   // cvmi.setVideoMixIndex(index);
   CabinVideoMixInfo cvmi(REGION_CABIN_A, reason, index);
   ETG_TRACE_USR4(("CameraSourceChangeHandler::sendCabinVideoMixRequest region(A/B/C), reason(N/U/R/A/G/C), index(N/M/P1/P1/P3)==>%d %d %d", region, reason, index));
   _hmiInfoServiceProxy->sendSetCabinVideoMixIndexRequest(*this, cvmi);
}


void CameraSourceChangeHandler::onSetCabinVideoMixIndexError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< SetCabinVideoMixIndexError >& error)
{
}


void CameraSourceChangeHandler::onSetCabinVideoMixIndexResponse(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< SetCabinVideoMixIndexResponse >& response)
{
}


/******************************************************************************
*NAME        : requestSourceActDeactOnCabinA2Sink
*SYSFL       :
******************************************************************************/
void CameraSourceChangeHandler::requestSourceActDeactOnCabinA2Sink(_Src SrcID, int request)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::requestSourceActDeactOnCabinA2Sink SrcID %d request=%d ", SrcID, request));
   sourceData srcData;
   srcData.setSrcId(SrcID);
   srcData.setSubSrcId(DEFAULT_SUBSRC_ID);
   _Sink currSinkId = CABIN_A_MEDIAGLASS2_SINK_ID;

   srcData.setSinkId(currSinkId);
   srcData.setConnectionState(STATE_UNKNOWN);

   if (request == REQ_ACTIVATE)
   {
      ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendActivateSourceRequest(*this, srcData, true)));
   }
   else if (request == REQ_DEACTIVATE)
   {
      ((m_audioSourceChangeProxy.get()) && (m_audioSourceChangeProxy->sendDeactivateSourceRequest(*this, srcData)));
   }
   else
   {
      // do nothing.
   }
}


/******************************************************************************
*NAME        : vSendSrcDeactivationOnPrevMixIndex
*SYSFL       :
******************************************************************************/
void CameraSourceChangeHandler::vSendSrcDeactivationOnPrevMixIndex(uint8 previousIndex, uint8 reason)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vSendSrcDeactivationOnPrevMixIndex previousIndex %d reason(NO/US/RS/EA/EG/EC) %d ", previousIndex, reason));
   std::string strCamSrc = "";
   _Src srcIdIPCam = SRC_INVALID;

   if (CABIN_VIDEO_MIX_CHANGE_REASON_CMP_ERROR == reason)
   {
      switch (previousIndex)
      {
         case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1:
            strCamSrc = "CMP1_IPCAM";
            srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
            requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_DEACTIVATE);
            break;

         case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_2:
            strCamSrc = "CMP2_IPCAM";
            srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
            requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_DEACTIVATE);
            break;

         case CABIN_VIDEO_MIX_INDEX_IP_CAMERA_3:
            strCamSrc = "CMP3_IPCAM";
            srcIdIPCam = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
            requestSourceActDeactOnCabinA2Sink(srcIdIPCam, REQ_DEACTIVATE);
            break;

         default:
            break;
      }
   }
}


void CameraSourceChangeHandler::vResetListIndexesToDefault()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexesToDefault"));

   //Check if any camera source is active in Cabin region. Checking with AudioSrc ID, since they are used in Cabin only.
   if (((SRC_CAMERAPORT1_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT1_SPLITSCREEN)) ||
         ((SRC_CAMERAPORT2_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT2_SPLITSCREEN)) ||
         ((SRC_CAMERAPORT3_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT3_SPLITSCREEN)) ||
         ((SRC_CAMERAPORT4_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT4_SPLITSCREEN)))
   {
      //check if the source is from primary camport in cabin
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexesToDefault -> Any CMP1 camera is active"));
      if (((SRC_CAMERAPORT2_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT2_SPLITSCREEN)) ||
            ((SRC_CAMERAPORT3_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT3_SPLITSCREEN)))
      {
         //reset camera list index of cabin list

         ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexesToDefault setSelectedCameraIndex is SETTING UP and Any CMP2/3 camera is active in cabin"));
         CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
      }
      //check if some camera from 2nd or 3rd cmp is streaming in cockpit
      if (true == bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit())
      {
         //reset camera list index of cockpit list.
         ETG_TRACE_USR4(("vResetListIndexesToDefault -> Any CMP2/3 camera is active in cockpit"));
         CameraListHandler::getInstance()->setSelectedCockpitCameraIndex(-1);
      }
   } // non camera source active
   else
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexesToDefault -> No CMP camera is active"));
      if (true == bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit())
      {
         //reset camera list index of cockpit list.

         ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexesToDefault No CMP camera is active. reset cockpit index setSelectedCameraIndex is SETTING UP"));
         CameraListHandler::getInstance()->setSelectedCockpitCameraIndex(-1);
      }

      if (true == bCheckWhetherCameraFromCMP2OrCMP3StreamingInCabin())
      {
         ETG_TRACE_USR4(("vResetListIndexesToDefault -> No CMP camera is active. reset cabin index"));
         CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
      }
   }
}


bool CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit"));
   int cptListIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
   std::string strCamSrc = "";
   int value = 0;
   bool streamStatus = false;
   bool bReturnVal = false;

   ETG_TRACE_USR4(("CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit"));
   CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(cptListIndex, value, strCamSrc, streamStatus);

   ETG_TRACE_USR4(("CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit strCamSRC %s", strCamSrc.c_str()));

   if ((0 == strcmp(strCamSrc.c_str(), "CMP2_CAM1")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM2")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM3")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP2_CAM4")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM1")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM2")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM3")) ||
         (0 == strcmp(strCamSrc.c_str(), "CMP3_CAM4")))
   {
      bReturnVal = true;
   }
   ETG_TRACE_USR4(("bCheckWhetherCameraFromCMP2OrCMP3StreamingInCockpit return value %d", bReturnVal));
   return bReturnVal;
}


bool CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCabin()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::bCheckWhetherCameraFromCMP2OrCMP3StreamingInCabin"));
   bool bReturnVal = false;
   if (((SRC_CAMERAPORT2_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT2_SPLITSCREEN)) ||
         ((SRC_CAMERAPORT3_CAM1 <= m_cameraSrcID) && (m_cameraSrcID <= SRC_CAMERAPORT3_SPLITSCREEN)))
   {
      bReturnVal = true;
   }
   ETG_TRACE_USR4(("bCheckWhetherCameraFromCMP2OrCMP3StreamingInCabin return value %d", bReturnVal));
   return bReturnVal;
}


void CameraSourceChangeHandler::vResetListIndexOnCMP1Disconnection()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vResetListIndexOnCMP1Disconnection -- reset list index for both list setSelectedCameraIndex is SETTING UP"));

   CameraListHandler::getInstance()->setSelectedCockpitCameraIndex(-1);
   CameraListHandler::getInstance()->setSelectedCameraIndex(-1);
}


int CameraSourceChangeHandler::getA2GlassCount()
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::getA2GlassCount %d", m_A2GlassCount));
   return m_A2GlassCount;
}


/************************************************************************
*FUNCTION: 		 vRequestVideoMixRestore
*DESCRIPTION: 	 Function to Restore  the source id (for the ipcamera of camport-1,2,3)
                 on SINK ID 2 (A2 CAMERA)
*PARAMETER:		 int16 srcid
*RETURNVALUE: 	 void

************************************************************************/
void CameraSourceChangeHandler::vRequestVideoMixRestore(int16 srcid)
{
   ETG_TRACE_USR4(("CameraSourceChangeHandler::vRequestVideoMixRestore srcid %d", srcid));
   int vidMixIndex = 0;
   bool IsCameraSource = false;
   switch (srcid)
   {
      case SRC_CAMERAPORT1_IPCAM:
         vidMixIndex = CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1;
         IsCameraSource = true;
         break;
      case SRC_CAMERAPORT2_IPCAM:
         vidMixIndex = CABIN_VIDEO_MIX_INDEX_IP_CAMERA_2;
         IsCameraSource = true;
         break;
      case SRC_CAMERAPORT3_IPCAM:
         vidMixIndex = CABIN_VIDEO_MIX_INDEX_IP_CAMERA_3;
         IsCameraSource = true;
         break;
      default:
         break;
   }
   if (IsCameraSource == true)
   {
      ETG_TRACE_USR4(("CameraSourceChangeHandler::vRequestVideoMixRestore Request vidMixIndex(N/M/P1/P1/P3)=%d", vidMixIndex));
      sendCabinVideoMixRequest(REG_B, CABIN_VIDEO_MIX_CHANGE_REASON_USER, vidMixIndex);
   }
}


int CameraSourceChangeHandler::getCMPConnectionStatus()
{
   return cmpConnDisconnectStatus;
}


}//core
}//App
