/**************************************************************************************
* @file         : CameraMain.cpp
* @author       : ECG5 - Deepak
* @addtogroup   : Camera  HALL
* @brief        : Class contains necessary business logic required to handle Camera data.
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "hall_std_if.h"
#include "CameraMain.h"
#include "CameraSourceChangeHandler/CameraSourceChangeHandler.h"
#include "Common/ApplicationSwitchClientComponent/ApplicationSwitchClientComponent.h"
#include "CameraConfig/CameraConfig.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::CameraMain::
#include "trcGenProj/Header/CameraMain.cpp.trc.h"
#endif

#define PORTNAME "applicationSwitchPort"
#define CAMERA_SRC_SELECTION 1
#define CAMERA_SRC_CABIN_SELECTION 2
#define CAMERA_SRC_CABIN_TILE_SELECTION 253
#define CAMERA_SRC_TILE_SELECTION 254
#define CAMERA_DUAL_SPLIT 4
#define CAMERA_QUAD_SPLIT 5
#define CAMERA_IP 6

namespace App {
namespace Core {
#define ANALOGCAM_START_INDEX 1
DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core/CameraMain", CameraMain, Info);
CameraMain* CameraMain::m_poCameraMain = NULL;

CameraMain::CameraMain():
   m_poApplicationSwitchProxy(ApplicationSwitchProxy::createProxy(PORTNAME, *this)),
   m_poRvcFiProxy(::rvc_cca_fi::Rvc_cca_fiProxy::createProxy("rvcFiPort", *this)),
   _camBlkScreen_Timer(*this, KDSHandler::poGetInstance()->u16GetCameraBlackScreenTimeValue())
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR3(("CameraMain : Constructor"));
   m_u8activeRegion = 2; //Default initialized to cockpit
   m_sourceID = SRC_INVALID;
   m_ToggleVisible = true;
   m_CamVisible = true;
   m_toggle = false;
   m_MixSrctoggle = false;
   m_control_id = 0;
   m_subSystemAvailability.clear();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   m_poCameraDataBindingHandler = CameraDataBindingHandler::poGetInstance();
   m_CMPPrevCount = 0;
   m_CMPCount = 0;
   currentKdsValue = 0; //CID_R2 215536
   cameraBlackScreenDelayTime = 1;

   _avRoutingParser = tclAvRoutingParser::pGetInstance();
   if (_avRoutingParser != NULL)
   {
      bool m_bParseState = _avRoutingParser->bParseXml();
   }
   m_selectedSplitSource = "";
   m_CMPPosition = 0;
   vAppflyinStatusInfo(false);
   m_pau8DelayTimeValue = KDSHandler::poGetInstance()->u16GetCameraBlackScreenTimeValue();
   m_RvcMsgPositionInfo = KDSHandler::poGetInstance()->u8GetRvcMsgPositionValue();
   ETG_TRACE_USR3(("CameraMain::CameraMain m_RvcMsgPositionInfo (%d) m_pau8DelayTimeValue (%d)", m_RvcMsgPositionInfo, m_pau8DelayTimeValue));
}


/**
* Description     : Destructor of class CameraMain
*/
CameraMain::~CameraMain()
{
   m_poApplicationSwitchProxy.reset();
   m_poRvcFiProxy.reset();
   if (m_poCameraMain != NULL)
   {
      delete m_poCameraMain;
      m_poCameraMain = NULL;
   }
   if (m_poCameraDataBindingHandler != NULL)
   {
      delete m_poCameraDataBindingHandler;
      m_poCameraDataBindingHandler = NULL;
   }
   ETG_I_UNREGISTER_FILE();
}


void CameraMain::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
}


void CameraMain::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
}


/**
* Description     : Request from Framework to start de-registering for properties.
*
* @param[in]      : proxy : reference to the base class of all proxies
* @param[in]      : stateChange : represents a state transition of a service
* @return         : void
*/
void CameraMain::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("CameraMain::UnAvailable()"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdDeregisterAll();
      m_poApplicationSwitchProxy->sendSig_ActivateContextDeregisterAll();
   }
}


/*****************************************************************************
* @brief   Add Function details here
*****************************************************************************/
void CameraMain::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR3(("CameraMain::onAvailable()"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdRegister(*this);
      m_poApplicationSwitchProxy->sendSig_ActivateContextRegister(*this);
   }
   if (m_poRvcFiProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, m_poRvcFiProxy->getPortName());
      ETG_TRACE_USR4(("RearViewCamera Constructor after creating proxy"));
   }
}


/**
* Description     : To handle back button in the Cockpit Camera list scene.
*/
bool CameraMain::onCourierMessage(const CameraBackButtonMsg& corfoMsg)
{
   const CameraBackButtonMsg* copoCameraBackButtonMsg = Courier::message_cast<const CameraBackButtonMsg*>(&corfoMsg);

   if (sm_gCurrentRegion == REGION_C)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_HOME);
   }
   else if (sm_gCurrentRegion == REGION_A)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CABINA_HOME);
   }
   else
   {
      //do nothinf.
   }
   return true;
}


bool CameraMain::checkCabinSplitCamAvailability(splitViewSrcIdCabinMap& currentsrcIdMap, const uint8& cabinButton)
{
   bool retVal = false;
   splitViewSrcIdCabinMap::const_iterator pos = currentsrcIdMap.find(cabinButton);

   if (pos != currentsrcIdMap.end())
   {
      retVal = true;
   }
   else
   {
      ETG_TRACE_USR4(("ERR: cabin source not handled (%d)", cabinButton));
   }
   return retVal;
}


bool CameraMain::onCourierMessage(const CabinSourceMsg& corfoMsg)
{
   ETG_TRACE_USR4(("CameraMain::onCourierMessage CabinSourceMsg& corfoMsg)"));//CID_R2 215536
   return true;
}


/**
* Description     : Function to toggle between Full Screen and Normal screen view.
*/
bool CameraMain::onCourierMessage(const CameraToggleButton& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::CameraToggleVisiblevalue(%d)", m_ToggleVisible));

   if (m_ToggleVisible == false) //enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW
   {
      ETG_TRACE_USR3(("CameraMain::CameraToggleVisiblevalue enActivityIDs__eActivityID_CAMERA_STREAM_NORMALVIEW m_ToggleVisible (%d)", m_ToggleVisible));
      m_ToggleVisible = true;
      m_poCameraDataBindingHandler -> vCameraStreamToggle(m_ToggleVisible);
      _applicationSwitchClientComponent->requestSendContext(m_u8activeRegion, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CAMERA_STREAM);
   }
   else //enActivityIDs__eActivityID_CAMERA_STREAM (NORMAL--> DEFAULT)
   {
      ETG_TRACE_USR3(("CameraMain::CameraToggleVisiblevalue enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW m_ToggleVisible (%d)", m_ToggleVisible));
      m_ToggleVisible = false;
      m_poCameraDataBindingHandler -> vCameraStreamToggle(m_ToggleVisible);
      _applicationSwitchClientComponent->requestSendContext(m_u8activeRegion, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW);
   }

   ApplicationSwitchClientComponent::poGetInstance()->requestShowHideAppFlyInScene(m_ToggleVisible);
   int listSeletedIndex =  CameraListHandler::getInstance()->getSelectedCockpitCameraIndex(); // get the  listselected ROW value (i.e to confirm RVC is at row 1st row )
   bool rvcAvailable = CameraListHandler::getInstance()->bCheckRearCameraAvailable();

// Applicable to Only RVC Mode  Fix for bug id 1488310
   if ((listSeletedIndex == 0) && (true == rvcAvailable))
   {
      m_poCameraDataBindingHandler->vShowCameraWarningText(true);
   }
   else
   {
      m_poCameraDataBindingHandler->vShowCameraWarningText(false);
   }
   return true;
}


bool CameraMain::SetkdsheaderfooterValue(bool kdsheaderfooter)
{
   currentKdsValue = kdsheaderfooter;
   ETG_TRACE_USR3(("CameraMain::SetkdsheaderfooterValue currentKdsValue==>%d kdsheaderfooter==>%d", currentKdsValue, kdsheaderfooter));
   return currentKdsValue;
}


bool CameraMain::Currentkdsheaderfooterupdate()
{
   ETG_TRACE_USR3(("CameraMain::Currentkdsheaderfooterupdate currentKdsValue==>%d ", currentKdsValue));
   return currentKdsValue;
}


/**
* Description     : Function to handle exit message for cockpit.
*/
bool CameraMain::onCourierMessage(const CameraExitMsg& corfoMsg)
{
   bool CANUpdate = VehicleClientHandler::getInstance()->GetCANUpdate();
   ETG_TRACE_USR4(("CameraMain::onCourierMessage(const CameraExitMsg getCurrentMic1Status sm_gRVCStatus CANUpdate m_u8activeRegion %d %d %d %d", CameraSourceChangeHandler::getInstance()->getCurrentMic1Status(), sm_gRVCStatus, CANUpdate, m_u8activeRegion));
   if (m_u8activeRegion == REGION_C)
   {
      if ((CANUpdate != true))
      {
         //POST_MSG((COURIER_MESSAGE_NEW(CameraSourceSelectionMsg)()));
      }
      //Bug 1474584: When camera is streaming in HU, if MIC1 is activated and then camera streaming is stopped, MIC1 cannot be de-activated
      //ContextSwitch to Default Scene should be blocked when (CANupate is true and MIC is active OR rvc is active)
      else if ((CANUpdate == true) && ((sm_gRVCStatus == false) || (false == CameraSourceChangeHandler::getInstance()->getCurrentMic1Status())))
      {
         VehicleClientHandler::getInstance()->SetCANUpdate(false);

         //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
         //Below Comment  -->to Resolve the priority order 1st RVC 2nd MIC 3rd Camera/CanCamera and Default Context switch is removed on Camera Streaming Exit-
         // and Context switch is handled in vehicleclienthandler file in handlecanRequest Method- Deactivation Region
         //  ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_STREAM, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0);
      }
      //    CameraListHandler::getInstance()->setCameraStreamStatusCockpit(sourceid, false);
   }
   return true;
}


/**
* Description     : Function to handle Exit message in the cabin.
*/
bool CameraMain::onCourierMessage(const CabinCameraExitMsg& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::CabinCameraExitMsg"));
   _Src sourceId;
   enCamera selCamera = INVALID;
   int SelectedIndex = CameraListHandler::getInstance()->getSelectedCameraIndex();
   if (m_toggle == true)
   {
      m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(0);
      CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight remove
      vRequestStopCabinCameraStream();
      m_toggle = false;
      ETG_TRACE_USR4(("CameraMain::CabinCameraExitMsg:vRequestStopCabinCameraStream toggle = %d", m_toggle));
      //CRQ 1366236 Enable Camera Power Management at EvoBus
      int value = 0;
      bool streamStatus = false;
      std::string strCamSrc = "";
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(SelectedIndex, value, strCamSrc, streamStatus);
      VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, DEACTIVATE);
   }
   else if (m_toggle == false)
   {
      if (SelectedIndex < 0)
      {
         SelectedIndex = 0;
      }
      else
      {
         SelectedIndex = CameraListHandler::getInstance()->getSelectedCameraIndex();
      }
      ETG_TRACE_USR4(("CameraMain::CabinCameraExitMsg:vStartCameraStream %d", SelectedIndex));
      ETG_TRACE_USR4(("CameraMain::CabinCameraExitMsg:vStartCameraStream toggle = %d", m_toggle));
      m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(1);  // ded6kor uncommented
      ETG_TRACE_USR4(("CameraMain::CabinCameraExitMsg:vStartCameraStream setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(SelectedIndex);
      CameraListHandler::getInstance()->setNoHighlightInList(false);  // cabin list highlight
      vStartCameraStream();
      m_toggle = true;

      //CRQ 1366236 Enable Camera Power Management at EvoBus
      // int value = 0;
      // bool streamStatus = false;
      // std::string strCamSrc = "";
      // CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(SelectedIndex, value, strCamSrc, streamStatus);
      // VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, ACTIVATE);
   }
   CameraListHandler::getInstance()->setCameraStreamStatusCabin(SelectedIndex, false);

   return true;
}


/**
* Description     : Function to handle Next button in cabin Camera tile.
*/
bool CameraMain::onCourierMessage(const NextCameraActivate& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::NextCameraActivate"));
   int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
   ETG_TRACE_USR4(("CameraMain::Cabin Camera activeCamera=%d", activeCamera));
   if ((activeCamera >= 0) && (activeCamera <= MAX_CABIN_CAMERA))
   {
      _Src sourceId = SRC_INVALID;
      (activeCamera == MAX_CABIN_CAMERA) ? activeCamera = MAX_CABIN_CAMERA : activeCamera = activeCamera + 1;
      ETG_TRACE_USR4(("CameraMain::onCourierMessage(const NextCameraActivate& corfoMsg) setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(activeCamera);
      m_poCameraDataBindingHandler -> updateCameraNameInTile(CameraListHandler::getInstance()->getActiveCameraNameCabin(activeCamera));
      HandleRequestCabin();
      sourceId = getSource();
      if (sourceId != SRC_INVALID)
      {
         CameraSourceChangeHandler::getInstance()->requestSourceActivation(sourceId, m_u8activeRegion);
      }
   }
   if (activeCamera == (MAX_CABIN_CAMERA - 1))
   {
      m_poCameraDataBindingHandler ->vShowHomeScreenNextPrevButtons(2);
   }
   else
   {
      m_poCameraDataBindingHandler ->vShowHomeScreenNextPrevButtons(1);
   }
   return true;
}


/**
* Description     : Function to handle previous button in cabin camera tile.
*/
bool CameraMain::onCourierMessage(const PreviousCameraActivate& corfoMsg)
{
   int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
   ETG_TRACE_USR4(("CameraListHandler::Cabin Camera activeCamera=%d", activeCamera));
   if ((activeCamera >= 0) && (activeCamera <= MAX_CABIN_CAMERA))
   {
      _Src sourceId = SRC_INVALID;
      (activeCamera == 0) ? activeCamera = 0 : activeCamera = activeCamera - 1;
      m_poCameraDataBindingHandler -> updateCameraNameInTile(CameraListHandler::getInstance()->getActiveCameraNameCabin(activeCamera));
      ETG_TRACE_USR4(("CameraMain::onCourierMessage(const PreviousCameraActivate& corfoMsg) setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(activeCamera);
      HandleRequestCabin();
      sourceId = getSource();
      if (sourceId != SRC_INVALID)
      {
         CameraSourceChangeHandler::getInstance()->requestSourceActivation(sourceId, m_u8activeRegion);
      }
   }
   if (activeCamera == 0)
   {
      m_poCameraDataBindingHandler ->vShowHomeScreenNextPrevButtons(3);
   }
   else
   {
      m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(1);
   }
   return true;
}


/**
* Description     : Function to handle Activation request of camera in Cabin.
*/
void CameraMain::vStartCameraStream()
{
   int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
   (activeCamera == -1) ? activeCamera = 0 : activeCamera = activeCamera;

   ETG_TRACE_USR4(("CameraMain::vStartCameraStream"));
   if ((activeCamera >= 0))
   {
      _Src sourceId = SRC_INVALID;
      enCamera selCamera = INVALID;
      ETG_TRACE_USR4(("CameraMain::vStartCameraStream setSelectedCameraIndex is SETTING UP"));
      CameraListHandler::getInstance()->setSelectedCameraIndex(activeCamera);
      m_poCameraDataBindingHandler -> updateCameraNameInTile(CameraListHandler::getInstance()->getActiveCameraNameCabin(activeCamera));
      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex=%d, sourceID=%d", selCamera, sourceId));
      ETG_TRACE_USR4(("CameraListHandler::Cabin Camera"));

      int value = 0;
      bool streamStatusCabin = false;
      std::string strCamSrc = "";
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(activeCamera, value, strCamSrc, streamStatusCabin);
      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         //set the split cam bits and splitcam contril id.
         CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, SPLTSCRNCAMS, value, strCamSrc);
         //depending on the position, change the cockpitSplitview1 to CMP1_SPLITSCREEN ( and so on )
         if (value == 1)
         {
            strCamSrc = "CMP1_SPLITSCREEN";
         }
         if (value == 2)
         {
            strCamSrc = "CMP2_SPLITSCREEN";
         }
         if (value == 3)
         {
            strCamSrc = "CMP3_SPLITSCREEN";
         }
      }

      sourceId = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
      if (sourceId != SRC_INVALID)
      {
         CameraSourceChangeHandler::getInstance()->requestSourceActivation(sourceId, m_u8activeRegion);
      }
   }
}


/**
* Description     : Function to send context request to master.
*/
bool CameraMain::onCourierMessage(const ContextToMaster& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::ContextToMaster -> m_activityID==> current region==>  m_ToggleVisible %d %d %d ", m_activityID, sm_gCurrentRegion, m_ToggleVisible));
   const ContextToMaster* copoContextToMaster = Courier::message_cast<const ContextToMaster*>(&corfoMsg);
   if ((NULL != copoContextToMaster))
   {
      switch (copoContextToMaster->GetContextScene())
      {
         case 250:
            m_activityID = enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION;
            break;
         case 251:
         case 255:

            if (m_ToggleVisible == true) //enActivityIDs__eActivityID_CAMERA_STREAM
            {
               m_activityID = enActivityIDs__eActivityID_CAMERA_STREAM;
            }
            else
            {
               m_activityID = enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW;
            }

            //Used to Display the Camera Streaming Text in enActivityIDs__eActivityID_CAMERA_STREAM(251)
            SelectedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
            ETG_TRACE_USR3(("CameraMain::ContextToMaster ->SelectedIndex=%d", SelectedIndex));
            m_poCameraDataBindingHandler -> updateCameraNameInStreamScreen(CameraListHandler::getInstance()->getActiveCameraName(SelectedIndex));
            break;
         case 252:
            m_activityID = enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT;
            break;
         case 256:
            m_activityID = enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT;
            break;
         case 257:
            m_activityID = enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT_FULLSCREEN;
            break;
         case 258:
            m_activityID = enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM;
            break;
         case 259:
            m_activityID = enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM_SPLIT;
            break;
         default:
            break;
      }
      ETG_TRACE_USR4(("CameraMain::ContextToMaster==%d", m_activityID));
      _applicationSwitchClientComponent->requestSendContext(m_u8activeRegion, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)m_activityID);
   }
   return true;
}


/**
* Description     : Function to return the value of toggle in stream screen.
*/
bool CameraMain::GetToggleValue()
{
   return m_ToggleVisible;
}


void CameraMain::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
                                       const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("CameraMain::onActiveRegionIdError"));
}


void CameraMain::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
                                        const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   m_u8activeRegion = update->getActiveRegionId();
   ETG_TRACE_USR4(("onActiveRegionIdUpdate::m_u8activeRegion: %d", m_u8activeRegion));
}


/**
* Description     : Function to return the value of region.
*/
uint8 CameraMain::getRegionValue()
{
   return m_u8activeRegion;
}


/**
* Description     : Virtual function implemented to get update of onSig_ActivateContextError Error
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : error : The error message of 'onSig_ActivateContextError'
* @return         : void
*/

void CameraMain::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextError"));
}


/**
* Description     : Virtual function implemented to get update of onSig_ActivateContextSignal
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : response : The response message of 'onSig_ActivateContextSignal'
* @return         : void
*/


void CameraMain::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   uint16 activityID = signal->getTargetactivityId();

   ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal  %d ", signal->getTargetactivityId()));
   switch (activityID)
   {
      case enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION:
      {
         POST_MSG((COURIER_MESSAGE_NEW(CameraSourceSelectionMsg)()));
         sm_gCurrentRegion = REGION_C;
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal Call Stop stream silently"));
         //CRQ 1366236 Enable Camera Power Management at EvoBus
         int value = 0;
         bool streamStatus = false;
         std::string strCamSrc = "";
         ETG_TRACE_USR3(("CameraMain::onSig_ActivateContextSignal"));
         int sourceid = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
         //CRQ 1366236 Enable Camera Power Management at EvoBus
         CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(sourceid, value, strCamSrc, streamStatus);
         ETG_TRACE_USR3(("CameraMain::onSig_ActivateContextSignal COCKPIT sourceid, value, streamStatus  strCamSrc =%d %d %d %s", sourceid, value,  streamStatus, strCamSrc.c_str()));
         VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, DEACTIVATE);
         CameraListHandler::getInstance()->setCameraStreamStatusCockpit(sourceid, false);
         vRequestStopCameraStream();
         VehicleClientHandler::getInstance()->vsendDispCamCls_Rq(1);
         break;
      }
      case enActivityIDs__eActivityID_CAMERA_STREAM: //251
      {
         //TO DO
         uint8 m_isBlackScreenEnabledValue = KDSHandler::poGetInstance()->u8GetCameraBlackScreenFeatureValue();
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));

         POST_MSG((COURIER_MESSAGE_NEW(ActivateCameraStreamMsg)()));

         sm_gCurrentRegion = REGION_C;
         break;
      }
      case enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT:
      {
         //TO DO
         ETG_TRACE_USR4(("CameraMain::enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT "));
         POST_MSG((COURIER_MESSAGE_NEW(CameraSourceSelectionCabinMsg)()));
         sm_gCurrentRegion = REGION_A;
         break;
      }
      case enActivityIDs__eActivityID_CAMERA_CABIN_VIEW:
      case enActivityIDs__eActivityID_CABINA_HOME:
      case enActivityIDs__eActivityID_CABINB_HOME:
      {
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__eActivityID_CABINA_HOME "));
         if (m_streamactive == true)
         {
            vRequestStopCameraStream();
            //vRequestStopCabinCameraStream();
         }
         sm_gCurrentRegion = REGION_A;
         m_activityID = enActivityIDs__eActivityID_NONE;  // added here as a reset variable.
         break;
      }
      case enActivityIDs__eActivityID_CAMERA_REAR_VIEW:
      {
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__eActivityID_CAMERA_REAR_VIEW "));
         break;
      }

      case enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM://258
      {
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM "));
         POST_MSG((COURIER_MESSAGE_NEW(CameraViewBlackScreenTileTransistionMsg)()));
         break;
      }

      case enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM_SPLIT://259
      {
         POST_MSG((COURIER_MESSAGE_NEW(ActivateBlackScreenSplitScreenMsg)()));
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__eActivityID_CAMERA_BLACKSCREEN_STREAM_SPLIT "));
         break;
      }
      // case enActivityIDs__eActivityID_HOME:
      // {
      // break;
      // }
      case enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT:
      {
         //TO DO
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__eActivityID_CAMERA_STREAM_SPLIT "));
         POST_MSG((COURIER_MESSAGE_NEW(ActivateSplitScreenMsg)()));
         sm_gCurrentRegion = REGION_C;
         break;
      }
      case enActivityIDs__eActivityID_RVC:
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal enActivityIDs__enActivityIDs__eActivityID_RVC "));
         //Do nothing, When RVC is active, the context local variable shall not be updated.
         break;

      default:
         ETG_TRACE_USR4(("CameraMain::onSig_ActivateContextSignal default "));
         m_activityID = enActivityIDs__eActivityID_NONE;  // added here as a reset variable.
         break;
   }
}


void CameraMain::onSetSystemStatusResult(const ::boost::shared_ptr< Rvc_cca_fiProxy >& proxy, const ::boost::shared_ptr< SetSystemStatusResult >& result)
{
}


void CameraMain::onSetSystemStatusError(const ::boost::shared_ptr< Rvc_cca_fiProxy >& proxy, const ::boost::shared_ptr< SetSystemStatusError >& error)
{
}


/**
* Description     : Function to request RVC to start streaming.
*/
void CameraMain::vRequestStartCameraStream()
{
   ETG_TRACE_USR4(("Requesting Camera Start Stream "));
   m_poRvcFiProxy->sendSetSystemStatusStart(*this, T_e8_RVC_SystemStatus__RVC_VISUALISATION_ON,
         static_cast<T_e8_RVC_StatusGuideLine>(0x02),
         T_e8_RVC_StatusWarningText__WarningTextOff);
}


/**
* Description     : Function to request RVC to stop streaming.
*/
void CameraMain::vRequestStopCameraStream()
{
   ETG_TRACE_USR4(("Requesting Camera Stop Stream"));
   m_poRvcFiProxy->sendSetSystemStatusStart(*this, T_e8_RVC_SystemStatus__VISUALISATION_OFF,
         static_cast<T_e8_RVC_StatusGuideLine>(0x02),
         T_e8_RVC_StatusWarningText__WarningTextOff);
}


/**
* Description     : Function to handle the request of stop camera stream in cabin.
*/
bool CameraMain::vRequestStopCabinCameraStream()
{
   ETG_TRACE_USR4(("Requesting Camera Stop Stream"));
   int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
   ETG_TRACE_USR4(("CameraListHandler::Cabin Camera activeCamera=%d", activeCamera));
   if ((activeCamera >= 0)) // && (activeCamera <= MAX_CABIN_CAMERA))   //Range check of available cameras
   {
      m_poCameraDataBindingHandler -> vShowHomeScreenNextPrevButtons(0);
      _Src sourceId = SRC_INVALID;
      enCamera selCamera = INVALID;
      int region = 0;

      CameraListHandler::getInstance()->setSelectedCameraIndex(activeCamera);
      CameraListHandler::getInstance()->setNoHighlightInList(true);  // cabin list highlight  remove
      ETG_TRACE_USR4(("CameraMain::vRequestStopCabinCameraStream() setSelectedCameraIndex is SETTING UP camListSelectedIndex=%d, sourceID=%d", selCamera, sourceId));

      int value = 0;
      bool streamStatusCabin = false;
      std::string strCamSrc = "";
      CameraListHandler::getInstance()->getCameraPositionAndSourceNameValuesCabin(activeCamera, value, strCamSrc, streamStatusCabin);
      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         //set the split cam bits and splitcam contril id.
         CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, SPLTSCRNCAMS, value, strCamSrc);
         //depending on the position, change the cockpitSplitview1 to CMP1_SPLITSCREEN ( and so on )
         if (value == 1)
         {
            strCamSrc = "CMP1_SPLITSCREEN";
         }
         if (value == 2)
         {
            strCamSrc = "CMP2_SPLITSCREEN";
         }
         if (value == 3)
         {
            strCamSrc = "CMP3_SPLITSCREEN";
         }
      }

      sourceId = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);

      if (sourceId != SRC_INVALID)
      {
         CameraSourceChangeHandler::getInstance()->requestSourceDeActivation(sourceId, region);
      }
   }
   return true;
}


/**
* Description     : Requesting Split view for Cabin split view to handle request for 2:1 and 4:1.
*/
void CameraMain::RequestPluginForCabinSplitViewSel(enCamera camSrc, int splitType)
{
   ETG_TRACE_USR4(("CameraMain::RequestPluginForCabinSplitViewSel"));
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      std::vector<uint8> vsourceId;

      if (4 == splitType)
      {
         splitViewSrcIdCabinMap::const_iterator pos = m_splitViewCabinMap.find(static_cast<uint8>(camSrc));
         if (pos == m_splitViewCabinMap.end())
         {
            ETG_TRACE_USR4(("CameraMain::RequestPluginForCabinSplitViewSel, cabin split view selection, Invalid Source Id"));
            return;
         }
         vsourceId = pos->second;
      }
      else
      {
         splitViewCockpitMap::const_iterator pos = m_splitViewCockpitMap.find(static_cast<uint8>(camSrc));
         if (pos == m_splitViewCockpitMap.end())
         {
            ETG_TRACE_USR4(("CameraMain::RequestPluginForCabinSplitViewSel, cabin split view selection, Invalid Source Id"));
            return;
         }
         vsourceId = pos->second;
      }

      for (int index = 0; index < vsourceId.size(); ++index)
      {
         pluginData->addEventData(static_cast<uint16>(vsourceId.at(index)));
      }
      POST_MSG((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_CAMERA, CTRLBLOCK_NAME_CAMERA_SPLITVIEW_SEL, pluginData)));
   }
}


/**
* Description     : Function to handle request from different camportss for split view.
*/
_Src CameraMain::getCabinSplitSrc(uint8 cameraButton)
{
   ETG_TRACE_USR4(("getCabinSplitSrc(selCamera %d)", cameraButton));
   _Src splitSrc = SRC_INVALID;
   switch (cameraButton)
   {
      case CMP1_CAM1:
      {
         splitSrc = SRC_CAMERAPORT1_SPLITSCREEN;
      }
      break;
      case CMP1_CAM2:
      {
         splitSrc = SRC_CAMERAPORT2_SPLITSCREEN;
      }
      break;
      case CMP1_CAM3:
      {
         splitSrc = SRC_CAMERAPORT3_SPLITSCREEN;
      }
      break;
      case CMP1_CAM4:
      {
         splitSrc = SRC_CAMERAPORT4_SPLITSCREEN;
      }
      break;
      default:
      {
         ETG_TRACE_USR4(("Invalid cabin source"));
      }
      break;
   }
   return splitSrc;
}


/**
* Description     : Courier message for Functions to run on startup.
*/
bool CameraMain::onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg)
{
   VehicleClientHandler::getInstance()->vsendDispCamCls_Rq(1);
   bool bPrimaryCMPStatus = CameraSourceChangeHandler::getInstance()->getPrimaryCMPConnectedStatus();
   ETG_TRACE_USR4(("CameraMain::GuiStartupFinishedUpdMsg() bPrimaryCMPStatus ==>%d", bPrimaryCMPStatus));
   if (bPrimaryCMPStatus == true)
   {
      m_poCameraDataBindingHandler -> vCamAvailability(true);
   }
   return false;
}


/**
* Description     : Method to handle the request for cockpit camera button and tile.
*/
bool CameraMain::onCourierMessage(const ButtonMainSrcActivate& oMsg)
{
   ETG_TRACE_USR3(("CameraMain : ButtonMainSrcActivate "));
   int _speedStatus = 0;
   vector<int> camTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();
   uint8 activityID = 0;
   const ButtonMainSrcActivate* copoButtonMainSrcActivate = Courier::message_cast<const ButtonMainSrcActivate*>(&oMsg);
   if ((NULL != copoButtonMainSrcActivate))
   {
      uint8 u8srcmode = copoButtonMainSrcActivate->GetSrcmode();
      ETG_TRACE_USR3(("CameraMain : u8srcmode %d", u8srcmode));
      if (camTextId.size() > 0)
      {
         if (u8srcmode == CAMERA_SRC_SELECTION)  // SOURCE_SELECTION CAMERA
         {
            activityID = enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION;
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID/*enActivityIDs__eActivityID_TUNER_FM1*/);
         }

         else if (u8srcmode == CAMERA_SRC_TILE_SELECTION)
         {
            int selectedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
            (selectedIndex == -1) ? selectedIndex = 0 : selectedIndex = selectedIndex;
            CameraListHandler::getInstance()->camListSelectedIndex((uint32)selectedIndex, HOMESCREEN_TILE);
            VehicleClientHandler::getInstance()->vsendDispCamCls_Rq(0);
         }
      }
      else
      {
         POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupNotConfigMsg)()));
      }
   }
   return true;
}


/**
* Description     : Method to handle the request for Cabin camera button and tile.
*/
bool CameraMain::onCourierMessage(const ButtonMainSrcCabinActivate& oMsg)
{
   ETG_TRACE_USR3(("CameraMain : ButtonMainSrcCabinActivate "));
   uint8 activityID = 0;
   const ButtonMainSrcCabinActivate* copoButtonMainSrcActivate = Courier::message_cast<const ButtonMainSrcCabinActivate*>(&oMsg);
   if ((NULL != copoButtonMainSrcActivate))
   {
      uint8 u8srcmode = copoButtonMainSrcActivate->GetSrcmode();
      ETG_TRACE_USR3(("CameraMain : u8srcmode %d", u8srcmode));

      if (u8srcmode == CAMERA_SRC_CABIN_SELECTION)
      {
         ETG_TRACE_USR3(("CameraMain : sm_gCurrentRegion == cabin A "));
         activityID = enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT;
         ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_CABINA_HOME, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
      }
      else if (u8srcmode == CAMERA_SRC_CABIN_TILE_SELECTION)
      {
         ETG_TRACE_USR3(("CameraMain : sm_gCurrentRegion == cabin A "));
         activityID = enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT;
         ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_CABINA_HOME, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
      }
   }
   return true;
}


/**
* Description     : Method to Handle the request from SM to start streaming in cockpit.
*/
bool CameraMain::onCourierMessage(const CameraStreamStartMsg& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::CameraStreamStartMsg -> current region %d", sm_gCurrentRegion));
   const CameraStreamStartMsg* copoCameraStreamStartMsg = Courier::message_cast<const CameraStreamStartMsg*>(&corfoMsg);
   if ((NULL != copoCameraStreamStartMsg))
   {
      if (m_u8activeRegion == 2)
      {
         vRequestStartCameraStream();
         VehicleClientHandler::getInstance()->vsendDispCamCls_Rq(0);
      }
   }
   return true;
}


/**
* Description     : Method to handle the request from SM to stop streaming in cockpit.
*/
bool CameraMain::onCourierMessage(const CameraStreamStopMsg& corfoMsg)
{
   ETG_TRACE_USR4(("CameraMain::onCourierMessage CameraStreamStopMsg getCurrentMic1Status %d sm_gRVCStatus %d", CameraSourceChangeHandler::getInstance()->getCurrentMic1Status(), sm_gRVCStatus));
   //Bug 1474790 Bug 1398113 Bug 1518561 - Resolving the Context Priority issue Between CANcAMERA-MIC-RVC
   //Below Comment  -->to Resolve the priority order 1st RVC 2nd MIC 3rd Camera/CanCamera and Default Context switch is removed on Camera Streaming Exit-
   if ((false == CameraSourceChangeHandler::getInstance()->getCurrentMic1Status()) && (sm_gRVCStatus == false))
   {
      int value = 0;
      bool streamStatus = false;
      std::string strCamSrc = "";

      const CameraStreamStopMsg* copoCameraStreamStopMsg = Courier::message_cast<const CameraStreamStopMsg*>(&corfoMsg);
      if ((NULL != copoCameraStreamStopMsg))
      {
         if (m_u8activeRegion == 2)
         {
            vRequestStopCameraStream();
            int canindex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
            CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(canindex, value, strCamSrc, streamStatus);
            ETG_TRACE_USR3(("bool CameraMain::onCourierMessage CameraStreamStopMsg COCKPIT canindex+1 value,streamStatus strCamSrc=%d %d  %d %s", canindex, value, streamStatus, strCamSrc.c_str()));
            VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, DEACTIVATE);
            VehicleClientHandler::getInstance()->vsendDispCamCls_Rq(1);
         }
      }
   }
   return true;
}


void CameraMain::HandleRequestCabin()
{
   int activeCamera = CameraListHandler::getInstance()->getSelectedCameraIndex();
   _Src sourceId = SRC_INVALID; //CID_R1 3412368
   enCamera selCamera = INVALID;
   if (activeCamera == CAMERA_DUAL_SPLIT)
   {
      enCamera selCamera = static_cast<enCamera>((activeCamera + 2));
      ETG_TRACE_USR4(("CameraMain::HandleRequestCabin  =%d, sourceID=%d", (selCamera), sourceId));
   }
   else if (activeCamera == CAMERA_QUAD_SPLIT)
   {
      enCamera selCamera = static_cast<enCamera>((activeCamera + 1));
      ETG_TRACE_USR4(("CameraMain::HandleRequestCabin  =%d, sourceID=%d", (selCamera), sourceId));
   }
   else if (activeCamera == CAMERA_IP)
   {
      enCamera selCamera = static_cast<enCamera>((activeCamera - 1));
      ETG_TRACE_USR4(("CameraMain::HandleRequestCabin  =%d, sourceID=%d", selCamera, sourceId));
   }
   else
   {
      enCamera selCamera = static_cast<enCamera>((activeCamera + 1));
      ETG_TRACE_USR4(("CameraMain::HandleRequestCabin  =%d, sourceID=%d", selCamera, sourceId));
   }
   //setSource(sourceId);
}


bool CameraMain::onCourierMessage(const PluginUpdateRespMsg& oMsg)
{
   const Candera::String& pluginName = oMsg.GetPluginName();
   if (0 == strcmp(PLUGIN_NAME_CAMERA, pluginName.GetCString()))
   {
      {
         const boost::shared_ptr<EventDataUtility>& dataUtility = oMsg.GetPluginData();
         if (NULL != dataUtility.get())
         {
            const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
            ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginUpdateRespMsg dataItems %d", dataItems.size()));
            if (dataItems.size() > 0)
            {
               EventDataItem* eventDataItem = dataItems.at(0);
               if (NULL != eventDataItem)
               {
                  const EventDataItem::Data& eventData = eventDataItem->getData();
                  m_control_id = eventData._value._uint16Value;
                  ETG_TRACE_USR4(("CameraMain::onCourierMessage:PluginUpdateRespMsg control block id : %d", m_control_id));
                  switch (m_control_id)
                  {
                     case CVBSSOURCE :
                     case SPLTSCRNCAMS :
                     {
                        //POST_MSG((COURIER_MESSAGE_NEW(CameraViewMsg)()));
                     }
                     break;
                     default:
                        ETG_TRACE_USR1(("CameraMain::onCourierMessage:PluginUpdateRespMsg controlindex default"));
                        break;
                  }
               }
            }
         }
      }
   }
   return true;
}


uint16 CameraMain::getCameraActivityID(std::string viewid)
{
   uint16 activityID = 0;
   if (0 == viewid.compare("AppHmi_CameraApp#Scenes#C_CameraStream"))
   {
      if (m_ToggleVisible == true)
      {
         activityID = enActivityIDs__eActivityID_CAMERA_STREAM;
      }
      else
      {
         activityID = enActivityIDs__eActivityID_CAMERA_STREAM_FULLSCREENVIEW;
      }
   }
   return activityID;
}


bool CameraMain::onCourierMessage(const ActiveRenderedView& msg)
{
   std::string sViewName = msg.GetViewName().GetCString();
   uint16 activityID = 0;
   activityID = getCameraActivityID(sViewName);
   sendStoreContextRequest(activityID);
   if ((0 == sViewName.compare("AppHmi_CameraApp#Scenes#C_CameraStream")) ||
         (0 == sViewName.compare("AppHmi_CameraApp#Scenes#C_CameraSplitScene")) ||
         (0 == sViewName.compare("AppHmi_CameraApp#Scenes#C_CameraFullSplit")))
   {
      m_streamactive = true;
   }
   else
   {
      m_streamactive = false;
   }
   ETG_TRACE_USR1(("CameraMain::onCourierMessage ActiveRenderedView ====>activityID =%d m_streamactive =%d surface id %d ,view name %s", activityID, m_streamactive, msg.GetSurfaceId(), sViewName.c_str()));
   return false;
}


void CameraMain::sendStoreContextRequest(uint16 activityID)
{
   ETG_TRACE_USR4(("CameraMain::sendStoreContextRequest: %d", activityID));
   if (activityID != 0)
   {
      _applicationSwitchClientComponent->requestSendContext(m_u8activeRegion, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
   }
}


/**
* Description     : Function to update the toggle value of Icon in camera cabin tile.
*/
void CameraMain::setToggleValue(bool toggle)
{
   m_toggle = toggle;
}


void CameraMain::setSource(_Src source)
{
   m_sourceID = source; //CID_R1 2795222
}


_Src CameraMain::getSource()
{
   return m_sourceID;
}


void CameraMain::sendPluginRequestToAVDECC(const char* pluginName, uint16 ControlId, uint16 position, std::string cameraSourceName)
{
   ETG_TRACE_USR4(("sendPluginRequestToAVDECC Plugin name : %s", pluginName));
   ETG_TRACE_USR4(("sendPluginRequestToAVDECC Control block name : %d", ControlId));
   ETG_TRACE_USR4(("sendPluginRequestToAVDECC CMP position : %d", position));
   ETG_TRACE_USR4(("sendPluginRequestToAVDECC Camera Source Name : %s", cameraSourceName.c_str()));
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(ControlId);
      pluginData->addEventData(position);
      pluginData->addEventData(cameraSourceName);
      POST_MSG((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(pluginName, "", pluginData)));
   }
   else
   {
      ETG_TRACE_USR4(("sendPluginRequestToAVDECC cascading of camport is not handled"));
   }
}


bool CameraMain::onCourierMessage(const PluginEndStationListUpdateMsg& oMsg)
{
   ETG_TRACE_USR1(("CameraMain::onCourierMessage:PluginEndStationListUpdateMsg Entered"));
   int index = 0;
   m_subSystemAvailability.clear();

   const boost::shared_ptr<EventDataUtility>& dataUtility = oMsg.GetEndStationListData();
   if (NULL != dataUtility.get())
   {
      const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
      ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginEndStationListUpdateMsg dataItems %d", dataItems.size()));
      if (dataItems.size() > 0)
      {
         for (index = 0; index < dataItems.size(); index++)
         {
            EventDataItem* eventDataItem = dataItems.at(index);
            if (NULL != eventDataItem)
            {
               EventDataItem::Data eventDataVal = eventDataItem->getData();
               std::string strVal = eventDataVal._value._stringValue->c_str();
               ETG_TRACE_USR4(("CameraMain::onCourierMessage:PluginEndStationListUpdateMsg strVal:  %s", strVal.c_str()));
               if (strVal == CMG || strVal == CMP || strVal == CMA)
               {
                  m_subSystemAvailability.push_back(strVal);
               }
            }
         }
      }
   }
#if 0
   bool checkCMPAvailabilty = getCurrentSubSystemAvailability(CMP);
   m_poCameraDataBindingHandler -> vCamAvailability(checkCMPAvailabilty);
   bool checkCMGAvailabilty = getCurrentSubSystemAvailability(CMG);
   m_poCameraDataBindingHandler -> vCamAvailabilityCabin(checkCMGAvailabilty, checkCMPAvailabilty);
   if ((checkCMPAvailabilty == false) && ((m_activityID == 250) || (m_activityID == 251) || (m_activityID == 255)))
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)m_activityID, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_HOME);
   }
   if (((checkCMPAvailabilty == false) || (checkCMGAvailabilty == false)) && (m_activityID == 252))
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)m_activityID, APPID_APPHMI_MASTER, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_CABINA_HOME);
   }
   //added as a extra workaround.

   if ((checkCMPAvailabilty == true))
   {
      if (m_subSystemAvailability.size() != 0)
      {
         m_CMPCount = std::count(m_subSystemAvailability.begin(), m_subSystemAvailability.end(), "CMP");
         ETG_TRACE_USR4(("CameraMain::onCourierMessage:PluginEndStationListUpdateMsg CMP count Curr %d Prev %d", m_CMPCount, m_CMPPrevCount));
         if (m_CMPPrevCount != m_CMPCount)
         {
            ETG_TRACE_USR4(("CameraMain::onCourierMessage:PluginEndStationListUpdateMsg Request Camera List"));
            sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CAMLIST_STARTUP, 0, "CAMERA");
            m_CMPPrevCount = m_CMPCount;
         }
      }
   }
#endif
   return true;
}


bool CameraMain::onCourierMessage(const PluginListUpdateRespMsg& oMsg)
{
   ETG_TRACE_USR4(("CameraMain::PluginListUpdateRespMsg "));
   int index = 0;
   cameraTextIdFromXML.clear();

   const Candera::String& pluginName = oMsg.GetPluginName();
   if (0 == strcmp(PLUGIN_NAME_CAMERA, pluginName.GetCString()))
   {
      const boost::shared_ptr<EventDataUtility>& dataUtility = oMsg.GetPluginData();
      if (NULL != dataUtility.get())
      {
         const std::vector<EventDataItem*>& dataItems = dataUtility->getData();
         ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg dataItems %d", dataItems.size()));
         if (dataItems.size() > 0)
         {
            EventDataItem* eventDataItem = dataItems.at(0);
            if (NULL != eventDataItem)
            {
               const EventDataItem::Data& eventData = eventDataItem->getData();
               m_control_id = eventData._value._uint16Value;
            }
         }
      }

      const boost::shared_ptr<EventListDataUtility>& dataListUtility = oMsg.GetPluginListData();
      if (NULL != dataListUtility.get())
      {
         const std::vector<const EventDataUtility*>& listItems = dataListUtility->getData();
         ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg dataListItems %d", listItems.size()));
         if (listItems.size() > 0)
         {
            stCameraData scd = {0, 0, "", false, false};//CID_R1 3412367
            vecCamData.clear();
            for (index = 0; index < listItems.size(); index++)
            {
               const EventDataUtility* listItem = listItems.at(index);
               if (NULL != listItem)
               {
                  const std::vector<EventDataItem*>& eventDataItems = listItem->getData();
                  EventDataItem* eventDataItem =  eventDataItems.at(0);
                  if (NULL != eventDataItem)
                  {
                     const EventDataItem::Data& eventData = eventDataItem->getData();
                     if (EventDataItem::UINT16 == eventData._type)
                     {
                        ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , position value is : %d", eventData._value._uint16Value));
                        scd.position = eventData._value._uint16Value;
                     }
                  }
                  eventDataItem =  eventDataItems.at(1);
                  if (NULL != eventDataItem)
                  {
                     const EventDataItem::Data& eventData = eventDataItem->getData();
                     if (EventDataItem::UINT16 == eventData._type)
                     {
                        ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , camTxt value is : %d", eventData._value._uint16Value));
                        scd.camTextID = eventData._value._uint16Value;
                        cameraTextIdFromXML.push_back(eventData._value._uint16Value);
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("CameraMain:onCourierMessage:data item 1 null"))
                  }
                  eventDataItem =  eventDataItems.at(2);
                  if (NULL != eventDataItem)
                  {
                     const EventDataItem::Data& eventData = eventDataItem->getData();
                     if (EventDataItem::STRING == eventData._type)
                     {
                        ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , String value is : %s", eventData._value._stringValue->c_str()));
                        scd.camerasource = eventData._value._stringValue->c_str();
                     }
                  }
                  else
                  {
                     ETG_TRACE_USR4(("CameraMain:onCourierMessage:data item 2 null"))
                  }
                  scd.streamStatus = false;
                  scd.streamStatusCabin = false;
                  vecCamData.push_back(scd);
               }
               else
               {
                  ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , listItem is null"));
               }
            }
         }
         else
         {
            ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , dataItems is zero"));
         }
      }
   }
   // printStructure();
//  CameraListHandler::getInstance()->vUpdateCameraListData();

   bool bPrimaryCMPStatus = CameraSourceChangeHandler::getInstance()->getPrimaryCMPConnectedStatus();

   ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg , vecCamData.size(%d), bPrimaryCMPStatus %d", vecCamData.size(), bPrimaryCMPStatus));

   if ((vecCamData.size() > 0)  && (bPrimaryCMPStatus == true))// check the list size and enable the tile
      //update Tile information as well.
   {
      CameraSourceChangeHandler::getInstance()->vCheckIfSourceToBeRestored();
      ETG_TRACE_USR4(("CameraMain:onCourierMessage:PluginListUpdateRespMsg enable the cockpit tile"));
      m_poCameraDataBindingHandler -> vCamAvailability(true);
      //m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
   }
   return true;
}


void CameraMain::vOnReceiveCameraList(tVecCamData vecCamDataToHmi)
{
   ETG_TRACE_USR4(("CameraMain::vOnReceiveCameraList "));
   int index = 0;
   cameraTextIdFromXML.clear();

   stCameraData scd;
   vecCamData.clear();
   for (index = 0; index < vecCamDataToHmi.size(); index++)
   {
      scd.position = vecCamDataToHmi[index].pos;
      scd.camTextID = vecCamDataToHmi[index].camtextid;;
      cameraTextIdFromXML.push_back(vecCamDataToHmi[index].camtextid);

      scd.camerasource = vecCamDataToHmi[index].srcNam.c_str();

      scd.streamStatus = false;
      scd.streamStatusCabin = false;
      vecCamData.push_back(scd);
   }
   ETG_TRACE_USR4(("CameraMain::vOnReceiveCameraList call print trace from here"));
   printStructure();
   CameraListHandler::getInstance()->vUpdateCameraListData();
   bool bPrimaryCMPStatus = CameraSourceChangeHandler::getInstance()->getPrimaryCMPConnectedStatus();

   ETG_TRACE_USR4(("CameraMain:onCourierMessage:vOnReceiveCameraList , vecCamData.size(%d), bPrimaryCMPStatus %d", vecCamData.size(), bPrimaryCMPStatus));

   if ((vecCamData.size() > 0)  && (bPrimaryCMPStatus == true))// check the list size and enable the tile
      //update Tile information as well.
   {
      CameraSourceChangeHandler::getInstance()->vCheckIfSourceToBeRestored();
      ETG_TRACE_USR4(("CameraMain:onCourierMessage:vOnReceiveCameraList enable the cockpit tile"));
      m_poCameraDataBindingHandler -> vCamAvailability(true);
      //m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
   }
}


void CameraMain::printStructure()
{
   ETG_TRACE_USR4(("CameraMain:printStructure"));
   for (int i = 0; i < vecCamData.size(); i++)
   {
      ETG_TRACE_USR4(("CameraMain:printStructure: position[%d] = %d", i, vecCamData.at(i).position));
      ETG_TRACE_USR4(("CameraMain:printStructure:position[%d] = %d", i, vecCamData.at(i).camTextID));
      ETG_TRACE_USR4(("CameraMain:printStructure:sourcename [%d] = %s", i, vecCamData.at(i).camerasource.c_str()));
   }

   for (int i = 0; i < cameraTextIdFromXML.size(); i++)
   {
      ETG_TRACE_USR4(("CameraMain:printStructure: cameraTextIdFromXML[%d]", cameraTextIdFromXML.at(i)));
   }
}


bool CameraMain::vAppflyinStatusInfo(bool FlyInStatus)
{
   ETG_TRACE_USR4(("CameraMain::vAppflyinStatusInfo %d", FlyInStatus));
   std::vector<AppFlyInStateInfo> temp_AppFlyInStateInfo;
   AppFlyInStateInfo temp_obj;

   temp_obj.setRegionId((uint8)2);
   temp_obj.setAppFlyInIndex((uint8)ITEM_CAMERA);

   /* When BT is enabled, enable Phone AppFlyIn else disable the same */
   if (FlyInStatus == true)
   {
      temp_obj.setStatus((uint8)BUTTON_STATE_ENABLED);
   }
   else
   {
      temp_obj.setStatus((uint8)BUTTON_STATE_DISABLED);
   }

   temp_AppFlyInStateInfo.clear();
   temp_AppFlyInStateInfo.push_back(temp_obj) ;

   if (m_poApplicationSwitchProxy)
   {
      ETG_TRACE_USR4(("CameraMain::vAppflyinStatusInfo %d %d %d", temp_obj.getRegionId(), temp_obj.getAppFlyInIndex(), temp_obj.getStatus()));
      m_poApplicationSwitchProxy->sendConfigureAppFlyInStatusRequest(temp_AppFlyInStateInfo);
   }

   return true;
}


bool CameraMain::getCurrentSubSystemAvailability(std::string strCurrentVal)
{
   ETG_TRACE_USR4(("CameraMain::getCurrentSubSystemAvailability strCurrentVal %s", strCurrentVal.c_str()));
   bool bSubSystemAvail = false;
   if (m_subSystemAvailability.size() != 0)
   {
      std::vector <std::string>::iterator itr = std::find(m_subSystemAvailability.begin(), m_subSystemAvailability.end(), strCurrentVal);
      if (itr != m_subSystemAvailability.end())
      {
         bSubSystemAvail = true;
      }
   }
   ETG_TRACE_USR4(("CameraMain::getCurrentSubSystemAvailability bSubSystemAvail %d", bSubSystemAvail));
   return bSubSystemAvail;
}


int CameraMain::vGetActivityID()
{
   ETG_TRACE_USR4(("CameraMain::vGetActivityID() m_activityID==>%d", m_activityID));
   return m_activityID;
}


bool CameraMain::onCourierMessage(const SplitViewLeftCamMsg& msg)       //To Implement
{
   ETG_TRACE_USR1(("CameraMain::onCourierMessageSplitViewLeftCam"));
   std::string camLeftsrc = getSplitCamName(m_selectedSplitSource, m_CMPPosition, 0);
   sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, m_CMPPosition, camLeftsrc);
   POST_MSG((COURIER_MESSAGE_NEW(ActivateSplitFullScreenMsg)()));
   return true;
}


bool CameraMain::onCourierMessage(const SplitViewRightCamMsg& msg)     //To Implement
{
   ETG_TRACE_USR1(("CameraMain::onCourierMessageSplitViewRightCam"));
   std::string camRightsrc = getSplitCamName(m_selectedSplitSource, m_CMPPosition, 1);
   sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, m_CMPPosition, camRightsrc);
   POST_MSG((COURIER_MESSAGE_NEW(ActivateSplitFullScreenMsg)()));
   return true;
}


bool CameraMain::onCourierMessage(const ButtonMixedSourceMsg& msg)     //To Implement
{
   int ipCamCount = CameraSourceChangeHandler::getInstance()-> getIPCamCount();//get method for number of IP cams available
   ETG_TRACE_USR4(("CameraMain::onCourierMessage ButtonMixedSourceMsg =%d ", ipCamCount));
   if (ipCamCount == 0)
   {
      //databinding to grey out the buttons or hide button
   }
   else if (ipCamCount == 1)
   {
      std::string strCamSrc = "CMP1_IPCAM";
      _Src sourceId = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
      if (m_MixSrctoggle == true)
      {
         CameraSourceChangeHandler::getInstance()->sendCabinVideoMixRequest(REG_B, CABIN_VIDEO_MIX_CHANGE_REASON_USER, CABIN_VIDEO_MIX_INDEX_NONE);
         m_MixSrctoggle = false;
         ETG_TRACE_USR4(("CameraMain::onCourierMessage ButtonMixedSourceMsg NONE = %d", m_MixSrctoggle));
      }
      else if (m_MixSrctoggle == false)
      {
         CameraSourceChangeHandler::getInstance()->sendCabinVideoMixRequest(REG_B, CABIN_VIDEO_MIX_CHANGE_REASON_USER, CABIN_VIDEO_MIX_INDEX_IP_CAMERA_1);
         m_MixSrctoggle = true;
         ETG_TRACE_USR4(("CameraMain::onCourierMessage ButtonMixedSourceMsg Panoramic = %d", m_MixSrctoggle));
      }
   }
   else
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(m_u8activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_SYSTEM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)193);
   }
   return true;
}


std::string CameraMain::getSplitCamName(std::string strSplitCamSrcName, int pos, int selection)
{
   std::string camSrcName = "";
   std::vector<std::string> vecSrNamesSplit =  _avRoutingParser->getSourceNamesForSplitView(strSplitCamSrcName, pos);
   if (vecSrNamesSplit.size() > 0)
   {
      if (selection == 0)
      {
         camSrcName = vecSrNamesSplit.at(0); 	// Left selection
      }
      else
      {
         camSrcName = vecSrNamesSplit.at(1); 	// Right selection
      }
   }
   ETG_TRACE_USR4(("CameraMain:getSplitCamName %d %d -> %s", pos, selection, camSrcName.c_str()));

   return camSrcName;
}


void CameraMain::setSplitCamSourceAndPosition(std::string camSrcName, int pos)
{
   m_selectedSplitSource = camSrcName;
   m_CMPPosition = pos;
}


bool CameraMain::onCourierMessage(const CameraSplitStreamStartMsg& corfoMsg)
{
   ETG_TRACE_USR3(("CameraMain::CameraSplitStreamStartMsg -> current region %d", sm_gCurrentRegion));
   const CameraSplitStreamStartMsg* copoCameraStreamStartMsg = Courier::message_cast<const CameraSplitStreamStartMsg*>(&corfoMsg);
   int value = 0;
   bool streamStatus = false;
   std::string strCamSrc = "";
   int listSeletedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
   CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(listSeletedIndex, value, strCamSrc, streamStatus);
   if ((NULL != copoCameraStreamStartMsg))
   {
      if (m_u8activeRegion == 2)
      {
         if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
         {
            sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, value, strCamSrc);
            setSplitCamSourceAndPosition(strCamSrc, value); //required for split view to set the source name.
            vRequestStartCameraStream();
            //POST_MSG((COURIER_MESSAGE_NEW(ActivateCameraSplitStreamMsg)()));
         }
      }
   }
   return true;
}


vector<int> CameraMain::getCameraTextIds()
{
   return cameraTextIdFromXML;
}


vector<stCameraData> CameraMain::getCamDataStructure()
{
   return vecCamData;
}


/*Function to set-reset the "m_MixSrctoggle" toggle flag*/
void CameraMain::setMixToggleFlag(int toggle)
{
   m_MixSrctoggle = toggle;
}


//this method is added to get the camera source button toggle status in cabin region.
//This method is required, and used as a check before sending camera ManAct signal.
bool CameraMain::getCabinTileCameraToggleStatus()
{
   return m_toggle;
}


void CameraMain::TestDataToHMI_1CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t ipcam)
{
   tU16 position[] = {1, 1, 1, 1, 1, 1, 1, 1};
   std::string srcName[] = {"CMP1_CAM1", "CMP1_CAM2", "CMP1_CAM3", "CMP1_CAM4", "CMP1_IPCAM", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3"};
   tU16 camTextID[8];
   camTextID[0] = cam1;
   camTextID[1] = cam2;
   camTextID[2] = cam3;
   camTextID[3] = cam4;
   camTextID[4] = ipcam;
   camTextID[5] = -1;
   camTextID[6] = -1;
   camTextID[7] = -1;
   vecCamData.clear();
   cameraTextIdFromXML.clear();
   stCameraData t = {0, 0, "", false, false}; //CID_R1 11378982
   for (int i = 0; i < 8; i++)
   {
      t.position = position[i];
      t.camTextID = camTextID[i];
      t.camerasource = srcName[i];
      vecCamData.push_back(t);
      cameraTextIdFromXML.push_back(camTextID[i]);
   }
   CameraListHandler::getInstance()->vUpdateCameraListData();
}


void CameraMain::TestDataToHMI_2CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t cam5, uint16_t cam6, uint16_t cam7, uint16_t ipcam1, uint16_t ipcam2)
{
   tU16 position[] = {1, 1, 1, 2, 2, 2, 2, 1, 2, 1, 1, 1, 2, 2, 2};
   std::string srcName[] = {"CMP1_CAM1", "CMP1_CAM2", "CMP1_CAM3", "CMP2_CAM1", "CMP2_CAM2", "CMP2_CAM3", "CMP2_CAM4", "CMP1_IPCAM", "CMP2_IPCAM", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3"};
   tU16 camTextID[15];
   camTextID[0] = cam1;
   camTextID[1] = cam2;
   camTextID[2] = cam3;
   camTextID[3] = cam4;
   camTextID[4] = cam5;
   camTextID[5] = cam6;
   camTextID[6] = cam7;
   camTextID[7] = ipcam1;
   camTextID[8] = ipcam2;
   camTextID[9] = -1;
   camTextID[10] = -1;
   camTextID[11] = -1;
   camTextID[12] = -1;
   camTextID[13] = -1;
   camTextID[14] = -1;

   vecCamData.clear();
   cameraTextIdFromXML.clear();
   stCameraData t = {0, 0, "", false, false}; //CID 11378978

   for (int i = 0; i < 15; i++)
   {
      t.position = position[i];
      t.camTextID = camTextID[i];
      t.camerasource = srcName[i];
      vecCamData.push_back(t);
      cameraTextIdFromXML.push_back(camTextID[i]);
   }
   CameraListHandler::getInstance()->vUpdateCameraListData();
}


void CameraMain::TestDataToHMI_3CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t cam5, uint16_t cam6, uint16_t cam7, uint16_t cam8, uint16_t cam9, uint16_t cam10, uint16_t ipcam1, uint16_t ipcam2, uint16_t ipcam3)
{
   tU16 position[] = {1, 1, 1, 2, 2, 2, 3, 3, 3, 3, 1, 2, 3, 1, 1, 1, 2, 2, 2, 3, 3, 3};
   std::string srcName[] = {"CMP1_CAM1", "CMP1_CAM2", "CMP1_CAM3", "CMP2_CAM1", "CMP2_CAM2", "CMP2_CAM3", "CMP3_CAM1", "CMP3_CAM2", "CMP3_CAM3", "CMP3_CAM4", "CMP1_IPCAM", "CMP2_IPCAM", "CMP3_IPCAM", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3", "cockpitSplitView1", "cockpitSplitView2", "cockpitSplitView3"};
   tU16 camTextID[22];
   camTextID[0] = cam1;
   camTextID[1] = cam2;
   camTextID[2] = cam3;
   camTextID[3] = cam4;
   camTextID[4] = cam5;
   camTextID[5] = cam6;
   camTextID[6] = cam7;
   camTextID[7] = cam8;
   camTextID[8] = cam9;
   camTextID[9] = cam10;
   camTextID[10] = ipcam1;
   camTextID[11] = ipcam2;
   camTextID[12] = ipcam3;
   camTextID[13] = -1;
   camTextID[14] = -1;
   camTextID[15] = -1;
   camTextID[16] = -1;
   camTextID[17] = -1;
   camTextID[18] = -1;
   camTextID[19] = -1;
   camTextID[20] = -1;
   camTextID[21] = -1;

   vecCamData.clear();
   cameraTextIdFromXML.clear();
   stCameraData t = {0, 0, "", false, false}; //CID_R1 11378983
   for (int i = 0; i < 22; i++)
   {
      t.position = position[i];
      t.camTextID = camTextID[i];
      t.camerasource = srcName[i];
      vecCamData.push_back(t);
      cameraTextIdFromXML.push_back(camTextID[i]);
   }
   CameraListHandler::getInstance()->vUpdateCameraListData();
}


ETG_I_CMD_DEFINE((TraceCmd_setCamTextFor1CMP, "setCamTextFor1CMP %d %d %d %d %d", int, int, int, int, int))
void CameraMain::TraceCmd_setCamTextFor1CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t ipcam)
{
   ETG_TRACE_USR4(("CameraMain::TestDataToHMI_1CMP"));
   m_poCameraMain->TestDataToHMI_1CMP(cam1, cam2, cam3, cam4, ipcam);
}


ETG_I_CMD_DEFINE((TraceCmd_setCamTextFor2CMP, "setCamTextFor2CMP %d %d %d %d %d %d %d %d %d", int, int, int, int, int, int, int, int, int))
void CameraMain::TraceCmd_setCamTextFor2CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t cam5, uint16_t cam6, uint16_t cam7, uint16_t ipcam1, uint16_t ipcam2)
{
   ETG_TRACE_USR4(("CameraMain::TestDataToHMI_2CMP"));
   m_poCameraMain->TestDataToHMI_2CMP(cam1, cam2, cam3, cam4, cam5, cam6, cam7, ipcam1, ipcam2);
}


ETG_I_CMD_DEFINE((TraceCmd_setCamTextFor3CMP, "setCamTextFor3CMP %d %d %d %d %d %d %d %d %d %d %d %d %d", int, int, int, int, int, int, int, int, int, int, int, int, int))
void CameraMain::TraceCmd_setCamTextFor3CMP(uint16_t cam1, uint16_t cam2, uint16_t cam3, uint16_t cam4, uint16_t cam5, uint16_t cam6, uint16_t cam7, uint16_t cam8, uint16_t cam9, uint16_t cam10, uint16_t ipcam1, uint16_t ipcam2, uint16_t ipcam3)
{
   ETG_TRACE_USR4(("CameraMain::TestDataToHMI_3CMP"));
   m_poCameraMain->TestDataToHMI_3CMP(cam1, cam2, cam3, cam4, cam5, cam6, cam7, cam8, cam9, cam10, ipcam1, ipcam2, ipcam3);
}


/**********************************************************************************************
*FUNCTION:          startCameraBlackScreenTimer
*DESCRIPTION:       Starts inactivity timer
*PARAMETER:         None
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CameraMain::startCameraBlackScreenTimer(void)
{
   ETG_TRACE_USR4(("void CameraMain::startCameraBlackScreenTimer"));

   _camBlkScreen_Timer.start();
}


/**********************************************************************************************
*FUNCTION:          stopCameraBlackScreenTimer
*DESCRIPTION:       Stops inactivity timer
*PARAMETER:         None
*RETURNVALUE:       None
*
*HISTORY:
*05.05.2020:Prashant BG

*0.1 Initial Revision.
**********************************************************************************************/
void CameraMain::stopCameraBlackScreenTimer(void)
{
   ETG_TRACE_USR4(("void CameraMain::stopCameraBlackScreenTimer(void)"));
   _camBlkScreen_Timer.stop();
}


void CameraMain::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   ETG_TRACE_USR4(("void CameraMain::onExpired(void) Timer is Running"));

   if ((timer.getAct()  == _camBlkScreen_Timer.getAct()) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
   {
      //Black Screen Loading Animation for few seconds before the Camera is Streamed

      POST_MSG((COURIER_MESSAGE_NEW(CameraBlackScreenStopTriggerMsg)()));
      ETG_TRACE_USR4(("void CameraMain::onExpired(void) CameraBlackScreenTimer is STOPPPED"));
   }
}


}// namespace Core
}// namespace App
