/**************************************************************************************
* @file         : CameraListHandler.cpp
* @author       : ECH2- Somanath Baddi
* @addtogroup   :
* @brief        : Class contains necessary business logic required to handle List Handling for the Camera.
* @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/


#include "hall_std_if.h"
#include "CameraListHandler.h"
#include "CameraMain.h"
#include "CameraHall.h"
#include "CameraConfig/CameraConfig.h"
#include <sstream>
#include "CameraSourceChangeHandler/CameraSourceChangeHandler.h"
#include "App/Core/HmiTranslation_TextIds.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::CameraListHandler::
#include "trcGenProj/Header/CameraListHandler.cpp.trc.h"
#endif

static const unsigned int CAMPORT_ICON_SIZE = 3;
static const char* CAMPORT1_NORMAL_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_1Bmp";
static const char* CAMPORT1_ACTIVE_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_1Bmp";
static const char* CAMPORT2_NORMAL_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_2Bmp";
static const char* CAMPORT2_ACTIVE_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_2Bmp";
static const char* CAMPORT3_NORMAL_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_10Bmp";
static const char* CAMPORT3_ACTIVE_ICON                     = "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_5Bmp";

namespace App {
namespace Core {
CameraListHandler* CameraListHandler::m_poCameraListHandler = NULL;

/**
* Description     : Constructor class for CamerListHandler.
*/
CameraListHandler::CameraListHandler()
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("CameraListHandler::Constructor()"));
   ListRegistry::s_getInstance().addListImplementation(LISTID_CAMERA, this);
   (*oHighlightButton).mHighlightList = 0;
   (*oUpdateIcon).mActiveIcon = false;

   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   m_poCameraDataBindingHandler = CameraDataBindingHandler::poGetInstance();
   vListUpdate();
   bTextIDOverwrittenByCAN = false;
   m_camtxtStatusReceived = false;
   _avRoutingParser = tclAvRoutingParser::pGetInstance();
   if (_avRoutingParser != NULL)
   {
      bool m_bParseState = _avRoutingParser->bParseXml();
   }
   m_bHighlightListItemInCabin = false;
   //CRQ 1366236 Enable Camera Power Management at EvoBus
   previousCanIndex4Cockpit = 0;
   previousCanIndex4Cabin = 0;
   m_isBlackScreenEnabledValue = KDSHandler::poGetInstance()->u8GetCameraBlackScreenFeatureValue();//CID_R2 920776
   m_prevCabinSelectedIndex = 0;
   m_prevCabinSourceId = 0;
}


/************************************************************************
*FUNCTION: 		 getPreviousCanIndex4Cabin
*DESCRIPTION: 	 Function to get the previous Selected Selected Index 4 Cabin
*PARAMETER:		 NULL
*RETURNVALUE: 	 uint8 previousCanIndex4Cabin
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
uint8 CameraListHandler::getPreviousCanIndex4Cabin()
{
   ETG_TRACE_USR3(("void CameraListHandler::getPreviousCanIndex4Cabin previousCanIndex4Cabin = %d", previousCanIndex4Cabin));
   return previousCanIndex4Cabin;
}


/************************************************************************
*FUNCTION: 		 updatePreiousCanIndex4Cabin
*DESCRIPTION: 	 Function to Update the previous Selected Index 4 Cabin
*PARAMETER:		 uint8 prvCanInd4Cab
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void CameraListHandler::updatePreiousCanIndex4Cabin(uint8 prvCanInd4Cab)
{
   ETG_TRACE_USR3(("void CameraListHandler::updatePreiousCanIndex4Cabin prvCanInd4Cab = %d", prvCanInd4Cab));
   previousCanIndex4Cabin = prvCanInd4Cab;
}


/************************************************************************
*FUNCTION: 		 getPreviousCanIndex4Cockpit
*DESCRIPTION: 	 Function to get the previous Selected Selected Index 4 Cockpit
*PARAMETER:		 NULL
*RETURNVALUE: 	 uint8 previousCanIndex4Cockpit
*
//CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
uint8 CameraListHandler::getPreviousCanIndex4Cockpit()
{
   ETG_TRACE_USR3(("void CameraListHandler::getPreviousCanIndex4Cockpit previousCanIndex4Cockpit = %d", previousCanIndex4Cockpit));
   return previousCanIndex4Cockpit;
}


/************************************************************************
*FUNCTION: 		 updatePreviousCanIndex4Cockpit
*DESCRIPTION: 	 Function to Update the previous Selected Index 4 Cockpit
*PARAMETER:		 uint8 prvCanInd4Ckpt
*RETURNVALUE: 	 void
*CRQ 1366236 Enable Camera Power Management at EvoBus
************************************************************************/
void CameraListHandler::updatePreviousCanIndex4Cockpit(uint8 prvCanInd4Ckpt)
{
   ETG_TRACE_USR3(("void CameraListHandler::updatePreviousCanIndex4Cockpit prvCanInd4Ckpt=%d", prvCanInd4Ckpt));
   previousCanIndex4Cockpit = prvCanInd4Ckpt;
}


static const char* const _CameraIconsPath[CAMPORT_ICON_SIZE][2] =
{
   {"Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_1Bmp", "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_1Bmp"},
   {"Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_2Bmp", "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_2Bmp"},
   {"Global#Bitmaps#List#CameraListIcons#Icon_Camera_normal_10Bmp", "Global#Bitmaps#List#CameraListIcons#Icon_Camera_active_5Bmp"}
};


static const unsigned int CAMERALIST_DATA_SIZE = 22;

/**
* Description     : Destructor class for CamerListHandler
*/
CameraListHandler::~CameraListHandler()
{
   ETG_TRACE_USR4(("CameraListHandler::Destructor()"));

   if (m_poCameraListHandler != NULL)
   {
      delete m_poCameraListHandler;
      m_poCameraListHandler = NULL;
   }
   if (m_poCameraDataBindingHandler != NULL)
   {
      delete m_poCameraDataBindingHandler;
      m_poCameraDataBindingHandler = NULL;
   }
   ListRegistry::s_getInstance().removeListImplementation(LISTID_CAMERA);
   ETG_I_UNREGISTER_FILE();
}


void CameraListHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
}


void CameraListHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
}


/**
* Description     : Function to Get the data of the list and filling the list with available cameras.
*/
tSharedPtrDataProvider CameraListHandler::getCameraProfileList()
{
   ETG_TRACE_USR3(("CameraListHandler::getCameraProfileList()"));
   const char* const CAMERA_LIST_ITEM = "Button_Text_Image_Icon_List";
   ListDataProviderBuilder listBuilder(LISTID_CAMERA, CAMERA_LIST_ITEM);
   uint8 m_u8activeRegion =  CameraMain::poGetInstance()->getRegionValue();
   HighlightButtonData item;
   UpdateIconData data;

   if (m_u8activeRegion == REGION_C)
   {
      for (_cockpitIndex = 0 ; _cockpitIndex < sizeof(_cameraname_cockpit) / sizeof(_cameraname_cockpit[0]) ; _cockpitIndex++) //(CRQ-SplitRemoval) changed from 6 to 4. since split cams are not required (CRQ)
      {
         ETG_TRACE_USR3(("CameraListHandler : list %u", _cameraname_cockpit[_cockpitIndex]));
         if (_cameraname_cockpit[_cockpitIndex] == 0)
         {
            break;
         }
         if (_CMPPostiion_cockpit[_cockpitIndex] == 1)
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 1"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT1_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT1_ACTIVE_ICON);
         }
         else if (_CMPPostiion_cockpit[_cockpitIndex] == 2)
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 2"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT2_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT2_ACTIVE_ICON);
         }
         else
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 3"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT3_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT3_ACTIVE_ICON);
         }
         listBuilder.AddItem(_cockpitIndex, 0UL, CAMERA_LIST_ITEM)
         .AddDataBindingUpdater<UpdateIconDataBindingSource>(data)
         .AddData(Candera::String((_cameraname_cockpit[_cockpitIndex])));
      }
   }
   else if (m_u8activeRegion == REGION_A)
   {
      for (_cabinIndex = 0 ; _cabinIndex < sizeof(_cameraname_cabin) / sizeof(_cameraname_cabin[0]); _cabinIndex++) //(CRQ-SplitRemoval) changed from 7 to 1. Since only IP cam is required (CRQ)
      {
         ETG_TRACE_USR3(("CameraListHandler : list %u", _cameraname_cabin[_cabinIndex]));
         if (_cameraname_cabin[_cabinIndex] == 0)
         {
            break;
         }
         if (_CMPPosition_cabin[_cabinIndex] == 1)
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 1"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT1_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT1_ACTIVE_ICON);
         }
         else if (_CMPPosition_cabin[_cabinIndex] == 2)
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 2"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT2_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT2_ACTIVE_ICON);
         }
         else
         {
            ETG_TRACE_USR3(("CameraListHandler : list camport 3"));
            data.mNormalPressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT3_NORMAL_ICON);
            data.mActivePressedIcon = ImageLoader::getAssetBitmapImage(CAMPORT3_ACTIVE_ICON);
         }

         listBuilder.AddItem(_cabinIndex, 0UL, CAMERA_LIST_ITEM)
         .AddDataBindingUpdater<UpdateIconDataBindingSource>(data)
         .AddData(Candera::String(_cameraname_cabin[_cabinIndex]))
         .AddData(bCameraIconHighlights(_cabinIndex, _CMPPosition_cabin[_cabinIndex]));
      }
   }
   return listBuilder.CreateDataProvider();
}


/**
* Description     : Function to get the List ID.
*/
tSharedPtrDataProvider CameraListHandler::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   ETG_TRACE_USR3(("CameraListHandler : getListDataProvider"));
   ETG_TRACE_USR3(("CameraListHandler : list ID %d", listDataInfo.GetListId()));
   tSharedPtrDataProvider ptr = tSharedPtrDataProvider();
   switch (listDataInfo.GetListId())
   {
      case LISTID_CAMERA:
      {
         ptr = getCameraProfileList();
         break;
      }
   }
   return ptr;
}


bool CameraListHandler::onCourierMessage(const ListDateProviderReqMsg& oMsg)
{
   ETG_TRACE_USR3(("CameraListHandler : onCourierMessage ListDateProviderReqMsg"));
   return ListRegistry::s_getInstance().updateList(oMsg.GetListId(), oMsg.GetStartIndex(), oMsg.GetWindowElementSize());//TODO : Confirm if this should be included after list data is received
}


/**
* Description     : Method to return the value of row and column for camListSelectedIndex function.
*/
bool CameraListHandler::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   ETG_TRACE_USR4(("CameraListHandler::onCourierMessage(ButtonReactionMsg)"));
   ListProviderEventInfo info;
   bool isRetVal = false;
   unsigned int listId = info.getListId();
   int hdlRow = info.getHdlRow();
   unsigned int hdlCol = info.getHdlCol();
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), info) == true)
   {
      ETG_TRACE_USR4(("CameraListHandler::ButtonReactionMsg listId:%d", info.getListId()));
      if (oMsg.GetEnReaction() == enRelease)
      {
         switch (info.getListId())
         {
            case 10:
            {
               camListSelectedIndex(info.getHdlRow(), CAMERA_LIST_SEL);
               isRetVal = true;
            }
            break;
            default:
               break;
         }
         if (isRetVal)
         {
            POST_MSG((COURIER_MESSAGE_NEW(ButtonListItemUpdMsg)(listId, hdlRow, hdlCol, enRelease)));
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("CameraListHandler::onCourierMessage(ButtonReactionMsg) : failure = %d", info.getListId()));
   }
   return isRetVal;
}


/**
* Description     : Function to handle the request based on the item selected on the list of cabin and cockpit seperately.
*/
void CameraListHandler::camListSelectedIndex(uint32 selectedIndex, activatedScreen screenID)
{
   uint8 m_u8activeRegion =  CameraMain::poGetInstance()->getRegionValue();
   m_isBlackScreenEnabledValue = KDSHandler::poGetInstance()->u8GetCameraBlackScreenFeatureValue();
   int _speedStatus = 0;
   bool speedAndParkStatus = VehicleClientHandler::getInstance()->getSpeedAndParkingStatus();
   bool bSpeedAndParkBrakeStatusValidity = VehicleClientHandler::getInstance()->bGetSpeedAndParkBrakeValidityStatus();
   ETG_TRACE_USR4(("void CameraListHandler::camListSelectedIndex speedAndParkStatus %d bSpeedAndParkBrakeStatusValidity %d selectedIndex %d m_isBlackScreenEnabledValue %d ", speedAndParkStatus, bSpeedAndParkBrakeStatusValidity, selectedIndex, m_isBlackScreenEnabledValue));
   if (m_u8activeRegion == REGION_C)		// Region is Cockpit
   {
      vector<int> camTextId;
      camTextId = getAvailableCameraTextIdOnSelection();   // GET the text id from the list of available cameras.

      if (camTextId.at(selectedIndex) == -1)
      {
         ETG_TRACE_USR4(("void CameraListHandler::camListSelectedIndex COURIER_MESSAGE_NEW(ActivatePopupNotConfigMsg)"));
         POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupNotConfigMsg)()));
         return;
      }

      CameraConfig::getInstance()-> getSpeedStatus(camTextId.at(selectedIndex), _speedStatus);
      //Added validity check for speed and parking brake status. Bug ID 1267971
      if (bSpeedAndParkBrakeStatusValidity == SPEED_AND_PARKBRAKE_STATUS_VALID)
      {
         if (speedAndParkStatus == ALLOW_CAMERA)    // no restrictions. Allow all cameras. ie, speed is 0
         {
            ETG_TRACE_COMP(("void CameraListHandler::camListSelectedIndex REGION_C ALLOW_CAMERA"));
            vCockpitListSelection(selectedIndex, screenID);
         }
         if (speedAndParkStatus == ALLOW_PARTIAL_OR_BLOCK_CAMERA) 	// restrictions on few cameras. ie, speed is > 0
         {
            // None, cp, cb. cpcb
            if ((_speedStatus == COCKPIT_ONLY) || (_speedStatus == COCKPIT_CABIN))    // ignore speed.
            {
               ETG_TRACE_COMP(("void CameraListHandler::camListSelectedIndex REGION_C ALLOW_PARTIAL_OR_BLOCK_CAMERA"));
               vCockpitListSelection(selectedIndex, screenID);
            }
            else
            {
               ETG_TRACE_COMP(("void CameraListHandler::camListSelectedIndex REGION_C COURIER_MESSAGE_NEW -ActivatePopupMsg Camera_WarningPopup"));
               POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
            }
         }
      }
      else
      {
         ETG_TRACE_COMP(("void CameraListHandler::camListSelectedIndex REGION_C COURIER_MESSAGE_NEW else ActivatePopupMsg Camera_WarningPopup"));
         POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
      }
   }
   //List selection in Cabin
   if (m_u8activeRegion == REGION_A)
   {
      _Src sourceId = SRC_INVALID;
      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A setSelectedCameraIndex is SETTING UP"));
      setSelectedCameraIndex(selectedIndex);
      setNoHighlightInList(false);  // cabin list highlight
      //new changes
      int value = 0;
      bool streamStatusCabin = false;
      std::string strCamSrc = "";
      getCameraPositionAndSourceNameValuesCabin(selectedIndex, value,  strCamSrc, streamStatusCabin);     //strCamSrc will return cockpitSplit1/2/3 for splitcam source

      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         //set the split cam bits and splitcam contril id.
         CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, SPLTSCRNCAMS, value, strCamSrc);
         //depending on the position, change the cockpitSplitview1 to CMP1_SPLITSCREEN ( and so on )
         if (value == 1)
         {
            strCamSrc = "CMP1_SPLITSCREEN";
         }
         if (value == 2)
         {
            strCamSrc = "CMP2_SPLITSCREEN";
         }
         if (value == 3)
         {
            strCamSrc = "CMP3_SPLITSCREEN";
         }
      }

      sourceId = CameraConfig::getInstance()->getCabinSrcIdForCam(strCamSrc);
      m_poCameraDataBindingHandler ->vShowHomeScreenNextPrevButtons(1);  // ded6kor uncommented
      CameraMain::poGetInstance()->setToggleValue(true);

      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A sourceID=%d", sourceId));

      if (sourceId != SRC_INVALID)
      {
         CameraSourceChangeHandler::getInstance()->requestSourceActivation(sourceId, m_u8activeRegion);
      }

      m_poCameraDataBindingHandler -> updateCameraNameInTile(_cameraname_cabin[selectedIndex]);
      _activeCameraNameCabin = _cameraname_cabin[selectedIndex];
      updateList(LISTID_CAMERA);
      setCameraStreamStatusCabin(selectedIndex, true);
      // added for CRQ 1366236: [EvoBus][INF4CV]: Enable Camera Power Management at EvoBus
      // send CAN request as active for selected index ( current camera )
      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A send list src activation sourceID=%d, src name %s", sourceId, strCamSrc.c_str()));
      VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, ACTIVATE); //manact is enabled corresponding to camsrc for efficient power utilization

      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A send list src Deactivation ? prevIdx %d selIdx %d", m_prevCabinSelectedIndex, selectedIndex));
      // send CAN power signal as off for previous index
      if (m_prevCabinSelectedIndex != selectedIndex)
      {
         std::string camCabinSrcName = "";
         getCameraPositionAndSourceNameValuesCabin(m_prevCabinSelectedIndex, value,  camCabinSrcName, streamStatusCabin);
         ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A send list src Deactivation src name m_prevCabinSelectedIndex %d value %d streamStatusCabin %d  camCabinSrcName %s", m_prevCabinSelectedIndex, value, streamStatusCabin,   camCabinSrcName.c_str()));
         VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(camCabinSrcName, DEACTIVATE);
      }
      //store the selected index, since this will used to send deactivate signal.
      m_prevCabinSelectedIndex = selectedIndex;
      m_prevCabinSourceId = sourceId;

      ETG_TRACE_USR4(("CameraListHandler::camListSelectedIndex REGION_A before exit prevIdx %d prevCabinSrcId %d", m_prevCabinSelectedIndex, m_prevCabinSourceId));
   }
}


void CameraListHandler::vCockpitListSelection(uint32 listSeletedIndex, activatedScreen screenID)
{
   ETG_TRACE_USR4(("CameraListHandler::vCockpitListSelection = %d", listSeletedIndex));
   //RVC CRQ -> Check if the first camera is configured for RVC to display warning message.
   if ((listSeletedIndex == 0) && (true == bCheckRearCameraAvailable()))
   {
      m_poCameraDataBindingHandler->vShowCameraWarningText(true);
   }
   else
   {
      m_poCameraDataBindingHandler->vShowCameraWarningText(false);
   }
   int value = 0;
   bool streamStatus = false;
   std::string strCamSrc = "";
   getCameraPositionAndSourceNameValues(listSeletedIndex, value, strCamSrc, streamStatus);

   CameraMain::poGetInstance()->sendPluginRequestToAVDECC(PLUGIN_NAME_CAMERA, CVBSSOURCE, value, strCamSrc);
   setSelectedCockpitCameraIndex(listSeletedIndex);

   //CRQ 1366236 Enable Camera Power Management at EvoBus
   int canIndex = getOriginalXMLIndex(strCamSrc);
   CameraListHandler::getInstance()->getCameraPositionAndSourceNameValues(canIndex, value, strCamSrc, streamStatus);
   ETG_TRACE_USR4(("CameraListHandler::vCockpitListSelection COCKPIT CurrentCandex  canIndex, value,  streamStatus, strCamSrc %d %d %d %s", canIndex, value,  streamStatus, strCamSrc));
   VehicleClientHandler::getInstance()->vUpdatetoCANfromHMI(strCamSrc, ACTIVATE); //manact is enabled corresponding to camsrc for efficient power utilization
   updatePreviousCanIndex4Cockpit(canIndex + 1);

   m_poCameraDataBindingHandler -> updateCameraNameInStreamScreen(_cameraname_cockpit[listSeletedIndex]);
   _activeCockpitflag = true;

   if (activatedScreen(CAMERA_LIST_SEL) == screenID)
   {
      ETG_TRACE_USR4(("CameraListHandler::vCockpitListSelection screenID strCamSrc %d %s ", screenID, strCamSrc.c_str()));
      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         CameraMain::poGetInstance()->setSplitCamSourceAndPosition(strCamSrc, value); //required for split view to set the source name.
         // Trigger Via External (CAN/Dig PIN) --> Go to the Split Camera (BlackScreen) [OR] Trigger Via List --> Go to Split Camera (No BlackScreen)
         vStateMachineExternalTriggerToCameraOrBlackScreen();
      }
      else
      {
         // Trigger Via List --> Go to Normal Camera - (BlackScreen) [OR] Trigger Via List --> Go to Normal Camera (No BlackScreen)
         vStateMachineTriggerToCameraOrBlackScreen();
      }
      setCameraStreamStatusCockpit(listSeletedIndex, true);  //cockpit
   }
   else
   {
      ETG_TRACE_USR4(("CameraListHandler::vCockpitListSelection ELSE screenID strCamSrc %d %s ", screenID, strCamSrc.c_str()));
      if ((strCamSrc == _strSplitCamSrcName[0]) || (strCamSrc == _strSplitCamSrcName[1]) || (strCamSrc == _strSplitCamSrcName[2]))
      {
         // Context switch to 259(Black screen of split camera screen) [or] 256 (directly to split screen)
         vContextSwitchRequestSplitOrBlackScreen();
      }
      else
      {
         // Context switch to 258(Black screen of normal camera screen) [or] 251 (directly to normal camera screen)
         vContextSwitchRequestCameraOrBlackScreen();
      }
      setCameraStreamStatusCockpit(listSeletedIndex, true);
   }

   //VehicleClientHandler::getInstance()->SetCANUpdate(true);  // added for Bug Id 1318897
}


/**
* Description     : Function to return the name of active camera name.
*/
uint32 CameraListHandler::getActiveCameraName(int selectedindex)
{
   for (int i = 0; i < sizeof(_cameraname_cockpit) / sizeof(_cameraname_cockpit[0]) ; i++)
   {
      if (i == selectedindex)
      {
         _activeCameraName = _cameraname_cockpit[i];
         break;
      }
   }
   return _activeCameraName;
}


uint32 CameraListHandler::getActiveCameraNameCabin(int selectedindex)
{
   for (int i = 0; i < sizeof(_cameraname_cabin) / sizeof(_cameraname_cabin[0]); i++)
   {
      if (i == selectedindex)
      {
         _activeCameraNameCabin = _cameraname_cabin[i];
         break;
      }
   }

   ETG_TRACE_USR4(("CameraListHandler::getActiveCameraNameCabin _activeCameraNameCabin=%d ", _activeCameraNameCabin));
   return _activeCameraNameCabin;
}


void CameraListHandler::vListUpdate()
{
   uint32 camvalue;
   int regionStatus = 0;
   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();  // get the vecStCamData of stCameraData type from xml file
   vector<int> camTextId = CameraMain::poGetInstance()->getCameraTextIds(); //get text id directly from xml
   ETG_TRACE_USR3(("vListUpdate(): size  is  [%d]", camTextId.size()));

   vector<int>::iterator itr;
   camAvailableSourceNames.clear();
   camAvailableSourceNamesCabin.clear();
   camAvailableXMLTextIdsCockpit.clear();
   camAvailableXMLTextIdsCabin.clear();

   camHexCamData.clear();
   camHexCamDataCabin.clear();

   for (itr = camTextId.begin(); itr != camTextId.end(); ++itr)
   {
      ETG_TRACE_USR3(("void CameraListHandler::vListUpdate() Value is  [%d]", *itr));
   }

   std::memset(_cameraname_cockpit, 0, sizeof(_cameraname_cockpit));
   std::memset(_cameraname_cabin, 0, sizeof(_cameraname_cabin));
   std::memset(_CMPPostiion_cockpit, 0, sizeof(_CMPPostiion_cockpit));
   std::memset(_CMPPosition_cabin, 0, sizeof(_CMPPosition_cabin));

   int j = 0;
   int k = 0;
   for (int index = 0; index < camTextId.size(); index++)
   {
      CameraConfig::getInstance()->getCameraValues(camTextId.at(index), camvalue); //based on cantextid index--> we get translation text id  example TextId_0x150B
      CameraConfig::getInstance()->getRegionStatus(camTextId.at(index), regionStatus);//based on camtextid index , finds in which region status is applicable or not (shown in cockpit/cabin/both)
      if (camvalue != 0xFF)
      {
         if ((regionStatus == COCKPIT) || (regionStatus == BOTH))
         {
            _cameraname_cockpit[j] = camvalue; //Translation ID --> TextId_0x150B
            _CMPPostiion_cockpit[j] = vecStCamData.at(index).position;// position value
            camHexCamData.insert(pair<int, stCameraData>(j, {vecStCamData.at(index).position, vecStCamData.at(index).camTextID, vecStCamData.at(index).camerasource.c_str(), vecStCamData.at(index).streamStatus, vecStCamData.at(index).streamStatusCabin}));
            camAvailableSourceNames.push_back(vecStCamData.at(index).camerasource.c_str()); //push available source names, from xml ex cmp1_cam1,cmp2_cam2 etc
            camAvailableXMLTextIdsCockpit.push_back(vecStCamData.at(index).camTextID);  // push only available camera text ids from xml ex 0x0158,0x35,0x48

            ++j;
         }
         if ((regionStatus == BOTH) || (regionStatus == CABIN))
         {
            _cameraname_cabin[k] = camvalue;
            _CMPPosition_cabin[k] = vecStCamData.at(index).position;
            camHexCamDataCabin.insert(pair<int, stCameraData>(k, {vecStCamData.at(index).position, vecStCamData.at(index).camTextID, vecStCamData.at(index).camerasource.c_str(), vecStCamData.at(index).streamStatus, vecStCamData.at(index).streamStatusCabin}));
            camAvailableSourceNamesCabin.push_back(vecStCamData.at(index).camerasource.c_str());
            camAvailableXMLTextIdsCabin.push_back(vecStCamData.at(index).camTextID);
            ++k;
         }
      }
   }
   vprintdata();
   vPrintCamHexCamDataMap();
}


vector<int> CameraListHandler::vgetCANTextId()
{
   return CANTextId;
}


void CameraListHandler::updateHMIAvailableCamerasFromCAN(uint32 cameraname_arr[], bool camTxtStatusReceived)
{
   ETG_TRACE_USR4(("CameraListHandler::updateHMIAvailableCamerasFromCAN"));
   std::memset(m_cameraname_arr, 0, sizeof(m_cameraname_arr));
   // CANTextId.clear();
   for (int i = 0; i < 8; i++)
   {
      m_cameraname_arr[i] = cameraname_arr[i];
   }

   m_camtxtStatusReceived = camTxtStatusReceived;
   if (sm_gCMPStatus == 1)
   {
      vPrepareCAMTextConfiguration(cameraname_arr);
      vUpdateCameraListData();
      //update the camera tiles when camera list is updated after startup.
      m_poCameraDataBindingHandler -> vCamAvailability(true);
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
   }
}


void CameraListHandler::vprintdata()
{
   for (int index = 0; index < sizeof(_cameraname_cockpit) / sizeof(int); index++)
   {
      ETG_TRACE_USR3(("CameraListHandler : vprintdata cockpit %u", _cameraname_cockpit[index]));
   }
   for (int index = 0; index < sizeof(_cameraname_cabin) / sizeof(int); index++)
   {
      ETG_TRACE_USR3(("CameraListHandler : vprintdata cabin %u", _cameraname_cabin[index]));
   }
}


/**
* Description     : Function to return the index of active camera in cabin.
*/
int CameraListHandler::getSelectedCameraIndex()
{
   ETG_TRACE_USR3(("CameraListHandler::getSelectedCameraIndex() cabin_A _activeCameraIndex==>%d", _activeCameraIndex));
   return _activeCameraIndex;
}


/**
* Description     : Function to return the index of active camera in cockpit.
*/
int CameraListHandler::getSelectedCockpitCameraIndex()
{
   ETG_TRACE_USR3(("CameraListHandler::getSelectedCockpitCameraIndex() Cockpit _activeCockpitCamera:==>%d", _activeCockpitCamera));
   return _activeCockpitCamera;
}


/**
* Description     : Function to set the value of active camera in cabin on request.
*/
void CameraListHandler::setSelectedCameraIndex(int index)
{
   _activeCameraIndex = index;
   ETG_TRACE_USR3(("CameraListHandler::setSelectedCameraIndex() cabin_A _activeCameraIndex/index:==>%d", _activeCameraIndex));
}


/**
* Description     : Function to set the value of active camera in cockpit.
*/
void CameraListHandler::setSelectedCockpitCameraIndex(int index)
{
   _activeCockpitCamera = index;
   ETG_TRACE_USR3(("CameraListHandler::setSelectedCockpitCameraIndex() Cockpit _activeCockpitCamera/index:==>%d", index));
}


bool CameraListHandler::getCockpitCameraIndex()
{
   ETG_TRACE_USR3(("CameraListHandler::getCockpitCameraIndex() Cockpit _activeCockpitflag:==>%d", _activeCockpitflag));
   return _activeCockpitflag;
}


int CameraListHandler::getCabinListsize()
{
   return camAvailableSourceNamesCabin.size();
}


int CameraListHandler::getCockpitListsize()
{
   return camAvailableSourceNames.size();
}


bool CameraListHandler::getSpeedLockStatus()
{
   return m_SpeedLockActive;
}


/**
* Description     : Function to update the list if any changes occur.
*/
void CameraListHandler::updateList(const uint32& listID)
{
   ETG_TRACE_USR3(("CameraListHandler : updateList"));
   ListRegistry::s_getInstance().updateList(listID);
}


/*vPrintCamHexCamDataMap() Contains Below Information
//camHexCamData Vector Contains Final Cockpit List for EVOBUS
//camHexCamDataCabin Vector Contains Final Cabin List for EVOBUS
*/
void CameraListHandler::vPrintCamHexCamDataMap()
{
   map<int, stCameraData>::iterator itr;
   ETG_TRACE_USR4(("vPrintCamHexCamDataMap Cockpit :: Print mapping data"));
   for (itr = camHexCamData.begin(); itr != camHexCamData.end(); ++itr)
   {
      ETG_TRACE_USR4(("CameraListHandler::vPrintCamHexCamDataMap{ %d %d %d %s}", itr->second.position, itr->second.camTextID, itr->second.streamStatus, itr->second.camerasource.c_str()));
   }
   ETG_TRACE_USR4(("vPrintCamHexCamDataMap Cabin :: Print cabin mapping data"));
   for (itr = camHexCamDataCabin.begin(); itr != camHexCamDataCabin.end(); ++itr)
   {
      ETG_TRACE_USR4(("CameraListHandler::vPrintCamHexCamDataMapCabin { %d %d %d %s}", itr->second.position, itr->second.camTextID, itr->second.streamStatusCabin, itr->second.camerasource.c_str()));
   }
}


//Get the position , cam source, stream status from key index (refering the filtered cocpit vector list i.e -1 )
// key referes to index
//stCameraData-->int position;    int camTextID;    std::string camerasource;    bool streamStatus;    bool streamStatusCabin;
void CameraListHandler::getCameraPositionAndSourceNameValues(int key, int& value, std::string& strCamSrc, bool& streamStatus)
{
   ETG_TRACE_USR4(("CameraListHandler::getCameraPositionAndSourceNameValues for list item  (%d)", key));
   map<int, stCameraData>::iterator itr;
   for (itr = camHexCamData.begin(); itr != camHexCamData.end(); ++itr)
   {
      if (itr->first == key)
      {
         value = itr->second.position;
         strCamSrc = itr->second.camerasource;
         streamStatus = itr->second.streamStatus;
         ETG_TRACE_USR3(("CameraListHandler::getCameraPositionAndSourceNameValues(): Position is %d, streamstatus = %d,  %s", value, streamStatus, strCamSrc.c_str()));
         break;
      }
   }
}


void CameraListHandler::getCameraPositionAndSourceNameValuesCabin(int key, int& value, std::string& strCamSrc, bool& streamStatusCabinflag)
{
   ETG_TRACE_USR4(("CameraListHandler::getCameraPositionAndSourceNameValuesCabin for list item  (%d)", key));
   map<int, stCameraData>::iterator itr;
   for (itr = camHexCamDataCabin.begin(); itr != camHexCamDataCabin.end(); ++itr)
   {
      if (itr->first == key)
      {
         value = itr->second.position;

         strCamSrc = itr->second.camerasource;
         streamStatusCabinflag = itr->second.streamStatusCabin;
         ETG_TRACE_USR3(("CameraListHandler::getCameraPositionAndSourceNameValuesCabin(): key==>%d Position==>%d, streamStatusCabinflag = %d and camsource %s ", key, value, streamStatusCabinflag, strCamSrc.c_str()));
         break;
      }
   }
}


int CameraListHandler::bCheckRequestedCamaraExists(std::string cameraSrcName)
{
   ETG_TRACE_USR4(("CameraListHandler::bCheckRequestedCamaraExists %s", cameraSrcName.c_str()));
   int index = -1;
   if (camAvailableSourceNames.size() != 0)
   {
      std::vector <std::string>::iterator itr = std::find(camAvailableSourceNames.begin(), camAvailableSourceNames.end(), cameraSrcName);
      if (itr != camAvailableSourceNames.end())
      {
         index = itr - camAvailableSourceNames.begin();
      }
      else
      {
         index = -1;
      }
   }
   ETG_TRACE_USR4(("CameraMain::bCheckRequestedCamaraExists bCameraSrcAvail %d", index));
   return index;
}


void CameraListHandler::vListUpdateByCamTxtSignal()
{
   uint32 camvalue;
   int regionStatus = 0;
   vector<int> cameraTextIDs;
   cameraTextIDs = vgetCANTextId();
   camHexCamData.clear();
   camHexCamDataCabin.clear();

   camAvailableCAMTextIdsCockpit.clear();
   camAvailableCAMTextIdsCabin.clear();
   camAvailableSourceNames.clear();
   camAvailableSourceNamesCabin.clear();

   ETG_TRACE_USR4(("void CameraListHandler::vListUpdateByCamTxtSignal() cameraTextIDs size [%d] m_vecCamDataToHmi_CANTxtSignal size[%d]", cameraTextIDs.size(), m_vecCamDataToHmi_CANTxtSignal.size()));
   if ((cameraTextIDs.size() > 0) && (m_vecCamDataToHmi_CANTxtSignal.size() > 0))
   {
      std::memset(_cameraname_cockpit, 0, sizeof(_cameraname_cockpit));
      std::memset(_cameraname_cabin, 0, sizeof(_cameraname_cabin));
      std::memset(_CMPPostiion_cockpit, 0, sizeof(_CMPPostiion_cockpit));
      std::memset(_CMPPosition_cabin, 0, sizeof(_CMPPosition_cabin));

      int j = 0;
      int k = 0;
      for (int index = 0; index < cameraTextIDs.size(); index++)
      {
         CameraConfig::getInstance()->getCameraValues(cameraTextIDs.at(index), camvalue);
         CameraConfig::getInstance()->getRegionStatus(cameraTextIDs.at(index), regionStatus);
         if ((camvalue != 0xFF))
         {
            if ((regionStatus == COCKPIT) || (regionStatus == BOTH))
            {
               _cameraname_cockpit[j] = camvalue;
               _CMPPostiion_cockpit[j] = m_vecCamDataToHmi_CANTxtSignal.at(index).position;
               camHexCamData.insert(pair<int, stCameraData>(j, {m_vecCamDataToHmi_CANTxtSignal.at(index).position, m_vecCamDataToHmi_CANTxtSignal.at(index).camTextID, m_vecCamDataToHmi_CANTxtSignal.at(index).camerasource.c_str(), m_vecCamDataToHmi_CANTxtSignal.at(index).streamStatus, m_vecCamDataToHmi_CANTxtSignal.at(index).streamStatusCabin}));
               camAvailableSourceNames.push_back(m_vecCamDataToHmi_CANTxtSignal.at(index).camerasource.c_str()); //push available source names, for CAN / NM activations.
               camAvailableCAMTextIdsCockpit.push_back(m_vecCamDataToHmi_CANTxtSignal.at(index).camTextID);
               ++j;
            }
            if ((regionStatus == BOTH) || (regionStatus == CABIN))
            {
               _cameraname_cabin[k] = camvalue;
               _CMPPosition_cabin[k] = m_vecCamDataToHmi_CANTxtSignal.at(index).position;
               camHexCamDataCabin.insert(pair<int, stCameraData>(k, {m_vecCamDataToHmi_CANTxtSignal.at(index).position, m_vecCamDataToHmi_CANTxtSignal.at(index).camTextID, m_vecCamDataToHmi_CANTxtSignal.at(index).camerasource.c_str(), m_vecCamDataToHmi_CANTxtSignal.at(index).streamStatus, m_vecCamDataToHmi_CANTxtSignal.at(index).streamStatusCabin}));
               camAvailableSourceNamesCabin.push_back(m_vecCamDataToHmi_CANTxtSignal.at(index).camerasource.c_str());
               camAvailableCAMTextIdsCabin.push_back(m_vecCamDataToHmi_CANTxtSignal.at(index).camTextID);
               ++k;
            }
         }
      }
      vprintdata();
      vPrintCamHexCamDataMap();
      bTextIDOverwrittenByCAN = true;  // set this flag to true, since xml is overwritten by CAN.

      vUpdateAvailableCameraTextIdsFromCANSignal();

      updateList(LISTID_CAMERA);
   }
   else
   {
      ETG_TRACE_USR4(("CameraListHandler::vListUpdateByCamTxtSignal() vector size are empty"));
   }
}


// Return vector of all text ids including -1
vector<int> CameraListHandler::getAllCameraTextIdOnSelection()
{
   vector<int> camTextId;
   if (bTextIDOverwrittenByCAN == TRUE)
   {
      camTextId = vgetCANTextId();   // Use text id from CAN CAMTXT signal.
   }
   else
   {
      camTextId = CameraMain::poGetInstance()->getCameraTextIds();   // Use text id from XML config.
   }
   return camTextId;
}


// get available vectortextids  image (of int format) only ,  except -1.
vector<int> CameraListHandler::getAvailableCameraTextIdOnSelection()
{
   vector<int> camTextId;
   if (bTextIDOverwrittenByCAN == TRUE) // for evobus --> get the analog camera TEXTID VIA CAN and rest from xml
   {
      camTextId = getAvailableCameraTextIdsFromCamSignal();
   }
   else // for generic --> "camAvailableXMLTextIdsCockpit"
   {
      camTextId = getAvailableCameraTextIds();
   }
   return camTextId;
}


void CameraListHandler::vUpdateCameraListData()
{
   int _variantInfo = KDSHandler::poGetInstance()->getVariantInfoFromKDS();
   ETG_TRACE_USR3(("CameraListHandler::vUpdateCameraListData() variant camtxtSigStatus [%d %d]", _variantInfo, m_camtxtStatusReceived));
   if ((_variantInfo == 4) && (m_camtxtStatusReceived == true))    // //List Update for EVOBUS, and CamTxt_Req signal is received with data, use this list
   {
      vPrepareCAMTextConfiguration(m_cameraname_arr);
      vListUpdateByCamTxtSignal();
      //update the camera tiles when camera list is updated after startup.
      m_poCameraDataBindingHandler -> vCamAvailability(true);
      m_poCameraDataBindingHandler -> vCamAvailabilityCabin(true, true);
   }
   else //List Update for Generic
   {
      vListUpdate();
   }
}


void CameraListHandler::vUpdateAvailableCameraTextIdsFromCANSignal()
{
   m_availableCameraTxtIdsFromCAN.clear();
   vector<int> cameraTextIds = vgetCANTextId();
   for (int i = 0; i < cameraTextIds.size(); i++)
   {
      if ((cameraTextIds.at(i) != 0xFFFF) || (cameraTextIds.at(i) != 0xFF))
      {
         m_availableCameraTxtIdsFromCAN.push_back(cameraTextIds.at(i));
      }
   }
   ETG_TRACE_USR4(("CameraListHandler:vUpdateAvailableCameraTextIdsFromCANSignal:[%d]", m_availableCameraTxtIdsFromCAN.size()));
   for (int i = 0; i < m_availableCameraTxtIdsFromCAN.size(); i++)
   {
      ETG_TRACE_USR4(("CameraListHandler:vUpdateAvailableCameraTextIdsFromCANSignal:[%d]", m_availableCameraTxtIdsFromCAN.at(i)));
   }
}


vector<int> CameraListHandler::getAvailableCameraTextIdsFromCamSignal()
{
   for (int i = 0; i < camAvailableCAMTextIdsCockpit.size()/* m_availableCameraTxtIdsFromCAN.size() */; i++)
   {
      ETG_TRACE_USR4(("CameraListHandler:getAvailableCameraTextIdsFromCamSignal:[%d]", camAvailableCAMTextIdsCockpit.at(i)));
   }
   return camAvailableCAMTextIdsCockpit; //m_availableCameraTxtIdsFromCAN;
}


//Fetch the original index of the camera list, to select the port name.
//Fetch the original xml index based on xml camera source string
int CameraListHandler::getOriginalXMLIndex(std::string camSrcName)
{
   int camSrcIndex = 0;
   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure(); //get original xml vector image of struct type stCameraData
   for (int index = 0; index < vecStCamData.size(); index++)
   {
      std::string xmlCamSrcName = vecStCamData.at(index).camerasource.c_str();
      if (0 == strcmp(camSrcName.c_str(), xmlCamSrcName.c_str()))
      {
         camSrcIndex = index;
         break;
      }
   }
   return camSrcIndex;
}


// print all the Cockpit camera CAN text ids except -1 and returns
vector<int> CameraListHandler::getAvailableCameraTextIds()
{
   for (int i = 0; i < camAvailableXMLTextIdsCockpit.size(); i++)
   {
      ETG_TRACE_USR4(("CameraListHandler:getAvailableCameraTextIds:[%d]", camAvailableXMLTextIdsCockpit.at(i)));
   }
   return camAvailableXMLTextIdsCockpit;
}


void CameraListHandler::fillDataInVector(int indexCMP, int loopSize)
{
   int u8Index = indexCMP;
   for (tU8 loop = 0 ; loop < loopSize; ++loop)
   {
      m_vecCamDataToHmi_CANTxtSignal.push_back(m_vecCamDataToHmi_CANTxt[u8Index]);
      CANTextId.push_back(m_vecCamDataToHmi_CANTxt.at(u8Index).camTextID);
      ++u8Index;
   }
}


void CameraListHandler::prepareConfigBasedOnOneCMP(uint32 cameraname_arr[], int cmp1IPCamIndex)
{
   stCameraData scd = {0, 0, "", false, false};//CID_R1 11378975
   m_vecCamDataToHmi_CANTxt.clear();
   int position[] = {1, 1, 1, 1, 1};
   std::string srcName[] = {"CMP1_CAM1", "CMP1_CAM2", "CMP1_CAM3", "CMP1_CAM4", "CMP1_IPCAM"};
   for (int i = 0; i < 4; i++)
   {
      scd.position = position[i];
      scd.camerasource = srcName[i];
      scd.camTextID = cameraname_arr[i];
      m_vecCamDataToHmi_CANTxt.push_back(scd);
   }
   scd.position = position[4];
   scd.camerasource = srcName[4];
   scd.camTextID = cmp1IPCamIndex;
   m_vecCamDataToHmi_CANTxt.push_back(scd);

   ETG_TRACE_USR4(("prepareConfigBasedOnOneCMP:: 1 CMP Tracing print"));
   for (int i = 0; i < m_vecCamDataToHmi_CANTxt.size(); i++)
   {
      ETG_TRACE_USR4(("1CMP config P->%d T->%d S->%s", m_vecCamDataToHmi_CANTxt.at(i).position, m_vecCamDataToHmi_CANTxt.at(i).camTextID, m_vecCamDataToHmi_CANTxt.at(i).camerasource.c_str()));
   }
}


void CameraListHandler::prepareConfigBasedOnTwoCMP(uint32 cameraname_arr[], int cmp1IPCamIndex, int cmp2IPCamIndex)
{
   stCameraData scd = {0, 0, "", false, false};//CID_R1 11378979
   m_vecCamDataToHmi_CANTxt.clear();
   int position[] = {1, 1, 1, 2, 2, 2, 2, 1, 2};
   std::string srcName[] = {"CMP1_CAM1", "CMP1_CAM2", "CMP1_CAM3", "CMP2_CAM1", "CMP2_CAM2", "CMP2_CAM3", "CMP2_CAM4", "CMP1_IPCAM", "CMP2_IPCAM"};
   for (int i = 0; i < 7; i++)
   {
      scd.position = position[i];
      scd.camerasource = srcName[i];
      scd.camTextID = cameraname_arr[i];
      m_vecCamDataToHmi_CANTxt.push_back(scd);
   }
   scd.position = position[7];
   scd.camerasource = srcName[7];
   scd.camTextID = cmp1IPCamIndex;
   m_vecCamDataToHmi_CANTxt.push_back(scd);

   scd.position = position[8];
   scd.camerasource = srcName[8];
   scd.camTextID = cmp2IPCamIndex;
   m_vecCamDataToHmi_CANTxt.push_back(scd);

   ETG_TRACE_USR4(("prepareConfigBasedOnTwoCMP:: 2 CMP Tracing print"));
   for (int i = 0; i < m_vecCamDataToHmi_CANTxt.size(); i++)
   {
      ETG_TRACE_USR4(("2CMP config P->%d T->%d S->%s", m_vecCamDataToHmi_CANTxt.at(i).position, m_vecCamDataToHmi_CANTxt.at(i).camTextID, m_vecCamDataToHmi_CANTxt.at(i).camerasource.c_str()));
   }
}


int CameraListHandler::getIPCameraValue(std::string strIpCamName)
{
   ETG_TRACE_USR4(("CameraListHandler:getIPCameraValue: %s", strIpCamName.c_str()));
   int ipCamValue = 0;
   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();  // get the structure received.
   vector<int> camTextId = CameraMain::poGetInstance()->getCameraTextIds();
   for (int index = 0; index < camTextId.size(); index++)
   {
      //ETG_TRACE_USR4((" = : %s", vecStCamData.at(index).camerasource.c_str()));
      if (vecStCamData.at(index).camerasource ==  strIpCamName)
      {
         ipCamValue = vecStCamData.at(index).camTextID;
         break;
      }
   }
   return ipCamValue;
}


vector<stCameraData> CameraListHandler::setConfigForCANTxtId(uint32 cameraname_arr[], int startIndex, int endIndex, int CANTextIDCount)
{
   stCameraData scd = {0, 0, "", false, false};//CID_R1 11378980
   m_vecCamDataToHmi_CANTxt.clear();

   ETG_TRACE_USR4(("CameraListHandler:setConfigForCANTxtId: CAN Text Count %d StartIndex %d EndIndex %d", CANTextIDCount, startIndex, endIndex));

   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();  // get the structure received.
   vector<int> camTextId = CameraMain::poGetInstance()->getCameraTextIds();
   ETG_TRACE_USR4(("CameraListHandler:setConfigForCANTxtId: vecStCamData size %d camTextId size %d", vecStCamData.size(), camTextId.size()));

   for (int i = 0; i < CANTextIDCount; i++)
   {
      scd.position = vecStCamData.at(i).position;
      scd.camerasource = vecStCamData.at(i).camerasource;
      scd.camTextID = cameraname_arr[i];
      m_vecCamDataToHmi_CANTxt.push_back(scd);
   }

   for (int j = startIndex; j < endIndex ; j++)
   {
      scd.position = vecStCamData.at(j).position;
      scd.camerasource = vecStCamData.at(j).camerasource;
      scd.camTextID = vecStCamData.at(j).camTextID;
      m_vecCamDataToHmi_CANTxt.push_back(scd);
   }

   ETG_TRACE_USR4(("setConfigForCANTxtId:: 1 CMP Tracing print"));
   for (int i = 0; i < m_vecCamDataToHmi_CANTxt.size(); i++)
   {
      CANTextId.push_back(m_vecCamDataToHmi_CANTxt.at(i).camTextID);
      ETG_TRACE_USR4(("1CMP config P->%d T->%d S->%s", m_vecCamDataToHmi_CANTxt.at(i).position, m_vecCamDataToHmi_CANTxt.at(i).camTextID, m_vecCamDataToHmi_CANTxt.at(i).camerasource.c_str()));
   }

   return m_vecCamDataToHmi_CANTxt;
}


/**
* Description     : In EVOBUS Prepare the List by using CAN AND XML
* @Method 		  : vPrepareCAMTextConfiguration
* @param[in]      : cameraname_arr : CAN Values
* @return         : void
*/
void CameraListHandler::vPrepareCAMTextConfiguration(uint32 cameraname_arr[])
{
   m_vecCamDataToHmi_CANTxt.clear();
   m_vecCamDataToHmi_CANTxtSignal.clear();
   stCameraData scd = {0, 0, "", false, false};//CID_R1 11378977

   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();  // get the structure received.
   uint8 index,  startPos = POS_ZERO;
   for (index = POS_ZERO; index < vecStCamData.size(); index++)
   {
      ETG_TRACE_USR4(("CameraListHandler::vPrepareCAMTextConfiguration XML DATA Before CAN Assignment P->%d T->%d S->%s", vecStCamData.at(index).position, vecStCamData.at(index).camTextID, vecStCamData.at(index).camerasource.c_str()));
   }

   uint8 sizeOfcameraname_arr = sizeof(cameraname_arr);// CID_R2 13831066 (sizeof(cameraname_arr) / sizeof(cameraname_arr[0]));
   for (index = POS_ZERO; index < sizeOfcameraname_arr; index++)
   {
      ETG_TRACE_USR4(("CameraListHandler::vPrepareCAMTextConfiguration CAN Values cameraname_arr CANtextIdvalue :%02X) ", cameraname_arr[index]));
   }

   uint8 numberOfCMPs = TOTAL_ZERO_CMP;
   if (_avRoutingParser != NULL)
   {
      numberOfCMPs = _avRoutingParser->vectorGetEntityID("CMP").size();
   }

   uint8 camportConnnectedDevice = TOTAL_ZERO_CMP;
   uint8 cmpConnDisconnectStatus  = CameraSourceChangeHandler::getInstance()->getCMPConnectionStatus();
   ETG_TRACE_USR4(("CameraListHandler::vPrepareCAMTextConfiguration::cmpConnDisconnectStatus %d sizeOfcameraname_arr=%d", cmpConnDisconnectStatus, sizeOfcameraname_arr));
   switch (cmpConnDisconnectStatus)
   {
      case CAMPORT_CONNECTED_STATUS_TER_OFF_SEC_OFF_PRI_ONN:   // 1
      case CAMPORT_CONNECTED_STATUS_TER_ONN_SEC_OFF_PRI_ONN:   // 5
         camportConnnectedDevice = ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_OFF_PRI_ONN; //1- Primary Camport is Connected and Available
         break;
      case CAMPORT_CONNECTED_STATUS_TER_OFF_SEC_ONN_PRI_ONN: //3
         camportConnnectedDevice = ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_ONN_PRI_ONN; //2- Primary,Secondary Camport is Connected and Available
         break;
      case CAMPORT_CONNECTED_STATUS_TER_ONN_SEC_ONN_PRI_ONN: //7
         camportConnnectedDevice = ACCESSIBLE_CAMPORT_DEVICE_TER_ONN_SEC_ONN_PRI_ONN; //3- Primary,Secondary,Teritory Camport is Connected and Available
         break;
      default: //0, 2, 4, 6,
         camportConnnectedDevice = ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_OFF_PRI_OFF; //0-clear list NO-Camport is Connected and Available
         break;
   }

   //number of CMP is 1, and CMP1 connected. Pick 4 textid from CAN array and replace in vector
   if ((numberOfCMPs == TOTAL_ONE_CMP) && (camportConnnectedDevice == ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_OFF_PRI_ONN))
   {
      //replace(4);
      for (index = POS_ZERO; index < POS_FOUR; index++)
      {
         scd.position = vecStCamData.at(index).position;
         scd.camerasource = vecStCamData.at(index).camerasource;
         scd.camTextID = cameraname_arr[index];
         m_vecCamDataToHmi_CANTxt.push_back(scd);
      }
      startPos = POS_FOUR;
   }

   //number of CMP is 2 or 3, and only CMP1 connected. Pick 3 textid from CAN array and replace in vector
   if ((numberOfCMPs > TOTAL_ONE_CMP) && (camportConnnectedDevice == ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_OFF_PRI_ONN))
   {
      //replace(3)
      for (index = POS_ZERO; index < POS_THREE; index++)
      {
         scd.position = vecStCamData.at(index).position;
         scd.camerasource = vecStCamData.at(index).camerasource;
         scd.camTextID = cameraname_arr[index];
         m_vecCamDataToHmi_CANTxt.push_back(scd);
      }
      startPos = POS_THREE;
   }

   //number of CMP is 2 , and both CMP1 & cmp2 connected. Pick 7 textid from CAN array and replace in vector
   if ((numberOfCMPs == TOTAL_TWO_CMP) && (camportConnnectedDevice == ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_ONN_PRI_ONN))
   {
      //replace(7)
      for (index = POS_ZERO; index < POS_SEVEN; index++)
      {
         scd.position = vecStCamData.at(index).position;
         scd.camerasource = vecStCamData.at(index).camerasource;
         scd.camTextID = cameraname_arr[index];
         m_vecCamDataToHmi_CANTxt.push_back(scd);
      }
      startPos = POS_SEVEN;
   }

   //number of CMP is 3, and CMP1, cmp2, CMP3 connected or only cmp1, cmp2 connected. Pick 6 textid from CAN array and replace in vector
   //ADDITIONALLY INSERT 7TH ITEM FROM ARRAY TO VECTOR
   if ((numberOfCMPs == TOTAL_THREE_CMP) &&
         ((camportConnnectedDevice == ACCESSIBLE_CAMPORT_DEVICE_TER_OFF_SEC_ONN_PRI_ONN) ||
          (camportConnnectedDevice == ACCESSIBLE_CAMPORT_DEVICE_TER_ONN_SEC_ONN_PRI_ONN)))
   {
      //replace(6) and manually insert 7
      for (index = POS_ZERO; index < POS_SIX; index++)
      {
         scd.position = vecStCamData.at(index).position;
         scd.camerasource = vecStCamData.at(index).camerasource;
         scd.camTextID = cameraname_arr[index];
         m_vecCamDataToHmi_CANTxt.push_back(scd);
      }

      // Assign Manually CMP2_CMP4 value in 6th position to Vector m_vecCamDataToHmi_CANTxt
      stCameraData scd_ = {2, cameraname_arr[POS_SIX], "CMP2_CAM4"};
      m_vecCamDataToHmi_CANTxt.push_back(scd_);
      //vecStCamData.insert(vecStCamData.begin() + 6, scd);
      startPos = POS_SIX;
   }

   for (uint8 j = startPos; j < vecStCamData.size(); ++j)
   {
      scd.position = vecStCamData.at(j).position;
      scd.camerasource = vecStCamData.at(j).camerasource;
      scd.camTextID = vecStCamData.at(j).camTextID;
      m_vecCamDataToHmi_CANTxt.push_back(scd);
   }

   m_vecCamDataToHmi_CANTxtSignal = m_vecCamDataToHmi_CANTxt;

   CANTextId.clear();
   for (uint8 index = POS_ZERO; index < m_vecCamDataToHmi_CANTxt.size(); index++)
   {
      CANTextId.push_back(m_vecCamDataToHmi_CANTxt.at(index).camTextID);
      ETG_TRACE_USR4(("CameraListHandler::vPrepareCAMTextConfiguration XML DATA After CAN Assignment CMP CAN config P->%d T->%d S->%s", m_vecCamDataToHmi_CANTxt.at(index).position, m_vecCamDataToHmi_CANTxt.at(index).camTextID, m_vecCamDataToHmi_CANTxt.at(index).camerasource.c_str()));
   }
}


// Return vector of all text ids which are available only in Cabin Region
vector<int> CameraListHandler::getAvailableCabinCamTxtIdOnSelection()
{
   vector<int> camTextId;
   if (bTextIDOverwrittenByCAN == TRUE)
   {
      camTextId = getAvailableCabinCamTxtIdsFromCamSignal();
   }
   else
   {
      camTextId = getAvailableCabinCamTxtIds();
   }
   return camTextId;
}


vector<int> CameraListHandler::getAvailableCabinCamTxtIdsFromCamSignal()
{
   for (int i = 0; i < camAvailableCAMTextIdsCabin.size(); i++)
   {
      ETG_TRACE_USR4(("CameraListHandler:getAvailableCabinCamTxtIdsFromCamSignal:[%d]", camAvailableCAMTextIdsCabin.at(i)));
   }
   return camAvailableCAMTextIdsCabin;
}


vector<int> CameraListHandler::getAvailableCabinCamTxtIds()
{
   for (int i = 0; i < camAvailableXMLTextIdsCabin.size(); i++)
   {
      ETG_TRACE_USR4(("CameraListHandler:getAvailableCabinCamTxtIds:[%d]", camAvailableXMLTextIdsCabin.at(i)));
   }
   return camAvailableXMLTextIdsCabin;
}


//RVC CRQ Implementation
bool CameraListHandler::bCheckRearCameraAvailable()
{
   bool bRearCameraAvailable = false;
   vector<int> camTextId;
   camTextId = getAvailableCameraTextIdOnSelection();   // GET the text id from the list of available cameras.

   if (0 < camTextId.size())
   {
      //The First entry in the list belongs to CMP1_CAM1. Hence checking 0th index.
      if (camTextId.at(0) == 35)
      {
         //RVC camera exists. Show warning text
         bRearCameraAvailable = true;
      }
      else
      {
         //May be some other camera. Donot show warning text.
         bRearCameraAvailable = false;
      }
   }
   ETG_TRACE_USR4(("CameraListHandler::bCheckRearCameraAvailable() bRearCameraAvailable:%d <1-RVC Exist 0-RVC Don't Exist>", bRearCameraAvailable));
   return bRearCameraAvailable;
}


bool CameraListHandler::bCameraIconHighlights(int selectedIndex, int cmpValue)
{
   bool selectedItem = false;
   if ((_activeCameraIndex == selectedIndex) && ((cmpValue == 1) || (cmpValue == 2) || (cmpValue == 3)) && (m_bHighlightListItemInCabin == false))
   {
      selectedItem = true;
   }
   else
   {
      selectedItem = false;
   }
   ETG_TRACE_USR3(("CameraListHandler::bCameraIconHighlights selectedIndex==> %d _activeCameraIndex==>%d m_bHighlightListItemInCabin==>%d cmpValue==>%d", selectedIndex, _activeCameraIndex, m_bHighlightListItemInCabin, cmpValue));
   return selectedItem;
}


int CameraListHandler::bCheckRequestedCabinCameraExists(std::string cameraSrcName)
{
   ETG_TRACE_USR4(("CameraListHandler::bCheckRequestedCabinCameraExists %s", cameraSrcName.c_str()));
   int index = -1;
   if (camAvailableSourceNamesCabin.size() != 0)
   {
      std::vector <std::string>::iterator itr = std::find(camAvailableSourceNamesCabin.begin(), camAvailableSourceNamesCabin.end(), cameraSrcName);
      if (itr != camAvailableSourceNamesCabin.end())
      {
         index = itr - camAvailableSourceNamesCabin.begin();
      }
      else
      {
         index = -1;
      }
   }
   ETG_TRACE_USR4(("CameraListHandler::bCheckRequestedCabinCameraExists bCameraSrcAvail at %d", index));
   return index;
}


void CameraListHandler::setCameraStreamStatusCockpit(int listindex, bool status)
{
   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();
   for (int index = 0; index < camHexCamData.size(); index++)
   {
      if (index == listindex)
      {
         vecStCamData.at(index).streamStatus = status;
         break;
      }
   }
   ETG_TRACE_USR4(("CameraMain::setCameraStreamStatusCockpit at %d is %d", listindex, status));
}


void CameraListHandler::setCameraStreamStatusCabin(int listindex, bool status)
{
   vector<stCameraData> vecStCamData = CameraMain::poGetInstance()->getCamDataStructure();
   for (int index = 0; index < camHexCamDataCabin.size(); index++)
   {
      if (index == listindex)
      {
         vecStCamData.at(index).streamStatusCabin = status;
         break;
      }
   }
   ETG_TRACE_USR4(("CameraMain::setCameraStreamStatusCabin at %d is %d", listindex, status));
}


void CameraListHandler::updateHMIAvailableCamerasFromCAN_TTFIs(uint32 cameraname_arr[], bool camTxtStatusReceived)
{
   ETG_TRACE_USR4(("CameraListHandler::updateHMIAvailableCamerasFromCAN"));
   std::memset(m_cameraname_arr, 0, sizeof(m_cameraname_arr));
   for (int i = 0; i < 8; i++)
   {
      m_cameraname_arr[i] = cameraname_arr[i];
   }
   m_camtxtStatusReceived = camTxtStatusReceived;
   vPrepareCAMTextConfiguration(cameraname_arr);
   vListUpdateByCamTxtSignal();
}


void CameraListHandler::setNoHighlightInList(bool value)
{
   m_bHighlightListItemInCabin = value;
}


//This method is only called from the restore source function. The previousCabin index is
//to be set, since this is required to send deactivation when list selection is done.
void CameraListHandler::setprevCabinSelectedIndex(int index)
{
   m_prevCabinSelectedIndex = index;
}


bool CameraListHandler::onCourierMessage(const CameraBlackScreenStartTimerMsg& /* oMsg */)
{
   ETG_TRACE_COMP(("CameraDataBindingHandler::CameraListHandler::onCourierMessage CameraBlackScreenStartTimerMsg"));
   CameraMain::poGetInstance()->startCameraBlackScreenTimer();
   return true; //CID_R1 16580823
   // m_poCameraDataBindingHandler ->RequestCameraBlackScreen();
   //DISABLE_CAM_ENABLE_BLACK_IMG 1
//#define ENABLE_CAM_DISABLE_BLACK_IMG 0
}


void CameraListHandler::vStateMachineExternalTriggerToCameraOrBlackScreen()
{
   ETG_TRACE_COMP(("void CameraListHandler::vStateMachineExternalTriggerToCameraOrBlackScreen() m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));
   if (m_isBlackScreenEnabledValue == true)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ActivateBlackScreenSplitScreenMsg)())); // Trigger Via External (CAN/Dig PIN) --> Go to the Split Camera (BlackScreen)
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(ActivateCameraSplitStreamMsg)())); // Trigger Via List --> Go to Split Camera (No BlackScreen)
   }
}


void CameraListHandler::vStateMachineTriggerToCameraOrBlackScreen()
{
   ETG_TRACE_COMP(("void CameraListHandler::vStateMachineTriggerToCameraOrBlackScreen() m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));
   if (m_isBlackScreenEnabledValue == true)
   {
      POST_MSG((COURIER_MESSAGE_NEW(CameraViewBlackScreenTransistionMsg)())); // Trigger Via List --> Go to Normal Camera -  (BlackScreen)
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(CameraViewMsg)())); // Trigger Via List --> Go to Normal Camera (No BlackScreen)
   }
}


// vContextSwitchRequestSplitOrBlackScreen--> Context switch to 259(Black screen of split camera screen) [or] 256 (directly to split screen)
void CameraListHandler::vContextSwitchRequestSplitOrBlackScreen()
{
   ETG_TRACE_COMP(("void CameraListHandler::vContextSwitchRequestSplitOrBlackScreen() m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));
   if (m_isBlackScreenEnabledValue == true)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)259);
   }
   else
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)256);
   }
}


// vContextSwitchRequestCameraOrBlackScreen --> Context switch to 258(Black screen of normal camera screen) [or] 251 (directly to normal camera screen)
void CameraListHandler::vContextSwitchRequestCameraOrBlackScreen()
{
   ETG_TRACE_COMP(("void CameraListHandler::vContextSwitchRequestCameraOrBlackScreen() m_isBlackScreenEnabledValue==>%d", m_isBlackScreenEnabledValue));
   if (m_isBlackScreenEnabledValue == true)
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)258);
   }
   else
   {
      ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(2, 0, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)0, APPID_APPHMI_CAMERA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)251);
   }
}


} //namespace Core
} //namespace App
