/*
 * Copyright (C) 2011 Benjamin Franzke
 * Copyright (C) 2010 Intel Corporation
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation, and
 * that the name of the copyright holders not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  The copyright holders make no representations
 * about the suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */
#include "Renderer.h"

#include <unistd.h>
#include <sys/mman.h>
#include "assert.h"

#if defined (_LINUXX86_64_)

typedef enum
{
   ADAPTER_CONFIGURATION_ALL = 0xffff
} compositor_shim_configuration_mask;

struct compositor_shim_context
{
};


struct compositor_shim_surface_context
{
};


int compositor_shim_surface_init(struct compositor_shim_surface_context* surf_ctx,
                                 struct wl_surface* wlSurface, uint32_t layer_id,
                                 uint32_t surface_id, uint32_t width, uint32_t height);
int compositor_shim_surface_configure(struct compositor_shim_context* ctx,
                                      struct compositor_shim_surface_context* surf_ctx, uint32_t mask);
int compositor_shim_surface_destroy(struct compositor_shim_surface_context* surf_ctx);
int compositor_shim_terminate(struct compositor_shim_context* ctx);
struct compositor_shim_context* compositor_shim_initialize(struct wl_display* wlDisplay);

#else
#include <compositor-shim.h>
#endif

#include <wayland-client.h>
#include "os-compatibility.h"

// FreeType stuff
#include <ft2build.h>
#include FT_FREETYPE_H

#include <vector>
#include <string.h>

#define NUM_BUFFERS 10
#define WIND_WIDTH 800
#define WIND_HEIGHT 480

#define PADDING_VALUE 20


namespace hmibase {

namespace alertdaemon {

std::vector<std::string> _myTexts;

unsigned int layerid = 71; // build-in layer for HMIBase internal usage
unsigned int surfaceid = 9;

/*******************************************************************************/
// BEGIN: Global variables
/*******************************************************************************/
FT_Library    _library;
FT_Face       _face;
FT_GlyphSlot  _slot;
FT_Error      _error;

#define WIDTH   WIND_WIDTH
#define HEIGHT  WIND_HEIGHT

unsigned char _fontImage[HEIGHT][WIDTH];


/*******************************************************************************/
// BEGIN: Freetype adaptions
/*******************************************************************************/
bool initFreeType()
{
   const char* filename = "/opt/bosch/hmibase/BoschJevaSans.ttf";

   FILE* fp = fopen(filename, "r");
   if (!fp)
   {
      return false;
   }
   else
   {
      fclose(fp);
   }

   _error = FT_Init_FreeType(&_library);                  /* initialize library */
   /* error handling omitted */

   _error = FT_New_Face(_library, filename, 0, &_face);   /* create face object */
   /* error handling omitted */

   /* use 12pt at 100dpi */
   _error = FT_Set_Char_Size(_face, 12 * 64, 0, 100, 0);   /* set character size */
   /* error handling omitted */

   _slot = _face->glyph;

   return true;
}


void deinitFreeType()
{
   FT_Done_Face(_face);
   FT_Done_FreeType(_library);
}


void draw_bitmap(const FT_Bitmap* bitmap, FT_Int x, FT_Int y)
{
   FT_Int  i, j, p, q;
   FT_Int  x_max = x + bitmap->width;
   FT_Int  y_max = y + bitmap->rows;

   for (i = x, p = 0; i < x_max; i++, p++)
   {
      for (j = y, q = 0; j < y_max; j++, q++)
      {
         if (i < 0      || j < 0       ||
               i >= WIDTH || j >= HEIGHT)
         {
            continue;
         }

         _fontImage[j][i] |= bitmap->buffer[q * bitmap->width + p];
      }
   }
}


void prepareTextImage()
{
   FT_Vector     pen;                    /* untransformed origin  */
   int           target_height = HEIGHT;

   /* the pen position in 26.6 cartesian space coordinates; */
   /* start at (30,60) relative to the upper left corner  */
   unsigned int xPos = PADDING_VALUE + 10;
   unsigned int yPos = PADDING_VALUE + 40;
   pen.x = static_cast<FT_Pos>(xPos * 64);
   pen.y = static_cast<FT_Pos>((static_cast<unsigned int>(target_height) - yPos) * 64);

   memset(_fontImage, 0, WIDTH * HEIGHT);

   for (size_t i = 0; i < _myTexts.size(); ++i)
   {
      if (_myTexts[i].empty())
      {
         continue;
      }

      const char* text = _myTexts[i].c_str();

      unsigned int pt = (i == 0) ? 25 : 16;
      /* use 12pt at 100dpi */
      FT_Set_Char_Size(_face, static_cast<FT_F26Dot6>(pt * 64), 0, 100, 0);   /* set character size */

      size_t len = strlen(text);
      for (size_t n = 0; n < len; n++)
      {
         if (text[n] == '\n')
         {
            pen.x = static_cast<FT_Pos>(xPos * 64);
            pen.y =  pen.y - _face->height;
            continue;
         }

         if (pen.x >= (WIDTH - PADDING_VALUE) * 64)
         {
            pen.x = static_cast<FT_Pos>(xPos * 64);
            pen.y =  pen.y - _face->height;//_slot->metrics.height;
         }

         if (text[n] == '\t')
         {
            pen.x += _slot->advance.x;
            pen.y += _slot->advance.y;
            continue;
         }

         /* set transformation */
         FT_Set_Transform(_face, 0, &pen);

         /* load glyph image into the slot (erase previous one) */
         _error = FT_Load_Char(_face, text[n], FT_LOAD_RENDER);
         if (_error)
         {
            continue;   /* ignore errors */
         }

         /* chek if  */
         if (((pen.x + _slot->advance.x) / 64) > (WIDTH - PADDING_VALUE))
         {
            pen.x = static_cast<FT_Pos>(xPos * 64);
            pen.y =  pen.y - _face->height;//_slot->metrics.height;_

            /* set transformation */
            FT_Set_Transform(_face, 0, &pen);

            /* load glyph image into the slot (erase previous one) */
            _error = FT_Load_Char(_face, text[n], FT_LOAD_RENDER);
            if (_error)
            {
               continue;
            }                 /* ignore errors */;
         }

         /* now, draw to our target surface (convert position) */
         draw_bitmap(&_slot->bitmap,
                     _slot->bitmap_left,
                     target_height - _slot->bitmap_top);

         /* increment pen position */
         pen.x += _slot->advance.x;
         pen.y += _slot->advance.y;
      }

      pen.x = static_cast<FT_Pos>(xPos * 64);
      pen.y =  pen.y - _face->height;//_slot->metrics.height;
   }
}


struct display
{
   struct wl_display* display;
   struct wl_registry* registry;
   struct wl_compositor* compositor;
   struct wl_shm* shm;
   struct serverinfo* wl_ext_serverinfo;
   unsigned int wl_connect_id;
   int c8_fmt_supported;
   uint32_t formats;
   struct compositor_shim_context* wlAdapterContext;
};


struct buffer
{
   struct wl_buffer* buffer;
   void* shm_data;
   int busy;
};


struct window
{
   struct display* display;
   int width, height;
   struct wl_surface* surface;
   struct buffer buffers[NUM_BUFFERS];
   struct buffer* prev_buffer;
   struct wl_callback* callback;

   unsigned int surface_id;
   unsigned int layer_id;
   struct compositor_shim_surface_context wlAdapterSurfaceContext;
};


static void
buffer_release(void* data, struct wl_buffer* buf)
{
   struct buffer* mybuf = static_cast<struct buffer*>(data);
   UNUSED(buf);
   mybuf->busy = 0;
}


static const struct wl_buffer_listener buffer_listener =
{
   buffer_release
};


static int
create_shm_buffer(const struct display* disp, struct buffer* buf,
                  int width, int height, uint32_t format)
{
   struct wl_shm_pool* pool;
   int fd, size, stride;
   void* data;

   stride = width * 4;
   size = stride * height;

   fd = os_create_anonymous_file(static_cast<off_t>(size));
   if (fd < 0)
   {
      fprintf(stderr, "creating a buffer file for %d B failed: %m\n",                                                   //lint !e557; The ‘%m’ conversion is a GNU C Library extension.; compatibly with other compiler: fprintf(stderr, "creating a buffer file for %d B failed: %s", size, strerror (errno));
              size);
      return -1;
   }

   /*pointer to the buffer memory which will be displayed*/
   data = mmap(NULL, static_cast<size_t>(size), PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
   if (data == MAP_FAILED)
   {
      fprintf(stderr, "mmap failed: %m\n");                                                                             //lint !e557; The ‘%m’ conversion is a GNU C Library extension.; compatibly with other compiler: fprintf(stderr, "mmap failed: %s\n", strerror (errno));
      close(fd);
      return -1;
   }

   pool = wl_shm_create_pool(disp->shm, fd, size);
   buf->buffer = wl_shm_pool_create_buffer(pool, 0,
                                           width, height,
                                           stride, format);
   wl_buffer_add_listener(buf->buffer, &buffer_listener, buf);
   wl_shm_pool_destroy(pool);
   close(fd);

   buf->shm_data = data;

   return 0;
}


static struct window*
create_window(struct display* disp, int width, int height)
{
   struct window* wdw;

   wdw = static_cast<struct window*>(calloc(1, sizeof * wdw));
   if (!wdw)
   {
      return NULL;
   }

   wdw->callback = NULL;
   wdw->prev_buffer = NULL;
   wdw->display = disp;
   wdw->width = width;
   wdw->height = height;
   wdw->surface = wl_compositor_create_surface(disp->compositor);

   wdw->layer_id = 0xFFFFFFFF;
   wdw->surface_id = 0xFFFFFFFF;

   return wdw;
}


static void
destroy_window(struct window* wdw)
{
   if (wdw->callback)
   {
      wl_callback_destroy(wdw->callback);
   }

   for (unsigned int i = 0; i < NUM_BUFFERS; i++)
   {
      if (wdw->buffers[i].buffer)
      {
         wl_buffer_destroy(wdw->buffers[i].buffer);
      }
   }
   wl_surface_destroy(wdw->surface);
   free(wdw);
}


static struct buffer*
window_next_buffer(struct window* wdw)
{
   struct buffer* buf = NULL;
   int ret = 0;
   int bpp = 4;  // for WL_SHM_FORMAT_ARGB8888

   for (unsigned int cur_buf = 0; cur_buf < NUM_BUFFERS; cur_buf++)
   {
      if (!wdw->buffers[cur_buf].busy)
      {
         buf = &wdw->buffers[cur_buf];
         break;
      }
   }
   if (buf == NULL)
   {
      /*all buffers are busy, no free buffer available*/
      return NULL;
   }

   if (!buf->buffer)
   {
      /*here the memory is actually allocated
       * only ones, in the first run*/
      ret = create_shm_buffer(wdw->display, buf,
                              wdw->width, wdw->height,
                              WL_SHM_FORMAT_ARGB8888);

      if (ret < 0)
      {
         return NULL;
      }

      /* paint the padding */
      memset(buf->shm_data, 0xff,
             static_cast<size_t>(wdw->width * wdw->height * bpp));
   }

   return buf;
}


static void
paint_pixels(void* image, int padding, int width, int height, uint32_t redrawtime)
{
   UNUSED(redrawtime);

   const int halfh = padding + (height - padding * 2) / 2;
   const int halfw = padding + (width  - padding * 2) / 2;
   int inner, outer;
   uint32_t* pixel = static_cast<uint32_t*>(image);
   int y;

   prepareTextImage();

   /* squared radii thresholds */
   outer = (halfw < halfh ? halfw : halfh) - 8;
   inner = outer - 32;
   outer *= outer;
   inner *= inner;

   pixel += padding * width;
   for (y = padding; y < height - padding; y++)
   {
      int x;
      //int y2 = (y - halfh) * (y - halfh);

      pixel += padding;
      for (x = padding; x < width - padding; x++)
      {
//         uint32_t v;
//
//         /* squared distance from center */
//         int r2 = (x - halfw) * (x - halfw) + y2;
//
//         if (r2 < ir)
//            v = (r2 / 32 + time / 64) * 0x0080401;
//         else if (r2 < or)
//            v = (y + time / 32) * 0x0080401;
//         else
//            v = (x + time / 16) * 0x0080401;
//         v &= 0x00ffffff;
//
//         /* cross if compositor uses X from XRGB as alpha */
//         if (abs(x - y) > 6 && abs(x + y - height) > 6)
//            v |= 0xff000000;
//
//         *pixel++ = v;

         if (_fontImage[y][x] != 0)
         {
            unsigned int color = 0x00ff0000; // red
            unsigned int alpha = (unsigned)_fontImage[y][x] << 24;
            *pixel++ = color | alpha;
         }
         else
         {
            *pixel++ = 0xb0806050;
         }
      }

      pixel += padding;
   }
}


/**
 * Use this dummy function to be called on every frame
 */
static void redraw_dummy(void* data, struct wl_callback* callback, uint32_t redrawtime)
{
   UNUSED(data);
   UNUSED(callback);
   UNUSED(redrawtime);
}


static const struct wl_callback_listener frame_listener =
{
   redraw_dummy
};


/**
 * Use this function to update screen content (not on frame rate base)
 */
static void redraw(void* data, struct wl_callback* callback, uint32_t redrawtime)
{
   struct window* wdw = static_cast<struct window*>(data);
   struct buffer* buf;
   buf = window_next_buffer(wdw);
   if (!buf)
   {
      fprintf(stderr, "%s",
              !callback ? "Failed to create the first buffer.\n" :
              "Both buffers busy at redraw(). Server bug?\n");
      return;
   }
   paint_pixels(buf->shm_data, PADDING_VALUE, wdw->width, wdw->height, redrawtime);

   wl_surface_attach(wdw->surface, buf->buffer, 0, 0);
   wl_surface_damage(wdw->surface,
                     100, 100, wdw->width - 40, wdw->height - 40);

   if (callback)
   {
      wl_callback_destroy(callback);
   }

   wdw->callback = wl_surface_frame(wdw->surface);
   wl_callback_add_listener(wdw->callback, &frame_listener, wdw);

   wl_surface_commit(wdw->surface);
   buf->busy = 1;
}


static void
shm_format(void* data, struct wl_shm* shm, uint32_t format)
{
   struct display* d = static_cast<struct display*>(data);
   UNUSED(shm);
   if (format == WL_SHM_FORMAT_C8)
   {
      d->c8_fmt_supported = 1;
   }
   else
   {
      d->formats |= (1u << format);
   }
}


struct wl_shm_listener shm_listenter =
{
   shm_format
};


static void
registry_handle_global(void* data, struct wl_registry* registry,
                       uint32_t id, const char* interface, uint32_t version)
{
   struct display* d = static_cast<display*>(data);
   UNUSED(version);

   if (strcmp(interface, "wl_compositor") == 0)
   {
      d->compositor = static_cast<struct wl_compositor*>(wl_registry_bind(registry, id, &wl_compositor_interface, 1));
   }
   else if (strcmp(interface, "wl_shm") == 0)
   {
      d->shm = static_cast<struct wl_shm*>(wl_registry_bind(registry, id, &wl_shm_interface, 1));
      wl_shm_add_listener(d->shm, &shm_listenter, d);
   }
}


static void
registry_handle_global_remove(void* data, struct wl_registry* registry,
                              uint32_t name)
{
   UNUSED(data);
   UNUSED(registry);
   UNUSED(name);
}


static const struct wl_registry_listener registry_listener =
{
   registry_handle_global,
   registry_handle_global_remove
};


static struct display*
create_display(void)
{
   struct display* disp;

   disp = static_cast<struct display*>(malloc(sizeof * disp));

   if (disp == 0)
   {
      return 0;
   }

   disp->display = wl_display_connect(NULL);
   assert(disp->display);

   disp->formats = 0;
   disp->registry = wl_display_get_registry(disp->display);
   wl_registry_add_listener(disp->registry,
                            &registry_listener, disp);
   wl_display_roundtrip(disp->display);
   if (disp->shm == NULL)
   {
      fprintf(stderr, "No wl_shm global\n");
      exit(1);
   }

   wl_display_roundtrip(disp->display);

   if (!(disp->formats & (1 << static_cast<unsigned int>(WL_SHM_FORMAT_XRGB8888))))
   {
      fprintf(stderr, "WL_SHM_FORMAT_XRGB8888 not available\n");
      exit(1);
   }

   disp->wlAdapterContext = compositor_shim_initialize(disp->display);
   if (!disp->wlAdapterContext)
   {
      fprintf(stderr, "cannot init compositor_shim 3\n");
      exit(1);
   }

   return disp;
}


static void
destroy_display(struct display* disp)
{
   if (disp->shm)
   {
      wl_shm_destroy(disp->shm);
   }

   if (disp->compositor)
   {
      wl_compositor_destroy(disp->compositor);
   }

   wl_registry_destroy(disp->registry);
   wl_display_flush(disp->display);
   wl_display_disconnect(disp->display);
   free(disp);
}


static void init_lm_client(struct window* w, uint layer_id, uint surface_id)
{
   if (NULL == getenv("XDG_RUNTIME_DIR"))
   {
      setenv("XDG_RUNTIME_DIR", "/tmp/", 1);
   }

//   printf("%s(%d): CREATE ON LAYERID: %u ,SURFACEID :%u\n",
//          __FUNCTION__, __LINE__,
//          layer_id,
//          surface_id);
   w->layer_id = layer_id;
   w->surface_id = surface_id;

   if (compositor_shim_surface_init(&w->wlAdapterSurfaceContext, w->surface,
                                    w->layer_id, w->surface_id, w->width, w->height) < 0)
   {
      fprintf(stderr, "compositor_shim_surface_init failed\n");
   }

   if (compositor_shim_surface_configure(w->display->wlAdapterContext, &w->wlAdapterSurfaceContext,
                                         ADAPTER_CONFIGURATION_ALL) < 0)
   {
      fprintf(stderr, "compositor_shim_surface_configure failed\n");
   }
}


static void deinit_lm_client(const struct window* w)
{
   wl_display_flush(w->display->display);
   wl_display_roundtrip(w->display->display);

   compositor_shim_surface_destroy(const_cast<compositor_shim_surface_context*>(&w->wlAdapterSurfaceContext));

   compositor_shim_terminate(w->display->wlAdapterContext);
}


IHMIAlert_Daemon_Plugin& Renderer::Create()
{
   static Renderer sInstance;
   return sInstance;
}


Renderer::Renderer() : _display(0), _window(0)
{
}


Renderer::~Renderer()
{
   Reset();
}


bool Renderer::Init()
{
   bool ret = true;
   _display = create_display();

   if (_display == 0)
   {
      fprintf(stderr, "%s(%d):display could not be created\n", __FUNCTION__, __LINE__);
      return false;
   }

   _window = create_window(_display, WIND_WIDTH, WIND_HEIGHT);
   if (!_window)
   {
      fprintf(stderr, "%s(%d):window could not be created\n", __FUNCTION__, __LINE__);
      return false;
   }
   init_lm_client(_window, layerid, surfaceid);

   /* Initialize damage to full surface, so the padding gets painted */
   wl_surface_damage(_window->surface, 0, 0, _window->width, _window->height);

   if (initFreeType() == false)
   {
      fprintf(stderr, "%s(%d):initFreeType() failed to open Font file\n", __FUNCTION__, __LINE__);
      ret = false;
   }

   return ret;
}


void Renderer::Reset()
{
   deinitFreeType();
   deinit_lm_client(_window);
   destroy_window(_window);
   destroy_display(_display);
}


bool Renderer::update(const std::vector<std::string>& texts)
{
   _myTexts = texts;
   _myTexts.insert(_myTexts.begin(), "!!! HMI Alert !!!\n");
   _myTexts.resize(10);

   redraw(_window, NULL, 0);

   if (wl_display_dispatch(_display->display) == -1)
   {
      fprintf(stderr, "wl_display_dispatch %p failed\n", _display->display);
      return false;
   }
   return true;
}


// ========================================================================
// Global registration for plugin
// ========================================================================
// ------------------------------------------------------------------------
class RendererProxy
{
   public:
      RendererProxy()
      {
         // Register the plugin instance creation at the global factory map
         gHMIAlertDaemonPluginFactoryMap[IRendererPlugin::Name()] = Renderer::Create;
      }
};


// Create one instance of the proxy to perform registration
RendererProxy gRendererPlugin;
}


}
