/* ***************************************************************************************
* FILE:          HMIAlert_Daemon.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIAlert_Daemon.cpp is part of HMI-Base delivery
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hmi_alert_defines.h"

// SystemD
#include <systemd/sd-daemon.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <vector>
#include <string>

#include <dlfcn.h>

// socket stuff
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

// Plugin stuff
#include <map>

#include "IHMIAlert_Daemon_Plugin.h"
#include "Plugin/IRenderer.h"

namespace hmibase {
namespace alertdaemon {

HMIAlertDaemonPluginFactoryMap gHMIAlertDaemonPluginFactoryMap;

class PluginManager
{
   public:

      PluginManager()
      {
      }

      ~PluginManager()
      {
         UnloadPlugin();
      }

      int LoadAndBind()
      {
         if (_sharedLibrary.handle != 0)
         {
            // is already loaded
            return 0;
         }

         _sharedLibrary.name = "/opt/bosch/hmibase/lib/libhmibase_alertdaemon_rendererplugin_so.so";

         // load plugin
         _sharedLibrary.handle = dlopen(_sharedLibrary.name, RTLD_NOW);

         if (_sharedLibrary.handle == 0)
         {
            fprintf(stderr, "Error loading plugin %40s, dlerror %s\n", _sharedLibrary.name, dlerror());
            return -1;
         }

         // bind plugin
         _plugins[PluginId::Renderer] = LoadPlugin<IRendererPlugin>();
         if (0 != _plugins[PluginId::Renderer])
         {
            if (_plugins[PluginId::Renderer]->Init() == false)
            {
               fprintf(stderr, "Initializing renderer plugin failed!");
               return -1;
            }
         }

         return 0;
      }

      // ------------------------------------------------------------------------
      IHMIAlert_Daemon_Plugin* GetPlugin(PluginId::Enum pluginId)
      {
         PluginMap::iterator it = _plugins.find(pluginId);
         return (_plugins.end().operator != (it)) ? (*it).second : 0;
      }

   private:

      void UnloadPlugin()
      {
         // Unload plugins in reverse add order
         for (PluginMap::reverse_iterator rit = _plugins.rbegin();
               rit != _plugins.rend();
               ++rit)
         {
            if (0 != (*rit).second)
            {
               (*rit).second->Reset();
            }
         }

         // Close shared libraries
         if (_sharedLibrary.handle != 0)
         {
            //      fprintf(stdout, "Unloading shared library %40s\n", lContext.name);
            dlclose(_sharedLibrary.handle);
         }
      }

      // ------------------------------------------------------------------------
      template <typename T> T* LoadPlugin()
      {
         // Plugin creation (if plugin exists)
         IHMIAlert_Daemon_Plugin::Create* lPluginFactory = gHMIAlertDaemonPluginFactoryMap[T::Name()];
         if (0 != lPluginFactory)
         {
            T& lPlugin = static_cast<T&>(lPluginFactory());
            //      fprintf(stdout, "Created plugin object '%40s'\n", T::Name());
            return &lPlugin;
         }
         else
         {
            fprintf(stderr, "Failed to create plugin object '%40s'\n", T::Name());
            return 0;
         }
      }

      struct SharedLibraryContext
      {
         void* handle;
         const char* name;

         SharedLibraryContext(): handle(0), name(0) {}
      };

      typedef std::map<PluginId::Enum, IHMIAlert_Daemon_Plugin*> PluginMap;

      PluginMap _plugins;
      SharedLibraryContext _sharedLibrary;
};


class SimpleSocket
{
   public:

      SimpleSocket(): _sockfd(0)
      {
         open_internal();
      }

      ~SimpleSocket()
      {
         close_internal();
      }

      void read_internal()
      {
         union
         {
            struct sockaddr_in  intStyle;
            struct sockaddr     sadr;
         } cli_adr;
         int newsockfd;
         socklen_t clilen;
         char buffer[257];

         memset(buffer, 0, sizeof(buffer));
         clilen = sizeof(cli_adr.intStyle);

         newsockfd = accept(_sockfd, &cli_adr.sadr, &clilen);
         if (newsockfd < 0)
         {
            printf("Error on accept\n");
            exit(1);
         }
         ssize_t n = read(newsockfd, buffer, sizeof(buffer) - 1);
         buffer[256] = '\0';
         if (n < 0)
         {
            printf("ERROR reading from socket\n");
            exit(1);
         }
         printf("Here is the message:'%s'\n", buffer);

         //addNewText(buffer);

         if (buffer[0] != 0)
         {
            _myTexts.push_back(buffer);
         }

         close(newsockfd);
      }

      const std::vector<std::string>& getData()
      {
         return _myTexts;
      }

   private:
      void open_internal()
      {
         union
         {
            struct sockaddr_in  intStyle;
            struct sockaddr     sadr;
         } r_adr;

         _sockfd = socket(AF_INET, static_cast<int>(SOCK_STREAM), 0);
         if (_sockfd < 0)
         {
            fprintf(stderr, "Error opening socket\n");
            exit(1);
         }
         memset((char*) &r_adr.sadr, 0, sizeof(r_adr.intStyle));

         r_adr.intStyle.sin_family      = AF_INET;
         r_adr.intStyle.sin_addr.s_addr = INADDR_ANY;
         r_adr.intStyle.sin_port        = htons(HMI_ALERT_PORT_NO);

         if (bind(_sockfd, &r_adr.sadr, sizeof(r_adr.intStyle)) < 0)
         {
            fprintf(stderr, "Error on binding \n");
            exit(1);
         }

         if (listen(_sockfd, 5) == -1)
         {
            fprintf(stderr, "listen on socket returned with error, errno %d\n", errno);
         }
      }

      void close_internal()
      {
         close(_sockfd);
      }

      int _sockfd;
      std::vector<std::string> _myTexts;

   private: // prevents copy construction and assignment
      SimpleSocket(const SimpleSocket& other);
      SimpleSocket& operator =(const SimpleSocket& other);
};


} // alertdaemon
} // hmibase


/**********************************************************************************/

int main(int /*argc*/, char** /*argv*/)
{
   int ret = 0;

#ifdef VARIANT_S_FTR_ENABLE_HMI_ALERT
   hmibase::alertdaemon::SimpleSocket sock;
   hmibase::alertdaemon::PluginManager pluginManager;
#endif

   sd_notifyf(0, "READY=1\n"  // Tells the init system that daemon startup is finished.
              // This is only used by systemd if the service definition file has Type=notify set.
              "STATUS=Processing requests...\n" // Passes a single-line status string back to the init system that describes the daemon state.
              // This is free-form and can be used for various purposes
              "MAINPID=%lu\nPARENTID=%lu",
              (unsigned long) getpid(),
              (unsigned long) getppid());

#ifdef VARIANT_S_FTR_ENABLE_HMI_ALERT
   if (getenv("LAYZ_PLUGIN_LOADING") != 0)
   {
      // initial read before loading plugin
      sock.read_internal();
   }
   else
   {
      // sleep before loading plugins, to prevent concurrency with screenbroker
      sleep(5);
   }
#endif

   for (;/*ever*/;) // start listening
   {
#ifdef VARIANT_S_FTR_ENABLE_HMI_ALERT
      if (pluginManager.LoadAndBind() != 0)
      {
         fprintf(stderr, "Error on Load or bind renderer update failed\n");
         ret = -1;
         break;
      }

      hmibase::alertdaemon::IRendererPlugin* renderer = static_cast<hmibase::alertdaemon::IRendererPlugin*>(pluginManager.GetPlugin(hmibase::alertdaemon::IRendererPlugin::Id()));
      if (renderer != 0)
      {
         if (renderer->update(sock.getData()) == false)
         {
            fprintf(stderr, "Executing renderer update failed\n");
            ret = -1;
            break;
         }
      }
      else
      {
         fprintf(stderr, "No renderer \n");
         ret = -1;
         break;
      }

      // read data from socket
      sock.read_internal();

#else
      sleep(1);
#endif
   }

   return ret;
}                                                                                                                       //lint !e818; standard function declaration for main(...)


/* EOF */
