/* ***************************************************************************************
* FILE:          WidgetFunctors.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetFunctors.h is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#if !defined(WidgetFunctors_h)
#define WidgetFunctors_h


namespace hmibase {
namespace widget {
class Widget;

namespace utils {

/*****************************************************************************/
class CompareUtils
{
   public:
      template <typename TNode>
      static bool areAncestorAndDescendent(const TNode* ancestor, const TNode* descendent)
      {
         while ((ancestor != NULL) && (descendent != NULL))
         {
            descendent = descendent->GetParent();
            if (ancestor == descendent)
            {
               return true;
            }
         }
         return false;
      }

      template <typename TNode>
      static bool areSameOrAncestorAndDescendent(const TNode* ancestor, const TNode* descendent)
      {
         if (ancestor == descendent)
         {
            return true;
         }
         while ((ancestor != NULL) && (descendent != NULL))
         {
            descendent = descendent->GetParent();
            if (ancestor == descendent)
            {
               return true;
            }
         }
         return false;
      }
};


/*****************************************************************************/
/* Encapsulates 4 ids.                                                       */
/*****************************************************************************/
template <typename TId>
struct ComplexId4
{
   typedef TId IdType;

   ComplexId4(TId id1 = TId(), TId id2 = TId(), TId id3 = TId(), TId id4 = TId())
      : Id1(id1), Id2(id2), Id3(id3), Id4(id4)
   {
   }

   bool operator== (const ComplexId4& other) const
   {
      return (Id1 == other.Id1) && (Id2 == other.Id2) && (Id3 == other.Id3) && (Id4 == other.Id4);
   }

   bool operator!= (const ComplexId4& other) const
   {
      return !operator==(other);
   }

   TId Id1;
   TId Id2;
   TId Id3;
   TId Id4;
};


typedef ComplexId4<unsigned int> ComplexId4UI;
typedef ComplexId4<float> ComplexId4F;

namespace functors {


/*****************************************************************************/
template <typename TWidget>
class MatchesId
{
   public:
      MatchesId(const Courier::Identifier& id) : _id(id) {}

      bool operator()(TWidget* widget)
      {
         return (widget != NULL) && (_id == Courier::Identifier(widget->GetLegacyName()));
      }

   private:
      Courier::Identifier _id;
};


/*****************************************************************************/
/* Filter which checks complex identifiers.
 * Depending on how it is initialized this filter will check 1, 2, 3 or 4 identifiers. */
/*****************************************************************************/
template <typename TId>
class MatchesComplexId4
{
   public:
      typedef TId IdType;

      MatchesComplexId4(const hmibase::widget::utils::ComplexId4<TId>& id, bool id1Enabled, bool id2Enabled, bool id3Enabled, bool id4Enabled)
         : _id(id), _id1Enabled(id1Enabled), _id2Enabled(id2Enabled), _id3Enabled(id3Enabled), _id4Enabled(id4Enabled)
      {
      }

      /* Creates a default filter.  */
      MatchesComplexId4() : _id1Enabled(false), _id2Enabled(false), _id3Enabled(false), _id4Enabled(false)
      {
      }
      /* Creates a filter which checks the first identifier of a ComplexId. */
      MatchesComplexId4(TId id1) : _id(id1), _id1Enabled(true), _id2Enabled(false), _id3Enabled(false), _id4Enabled(false)
      {
      }
      /* Creates a filter which checks the first and the second identifiers of a ComplexId. */
      MatchesComplexId4(TId id1, TId id2) : _id(id1, id2), _id1Enabled(true), _id2Enabled(true), _id3Enabled(false), _id4Enabled(false)
      {
      }
      /* Creates a filter which checks the first, the second and the third identifiers of a ComplexId. */
      MatchesComplexId4(TId id1, TId id2, TId id3) : _id(id1, id2, id3), _id1Enabled(true), _id2Enabled(true), _id3Enabled(true), _id4Enabled(false)
      {
      }
      /* Creates a filter which checks all four identifiers of a ComplexId. */
      MatchesComplexId4(TId id1, TId id2, TId id3, TId id4) : _id(id1, id2, id3, id4), _id1Enabled(true), _id2Enabled(true), _id3Enabled(true), _id4Enabled(true)
      {
      }

      bool operator()(const hmibase::widget::utils::ComplexId4<TId>& id)
      {
         return (!_id1Enabled || _id.Id1 == id.Id1)
                && (!_id2Enabled || _id.Id2 == id.Id2)
                && (!_id3Enabled || _id.Id3 == id.Id3)
                && (!_id4Enabled || _id.Id4 == id.Id4);
      }

      bool operator== (const MatchesComplexId4<TId>& other) const
      {
         return (_id == other._id) && (_id1Enabled == other._id1Enabled) && (_id2Enabled == other._id2Enabled) && (_id3Enabled == other._id3Enabled) && (_id4Enabled == other._id4Enabled);
      }

      bool operator!= (const MatchesComplexId4<TId>& other) const
      {
         return !operator==(other);
      }

   private:
      hmibase::widget::utils::ComplexId4<TId> _id;
      bool _id1Enabled;
      bool _id2Enabled;
      bool _id3Enabled;
      bool _id4Enabled;
};


/*****************************************************************************/
template <typename TWidget, typename TNode>
class IsAncestorOf
{
   public:
      IsAncestorOf(const TNode* node) : _node(node) {}
      ~IsAncestorOf()
      {
         _node = NULL;
      }

      bool operator()(TWidget* widget)
      {
         return (widget != NULL) && hmibase::widget::utils::CompareUtils::areAncestorAndDescendent(widget->GetNode(), _node);
      }

   private:
      const TNode* _node;
};


/*****************************************************************************/
template <typename TWidget, typename TNode>
class IsSameOrAncestorOf
{
   public:
      IsSameOrAncestorOf(const TNode* node) : _node(node) {}
      ~IsSameOrAncestorOf()
      {
         _node = NULL;
      }

      bool operator()(TWidget* widget)
      {
         return (widget != NULL) && hmibase::widget::utils::CompareUtils::areSameOrAncestorAndDescendent(widget->GetNode(), _node);
      }

   private:
      const TNode* _node;
};


/*****************************************************************************/
template <typename TWidget, typename TNode>
class IsDescendentOf
{
   public:
      IsDescendentOf(const TNode* node) : _node(node) {}
      ~IsDescendentOf()
      {
         _node = NULL;
      }

      bool operator()(TWidget* widget)
      {
         return (widget != NULL) && hmibase::widget::utils::CompareUtils::areAncestorAndDescendent(_node, widget->GetNode());
      }

   private:
      const TNode* _node;
};


/*****************************************************************************/
template <typename TWidget, typename TNode>
class IsSameOrDescendentOf
{
   public:
      IsSameOrDescendentOf(const TNode* node) : _node(node) {}
      ~IsSameOrDescendentOf()
      {
         _node = NULL;
      }

      bool operator()(TWidget* widget)
      {
         return (widget != NULL) && hmibase::widget::utils::CompareUtils::areSameOrAncestorAndDescendent(_node, widget->GetNode());
      }

   private:
      const TNode* _node;
};


/*****************************************************************************/
template <typename TWidget>
class AccessorMatchesWidget
{
   public:
      AccessorMatchesWidget(const hmibase::widget::Widget* widget) : _widget(widget) {}
      ~AccessorMatchesWidget()
      {
         _widget = NULL;
      }

      bool operator()(const hmibase::util::ObjectAccessor<TWidget>& accessor)
      {
         return accessor.getObjectSafely() == _widget;
      }

   private:
      const hmibase::widget::Widget* _widget;
};


}//functors
}//utils
}//widget
}//hmibase

#endif
