/* ***************************************************************************************
* FILE:          IndexRangeSet.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IndexRangeSet is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "IndexRangeSet.h"

#include <algorithm>
#include <bitset>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/IndexRangeSet.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace utils {


/*****************************************************************************/
struct Range
{
   Range(size_t begin = 0, size_t end = 0, Range* next = NULL) : _begin(begin), _end(end), _next(next)
   {
   }

   ~Range()
   {
      if (_next != NULL)
      {
         FEATSTD_DELETE(_next);
         _next = NULL;
      }
   }

   size_t _begin;
   size_t _end;
   Range* _next;
};


/*****************************************************************************/
DynamicIndexRangeSet::DynamicIndexRangeSet() : _set(NULL), _first(NULL)
{
}


/*****************************************************************************/
DynamicIndexRangeSet::~DynamicIndexRangeSet()
{
   if (_set != NULL)
   {
      FEATSTD_DELETE(_set);
      _set = NULL;
   }
   if (_first != NULL)
   {
      FEATSTD_DELETE(_first);
      _first = NULL;
   }
}


/*****************************************************************************/
bool DynamicIndexRangeSet::contains(size_t index) const
{
   for (Range* r = _first; r != NULL; r = r->_next)
   {
      if (r->_begin <= index && index <= r->_end)
      {
         return true;
      }
   }
   return _set != NULL ? (_set->find(index) != _set->end()) : false;
}


/*****************************************************************************/
void DynamicIndexRangeSet::clear()
{
   if (_set != NULL)
   {
      FEATSTD_DELETE(_set);
      _set = NULL;
   }
   if (_first != NULL)
   {
      FEATSTD_DELETE(_first);
      _first = NULL;
   }
}


/*****************************************************************************/
void DynamicIndexRangeSet::add(size_t index)
{
   if (_set == NULL)
   {
      _set = FEATSTD_NEW(std::set<size_t>);
   }
   if (_set != NULL)
   {
      _set->insert(index);
   }
}


/*****************************************************************************/
void DynamicIndexRangeSet::addRange(size_t begin, size_t end)
{
   _first = FEATSTD_NEW(Range)(begin, end, _first);
}


/*****************************************************************************/
bool IndexRangeSetFactory::fillRanges(IndexRangeSet& rangeSet, const char* text)
{
   rangeSet.clear();

   FeatStd::SizeType textLength = FeatStd::Internal::Generic::GenericString::Length(text);

   int numberInProgress = -1;
   int availableNumber = -1;
   int rangeBegin = -1;

   for (FeatStd::SizeType i = 0; i <= textLength; ++i)
   {
      FeatStd::TChar c = text[i];
      if ('0' <= c && c <= '9')
      {
         if (availableNumber != -1)
         {
            ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "Unused number '%d' from '%s'!", availableNumber, text));
            availableNumber = -1;
         }

         if (numberInProgress == -1)
         {
            numberInProgress = 0;
         }
         else
         {
            numberInProgress *= 10;
         }

         numberInProgress += static_cast<int>(c - '0');
      }
      else if (c == ' ')
      {
         if (numberInProgress != -1)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Number '%d' available from '%s'!", numberInProgress, text));
            availableNumber = numberInProgress;
            numberInProgress = -1;
         }
      }
      else if (c == '-' || c == ',' || c == ';' || c == '\0')
      {
         int numberToUse = -1;
         //after a space
         if (availableNumber != -1)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Use available number '%d' from '%s'!", availableNumber, text));
            numberToUse = availableNumber;
            availableNumber = -1;
         }
         else if (numberInProgress != -1)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Use number in progress '%d' from '%s'!", numberInProgress, text));
            numberToUse = numberInProgress;
            numberInProgress = -1;
         }
         else
         {
            ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "Unexpected separator '%c' in '%s'!", c, text));
            continue;
         }

         if (c == '-')
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Found range begin '%d' in '%s'!", numberToUse, text));
            rangeBegin = numberToUse;
         }
         else //if (c == ',' || c == ';')
         {
            if (rangeBegin != -1)
            {
               int rangeEnd = numberToUse;
               ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Found range end '%d' in '%s'!", numberToUse, text));

               if (rangeBegin > rangeEnd)
               {
                  ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "Wrong range '%d' > '%d' in '%s'!", rangeBegin, rangeEnd, text));
                  rangeEnd = rangeBegin;
               }
               const int MAX_RANGE = 1000;
               if (rangeEnd - rangeBegin > MAX_RANGE)
               {
                  ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "Range '%d' exceeds max range '%d' from '%s'!", rangeEnd - rangeBegin, MAX_RANGE, text));
                  rangeEnd = rangeBegin + MAX_RANGE;
               }

               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Insert range '%d' to '%d' from '%s'!", rangeBegin, rangeEnd, text));
               rangeSet.addRange(static_cast<size_t>(rangeBegin), static_cast<size_t>(rangeEnd));

               rangeBegin = -1;
            }
            else
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Insert number '%d' from '%s'!", numberToUse, text));
               rangeSet.add(static_cast<size_t>(numberToUse));
            }
         }
      }
      else
      {
         ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "Unexpected character '%c' in '%s'!", c, text));
         numberInProgress = -1;
         availableNumber = -1;
         rangeBegin = -1;
      }
   }
   return true;
}


}
}


}
