/* ***************************************************************************************
* FILE:          BlurShaderStatic.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurShaderStatic is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "BlurShaderStatic.h"
#include <iomanip>
#include <FeatStd/Util/Guid.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurShaderStatic.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace blur {

using namespace Candera;


BlurShaderStatic::BlurShaderStatic(bool initHorizontal, bool initVertical) :
   m_vertShaderSource(),
   m_fragShaderSource(),
   m_shader(),
   m_shaderParams(),
   m_horizontal(initHorizontal),
   m_vertical(initVertical)
{
   m_guidStringBase = "56032E5B-ABE5-445D-B9E9-109796C9543D";
}


SharedPointerShaderType BlurShaderStatic::GetShader()
{
   if (m_shader.PointsToNull() == true)
   {
      m_shader = Candera::Shader::Create();

      m_vertShaderSource = GenerateVertexShader1D();

      if (m_horizontal && m_vertical)
      {
         m_fragShaderSource = GenerateFragmentShader2D(m_textureWidth, m_textureHeight, m_sigma, m_radius);
      }
      else if (m_horizontal)
      {
         m_fragShaderSource = GenerateFragmentShader1D(true, m_textureWidth, m_sigma, m_radius);
      }
      else
      {
         m_fragShaderSource = GenerateFragmentShader1D(false, m_textureHeight, m_sigma, m_radius);
      }

      if (!m_shader->SetVertexShader(m_vertShaderSource.c_str(), 0))
      {
         ETG_TRACE_ERR(("Setting the Vertex Shader of Static Blur Shader failed!"));
      }

      if (!m_shader->SetFragmentShader(m_fragShaderSource.c_str(), 0))
      {
         ETG_TRACE_ERR(("Setting the Fragment Shader of Static Blur Shader failed!"));
      }
   }

   if ((m_shader->GetVertexShader() == 0) || (m_shader->GetFragmentShader() == 0))
   {
      ETG_TRACE_ERR(("Static Blur Shader: Setting the Shader Source failed! Vertex NULL: %p, Fragment NULL: %p", m_shader->GetVertexShader(), m_shader->GetFragmentShader()));
   }

   m_shader->SetName("BlurShaderStatic");
   UpdateGuidString();

   m_shader->SetGuid(MakeGuid());

   return m_shader;
}


SharedPointerParamSetterType BlurShaderStatic::GetShaderParams()
{
   if (m_shaderParams.PointsToNull() == true)
   {
      m_shaderParams = Candera::GenericShaderParamSetter::Create();
   }

   if (!m_shaderParams->SetUniform(m_uniformDarken.c_str(), Shader::FloatVec4, &m_darken[0], 1))
   {
      ETG_TRACE_ERR(("Setting the Darken uniform failed!"));
   }

   Candera::GenericShaderParamSetter* gsps = static_cast<Candera::GenericShaderParamSetter*>(m_shaderParams.GetPointerToSharedInstance());
   if (NULL != gsps)
   {
      gsps->SetTextureActivationEnabled(true);
   }

   return m_shaderParams;
}


bool BlurShaderStatic::HasValidShader()
{
   return false;
}


bool BlurShaderStatic::HasValidParams()
{
   return false;
}


std::string BlurShaderStatic::GenerateVertexShader1D() const
{
   std::string result = "uniform mat4 u_MVPMatrix;\n";
   result += "attribute vec4 a_Position;\n";
   result += "attribute vec2 a_TextureCoordinate;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "void main(void)\n";
   result += "{\n";
   result += "gl_Position = u_MVPMatrix * a_Position;\n";
   result += m_varyingTexCoord + " = a_TextureCoordinate;\n";
   result += "}\n";

   return result;
}


std::string BlurShaderStatic::GenerateVertexShader2D() const
{
   std::string result;
   result = "uniform mat4 u_MVPMatrix;\n";
   result += "attribute vec4 a_Position;\n";
   result += "attribute vec2 a_TextureCoordinate;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "void main(void)\n";
   result += "{\n";
   result += "gl_Position = u_MVPMatrix * a_Position;\n";
   result += m_varyingTexCoord + " = a_TextureCoordinate;\n";
   result += "}\n";

   return result;
}


std::string BlurShaderStatic::GenerateFragmentShader1D(bool inHorizontal, Candera::Int inTextureDimension, Candera::Double inSigma, Candera::Int inKernelRadius) const
{
   std::string result;
   Candera::Int kernelSize = ((2 * inKernelRadius) + 1);
   KernelRow blurFactors = Produce1dGaussianKernel(inSigma, inKernelRadius);
   TexCoordRow texCoords = Produce1dTexCoords(inHorizontal, inTextureDimension, inKernelRadius);

   result = "precision mediump int;\n";
   result += "precision mediump float;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "uniform sampler2D " + m_uniformTexSampler + ";\n";

   /* Add texture units in case a mask exist.*/
   result += GenerateMaskUnformCode();

   result += "uniform vec4 " + m_uniformDarken + "; \n";
   result += "void main()\n";
   result += "{\n";

   /* Add code in case a mask exist.*/
   result += GenerateMaskDecisionCode();

   result += "lowp vec4 " + m_blurAccumulator + ";\n";
   std::string sCoordString;
   std::string tCoordString;
   std::string gaussFactorString;
   Candera::Char buffer[FLT_DIG + FLT_DIG];

   for (Candera::Int x = 0; x < kernelSize; x++)
   {
      if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, texCoords[x].GetX()))
      {
         ETG_TRACE_ERR(("Writing the texture X-coordinate into char array failed!"));
         sCoordString = "0.0";
      }
      else
      {
         sCoordString = std::string(buffer);
      }

      if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, texCoords[x].GetY()))
      {
         ETG_TRACE_ERR(("Writing the texture Y-coordinate into char array failed!"));
         tCoordString = "0.0";
      }
      else
      {
         tCoordString = std::string(buffer);
      }

      if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, static_cast<Candera::Float>(blurFactors[x])))
      {
         ETG_TRACE_ERR(("Writing the blur factors into char array failed!"));
         gaussFactorString = "0.0";
      }
      else
      {
         gaussFactorString = std::string(buffer);
      }
      result += m_blurAccumulator + " = ";

      if (x > 0)
      {
         result += m_blurAccumulator + " +";
      }
      result += "(texture2D(" + m_uniformTexSampler + ", " + m_varyingTexCoord + " + vec2(" + sCoordString + ", " + tCoordString + ")) * " + gaussFactorString + "); ";
   }
   result += m_blurAccumulator + " = " + m_blurAccumulator + " * " + m_uniformDarken + ";\n";
   std::string fragColorCode = GenerateMaskFragColorCode();

   if (fragColorCode.empty())
   {
      result += "gl_FragColor = " + m_blurAccumulator + ";\n";
   }
   else
   {
      result += fragColorCode;
   }
   result += "}\n";

   return result;
}


std::string BlurShaderStatic::GenerateFragmentShader2D(Candera::Int inTextureDimensionX, Candera::Int inTextureDimensionY, Candera::Double inSigma, Candera::Int inKernelRadius) const
{
   std::string result;
   KernelType blurFactors = Produce2dGaussianKernel(inSigma, inKernelRadius);
   TexCoords2D texCoords = Produce2dTexCoords(inTextureDimensionX, inTextureDimensionY, inKernelRadius);

   result = "precision mediump int;\n";
   result += "precision mediump float;\n";
   result += "varying mediump vec2 " + m_varyingTexCoord + ";\n";
   result += "uniform sampler2D " + m_uniformTexSampler + ";\n";

   /* Add texture units in case a mask exist.*/
   result += GenerateMaskUnformCode();

   result += "uniform vec4 " + m_uniformDarken + "; \n";
   result += "void main()\n";
   result += "{\n";

   /* Add code in case a mask exist.*/
   result += GenerateMaskDecisionCode();

   result += "vec4 " + m_blurAccumulator + " = vec4(0.0);\n";

   std::string sCoordString;
   std::string tCoordString;
   std::string gaussFactorString;
   Candera::Char buffer[FLT_DIG + FLT_DIG];

   for (size_t x = 0; x < blurFactors.size(); x++)
   {
      for (size_t y = 0; y < blurFactors[x].size(); y++)
      {
         if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, texCoords[x][y].GetX()))
         {
            ETG_TRACE_ERR(("Writing the texture X-coordinate into char array failed!"));
            sCoordString = "0.0";
         }
         else
         {
            sCoordString = std::string(buffer);
         }

         if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, texCoords[x][y].GetY()))
         {
            ETG_TRACE_ERR(("Writing the texture Y-coordinate into char array failed!"));
            tCoordString = "0.0";
         }
         else
         {
            tCoordString = std::string(buffer);
         }

         if (0 > Candera::StringPlatform::StringPrintf(buffer, sizeof(buffer), "%.*f", FLT_DIG, static_cast<Candera::Float>(blurFactors[x][y])))
         {
            ETG_TRACE_ERR(("Writing the blur factors into char array failed!"));
            gaussFactorString = "0.0";
         }
         else
         {
            gaussFactorString = std::string(buffer);
         }
         result += m_blurAccumulator + " = " + m_blurAccumulator + " + ";
         result += "(texture2D(" + m_uniformTexSampler + ", " + m_varyingTexCoord + " + vec2(" + sCoordString + ", " + tCoordString + ")) * " + gaussFactorString + "); ";
      }
   }
   result += m_blurAccumulator + " = " + m_blurAccumulator + " * " + m_uniformDarken + ";\n";
   std::string fragColorCode = GenerateMaskFragColorCode();

   if (fragColorCode.empty())
   {
      result += "gl_FragColor = " + m_blurAccumulator + ";\n";
   }
   else
   {
      result += fragColorCode;
   }
   result += "}\n";
   return result;
}


Candera::Double BlurShaderStatic::Gaussian(Candera::Double x, Candera::Double mu, Candera::Double sigma) const
{
   if (sigma < 0.0)
   {
      sigma = 1.0;
   }
   return exp(-(((x - mu) / (sigma)) * ((x - mu) / (sigma))) / 2.0);
}


KernelType BlurShaderStatic::Produce2dGaussianKernel(Candera::Double sigma, Candera::Int kernelRadius) const
{
   size_t kernelSize = static_cast<size_t>(2 * kernelRadius + 1);
   KernelType kernel2d(kernelSize, KernelRow(kernelSize));
   Candera::Double sum = 0.0;

   /* compute values */
   for (size_t row = 0; row < kernel2d.size(); row++)
   {
      for (size_t col = 0; col < kernel2d[row].size(); col++)
      {
         Candera::Double x = Gaussian(static_cast<Candera::Double>(row), static_cast<Candera::Double>(kernelRadius), sigma);
         x *= Gaussian(static_cast<Candera::Double>(col), static_cast<Candera::Double>(kernelRadius), sigma);
         kernel2d[row][col] = x;
         sum += x;
      }
   }

   /* normalize */
   for (size_t row = 0; row < kernel2d.size(); row++)
   {
      for (size_t col = 0; col < kernel2d[row].size(); col++)
      {
         kernel2d[row][col] /= sum;
      }
   }
   return kernel2d;
}


KernelRow BlurShaderStatic::Produce1dGaussianKernel(Candera::Double sigma, Candera::Int kernelRadius) const
{
   size_t kernelSize = static_cast<size_t>(2 * kernelRadius + 1);
   KernelRow kernel1d(kernelSize);
   Candera::Double sum = 0.0;

   /* compute values */
   for (size_t col = 0; col < kernel1d.size(); col++)
   {
      Candera::Double x = Gaussian(static_cast<Candera::Double>(col), static_cast<Candera::Double>(kernelRadius), sigma);

      kernel1d[col] = x;
      sum += x;
   }

   /* normalize */
   for (size_t col = 0; col < kernel1d.size(); col++)
   {
      kernel1d[col] /= sum;
   }
   return kernel1d;
}


TexCoordRow BlurShaderStatic::Produce1dTexCoords(bool inHorizontal, Candera::Int inTextureDimension, Candera::Int kernelRadius) const
{
   size_t kernelSize = static_cast<size_t>(2 * kernelRadius + 1);
   TexCoordRow texCoords1d(kernelSize);
   Candera::Double texCoordStepWidth = 1.0 / static_cast<Candera::Double>(inTextureDimension);
   Candera::Double texCoordStart = -1.0 * texCoordStepWidth * static_cast<Candera::Double>(kernelRadius);

   for (size_t col = 0; col < texCoords1d.size(); col++)
   {
      if (inHorizontal)
      {
         texCoords1d[col] = Candera::Vector2(static_cast<Candera::Float>(texCoordStart + (static_cast<Candera::Double>(col) * texCoordStepWidth)), 0.0F);
      }
      else
      {
         texCoords1d[col] = Candera::Vector2(0.0F, static_cast<Candera::Float>(texCoordStart + (static_cast<Candera::Double>(col) * texCoordStepWidth)));
      }
   }
   return texCoords1d;
}


TexCoords2D BlurShaderStatic::Produce2dTexCoords(Candera::Int inTextureDimensionS, Candera::Int inTextureDimensionT, Candera::Int kernelRadius) const
{
   size_t kernelSize = static_cast<size_t>(2 * kernelRadius + 1);
   TexCoords2D texCoords2d(kernelSize, TexCoordRow(kernelSize));
   Candera::Double texCoordStepWidthS = 1.0 / static_cast<Candera::Double>(inTextureDimensionS);
   Candera::Double texCoordStartS = -1.0 * texCoordStepWidthS * static_cast<Candera::Double>(kernelRadius);
   Candera::Double texCoordStepWidthT = 1.0 / static_cast<Candera::Double>(inTextureDimensionT);
   Candera::Double texCoordStartT = -1.0 * texCoordStepWidthT * static_cast<Candera::Double>(kernelRadius);
   Candera::Vector2 currentTexCoord;

   /* compute values */
   for (size_t row = 0; row < texCoords2d.size(); row++)
   {
      for (size_t col = 0; col < texCoords2d[row].size(); col++)
      {
         Candera::Float sCoord = static_cast<Candera::Float>(texCoordStartS + (static_cast<Candera::Double>(col) * texCoordStepWidthS));
         Candera::Float tCoord = static_cast<Candera::Float>(texCoordStartT + (static_cast<Candera::Double>(row) * texCoordStepWidthT));

         texCoords2d[row][col] = Candera::Vector2(sCoord, tCoord);
      }
   }
   return texCoords2d;
}


void BlurShaderStatic::UpdateGuidString()
{
   BlurShaderBase::UpdateGuidString();
   std::stringstream target;
   //make a string out of all the members
   target << m_guidString << "/BlurShaderStratic:" << m_horizontal << ":" << m_vertical;
   m_guidString += target.str();
}


}
}


}   /* namespace CgiWidgetLibrary */
