/* ***************************************************************************************
* FILE:          BlurCameraSetter.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  FocusGroupWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "BlurCameraSetter.h"
#include "BlurUtilsCommon.h"
#include "WindowRect.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurCameraSetter.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace blur {


void CopyCameraSettings(Candera::Camera2D const* source, Candera::Camera* target, bool copyViewport, bool copyScissor)
{
   if (NULL == source)
   {
      ETG_TRACE_ERR(("The source camera is missing (Null Pointer)!"));
      return;
   }

   if (NULL == target)
   {
      ETG_TRACE_ERR(("The target camera is missing (Null Pointer)!"));
      return;
   }

   target->SetSwapEnabled(source->IsSwapEnabled());
   target->SetScissoringEnabled(source->IsScissoringEnabled());
   target->SetAlphaValue(source->GetAlphaValue());

   // Candera::Vector3 position(source->GetPosition().GetX(), source->GetPosition().GetY(), 0.0F);
   // target->SetPosition(position);
   target->SetRenderingEnabled(source->IsRenderingEnabled());

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Blur::CopyCameraSettings isClearColorEnabled=%u, camera=%s",
                       source->IsClearColorEnabled(), target->GetName() != NULL ? target->GetName() : "null"));

   Candera::ClearMode clearMode = target->GetClearMode();
   clearMode.SetClearColor(source->GetClearColor());
   clearMode.SetColorClearEnabled(source->IsClearColorEnabled());
   clearMode.SetStencilClearEnabled(false);
   clearMode.SetDepthClearEnabled(false);
   clearMode.SetClearDepth(1.0);
   target->SetClearMode(clearMode);

   if (copyViewport)
   {
      WindowRect viewportRect;
      GetViewport(source, viewportRect);
      SetViewport(target, viewportRect);
   }

   if (copyScissor)
   {
      WindowRect scissorRect;
      GetViewport(source, scissorRect);
      SetViewport(target, scissorRect);
   }
}


void CopyCameraSettings(Candera::Camera const* source, Candera::Camera* target, bool copyViewport, bool copyScissor)
{
   if (NULL == source)
   {
      ETG_TRACE_ERR(("The source camera is missing (Null Pointer)!"));
      return;
   }

   if (NULL == target)
   {
      ETG_TRACE_ERR(("The target camera is missing (Null Pointer)!"));
      return;
   }

   target->SetSwapEnabled(source->IsSwapEnabled());
   target->SetScissoringEnabled(source->IsScissoringEnabled());
   target->SetAlphaValue(source->GetAlphaValue());
   // target->SetPosition(source->GetPosition()); // 3D Camera coordinates do not match to blurred rendertarget
   target->SetRenderingEnabled(source->IsRenderingEnabled());
   target->SetClearMode(source->GetClearMode());
   target->SetViewingFrustumCullingEnabled(source->IsViewingFrustumCullingEnabled());

   if (copyViewport)
   {
      WindowRect viewportRect;
      GetViewport(source, viewportRect);
      SetViewport(target, viewportRect);
   }

   if (copyScissor)
   {
      WindowRect scissorRect;
      GetViewport(source, scissorRect);
      SetViewport(target, scissorRect);
   }
}


void GetViewport(Candera::Camera2D const* camera, WindowRect& outViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   Candera::RenderTarget2D const* renderTarget = camera->GetRenderTarget();
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outViewport.SetRectInPixel(camera->GetViewport(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void GetViewport(Candera::Camera2D const* camera, Candera::RenderTarget2D const* renderTarget, WindowRect& outViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outViewport.SetRectInPixel(camera->GetViewport(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void GetViewport(Candera::Camera const* camera, WindowRect& outViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   Candera::RenderTarget const* renderTarget = camera->GetRenderTarget();
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outViewport.SetRectNormalized(camera->GetViewport(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void GetViewport(Candera::Camera const* camera, Candera::RenderTarget3D const* renderTarget, WindowRect& outViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outViewport.SetRectNormalized(camera->GetViewport(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void SetViewport(Candera::Camera2D* camera, WindowRect& inViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   camera->SetViewport(inViewport.GetRectInPixel());
}


void SetViewport(Candera::Camera* camera, WindowRect& inViewport)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   camera->SetViewport(inViewport.GetRectNormalized());
}


void GetScissor(Candera::Camera2D const* camera, WindowRect& outScissor)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   Candera::RenderTarget2D const* renderTarget = camera->GetRenderTarget();
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outScissor.SetRectInPixel(camera->GetScissorRectangle(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void GetScissor(Candera::Camera const* camera, WindowRect& outScissor)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   Candera::RenderTarget const* renderTarget = camera->GetRenderTarget();
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("The camera has no render target assigned!"));
      return;
   }

   outScissor.SetRectNormalized(camera->GetViewport(), static_cast<Candera::Float>(renderTarget->GetWidth()), static_cast<Candera::Float>(renderTarget->GetHeight()));
}


void SetScissor(Candera::Camera2D* camera, WindowRect& inScissor)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   camera->SetScissorRectangle(inScissor.GetRectInPixel());
}


void SetScissor(Candera::Camera* camera, WindowRect& inScissor)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("The camera is missing (Null Pointer)!"));
      return;
   }

   camera->SetScissorRectangle(inScissor.GetRectNormalized());
}


}
}


}
