/* ***************************************************************************************
* FILE:          TraceQueryHelper.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TraceQueryHelper.cpp is part of HMI-Base widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "sys_std_if.h"

#include "TraceQueryHelper.h"
#include "TextWidgetFinderCallback.h"
#include "TextIdWidgetFinderCallback.h"
#include "ListWidgetFinderCallback.h"
#include "IndexedWidgetFinderCallback.hpp"

#include "Widgets/2D/List/ListWidget2D.h"
#include "Widgets/2D/WidgetFinder2D.h"

#include "PrintAllVisibleTextCallback.h"
#include "PrintTextInListCallback.h"

#include <CanderaPlatform/Device/Common/Effects/TextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColorBlend.h>

#include "Widgets/2D/ScrollableText/ScrollableTextWidget2D.h"
namespace hmibase {
namespace widget {
namespace trace {

bool TraceQueryHelper::ProcessSimTouchRequest(const SimTouchReqMsg& msg, ::Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   bool consumed = true;
   switch (msg.GetIdentifier())
   {
      case hmibase::SIM_TOUCH_BY_TEXT:
         ::hmibase::trace::TraceQueryHelper::GetInstance().simTouchByText<TextWidgetFinderCallback>(msg, viewHandler, hmiRenderViewVisitor);
         break;
      case hmibase::SIM_TOUCH_BY_TEXT_ID:
         ::hmibase::trace::TraceQueryHelper::GetInstance().simTouchByTextId<TextIdWidgetFinderCallback>(msg, viewHandler, hmiRenderViewVisitor);
         break;
      case hmibase::SIM_TOUCH_BY_INDEX:
         ::hmibase::trace::TraceQueryHelper::GetInstance().simTouchByIndex<IndexedWidgetFinderCallback<BaseButtonWidget2D>, BaseButtonWidget2D, ListWidget2D, ListWidgetFinderCallback>(msg, viewHandler, hmiRenderViewVisitor);
         break;
      default:
         consumed = false;
         break;
   }

   return consumed;
}


void TraceQueryHelper::ProcessQueryTraceMsgToPrintVisibleText(const QueryTraceInfoReqMsg& msg, ::Courier::View* view, std::string& printData)
{
   switch (msg.GetQueryId())
   {
      case hmibase::GET_ALL_VISIBLE_TEXT_IN_GIVEN_LISTWIDGET:
         ::hmibase::trace::TraceQueryHelper::GetInstance().printAllVisibleTextInList<PrintTextInListCallback, ListWidget2D, ListWidgetFinderCallback>(msg.GetUserData1(), view, printData);
         break;

      case hmibase::GET_ALL_VISIBLE_TEXT_IN_VIEW:
         ::hmibase::trace::TraceQueryHelper::GetInstance().printAllVisibleTextInView<PrintAllVisibleTextCallback>(view, printData);
         break;

      case hmibase::GET_ALL_VISIBLE_TEXT_IN_ALL_LISTWIDGETS:
         ::hmibase::trace::TraceQueryHelper::GetInstance().printAllVisibleTextInList<PrintTextInListCallback>(view, printData);
         break;

      default:
         break;
   }
}


bool TraceQueryHelper::ProcessTraceQueryRequest(const QueryTraceInfoReqMsg& msg, ::Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   bool consumed = true;
   switch (msg.GetQueryId())
   {
      case hmibase::GET_ALL_VISIBLE_TEXTID_IN_LISTITEM:
      {
         ::hmibase::trace::TraceQueryHelper::GetInstance().getTextIdsInListItem<BaseWidget2D, TextWidget2D, WidgetFinder, ScrollableTextWidget2D>(msg, viewHandler, hmiRenderViewVisitor);
      }
      break;

      case hmibase::GET_WIDGET_INSTANCE_INFO:
      {
         ::hmibase::trace::TraceQueryHelper::GetInstance().getWidgetInstanceInfo<BaseWidget2D, TextWidget2D, WidgetFinder, ScrollableTextWidget2D>(msg, viewHandler, hmiRenderViewVisitor);
      }
      break;
      case hmibase::GET_WIDGET_PROPERTY_INFO:
      {
         ::hmibase::trace::TraceQueryHelper::GetInstance().getWidgetPropertyInfo<BaseWidget2D, TextWidget2D, WidgetFinder, ScrollableTextWidget2D>(msg, viewHandler, hmiRenderViewVisitor);
      }
      break;

      default:
      {
         consumed = false;
      }
      break;
   }

   return consumed;
}


bool TraceQueryHelper::ProcessWidgetPropertySetRequest(const WidgetPropertySetterReqMsg& msg, ::Courier::IViewHandler& viewHandler, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   return ::hmibase::trace::TraceQueryHelper::GetInstance().ProcessWidgetPropertySetRequest<BaseWidget2D, WidgetFinder>(msg, viewHandler, hmiRenderViewVisitor);
}


static bool GetNodeDimension(Candera::Float& fWidth, Candera::Float& fHeight, Candera::Node2D* nChangeNode)
{
   fHeight = 0.0F;
   fWidth = 0.0F;

   if (nChangeNode)
   {
      Candera::RenderNode* rNode = Candera::Dynamic_Cast<Candera::RenderNode*>(nChangeNode);

      if (rNode)
      {
         Candera::Vector2 v2OriginalScale = rNode->GetScale();
         Candera::Effect2D* eNodeEffect = rNode->GetEffect(0);

         if (eNodeEffect->IsTypeOf(Candera::BitmapBrushBlend::GetTypeId()))
         {
            Candera::BitmapBrushBlend* bEffect = dynamic_cast<Candera::BitmapBrushBlend*>(eNodeEffect);

            if (bEffect != NULL)
            {
               Candera::Image2D* iImage = bEffect->GetBitmapBrush().Image().Get();

               fHeight = v2OriginalScale.GetY() * static_cast<Candera::Float>(iImage->GetHeight());
               fWidth = v2OriginalScale.GetX() * static_cast<Candera::Float>(iImage->GetWidth());
               return true;
            }
         }
         else if (eNodeEffect->IsTypeOf(Candera::SolidColorBrushBlend::GetTypeId()))
         {
            Candera::SolidColorBrushBlend* sEffect = dynamic_cast<Candera::SolidColorBrushBlend*>(eNodeEffect);

            if (sEffect != NULL)
            {
               Candera::Vector2 v2ColorSize = sEffect->GetSolidColorBrush().Size();

               fHeight = v2OriginalScale.GetY() * (v2ColorSize.GetY());
               fWidth = v2OriginalScale.GetX() * (v2ColorSize.GetX());
               return true;
            }
         }
         else if (eNodeEffect->IsTypeOf(Candera::TextBrushBlend::GetTypeId()))
         {
            Candera::TextBrushBlend* sEffect = dynamic_cast<Candera::TextBrushBlend*>(eNodeEffect);

            if (sEffect != NULL)
            {
               Candera::Rectangle  r;
               sEffect->GetBoundingRectangle(r);
               fHeight = v2OriginalScale.GetY() * r.GetHeight();
               fWidth = v2OriginalScale.GetX() * r.GetWidth();
               return true;
            }
         }
      }
   }
   return false;
}


void TraceQueryHelper::GetTextWidgetProperties(::Candera::Widget2D* widget, TextProperties& textProperties)
{
   TextWidget2D* textWidget2D = ::Candera::Dynamic_Cast<TextWidget2D*>(widget);

   if (textWidget2D)
   {
      textProperties.isValid = true;

      textProperties.text = textWidget2D->GetText();

      float width = 0.0f;
      float height = 0.0f;

      GetNodeDimension(width, height, textWidget2D->GetNode());
      textProperties.dimension = ::Candera::Vector2(width, height);
      textProperties.maxNumOfLines = textWidget2D->GetMaximumNumberOfLines();
      textProperties.style = textWidget2D->GetStyle();
      textProperties.maxDimension = textWidget2D->GetMaximumSize();
      textProperties.referredTextId = textWidget2D->GetTransRefs().GetCString();
   }
   else
   {
      textProperties.isValid = false;
   }
}


void TraceQueryHelper::ProcessQueryTraceMsgToGetPossibleTouchPoints(const QueryTraceInfoReqMsg& msg, Courier::IViewHandler& viewHandler, Courier::View* view, const ::hmibase::view::HMIRenderViewVisitor& hmiRenderViewVisitor)
{
   switch (msg.GetQueryId())
   {
      case GET_POSSIBLE_TOUCHPOINT_ELEM_NAME:
         ::hmibase::trace::TraceQueryHelper::GetInstance().getPossibleTouchPointsForElementWithName<Candera::WidgetBase*, BaseWidget2D, WidgetFinder>(msg, viewHandler, hmiRenderViewVisitor);
         break;

      case GET_POSSIBLE_TOUCHPOINT_ELEM_TEXT:
         ::hmibase::trace::TraceQueryHelper::GetInstance().getPossibleTouchPointsForElementsWithTextData<TextWidgetFinderCallback>(msg, view);
         break;

      case GET_POSSIBLE_TOUCHPOINT_ELEM_TEXTID:
         ::hmibase::trace::TraceQueryHelper::GetInstance().getPossibleTouchPointsForElementsWithTextData<TextIdWidgetFinderCallback>(msg, view);
         break;

      case GET_POSSIBLE_TOUCHPOINT_LISTELEM:
         ::hmibase::trace::TraceQueryHelper::GetInstance().getPossibleTouchPointsForListWithIndex<IndexedWidgetFinderCallback<BaseButtonWidget2D>, BaseButtonWidget2D, ListWidget2D, ListWidgetFinderCallback>(msg, view);
         break;
      default:
         break;
   }
}


}
}


}
