/* ***************************************************************************************
* FILE:          Touchable3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Touchable3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"

#include "BaseContract/generated/BaseTouchSessionMsgs.h"
#include "Courier/Visualization/ViewScene3D.h"
#include "NodeSelector.h"
#include "Touchable3D.h"
#include <View/CGI/Widget/WidgetController.h>

#if !defined(WIN32)
#include "CanderaPlatform/Device/Genivi/GeniviWindowSurface.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TOUCHABLE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Touchable3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(Touchable3D)

Touchable3D::Touchable3D() :
   _bIsTouchable(false),
   _touchPriority(TouchPriorityType(0)),
   _touchSession(NULL)
{
}


Touchable3D::~Touchable3D()
{
   if (_touchSession != NULL)
   {
      _touchSession->Deregister(this);
      _touchSession = NULL;
   }
}


/****************************************************************************
*  OnParentViewLoad is invoked by F/W when parent view is loaded or unloaded
*     Parameters  : true  - Scene context was loaded
*                   false - Scene context was unloaded
****************************************************************************/
void Touchable3D::OnParentViewLoad(bool load)
{
   if (!load)
   {
      if (_touchSession != NULL)
      {
         _touchSession->Deregister(this);

         _TODO("check why TouchSession::deregister myself is not enough")
         _touchSession->Cleanup();   // with commit a4487852 the issue is not solved in al cases, therefore Cleanup() method re-enabled
         // see also -> https://hi-cmts.apps.intranet.bosch.com:8443/browse/CMG3GB-2395

         _touchSession = NULL;
      }
   }
}


/****************************************************************************
*  OnParentViewActivate is invoked by F/W when parent view changes its
*  activation state
*     Parameters  : true  - View is activated
*                   false - View is deactivated
****************************************************************************/
void Touchable3D::OnParentViewActivate(bool activate)
{
   if (!activate)
   {
      if (_touchSession != NULL)
      {
         _touchSession->Deregister(this);

         _TODO("check why TouchSession::deregister myself is not enough")
         _touchSession->Cleanup();   // with commit a4487852 the issue is not solved in al cases, therefore Cleanup() method re-enabled
         // see also -> https://hi-cmts.apps.intranet.bosch.com:8443/browse/CMG3GB-2395

         _touchSession = NULL;
      }
   }
}


bool Touchable3D::OnMessage(const Message& msg)
{
   if (DELEGATE_WIDGET_BOOL_FUNC(OnMessage(*this, msg)))
   {
      return true;
   }

   if (Base::OnMessage(msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case hmibase::input::TouchSessionStartEvent::ID:
      {
         const hmibase::input::TouchSessionStartEvent* startEvent = Courier::message_cast<const hmibase::input::TouchSessionStartEvent*>(&msg);
         if (startEvent != NULL)
         {
            const Courier::TouchInfo& touchInfo = startEvent->GetTouchInfo();
            if (IsInsideBoundingRect(touchInfo) && IsTouchable() && IsEnabled())
            {
               if (_touchSession != NULL)
               {
                  _touchSession->Deregister(this);
                  _touchSession = NULL;
               }

               hmibase::input::TouchSession& touchSession = const_cast<hmibase::input::TouchSession&>(startEvent->GetTouchSession().Get());
               _touchSession = &touchSession;
               touchSession.Register(this);

               postTouchUpdMessage(hmibase::widget::enTouchState::Down);
            }
         }
      }
      break;

      case hmibase::input::TouchSessionStopEvent::ID:
      {
         const hmibase::input::TouchSessionStopEvent* stopEvent = Courier::message_cast<const hmibase::input::TouchSessionStopEvent*>(&msg);
         if (stopEvent != NULL)
         {
            hmibase::input::TouchSession& touchSession = const_cast<hmibase::input::TouchSession&>(stopEvent->GetTouchSession().Get());
            if (_touchSession == &touchSession)
            {
               touchSession.Deregister(this);
               _touchSession = NULL;

               postTouchUpdMessage(hmibase::widget::enTouchState::Up);
            }
         }
      }
      break;

      case Courier::TouchMsg::ID:
      {
         const Courier::TouchMsg* touchMsg = Courier::message_cast<const Courier::TouchMsg*>(&msg);
         if (touchMsg != NULL)
         {
         }
      }
      break;

      default:
         break;
   }

   return consumed;
}


void Touchable3D::postTouchUpdMessage(hmibase::widget::enTouchState::Enum state)
{
   Courier::View* view = GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "Touchable3D: postTouchUpdMessage state=%d userData=%u [%50s.%s]",
                          ETG_CENUM(hmibase::widget::enTouchState::Enum, state), GetUserData(), GetLegacyName(), view->GetId().CStr()));

      hmibase::widget::TouchUpdMsg* msg = COURIER_MESSAGE_NEW(hmibase::widget::TouchUpdMsg)(view->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), state);
      if (msg != NULL)
      {
         msg->Post();
      }
   }
}


/****************************************************************************
* IsInsideBoundingRec function tells whether the touch coordinate is within
* the widget's bounding rectangle or not.
* Return true, if the touch event happens with the bounding rectangle
****************************************************************************/
bool Touchable3D::IsInsideBoundingRect(const Courier::TouchInfo& msg)
{
   if (GetParentView() != NULL)
   {
      Courier::ViewScene3D* viewScene3D = GetParentView()->ToViewScene3D();
      if (viewScene3D != NULL)
      {
         const Courier::ViewScene3D::CameraPtrVector& cameras = viewScene3D->GetCameraPtrVector();
         const Candera::Vector2 point(static_cast<FeatStd::Float>(msg.mX), static_cast<FeatStd::Float>(msg.mY));

         for (Courier::ViewScene3D::CameraPtrVector::ConstIterator it = cameras.ConstBegin(); it != cameras.ConstEnd(); ++it)
         {
            Candera::Camera* camera3D = *it;
            if ((camera3D != NULL) && camera3D->IsRenderingEnabled())
            {
#if defined(WIN32)
               return IsNodePicked(camera3D, (FeatStd::Int16)msg.mX, (FeatStd::Int16)msg.mY);
#else

               if (camera3D->GetRenderTarget() != NULL)
               {
                  Candera::GeniviWindowSurface* surface = dynamic_cast<Candera::GeniviWindowSurface*>(camera3D->GetRenderTarget()->GetGraphicDeviceUnit());     //lint !e740

                  //The following issues are generated because Lint can not understand the complex definition of the class GeniviWindowSurface
                  // Lint prio 1/2 are authorized deactivated for this function
                  //Info 737: prio3: Loss of sign in promotion from int to unsigned int
                  CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(737, "symbol GetSupportProperties is defined in base class")
                  if ((surface != NULL) && (msg.mSourceId == surface->GetSupportProperties().GetSurfaceId()))
                  {
                     ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BaseWidget2D : In IsInsideBoundingRect() sourceId: [ %d ]", msg.mSourceId));

                     FeatStd::Float dst = 0.0f;
                     if (OnTouch(*camera3D, point, dst))
                     {
                        return true;
                     }
                     return IsNodePicked(camera3D, (FeatStd::Int16)msg.mX, (FeatStd::Int16)msg.mY);
                  }
               }

#endif
            }
         }
      }
   }

   return false;
}


bool Touchable3D::IsNodePicked(Candera::Camera* camera, FeatStd::Int16 x, FeatStd::Int16 y)
{
   if (GetNode() != NULL)
   {
      Candera::Node* tmpNode = NodeSelector::SelectNode(GetNode(), camera, x, y, GetNode()->GetScopeMask(), true);
      return (tmpNode != 0 && tmpNode->GetName() == GetNode()->GetName());
   }

   return false;
}


bool Touchable3D::OnTouch(const Candera::Camera& camera3D, const Candera::Vector2& pos, FeatStd::Float& dist)
{
   FEATSTD_UNUSED3(camera3D, pos, dist);

   return false ;
}


bool Touchable3D::processGestureEvent(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool bConsumed = false;

   if (DELEGATE_WIDGET_BOOL_FUNC(OnGesture(*this, gestureData)))
   {
      return true;
   }

   if (gestureData._gestureType == hmibase::input::gesture::GestureEvent::GT_PressHoldRepeatTap)
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            onTouchPress();
            break;

         case hmibase::input::gesture::GestureEvent::ET_MOVE:
            break;

         case hmibase::input::gesture::GestureEvent::ET_ABORT:
         case hmibase::input::gesture::GestureEvent::ET_END:
            onTouchRelease();
            break;

         default:
            break;
      }

      bConsumed = true;
   }
   return bConsumed;
}


/****************************************************************************
* Returns the gesture configuration as required by the gesture framework.
*****************************************************************************/
hmibase::input::gesture::GestureListener::Gestures Touchable3D::GetGestureList() const
{
   hmibase::input::gesture::GestureListener::Gestures gestures;
   if (GetTouchableStatus())
   {
      // Press and Tap
      // gesture must be enabled both in the widget and in the configuration
      hmibase::input::gesture::GestureListener::GestureConfig config;

      config.tapPressHoldRepeatGestureParameter.maxDistance = 0;
      config.tapPressHoldRepeatGestureParameter.doubleTapEnabled = true;
      config.tapPressHoldRepeatGestureParameter.doubleTapTime = 1000;
      config.tapPressHoldRepeatGestureParameter.holdTime = 0;
      config.tapPressHoldRepeatGestureParameter.repeatTime = 0;

      gestures.push_back(Gesture(hmibase::input::gesture::GestureEvent::GT_PressHoldRepeatTap, config));
   }
   return gestures;
}


hmibase::input::gesture::GestureListener::GesturePriorizationParameter Touchable3D::getGesturePriorizationParameter() const
{
   GesturePriorizationParameter params;
   params.effectiveTouchPriority = GetEffectiveTouchPriority();
   params.effectiveRenderOrderRank = 0; // no render order rank in 3D

   return params;
}


void Touchable3D::SetTouchableStatus(const bool touchableStatus)
{
   _bIsTouchable = touchableStatus;
}


bool Touchable3D::GetTouchableStatus() const
{
   return _bIsTouchable;
}


void Touchable3D::SetTouchPriority(TouchPriorityType touchPriority)
{
   _touchPriority = touchPriority;
}


void Touchable3D::onTouchPress()
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Touchable3D: onTouchPress"));
}


void Touchable3D::onTouchRelease()
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Touchable3D: onTouchRelease"));
}


bool Touchable3D::OnTouchGeneric(const Candera::CanderaObject& camera, const Candera::Vector2& pos, FeatStd::Float dist)
{
   if (camera.IsTypeOf(Candera::Camera::GetTypeId()))
   {
      const Candera::Camera& camera3D = dynamic_cast<const Candera::Camera&>(camera);
      return OnTouch(camera3D, pos, dist);
   }
   return false;
}


Courier::TouchInfo Touchable3D::getCurrentSessionTouchInfo() const
{
   return (_touchSession != NULL) ? _touchSession->GetTouchInfo() : Courier::TouchInfo(0, 0, 0, 0, 0);
}


Courier::TouchInfo Touchable3D::getCurrentSessionStartTouchInfo() const
{
   return (_touchSession != NULL) ? _touchSession->GetStartTouchInfo() : Courier::TouchInfo(0, 0, 0, 0, 0);
}
