/* ***************************************************************************************
* FILE:          MultiSliderHelperWidget3D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MultiSliderHelperWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "MultiSliderHelperWidget3D.h"
#include <Candera/Engine3D/Core/Appearance.h>
#include <Candera/Engine3D/Core/Mesh.h>
#include <Candera/Engine3D/ShaderParamSetters/ShaderParamSetter.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/MultiSliderHelperWidget3D.cpp.trc.h"
#endif

using Candera::Appearance;
using Candera::Dynamic_Cast;
using Candera::Matrix4;
using Candera::Mesh;
using Candera::Node;
using Candera::Shader;
using Candera::ShaderParamSetter;
using Candera::Vector2;
using Candera::Vector3;
using Candera::VertexBuffer;
using Candera::VertexGeometry;
using FeatStd::Float;
using FeatStd::UInt;
using FeatStd::UInt16;
using FeatStd::UInt32;
using FeatStd::SizeType;

CGI_WIDGET_RTTI_DEFINITION(MultiSliderHelperWidget3D)

MultiSliderHelperWidget3D::Billboard::Billboard()
{
   m_geometry[0] = GeometryData(Vector3(-1.0f, 0.0f, -1.0f), Vector2(0.0f, 0.0f));
   m_geometry[1] = GeometryData(Vector3(-1.0f, 0.0f,  1.0f), Vector2(0.0f, 1.0f));
   m_geometry[2] = GeometryData(Vector3(0.0f, 0.0f,  0.0f), Vector2(0.5f, 0.5f));
   m_geometry[3] = GeometryData(Vector3(1.0f, 0.0f, -1.0f), Vector2(1.0f, 0.0f));
   m_geometry[4] = GeometryData(Vector3(1.0f, 0.0f,  1.0f), Vector2(1.0f, 1.0f));
}


void MultiSliderHelperWidget3D::Billboard::Move(Float x, Float y)
{
   for (SizeType i = 0; i < static_cast<SizeType>(c_vertexCount); ++i)
   {
      m_geometry[i].m_pos += Vector3(x, y, 0);
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D %.3f / %.3f / %.3f", m_geometry[i].m_pos.GetX(), m_geometry[i].m_pos.GetY(), m_geometry[i].m_pos.GetZ()));
   }
}


void MultiSliderHelperWidget3D::Billboard::Rotate(Float angle)
{
   Matrix4 m;
   m.SetRotationZ(angle);
   for (SizeType i = 0; i < static_cast<SizeType>(c_vertexCount); ++i)
   {
      m_geometry[i].m_pos.TransformCoordinate(m);
   }
}


void MultiSliderHelperWidget3D::Billboard::SetGeometry(SizeType index, Float x, Float y)
{
   m_geometry[index].m_pos = Vector3(x, y, 0.0f);
   SizeType u = index / 2;
   SizeType v = index % 2;
   m_geometry[index].m_texCoord = Vector2(Float(u), Float(v));
}


/** Billboard vertices and triangles:
 *  1-----4
 *  |\   /|
 *  | \ / |
 *  |  2  |
 *  | / \ |
 *  |/   \|
 *  0-----3
 */
const UInt16 MultiSliderHelperWidget3D::Billboard::c_index[] =
{
   0, 1, 2,
   2, 1, 4,
   4, 3, 2,
   2, 3, 0
};


void MultiSliderHelperWidget3D::DisposeBillboards(const void* ptr)
{
   const Billboard* billboard = reinterpret_cast<const Billboard*>(ptr);
   FEATSTD_DELETE_ARRAY(billboard);
}


void MultiSliderHelperWidget3D::DisposeIndizes(const void* indizes)
{
   FEATSTD_DELETE_ARRAY(indizes);
}


void MultiSliderHelperWidget3D::SetIndizes(SizeType billboardIndex, UInt16* index)
{
   const SizeType offset = billboardIndex * static_cast<SizeType>(Billboard::c_vertexCount);
   SizeType destIndex = billboardIndex * static_cast<SizeType>(Billboard::c_indexCount);
   for (SizeType i = 0; i < static_cast<SizeType>(Billboard::c_indexCount); ++i)
   {
      index[destIndex++] = static_cast<UInt16>(Billboard::c_index[i] + offset);
   }
}


Candera::Color MultiSliderHelperWidget3D::Mix(const Candera::Color& color1, const Candera::Color& color2, Float value)
{
   return (color1 * (1 - value)) + (color2 * value);
}


static VertexGeometry::VertexElementFormat s_vertexElementFormat[] =
{
   { 0, VertexGeometry::Float32_3, VertexGeometry::Position, 0 },
   { 12, VertexGeometry::Float32_2, VertexGeometry::TextureCoordinate, 0 }
};


#define ARRAY_SIZE(array) \
    sizeof(array) / sizeof(array[0])

MultiSliderHelperWidget3D::MultiSliderHelperWidget3D() :
   m_createVertexBuffer(false),
   m_reCalculate(false),
   m_f(1.0f)
{
}


MultiSliderHelperWidget3D::~MultiSliderHelperWidget3D()
{
}


void MultiSliderHelperWidget3D::UpdateValues(const ValueArrayProperty& valueArray)
{
   const SizeType count = valueArray.GetCount();

   if (m_valueArray.GetCount() != count)
   {
      m_valueArray.Clear();
      for (SizeType i = 0; i < count; ++i)
      {
         m_valueArray.Add(CalculateValue(valueArray.Get(i)));
      }
      m_createVertexBuffer = true;
   }
   else
   {
      for (SizeType i = 0; i < count; ++i)
      {
         m_valueArray.Get(i) = CalculateValue(valueArray.Get(i));
      }
   }

   m_reCalculate = true;
}


void MultiSliderHelperWidget3D::Update()
{
   Base::Update();

   if (m_createVertexBuffer)
   {
      m_vertexBuffer = CreateVertexBuffer();
      UpdateMesh();
      m_createVertexBuffer = false;
   }

   if (m_reCalculate)
   {
      ReCalculate();
      m_reCalculate = false;
   }
}


void MultiSliderHelperWidget3D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case ScalePropertyId:
      case RadiusOffsetPropertyId:
         m_createVertexBuffer = true;
         break;

      case NonLinearFactorPropertyId:
         m_f = 1.0f - Candera::Math::Power(Candera::Math::EulerConstant(), -GetNonLinearFactor());
         break;

      default:
         break;
   }
}


void MultiSliderHelperWidget3D::OnNodeChanged()
{
   UpdateMesh();
   Base::OnNodeChanged();
}


VertexBuffer::SharedPointer MultiSliderHelperWidget3D::CreateVertexBuffer()
{
   VertexBuffer::SharedPointer vertexBuffer;

   bool autoDisposeBillboard = false;

   const SizeType billboardCount = m_valueArray.GetCount();
   Billboard* billboard = FEATSTD_NEW_ARRAY(Billboard, billboardCount);

   const SizeType c_indexCount = static_cast<SizeType>(Billboard::c_indexCount) * billboardCount;
   UInt16* index = FEATSTD_NEW_ARRAY(FeatStd::UInt16, c_indexCount);

   if ((billboard != 0) && (index != 0))
   {
      const Float a = 2.0f;   // width of billboard  ( -1 .. +1)

      // calculate inner radius of the regular polygon
      // RadiusOffset can be used to create a gap on the edges of the polygon
      const Float r = (a / (2.0f * Candera::Math::Tangent(Candera::Math::Pi() / static_cast<Float>(billboardCount))) + GetRadiusOffset());

      const Float deltaAngle = 360.0f / static_cast<Float>(billboardCount);
      const Float offsetAngle = deltaAngle / 2.0F; // to have value with index 0 at north position

      for (SizeType i = 0; i < billboardCount; i++)
      {
         const Float angle = deltaAngle * static_cast<Float>(i) + offsetAngle;
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D %u: %.3f", i, angle));
         billboard[i].Rotate(-angle);

         const Float angleRad = Candera::Math::DegreeToRadian(angle);
         const Float x = r * Candera::Math::Sine(angleRad);
         const Float y = r * Candera::Math::Cosine(angleRad);
         billboard[i].Move(x, y);

         SetIndizes(i, index);
      }

      VertexGeometry* vertexGeometry = CANDERA_NEW(VertexGeometry)(
                                          billboard, DisposeBillboards,
                                          s_vertexElementFormat, 0,
                                          index, DisposeIndizes,
                                          static_cast<UInt16>(billboardCount * Billboard::c_vertexCount),
                                          sizeof(GeometryData),
                                          ARRAY_SIZE(s_vertexElementFormat),
                                          static_cast<UInt32>(c_indexCount),
                                          VertexGeometry::VideoMemory,
                                          VertexGeometry::IndexedArrayBuffer,
                                          VertexGeometry::DynamicWrite);

      if (vertexGeometry != 0)
      {
         autoDisposeBillboard = true;
         vertexBuffer = VertexBuffer::Create();
         if (vertexBuffer != 0)
         {
            if (!vertexBuffer->SetVertexGeometry(vertexGeometry, VertexBuffer::VertexGeometryDisposer::Dispose))
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D: SetVertexGeometry() failed!"));
            }
         }
         else
         {
            FEATSTD_SAFE_DELETE(vertexGeometry);
         }
      }
   }

   if (!autoDisposeBillboard)
   {
      FEATSTD_SAFE_DELETE(billboard);
      FEATSTD_SAFE_DELETE(index);
   }

   return vertexBuffer;
}


void MultiSliderHelperWidget3D::UpdateMesh()
{
   Mesh* mesh = GetMesh();
   if (mesh != 0)
   {
      mesh->SetVertexBuffer(m_vertexBuffer);
      if (!mesh->UploadAll())
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D: mesh->UploadAll() failed!"));
      }
      Invalidate();
   }
}


void MultiSliderHelperWidget3D::ReCalculate()
{
   if (m_vertexBuffer != 0)
   {
      const SizeType billboardCount = m_valueArray.GetCount();

      VertexGeometry::VertexArrayResource vertexResource(m_vertexBuffer->GetVertexGeometry()->GetVertexArrayResourceHandle());
      Billboard* billboard = const_cast<Billboard*>(FeatStd::Internal::PointerToPointer<const Billboard*>(vertexResource.GetData()));

      if (billboard != 0)
      {
         const Float& valueMin = GetValueMin();
         Vector2 sum;
         for (SizeType index = 0; index < billboardCount; ++index)
         {
            static const int iCenter = 2;

            const Float valueLeft = Candera::Math::Maximum(valueMin, m_valueArray.Get(index));
            GeometryData& geometryTopLeft = billboard[index].m_geometry[1];
            geometryTopLeft.m_pos.SetZ(2.0f * valueLeft - 1.0f);
            geometryTopLeft.m_texCoord.SetY(GetScale() ? 1.0f : valueLeft);

            const Float valueRight = Candera::Math::Maximum(valueMin, m_valueArray.Get((index + 1) % billboardCount));
            GeometryData& geometryTopRight = billboard[index].m_geometry[4];
            geometryTopRight.m_pos.SetZ(2.0f * valueRight - 1.0f);
            geometryTopRight.m_texCoord.SetY(GetScale() ? 1.0f : valueRight);

            const Float center = (valueLeft + valueRight) / 2;
            GeometryData& geometryCenter = billboard[index].m_geometry[iCenter];
            geometryCenter.m_pos.SetZ(center - 1.0f);
            geometryCenter.m_texCoord.SetY(GetScale() ? 0.5f : center / 2.0f);

            GeometryData& geo = billboard[index].m_geometry[0];
            sum += Vector2(geo.m_pos.GetX(), geo.m_pos.GetY()) * valueLeft;
         }

         // color changes with calculated center of polygon area
         UpdateColor(sum / Float(billboardCount));

         const FeatStd::UInt32 vertexCount = m_vertexBuffer->GetVertexGeometry()->GetVertexCount();
         if (!m_vertexBuffer->Update(0, vertexCount))
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D: m_vertexBuffer->Update() failed!"));
         }

         Mesh* mesh = GetMesh();
         if (mesh != 0)
         {
            if (!mesh->UploadAll())
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "MultiSliderHelperWidget3D: mesh->UploadAll() failed!"));
            }
            Invalidate();
         }
      }
   }
}


void MultiSliderHelperWidget3D::UpdateColor(const Vector2& center)
{
   Node* node = GetNode();
   if (node != 0)
   {
      Appearance::SharedPointer appearance = node->GetAppearance();
      if (!appearance.PointsToNull())
      {
         ShaderParamSetter::SharedPointer shaderParamSetter(Dynamic_Cast<ShaderParamSetter*>(appearance->GetShaderParamSetter().GetPointerToSharedInstance()));
         if (!shaderParamSetter.PointsToNull())
         {
            const Candera::Color colorX = Mix(GetColorTopLeft(), GetColorTopRight(), center.GetX() + 0.5f);
            const Candera::Color color = Mix(GetColorBottom(), colorX, center.GetY() + 0.5f);
            shaderParamSetter->SetUniform("u_color", Shader::FloatVec4, const_cast<Float*>(&color.GetData().red), 1, true);
         }
      }
   }
}


Float MultiSliderHelperWidget3D::CalculateValue(const Float value) const
{
   Float v = (1 - Candera::Math::Power(1 - m_f, value)) / m_f;

   if (v < 0.0F)
   {
      v = 0.0F;
   }
   else if (v > 1.0F)
   {
      v = 1.0F;
   }

   return v;
}
