/* ***************************************************************************************
* FILE:          DirectTextureProvider3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DirectTextureProvider3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "DirectTextureProvider3D.h"
#include "CanderaAssetLoader/AssetLoaderBase/AssetProvider.h"

using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_DIRECTTEXTURE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DirectTextureProvider3D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(DirectTextureProvider3D)


// Widget to be reworked
#define VARIANT_S_FTR_WIDGET_DEACTIVATED


DirectTextureProvider3D::DirectTextureProvider3D() :
   m_directTextureImage(0),
   m_gdu(0),
   m_invalidated(false),
   m_camera(0),
   m_drmBuffer(0)
{
}


DirectTextureProvider3D::~DirectTextureProvider3D()
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Start DirectTextureProvider3D Destructor."));
#if !defined(CGI_GPU_SIMULATION)

   // Clear external color texture in framebuffer before deallocating associated buffer
   GeniviFrameBufferObject* fbo = FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(m_gdu);
   fbo->GetProperties().SetExternalColorTexture(0);   //lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages

   if (m_drmBuffer != 0)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Deallocating DRM buffer at %X with size %u.", m_drmBuffer->getDataPtr(), m_drmBuffer->getDataSize()));
      hmibase::gadget::DrmAccessor accessor;
      accessor.free(m_drmBuffer);
      m_drmBuffer = 0;
   }
   if (m_gdu != 0)
   {
      m_gdu = 0;
   }
   if (m_camera != 0)
   {
      m_camera = 0;
   }
#endif
#endif
};


void DirectTextureProvider3D::Init(Candera::AssetProvider* /*assetProvider*/)
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   m_invalidated = true;
   Base::Init(assetProvider);
#endif
}


void DirectTextureProvider3D::Update()
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
   if (m_invalidated)
   {
      m_invalidated = !Upload();
   }
#endif
}


bool DirectTextureProvider3D::Upload()
{
#ifdef VARIANT_S_FTR_WIDGET_DEACTIVATED // widget to be reworked
#else
#if !defined(CGI_GPU_SIMULATION)
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Start DirectTextureProvider3D::Upload()..."));
   if (GetNode() == 0)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No Node set on DirectTextureProvider3D."));
      return false;
   }
   if (m_gdu == 0 || m_gdu->GetUnitType() != c_geniviFrameBufferObjectType)
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Graphic Device Unit not set or set to a wrong unit type."));
      return false;
   }

   if (m_camera != 0 && m_camera->IsTypeOf(Candera::Camera::GetTypeId()))
   {
      Candera::Camera* camera = static_cast<Candera::Camera*>(m_camera);
      Candera::RenderTarget3D* rt = camera->GetRenderTarget();
      if (rt != 0)
      {
         //ensure correct context is loaded
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Activating render target of camera \"%s\"...", camera->GetName()));
         rt->Activate();
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No render target defined for given camera."));
      }
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "No display camera defined for direct texture widget."));
   }

   Candera::MemoryManagement::SharedPointer<Texture> texture;
   if (m_directTextureImage == 0)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Creating DirectTextureImage..."));
      m_directTextureImage = DirectTextureImage::Create();
      texture = Texture::Create();
      GetNode()->GetAppearance()->SetTexture(texture);
   }

   GeniviFrameBufferObject* fbo = FeatStd::Internal::PointerToPointer<GeniviFrameBufferObject*>(m_gdu);

   m_directTextureImage->SetWidth(fbo->GetProperties().GetWidth());     //lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
   m_directTextureImage->SetHeight(fbo->GetProperties().GetHeight());   //lint !e10 !e746 !e1013 !e1055 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages
   m_directTextureImage->SetFormat(DirectTextureImage::FormatRGBA);

   Candera::UInt size = m_directTextureImage->GetSize();

   hmibase::gadget::DrmAccessor accessor;
   if (m_drmBuffer != 0)
   {
      //a new buffer will be allocated.
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Deallocating DRM buffer at %X with size %u.", m_drmBuffer->data, m_drmBuffer->size));
      accessor.free(m_drmBuffer);
      m_drmBuffer = 0;
   }
   //Allocate contiguous memory in kernel space using the linux direct rendering manager and retrieve logical address.
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Allocating drm buffer with %u...", size));
   //drm_access_buffer* drmBuffer = drm_access_buffer_allocate(size);
   m_drmBuffer = accessor.allocate(m_directTextureImage->GetWidth(), m_directTextureImage->GetHeight(), 24, 32);

   if (m_drmBuffer == NULL)
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Memory allocation of drm buffer failed!"));
      return false;
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Drm buffer: FB: %u, Handle: %X, Data*: %X, Size: %u", m_drmBuffer->fb, m_drmBuffer->handle, m_drmBuffer->data, m_drmBuffer->size));

   //Set logical address of allocated memory for direct texture image.
   void* logical[] = { m_drmBuffer->data, NULL, NULL, NULL };
   Candera::UInt physical[4] = { ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U), ~static_cast<UInt>(0U) };
   m_directTextureImage->SetLogicalAddress(logical);
   m_directTextureImage->SetPhysicalAddress(physical);
   texture->SetTextureImage(m_directTextureImage);

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Starting Upload of direct texture..."));
   if ((texture.PointsToNull() == false) && (!texture->Upload()))
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Failed to upload direct texture."));
      return false;
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "DirectTexture upload successful."));

   //Associate direct texture image with FBO.
   fbo->GetProperties().SetExternalColorTexture(m_directTextureImage->ToImageSource3D()); //lint !e10 !e746 !e1013 ToDo: If this line stays in final version we must apply for a permission to deactivate the lint msessages

   //Need to re-upload FBO.
   fbo->Unload();
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Starting Upload of FBO..."));
   if (!fbo->Upload())
   {
      ETG_TRACE_FATAL_DCL((APP_TRACECLASS_ID(), "Failed to upload frame buffer object!"));
      return false;
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "FBO upload successful."));

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Direct texture image successfully set as external source for frame buffer object."));
#endif
#endif
   return true;
}
