/* ***************************************************************************************
* FILE:          BlurWidget3D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  BlurWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"

#include "BlurWidget3D.h"
#include "Widgets/utils/Blur/BlurUtilsCommon.h"
#include "Widgets/utils/Blur/BlurCameraSetter.h"
#include <View/CGI/CgiExtensions/SCHostUtils.h>

#ifdef COURIER_ENHANCED_ENABLED
#include "BlurViewScene3D.h"
#include <Courier/Visualization/ViewHandler.h>
#endif  // ifdef COURIER_ENHANCED_ENABLED
#include <View/CGI/CgiExtensions/Renderer.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_BLUR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/BlurWidget3D.cpp.trc.h"
#endif  // ifdef VARIANT_S_FTR_ENABLE_TRC_GEN

#include <CanderaPlatform/Device/Genivi/GeniviFrameBufferObject.h>

using namespace Candera;

namespace hmibase {
namespace widget {
namespace blur {


CGI_WIDGET_RTTI_DEFINITION(BlurWidget3D)


const Candera::Int32 BlurWidget3D::m_minRadius = 1;

OriginalCamera3DVectorType BlurWidget3D::s_originalCameraVector = OriginalCamera3DVectorType();

BlurWidget3D::BlurWidget3D() :
   m_originalNodeRenderingEnabled(false),
   m_bitmapMaskRectangle(0.0F, 0.0F, 0.0F, 0.0F),
   m_billboardViewport(0.0F, 0.0F, 1.0F, 1.0F),
   m_bitmapMaskTexture(Candera::BitmapTextureImage::Create()),
   m_outputRenderModeBlending(Candera::RenderMode::Create()),
   m_outputRenderModeOverwrite(Candera::RenderMode::Create()),
   m_oldBillboardMask(0),
   m_validateBlurParameters(true),
   m_validateShaderType(true),
   m_skipSetup(true),
   m_isRunningInSceneComposer(false),
   m_isOriginalCameraModifiedBySceneComposer(false),
   m_blurEnabledInternal(false),
   m_blurCameraMap(),
   m_maskTextureUnit(1),
   m_sourceTextureUnit(2)
#ifdef COURIER_ENHANCED_ENABLED
   , m_viewSceneActive(false),
   m_viewSceneRenderingEnabled(false),
   m_updateAfterInactive(false)
#endif
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Ctor Blur Widget: [%s]", GetLegacyName()));
   m_darkenUniform[0] = 1.0F;
   m_darkenUniform[1] = 1.0F;
   m_darkenUniform[2] = 1.0F;
   m_darkenUniform[3] = 1.0F;

   m_noDarkenUniform[0] = 1.0F;
   m_noDarkenUniform[1] = 1.0F;
   m_noDarkenUniform[2] = 1.0F;
   m_noDarkenUniform[3] = 1.0F;

   if (m_outputRenderModeBlending.PointsToNull() == false)
   {
      // Inherit everything
      m_outputRenderModeBlending->SetInheritanceBitsToOne(0xFFFFFFFF);

      // Except this
      m_outputRenderModeBlending->SetBlendingEnabled(true);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::BlendingEnabledBit);

      m_outputRenderModeBlending->SetDepthTestEnabled(false);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::DepthTestEnabledBit);

      m_outputRenderModeBlending->SetDepthWriteEnabled(false);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::DepthWriteEnabledBit);

      m_outputRenderModeBlending->SetBlendModeSeparate(Candera::RenderMode::SourceAlpha, Candera::RenderMode::InverseSourceAlpha, Candera::RenderMode::Add,
            Candera::RenderMode::One, Candera::RenderMode::InverseSourceAlpha, Candera::RenderMode::Add);
      m_outputRenderModeBlending->SetInheritanceBitsToZero(RenderMode::BlendModeBit);
   }

   if (m_outputRenderModeOverwrite.PointsToNull() == false)
   {
      // Inherit everything
      m_outputRenderModeOverwrite->SetInheritanceBitsToOne(0xFFFFFFFF);

      // Except this
      m_outputRenderModeOverwrite->SetBlendingEnabled(true);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::BlendingEnabledBit);

      m_outputRenderModeOverwrite->SetDepthTestEnabled(false);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::DepthTestEnabledBit);

      m_outputRenderModeOverwrite->SetDepthWriteEnabled(false);
      m_outputRenderModeOverwrite->SetInheritanceBitsToZero(RenderMode::DepthWriteEnabledBit);
   }
}


BlurWidget3D::~BlurWidget3D()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Dtor Blur Widget: [%s]", GetLegacyName()));
   Finalize();
}


void BlurWidget3D::Finalize()
{
   if (m_blurEnabledInternal == true)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Finalize Blur Widget: [%s]", GetLegacyName()));
      ActivateContext();
      CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
      SetupCourier(false);
      SetupBlurScene(false);
      SetupRenderTarget(false);
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Skipping Finalize (disabled internal) Blur Widget: [%s]", GetLegacyName()));
   }
}


bool BlurWidget3D::IsRunningInSceneComposer() const
{
   return hmibase::view::util::SCHostUtils::IsSCHostLoaded();
}


bool BlurWidget3D::CheckForOriginalCameraModifications()
{
   bool isModified = false;

   if (m_isRunningInSceneComposer)
   {
      BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      while (itBlurCameraMap != m_blurCameraMap.end())
      {
         FeatStd::UInt32 cameraCount = Candera::Renderer::GetCameraCount();
         for (FeatStd::UInt32 cameraIndex = 0; cameraIndex < cameraCount; cameraIndex++)
         {
            OriginalCamera3D* camera = Candera::Renderer::GetCamera(cameraIndex);
            OriginalCamera3D* cameraWidget = itBlurCameraMap->first;
            if ((camera == cameraWidget) &&
                  ((camera->GetRenderTarget() != itBlurCameraMap->second.passOneGdu->ToRenderTarget3D()) ||
                   (camera->GetRenderTarget() == itBlurCameraMap->second.passTwoGdu->ToRenderTarget3D())))
            {
               isModified = true;
            }
         }
         ++itBlurCameraMap;
      }

      if (isModified)
      {
         m_isOriginalCameraModifiedBySceneComposer = true;
         CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
         SetupCourier(false);
         SetupBlurScene(false);
         SetupRenderTarget(false);
         InvalidateBlurWidget();
         m_isOriginalCameraModifiedBySceneComposer = false;
      }
   }
   return isModified;
}


Candera::RenderTarget3D* BlurWidget3D::RetrieveOwnerRenderTarget(Candera::GraphicDeviceUnit* gdu) const
{
   Candera::RenderTarget3D* result = 0;
   if (gdu != 0)
   {
      Candera::Int displayId = gdu->GetDisplay();
      Candera::Int count = 0;

      // GDU is display -> Is owner
      if ((gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::DisplayTarget2D, displayId, count))) ||
            (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::DisplayTarget3D, displayId, count))) ||
            (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DDisplayTarget, displayId, count))))
      {
         result = gdu->ToRenderTarget3D();
      } // GDU itself is offscreen render target -> Has owner -> But should not be set, as position of offscreen pixel has no relevance to touch input
      else if ((gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget2D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget3D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DOffscreenTarget, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget2Dto3D, displayId, count))) ||
               (gdu->GetUnitType() == *(DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::OffscreenTarget3Dto2D, displayId, count))))
      {
         // Do NOT set owner for offscreen RT, as it interferes with touch input position
         //result = 0;

         //fet2hi:without an owner rendering into on offscreen render target will get corrupted
         Candera::GeniviFrameBufferObject* fbo = dynamic_cast<Candera::GeniviFrameBufferObject*>(gdu);
         result = fbo != NULL ? fbo->GetProperties().GetOwner() : NULL;
      } // GDU is of unknown type
      else
      {
         result = 0;
      }
   }

   return result;
}


bool BlurWidget3D::IsCameraPartOfBlur(Candera::Camera* camera)
{
   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      return true;
   }

   return false;
}


bool BlurWidget3D::IsRenderTargetPartOfBlur(Candera::RenderTarget* renderTarget)
{
   BlurCamera3DMapType::iterator renderSetupIterator = m_blurCameraMap.begin();
   BlurCamera3DMapType::iterator renderSetupIteratorEnd = m_blurCameraMap.end();

   while (renderSetupIterator != renderSetupIteratorEnd)
   {
      if (renderTarget == renderSetupIterator->second.originalRenderTarget)
      {
         return true;
      }
      ++renderSetupIterator;
   }

   return false;
}


void BlurWidget3D::Update()
{
   if (GetBlurEnabled() == true)
   {
      // During Scene Composer update, the updated scene is first created and then the old scene deleted.
      // To first clean up the blurred scene before setting up the new blur, the setup is delayed by one frame
      // in Scene Composer.
      m_isRunningInSceneComposer = IsRunningInSceneComposer();
   }
#ifdef COURIER_ENHANCED_ENABLED
   bool blurEnabled = ((GetBlurEnabled()) && (m_viewSceneActive) && (m_viewSceneRenderingEnabled));

   if (m_isRunningInSceneComposer == true)
   {
      blurEnabled = (GetBlurEnabled());
   }
#else
   bool blurEnabled = GetBlurEnabled();
#endif
   if (blurEnabled && m_blurEnabledInternal)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Update BlurWidget: Blur running: [%s]", GetLegacyName()));
      ActivateContext();
      if (m_isRunningInSceneComposer && CheckForOriginalCameraModifications())
      {
         ETG_TRACE_ERR(("BlurWidget3D: Original camera has been modified in background: [%s]", GetLegacyName()));
      }
      else
      {
         UpdateBlurBillboardTransform();
         ValidateBlurParameters();
      }
   }
   else if (blurEnabled && (!m_blurEnabledInternal))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Update BlurWidget: Blur being enabled: [%s]", GetLegacyName()));
      ActivateContext();
      if (m_skipSetup && m_isRunningInSceneComposer)
      {
         m_skipSetup = false;
         UpdateBlurBillboardTransform();
         InvalidateBlurWidget();
      }
      else
      {
         SetupRenderTarget(true);
         UpdateBlurBillboardTransform();
         SetupBlurScene(true);

#ifdef COURIER_ENHANCED_ENABLED
         SetupCourier(true);
#endif  //#ifdef COURIER_ENHANCED_ENABLED
         InvalidateBlurWidget();
      }
   }
   else if ((!blurEnabled) && m_blurEnabledInternal)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Update BlurWidget: Blur being disabled: [%s]", GetLegacyName()));
      ActivateContext();
      CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap>(0));
#ifdef COURIER_ENHANCED_ENABLED
      SetupCourier(false);
#endif  //#ifdef COURIER_ENHANCED_ENABLED
      SetupBlurScene(false);
      SetupRenderTarget(false);
      InvalidateBlurWidget();
   }
   else
   {
      /* do nothing */
   }
}


void BlurWidget3D::OnChanged(FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);//calls triggerUpdate

   switch (propertyId)
   {
      case BlurEnabledPropertyId:
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: BlurEnabled Changed: [%s]", GetLegacyName()));
         break;
      case DarkenPropertyId:
         OnDarkenPropertyChanged();
         break;

      case BlurRadiusPropertyId:
      case BlurRadiusToSigmaPropertyId:
         m_validateBlurParameters = true;
         break;

      case BlurMaskVariantPropertyId:
         OnBlurMaskVariantPropertyChanged();
         break;

      case BillboardMaskPropertyId:
         OnBillboardMaskPropertyChanged();
         break;

      case BlurShaderVariantPropertyId:
         OnBlurShaderVariantPropertyChanged();
         break;
      case BitmapMaskPropertyId:
         OnBitmapMaskPropertyChanged();
         break;
      case BlendOutputPropertyId:
         OnBlendOutputPropertyChanged();
         break;
      case MultipassPropertyId:
         OnMultipassPropertyChanged();
         break;
   }
}


void BlurWidget3D::SetupCameraNode(OriginalNode3D* node, bool blurEnabled)
{
   if (m_isRunningInSceneComposer && (IsSceneComposerCamera(node)))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Scene Composer Blur Preview not active!: [%s]", GetLegacyName()));
   }
   else
   {
      SetupRenderTargetForCamera(node, blurEnabled);
   }
}


void BlurWidget3D::SetupRenderTarget(bool blurEnabled)
{
   /* Only enter function, if we actually need to change something */
   if (blurEnabled != m_blurEnabledInternal)
   {
      OriginalNode3D* node = GetNode();
      if (NULL != node)
      {
         if (node->GetDynamicTypeId() == OriginalCamera3D::GetTypeId())
         {
            // 3D Only: Retrieve billboard viewport for billboard mask
            if (blurEnabled == true)
            {
               Candera::Camera* camera = Dynamic_Cast<Candera::Camera*>(node);
               if (camera != 0)
               {
                  m_billboardViewport = camera->GetViewport();
               }
            }
            SetupCameraNode(node, blurEnabled);
         }
         else if (node->GetDynamicTypeId() == OriginalScene3D::GetTypeId())
         {
            OriginalNode3D* child = node->GetFirstChild();
            while (child != 0)
            {
               if (child->GetDynamicTypeId() == OriginalCamera3D::GetTypeId())
               {
                  SetupCameraNode(child, blurEnabled);
               }
               child = child->GetNextSibling();
            }
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget3D: Found no scene or camera that could be blurred!: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget3D: There is no Node assigned to the Blur Widget: [%s]", GetLegacyName()));
      }
   }
}


void BlurWidget3D::SetupRenderTargetForCamera(OriginalNode3D* child, bool blurEnabled)
{
   OriginalCamera3D* camera = Dynamic_Cast<OriginalCamera3D*>(child);

   if ((camera != 0) && blurEnabled) /* load blur render targets */
   {
      if (IsCameraAlreadyBlurred(camera))
      {
         ETG_TRACE_ERR(("BlurWidget3D: Try to blur camera multiple times: [%s]", GetLegacyName()));
         return;
      }

      OriginalRenderTarget3D* originalRT = camera->GetRenderTarget();

      if (originalRT != 0)
      {
         if (m_blurCameraMap.count(camera) == 0) /* Camera not already known */
         {
            /* RT not known -> create it */
            Candera::GraphicDeviceUnit* passOneGdu = CreateOffscreenRendertarget(camera);
            Candera::GraphicDeviceUnit* passTwoGdu = NULL;

            if (GetMultipass())
            {
               passTwoGdu = CreateOffscreenRendertarget(camera);
            }

            if (passOneGdu != 0)
            {
               m_blurEnabledInternal = true;

               RenderTargetSetup3D renderTargetSetup;
               renderTargetSetup.passOneGdu = passOneGdu;
               renderTargetSetup.passTwoGdu = passTwoGdu;
               renderTargetSetup.passOneShader = NULL;
               renderTargetSetup.passTwoShader = NULL;
               renderTargetSetup.passOneScene = NULL;
               renderTargetSetup.passTwoScene = NULL;
               renderTargetSetup.originalRenderTarget = originalRT;

               SetOriginalCamera(camera, passOneGdu->ToRenderTarget3D(), renderTargetSetup);

               std::pair<OriginalCamera3D*, RenderTargetSetup3D> pairCameraRenderTargetSetup(camera, renderTargetSetup);
               std::pair<BlurCamera3DMapType::iterator, bool> result = m_blurCameraMap.insert(pairCameraRenderTargetSetup);
               s_originalCameraVector.push_back(camera);

               if (!result.second)
               {
                  ETG_TRACE_ERR(("BlurWidget3D: Insert of the original render target into blurring GDU map failed: [%s]", GetLegacyName()));
               }
            }
         }
      }
   }
   else if ((camera != 0) && (!blurEnabled))
   {
      BlurCamera3DMapType::iterator iterator = m_blurCameraMap.begin();
      while (iterator != m_blurCameraMap.end())
      {
         ResetOriginalCamera(iterator->first);

         GraphicDeviceUnit* firstPass = iterator->second.passOneGdu;
         GraphicDeviceUnit* secondPass = iterator->second.passTwoGdu;

         if (firstPass != 0)
         {
            Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(firstPass, 0);
            firstPass->Unload();
            DevicePackageInterface::DestroyGraphicDeviceUnit(firstPass);
         }

         if (secondPass != 0)
         {
            Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(secondPass, 0);
            secondPass->Unload();
            DevicePackageInterface::DestroyGraphicDeviceUnit(secondPass);
         }

         //iterator = m_blurCameraMap.erase(iterator);//not compilable on the target
         m_blurCameraMap.erase(iterator);
         iterator = m_blurCameraMap.begin();

         //Remove camera from global blur list
         OriginalCamera3DVectorType::iterator iteratorCam = s_originalCameraVector.begin();
         while (iteratorCam != s_originalCameraVector.end())
         {
            if (*iteratorCam == camera)
            {
               iteratorCam = s_originalCameraVector.erase(iteratorCam);
            }
            else
            {
               ++iteratorCam;
            }
         }
      }

      if (m_blurCameraMap.empty())
      {
         m_blurCameraMap.clear();
         m_blurEnabledInternal = false;
      }
   }
   else
   {
      //do nothing
   }
}


bool BlurWidget3D::IsCameraAlreadyBlurred(OriginalCamera3D const* camera) const
{
   OriginalCamera3DVectorType::iterator iterator = s_originalCameraVector.begin();
   while (iterator != s_originalCameraVector.end())
   {
      if (*iterator == camera)
      {
         return true;
      }
      ++iterator;
   }
   return false;
}


Candera::GraphicDeviceUnit* BlurWidget3D::CreateOffscreenRendertarget(OriginalCamera3D const* camera)
{
   /* Create a similar render target to the one given */
   GraphicDeviceUnit* blurringGdu = 0;
   const OriginalRenderTarget3D* renderTarget = camera->GetRenderTarget();

   if (renderTarget != 0)
   {
      GraphicDeviceUnit* originalGDU = renderTarget->GetGraphicDeviceUnit();

      Int count = 0;

      if (originalGDU != 0)
      {
         const GraphicDeviceUnitTypeHandle* handle = DevicePackageDescriptor::GetUnitTypes(DevicePackageDescriptor::Mixed2D3DOffscreenTarget, originalGDU->GetDisplay(), count);

         if ((handle != 0) && (count > 0))
         {
            blurringGdu = DevicePackageInterface::CreateGraphicDeviceUnit(handle[0]);

            if (blurringGdu != NULL)
            {
               if (IsRunningInSceneComposer() == false)
               {
                  Candera::RenderTarget3D* ownerRenderTarget = RetrieveOwnerRenderTarget(originalGDU);
                  if (ownerRenderTarget != 0)
                  {
                     Candera::Internal::GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(blurringGdu, ownerRenderTarget);
                  }
               }

               Candera::Int newWidth = 0;
               Candera::Int newHeight = 0;
               GetBlurSceneSize(renderTarget, camera, newWidth, newHeight);

               FeatStd::Int subsamplingFactor = GetSubsamplingFactor();

               if (subsamplingFactor != 0)
               {
                  newWidth /= subsamplingFactor;
                  newHeight /= subsamplingFactor;
               }

               if (newWidth > 0)
               {
                  char metaInfoWidth[] = "Width\0";
                  SetMetaInfoInt(metaInfoWidth, newWidth, blurringGdu);
               }
               else
               {
                  ETG_TRACE_ERR(("BlurWidget3D: The width of the render target is not valid: [%s]", GetLegacyName()));
               }

               if (newHeight > 0)
               {
                  char metaInfoHeight[] = "Height\0";
                  SetMetaInfoInt(metaInfoHeight, newHeight, blurringGdu);
               }
               else
               {
                  ETG_TRACE_ERR(("BlurWidget3D: The height of the render target is not valid: [%s]", GetLegacyName()));
               }

               blurringGdu->SetDisplay(originalGDU->GetDisplay());

               if (!blurringGdu->Upload())
               {
                  ETG_TRACE_ERR(("BlurWidget3D: Upload of the blurring GDU failed: [%s]", GetLegacyName()));
               }
               else
               {
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Blurring GDU successfully uploaded: [%s]", GetLegacyName()));
               }
            }
            else
            {
               ETG_TRACE_ERR(("BlurWidget3D: Could create offscreen rendertarget: [%s]", GetLegacyName()));
            }
         }
         else
         {
            ETG_TRACE_ERR(("BlurWidget3D: Could not obtain handle to create offscreen rendertarget or"));
            ETG_TRACE_ERR(("The number of available graphic device units with this category on the specified display <= 0: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget3D: Rendertarget of Original Camera cannot retrieve GDU: [%s]", GetLegacyName()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("BlurWidget3D: Rendertarget of Original Camera cannot be retrieved: [%s]", GetLegacyName()));
   }
   return blurringGdu;
}


void BlurWidget3D::UpdateBlurBillboardTransform()
{
   if (m_blurEnabledInternal == true)
   {
      Candera::Rectangle boundingRectangle;
      bool billboardApplied = false;

      if ((GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask) && (GetBillboardMask()->GetDynamicTypeId() == Candera::Billboard::GetTypeId()))
      {
         Candera::Billboard* billboard = Dynamic_Cast<Candera::Billboard*>(GetBillboardMask());
         if (billboard != 0)
         {
            if (m_billboardViewport.GetWidth() > 0)
            {
               boundingRectangle.SetWidth(billboard->GetWidth() / m_billboardViewport.GetWidth());
               boundingRectangle.SetLeft(billboard->GetPosition().GetX() / m_billboardViewport.GetWidth());
            }

            if (m_billboardViewport.GetHeight() > 0)
            {
               boundingRectangle.SetHeight(billboard->GetHeight() / m_billboardViewport.GetHeight());
               boundingRectangle.SetTop(billboard->GetPosition().GetY() / m_billboardViewport.GetHeight());
            }
            billboardApplied = true;
         }
      }

      if (billboardApplied == false)
      {
         boundingRectangle.SetLeft(0.0);
         boundingRectangle.SetTop(0.0);
         boundingRectangle.SetWidth(0.0);
         boundingRectangle.SetHeight(0.0);
      }

      if (boundingRectangle != m_bitmapMaskRectangle)
      {
         FeatStd::Int subsamplingFactor = GetSubsamplingFactor();
         if (subsamplingFactor > 1)
         {
            boundingRectangle.SetLeft(boundingRectangle.GetLeft() / subsamplingFactor);
            boundingRectangle.SetTop(boundingRectangle.GetTop() / subsamplingFactor);
            boundingRectangle.SetWidth(boundingRectangle.GetWidth() / subsamplingFactor);
            boundingRectangle.SetHeight(boundingRectangle.GetHeight() / subsamplingFactor);
         }
         m_bitmapMaskRectangle = boundingRectangle;
         m_validateBlurParameters = true;
      }
   }
}


void BlurWidget3D::CreateTextureFromBitmap(FeatStd::MemoryManagement::SharedPointer<Candera::Bitmap> bitmap)
{
   if ((m_bitmapMaskTexture.PointsToNull() == false) &&
         (m_bitmapMaskTexture->IsUploaded() == true))
   {
      m_bitmapMaskTexture->Unload();
   }

   if (bitmap.PointsToNull() == false)
   {
      bool result = m_bitmapMaskTexture->SetBitmap(bitmap);
      if (result == true)
      {
         result = m_bitmapMaskTexture->Upload();

         if (result != true)
         {
            ETG_TRACE_ERR(("BlurWidget3D: Uploading blur mask texture failed: [%s]", GetLegacyName()));
         }
      }
   }
}


Candera::Scene* BlurWidget3D::SetupBlurPass(Candera::RenderTarget3D* dstRenderTarget, const Candera::RenderTarget3D* srcRenderTarget, Candera::ImageSource3D* mskRenderTarget,
      Candera::Int width, Candera::Int height, Candera::ImageSource3D* imageSource, Candera::String cameraName, Candera::Int sequenceNumber,
      IBlurShader* passShaderGenerator, bool darken)
{
   Scene* blurScene = Scene::Create();
   blurScene->SetName(cameraName.GetCString());

   Camera* blurCamera = Camera::Create();
   MemoryManagement::SharedPointer<OrthographicProjection> projection = OrthographicProjection::Create();

   projection->SetNearZ(0.001F);
   projection->SetFarZ(10.0F);

   Candera::Float projectionWidth = static_cast<Candera::Float>(width);
   projection->SetWidth(projectionWidth);
   Candera::Float projectionHeight = static_cast<Candera::Float>(height);
   projection->SetHeight(projectionHeight);

   Candera::ClearMode clearMode;
   clearMode.SetClearColor(Candera::Color(0.0F, 0.0F, 0.0F, 0.0F));
   clearMode.SetColorClearEnabled(true);
   clearMode.SetDepthClearEnabled(false);
   clearMode.SetClearDepth(1.0F);

   Candera::Rectangle rect(0.0F, 0.0F, 1.0F, 1.0F);
   blurCamera->SetProjection(projection);
   blurCamera->SetName(cameraName.GetCString());
   blurCamera->SetSequenceNumber(sequenceNumber);
   blurCamera->SetRenderTarget(dstRenderTarget);
   blurCamera->SetViewport(rect);
   blurCamera->SetClearMode(clearMode);
   blurCamera->SetSwapEnabled(true);
   blurCamera->SetRenderingEnabled(true);

   blurCamera->SetAppearance(Candera::Appearance::Create());
   FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> cameraAppearance = blurCamera->GetAppearance();
   cameraAppearance->SetRenderMode(m_outputRenderModeOverwrite);

   Candera::Float billboardWidth = static_cast<Candera::Float>(width);
   Candera::Float billboardHeight = static_cast<Candera::Float>(height);
   Billboard* blurBillboard = Billboard::Create(billboardWidth, billboardHeight);
   Vector3 position(0.0F, 0.0F, -1.0F);
   blurBillboard->SetPosition(position);

   FeatStd::MemoryManagement::SharedPointer<MultiPassAppearance> appearance = MultiPassAppearance::Create();
   FeatStd::MemoryManagement::SharedPointer<Texture> proxyTex = Texture::Create();
   FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> proxyTexImg = ProxyTextureImage::Create(imageSource);

   proxyTex->SetWrapModeU(Texture::ClampToEdge);
   proxyTex->SetWrapModeV(Texture::ClampToEdge);
   proxyTex->SetTextureImage(proxyTexImg);

   if (!appearance->SetTexture(proxyTex, 0))
   {
      ETG_TRACE_ERR(("BlurWidget3D: Setting the blur source texture failed: [%s]", GetLegacyName()));
   }
   bool applyMask = false;

   if ((NULL != srcRenderTarget) && (NULL != mskRenderTarget))
   {
      Candera::ImageSource3D* mskImage = mskRenderTarget;
      FeatStd::MemoryManagement::SharedPointer<Texture> mskTex = Texture::Create();
      FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> mskTexImg = ProxyTextureImage::Create(mskImage);

      mskTex->SetWrapModeU(Texture::ClampToEdge);
      mskTex->SetWrapModeV(Texture::ClampToEdge);
      mskTex->SetTextureImage(mskTexImg);

      if (!appearance->SetTexture(mskTex, m_maskTextureUnit))
      {
         ETG_TRACE_ERR(("BlurWidget3D: Setting the mask texture unit failed: [%s]", GetLegacyName()));
      }
      Candera::ImageSource3D* srcImage = srcRenderTarget->GetGraphicDeviceUnit()->ToImageSource3D();
      FeatStd::MemoryManagement::SharedPointer<Texture> srcTex = Texture::Create();
      FeatStd::MemoryManagement::SharedPointer<ProxyTextureImage> srcTexImg = ProxyTextureImage::Create(srcImage);

      srcTex->SetWrapModeU(Texture::ClampToEdge);
      srcTex->SetWrapModeV(Texture::ClampToEdge);
      srcTex->SetTextureImage(srcTexImg);

      if (!appearance->SetTexture(srcTex, m_sourceTextureUnit))
      {
         ETG_TRACE_ERR(("BlurWidget3D: Setting the source texture unit failed: [%s]", GetLegacyName()));
      }
      applyMask = true;
   }

   if (!SetupBlurShader(passShaderGenerator, width, height, darken, applyMask))
   {
      ETG_TRACE_ERR(("BlurWidget3D: Setup of blur shader for render pass failed: [%s]", GetLegacyName()));
   }
   appearance->SetShader(passShaderGenerator->GetShader());
   appearance->SetShaderParamSetter(passShaderGenerator->GetShaderParams());

   blurBillboard->SetAppearance(appearance);
   bool result = blurScene->AddChild(blurCamera);

   if (result == false)
   {
      return NULL;
   }

   result = blurScene->AddChild(blurBillboard);

   if (result == false)
   {
      return NULL;
   }

   result = blurScene->UploadAll();

   if (result == false)
   {
      return NULL;
   }

   return blurScene;
}


void BlurWidget3D::GetBlurSceneSize(OriginalRenderTarget3D const* renderTarget, OriginalCamera3D const* camera, Candera::Int& width, Candera::Int& height)
{
   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("BlurWidget3D: Missing render target: [%s]", GetLegacyName()));
   }

   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget3D: Missing camera: [%s]", GetLegacyName()));
   }

   WindowRect viewport;
   GetViewport(camera, renderTarget, viewport);
   width = static_cast<Candera::Int>(viewport.GetRectInPixel().GetWidth());
   height = static_cast<Candera::Int>(viewport.GetRectInPixel().GetHeight());
}


void BlurWidget3D::RemoveRendertargetsFromBillboard(Candera::Node* node)
{
   while (node != 0)
   {
      if (node->GetDynamicTypeId() == Candera::Billboard::GetTypeId())
      {
         Candera::Billboard* billboard = Dynamic_Cast<Candera::Billboard*>(node);
         if (billboard != 0)
         {
            billboard->SetAppearance(FeatStd::MemoryManagement::SharedPointer<Candera::Appearance>(0));
         }
      }

      RemoveRendertargetsFromBillboard(node->GetFirstChild());
      node = node->GetNextSibling();
   }
}


void BlurWidget3D::SetupBlurScene(bool blurEnabled)
{
   if (blurEnabled && (!m_blurCameraMap.empty()))
   {
      BlurCamera3DMapType::iterator iterator = m_blurCameraMap.begin();
      BlurCamera3DMapType::iterator iteratorEnd = m_blurCameraMap.end();

      while (iterator != iteratorEnd)
      {
         if ((iterator->second.passOneScene == NULL) && (iterator->second.passTwoScene == NULL))
         {
            RenderTarget3D*         renderTargetFirstPass = 0;
            RenderTarget3D*         renderTargetSecondPass = 0;
            ImageSource3D*          imageSourceFirstPass = 0;
            ImageSource3D*          imageSourceSecondPass = 0;
            RenderTarget3D*         originalRenderTarget = iterator->second.originalRenderTarget->GetGraphicDeviceUnit()->ToRenderTarget3D();
            GraphicDeviceUnit*      gduFirstPass = iterator->second.passOneGdu;
            GraphicDeviceUnit*      gduSecondPass = iterator->second.passTwoGdu;

            if (gduFirstPass != 0)
            {
               renderTargetFirstPass = gduFirstPass->ToRenderTarget3D();
               imageSourceFirstPass = gduFirstPass->ToImageSource3D();

               if (GetMultipass() && (gduSecondPass != 0))
               {
                  renderTargetSecondPass = gduSecondPass->ToRenderTarget3D();
                  imageSourceSecondPass = gduSecondPass->ToImageSource3D();
               }
            }

            if ((0 != originalRenderTarget) && (0 != renderTargetFirstPass) && (0 != imageSourceFirstPass))
            {
               Candera::Int16 cameraSequenceNumber = static_cast<Candera::Int16>(iterator->first->GetSequenceNumber());

               // The original camera sequence number gets lowered by 2 (there is no difference regarding this between single and multipass
               // Thus, multipass number (worst case) is chosen
               iterator->first->SetSequenceNumber(cameraSequenceNumber - 2);

               iterator->second.passOneCameraName = GetBlurSceneName(iterator->first->GetName(), 1);

               CreateShaderType(&iterator->second.passOneShader, &iterator->second.passTwoShader, GetBlurShaderVariant(), GetMultipass());

               Candera::Int width = 0;
               Candera::Int height = 0;
               GetBlurSceneSize(iterator->second.originalRenderTarget, (*iterator).first, width, height);
               Candera::ImageSource3D* imagesourceMask = 0;

               if (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)
               {
                  CreateTextureFromBitmap(GetBitmapMask());
                  if (m_bitmapMaskTexture.PointsToNull() == false)
                  {
                     imagesourceMask = m_bitmapMaskTexture->ToImageSource3D();
                  }

                  if (imagesourceMask == 0)
                  {
                     ETG_TRACE_ERR(("BlurWidget3D: Bitmap Blur Mask could not be retrieved as ImageSource3D: [%s]", GetLegacyName()));
                  }
               }
               else if (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)
               {
                  OriginalRenderTarget3D* renderTargetMask = GetRenderTargetMask();
                  if (renderTargetMask != 0)
                  {
                     if (renderTargetMask->GetGraphicDeviceUnit() != 0)
                     {
                        imagesourceMask = renderTargetMask->GetGraphicDeviceUnit()->ToImageSource3D();
                     }
                  }

                  if (imagesourceMask == 0)
                  {
                     ETG_TRACE_ERR(("BlurWidget3D: Rendertarget Blur Mask could not be retrieved as ImageSource3D: [%s]", GetLegacyName()));
                  }
               }
               else
               {
                  // Do nothing (no mask applied)
               }

               if (GetMultipass())
               {
                  // The first pass of a multipass process needs to render after the original camera, but before the second pass camera
                  iterator->second.passOneScene = SetupBlurPass(renderTargetSecondPass, NULL, NULL, width, height, imageSourceFirstPass, iterator->second.passOneCameraName, cameraSequenceNumber - 1, iterator->second.passOneShader, false);
               }
               else
               {
                  // In single pass, the blur pass receives the original camera sequence number, so the resulting draw order is the same as with the original camera
                  iterator->second.passOneScene = SetupBlurPass(originalRenderTarget, renderTargetFirstPass, imagesourceMask, width, height, imageSourceFirstPass, iterator->second.passOneCameraName, cameraSequenceNumber, iterator->second.passOneShader, true);
                  Candera::Camera* passOneCamera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(iterator->second.passOneScene, Candera::Camera::GetTypeId()));
                  if (NULL != passOneCamera)
                  {
                     SetFinalPassCamera(iterator->first, passOneCamera);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("BlurWidget3D: Final camera for single pass is missing (Null Pointer): [%s]", GetLegacyName()));
                  }
               }

               /* Scene setup second pass */
               if (GetMultipass() && (0 != renderTargetSecondPass) && (0 != imageSourceSecondPass))
               {
                  // In multipass rendering, the second pass is to be rendered at the same sequence number as the original camera (so the render order stays the same)
                  iterator->second.passTwoCameraName = GetBlurSceneName(iterator->first->GetName(), 2);

                  iterator->second.passTwoScene = SetupBlurPass(originalRenderTarget, renderTargetFirstPass, imagesourceMask, width, height, imageSourceSecondPass, iterator->second.passTwoCameraName, cameraSequenceNumber, iterator->second.passTwoShader, true);
                  Candera::Camera* passTwoCamera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(iterator->second.passTwoScene, Candera::Camera::GetTypeId()));
                  if (NULL != passTwoCamera)
                  {
                     SetFinalPassCamera(iterator->first, passTwoCamera);
                  }
                  else
                  {
                     ETG_TRACE_ERR(("BlurWidget3D: Final camera for two pass is missing (Null Pointer): [%s]", GetLegacyName()));
                  }
               }
            }
         }
         ++iterator;
      }

      // Shader setup now correct
      m_validateBlurParameters = false;
      m_validateShaderType = false;
   }

   if (blurEnabled == false)
   {
      BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      BlurCamera3DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

      while (itBlurCameraMap != itBlurCameraMapEnd)
      {
         Scene* scene = itBlurCameraMap->second.passOneScene;
         if (NULL != scene)
         {
            Node* node = scene->GetFirstChild();
            RemoveRendertargetsFromBillboard(node);

            if (!scene->UnloadAll())
            {
               ETG_TRACE_ERR(("BlurWidget3D: Unload All failed: [%s]", GetLegacyName()));
            }
            scene->Dispose();
         }
         itBlurCameraMap->second.passOneScene = NULL;

         scene = itBlurCameraMap->second.passTwoScene;
         if (NULL != scene)
         {
            if (!scene->UnloadAll())
            {
               ETG_TRACE_ERR(("BlurWidget3D: Unload All failed: [%s]", GetLegacyName()));
            }
            scene->Dispose();
         }
         itBlurCameraMap->second.passTwoScene = NULL;

         delete itBlurCameraMap->second.passOneShader;
         itBlurCameraMap->second.passOneShader = NULL;

         delete itBlurCameraMap->second.passTwoShader;
         itBlurCameraMap->second.passTwoShader = NULL;

         ++itBlurCameraMap;
      }
   }
}


bool BlurWidget3D::ValidateShaderType()
{
   bool result = true;

   BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera3DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      CreateShaderType(&itBlurCameraMap->second.passOneShader, &itBlurCameraMap->second.passTwoShader, GetBlurShaderVariant(), GetMultipass());

      if ((NULL == itBlurCameraMap->second.passOneShader) || (GetMultipass() && (NULL == itBlurCameraMap->second.passTwoShader)))
      {
         ETG_TRACE_ERR(("BlurWidget3D: The validation of a shader type failed: [%s]", GetLegacyName()));
         result = false;
      }
      ++itBlurCameraMap;
   }
   return result;
}


bool BlurWidget3D::ValidateBlurShaders()
{
   bool result = true;

   BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera3DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      /* Search for Pass One Billboard to apply the shader */
      Billboard* passOneBillboard = Dynamic_Cast<Billboard*>(GetFirstChildNodeOfType(itBlurCameraMap->second.passOneScene, Billboard::GetTypeId()));

      if (NULL == passOneBillboard)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The billboard for the first pass blur scene is missing: [%s]", GetLegacyName()));
         return false;
      }
      Candera::Int width = static_cast<Candera::Int>(passOneBillboard->GetWidth());
      Candera::Int height = static_cast<Candera::Int>(passOneBillboard->GetHeight());

      bool applyMask = ((((GetRenderTargetMask() != 0) && (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)) ||
                         ((m_bitmapMaskTexture.PointsToNull() == false) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask))) &&
                        (GetMultipass() == false));

      if (!SetupBlurShader(itBlurCameraMap->second.passOneShader, width, height, !GetMultipass(), applyMask))
      {
         ETG_TRACE_ERR(("BlurWidget3D: Setup of blur shader for first pass failed: [%s]", GetLegacyName()));
         result = false;
      }
      SharedPointerShaderType passOneShader = itBlurCameraMap->second.passOneShader->GetShader();

      passOneBillboard->GetAppearance()->SetShader(passOneShader);
      passOneBillboard->GetAppearance()->SetShaderParamSetter(itBlurCameraMap->second.passOneShader->GetShaderParams());

      //Activate the camera to make sure the upload is done into the right rendering context.
      itBlurCameraMap->second.passOneGdu->ToRenderTarget3D()->Activate();
      if ((!passOneShader->IsUploaded()) && (!passOneShader->Upload()))
      {
         ETG_TRACE_ERR(("BlurWidget3D: Upload of the first pass blur shader failed: [%s]", GetLegacyName()));
         result = false;
      }

      /* Pass Two */
      if (GetMultipass())
      {
         /* Search for Pass Two Billboard to apply the shader */
         Billboard* passTwoBillboard = Dynamic_Cast<Billboard*>(GetFirstChildNodeOfType(itBlurCameraMap->second.passTwoScene, Billboard::GetTypeId()));

         if (NULL == passTwoBillboard)
         {
            ETG_TRACE_ERR(("BlurWidget3D: The billboard of the second pass blur scene is missing: [%s]", GetLegacyName()));
            return false;
         }
         Candera::Int width = static_cast<Candera::Int>(passTwoBillboard->GetWidth());
         Candera::Int height = static_cast<Candera::Int>(passTwoBillboard->GetHeight());

         bool applyMask = (((GetRenderTargetMask() != 0) && (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)) ||
                           ((m_bitmapMaskTexture.PointsToNull() == false) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)));

         if (!SetupBlurShader(itBlurCameraMap->second.passTwoShader, width, height, true, applyMask))
         {
            ETG_TRACE_ERR(("BlurWidget3D: Setup of blur shader for second pass failed: [%s]", GetLegacyName()));
            result = false;
         }
         SharedPointerShaderType passTwoShader = itBlurCameraMap->second.passTwoShader->GetShader();

         passTwoBillboard->GetAppearance()->SetShader(passTwoShader);
         passTwoBillboard->GetAppearance()->SetShaderParamSetter(itBlurCameraMap->second.passTwoShader->GetShaderParams());
         //Activate the camera to make sure the upload is done into the right rendering context.
         itBlurCameraMap->second.passTwoGdu->ToRenderTarget3D()->Activate();

         if ((!passTwoShader->IsUploaded()) && (!passTwoShader->Upload()))
         {
            ETG_TRACE_ERR(("BlurWidget3D: Upload of the second pass blur shader failed: [%s]", GetLegacyName()));
            result = false;
         }
      }
      ++itBlurCameraMap;
   }
   return result;
}


void BlurWidget3D::ValidateBlurParameters()
{
   bool invalidateWidget = false;

   if (m_validateShaderType)
   {
      if (ValidateShaderType())
      {
         m_validateShaderType = false;
      }
      invalidateWidget = true;
   }

   if (m_validateBlurParameters)
   {
      if (ValidateBlurShaders())
      {
         m_validateBlurParameters = false;
      }
      invalidateWidget = true;
   }

   //To force redraw
   if (invalidateWidget)
   {
      InvalidateBlurWidget();
   }
}


bool BlurWidget3D::SetupBlurShader(IBlurShader* shaderGenerator, Candera::Int newWidth, Candera::Int newHeight, bool darken, bool mask)
{
   /* Configure the shader */
   Candera::Int32 blurRadius = GetBlurRadius() / GetSubsamplingFactor();
   if (blurRadius < m_minRadius)
   {
      blurRadius = m_minRadius;
   }

   Candera::Float sigma = static_cast<Candera::Float>(blurRadius) * GetBlurRadiusToSigma();

   shaderGenerator->SetRadius(blurRadius);
   shaderGenerator->SetSigma(sigma);
   shaderGenerator->SetTextureWidth(newWidth);
   shaderGenerator->SetTextureHeight(newHeight);

   bool maskEnabled = GetBlurMaskVariant() != enBlurMaskVariant::NoBlurMask;

   shaderGenerator->SetBlurMaskRectangle(m_bitmapMaskRectangle, GetKeepFrameBufferContent(), maskEnabled);

   if (darken)
   {
      shaderGenerator->SetDarken(m_darkenUniform);
   }
   else
   {
      shaderGenerator->SetDarken(m_noDarkenUniform);
   }

   if (mask)
   {
      shaderGenerator->SetMaskTextureUnit(m_maskTextureUnit, GetCombineMask());
      shaderGenerator->SetSourceTextureUnit(m_sourceTextureUnit);
   }
   else
   {
      shaderGenerator->SetMaskTextureUnit(-1, false);
      shaderGenerator->SetSourceTextureUnit(-1);
   }
   return true;
}


void BlurWidget3D::SetOriginalCamera(OriginalCamera3D* camera, OriginalRenderTarget3D* renderTarget, RenderTargetSetup3D& renderTargetSetup)
{
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget3D: Camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   if (NULL == renderTarget)
   {
      ETG_TRACE_ERR(("BlurWidget3D: Render Target is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   GetViewport(camera, renderTargetSetup.backupViewportRect);
   GetScissor(camera, renderTargetSetup.backupScissorRect);

   Candera::Int renderTargetWidth = 0;
   Candera::Int renderTargetHeight = 0;
   GetBlurSceneSize(renderTargetSetup.originalRenderTarget, camera, renderTargetWidth, renderTargetHeight);

   FeatStd::Int subsamplingFactor = GetSubsamplingFactor();
   renderTargetWidth /= subsamplingFactor;
   renderTargetHeight /= subsamplingFactor;

   WindowRect scissorRect;
   GetScissor(camera, scissorRect);
   Candera::Rectangle pixelScissorRect = scissorRect.GetRectInPixel();

   WindowRect viewportRect;
   GetViewport(camera, viewportRect);
   Candera::Rectangle pixelViewportRect = viewportRect.GetRectInPixel();

   // Shift scissor and viewport to 0/0 pixelViewportRect
   FeatStd::Float left = pixelScissorRect.GetLeft() - pixelViewportRect.GetLeft();
   if (left < 0.0F)
   {
      left = 0.0F;
   }
   pixelScissorRect.SetLeft(left);

   FeatStd::Float top = pixelScissorRect.GetTop() - pixelViewportRect.GetTop();
   if (top < 0.0F)
   {
      top = 0.0F;
   }
   pixelScissorRect.SetTop(top);

   pixelScissorRect.SetWidth(pixelScissorRect.GetWidth() / subsamplingFactor);
   pixelScissorRect.SetHeight(pixelScissorRect.GetHeight() / subsamplingFactor);

   pixelViewportRect.SetLeft(0.0F);
   pixelViewportRect.SetTop(0.0F);
   pixelViewportRect.SetWidth(pixelViewportRect.GetWidth() / subsamplingFactor);
   pixelViewportRect.SetHeight(pixelViewportRect.GetHeight() / subsamplingFactor);

   scissorRect.SetRectInPixel(pixelScissorRect, static_cast<Candera::Float>(renderTargetWidth), static_cast<Candera::Float>(renderTargetHeight));
   viewportRect.SetRectInPixel(pixelViewportRect, static_cast<Candera::Float>(renderTargetWidth), static_cast<Candera::Float>(renderTargetHeight));

   SetViewport(camera, viewportRect);
   SetScissor(camera, scissorRect);

   if (NULL != camera)
   {
      // Finally change the render target of the original camera to the new render target
      camera->SetRenderTarget(renderTarget);
      renderTargetSetup.originalClearMode = camera->GetClearMode();

      // Since we set the original camera to a lower sequence number, we want to keep a backup of the original number
      renderTargetSetup.backupSequenceNumber = static_cast<Candera::Int16>(camera->GetSequenceNumber());
   }
}


void BlurWidget3D::ResetOriginalCamera(OriginalCamera3D* camera)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Reset Original Camera: [%s]", GetLegacyName()));
   if (NULL == camera)
   {
      ETG_TRACE_ERR(("BlurWidget3D: Camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }
   else
   {
      RenderTargetSetup3D renderTargetSetup = m_blurCameraMap[camera];

      SetViewport(camera, renderTargetSetup.backupViewportRect);
      SetScissor(camera, renderTargetSetup.backupScissorRect);

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Reset Original Camera: modifiedBySC=%u, origRT=%p, blurRT=%p, camera=%s",
                          m_isOriginalCameraModifiedBySceneComposer, renderTargetSetup.originalRenderTarget,
                          camera->GetRenderTarget(), camera->GetName() != NULL ? camera->GetName() : "null"));
      if (!m_isOriginalCameraModifiedBySceneComposer)
      {
         Courier::Gdu(0).SetActivate(true);//fet2hi:force increment the invalidation id

         camera->SetRenderTarget(renderTargetSetup.originalRenderTarget);
         camera->SetClearMode(renderTargetSetup.originalClearMode);
         camera->SetSequenceNumber(renderTargetSetup.backupSequenceNumber);
      }
   }
}


void BlurWidget3D::SetFinalPassCamera(OriginalCamera3D* originalCamera, Candera::Camera* finalCamera)
{
   if (NULL == originalCamera)
   {
      ETG_TRACE_ERR(("BlurWidget3D: The camera is missing (Null Pointer): [%s]", GetLegacyName()));
      return;
   }

   if (NULL == finalCamera)
   {
      ETG_TRACE_ERR(("BlurWidget3D: The camera is missing (Null Pointer): [%s]", GetLegacyName()));
   }

   RenderTargetSetup3D renderTargetSetup = m_blurCameraMap[originalCamera];

   SetViewport(finalCamera, renderTargetSetup.backupViewportRect);
   SetScissor(finalCamera, renderTargetSetup.backupScissorRect);

   if (!(m_outputRenderModeBlending.PointsToNull() || (NULL == finalCamera)))
   {
      if (finalCamera->GetAppearance().PointsToNull() == true)
      {
         finalCamera->SetAppearance(Candera::Appearance::Create());
      }

      if (GetBlendOutput() == true)
      {
         FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> appearance = finalCamera->GetAppearance();
         appearance->SetRenderMode(m_outputRenderModeBlending);
      }
      else
      {
         FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> appearance = finalCamera->GetAppearance();
         appearance->SetRenderMode(m_outputRenderModeOverwrite);
      }
   }

   CopyCameraSettings(originalCamera, finalCamera, false, false);

   //enable color clear so that the first pass buffer is cleared before each rendering
   Candera::ClearMode clearMode = originalCamera->GetClearMode();
   clearMode.SetColorClearEnabled(true);
   originalCamera->SetClearMode(clearMode);
}


#ifdef COURIER_ENHANCED_ENABLED
void BlurWidget3D::AddBlurPassGdus()
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      Courier::ViewHandler* blurViewHandler = dynamic_cast<Courier::ViewHandler*>(viewScene->GetViewHandler());

      if (NULL == blurViewHandler)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The View Handler for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      //BlurRenderer* blurRenderer = dynamic_cast<BlurRenderer*>( blurViewHandler->GetRenderer());
      hmibase::view::Renderer* blurRenderer = dynamic_cast<hmibase::view::Renderer*>(blurViewHandler->GetRenderer());

      if (NULL == blurRenderer)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The Renderer for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
      BlurCamera3DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

      while (itBlurCameraMap != itBlurCameraMapEnd)
      {
         GraphicDeviceUnit* firstPass = itBlurCameraMap->second.passOneGdu;
         GraphicDeviceUnit* secondPass = itBlurCameraMap->second.passTwoGdu;

         if (NULL != firstPass)
         {
            if (!blurRenderer->AddGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Adding pass-one GDU failed: [%s]", GetLegacyName()));
            }

            if (!blurRenderer->LoadGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Loading pass-one GDU failed: [%s]", GetLegacyName()));
            }
         }

         if (GetMultipass() && (NULL != secondPass))
         {
            if (!blurRenderer->AddGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Adding pass-two GDU failed: [%s]", GetLegacyName()));
            }

            if (!blurRenderer->LoadGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Loading pass-two GDU failed: [%s]", GetLegacyName()));
            }
         }
         ++itBlurCameraMap;
      }
   }
}


void BlurWidget3D::RemoveBlurPassGdus()
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      Courier::ViewHandler* blurViewHandler = dynamic_cast<Courier::ViewHandler*>(viewScene->GetViewHandler());

      if (NULL == blurViewHandler)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The View Handler for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      //BlurRenderer* blurRenderer = dynamic_cast<BlurRenderer*>( blurViewHandler->GetRenderer());
      hmibase::view::Renderer* blurRenderer = dynamic_cast<hmibase::view::Renderer*>(blurViewHandler->GetRenderer());

      if (NULL == blurRenderer)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The Renderer for the blur scene is missing: [%s]", GetLegacyName()));
         return;
      }

      BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();

      while (itBlurCameraMap != m_blurCameraMap.end())
      {
         GraphicDeviceUnit* firstPass = itBlurCameraMap->second.passOneGdu;
         GraphicDeviceUnit* secondPass = itBlurCameraMap->second.passTwoGdu;

         if (NULL != firstPass)
         {
            if (!blurRenderer->RemoveGdu(firstPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Removing pass-one GDU failed: [%s]", GetLegacyName()));
            }
         }

         if (GetMultipass() && (NULL != secondPass))
         {
            if (!blurRenderer->RemoveGdu(secondPass))
            {
               ETG_TRACE_ERR(("BlurWidget3D: Removing pass-two GDU failed: [%s]", GetLegacyName()));
            }
         }
         ++itBlurCameraMap;
      }
   }
}


void BlurWidget3D::SetupCourier(bool inBlurEnabled)
{
   Courier::View* viewScene = GetParentView();

   if (NULL != viewScene)
   {
      BlurViewScene3D* blurViewScene = dynamic_cast<BlurViewScene3D*>(viewScene);

      if (NULL == blurViewScene)
      {
         ETG_TRACE_ERR(("BlurWidget3D: The Blur View Scene is missing: [%s]", GetLegacyName()));
         return;
      }

      if (inBlurEnabled)
      {
         blurViewScene->AddBlurWidget(this);
      }
      else
      {
         blurViewScene->RemoveBlurWidget(this);
      }
   }
}


void BlurWidget3D::OnParentViewActivate(bool activate)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: OnParentViewActivate = %5s: [%s]", activate ? "true" : "false", GetLegacyName()));
   if (m_viewSceneActive != activate)
   {
      m_viewSceneActive = activate;

      if (((m_viewSceneActive == false) || (m_viewSceneRenderingEnabled == false)) && (m_blurEnabledInternal == true))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: OnParentViewActivate -> Disabling Blur Widget [%s]", GetLegacyName()));
         Finalize();
         m_updateAfterInactive = true;
      }
   }
}


void BlurWidget3D::OnParentViewRenderingEnabled(bool enable)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: OnParentViewRenderingEnabled = %5s: [%s]", enable ? "true" : "false", GetLegacyName()));
   if (m_viewSceneRenderingEnabled != enable)
   {
      m_viewSceneRenderingEnabled = enable;

      if (((m_viewSceneActive == false) || (m_viewSceneRenderingEnabled == false)) && (m_blurEnabledInternal == true))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: OnParentViewRenderingEnabled -> Disabling Blur Widget [%s]", GetLegacyName()));
         Finalize();
         m_updateAfterInactive = true;
      }
   }
}


Camera3DVectorType BlurWidget3D::GetBlurCameras(OriginalCamera3D* camera, Candera::Int passIndex)
{
   Camera3DVectorType result;

   if ((passIndex <= 0) || (passIndex > 2))
   {
      ETG_TRACE_ERR(("BlurWidget3D: Invalid blur pass index: [%s]", GetLegacyName()));
      return result;
   }

   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      Candera::Scene* passOneScene = m_blurCameraMap[camera].passOneScene;
      Candera::Scene* passTwoScene = m_blurCameraMap[camera].passTwoScene;

      if (1 == passIndex)
      {
         Candera::Camera* camera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(passOneScene, Camera::GetTypeId()));

         if (NULL != camera)
         {
            result.push_back(camera);
         }
      }
      else if ((2 == passIndex) && (NULL != passTwoScene))
      {
         Candera::Camera* camera = Dynamic_Cast<Candera::Camera*>(GetFirstChildNodeOfType(passTwoScene, Camera::GetTypeId()));

         if (NULL != camera)
         {
            result.push_back(camera);
         }
      }
   }
   return result;
}


OriginalCamera3DVectorType BlurWidget3D::GetOriginalCameras()
{
   OriginalCamera3DVectorType result;
   BlurCamera3DMapType::iterator itBlurCameraMap = m_blurCameraMap.begin();
   BlurCamera3DMapType::iterator itBlurCameraMapEnd = m_blurCameraMap.end();

   while (itBlurCameraMap != itBlurCameraMapEnd)
   {
      result.push_back(itBlurCameraMap->first);
      ++itBlurCameraMap;
   }

   return result;
}


hmibase::widget::blur::WindowRect BlurWidget3D::GetOriginalCameraViewport(OriginalCamera3D* camera)
{
   if ((camera != 0) && (m_blurCameraMap.count(camera) > 0))
   {
      return m_blurCameraMap[camera].backupViewportRect;
   }

   return WindowRect();
}


#endif // COURIER_ENHANCED_ENABLED


void BlurWidget3D::SetMetaInfoInt(char* item, Candera::Int newValue, GraphicDeviceUnit* blurringGdu)
{
   const MetaInfo::GraphicDeviceUnitMetaInfo* metaInfoBlurring = DevicePackageDescriptor::GetMetaInformation(blurringGdu->GetUnitType());

   Candera::Char temp[25];
   if (0 < Candera::StringPlatform::StringPrintf(temp, sizeof(temp), "%d", newValue))
   {
      if (NULL != metaInfoBlurring->LookupItem(item))
      {
         if (!metaInfoBlurring->LookupItem(item)->Set(blurringGdu, temp))
         {
            ETG_TRACE_ERR(("BlurWidget3D: Setting a value of the off-screen render target failed: [%s]", GetLegacyName()));
         }
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget3D: Meta Info item not found: [%s]", GetLegacyName()));
      }
   }
   else
   {
      ETG_TRACE_ERR(("BlurWidget3D: Writing the height of an render target into char array failed: [%s]", GetLegacyName()));
   }
}


void BlurWidget3D::OnDarkenPropertyChanged()
{
   FeatStd::Float darken = GetDarken();
   if (fabs(m_darkenUniform[0] - darken) > FLT_EPSILON)
   {
      m_darkenUniform[0] = darken;
      m_darkenUniform[1] = darken;
      m_darkenUniform[2] = darken;
      m_darkenUniform[3] = 1.0F;
      m_validateBlurParameters = true;
   }
}


void BlurWidget3D::OnBlurMaskVariantPropertyChanged()
{
   // Reset everything when blur mask variant changes
   Finalize();

   if (GetBlurMaskVariant() == enBlurMaskVariant::RenderTargetBlurMask)
   {
      SetSubsamplingFactor(1);
   }
   else if (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask)
   {
      SetSubsamplingFactor(1);
   }
   else if (GetBlurMaskVariant() == enBlurMaskVariant::NoBlurMask)
   {
      // do nothing
   }
   else
   {
      // Unknown variant -> do nothing
   }
}


void BlurWidget3D::OnBillboardMaskPropertyChanged()
{
   Candera::Node* billboardMask = GetBillboardMask();

   if (billboardMask != NULL)
   {
      if (billboardMask->GetDynamicTypeId() == Candera::Billboard::GetTypeId())
      {
         if (m_oldBillboardMask != 0)
         {
            m_oldBillboardMask->SetRenderingEnabled(m_originalNodeRenderingEnabled);
         }

         UpdateBlurBillboardTransform();

         m_originalNodeRenderingEnabled = billboardMask->IsRenderingEnabled();
         billboardMask->SetRenderingEnabled(false);

         m_oldBillboardMask = billboardMask;
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget3D: Added Node ist not of type billboard for Billboard Mask [%s]", GetLegacyName()));
      }
   }
}


void BlurWidget3D::OnBlurShaderVariantPropertyChanged()
{
   enShaderVariant::Enum shaderVariant = GetBlurShaderVariant();
   if ((enShaderVariant::DynamicShader == shaderVariant) && (!GetMultipass()))
   {
      ETG_TRACE_ERR(("BlurWidget3D: The Dynamic Shader does not support single pass blur: [%s]", GetLegacyName()));
   }

   if ((enShaderVariant::StaticShader == shaderVariant) && (GetBlurMaskVariant() == enBlurMaskVariant::BitmapBlurMask))
   {
      ETG_TRACE_ERR(("BlurWidget3D: Bitmap Blurring is not supported with the static shader variant: [%s]", GetLegacyName()));
   }

   m_validateShaderType = true;
}


void BlurWidget3D::OnBitmapMaskPropertyChanged()
{
   CreateTextureFromBitmap(GetBitmapMask());
   UpdateBlurBillboardTransform();
}


void BlurWidget3D::OnBlendOutputPropertyChanged()
{
   // As a basic setting is changed, the widget has to be reinitialized
   Finalize();
}


void BlurWidget3D::OnMultipassPropertyChanged()
{
   // As a basic setting is changed, the widget has to be reinitialized
   Finalize();
}


void BlurWidget3D::InvalidateBlurWidget()
{
   // On property changes of the widget, the blur has to be redrawn from the beginning, as all parameters except
   // darken need to redo the first blur pass
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Invalidate BlurWidget: [%s]", GetLegacyName()));

   FeatStd::Optional<Candera::Rectangle> dirtyArea;
   bool dirtyAreaInvalidationEnabled = GetUseManualDirtyArea();
   bool blurEnabled = GetBlurEnabled();
   if (blurEnabled == true)
   {
      if (dirtyAreaInvalidationEnabled == true)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: Invalidate BlurWidget: Manual Dirty Area: [%s]", GetLegacyName()));
         dirtyArea = GetDirtyAreaRectangle();
      }
   }

#ifdef COURIER_ENHANCED_ENABLED
   if (GetParentView() != 0)
   {
      GetParentView()->Invalidate(dirtyArea);
   }
#else
   Invalidate();
#endif
}


void BlurWidget3D::ActivateContext()
{
   OriginalCamera3D* camera = Dynamic_Cast<OriginalCamera3D*>(GetNode());
   if (camera != 0)
   {
      if ((camera->GetRenderTarget() != 0) && (camera->GetRenderTarget()->Activate()))
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "BlurWidget3D: ActivateContext: [%s]", GetLegacyName()));
      }
      else
      {
         ETG_TRACE_ERR(("BlurWidget3D: ActivateContext failed: [%s]", GetLegacyName()));
      }
   }
}


}   /* hmibase */
}   /* widget */
}   /* blur */
