/* ***************************************************************************************
* FILE:          LineListWidget3D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LineListWidget3D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

/** /
* This is a 3D widget which will draw lines in a 3D view. The inputs to this widget will be a set of points
* representing the X and Y coordinates.
*/
#if !defined(LINELIST_WIDGET3D_H)
#define LINELIST_WIDGET3D_H

#include "Candera/Engine3D/Core/VertexGeometryBuilder.h"
#include "Candera/Engine3D/Core/VertexBuffer.h"

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(enDrawingMode)   \
    ENUM_DATA_TYPE_ITEM(CurveMode)        \
    ENUM_DATA_TYPE_ITEM(StraightMode)     \
    ENUM_DATA_TYPE_END(enDrawingMode)
#include "Candera/System/MetaInfo/EnumDataType.h"

namespace Candera {
class Appearance;
class Camera;
class Mesh;
class VertexGeometry;
}


class LineListWidget3D : public Candera::Widget
{
   public:
      LineListWidget3D();
      virtual ~LineListWidget3D();

      CGI_WIDGET_RTTI_DECLARATION(LineListWidget3D, Candera::Widget);

      /**
      * Initializes the widget so that all referred resource can be resolved
      * Parameters[in] : assetProvider - Asset provider to resolve all referred resources
      */
      virtual void Init(Candera::AssetProvider* assetProvider);

      /**
      * If there is any changes in Vertex Goemetry, then draw the lines.
      */
      virtual void Update();

      /**
      *  Add New Vertex to the Geometry
      *  Parameters[in]  : x,y,z coordinate values of the point(vertex) of type Float
      *  Return      : void
      */
      void addVertex(Candera::Float x, Candera::Float y, Candera::Float z = 0.0f);

      void setLineColor(const Candera::Color& color)
      {
         _lineColor = color;
      }
      const Candera::Color& getLineColor() const
      {
         return _lineColor;
      }

      void setLineWidth(Candera::Float lineWidth)
      {
         _lineWidth = lineWidth;
      }
      Candera::Float getLineWidth() const
      {
         return _lineWidth;
      }

      void triggerUpdateFlag(unsigned int count = 5)
      {
         _incUpdateCounter = count;
         triggerUpdate();
      }

      void createNewLineList();
      bool clearLines();

      CdaWidgetDef(LineListWidget3D, Candera::Widget)
      CdaDescription("A Widget for Drawing a list of lines.")
      CdaReadableName("LineListWidget3D")
      CdaCategory("Polygon")
      CdaProperties()
      CdaPropertiesEnd()
      CdaWidgetDefEnd()

   private:
      Candera::AssetProvider*                                         _assetProvider;
      Candera::VertexGeometryBuilder                                  _builder;
      // Flag used to indicate an update in the Geometry(Adding a vertex or Clearing the vertex)
      bool                                                            _modified;
      bool                                                            _checkRenderTargetActivated;
      Candera::Color                                                  _lineColor;
      Candera::Float                                                  _lineWidth;
      Candera::Group*                                                 _group;
      // Holds the Index of the current linelist
      Candera::UInt32                                                 _lineListIndex;
      Candera::Internal::Vector<Candera::Mesh*>                       _lineLists;
      Candera::MemoryManagement::SharedPointer<Candera::Shader>       _shader;
      Candera::MemoryManagement::SharedPointer<Candera::Appearance>   _appearance;
      Candera::MemoryManagement::SharedPointer<Candera::VertexBuffer> _vertexBuffer;
      //Add new varibles for smoothning
      Candera::Float                                                  _previous_X;
      Candera::Float                                                  _previous_Y;
      unsigned                                                        _incUpdateCounter;

      static const char* SMOOTH_VERT_SHADER;
      static const char* SMOOTH_FRAG_SHADER;

      Candera::AssetProvider* getAssetProvider() const
      {
         return _assetProvider;
      }

      /**
      * Creates a linelist to connect two points(pixels)
      * Return      : Returns the pointer to the created LineList object
      */
      Candera::Mesh* createLineList();
      //End 4933

      /**
      * Creates an Appearance to display the linelist
      * Return      : appearance
      */
      FeatStd::MemoryManagement::SharedPointer<Candera::Appearance> createAppearance();

      /**
      *  Get the shader from the Asset file
      *  Return      : returns the shader that is present in the asset
      */
      FeatStd::MemoryManagement::SharedPointer<Candera::Shader> getShader();

      /**
      * Activate if a valid instance of Render Target is there
      */
      void renderTargetActivate();

      void triggerUpdate() const;
};


#endif // !defined(LINELIST_WIDGET3D_H)
