/* ***************************************************************************************
* FILE:          HandWriting_Util_Widget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HandWriting_Util_Widget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
/**
* HandWriting_Util_Widget2D is a 2D widget which will make use of LineListWidget3D, which is
* a 3D widget to draw random shapes which will be displayed in a 2D view. The inputs to
* this widget will be a set of points representing the X and Y coordinates.
*
* The X and Y coordinates collected from the application and passed to the
* LineListWidget3D to draw the lines. The same Coordinates are stored in an array
* which can be passed to the HandWriting Recognition Engine as input for recognition
* of Characters.
*
* 1. Added properties to include Numbers, Symbols, UpperCase and LowerCase Latin
*    Alphabets, Gestures for English and Chinese Languages.
* 2. Property to assign the priorityMode for recognition.
* 3. Post a courier message which contains the first recognized character
*    to be displayed on the TextWidget present on the DrawingArea.
* 4. Interface to get the status of the Recognition Timer.
* 5. Interface to Clear the Drawing Area is made Public.
* 6. Stop the recognition timer when Clearing the Drawing Area. Cda properties made Public.
* 7. Prediction feature  implementation for new hanvon interface
* 8. Last six charecters as input for Prediction
*
* This widget is mainly designed to support Handwriting recognition application.
*/

#if !defined(HANDWRITING_WIDGET_2D)
#define HANDWRITING_WIDGET_2D

#define MAX_TOUCH_POINTS   5000
#define STROKE_END_MARK    -1
#define MAX_CHAR_LIMITS    192
#define CHINESE_LANGUAGE   -1

/**start of String CRQ */
#define MAX_TEXT_LENGTH 30
#define MAX_BYTE_FOR_A_CHARACTER 4
/**End of String CRQ */

#define CHAR_DICTIONARY_PATH_CHINESE   "Dic_GB18030.bin"
#define WORD_DICTIONARY_PATH_CHINESE   "ChsSysDic.dic"
#define DICTIONARY_PATH_EUROPE         "Dic_Europ.bin"

#include "Widgets/2D/BaseWidget2D.h"
#include "LineListWidget3D.h"
#include "HanWang_Interface.h"
#include "Widgets/2Dext/HandWritingRecognition/generated/HandWriting_Util_Widget2D_Base.h"
#include "HWRBackendInterface.h"
#include "Widgets/2Dext/HandWritingRecognition/generated/HandWriting_Util_Widget2D_Messages.h"


class HandWriting_Util_Widget2D : public HandWriting_Util_Widget2DBase
{
   public:
      CGI_WIDGET_RTTI_DECLARATION(HandWriting_Util_Widget2D, HandWriting_Util_Widget2DBase);

      HandWriting_Util_Widget2D();

      ~HandWriting_Util_Widget2D();

      void setCharactersToPredictedWord(const FeatStd::String& characterstopredictedword);
      const FeatStd::String& getCharactersToPredictedWord() const
      {
         return _charactersToPredictedWord;
      }

      void setSourcePointer(const Candera::MemoryManagement::SharedPointer<HWRBackendInterface>& pSourcePointer);
      const Candera::MemoryManagement::SharedPointer<HWRBackendInterface>& getSourcePointer() const;

      void setAccessMode(Candera::ACCESS_MODE AccessMode);
      Candera::ACCESS_MODE getAccessMode() const;

      /**
      * Depending on the item selected from the scene composer property (Language), the corresponding Langauage Binary
      * file will be used for character recognition.
      * @Parameters  : Language name
      */
      void assignLanguage();

      /**
      * Depending on the item selected from the scene composer property (Language), the corresponding Langauage Binary
      * file will be used for character recognition.
      * @Parameters  : Prediction Mode
      */
      void assignenIntelligentPredictionMode();

      //End for the Pridicted for the same
      void OnParentViewRenderingEnabled(bool enable);

      /**
      * Interface to Clear the Drawing on the Drawing Area
      */
      void clearDrawingArea();

      /**
      * Returns the status of the HandWritingRecognition timer
      * @returns true :  if the timer is running else false
      */
      bool isRecognitionTimerRunning();

      /**
      * Calculate Number of Bytes occupied by a String with the index iStartIndex upto nCharCount (UTF8) characters
      * @Parameters [in] : iStartIndex, uCharCount, string
      * @Return          : Candera::UInt8 - Number of bytes the UTF8 characters occupies
      */
      Candera::UInt8 calculateUTF8Bytes(Candera::UInt8 uStartIndex, Candera::UInt8 nCharCount, Candera::String string) const;

      //Gets the last MAX_HINT_UTF8CHAR (6 By default) UTF-8 Characters in a predict hint word
      /**
      * Copy last MAX_HINT_UTF8CHAR UTF8 Characters
      * @Parameters[in]: string - Source string
      * @Candera::UInt8: Number of bytes copied
      */
      bool getlastHintUTF8Chars(Candera::String string);

      /**
      *  Length of a utf8 string
      *  @Parameters[in]: Candera::String sText
      *  @Return        : UInt8 u8Length
      */
      Candera::UInt8 utfLength(Candera::String sText) const;

      //Cda Properties
      CdaWidget2DDef(HandWriting_Util_Widget2D, HandWriting_Util_Widget2DBase)
      CdaDescription("HandWritingRecognition widget")
      CdaReadableName("HandWriting Recognition Widget")
      CdaCategory("Input")

      CdaProperties()
      CdaBindableProperty(CharactersToPredictedWord, FeatStd::String, getCharactersToPredictedWord, setCharactersToPredictedWord)
      CdaDescription("Gives the characters entered in edit field by user, which will be used to precdited words, IsBindable = true, Type=FeatStd::String")
      CdaCategory("WordPredictionMode")
      CdaBindablePropertyEnd()

      /**
      * Source Pointer of the HDRBackEndInterface
      */
      CdaBindableProperty(PredictedWordsSourcePointer, Candera::MemoryManagement::SharedPointer<HWRBackendInterface>, getSourcePointer, setSourcePointer)
      CdaDescription("Shared pointer instance of the HDRBackEndInterface")
      CdaCategory("WordPredictionMode")
      CdaBindablePropertyEnd()

      /**
      * Access Mode read & write
      */
      CdaBindableProperty(AccessMode, Candera::ACCESS_MODE, getAccessMode, setAccessMode)
      CdaDescription("DynamicImgReadWrite status")
      CdaCategory("WordPredictionMode")
      CdaBindablePropertyEnd()

      CdaPropertiesEnd()
      CdaWidgetDefEnd()

   private:
      LineListWidget3D*       _lineListWidgetPtr;          // Pointer to the LineListWidget3D
      HanWangInterface*       _hanWangInterface;
      short*                  _pointData;                  // collected points from touch
      Candera::UInt32         _yPointData;
      Candera::Char*          _charList;                   // list of characters recognizd.
      static  Courier::Char   _defaultChar[];              // list of characters recognizd.
      Courier::UInt32         _nodePosition_X;
      Courier::UInt32         _nodePosition_Y;
      Candera::UInt32         _index;                      // index to set point data of handwriting recognition widget
      Candera::Vector3        _worldPosition;
      Candera::Char           _utf8CharBuffer[5];          // init char buffers
      Candera::String         _charecterRecDictionaryPath;
      Candera::String         _wordRecDictionaryPath;
      Candera::Int32          _currentLanguage;
      Candera::UInt32         _languageRange;
      bool                    _strokeEnded;
      bool                    _syncFlag;
      bool                    _processTouchPoints;
      Util::Timer             _characterRecogniton_Timer;
      Candera::Float          _drawingAreaWidth;
      Candera::Float          _drawingAreaHeight;
      int                     _numericRange;
      int                     _symbolRange;
      int                     _lowerCaseLatinLetterRange;
      int                     _upperCaseLatinLetterRange;
      int                     _gestureRecognitionRange;
      FeatStd::UInt32         _previous_X;
      FeatStd::UInt32         _previous_Y;
      FeatStd::MemoryManagement::SharedPointer<HWRBackendInterface> _sourcePointer;  //source pointer obj
      FeatStd::String         _charactersToPredictedWord;                            //Predicted word string
      Candera::ACCESS_MODE    _accessMode;                                           //use

      bool                    _update;
      std::string             _temp;
      int _temp2;
      Candera::UInt8          _curStrLength;
      Candera::UInt8          _maxTextLength;

      /**
      * Initialize the widget like
      * 1. Setting the line color
      * 2. Line list widget
      * 3. Position of drawing area
      * 4. Width and height for drawing area
      * 5. Post message with empty string to application so that Text Widget is set with default text
      * Parameters[in]: assetFactory - Asset factory
      */
      virtual void Init(Candera::AssetProvider* assetFactory);

   public:      //ToDo: Check why this 2 functions were private before. In base class (HandWriting_Util_Widget2DBase) they are public -> private has no effect beside confusion
      /** Message contains the Touch information and TimerExpiry notification message.
      returns true, if the message was consumed, else false. */
      virtual bool OnMessage(const Courier::Message& msg);

      /**
      * Use this function to Invalidate the scene
      */
      virtual void Update();

   private:
      /**
      * Handles the TouchDown Event by the Widget.
      * - Creates a new line each time there is a TouchDown Event.
      * - Kill the Recognition timer if running.
      * @Parameters[in] :   X and Y Coordinates of the Points to be drawn.
      */
      void handleTouchDown(Candera::UInt32, Candera::UInt32);

      /**
      * Handles the TouchMove Event.Store the Coordinate information and also use it to connect them by a LineList.
      * @Parameters[in] :   X and Y Coordinates
      */
      void handleTouchMove(Candera::UInt32, Candera::UInt32);

      /**
      * Used to mark the end of a stroke and start a recognition timer
      */
      void handleTouchUp();

      /**
      * Post a courier message which contains all the characters recognized by the
      * HandWritingRecognition engine.
      */
      void postCandidateUpdateMsg();

      /**
      * If the completion of the pattern to be recognized by the HandWriting Recognition Engine is detected, then mark
      * the end of the stroke and send the pattern to the Engine for processing.
      * Get the Candidates from the Pattern and pass to Text Widget
      */
      bool processRecognizedCharacters();

      /**
      * If the completion of the pattern to be recognized by the HandWriting Recognition Engine is detected, then mark
      * the end of the stroke and send the pattern to the Engine for processing.
      * Get the text from the Pattern and pass to Text Widget
      */
      bool getPredictedText();

      /**
      * Encoding UTF16 to UTF8.
      * @Returns the number of bytes occupied by the UTF8 Character
      */
      int convertUTF16toUTF8(char* pcString, int nCharacter);

      /**
      * Encoding UTF8 to UTF16.
      * @Returns the number of bytes occupied by the UTF8 Character
      */
      //int convertUTF8toUTF16(UInt8* pInput, unsigned short* unic) const;
      int convertUTF8toUTF16(const Candera::UInt8* pInput, Candera::UInt16* unic);

      /**
      * Post a courier message which contains the first character recognized by the HandWritingRecognition engine.
      */
      void postHWRTextWidgetUpdateMsg();

      /**
      * Reset the member variable which stores the recognized characters.Clear the Text Area which is used to display
      * the recognized character on the Drawing Area by sending a courier message.
      */
      void clearHWRTextWidget();

      /**
      * Interface for backend interface
      */
      void extractStrings(const Candera::UInt16* f_array, Candera::UInt32 f_size, FeatStd::Int32 predictedResultCount);

      void getNodePosition();
      void getNodeDimension();
};


#endif
