/* ***************************************************************************************
* FILE:          ViewSwitchWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ViewSwitchWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ViewSwitchWidget2D.h"
#include <Courier/Visualization/ViewFacade.h>
#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <hmibase/util/StringUtils.h>

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SWITCH
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ViewSwitchWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ViewSwitchWidget2D);


/*****************************************************************************/
ViewSwitchWidget2D::ViewSwitchWidget2D() : _selectedViewIndex(0), _invalid(true)
{
}


/*****************************************************************************/
ViewSwitchWidget2D::~ViewSwitchWidget2D()
{
}


/*****************************************************************************/
static Courier::ViewId getViewId(const FeatStd::Char* viewName)
{
   if (viewName == NULL)
   {
      return Courier::ViewId();
   }

   typedef std::map<Courier::ViewId, const FeatStd::Char*> MapType;
   static MapType _map;

   MapType::iterator it = _map.find(Courier::ViewId(viewName));
   if (it != _map.end())
   {
      return Courier::ViewId(it->second);
   }
   else
   {
      FeatStd::Char* buffer = FEATSTD_NEW_ARRAY(FeatStd::Char, FeatStd::Internal::String::Length(viewName) + 1);
      if (buffer != NULL)
      {
         FeatStd::Internal::String::Copy(buffer, viewName);
         Courier::ViewId viewId(buffer);
         _map[viewId] = buffer;
         return viewId;
      }
      return Courier::ViewId();
   }
}


/*****************************************************************************/
void ViewSwitchWidget2D::Finalize()
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Finalize index=%u %s", GetIndex(), HMIBASE_TO_STRING_VW(this)));

   Base::Finalize();

   unloadAllViews();
}


/*****************************************************************************/
void ViewSwitchWidget2D::Update()
{
   if (_invalid && (GetNode() != NULL) && (GetParentView() != NULL) && GetNode()->IsEffectiveRenderingEnabled())
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update index=%u %s", GetIndex(), HMIBASE_TO_STRING_VW(this)));

      Courier::ViewId viewId;
      if ((GetIndex() < GetViewIds().GetCount()) && !GetViewIds().Get(GetIndex()).IsEmpty())
      {
         std::string viewName(GetViewIds().Get(GetIndex()).GetCString());
         hmibase::util::trim(viewName);
         if (!viewName.empty())
         {
            if (!GetViewPrefix().IsEmpty())
            {
               std::string viewPrefix(GetViewPrefix().GetCString());
               hmibase::util::trim(viewPrefix);
               viewName.insert(0, viewPrefix);
            }
            viewId = Courier::ViewId(getViewId(viewName.c_str()));
         }
      }

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Update oldView=%50s newView=%50s thisView=%s",
                          HMIBASE_TO_STRING(_selectedViewId), HMIBASE_TO_STRING(viewId), HMIBASE_TO_STRING(GetParentView()->GetId())));

      //if view changed and is different from this view
      if ((viewId != GetParentView()->GetId()) && (viewId != _selectedViewId))
      {
         AppViewHandler& viewHandler(AppViewHandler::getInstance());
         Courier::ViewFacade viewFacade;
         viewFacade.Init(&viewHandler);

         //deactivate or destroy the old view
         if (_selectedViewId != Courier::ViewId())
         {
            if ((_selectedViewIndex < GetCacheViews().GetCount()) && GetCacheViews().Get(_selectedViewIndex))
            {
               activateView(_selectedViewId, false);
            }
            else
            {
               loadView(_selectedViewId, false);
            }
         }

         _selectedViewId = viewId;
         _selectedViewIndex = GetIndex();

         //create and activate the new view
         if (_selectedViewId != Courier::ViewId())
         {
            loadView(_selectedViewId, true);
            activateView(_selectedViewId, true);
         }

         //make sure the screen is repainted
         Invalidate();
      }

      _invalid = false;
   }
}


/*****************************************************************************/
void ViewSwitchWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnParentViewRenderingEnabled enable=%u %s", enable, HMIBASE_TO_STRING_VW(this)));
}


/*****************************************************************************/
void ViewSwitchWidget2D::OnParentViewActivate(bool activate)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnParentViewActivate activate=%u %s", activate, HMIBASE_TO_STRING_VW(this)));

   activateView(_selectedViewId, activate);
}


/*****************************************************************************/
void ViewSwitchWidget2D::OnParentViewLoad(bool load)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnParentViewLoad load=%u %s", load, HMIBASE_TO_STRING_VW(this)));

   if (load)
   {
      loadView(_selectedViewId, true);
   }
   else
   {
      unloadAllViews();
   }
}


/*****************************************************************************/
void ViewSwitchWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   _invalid = true;
}


/*****************************************************************************/
void ViewSwitchWidget2D::loadView(const Courier::ViewId& viewId, bool load)
{
   if (viewId != Courier::ViewId())
   {
      AppViewHandler& viewHandler(AppViewHandler::getInstance());
      Courier::ViewFacade viewFacade;
      viewFacade.Init(&viewHandler);

      if (load)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "LoadView view=%s", HMIBASE_TO_STRING(viewId)));

         Courier::ViewReqMsg viewReqMsg(Courier::ViewAction::Create, viewId, true, true);
         viewFacade.OnViewComponentMessage(viewReqMsg);

         if (GetAddInvalidationDependency())
         {
            //set invalidation dependency so that both scenes are rendered all the time
            Courier::View* selectedView = viewHandler.FindView(viewId);
            if ((selectedView != NULL) && (GetParentView() != NULL))
            {
               selectedView->AddInvalidationDependency(GetParentView()->GetId());
               GetParentView()->AddInvalidationDependency(viewId);
            }
         }
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UnloadView view=%s", HMIBASE_TO_STRING(viewId)));

         if (viewHandler.FindView(viewId) != NULL)
         {
            Courier::ViewReqMsg viewReqMsg(Courier::ViewAction::Destroy, viewId, false, false);
            viewFacade.OnViewComponentMessage(viewReqMsg);
         }
      }
   }
}


/*****************************************************************************/
void ViewSwitchWidget2D::activateView(const Courier::ViewId& viewId, bool activate)
{
   if (viewId != Courier::ViewId())
   {
      AppViewHandler& viewHandler(AppViewHandler::getInstance());
      Courier::ViewFacade viewFacade;
      viewFacade.Init(&viewHandler);

      if (activate)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ActivateView view=%s", HMIBASE_TO_STRING(viewId)));

         Courier::ActivationReqMsg activationReqMsg(viewId, true, true);
         viewFacade.OnViewComponentMessage(activationReqMsg);
      }
      else
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "DeactivateView view=%s", HMIBASE_TO_STRING(viewId)));

         Courier::ActivationReqMsg activationReqMsg(viewId, false, false);
         viewFacade.OnViewComponentMessage(activationReqMsg);
      }
   }
}


/*****************************************************************************/
void ViewSwitchWidget2D::unloadAllViews()
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "UnloadAllViews %s", HMIBASE_TO_STRING_VW(this)));

   AppViewHandler& viewHandler(AppViewHandler::getInstance());
   Courier::ViewFacade viewFacade;
   viewFacade.Init(&viewHandler);

   for (FeatStd::UInt i = 0; i < GetViewIds().GetCount(); ++i)
   {
      std::string viewFullName(GetViewPrefix().GetCString());
      viewFullName.append(GetViewIds().Get(i).GetCString());

      Courier::ViewId viewId(getViewId(viewFullName.c_str()));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "UnloadAllViews view=%s", HMIBASE_TO_STRING(viewId)));

      if (viewHandler.FindView(viewId) != NULL)
      {
         Courier::ViewReqMsg viewReqMsg(Courier::ViewAction::Destroy, viewId, false, false);
         viewFacade.OnViewComponentMessage(viewReqMsg);
      }
   }
}
