/* ***************************************************************************************
* FILE:          TwoDimensionSliderWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TwoDimensionSliderWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "TwoDimensionSliderWidget2D.h"
#include "Widgets/2D/TwoDimensionSlider/generated/TwoDimensionSliderWidget2DMessages.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TwoDimensionSliderWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TwoDimensionSliderWidget2D);


TwoDimensionSliderWidget2D::TwoDimensionSliderWidget2D() :
   _touchStartPosition(0.0f, 0.0f),
   _moved(false),
   _invalidateValueIntern(false),
   _invalidateValueExtern(true),
   _invalidateAll(true),
   _noOfPixelsPerUnitInHorz(0),
   _noOfPixelsPerUnitInVertical(0),
   _crosshairKnobNode(NULL),
   _crosshairHorNode(NULL),
   _crosshairVerNode(NULL),
   _horInternalValue(0),
   _verInternalValue(0)
{
   SetTouchable(true);
}


TwoDimensionSliderWidget2D::~TwoDimensionSliderWidget2D()
{
   _crosshairKnobNode = NULL;
   _crosshairHorNode = NULL;
   _crosshairVerNode = NULL;
}


// InitWidget (widget interface)
void TwoDimensionSliderWidget2D::InitWidget()
{
   Base::InitWidget();
   updateNodes();
   calcNoOfPixelsPerUnit();
}


// get node pointers for cross hair
void TwoDimensionSliderWidget2D::updateNodes()
{
   _crosshairKnobNode = 0;
   _crosshairHorNode = 0;
   _crosshairVerNode = 0;

   if (GetNode() != NULL)
   {
      _crosshairKnobNode = GetNode()->GetFirstChild();
      if (_crosshairKnobNode)
      {
         _crosshairHorNode = _crosshairKnobNode->GetNextSibling();
         if (_crosshairHorNode)
         {
            _crosshairVerNode = _crosshairHorNode->GetNextSibling();
         }
      }
   }
}


// OnParentViewLoad (widget interface)
void TwoDimensionSliderWidget2D::OnParentViewLoad(bool activate)
{
   BaseWidget2D::OnParentViewLoad(activate);
   if (activate)
   {
   }
}


// Update (widget interface)
void TwoDimensionSliderWidget2D::Update()
{
   Base::Update();

   if (_invalidateAll)
   {
      calcNoOfPixelsPerUnit();
   }

   if (_invalidateValueExtern || _invalidateAll)
   {
      _horInternalValue = GetHorizontalCurValue();
      _verInternalValue = GetVerticalCurValue();
      checkValueRange();
      _invalidateValueExtern = false;
      _invalidateValueIntern = true;
   }

   if (_invalidateValueIntern || _invalidateAll)
   {
      Candera::Vector2 sliderPos = getPixelPosFromInternValue();

      Candera::Rectangle rect;
      if (_crosshairHorNode)
      {
         // set x,y=const
         _crosshairHorNode->GetEffectiveBoundingRectangle(rect);
         _crosshairHorNode->SetPosition(_area.GetLeft(), sliderPos.GetY() - rect.GetHeight() / 2);
      }
      if (_crosshairVerNode)
      {
         // set y, x=const
         _crosshairVerNode->GetEffectiveBoundingRectangle(rect);
         _crosshairVerNode->SetPosition(sliderPos.GetX() - rect.GetWidth() / 2, _area.GetTop());
      }
      if (_crosshairKnobNode)
      {
         // set x and y
         _crosshairKnobNode->GetEffectiveBoundingRectangle(rect);
         _crosshairKnobNode->SetPosition(sliderPos.GetX() - rect.GetWidth() / 2, sliderPos.GetY() - rect.GetHeight() / 2);
      }
      Invalidate();
      _invalidateValueIntern = false;
      _invalidateAll = false;
   }
}


// OnChanged (widget interface)
void TwoDimensionSliderWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      // todo optimize performance

      case SkinNodePropertyId:
      case HorizontalRangeMinPropertyId:
      case HorizontalRangeMaxPropertyId:
      case HorizontalPermittedValueMinPropertyId:
      case HorizontalPermittedValueMaxPropertyId:
      case HorizontalStepValuePropertyId:
      case HorizontalMovementLockedPropertyId:
      case VerticalRangeMinPropertyId:
      case VerticalRangeMaxPropertyId:
      case VerticalPermittedValueMinPropertyId:
      case VerticalPermittedValueMaxPropertyId:
      case VerticalStepValuePropertyId:
      case VerticalMovementLockedPropertyId:
         _invalidateAll = true;
         break;

      case HorizontalCurValuePropertyId:
         _horInternalValue = GetHorizontalCurValue();
         _invalidateValueExtern = true;
         break;
      case VerticalCurValuePropertyId:
         _verInternalValue = GetVerticalCurValue();
         _invalidateValueExtern = true;
         break;

      default:
         _invalidateAll = true;
         break;
   }
}


// OnMessage (widget interface)
bool TwoDimensionSliderWidget2D::OnMessage(const Message& oMsg)
{
   switch (oMsg.GetId())
   {
      case Widgets::TwoDimensionSilder::MoveMsg::ID:
         return moveMessage(Courier::message_cast<const Widgets::TwoDimensionSilder::MoveMsg*>(&oMsg));
      case Courier::TouchMsg::ID:
         return touchMessage(Courier::message_cast<const Courier::TouchMsg*>(&oMsg));
      default:
         break;
   }
   return Base::OnMessage(oMsg);
}


#define TOFF 20 // pixel

bool TwoDimensionSliderWidget2D::touchMessage(const Courier::TouchMsg* touchMsg)
{
   if ((touchMsg != 0) && (_crosshairKnobNode != NULL))
   {
      FeatStd::Float fTouchedPosX = (FeatStd::Float)(touchMsg->GetXPos()) - getWorldPos().GetX();
      FeatStd::Float fTouchedPosY = (FeatStd::Float)(touchMsg->GetYPos()) - getWorldPos().GetY();

      if (((_area.GetLeft() - TOFF) <= fTouchedPosX)
            && (fTouchedPosX <= (_area.GetLeft() + _area.GetWidth() + TOFF))
            && ((_area.GetTop() - TOFF) <= fTouchedPosY)
            && (fTouchedPosY <= (_area.GetTop() + _area.GetHeight() + TOFF)))
      {
         switch (touchMsg->GetState())
         {
            case Courier::TouchMsgState::Down:
               // Touch state down received
               _touchStartPosition.SetX((FeatStd::Float)(touchMsg->GetXPos()));
               _touchStartPosition.SetY((FeatStd::Float)(touchMsg->GetYPos()));
               setInternValueFromPixelPos((FeatStd::Float)(touchMsg->GetXPos()), (FeatStd::Float)(touchMsg->GetYPos()));
               _moved = false;
               return true;

            case  Courier::TouchMsgState::Move:
               // Touch state move received
               setInternValueFromPixelPos((FeatStd::Float)(touchMsg->GetXPos()), (FeatStd::Float)(touchMsg->GetYPos()));
               if (GetValueUpdateOnDrag())
               {
                  _moved = true;
                  postCurrentValue(true);
               }
               return true;

            case Courier::TouchMsgState::Up:
               // Touch state up received
               if (!_moved)
               {
                  postCurrentValue(true);
               }
               _moved = false;
               return true;

            default:
               break;
         }
      }
   }
   return false;
}


//  incoming Widgets::TwoDimensionSilder::MoveMsg event from DataModel
bool TwoDimensionSliderWidget2D::moveMessage(const Widgets::TwoDimensionSilder::MoveMsg* moveMsg)
{
   if (moveMsg->GetWidgetName() == GetLegacyName())
   {
      switch (moveMsg->GetMoveType())  // TwoDimesionMoveType
      {
         case Widgets::TwoDimensionSilder::MoveUp:
            _verInternalValue = static_cast<FeatStd::Int32>(GetVerticalCurValue() - GetVerticalStepValue());                  //lint !e737
            break;

         case Widgets::TwoDimensionSilder::MoveDown:
            _verInternalValue = static_cast<FeatStd::Int32>(GetVerticalCurValue() + GetVerticalStepValue());                  //lint !e737
            break;

         case Widgets::TwoDimensionSilder::MoveLeft:
            _horInternalValue = static_cast<FeatStd::Int32>(GetHorizontalCurValue() - GetHorizontalStepValue());              //lint !e737
            break;

         case Widgets::TwoDimensionSilder::MoveRight:
            _horInternalValue = static_cast<FeatStd::Int32>(GetHorizontalCurValue() + GetHorizontalStepValue());              //lint !e737
            break;

         case Widgets::TwoDimensionSilder::MoveCenter:
            if (GetUsePermittedValues())
            {
               _horInternalValue = (GetHorizontalPermittedValueMax() - GetHorizontalPermittedValueMin()) / 2;
               _verInternalValue = (GetVerticalPermittedValueMax() - GetVerticalPermittedValueMin()) / 2;
            }
            else
            {
               _horInternalValue = (GetHorizontalRangeMax() - GetHorizontalRangeMin()) / 2;
               _verInternalValue = (GetVerticalRangeMax() - GetVerticalRangeMin()) / 2;
            }
            break;

         default:
            // todo : set a value by message instead of data binding
            break;
      }
      _invalidateValueIntern = true;
      checkValueRange();
      triggerUpdate();
      return true;
   }
   return false;
}


// calculate the number of pixels per unit of given value range
void TwoDimensionSliderWidget2D::calcNoOfPixelsPerUnit()
{
   checkValueRange();

   _area = getTouchAreaRect();
   FeatStd::Int32 horRange = GetHorizontalRangeMax() - GetHorizontalRangeMin();
   if (horRange > 0)
   {
      _noOfPixelsPerUnitInHorz = _area.GetWidth() / static_cast<FeatStd::Float>(horRange);
   }

   FeatStd::Int32 verRange = GetVerticalRangeMax() - GetVerticalRangeMin();
   if (verRange > 0)
   {
      _noOfPixelsPerUnitInVertical = _area.GetHeight() / static_cast<FeatStd::Float>(verRange);
   }
}


// get the position on the screen, source is the current value
Candera::Vector2 TwoDimensionSliderWidget2D::getPixelPosFromInternValue()
{
   const FeatStd::Int32 x = _horInternalValue - GetHorizontalRangeMin();
   const FeatStd::Int32 y = _verInternalValue - GetVerticalRangeMin();
   Candera::Vector2 position;
   position.SetX(_area.GetLeft() + (_noOfPixelsPerUnitInHorz * static_cast<FeatStd::Float>(x)));
   position.SetY(_area.GetTop() + (_noOfPixelsPerUnitInVertical * static_cast<FeatStd::Float>(y)));
   return position;
}


// set internal value base on a touch position on the screen (relative positon to the node)
void TwoDimensionSliderWidget2D::setInternValueFromPixelPos(const FeatStd::Float fPosX, const FeatStd::Float fPosY)
{
   if (_noOfPixelsPerUnitInHorz == 0.0f || _noOfPixelsPerUnitInVertical == 0.0f)
   {
      calcNoOfPixelsPerUnit();
   }

   if ((_noOfPixelsPerUnitInHorz != 0) && (_noOfPixelsPerUnitInVertical != 0))
   {
      Candera::Vector2 pos;
      // calculate at first the relative position in pixel
      pos.SetX(fPosX - (getWorldPos().GetX() + _area.GetLeft()));
      pos.SetY(fPosY - (getWorldPos().GetY() + _area.GetTop()));

      if (!GetHorizontalMovementLocked())
      {
         _horInternalValue = (Candera::Int32)((pos.GetX() / _noOfPixelsPerUnitInHorz) + 0.5f) + GetHorizontalRangeMin();
      }
      if (!GetVerticalMovementLocked())
      {
         _verInternalValue = (Candera::Int32)((pos.GetY() / _noOfPixelsPerUnitInVertical) + 0.5f) + GetVerticalRangeMin();
      }
      checkValueRange();
      _invalidateValueIntern = true;
      triggerUpdate();
   }
}


// range checks for to the internal value for different work modes
void TwoDimensionSliderWidget2D::checkValueRange()
{
   if (_horInternalValue < GetHorizontalRangeMin())
   {
      _horInternalValue  = GetHorizontalRangeMin();
   }
   if (_horInternalValue > GetHorizontalRangeMax())
   {
      _horInternalValue = GetHorizontalRangeMax();
   }

   if (_verInternalValue < GetVerticalRangeMin())
   {
      _verInternalValue = GetVerticalRangeMin();
   }
   if (_verInternalValue > GetVerticalRangeMax())
   {
      _verInternalValue = GetVerticalRangeMax();
   }
   if (GetUsePermittedValues())
   {
      if (GetHorizontalPermittedValueMin() > GetHorizontalRangeMin())
         if (_horInternalValue < GetHorizontalPermittedValueMin())
         {
            _horInternalValue = GetHorizontalPermittedValueMin();
         }

      if (GetHorizontalPermittedValueMax() < GetHorizontalRangeMax())
         if (_horInternalValue > GetHorizontalPermittedValueMax())
         {
            _horInternalValue = GetHorizontalPermittedValueMax();
         }

      if (GetVerticalPermittedValueMin() > GetVerticalRangeMin())
         if (_verInternalValue < GetVerticalPermittedValueMin())
         {
            _verInternalValue = GetVerticalPermittedValueMin();
         }

      if (GetVerticalPermittedValueMax() < GetVerticalRangeMax())
         if (_verInternalValue > GetVerticalPermittedValueMax())
         {
            _verInternalValue = GetVerticalPermittedValueMax();
         }
   }
}


// post current values to the DataModel
void TwoDimensionSliderWidget2D::postCurrentValue(bool draging)
{
   Widgets::TwoDimensionSilder::UpdMsg* msg = COURIER_MESSAGE_NEW(Widgets::TwoDimensionSilder::UpdMsg)(GetLegacyName(), _horInternalValue, _verInternalValue,
         draging ? Widgets::TwoDimensionSilder::SliderDrag : Widgets::TwoDimensionSilder::SliderRelease);
   if (NULL != msg)
   {
      msg->Post();
   }
}


// get root position of the object in world coordinates
Candera::Vector2 TwoDimensionSliderWidget2D::getWorldPos() const
{
   if (GetSkinNode())
   {
      return GetSkinNode()->GetWorldPosition();
   }
   if (GetNode())
   {
      return GetNode()->GetWorldPosition();
   }
   return Candera::Vector2(1.0f, 1.0f);
}


// get rectangle size of the widget area, bitmap size or manual defined by SliderArea
Candera::Rectangle TwoDimensionSliderWidget2D::getTouchAreaRect() const
{
   //  make use of the size of the manual configured area
   Candera::Rectangle r = GetSliderArea();
   if (r.GetWidth() > 0.0f && r.GetHeight() > 0.0f)
   {
      return r;
   }
   //  make use of the size of the bitmap
   if (GetSkinNode())
   {
      GetSkinNode()->GetEffectiveBoundingRectangle(r);
      if (r.GetWidth() > 0.0f && r.GetHeight() > 0.0f)
      {
         return r;
      }
   }
   // make use of the size of the node
   if (GetNode())
   {
      GetNode()->GetEffectiveBoundingRectangle(r);
   }
   return r;
}


// #########################################################################################
// Scenecomposer property filter
// #########################################################################################


bool TwoDimensionSliderWidget2D::composerPropVisibleFilterPermittedArea() const
{
   return (GetUsePermittedValues() == true);
}
