/* ***************************************************************************************
* FILE:          Touchable2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  Touchable2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Touchable2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h"
#include "Widgets/2D/DragDrop/DDManager.h"
#include "Courier/Visualization/ViewScene2D.h"
#include <View/CGI/CgiExtensions/SurfaceUtils.h>
#include <View/CGI/CgiExtensions/TouchInput.h>
#include <View/CGI/Widget/WidgetController.h>

#if !defined(WIN32)
#include "CanderaPlatform/Device/Genivi/GeniviWindowSurface.h"
#endif

#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TOUCHABLE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/Touchable2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(Touchable2D)

Touchable2D::Touchable2D() :
   _touchSession(NULL)
{
}


Touchable2D::~Touchable2D()
{
   if (_touchSession != NULL)
   {
      SetTouched(false);

      _touchSession->Deregister(this);
      _touchSession = NULL;
   }
}


/****************************************************************************
* Gesture config is attached to a widget using the property GestureConfigId.
*****************************************************************************/
WidgetGestureConfig Touchable2D::getGestureConfig() const
{
   hmibase::widget::WidgetController* controller = GetController();

   // first, check the config set explicitly on the widget
   if (GetGestureConfigId() != 0)
   {
      const WidgetGestureConfig* explicitConfig = WidgetGestureConfigRegistry::getItem(GetGestureConfigId());
      if (explicitConfig != NULL)
      {
         WidgetGestureConfig config(*explicitConfig);
         if (controller != NULL)
         {
            controller->CheckGestureConfig(*this, config);
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureConfig use gesture config by id=%u %s",
                             GetGestureConfigId(), HMIBASE_TO_STRING_VW(this)));
         return config;
      }
   }

   // second, check the default config of the controller
   if (controller != NULL)
   {
      FeatStd::Optional<WidgetGestureConfig> controllerDefaultConfig = controller->GetDefaultGestureConfig(*this);

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
      if (controllerDefaultConfig.IsSet())
#else
      if (controllerDefaultConfig == true)
#endif
      {
         WidgetGestureConfig config(*controllerDefaultConfig);
         controller->CheckGestureConfig(*this, config);

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureConfig use gesture config from controller=%d %s",
                             GetControllerId(), HMIBASE_TO_STRING_VW(this)));
         return config;
      }
   }

   // third, check the config set for the widget type or base classes
   const WidgetGestureConfig* typeConfig = WidgetGestureConfigByTypeRegistry::get(*this);
   if (typeConfig != NULL)
   {
      WidgetGestureConfig config(*typeConfig);
      if (controller != NULL)
      {
         controller->CheckGestureConfig(*this, config);
      }

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureConfig use gesture config by type=%25s (or base classes) %s",
                          GetTypeName(), HMIBASE_TO_STRING_VW(this)));
      return config;
   }

   // at last, return the default config
   WidgetGestureConfig defaultConfig(getDefaultGestureConfig());
   if (controller != NULL)
   {
      controller->CheckGestureConfig(*this, defaultConfig);
   }

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureConfig using default gesture config %s", HMIBASE_TO_STRING_VW(this)));
   return defaultConfig;
}


/****************************************************************************
* Subclasses can overwrite this to provide custom default configuration.
*****************************************************************************/
WidgetGestureConfig Touchable2D::getDefaultGestureConfig() const
{
   // enable all gestures here, the controller can overwrite this default config
   return WidgetGestureConfig::getDefaults();
}


/****************************************************************************
* Returns the gesture configuration as required by the gesture framework.
*****************************************************************************/
hmibase::input::gesture::GestureListener::Gestures Touchable2D::GetGestureList() const
{
   hmibase::input::gesture::GestureListener::Gestures gestures;

   if (GetTouchable())
   {
      WidgetGestureConfig widgetGestureConfig(getGestureConfig());

      // adjust the thresholds based on the factor associated to the touched display
      unsigned int surfaceId = static_cast<unsigned int>(getCurrentSessionTouchInfo().mSourceId);
      if (surfaceId != 0)
      {
         //unsigned int displayId = ScreenBrokerClient::GetInstance().GetDisplayID(surfaceId);
         unsigned int displayId = hmibase::view::util::SurfaceUtils::getDisplayId(surfaceId);
         const WidgetGestureFactor* gestureFactor = DisplayGestureFactorRegistry::getItem(displayId);
         if (gestureFactor != NULL)
         {
            widgetGestureConfig *= (*gestureFactor);
         }
      }

      // Press and Tap
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getTap().Enabled && GetTap())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.tapPressHoldRepeatGestureParameter.maxDistance = widgetGestureConfig.getTap().MaxDistance;
         config.tapPressHoldRepeatGestureParameter.doubleTapEnabled = GetDoubleTap();
         config.tapPressHoldRepeatGestureParameter.doubleTapTime = widgetGestureConfig.getTap().DoubleTapTime;
         config.tapPressHoldRepeatGestureParameter.holdTime = GetPressHold() ? widgetGestureConfig.getTap().HoldTime : 0;
         config.tapPressHoldRepeatGestureParameter.repeatTime = GetPressRepeat() ? widgetGestureConfig.getTap().RepeatTime : 0;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Tap maxDist=%d doubleTap=%u doubleTapTime=%u holdTime=%u repeatTime=%u %s",
                             config.tapPressHoldRepeatGestureParameter.maxDistance,
                             config.tapPressHoldRepeatGestureParameter.doubleTapEnabled,
                             config.tapPressHoldRepeatGestureParameter.doubleTapTime,
                             config.tapPressHoldRepeatGestureParameter.holdTime,
                             config.tapPressHoldRepeatGestureParameter.repeatTime,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_PressHoldRepeatTap, config));
      }
#ifdef VARIANT_S_FTR_ENABLE_LEGACY_DRAG_DROP
      //legacy support for drag and drop requires Tap instead of Drag
      else if (GetDragDropSourceEnabled())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.tapPressHoldRepeatGestureParameter.maxDistance = 0;
         config.tapPressHoldRepeatGestureParameter.doubleTapEnabled = false;
         config.tapPressHoldRepeatGestureParameter.doubleTapTime = 0;
         config.tapPressHoldRepeatGestureParameter.holdTime = 0;
         config.tapPressHoldRepeatGestureParameter.repeatTime = 0;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Tap(for DragDrop) maxDist=%d doubleTap=%u doubleTapTime=%u holdTime=%u repeatTime=%u %s",
                             config.tapPressHoldRepeatGestureParameter.maxDistance,
                             config.tapPressHoldRepeatGestureParameter.doubleTapEnabled,
                             config.tapPressHoldRepeatGestureParameter.doubleTapTime,
                             config.tapPressHoldRepeatGestureParameter.holdTime,
                             config.tapPressHoldRepeatGestureParameter.repeatTime,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_PressHoldRepeatTap, config));
      }
      else
      {
         //nothing here
      }
#endif
      // Drag
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getDrag().Enabled && GetDrag())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.dragGestureParameter.holdTime = widgetGestureConfig.getDrag().HoldTime;
         config.dragGestureParameter.holdMaxDistance = widgetGestureConfig.getDrag().HoldMaxDistance;
         config.dragGestureParameter.minDistance = widgetGestureConfig.getDrag().MinDistance;
         switch (GetDragDirection())
         {
            case Candera::enHorizontal:
               config.dragGestureParameter.direction = GestureEvent::DIR_HORIZONTAL;
               break;
            case Candera::enVertical:
               config.dragGestureParameter.direction = GestureEvent::DIR_VERTICAL;
               break;
            default:
               config.dragGestureParameter.direction = GestureEvent::DIR_2D;
               break;
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Drag holdTime=%u holdMaxDist=%d minDist=%d direction=%u parallel=%u %s",
                             config.dragGestureParameter.holdTime,
                             config.dragGestureParameter.holdMaxDistance,
                             config.dragGestureParameter.minDistance,
                             config.dragGestureParameter.direction,
                             widgetGestureConfig.getDrag().ParallelEnabled,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_DragNudge, config, widgetGestureConfig.getDrag().ParallelEnabled));
      }

      // Swipe
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getSwipe().Enabled && GetSwipe())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.swipeGestureParameter.minDistance = widgetGestureConfig.getSwipe().MinDistance;
         config.swipeGestureParameter.minSpeed = widgetGestureConfig.getSwipe().MinSpeed;
         switch (GetSwipeDirection())
         {
            case Candera::enHorizontal:
               config.swipeGestureParameter.direction = GestureEvent::DIR_HORIZONTAL;
               break;
            case Candera::enVertical:
               config.swipeGestureParameter.direction = GestureEvent::DIR_VERTICAL;
               break;
            default:
               config.swipeGestureParameter.direction = GestureEvent::DIR_2D;
               break;
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Swipe minDist=%d minSpeed=%d direction=%u %s",
                             config.swipeGestureParameter.minDistance,
                             config.swipeGestureParameter.minSpeed,
                             config.swipeGestureParameter.direction,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_SwipeFling, config));
      }

      // Pinch and Spread
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getPinchSpread().Enabled && GetPinchSpread())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.pinchGestureParameter.doubleTouchMaxTime = widgetGestureConfig.getPinchSpread().MaxTime;
         config.pinchGestureParameter.spreadMinFactor = widgetGestureConfig.getPinchSpread().MinSpreadFactor;
         config.pinchGestureParameter.pinchMaxFactor = widgetGestureConfig.getPinchSpread().MaxPinchFactor;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Pinch doubleTouchMaxTime=%u minFactor=%f maxFactor=%f %s",
                             config.pinchGestureParameter.doubleTouchMaxTime,
                             config.pinchGestureParameter.spreadMinFactor,
                             config.pinchGestureParameter.pinchMaxFactor,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_PinchSpread, config));
      }

      // Rotate
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getRotate().Enabled && GetRotate())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;

         config.rotateGestureParameter.doubleTouchMaxTime = widgetGestureConfig.getRotate().MaxTime;
         config.rotateGestureParameter.minRotationAngle = widgetGestureConfig.getRotate().MinAngle;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList Rotate doubleTouchMaxTime=%u minRotAngle=%d %s",
                             config.rotateGestureParameter.doubleTouchMaxTime,
                             config.rotateGestureParameter.minRotationAngle,
                             HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_Rotate, config));
      }

      // RawTouch
      // gesture must be enabled both in the widget and in the configuration
      if (widgetGestureConfig.getRawTouch().Enabled && GetRawTouch())
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;
         config.rawGestureParameter.dummy = 0;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList RawTouch %s", HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_RawTouchData, config));
      }

      // legacy gesture handling
      if (gestures.size() == 0)
      {
         hmibase::input::gesture::GestureListener::GestureConfig config;
         config.legacyTouchParameter.dummy = 0;

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "GetGestureList LegacyTouch %s", HMIBASE_TO_STRING_VW(this)));
         gestures.push_back(Gesture(GestureEvent::GT_LegacyTouch, config));
      }
   }
   return gestures;
}


hmibase::input::gesture::GestureListener::GesturePriorizationParameter Touchable2D::getGesturePriorizationParameter() const
{
   GesturePriorizationParameter params;
   params.effectiveTouchPriority = GetEffectiveTouchPriority();
   params.effectiveRenderOrderRank = GetNode2D() != 0 ? GetNode2D()->GetEffectiveRenderOrderRank() : 0;

   return params;
}


/****************************************************************************
* GetEffectiveTouchPriority returns the priority of handling touch messages
* for this widget. The TouchSession has to use this priority as an additional
* sorting criteria.
*  Returns 0 is less prior than 1.
****************************************************************************/
Touchable2D::TouchPriorityType Touchable2D::GetEffectiveTouchPriority() const
{
   return GetTouchPriority();
}


/****************************************************************************
*    CloneFrom is required for FlexList cloning, we are not extending
*    ControlTemplateCloneableWidget yet, just to have the base properties cloned
*****************************************************************************/
bool Touchable2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const Touchable2D* original = CLONEABLE_WIDGET_CAST<const Touchable2D*>(originalWidget);                                   //lint !e740
      if (original == NULL)
      {
         return false;
      }

      SetTouchable(original->GetTouchable());
      SetTouchPriority(original->GetTouchPriority());
      SetDisabledTouching(original->IsDisabledTouching());

      SetGestureConfigId(original->GetGestureConfigId());
      SetTap(original->GetTap());
      SetDoubleTap(original->GetDoubleTap());
      SetPressHold(original->GetPressHold());
      SetPressRepeat(original->GetPressRepeat());
      SetDrag(original->GetDrag());
      SetDragDirection(original->GetDragDirection());
      SetSwipe(original->GetSwipe());
      SetSwipeDirection(original->GetSwipeDirection());
      SetPinchSpread(original->GetPinchSpread());
      SetRotate(original->GetRotate());
      SetRawTouch(original->GetRawTouch());

      SetDragDropSourceEnabled(original->GetDragDropSourceEnabled());
      SetDragDropDestinationEnabled(original->GetDragDropDestinationEnabled());

      CloneFocusableFrom(*original, controlTemplateMap);

      return true;
   }
   return false;
}


/****************************************************************************
*  OnParentViewLoad is invoked by F/W when parent view is loaded or unloaded
*     Parameters  : true  - Scene context was loaded
*                   false - Scene context was unloaded
****************************************************************************/
void Touchable2D::OnParentViewLoad(bool load)
{
   if (!load)
   {
      if (_touchSession != NULL)
      {
         SetTouched(false);

         _touchSession->Deregister(this);

         _TODO("check why TouchSession::deregister myself is not enough")
         _touchSession->Cleanup();   // with commit a4487852 the issue is not solved in al cases, therefore Cleanup() method re-enabled
         // see also -> https://hi-cmts.apps.intranet.bosch.com:8443/browse/CMG3GB-2395

         _touchSession = NULL;
      }
   }
}


/****************************************************************************
*  OnParentViewActivate is invoked by F/W when parent view changes its
*  activation state
*     Parameters  : true  - View is activated
*                   false - View is deactivated
****************************************************************************/
void Touchable2D::OnParentViewActivate(bool activate)
{
   if (!activate)
   {
      if (_touchSession != NULL)
      {
         SetTouched(false);

         _touchSession->Deregister(this);

         _TODO("check why TouchSession::deregister myself is not enough")
         _touchSession->Cleanup();   // with commit a4487852 the issue is not solved in al cases, therefore Cleanup() method re-enabled
         // see also -> https://hi-cmts.apps.intranet.bosch.com:8443/browse/CMG3GB-2395

         _touchSession = NULL;
      }
   }
}


/****************************************************************************
*  OnMessage is called when a message shall be distributed through the
*  view tree and its views and widgets
*  Returns true if the Message is consumed
****************************************************************************/
bool Touchable2D::OnMessage(const Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   bool consumed = false;
   switch (msg.GetId())
   {
      case hmibase::input::TouchSessionStartEvent::ID:
      {
         const hmibase::input::TouchSessionStartEvent* startEvent = Courier::message_cast<const hmibase::input::TouchSessionStartEvent*>(&msg);
         if (startEvent != NULL)
         {
            const Courier::TouchInfo& touchInfo = startEvent->GetTouchInfo();
            if (IsInsideBoundingRect(touchInfo) && IsTouchable() && (IsEnabled() || IsDisabledTouching()))
            {
               if (_touchSession != NULL)
               {
                  _touchSession->Deregister(this);
                  _touchSession = NULL;
               }

               hmibase::input::TouchSession& touchSession = const_cast<hmibase::input::TouchSession&>(startEvent->GetTouchSession().Get());
               _touchSession = &touchSession;
               touchSession.Register(this);

               SetTouched(true);

               postTouchUpdMessage(hmibase::widget::enTouchState::Down);
            }
         }
      }
      break;

      case hmibase::input::TouchSessionStopEvent::ID:
      {
         const hmibase::input::TouchSessionStopEvent* stopEvent = Courier::message_cast<const hmibase::input::TouchSessionStopEvent*>(&msg);
         if (stopEvent != NULL)
         {
            hmibase::input::TouchSession& touchSession = const_cast<hmibase::input::TouchSession&>(stopEvent->GetTouchSession().Get());
            if (_touchSession == &touchSession)
            {
               SetTouched(false);

               touchSession.Deregister(this);
               _touchSession = NULL;

               postTouchUpdMessage(hmibase::widget::enTouchState::Up);
            }
         }
      }
      break;

      case Courier::TouchMsg::ID:
      {
         const Courier::TouchMsg* touchMsg = Courier::message_cast<const Courier::TouchMsg*>(&msg);
         if (touchMsg != NULL)
         {
            consumed = OnTouchMessage(*touchMsg);
         }
      }
      break;

      case RegisterFocusableWidgetsReqMsg::ID:
         RegisterToFocusManager(*this);
         break;

      default:
         break;
   }

   return consumed;
}


void Touchable2D::Update()
{
   Base::Update();
   Focusable2DBase::UpdateFocusable(*this);
}


void Touchable2D::postTouchUpdMessage(hmibase::widget::enTouchState::Enum state)
{
   Courier::View* view = GetParentView();
   if (view != NULL)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostTouchUpdMessage state=%d userData=%u %s",
                          ETG_CENUM(hmibase::widget::enTouchState::Enum, state), GetUserData(),
                          HMIBASE_TO_STRING_VW(this)));

      hmibase::widget::TouchUpdMsg* msg = COURIER_MESSAGE_NEW(hmibase::widget::TouchUpdMsg)(view->GetId(), Courier::Identifier(GetLegacyName()), GetUserData(), state);
      if (msg != NULL)
      {
         msg->Post();
      }
   }
}


bool Touchable2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnTapGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnDragGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnSwipeGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnPinchGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnPinchGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnRotateGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnRotateGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnRawTouchGesture(const hmibase::input::gesture::GestureEvent& /*msg*/)
{
   ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "OnRawTouchGesture not supported %s", HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::OnTouchMessage(const Courier::TouchMsg& msg)
{
   bool consumed = false;

   Courier::TouchInfo touchInfo(msg.GetXPos(), msg.GetYPos(), msg.GetTimeStamp(), msg.GetPointerId(), msg.GetSourceId());
   switch (msg.GetState())
   {
      case Courier::TouchMsgState::Down:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchMessage Down touchInfo=%50s %s",
                             HMIBASE_TO_STRING(touchInfo),
                             HMIBASE_TO_STRING_VW(this)));
         consumed = DragDrop::DDManager::getInstance().onTouchPress(*this, touchInfo);
      }
      break;

      case Courier::TouchMsgState::Move:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchMessage Move touchInfo=%50s %s",
                             HMIBASE_TO_STRING(touchInfo),
                             HMIBASE_TO_STRING_VW(this)));
         consumed = DragDrop::DDManager::getInstance().onTouchMove(*this, touchInfo);
      }
      break;

      case Courier::TouchMsgState::Up:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "OnTouchMessage Up touchInfo=%50s %s",
                             HMIBASE_TO_STRING(touchInfo),
                             HMIBASE_TO_STRING_VW(this)));
         consumed = DragDrop::DDManager::getInstance().onTouchRelease(*this, touchInfo);
      }
      break;

      default:
         break;
   }

   return consumed;
}


/****************************************************************************
*  OnTouch, overloaded function to be used only if the widget should perform
*  a custom hit test on a Touch message
*  Parameters  : Camera reference and Touch coordinates
*  Return      : true  - Touch intersects the node
*                false - Touch did not intersect the node
****************************************************************************/
bool Touchable2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool nodeTouched = false;
   Candera::Node2D* node = GetNode();
   if (GetTouchable() && (node != NULL) && node->IsEffectiveRenderingEnabled())
   {
      nodeTouched = IsPickIntersectingNode(camera2D, node, point);
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTouch intersects=%u camera=%50s point=%12s %s",
                          nodeTouched, HMIBASE_TO_STRING_N2D(&camera2D),
                          HMIBASE_TO_STRING(point),
                          HMIBASE_TO_STRING_VW(this)));

      //if a controller is attached allow it to also check the touch coordinates
      if (nodeTouched && (GetController() != NULL) && !DELEGATE_WIDGET_BOOL_FUNC(OnTouch(*this, camera2D, point)))
      {
         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnTouch disabled by controller with id=%d %s",
                             GetControllerId(),
                             HMIBASE_TO_STRING_VW(this)));
         nodeTouched = false;
      }
   }
   return nodeTouched;
}


#if defined(_lint) && (_lint <= 912) // Versions up to and including 9.00L
//lint -e10 PQM_authorized_517         Reason: Lint cannot dissolve FEAT macros
//lint -e1013 PQM_authorized_multi_518 Reason: Lint cannot dissolve FEAT macros
//lint -e1055 PQM_authorized_multi_519 Reason: Lint cannot dissolve FEAT macros
//lint -e746 PQM_authorized_multi_520  Reason: Lint cannot dissolve FEAT macros
#elif defined(_lint)
#error Check if above problems still exists with new Lint version
#endif

/****************************************************************************
* IsInsideBoundingRec function tells whether the touch coordinate is within
* the widget's bounding rectangle or not.
* Return true, if the touch event happens with the bounding rectangle
****************************************************************************/
bool Touchable2D::IsInsideBoundingRect(const Courier::TouchInfo& msg)
{
   if (_fkt_IsInsideBoundingBox2DFunction)
   {
      bool result = _fkt_IsInsideBoundingBox2DFunction(this, msg);

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "IsInsideBoundingRect(func) inside=%u touchInfo=%50s %s",
                          result, HMIBASE_TO_STRING(msg),
                          HMIBASE_TO_STRING_VW(this)));

      return result;
   }

   if (GetParentView() != NULL)
   {
      Courier::ViewScene2D* viewScene2D = GetParentView()->ToViewScene2D();
      if (viewScene2D != NULL)
      {
         const Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
         const Candera::Vector2 point(static_cast<Courier::Float>(msg.mX), static_cast<Courier::Float>(msg.mY));

         for (Courier::ViewScene2D::CameraPtrVector::ConstIterator it = cameras.ConstBegin(); it != cameras.ConstEnd(); ++it)
         {
            Candera::Camera2D* camera2D = *it;
            if ((camera2D != NULL) && camera2D->IsRenderingEnabled() && hmibase::input::ShouldUseCameraForTouch(camera2D))
            {
#if defined(WIN32)

               // on windows there is only one source-ID
               if ((msg.mSourceId == 0) && OnTouch(*camera2D, point))
               {
                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "IsInsideBoundingRect(1) inside=true touchInfo=%50s camera=%50s %s",
                                      HMIBASE_TO_STRING(msg),
                                      HMIBASE_TO_STRING_N2D(camera2D),
                                      HMIBASE_TO_STRING_VW(this)));
                  return true;
               }

#else

               if (camera2D->GetRenderTarget() != NULL)
               {
                  Candera::GeniviWindowSurface* surface = dynamic_cast<Candera::GeniviWindowSurface*>(camera2D->GetRenderTarget()->GetGraphicDeviceUnit());     //lint !e740

                  //The following issues are generated because Lint can not understand the complex definition of the class GeniviWindowSurface
                  // Lint prio 1/2 are authorized deactivated for this function
                  //Info 737: prio3: Loss of sign in promotion from int to unsigned int
                  CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(737, "symbol GetSupportProperties is defined in base class")
                  if ((surface != NULL) && (msg.mSourceId == surface->GetSupportProperties().GetSurfaceId()))
                  {
                     if (OnTouch(*camera2D, point))
                     {
                        ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "IsInsideBoundingRect(2) inside=true touchInfo=%50s camera=%50s %s",
                                            HMIBASE_TO_STRING(msg),
                                            HMIBASE_TO_STRING_N2D(camera2D),
                                            HMIBASE_TO_STRING_VW(this)));
                        return true;
                     }
                  }
               }

#endif
            }
         }
      }
   }

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "IsInsideBoundingRect inside=false touchInfo=%50s %s",
                       HMIBASE_TO_STRING(msg),
                       HMIBASE_TO_STRING_VW(this)));
   return false;
}


bool Touchable2D::processGestureEvent(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool consumed = false;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "ProcessGestureEvent gesture=%100s %s",
                       HMIBASE_TO_STRING(gestureData),
                       HMIBASE_TO_STRING_VW(this)));

   if (DELEGATE_WIDGET_BOOL_FUNC(OnGesture(*this, gestureData)))
   {
      return true;
   }

#ifdef VARIANT_S_FTR_ENABLE_LEGACY_DRAG_DROP
   //legacy support for drag and drop requires Tap instead of Drag
   if (gestureData._gestureType == GestureEvent::GT_PressHoldRepeatTap)
   {
      switch (gestureData._gestureState)
      {
         case hmibase::input::gesture::GestureEvent::ET_START:
            DragDrop::DDManager::getInstance().onTouchPress(*this, getCurrentSessionTouchInfo());
            break;

         case hmibase::input::gesture::GestureEvent::ET_MOVE:
            DragDrop::DDManager::getInstance().onTouchMove(*this, getCurrentSessionTouchInfo());
            break;

         case hmibase::input::gesture::GestureEvent::ET_ABORT:
         case hmibase::input::gesture::GestureEvent::ET_END:
            DragDrop::DDManager::getInstance().onTouchRelease(*this, getCurrentSessionTouchInfo());
            break;

         default:
            break;
      }
   }
#endif

   switch (gestureData._gestureType)
   {
      case hmibase::input::gesture::GestureEvent::GT_DragNudge:
         consumed = OnDragGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_PressHoldRepeatTap:
         consumed = OnTapGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_PinchSpread:
         consumed = OnPinchGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_SwipeFling:
         consumed = OnSwipeGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_Rotate:
         consumed = OnRotateGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_RawTouchData:
         consumed = OnRawTouchGesture(gestureData);
         break;
      case hmibase::input::gesture::GestureEvent::GT_LegacyTouch:
      {
         if (_touchSession != 0)
         {
            const Courier::Message* msg = _touchSession->GetLegayTouchInfo();
            if (msg != 0)
            {
               if (gestureData._gestureState == GestureEvent::ET_ABORT)
               {
                  if (msg->GetId() == hmibase::input::TouchAbort::ID)
                  {
                     consumed = OnMessage(*msg);
                  }
               }
               else
               {
                  consumed = OnMessage(*msg);
               }
            }
         }
      }
      break;
      default:
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessGestureEvent Not supported gesture=%100s %s",
                            HMIBASE_TO_STRING(gestureData),
                            HMIBASE_TO_STRING_VW(this)));
         break;
   }
   return consumed;
}


hmibase::input::gesture::GUI_Rect Touchable2D::getReceiverAbsoluteRect()
{
   Candera::Node2D* node = GetNode();
   if (node != NULL)
   {
      if (node->IsEffectiveRenderingEnabled())
      {
         Candera::Rectangle rect;
         node->GetEffectiveBoundingRectangle(rect);
         rect.SetPosition(node->GetWorldPosition() + rect.GetPosition());

         hmibase::input::gesture::Vector2D pos(static_cast<hmibase::input::gesture::TouchCoord>(rect.GetPosition().GetX()), static_cast<hmibase::input::gesture::TouchCoord>(rect.GetPosition().GetY()));
         hmibase::input::gesture::GUI_Size size(static_cast<hmibase::input::gesture::TouchCoord>(rect.GetSize().GetX()), static_cast<hmibase::input::gesture::TouchCoord>(rect.GetSize().GetY()));

         return hmibase::input::gesture::GUI_Rect(pos, size);
      }
   }

   return hmibase::input::gesture::GUI_Rect();
}


Candera::Rectangle Touchable2D::GetWorldTouchableRectangle() const
{
   const Candera::Node2D* node = GetNode();
   if (node != NULL)
   {
      Candera::Rectangle rect;
      node->GetEffectiveBoundingRectangle(rect);
      rect.SetPosition(rect.GetPosition() + node->GetWorldPosition());
      return rect;
   }
   return Candera::Rectangle();
}


Courier::TouchInfo Touchable2D::getCurrentSessionTouchInfo() const
{
   return (_touchSession != NULL) ? _touchSession->GetTouchInfo() : Courier::TouchInfo(0, 0, 0, 0, 0);
}


Courier::TouchInfo Touchable2D::getCurrentSessionStartTouchInfo() const
{
   return (_touchSession != NULL) ? _touchSession->GetStartTouchInfo() : Courier::TouchInfo(0, 0, 0, 0, 0);
}


bool Touchable2D::OnTouchGeneric(const Candera::CanderaObject& camera, const Candera::Vector2& pos, FeatStd::Float)
{
   if (camera.IsTypeOf(Candera::Camera2D::GetTypeId()))
   {
      const Candera::Camera2D& camera2D = dynamic_cast<const Candera::Camera2D&>(camera);
      return OnTouch(camera2D, pos);
   }
   return false;
}


#if defined(_lint) && (_lint <= 912) // Versions up to and including 9.00L
//lint +e10   re-activate due to above deactivation.
//lint +e1013 re-activate due to above deactivation.
//lint +e1055 re-activate due to above deactivation.
//lint +e746  re-activate due to above deactivation.
#elif defined(_lint)
#error Check if deactivation / re-activation is necessary with new Lint version
#endif
