/* ***************************************************************************************
* FILE:          TextExtensionWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextExtensionWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/TextExtension/TextExtensionWidget2D.h"
#include "Widgets/2D/WidgetFinder2D.h"

using namespace Courier;
using namespace Candera;


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextExtensionWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TextExtensionWidget2D);

#define MAX_STRINGS         5


/******************************************************************************
*  Constructor
******************************************************************************/
TextExtensionWidget2D::TextExtensionWidget2D() :
   _ptrCurrentCulture(NULL),
   _textWidget(NULL),
   _switchSelectedText(0),
   _dirty(true),
   _ancestorSearchDone(0),
   _timer(0)
{
}


/***************************************************************************
*     destructor
****************************************************************************/
TextExtensionWidget2D::~TextExtensionWidget2D()
{
   TimerStop(true);
   _textWidget = NULL;
   _ptrCurrentCulture = NULL;
} //lint !e1579 Pointer _timer is deleted in TimerStop()


/***************************************************************************
*  InitWidget
****************************************************************************/
void TextExtensionWidget2D::InitWidget()
{
   Base::InitWidget();
   _ptrCurrentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();
}


static std::string GetStdSTR(const FeatStd::String& str)
{
   std::string s;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(str);
   s = str.GetCString();
   SECURE_FEATSTD_STRING_ACCESS_END();
   return s;
}


/******************************************************************************
*  Update
******************************************************************************/
void TextExtensionWidget2D::Update()
{
   Base::Update();

   // if timer is not running check if it must be started
   if ((_timer == NULL) || (!_timer->running()))
   {
      // check if node is effective visible as last step to increase performance
      if ((GetExtensionMethod() == TEXTSWITCH) && (GetNode() != NULL) && (GetNode()->IsEffectiveRenderingEnabled()))
      {
         TimerStart();
      }
   }
   //if timer is already running check if it must be stopped
   else
   {
      // check if node is effective visible as last step to increase performance
      if ((GetExtensionMethod() != TEXTSWITCH) || (GetNode() == NULL) || (!GetNode()->IsEffectiveRenderingEnabled()))
      {
         TimerStop();
      }
   }

   if (IsDirty() && GetNode())
   {
      if (FindAncestorWidget() == true)
      {
         if (GetExtensionMethod() == TEXTCONCAT)
         {
            _effText =  GetStdSTR(GetText1());
            _effText += GetStdSTR(GetText2());
            _effText += GetStdSTR(GetText3());
            _effText += GetStdSTR(GetText4());
            _effText += GetStdSTR(GetText5());
         }
         else if (GetExtensionMethod() == TEXTWITHARGS)
         {
            _effText = FormatTextWithArgs();
         }
         else if (GetExtensionMethod() == TEXTSWITCH)
         {
            for (int ii = 0; ii < MAX_STRINGS; ii++)
            {
               if (_switchSelectedText >= MAX_STRINGS)
               {
                  _switchSelectedText = 0;
               }

               switch (_switchSelectedText)
               {
                  case 0:
                     _effText = GetStdSTR(GetText1());
                     break;
                  case 1:
                     _effText = GetStdSTR(GetText2());
                     break;
                  case 2:
                     _effText = GetStdSTR(GetText3());
                     break;
                  case 3:
                     _effText = GetStdSTR(GetText4());
                     break;
                  case 4:
                     _effText = GetStdSTR(GetText5());
                     break;
                  default:
                     break;
               }
               if (!_effText.empty())
               {
                  break;
               }
               _switchSelectedText++;
            }
         }
         if (_textWidget->GetText() != _effText.c_str())
         {
            _textWidget->SetText(_effText.c_str());
         }
      }
      ResetDirty();
   }
}


/******************************************************************************
*  CloneFrom
******************************************************************************/
bool TextExtensionWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget,
                                      ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const TextExtensionWidget2D* ow = CLONEABLE_WIDGET_CAST<const TextExtensionWidget2D*>(originalWidget);
      if (ow == NULL)
      {
         return false;
      }

      SetFormatText(ow->GetFormatText());
      SetText1(ow->GetText1());
      SetText2(ow->GetText2());
      SetText3(ow->GetText3());
      SetText4(ow->GetText4());
      SetText5(ow->GetText5());
      SetExtensionMethod(ow->GetExtensionMethod());
      SetAnimationTime(ow->GetAnimationTime());

      cloned = true;
   }
   return cloned;
}


/***************************************************************************
*  TimerStart
****************************************************************************/
void TextExtensionWidget2D::TimerStart()
{
   if (_timer == NULL)
   {
      _timer = CANDERA_NEW(Util::Timer);
      FEATSTD_DEBUG_ASSERT(_timer != 0);
      if (_timer)
      {
         _timer->setName("TextExtensionWidget2D", GetLegacyName());
      }
   }
   if (_timer != NULL)
   {
      if (!_timer->stopped())
      {
         _timer->stop();
      }
      if (GetAnimationTime() < 50)
      {
         SetAnimationTime(50);
      }
      _timer->setTimeoutWithRepeat(GetAnimationTime(), GetAnimationTime());
      _timer->start();
   }
}


/***************************************************************************
*  TimerStop
****************************************************************************/
void TextExtensionWidget2D::TimerStop(bool deleteInstance)
{
   if (_timer != NULL)
   {
      _timer->stop();
      if (deleteInstance)
      {
         CANDERA_DELETE(_timer);
         _timer = NULL;
      }
   }
}


/***************************************************************************
*  OnMessage
****************************************************************************/
bool TextExtensionWidget2D::OnMessage(const Message& msg)
{
   if (msg.GetId() == TimerExpiredMsg::ID)
   {
      const TimerExpiredMsg* timerMsg = message_cast<const TimerExpiredMsg*>(&msg);
      if (timerMsg != 0)
      {
         ::Util::Timer* timer = timerMsg->GetTimer();
         if (timer == _timer)
         {
            if (GetExtensionMethod() == TEXTSWITCH)
            {
               _switchSelectedText++;
               SetDirty();
               Invalidate();
            }
            return true;
         }
      }
   }
   else if (msg.GetId() == Courier::CultureChangeIndMsg::ID)
   {
      CultureChanged();
   }

   return false;
}


/***************************************************************************
*  OnNodeChanged
****************************************************************************/
void TextExtensionWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();
   _textWidget = NULL;
   TimerStop();
   SetDirty();
}


/******************************************************************************
*  OnChanged
******************************************************************************/
void TextExtensionWidget2D::OnParentViewRenderingEnabled(bool viewLoaded)
{
   Base::OnParentViewRenderingEnabled(viewLoaded);

   if (!viewLoaded)
   {
      // stop animation on view hide
      TimerStop();
   }
   else
   {
      //if a culture is known by this widget and it is different from the current one it means that the culture was changed while the view was inactive => trigger culture changed
      if ((_ptrCurrentCulture != NULL) && (_ptrCurrentCulture != Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()))
      {
         CultureChanged();
      }
   }
}


/******************************************************************************
*  OnChanged
******************************************************************************/
void TextExtensionWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   switch (propertyId)
   {
      case VisiblePropertyId:
      case EnabledPropertyId:
         Base::OnChanged(propertyId);
         break;

      case FormatTextPropertyId:
      case Text1PropertyId:
      case Text2PropertyId:
      case Text3PropertyId:
      case Text4PropertyId:
      case Text5PropertyId:
      case AnimationTimePropertyId:
      case ExtensionMethodPropertyId:
         TimerStop();
         _switchSelectedText = 0;
         break;
      default:
         break;
   };
   SetDirty();
   Invalidate();
}


/***************************************************************************
*  FindAncestorWidget
****************************************************************************/
bool TextExtensionWidget2D::FindAncestorWidget()
{
   if ((_textWidget == NULL) && (GetNode() != NULL))
   {
      _textWidget = WidgetFinder::FindAncestorWidget<TextWidget2D>(this);
      if (_textWidget != NULL)
      {
         _dirty = true;
      }

      else if (++_ancestorSearchDone == 2)
      {
         ETG_TRACE_ERR(("SceneTree config error: Extension called FindAncestorWidget without TextWidget2D ancestor!!"));
      }
   }
   return _textWidget != 0;
}


/***************************************************************************
*  ComposerPropFilterTimer
****************************************************************************/
bool TextExtensionWidget2D::ComposerPropFilterTimer() const
{
   return GetExtensionMethod() == TEXTSWITCH;
}


/***************************************************************************
*  ComposerPropFilterArgs
****************************************************************************/
bool TextExtensionWidget2D::ComposerPropFilterArgs() const
{
   return GetExtensionMethod() == TEXTWITHARGS;
}


/***************************************************************************
  FormatTextWithArgs - printf feature
    "hello Binu, how is the ticket status today?"
    SetFormatText("hello $(2:), how is the ticket status $(1:)?");
    SetText1("today");
    SetText2("Binu");
    FormatTextWithArgs();
****************************************************************************/
std::string TextExtensionWidget2D::FormatTextWithArgs()
{
   std::string sResult;
   std::string frm;
   frm = GetStdSTR(GetFormatText());
   const char* s = frm.c_str();

   _TODO("FormatTextWithArgs() is not UTF8 safety!")

   while (s && s[0] != 0)
   {
      // find '$(1:xxx)' or '$(2:)' ...
      const char* s1 = strstr(s, "$(");
      if (s1 == NULL)
      {
         sResult.append(s);  // finaly copy the tail
         break;              // finished
      }

      const char* s2 = strstr(s1 + 2, ")");
      if (s2 == NULL)
      {
         ETG_TRACE_ERR(("TextExtensionWidget2D ERROR: close bracket of $( missing"));
         break;
      }

      int ii = atoi(s1 + 2);
      sResult.append(s, (unsigned int)(s1 - s));

      switch (ii)
      {
         case 1:
            sResult += GetStdSTR(GetText1());
            break;
         case 2:
            sResult += GetStdSTR(GetText2());
            break;
         case 3:
            sResult += GetStdSTR(GetText3());
            break;
         case 4:
            sResult += GetStdSTR(GetText4());
            break;
         case 5:
            sResult += GetStdSTR(GetText5());
            break;
         default:
            ETG_TRACE_ERR(("TextExtensionWidget2D ERROR: supports only 1..5 Args arg=%d", ii));
            return sResult;
      }
      s = s2 + 1; // set pointer after argument close bracket '$(1:)'
   }
   return sResult;
}


void TextExtensionWidget2D::CultureChanged()
{
   SetDirty();

   //remember the last culture here in order to detect culture changes while the view is inactive
   _ptrCurrentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();
}
