/* #**************************************************************************************
* FILE:          TextWidgetUtils.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextWidgetUtils is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(hmiabse_TextWidgetUtils_H)
#define hmiabse_TextWidgetUtils_H

#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/TextEngine/Types.h>
#include <Candera/TextEngine/PreprocessingContext.h>
#include <Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h>

namespace Candera {
class TextBrush;
}


namespace hmibase {
namespace widget {
namespace text {


typedef Candera::TextRendering::PreprocessingContext::GlyphData CanderaGlyphData;
typedef Candera::TextRendering::PreprocessingContext::Iterator GlyphDataIterator;


/*****************************************************************************/
class BaseGlyphData : public CanderaGlyphData
{
   public:
      BaseGlyphData() : m_glyphIndex(0), m_characterPosition(0)
      {
         m_fontIdentifier.m_value = 0;
         m_position.x = 0;
         m_position.y = 0;
      }

      BaseGlyphData(const CanderaGlyphData& g) :
         m_fontIdentifier(g.GetFontIdentifier()),
         m_glyphIndex(g.GetGlyphIndex()),
         m_characterPosition(g.GetCharacterPosition()),
         m_position(g.GetPosition())
      {
      }

      BaseGlyphData& operator =(const CanderaGlyphData& g)
      {
         if (this != &g)
         {
            m_fontIdentifier = g.GetFontIdentifier();
            m_glyphIndex = g.GetGlyphIndex();
            m_characterPosition = g.GetCharacterPosition();
            m_position = g.GetPosition();
         }
         return *this;
      }

      bool matches(const CanderaGlyphData& g) const
      {
         return (this == &g) || ((m_fontIdentifier == g.GetFontIdentifier()) && (m_glyphIndex == g.GetGlyphIndex()) && (m_characterPosition == g.GetCharacterPosition()) && (m_position == g.GetPosition()));
      }

      /*****************************************************************************/
      virtual Candera::TextRendering::FontIdentifier GetFontIdentifier() const
      {
         return m_fontIdentifier;
      }

      virtual Candera::TextRendering::GlyphIndex GetGlyphIndex() const
      {
         return m_glyphIndex;
      }

      virtual Candera::TextRendering::TextPosition GetCharacterPosition() const
      {
         return m_characterPosition;
      }

      virtual Candera::TextRendering::PixelPosition2D GetPosition() const
      {
         return m_position;
      }

      /*****************************************************************************/
      Candera::TextRendering::FontIdentifier m_fontIdentifier;
      Candera::TextRendering::GlyphIndex m_glyphIndex;
      Candera::TextRendering::TextPosition m_characterPosition;
      Candera::TextRendering::PixelPosition2D m_position;
};


/*****************************************************************************/
class GlyphData : public BaseGlyphData
{
      typedef BaseGlyphData Base;

   public:
      GlyphData() : Base(),
         m_glyphBaseLine(0), m_glyphTop(0),
         m_left(0), m_top(0), m_right(0), m_bottom(0),
         m_leftToRight(true), m_hidden(false), m_offsetX(0), m_offsetY(0)
      {
         m_size.width = 0;
         m_size.height = 0;
      }

      /*****************************************************************************/
      FeatStd::Int32 GetScrollbaleX()
      {
         return GetScrollbale(m_position.x, m_offsetX);
      }

      FeatStd::Int32 GetScrollbaleY()
      {
         return GetScrollbale(m_position.y, m_offsetY);
      }

      FeatStd::Int32 GetScrollbaleRight()
      {
         return GetScrollbale(m_right, m_offsetX);
      }

      FeatStd::Int32 GetScrollbaleBottom()
      {
         return GetScrollbale(m_bottom, m_offsetY);
      }

      FeatStd::Int32 GetScrollbaleBaseLine()
      {
         return GetScrollbale(m_glyphBaseLine, m_offsetY);
      }

      FeatStd::Int32 GetScrollbaleTop()
      {
         return GetScrollbale(m_glyphTop, m_offsetY);
      }

      FeatStd::Int32 GetScrollbale(FeatStd::Int32 value, FeatStd::Int32 offset)
      {
         return value + (offset << 16);
      }

      /*****************************************************************************/
      Candera::TextRendering::PixelSize2D m_size;
      Candera::TextRendering::PixelPosition m_glyphBaseLine;
      Candera::TextRendering::PixelPosition m_glyphTop;
      Candera::TextRendering::PixelPosition m_left;
      Candera::TextRendering::PixelPosition m_top;
      Candera::TextRendering::PixelPosition m_right;
      Candera::TextRendering::PixelPosition m_bottom;
      bool m_leftToRight;
      bool m_hidden;
      FeatStd::Int32 m_offsetX;
      FeatStd::Int32 m_offsetY;
};


/*****************************************************************************/
class GlyphDataIteratorFactory : private Candera::TextRendering::PreprocessingContext
{
   public:
      template <typename TGlyph>
      static GlyphDataIterator CreateIterator(const typename FeatStd::Internal::Vector<TGlyph>& glyphDataVector, const size_t index = 0)
      {
         ReferenceGlyphDataIterator<TGlyph, typename FeatStd::Internal::Vector<TGlyph> > temp(glyphDataVector, index);
         return GlyphDataIterator(temp.Clone());
      }

      template <typename TGlyph>
      static GlyphDataIterator CreateIterator(const typename std::vector<TGlyph>& glyphDataVector, const size_t index = 0)
      {
         ReferenceGlyphDataIterator<TGlyph, typename std::vector<TGlyph> > temp(glyphDataVector, index);
         return GlyphDataIterator(temp.Clone());
      }

   private:
      FEATSTD_MAKE_CLASS_STATIC(GlyphDataIteratorFactory);
      FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphDataIteratorFactory);

      /*****************************************************************************/
      template <typename T> struct PtrTrait
      {
         static T* Get(T& g)
         {
            return &g;
         }

         static const T* Get(const T& g)
         {
            return &g;
         }
      };

      template <typename T> struct PtrTrait<T*>
      {
         static T& Get(T& g)
         {
            return g;
         }

         static const T& Get(const T& g)
         {
            return g;
         }
      };

      /*****************************************************************************/
      template <typename TGlyph>
      static bool IsGlyphHidden(const TGlyph& /*g*/)
      {
         return false;
      }

      template <typename TGlyph>
      static size_t GetVectorSize(const FeatStd::Internal::Vector<TGlyph>& v)
      {
         return v.Size();
      }

      template <typename TGlyph>
      static size_t GetVectorSize(const std::vector<TGlyph>& v)
      {
         return v.size();
      }

      template <typename TGlyph, typename TVector>
      class ReferenceGlyphDataIterator : public ReferenceIterator
      {
         public:
            ReferenceGlyphDataIterator(const TVector& glyphs, const size_t& index = 0) : _glyphDataVector(glyphs), _index(index) {}
            ReferenceGlyphDataIterator(const ReferenceGlyphDataIterator& other) : _glyphDataVector(other._glyphDataVector), _index(other._index) {}

            // Override functions from ReferenceIterator.
            virtual ReferenceIterator* Clone() const override final
            {
               return FEATSTD_NEW(ReferenceGlyphDataIterator)(*this);
            }

            virtual void Destroy() override final
            {
               return FEATSTD_DELETE(this);
            }

            virtual bool IsValid() const override final
            {
               return _index < GlyphDataIteratorFactory::GetVectorSize(_glyphDataVector);
            }

            virtual const CanderaGlyphData* GetData() const override final
            {
               return GlyphDataIteratorFactory::PtrTrait<TGlyph>::Get(_glyphDataVector[_index]);
            }

            virtual void Advance() override final
            {
               FEATSTD_DEBUG_ASSERT(IsValid());
               ++_index;

               while (IsValid() && GlyphDataIteratorFactory::IsGlyphHidden(*GlyphDataIteratorFactory::PtrTrait<TGlyph>::Get(_glyphDataVector[_index])))
               {
                  ++_index;
               }
            }

         private:
            ReferenceGlyphDataIterator& operator=(const ReferenceGlyphDataIterator& other);

            const TVector& _glyphDataVector;
            size_t _index;
      };
};


/*****************************************************************************/
template <> bool GlyphDataIteratorFactory::IsGlyphHidden<hmibase::widget::text::GlyphData>(const hmibase::widget::text::GlyphData& g);


/*****************************************************************************/
struct TextBrushData
{
      FEATSTD_TYPEDEF_SHARED_POINTER(TextBrushData);

      typedef FeatStd::Internal::Vector<GlyphData> GlyphDataVector;

      static SharedPointer Create();
      static SharedPointer Create(const GlyphDataVector& data);

      GlyphDataVector _GlyphDataVector;
      size_t _UpdateCount;

   private:
      TextBrushData();
      TextBrushData(const GlyphDataVector& data);

      FEATSTD_MAKE_CLASS_UNCOPYABLE(TextBrushData);
      FEATSTD_SHARED_POINTER_DECLARATION();
};


/*****************************************************************************/
class DynamicPropertiesAccessor
{
   public:
      static bool ClearTextBrushData(Candera::TextBrush& object);
      static bool SetTextBrushData(Candera::TextBrush& object, const TextBrushData::SharedPointer& value);
      static TextBrushData::SharedPointer GetTextBrushData(const Candera::TextBrush& object);
      static TextBrushData::SharedPointer GetOrCreateTextBrushData(Candera::TextBrush& object);
};


/*****************************************************************************/
class TextUtils
{
   public:
      static size_t GetByteCount(const Candera::TChar* text);

      template <typename TGlyph>
      static Candera::TextRendering::PreprocessingContext::Iterator CreateGlyphDataIterator(const typename FeatStd::Internal::Vector<TGlyph>& glyphDataVector)
      {
         return GlyphDataIteratorFactory::CreateIterator<TGlyph>(glyphDataVector);
      }

      template <typename TGlyph>
      static Candera::TextRendering::PreprocessingContext::Iterator CreateGlyphDataIterator(const typename std::vector<TGlyph>& glyphDataVector)
      {
         return GlyphDataIteratorFactory::CreateIterator<TGlyph>(glyphDataVector);
      }

      static void PrintDebug(const TextBrushData::GlyphDataVector& glyphDataVector, const Candera::TChar* text = NULL);

      template <typename TGlyph>
      static bool CompareGlyphsByCharacterPosition(TGlyph* x, TGlyph* y)
      {
         if ((x != NULL) && (y != NULL))
         {
            return x->GetCharacterPosition() < y->GetCharacterPosition();
         }
         return true;
      }

   private:
      FEATSTD_MAKE_CLASS_STATIC(TextUtils);
      FEATSTD_MAKE_CLASS_UNCOPYABLE(TextUtils);
};


}
}


}
#endif//hmiabse_TextWidgetUtils_H
