/* #**************************************************************************************
* FILE:          TextWidget2DV2.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextWidget2DV2 is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(TextWidget2DV2_H)
#define TextWidget2DV2_H

#include "CanderaPlatform/Device/Common/Effects/TextBrushBlend.h"
#include "Candera/System/GlobalizationBase/CultureManager.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateCloneableWidget.h"
#include "Widgets/2D/Text/generated/TextWidget2DBase.h"

#if defined SESA_ARABIC_LAYOUT_FIX
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"
#else
#include "Candera/Engine2D/Layout/ArabicLayouterPatch.h"
#endif
#endif

#include "TextStyleConfiguratorImpl.h"
#include "Candera/System/GlobalizationBase/Culture.h"
#include "TextLayoutingInfoListener.h"

typedef ::hmibase::widget::text::TextStyleConfiguratorImpl TextStyleManager;
typedef ::hmibase::widget::text::TextLayoutingInfoListener TextLayoutingListener;

class TextWidget2DV2 : public TextWidget2DBase
   IMPLEMENTS_CLONEABLE_WIDGET
{
   public:
      TextWidget2DV2();
      virtual ~TextWidget2DV2();

      CGI_WIDGET_RTTI_DECLARATION(TextWidget2DV2, TextWidget2DBase);

      typedef Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> SharedStylePointer;

      enum ExtensionStyleType
      {
         Bold,
         Italic,
         BoldItalic,
         NoExtensionStyleType
      } m_extensionStyleType;
      typedef const std::string StyleName;
      typedef FeatStd::Int16 StyleId;

      virtual void InitWidget();
      virtual void CultureChanged();
      virtual void Update();
      virtual void OnChanged(::FeatStd::UInt32 propertyId);
      virtual void OnNodeChanged();
      virtual void OnBeforeNodeChanged();
      virtual bool OnMessage(const Message& msg);
      virtual bool CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap);
      virtual void OnParentViewRenderingEnabled(bool enable);

      //// todo virtual void OnEvent(const Candera::Event& event, Candera::EventDispatchResult& dispatchResult);

      void startScroll();
      void stopScroll();
      void startScrollDelayTimer();
      void decideExtensionStyleType(TextWidget2DV2::ExtensionStyleType& m_extensionStyleType);
      bool getExtensionStyle(StyleId StyleId, SharedStylePointer& extensionStyle, TextStyleManager::ExtensionType& textShrinkStyles, bool& isShrinkEnabled);
      bool getExtensionStyleWithShrink(StyleId StyleId, SharedStylePointer& style, SharedStylePointer& extensionStyle, TextStyleManager::ExtensionType& textShrinkStyles, bool& isShrinkEnabled);
      Candera::Vector2 getAdditionalSizeforEffects(Candera::Vector2& additionalSize);

      void AddListener(TextLayoutingListener* layoutingInfoListener);

      CdaWidget2DDef(TextWidget2DV2, TextWidget2DBase)
      CdaDescription("Label widget capable for rendering text in 2D scenes, single line and multiline. TextWidget2DV2 is bindable to a RenderNode or TextNode2D. In case of TextNode2D most of the properties are provided by the Node, otherwise use the properties of the widget.")
      CdaReadableName("TextWidget2DV2")
      CdaCategory("Under construction")
      CdaProperties()
      CdaProperty_AppearanceId()
      CdaPropertiesEnd()
      CdaWidgetDefEnd()

   private:
      Candera::Globalization::Culture* _ptrCurrentCulture;

      virtual Candera::Vector2 OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea);
      virtual void OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea);
#if defined SESA_ARABIC_LAYOUT_FIX
      virtual void OnClipping(Candera::Node2D& node, const Candera::Rectangle& clientArea);
#endif
      friend class TextWidgetTextRenderContext;
      friend class TextWidget2DV2PreprocessingContext;

      class TextWidgetsLayouter : public Candera::Layouter
      {
         public:
            TextWidgetsLayouter();
            virtual ~TextWidgetsLayouter();

            void Init(TextWidget2DV2* widget);

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3)))
            LayoutAlignment::LayoutDirection::Enum GetTextLayoutDirection(Candera::Node2D& node);
#else
            Candera::LayoutDirection GetTextLayoutDirection(Candera::Node2D& node);
#endif

            virtual void Dispose();
            virtual Layouter* Clone() const;

         protected:
#if defined SESA_ARABIC_LAYOUT_FIX
            friend class TextWidget2DV2;
            using Layouter::SetArrangeActualSize;
#endif

            virtual Candera::Vector2 OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea);

            /**
            *  This method is called as second layout pass to arrange the child nodes (position and scale).
            *  node: which children shall be arranged.
            *  clientArea: Rectangle where layouting shall be applied.
            */
            virtual void OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea);

#if ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR > 4))
            virtual Candera::Vector2 OnMeasure(const Candera::AbstractNodePointer& node, const Candera::Vector2& clientArea)
            {
               return OnMeasure(*(node.ToNode2D()), clientArea);
            }
            virtual void OnArrange(const Candera::AbstractNodePointer& node, const Candera::Rectangle& clientArea)
            {
               OnArrange(*(node.ToNode2D()), clientArea);
            }
#endif

#if defined SESA_ARABIC_LAYOUT_FIX
            virtual void OnClipping(Candera::Node2D& node, const Candera::Rectangle& clientArea);
#endif
#ifdef CANDERA_2D_ENABLED
            virtual void OnLimitPreferredSize(Candera::Node2D& /*node*/, Candera::Vector2& /*preferredSize*/, const Candera::Vector2& /*nodeSize*/, const Candera::Vector2& /*nodeMinSize*/, const Candera::Vector2& /*nodeMaxSize*/);
#endif
            virtual void OnLimitPreferredSize(Candera::CanderaObject& /*node*/, Candera::Vector2& /*preferredSize*/, const Candera::Vector2& /*nodeSize*/, const Candera::Vector2& /*nodeMinSize*/, const Candera::Vector2& /*nodeMaxSize*/) override;

         private:
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1725, CANDERA_LINT_REASON_ASSOCIATION)
            TextWidget2DV2* _textWidget;
            FEATSTD_MAKE_CLASS_UNCOPYABLE(TextWidgetsLayouter);
      };

      class MinimalGlyphData : public Candera::TextRendering::PreprocessingContext::GlyphData
      {
            friend class TextWidget2DV2;
            friend class TextWidgetTextRenderContext;
         public:
            MinimalGlyphData() :
               m_glyphIndex(0),
               m_characterPosition(0),
               m_glyphBaseLine(0),
               m_glyphTop(0),
               m_right(0),
               m_bottom(0),
               m_offsetX(0),
               m_offsetY(0)
            {
               m_fontIdentifier.m_value = 0;
               m_position.x = 0;
               m_position.y = 0;
               m_size.width = 0;
               m_size.height = 0;
            }

            virtual Candera::TextRendering::FontIdentifier GetFontIdentifier() const
            {
               return m_fontIdentifier;
            }

            virtual Candera::TextRendering::GlyphIndex GetGlyphIndex() const
            {
               return m_glyphIndex;
            }

            virtual Candera::TextRendering::TextPosition GetCharacterPosition() const
            {
               return m_characterPosition;
            }

            virtual Candera::TextRendering::PixelPosition2D GetPosition() const
            {
               return m_position;
            }
            FeatStd::Int32 GetScrollbaleX()
            {
               return GetScrollbale(m_position.x, m_offsetX);
            }

            FeatStd::Int32 GetScrollbaleY()
            {
               return GetScrollbale(m_position.y, m_offsetY);
            }

            FeatStd::Int32 GetScrollbaleRight()
            {
               return GetScrollbale(m_right, m_offsetX);
            }

            FeatStd::Int32 GetScrollbaleBottom()
            {
               return GetScrollbale(m_bottom, m_offsetY);
            }

            FeatStd::Int32 GetScrollbaleBaseLine()
            {
               return GetScrollbale(m_glyphBaseLine, m_offsetY);
            }

            FeatStd::Int32 GetScrollbaleTop()
            {
               return GetScrollbale(m_glyphTop, m_offsetY);
            }

         private:
            FeatStd::Int32 GetScrollbale(FeatStd::Int32 value, FeatStd::Int32 offset)
            {
               return value + (offset << 16);
            }
            Candera::TextRendering::FontIdentifier m_fontIdentifier;
            Candera::TextRendering::GlyphIndex m_glyphIndex;
            Candera::TextRendering::TextPosition m_characterPosition;
            Candera::TextRendering::PixelPosition2D m_position;
            Candera::TextRendering::PixelPosition m_glyphBaseLine;
            Candera::TextRendering::PixelPosition m_glyphTop;
            Candera::TextRendering::PixelSize2D m_size;
            Candera::TextRendering::PixelPosition m_right;
            Candera::TextRendering::PixelPosition m_bottom;
            FeatStd::Int32 m_offsetX;
            FeatStd::Int32 m_offsetY;
      };

      /////////////////////////////////////////////////////////////////////////////
      // scroll start
      /////////////////////////////////////////////////////////////////////////////
      struct ScrollLimits
      {
         ScrollLimits() :
            m_right(0),
            m_bottom(0)
         {
         }
         FeatStd::Int32 m_right;
         FeatStd::Int32 m_bottom;
      };

      struct ScrollPosition
      {
         ScrollPosition() :
            x(0),
            y(0)
         {
         }
         FeatStd::Int32 x;
         FeatStd::Int32 y;
      };

      void ScrollUpdate();
      ScrollLimits ProjectText(FeatStd::Int32 left, FeatStd::Int32 top, Candera::TextRendering::PixelSize width, Candera::TextRendering::PixelSize height);
      void ScrollInitLimits();
      void ScrollUpdatePosition();
      void ScrollTimerStart(ScrollBehaviorType scrollBehavior, ScrollSpeedType speed);
      void ScrollTimerStop(bool deleteInstance = false);
      void ScrollInvokeOnTouch(bool bSet);
      void ScrollInvokeOnAuto();
      void ScrollInvokeOnExternal();
      bool ScrollRequired();
      bool ScrollOnMessage(const Message& msg);
      static void setScrollVisibleEndDuration(ScrollSpeedType duration);
      //
      // scroll ende
      //

      bool isInvalid() const
      {
         return _isInvalid || (GetText().HasChanged()) || (GetTruncationText().HasChanged());
      }

      bool bindToTextNode() const;
      Candera::TextBrush* getBrush() const;
      void brushpropertyUpdate();
      FeatStd::Int16 getClientSizeX() const
      {
         return static_cast<FeatStd::Int16>(_previousLayoutClientArea.GetX());
      }
      FeatStd::Int16 getClientSizeY() const
      {
         return static_cast<FeatStd::Int16>(_previousLayoutClientArea.GetY());
      }

      // Visible filter, only for Scenecomposer
      virtual bool ComposerPropVisibleFilterTruncation() const;
      virtual bool ComposerPropVisibleFilterMultiline() const;
      virtual bool ComposerPropVisibleFilterScrolling() const;
      virtual bool ComposerPropVisibleFilterNotTextNode2D() const;

      HorizontalAlignmentType decideHorizontalAlginment(void);
      Candera::TextRendering::BidiBaseLevel::Enum getBidiBaseLevel(DirectionType& directionType);
      Candera::TextRendering::BidiBaseLevel::Enum getBidiBaseLevelForNode(const Candera::Node2D& node);
      void updateScrolldistance();
      HorizontalAlignmentType decideCultureDepHorAlginment(bool maintainHorAlignment = false);
   private:
      // layout
      FeatStd::Internal::Vector<MinimalGlyphData> _truncationTextGlyphData;
      FeatStd::Internal::Vector<MinimalGlyphData> _textGlyphData;
      TextWidgetsLayouter _layouter;
      Candera::Vector2 _previousLayoutClientArea;
      Candera::Vector2 _previousLayoutClientAreaScroll;
      Candera::Vector2 _preferredSize;
      unsigned _isInvalid;
      unsigned _isInvalidScroll;
      bool _isInvalidateReq;
      bool _isInvalidateReqUpdate;

      // scrolling
      ScrollPosition _scrollOrigin;
      ScrollPosition _scrollTarget;
      FeatStd::Float _normalizedScrollPosition;
      ScrollLimits   _scrollLimits;
      FeatStd::Internal::Vector<MinimalGlyphData> _projectedTextGlyphData;
      ::Util::Timer*  _scrollTimer;
      ::Util::Timer*  _scrollDelayTimer;
      bool _scrollingActive;
      enum ScrollingMode
      {
         ScrollToLineEnd,
         ScrollToLineBegin,
         ScrollToNextPage,
         ScrollToTop
      };
      enum ScrollingBehaviour
      {
         OnExternal,
         OnAuto,
         OnTouch,
         None
      };
      ScrollingMode _scrollingMode;
      ScrollingBehaviour m_scrollingBehaviour;
      FeatStd::Int16	_truncationTextWidth;
      ScrollPosition m_lastScrollPosition;
      FeatStd::UInt32 m_scrollDistance;
      Candera::TextRendering::TextRect m_textRectangle;
      Candera::Globalization::TextDirection m_textDirection;

#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST
      FeatStd::Int32 _scrollDistance;
      void ScrollUpdate2();
      void ScrollUpdateDistance();
#endif


      static ScrollSpeedType _scrollVisibleEndDuration;  // time to show the tail,
      SharedStylePointer m_sharedStyle;
      std::vector<TextLayoutingListener*> m_listenerList;
      Candera::TruncationMethod m_truncationMethodInternal;
};


// Added for distorted aligment in Menu (Grid Menu)
#if defined SESA_ARABIC_LAYOUT_FIX
class LayouterHelper : public Candera::Layouter
{
   public:
      using Layouter::SetArrangeActualSize;
      friend class TextWidget2DV2;
};


#endif

/// Not Required after add onClipping funcation and LayouterHelper class : Todo confirm after testing and delete
//class BoundingRectTraverser : public ConstTreeTraverser2D {
//public:
//	BoundingRectTraverser(const Node2D& root) :
//		m_initializedBounds(false),
//		m_initializedFallbackPosition(false),
//		m_root(root)
//	{
//		m_bounds.minX = Math::MaxFloat();
//		m_bounds.minY = Math::MaxFloat();
//		m_bounds.maxX = -Math::MaxFloat();
//		m_bounds.maxY = -Math::MaxFloat();
//	}
//
//	void GetBoundingRectangle(Candera::Rectangle& rect) const
//	{
//		if (m_initializedBounds) {
//			rect.SetPosition(m_bounds.minX, m_bounds.minY);
//			rect.SetSize((m_bounds.maxX - m_bounds.minX), (m_bounds.maxY - m_bounds.minY));
//		}
//		else {
//			rect.SetPosition(m_fallbackPosition);
//			rect.SetSize(0.0F, 0.0F);
//		}
//	}
//
//protected:
//	virtual TraverserAction ProcessNode(const Node2D& node) override
//	{
//		if ((node.IsLayoutingRectangleSet()) || ((!node.IsTypeOf(Group2D::GetTypeId())) && (!node.IsTypeOf(Camera2D::GetTypeId())))) {
//			Vector2 point[4];
//
//			Candera::Rectangle boundingRect;
//			node.GetEffectiveLayoutingRectangle(boundingRect);
//
//			Matrix3x2 transform(node.GetCompositeTransform());
//
//			const Node2D* parent = node.GetParent();
//			while ((parent != 0) && (parent != &m_root)) {
//				transform *= parent->GetCompositeTransform();
//				parent = parent->GetParent();
//			}
//
//			const Float left = boundingRect.GetLeft();
//			const Float top = boundingRect.GetTop();
//			const Float width = boundingRect.GetWidth();
//			const Float height = boundingRect.GetHeight();
//
//			const Float right = left + width;
//			const Float bottom = top + height;
//
//			point[0] = transform.Multiply(Vector2(left, top));
//			point[1] = transform.Multiply(Vector2(right, top));
//			point[2] = transform.Multiply(Vector2(right, bottom));
//			point[3] = transform.Multiply(Vector2(left, bottom));
//
//			m_initializedBounds = true;
//			for (Int i = 0; i < 4; i++) {
//				const Float x = point[i].GetX();
//				const Float y = point[i].GetY();
//
//				m_bounds.minX = Math::Minimum(m_bounds.minX, x);
//				m_bounds.minY = Math::Minimum(m_bounds.minY, y);
//				m_bounds.maxX = Math::Maximum(m_bounds.maxX, x);
//				m_bounds.maxY = Math::Maximum(m_bounds.maxY, y);
//			}
//		}
//		else {
//			if ((!m_initializedBounds) && (!m_initializedFallbackPosition)) {
//				m_initializedFallbackPosition = true;
//				Matrix3x2 transform(node.GetCompositeTransform());
//				const Node2D* parent = node.GetParent();
//				while ((parent != 0) && (parent != &m_root)) {
//					transform *= parent->GetCompositeTransform();
//					parent = parent->GetParent();
//				}
//
//				m_fallbackPosition = transform.Multiply(node.GetPosition());
//			}
//		}
//
//		return ProceedTraversing;
//	}
//private:
//	struct Bounds {
//		Float minX;
//		Float minY;
//		Float maxX;
//		Float maxY;
//	};
//	bool m_initializedBounds;
//	bool m_initializedFallbackPosition;
//	Vector2 m_fallbackPosition;
//	Bounds m_bounds;
//	const Node2D& m_root;
//
//	//Forbid assignment, make it private
//	BoundingRectTraverser& operator=(const BoundingRectTraverser&);
//};
#endif
