/* #***************************************************************************************
* FILE:          TextWidget2DV2.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextWidget2DV2 is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Candera/TextEngine/TextLayoutStrategy.h"
#include "Candera/Engine2D/Core/TextNode2D.h"
#include "hmibase/util/Ticker.h"
#include "Widgets/2D/Text/TextWidget2DV2.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include "Candera/TextEngine/MinimalPreprocessingContext.h"
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRenderer.h>
#include "Candera/Engine2D/Core/TextNodeRenderer/TextNode2DLayouter.h"
#include "Candera/TextEngine/Internal/PreprocessingContextVectorIterator.h"
#include "TextWidget2DV2Helper.h"
#include "CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowTextBrushBlend.h"

typedef ::hmibase::widget::text::TextLayoutingInformation TextLayoutingCharCount;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/TextWidget2DV2.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(TextWidget2DV2);


#define DEFAULT_SCROLLSPEED 300   // 300 msec default value fro scrolling 
TextWidget2DBase::ScrollSpeedType TextWidget2DV2::_scrollVisibleEndDuration = 2000u;
#define MIN_PIXEL_FOR_NORMAL_SCROLL 40

using namespace Courier;
using namespace Candera;
#define ONELINE 1

// NOTE: further performance optimization for scrolling may be achieved by using more advanced caching of rendered text as additional render nodes.
// example1: if only page oriented scrolling is possible then a cache for each line can improve the performance (e.g. a render node for each column)
// example2: if only line scrolling is possible (no pages) then a cache for columns (a glyph has to be rendered into all caches that intersect with that glyph) can improve the performance (e.g. a render node for each line)
// example3: if scrolling is possible on both axis then a tile cache (a glyph has to be rendered into all caches that intersect with that glyph) can improve the performance (e.g. a render node for each tile)
// as part of the projection of the preprocessed text glyphs the update of the caches has to be managed (moving the caches position, clear caches that move out of the visible area and reuse them for new caches)

/******************************************************************************
*  Constructor
******************************************************************************/
TextWidget2DV2::TextWidget2DV2() :
   _ptrCurrentCulture(),
   _previousLayoutClientArea(0.0f, 0.0f),
   _previousLayoutClientAreaScroll(0.0f, 0.0f),
   _preferredSize(0.0f, 0.0f),
   _isInvalid(0),
   _normalizedScrollPosition(0.0f),
   _scrollTimer(0),
   _scrollDelayTimer(0),
   _scrollingActive(false),
   _scrollingMode(ScrollToLineEnd),
   m_scrollingBehaviour(None),
   _truncationTextWidth(0),
   _isInvalidScroll(0),
   m_extensionStyleType(NoExtensionStyleType),
   m_sharedStyle(NULL),
   m_truncationMethodInternal(Candera::NoneTruncation),
   _isInvalidateReq(false),
   _isInvalidateReqUpdate(false),
   m_lastScrollPosition(ScrollPosition()),
   m_scrollDistance(0),
   m_textRectangle(Candera::TextRendering::TextRect::GetMin()),
   m_textDirection(Candera::Globalization::_undefined)
#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST
   ,
   _scrollDistance(1),
#endif
{
   _layouter.Init(this);
}


/***************************************************************************
*     destructor
****************************************************************************/
TextWidget2DV2::~TextWidget2DV2()
{
   ScrollTimerStop(true);

   if (0 != GetNode())
   {
      GetNode()->SetLayouter(0);
   }
   Candera::TextBrush* brush = getBrush();
   if (brush)
   {
      brush->Text().Set("", 0);
      brush->PreprocessedText().Set(Candera::TextRendering::PreprocessingContext::Iterator());
   }
   _ptrCurrentCulture = NULL;
}


/***************************************************************************
*  InitWidget
****************************************************************************/
void TextWidget2DV2::InitWidget()
{
   Base::InitWidget();
   decideExtensionStyleType(m_extensionStyleType);
}


/******************************************************************************
*  Update
******************************************************************************/
void TextWidget2DV2::Update()
{
   Base::Update();
   if (_isInvalidateReqUpdate == true)
   {
      brushpropertyUpdate();
      Invalidate();
      _isInvalidateReqUpdate = false;
   }

   if (_isInvalid || _isInvalidScroll)
   {
      brushpropertyUpdate();

      if (GetScrollBehavior() != Off)
      {
#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST
         ScrollUpdate2();
#else
         ScrollUpdate();
#endif
      }
   }
}


void TextWidget2DV2::ScrollUpdate()
{
   if (_scrollingActive && (_textGlyphData.Size() > 0))
   {
      _isInvalidScroll = 1;
      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style;
      Candera::TextNode2D* textNode2D = Candera::Dynamic_Cast<Candera::TextNode2D*>(GetNode());
      Candera::TextBrush* brush = 0;

      //const Candera::TextRendering::TextRenderContext* referenceTextRenderContext = 0;

      if (0 != textNode2D)
      {
         style = textNode2D->GetStyle();
      }
      else
      {
         brush = getBrush();
         if (0 != brush)
         {
            style = brush->Style().Get();
         }
      }

      if ((0 != GetNode()) && (!style.PointsToNull()))
      {
         ScrollPosition scrollPosition;
         bool scrollPixelValid = false;
         if (GetScrollCharCount() > 0)
         {
            if (m_scrollDistance == 0)
            {
               updateScrolldistance();
            }
            scrollPixelValid = (m_lastScrollPosition.x >= _scrollTarget.x) ? true : false;
         }
         else
         {
            scrollPixelValid = (_normalizedScrollPosition >= 1.0F) ? true : false;
         }
         if (true == scrollPixelValid)
         {
            if (GetScrollBehavior() == EndOfLineRemain)
            {
               // we have reached the target position
               _scrollOrigin = _scrollTarget;
               scrollPosition = _scrollOrigin;
               scrollPosition.y = _scrollOrigin.y;
               ScrollTimerStop();
            }

            if (GetScrollBehavior() == EndOfLineOnce || GetScrollBehavior() == EndOfLineRepeat)
            {
               ScrollTimerStop();
               _scrollingMode = ScrollToLineBegin;
               ScrollTimerStart(GetScrollBehavior(), _scrollVisibleEndDuration);
               _isInvalidScroll = 0;
            }
         }
         if (_normalizedScrollPosition <= 0.0F)
         {
            // just in case of an error we reset the position to the origin position
            _normalizedScrollPosition = 0.0F;
            scrollPosition.x = _scrollOrigin.x;
            scrollPosition.y = _scrollOrigin.y;
            if (_scrollingMode == ScrollToLineBegin)
            {
               ScrollTimerStop();
               if (GetScrollBehavior() == EndOfLineRepeat)
               {
                  ScrollTimerStart(GetScrollBehavior(), GetScrollSpeed());
               }
               ScrollInitLimits();
            }
         }
         else
         {
            if (GetScrollCharCount() > 0)
            {
               scrollPosition.x += static_cast<FeatStd::Int32>(_normalizedScrollPosition);
               if (scrollPosition.x > _scrollTarget.x)
               {
                  scrollPosition.x = _scrollTarget.x;
               }
               m_lastScrollPosition = scrollPosition;
            }
            else
            {
               /* Below lines are to resolve slow scrolling happening when the value = (_scrollTarget.x - _scrollOrigin.x) is
                  less than 40, since if value < 40, the (_scrollTarget.x - _scrollOrigin.x) * _normalizedScrollPosition produces
                  a number less than 1 which after type casting to FeatStd::Int8 produces 0, thus resulting in slow scrolling(which looks as if jump of character)
               */
               if ((Candera::Globalization::RightToLeft != m_textDirection) && (_scrollTarget.x - _scrollOrigin.x < MIN_PIXEL_FOR_NORMAL_SCROLL))
               {
                  _normalizedScrollPosition = 1.0F;
               }
               // the normal update between origin position and target position
               scrollPosition.x = _scrollOrigin.x + FeatStd::Int32(FeatStd::Float(_scrollTarget.x - _scrollOrigin.x) * _normalizedScrollPosition);
               scrollPosition.y = _scrollOrigin.y + FeatStd::Int32(FeatStd::Float(_scrollTarget.y - _scrollOrigin.y) * _normalizedScrollPosition);
            }
         }

         // a float size that exceeds the max PixelSize will be mapped to the max PixelSize
         ProjectText(scrollPosition.x, scrollPosition.y,
                     (_previousLayoutClientAreaScroll.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(_previousLayoutClientAreaScroll.GetX()),
                     (_previousLayoutClientAreaScroll.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(_previousLayoutClientAreaScroll.GetY()));

         // update the brush with the projected glyphs, update the brush and schedule the scene for rendering

         Candera::TextRendering::LayoutingOptions layoutingOptions = Candera::TextRendering::LayoutingOptions();
         if (_projectedTextGlyphData.Size() > 0)
         {
            Candera::Float additionalWidth = 0.0f;
            if (_projectedTextGlyphData[0].m_position.x < 0)
            {
               additionalWidth = Candera::Float(abs(_projectedTextGlyphData[0].m_position.x));
            }

            layoutingOptions.SetSize(Candera::TextRendering::TextSize((_previousLayoutClientAreaScroll.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(_previousLayoutClientAreaScroll.GetX() + additionalWidth),
                                     (_previousLayoutClientAreaScroll.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(_previousLayoutClientAreaScroll.GetY())));
         }

         TextWidgetTextRenderContext::SetPreprocessedText(_projectedTextGlyphData, layoutingOptions, textNode2D, brush);
      }
   }
   else
   {
      _isInvalidScroll = 0;
   }
}


void TextWidget2DV2::ScrollUpdatePosition()
{
#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST
   // this is currently a frame based fix increment. the speed is dependent on the frame rate.
   FeatStd::Float speed = 1.0; //  GetScrollingSpeed();
   _normalizedScrollPosition += speed / FeatStd::Float(_scrollDistance);
#else
   if (_scrollingMode == ScrollToLineBegin)
   {
      _scrollTarget.x = 0;
      _scrollTarget.y = 0;
      _scrollOrigin = _scrollTarget;
      _normalizedScrollPosition = 0.0f;
      m_lastScrollPosition = _scrollOrigin;
   }
   if (_scrollingMode == ScrollToLineEnd)
   {
      // 0.025f -> Timer based increament,m_scrollDistance ->fixed distance scrolled
      _normalizedScrollPosition += (GetScrollCharCount() > 0) ? static_cast<FeatStd::Float>(m_scrollDistance) : 0.025f;
   }
   _isInvalidScroll = 1;
   Invalidate();
#endif
}


/******************************************************************************
*  bindToTextNode
******************************************************************************/
bool TextWidget2DV2::bindToTextNode() const
{
   return (GetNode() && GetNode()->IsTypeOf(Candera::TextNode2D::GetTypeId()));
}


/******************************************************************************
*  OnNodeChanged
******************************************************************************/
void TextWidget2DV2::OnNodeChanged()
{
   // the node has been changed. we have to set our text layouter on that node.
   Candera::Node2D* node = GetNode();
   if (0 != node)
   {
      node->SetLayouter(&_layouter);
   }

   // the node has been changed. we have to set our text layouter on that node.
   if (bindToTextNode())
   {
      Base::OnNodeChanged();
   }
   else
   {
      Candera::TextBrush* brush = getBrush();
      if (brush)
      {
         brush->Text().Set("", 0);
      }
   }
}


/******************************************************************************
*  OnBeforeNodeChanged
******************************************************************************/
void TextWidget2DV2::OnBeforeNodeChanged()
{
   if (bindToTextNode())
   {
      Base::OnBeforeNodeChanged();
      return; // connected to new TextNode2D
   }

   // If widget is already attached to a RenderNode, remove the widget effects and layouter and
   // set the cached ones back to the RenderNode.

   Candera::Node2D* node = GetNode();
   if (0 != node)
   {
      if (node->GetLayouter() == &_layouter)
      {
         node->SetLayouter(Candera::DefaultLayouter::GetInstance());
      }
   }
}


/******************************************************************************
*  AxisIntersectionTest
******************************************************************************/
static bool AxisIntersectionTest(FeatStd::Int32 left1, FeatStd::Int32 right1, FeatStd::Int32 left2, FeatStd::Int32 right2)
{
   // fast axis aligned intersection test of two segments.
   // can be combined for each axis to implement 2d axis aligned rectangle intersection test and 3D axis aligned box intersection test
   return ((right2 >= left1) && (left2 <= right1));
}


/******************************************************************************
* ProjectText
******************************************************************************/
TextWidget2DV2::ScrollLimits TextWidget2DV2::ProjectText(FeatStd::Int32 left, FeatStd::Int32 top,
      Candera::TextRendering::PixelSize width, Candera::TextRendering::PixelSize height)
{
   TextWidget2DV2::ScrollLimits scrollLimits;
   if (Candera::Globalization::RightToLeft == m_textDirection)
   {
      left = -left;
   }
   FeatStd::Int32 right = left + FeatStd::Int32(width);
   FeatStd::Int32 bottom = top + FeatStd::Int32(height);
   FeatStd::Int32 lastBottom = 0;
   bool addTruncationText = true;

   // first the previous content is cleared.
   _projectedTextGlyphData.Clear();

   // This block is to decide the truncation text to be displayed while scrolling....   addTruncationText's value will make sure to show the "..."
   if (GetScrollBehavior() != Off && (Candera::Text == m_truncationMethodInternal/*GetTruncationMethod()*/) && (_textGlyphData.Size() > 0))
   {
      if (Candera::Globalization::RightToLeft != m_textDirection)
      {
         FeatStd::Int32 size = _textGlyphData[_textGlyphData.Size() - 1].m_right - static_cast<FeatStd::Int32>(GetMaximumSize().GetX());
         // this condition is to decide if the last available characters can replace the truncation text (can fit in the width)
         ((left /*+ _truncationTextWidth*/) >= size) ? addTruncationText = false : right -= (_truncationTextWidth);
      }
      else
      {
         addTruncationText = ((abs(left) < _scrollTarget.x) || (left == 0 && _textGlyphData[0].m_position.x < 0)) ? true : false;
      }
   }
   // then we search the glyphs that intersect the provided window completely or partially.
   // we use FeatStd::Int32 instead of the PixelPosition to enable a virtual scrollable area of ~ +/- 2^31 by +/- 2^31 pixels.
   // NOTE: the final position within the window is still in PixelPosition and has to be in the area of ~ +/- 2^15 by +/- 2^15 pixels.
   for (FeatStd::SizeType i = 0; i < _textGlyphData.Size(); ++i)
   {
      TextWidget2DV2::MinimalGlyphData& glyph = _textGlyphData[i];
      // we check the y axis first
      if (AxisIntersectionTest(top, bottom, glyph.GetScrollbaleY(), glyph.GetScrollbaleY() + FeatStd::Int32(glyph.m_size.height)))
      {
         // then we check the x axis
         if (AxisIntersectionTest(left, right, glyph.GetScrollbaleX() + FeatStd::Int32(glyph.m_size.width), glyph.GetScrollbaleX() + FeatStd::Int32(glyph.m_size.width)))
         {
            TextWidget2DV2::MinimalGlyphData projectedGlyph = glyph;
            // the projected glyph is positioned relative to the provided window.
            projectedGlyph.m_position.x = Candera::TextRendering::PixelPosition(projectedGlyph.GetScrollbaleX() - left);
            projectedGlyph.m_position.y = Candera::TextRendering::PixelPosition(projectedGlyph.GetScrollbaleY() - top);
            _projectedTextGlyphData.Add(projectedGlyph);
            if ((scrollLimits.m_bottom < glyph.GetScrollbaleBottom()) && (glyph.GetScrollbaleBottom() <= bottom))
            {
               // update the last visible bottom line (the glyph has to be completely in the window)
               scrollLimits.m_bottom = glyph.GetScrollbaleBottom();
            }
         }
         if (scrollLimits.m_right < glyph.GetScrollbaleRight())
         {
            // update the most outer right position. it is the limit for any scrolling mechanism (like a scrollbar or an automatic scrolling)
            scrollLimits.m_right = glyph.GetScrollbaleRight();
         }
      }
      if (lastBottom < glyph.GetScrollbaleBottom())
      {
         // update the most bottom position. it is required to detect if there are more pages left.
         lastBottom = glyph.GetScrollbaleBottom();
      }
   }
   // this block is to add truncation text "..." in the end of the projectedgyplh if truncation is text and scrolling is enabled
   if ((GetScrollBehavior() != Off) && (Candera::Text == m_truncationMethodInternal/*GetTruncationMethod()*/) && addTruncationText && (_projectedTextGlyphData.Size() > 0))
   {
      if (Candera::Globalization::RightToLeft != m_textDirection)
      {
         TextWidget2DV2::MinimalGlyphData lastGlyphData = _projectedTextGlyphData[_projectedTextGlyphData.Size() - 1];
         for (FeatStd::SizeType i = 0; i < _truncationTextGlyphData.Size(); i++)
         {
            TextWidget2DV2::MinimalGlyphData glyphData = (_truncationTextGlyphData)[i];
            glyphData.m_characterPosition = lastGlyphData.m_characterPosition + 1;
            // position at the end of the last line
            glyphData.m_position.x = static_cast<Candera::TextRendering::PixelPosition>(lastGlyphData.m_position.x + lastGlyphData.m_size.width + (_truncationTextGlyphData)[i].m_position.x);  //Truncation text appended next to last possible visible character
            // use the same base line as the last line
            glyphData.m_position.y = static_cast<Candera::TextRendering::PixelPosition>(lastGlyphData.m_glyphBaseLine - glyphData.m_glyphTop);
            _projectedTextGlyphData.Add(glyphData);
            //scrollLimits.m_right += _truncationTextGlyphData[0].GetScrollbaleRight();		// increase the scroll limit right value
         }
      }
      else
      {
         if (_projectedTextGlyphData.Size() > 0)
         {
            FeatStd::UInt16 truncTextWidth = _truncationTextWidth;
            size_t count = 0;
            for (size_t i = 0; i < _projectedTextGlyphData.Size(); ++i)
            {
               if (_projectedTextGlyphData[i].m_position.x > _truncationTextWidth)
               {
                  count = i;
                  break;
               }
            }
            for (size_t i = 0; i < count; ++i)
            {
               _projectedTextGlyphData.Remove(0);                                    // These many characters can be removed from glyph data to fit in the elipses
            }

            TextWidget2DV2::MinimalGlyphData lastGlyphData = _projectedTextGlyphData[0];
            for (size_t i = 0; i < _truncationTextGlyphData.Size(); ++i)
            {
               TextWidget2DV2::MinimalGlyphData glyphData = (_truncationTextGlyphData)[i];
               glyphData.m_position.x = static_cast<Candera::TextRendering::PixelPosition>(glyphData.m_position.x + (lastGlyphData.m_position.x - _truncationTextWidth));        // elipses to be just next to text and not at the boundary of left most in x axis
               glyphData.m_right = static_cast<Candera::TextRendering::PixelPosition>(glyphData.m_right + (lastGlyphData.m_position.x - _truncationTextWidth));
               _projectedTextGlyphData.Insert(0, glyphData);
            }
         }
      }
   }
   if (0 == scrollLimits.m_bottom)
   {
      // if the page has only newlines at the bottom the projection will result in a bottom limit of 0
      if (bottom < lastBottom)
      {
         // it is not the last page so the provided bottom will be used as limit
         scrollLimits.m_bottom = bottom;
      }
      else
      {
         // there are no more lines. so, we use the last bottom as limit
         scrollLimits.m_bottom = lastBottom;
      }
   }
   return scrollLimits;
}


/******************************************************************************
* ScrollInitLimits
******************************************************************************/
void TextWidget2DV2::ScrollInitLimits()
{
   Candera::Vector2 size = _previousLayoutClientAreaScroll;
   _normalizedScrollPosition = 0;
   if (GetScrollBehavior() != Off)
   {
      _scrollOrigin.x = 0;
      _scrollOrigin.y = 0;
      // initialize scroll position top left
      if (Candera::Globalization::RightToLeft != m_textDirection)
      {
         // first initial projection of the preprocessed glyphs to the target window
         _scrollLimits = ProjectText(_scrollOrigin.x, _scrollOrigin.y,
                                     (size.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetX()),
                                     (size.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetY()));
         // set the scroll target to a position that show the end of the text (out most right glyph if right aligned with the window)
         if (FeatStd::Float(_scrollLimits.m_right) > size.GetX())
         {
            _scrollTarget.x = _scrollLimits.m_right - Candera::TextRendering::PixelSize(size.GetX());
            _scrollTarget.y = _scrollOrigin.y;
         }
         else
         {
            _scrollTarget.x = 0;
            _scrollTarget.y = 0;
         }
      }
      else
      {
         _scrollLimits = ProjectText(_scrollOrigin.x, _scrollOrigin.y,
                                     (size.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetX()),
                                     (size.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetY()));
         _scrollTarget.x = ((m_textRectangle.GetLeft() < 0) ? (-m_textRectangle.GetLeft()) : 0);
         _scrollTarget.y = _scrollOrigin.y;
      }
      _scrollingMode = ScrollToLineEnd;
#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST
      ScrollUpdateDistance();
#endif
   }
}


void TextWidget2DV2::OnParentViewRenderingEnabled(bool enable)
{
   if (enable)
   {
      //if a culture is known by this widget and it is different from the current one it means that the culture was changed while the view was inactive => trigger culture changed
      if ((_ptrCurrentCulture != NULL) && (_ptrCurrentCulture != Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()))
      {
         CultureChanged();
      }
   }
}


/******************************************************************************
*  OnMeasure, called from internal TextLayouter class
******************************************************************************/
Candera::Vector2 TextWidget2DV2::OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea)
{
   FEATSTD_UNUSED(node);

   if ((_isInvalid > 0) || (clientArea != _previousLayoutClientArea))
   {
      //remember the culture from the moment when the text was processed last time, we will use it later on view rendering enabled to detect if the culture was changed while the view was inactive
      _ptrCurrentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();

      _previousLayoutClientArea = clientArea;
      _previousLayoutClientAreaScroll = clientArea;

      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style;
      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> extensionStyle;
      Candera::TextNode2D* textNode2D = Candera::Dynamic_Cast<Candera::TextNode2D*>(GetNode());
      Candera::TextBrush* brush = 0;
      bool needUpdatePreferredSize = true;
      Candera::Globalization::TextDirection textDirection = Candera::Globalization::_undefined;
      if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().PointsToNull() == false)
      {
         textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
         m_textDirection = textDirection;
      }
      TextStyleManager::ExtensionType textShrinkStyles = TextStyleManager::Condensed;					//This will be used to decide for various extension styles. For Initial : condensed if shrink enabled
      bool isShrinkEnabled = (GetShrinkMethod() == CondenseAndShrink) ? true : false;						// to distinguish between GetShrinkMethod() and normal text style in case of condensed and shrinked does not fit

      const Candera::TextRendering::TextRenderContext* referenceTextRenderContext = 0;

      if (0 != textNode2D)
      {
         style = textNode2D->GetStyle();
         m_sharedStyle = style;
         if (GetText() != textNode2D->GetText())
         {
            textNode2D->SetText(GetText());
         }
         Candera::TextNodeRenderer* textNodeRenderer = textNode2D->GetTextNodeRenderer();
         if (0 != textNodeRenderer)
         {
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
            referenceTextRenderContext = textNodeRenderer->GetMeasureReferenceContext();
#endif
         }
      }
      else
      {
         brush = getBrush();
         if (0 != brush)
         {
            style = GetStyle();										// setting the text style from scene composer as initial style
            m_sharedStyle = style;
            SizeType strLen = StringPlatform::Length(GetText().GetCString());
            TChar* dupString = FEATSTD_NEW_ARRAY(TChar, strLen + 1);
            StringPlatform::Copy(dupString, GetText().GetCString());
            brush->Text().Set(dupString, MemoryManagement::ArrayDisposer<const TChar*>::Dispose);

            //brush->Text().Set(GetText().GetCString(), 0);
            referenceTextRenderContext = brush->GetTextRenderContext();
         }
      }
      if (!style.PointsToNull())
      {
         std::string styleName = "";
         if (!GetStyle().PointsToNull() && GetStyle().GetPointerToSharedInstance()->GetName())
         {
            styleName = GetStyle().GetPointerToSharedInstance()->GetName();
         }
         Candera::Vector2 size = clientArea;
         if (GetMaximumSize().GetY() == -1 && clientArea.GetY() > 0)
         {
            size.SetY(FeatStd::Float(Candera::TextRendering::PixelSize(-1)));
         }
         Candera::Vector2 additionalSize(0.0, 0.0);
         Candera::Vector2 additionalEffectSize = getAdditionalSizeforEffects(additionalSize);
         size.SetX(size.GetX() + additionalEffectSize.GetX());
         size.SetY(size.GetY() + additionalEffectSize.GetY());

         //Below step will be used for drop shadow when required for any project // TODO to check if same requirement for outline effect
         //Current requirement is text should not move and only shadow will be clipped if outside the maximum size
         //size.SetX(size.GetX() - additionalEffectSize.GetX());
         //size.SetY(size.GetY() - additionalEffectSize.GetY());

         _scrollingActive = false;

         TextWidgetTextRenderContext::Truncation truncation = TextWidgetTextRenderContext::None;
         TextWidgetTextRenderContext::Truncation truncationWithShrink = TextWidgetTextRenderContext::None;
         TruncationDirectionType truncDirection = GetTruncationDirection();
         std::string truncationPostString;

         switch (m_truncationMethodInternal /*GetTruncationMethod()*/)
         {
            case Candera::Hard:
               truncation = TextWidgetTextRenderContext::Hard;
               break;
            case Candera::Soft:
               truncation = TextWidgetTextRenderContext::Soft;
               break;
            case Candera::NoneTruncation:
               truncation = TextWidgetTextRenderContext::None;
               truncDirection = Candera::Right;
               break;
            /*case Candera::Shrink:
               truncation = TextWidgetTextRenderContext::None;
               truncationPostString = GetTruncationText().GetCString();
               break;*/
            case Candera::Text:
               truncation = TextWidgetTextRenderContext::Text;
               truncationPostString = GetTruncationText().GetCString();
               break;
            default:
               break;
         }
         bool getStyle = false;
         decideExtensionStyleType(m_extensionStyleType);
         TextStyleManager::StyleId styleId = TextStyleManager::getInstance().getTextStyleId(styleName);
         if (true == isShrinkEnabled)
         {
            truncationWithShrink = truncation;
            truncation = TextWidgetTextRenderContext::None;
         }
         getStyle = getExtensionStyle(styleId, extensionStyle, textShrinkStyles, isShrinkEnabled);
         if (getStyle)
         {
            style = extensionStyle;
            m_sharedStyle = style;
         }
         ////////////////////////////////////////////////////////////////////////////////////
         //
         // truncation string "..."
         //
         // TODO: perform the truncation text pre-shaping only if the truncation text or the style has been changed
         // the context for the truncation text is always processed with no truncation. the glyphs are stored in m_truncationTextGlyphData
         Candera::TextRendering::TextRenderer textRenderer;
         TextWidgetTextRenderContext truncationTextRenderContext(TextWidgetTextRenderContext::None, style, referenceTextRenderContext, _truncationTextGlyphData, 0, 0, truncDirection, Candera::Vector2(0, 0) , GetScrollBehavior(), 0);
         // the preprocessed glyphs are cleared because they will be updated by the following lines.
         _truncationTextGlyphData.Clear();
         Candera::TextRendering::TextProperties truncationTextProperties(truncationPostString.c_str());
         // update the preprocessed glyphs of the truncation text. this can be cached and only updated if the style of the truncation text is changed.
         textRenderer.Render(truncationTextRenderContext, truncationTextRenderContext.LayoutingOptions(), truncationTextRenderContext.ShapingOptions(), truncationTextProperties);
         // in this case calling finish is not mandatory. but, to ensure a correct behavior for possible future changes we will call it anyway.
         truncationTextRenderContext.Finish();
         _truncationTextWidth = truncationTextRenderContext.GetTextRectangle().GetRight();

         ////////////////////////////////////////////////////////////////////////////////////
         //
         // text content string "Text"
         //
         FeatStd::Internal::CriticalSectionLocker textLock(GetText().GetCriticalSection());
         const FeatStd::Char* text = GetText().GetCString();
         Candera::TextRendering::TextProperties textProperties(text);
         bool processForArabic = true;

         for (int shrinkFont = 20; shrinkFont > 0; --shrinkFont)
         {
            if (textShrinkStyles == TextStyleManager::None)
            {
               Candera::TextRendering::TextRenderer textRenderer;
               TextWidgetTextRenderContext truncationTextRenderContext(TextWidgetTextRenderContext::None, style, referenceTextRenderContext, _truncationTextGlyphData, 0, 0, truncDirection, Candera::Vector2(0, 0), GetScrollBehavior(), 0);
               // the preprocessed glyphs are cleared because they will be updated by the following lines.
               _truncationTextGlyphData.Clear();
               Candera::TextRendering::TextProperties truncationTextProperties(truncationPostString.c_str());
               // update the preprocessed glyphs of the truncation text. this can be cached and only updated if the style of the truncation text is changed.
               textRenderer.Render(truncationTextRenderContext, truncationTextRenderContext.LayoutingOptions(), truncationTextRenderContext.ShapingOptions(), truncationTextProperties);
               // in this case calling finish is not mandatory. but, to ensure a correct behavior for possible future changes we will call it anyway.
               truncationTextRenderContext.Finish();
               _truncationTextWidth = truncationTextRenderContext.GetTextRectangle().GetRight();
            }
            _textGlyphData.Clear();
            // now we can create the context for preprocessing the text itself. the preprocessed truncation glyphs and its pixel width are provided as additional parameters.
            TextWidgetTextRenderContext textRenderContext(truncation, style, referenceTextRenderContext, _textGlyphData, &_truncationTextGlyphData, _truncationTextWidth, truncDirection, additionalSize, GetScrollBehavior(), GetMaximumNumberOfLines(), GetMultiLineLayouting());
            if (GetScrollBehavior() == Off)
            {
               // provide some layout options as properties e.g. word wrap
               textRenderContext.LayoutingOptions().SetWordWrapEnabled(GetMultiLineLayouting() ? GetWordWrap() : false);
               // if scrolling is not enabled we will provide the limitation size for the text layout
               textRenderContext.LayoutingOptions().SetSize(Candera::TextRendering::TextSize(
                        (size.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetX()),
                        (size.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetY())));
               /// SetHorizontalAlignment, SetVerticalAlignment, LineSpacingFactor, MultiLineLayouting  ...
               textRenderContext.LayoutingOptions().SetHorizontalAlignment(decideCultureDepHorAlginment());
               textRenderContext.LayoutingOptions().SetVerticalAlignment(GetVerticalAlignment());
               textRenderContext.LayoutingOptions().SetOrientationRealignmentEnabled(true);
               textRenderContext.SetCultureDependentAlginment(GetCultureDependentAlignment());
               if (GetDirection() == hmibase::widget::text::enDirection::Legacy)
               {
                  if (Candera::Globalization::RightToLeft == textDirection)
                  {
                     textRenderContext.LayoutingOptions().SetRightToLeftDirection(true);
                  }
               }
               else
               {
                  hmibase::widget::text::enDirection::Enum direction = GetDirection();
                  textRenderContext.ShapingOptions().SetBidiBaseLevel(getBidiBaseLevel(direction));
               }
               if (GetMultiLineLayouting())
               {
                  textRenderContext.LayoutingOptions().SetMultilineTextEnabled(true);
                  if (::hmibase::widget::text::enMultiLineHorizontalAlignment::Default != GetMultiLineHorizontalAlignment())
                  {
                     TextWidget2DBase::HorizontalAlignmentType horAlignment = decideHorizontalAlginment();
                     textRenderContext.LayoutingOptions().SetHorizontalAlignment(horAlignment);
                  }
                  textRenderContext.LayoutingOptions().SetWordWrapEnabled(GetWordWrap() ? true : false);
                  if (GetLineSpacingInPixels() <= 0)
                  {
                     textRenderContext.LayoutingOptions().SetLineSpacing(Candera::TextRendering::PixelSize(((style->GetMetrics().lineHeight * GetLineSpacingFactor()) > 0) ? (style->GetMetrics().lineHeight * GetLineSpacingFactor()) : 1.0f));
                  }
                  else
                  {
                     textRenderContext.LayoutingOptions().SetLineSpacing(Candera::TextRendering::PixelSize((GetLineSpacingInPixels() > 0) ? GetLineSpacingInPixels() : 1.0f));
                  }
               }
            }
            else
            {
               // provide some layout options as properties e.g. word wrap
               textRenderContext.LayoutingOptions().SetSize(Candera::TextRendering::TextSize(
                        (size.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetX()),
                        (size.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetY())));
               textRenderContext.LayoutingOptions().SetHorizontalAlignment(decideCultureDepHorAlginment());
               textRenderContext.LayoutingOptions().SetVerticalAlignment(GetVerticalAlignment());
               textRenderContext.LayoutingOptions().SetWordWrapEnabled(GetMultiLineLayouting() ? GetWordWrap() : false);
               textRenderContext.LayoutingOptions().SetOrientationRealignmentEnabled(true);
            }
            // now update the the preprocessed text glyphs
            textRenderer.Render(textRenderContext, textRenderContext.LayoutingOptions(), textRenderContext.ShapingOptions(), textProperties);
            if ((true == GetMultiLineLayouting() && (ONELINE == textRenderContext.GetNumberOfLines()))
                  && (::hmibase::widget::text::enMultiLineHorizontalAlignment::Default != GetMultiLineHorizontalAlignment()))
            {
               _textGlyphData.Clear();
               textRenderContext.LayoutingOptions().SetHorizontalAlignment(decideCultureDepHorAlginment());
               textRenderer.Render(textRenderContext, textRenderContext.LayoutingOptions(), textRenderContext.ShapingOptions(), textProperties);
            }
            // Re Render with alignment change so layouter change it back to current alignment if text is more than max size and full content is Latin in arabic culture
            if ((Candera::Globalization::RightToLeft == textDirection) && true == textRenderContext.IsBlitSkipped())
            {
               if (true == textRenderContext.IsTextEnglish())
               {
                  processForArabic = false;
                  _textGlyphData.Clear();
                  textRenderContext.ClearPendingGlyphData();
                  textRenderContext.SetInitialGlyphProcessing(false);
                  textRenderContext.SetCurrentLine(0);
                  textRenderContext.LayoutingOptions().SetHorizontalAlignment((GetHorizontalAlignment() == Candera::HCenter) ? Candera::HRight : decideCultureDepHorAlginment(true));
                  textRenderer.Render(textRenderContext, textRenderContext.LayoutingOptions(), textRenderContext.ShapingOptions(), textProperties);
               }
               else
               {
                  if (!GetCultureDependentAlignment())
                  {
                     _textGlyphData.Clear();
                     textRenderContext.ClearPendingGlyphData();
                     textRenderContext.SetInitialGlyphProcessing(false);
                     textRenderContext.SetCurrentLine(0);
                     textRenderContext.LayoutingOptions().SetHorizontalAlignment(GetHorizontalAlignment());
                     textRenderer.Render(textRenderContext, textRenderContext.LayoutingOptions(), textRenderContext.ShapingOptions(), textProperties);
                  }
               }
            }
            if (false == GetFixedTextBoxSize())
            {
               if (textDirection != Candera::Globalization::RightToLeft)
               {
                  needUpdatePreferredSize = ((textRenderContext.GetTextRectangle().GetLeft() - textRenderContext.GetTextRectangle().GetRight()) < static_cast<Candera::TextRendering::PixelPosition>(size.GetX())) ? false : true;
               }
               else
               {
                  needUpdatePreferredSize = (textRenderContext.GetTextRectangle().GetRight() < static_cast<Candera::TextRendering::PixelPosition>(size.GetX())) ? false : true;
               }
            }
            if (CondenseAndShrink == GetShrinkMethod() && !styleName.empty())
            {
               if (textRenderContext.FitsIntoRectangle() == false)
               {
                  bool considerNormalStyle = false;
                  considerNormalStyle = getExtensionStyleWithShrink(styleId, style, extensionStyle, textShrinkStyles, isShrinkEnabled);
                  if (true == considerNormalStyle)
                  {
                     truncation = truncationWithShrink;
                  }
                  if (brush != NULL)
                  {
                     brush->Style().Set(style);
                  }
                  m_sharedStyle = style;
                  if (TextStyleManager::None != textShrinkStyles)
                  {
                     if (isShrinkEnabled == false)
                     {
                        textShrinkStyles = TextStyleManager::None;
                     }
                     continue;
                  }
               }
            }

            // This call of Finish is important. because (depending on the truncation mode) it will either add pending glyphs of the text
            // if it fits into the windows or the truncation glyphs if it is truncated.
            if (Candera::NoneTruncation != m_truncationMethodInternal/*GetTruncationMethod()*/ && Candera::Left == GetTruncationDirection())
            {
               textRenderContext.ProcessGlyphForLeft();			// This is to check the position in complete text from which it should be diplayed in the given width in Left Truncation
            }

            textRenderContext.Finish(processForArabic);

            m_textRectangle = textRenderContext.GetTextRectangle();

            std::vector<TextLayoutingListener*>::iterator it = m_listenerList.begin();
            if (it != m_listenerList.end())
            {
               TextLayoutingCharCount info;
               info.SetCharCountEachLine(textRenderContext.GetCharCountInEachLineHelper());
               info.SetRectTopEachLine(textRenderContext.GetRectTopEachLine());
               while (it != m_listenerList.end())
               {
                  if (NULL != *it)
                  {
                     (*it)->OnLayoutingInfoChanged(info);
                  }
                  it++;
               }
            }

            ////////////////////////////////////////////////////////////////////////////////////
            //
            // finally set the preprocessed glyphs (possible truncated) at the brush
            //
            if (GetScrollBehavior() != Off)
            {
               ScrollInitLimits();
               TextWidgetTextRenderContext::SetPreprocessedText(_projectedTextGlyphData, textRenderContext.LayoutingOptions(), textNode2D, brush);
               (GetScrollStartDelay() > 0) ? m_scrollingBehaviour = OnAuto, startScrollDelayTimer() : ScrollInvokeOnAuto();
               _isInvalidScroll = 1;
            }
            else
            {
               TextWidgetTextRenderContext::SetPreprocessedText(_textGlyphData, textRenderContext.LayoutingOptions(), textNode2D, brush);
               _isInvalidScroll = 0;
            }

            // the measured size is the preferred size
            if (GetScrollBehavior() == Off)
            {
               if (textDirection != Candera::Globalization::RightToLeft)
               {
                  // the measured size is the preferred size
                  _preferredSize.SetX(static_cast<float>(textRenderContext.GetTextRectangle().GetRight()));
                  _preferredSize.SetY(static_cast<float>(textRenderContext.GetTextRectangle().GetBottom()));
               }
               else
               {
                  _preferredSize.SetX(static_cast<float>(textRenderContext.GetTextRectangle().GetRight() - textRenderContext.GetTextRectangle().GetLeft()));
                  _preferredSize.SetY(static_cast<float>(textRenderContext.GetTextRectangle().GetBottom() - textRenderContext.GetTextRectangle().GetTop()));
               }
            }
            else
            {
               _preferredSize.SetX(static_cast<float>(size.GetX()));
               _preferredSize.SetY(static_cast<float>(size.GetY()));
            }

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextWidget2DV2::OnMeasure() Input Text Id=(%d),Is Text Truncated=(%5s),Input Text=(%s)", GetText().GetId(), (true == textRenderContext.IsTextTruncated()) ? "true" : "false", GetText().GetCString()));
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextWidget2DV2::OnMeasure() Calculated text size(width, height)=(%f, %f),widget=%p,ViewName=(%50s),WidgetName=(%s)", textRenderContext.GetTextRectangle().GetRight(), textRenderContext.GetTextRectangle().GetBottom(), this, GetViewName(), GetLegacyName()));
            break;
         }
         _preferredSize += additionalEffectSize;
         if (brush != NULL)
         {
            brush->Style().Set(style);
            brush->CacheArea().Set(_preferredSize);
            brush->Update();
         }
         // store the current layout size to cache the preprocessed glyphs until the layout size changes from outside
         //_previousLayoutClientArea = size;
         _previousLayoutClientAreaScroll = size;
         //The Below conditions of Bitmap cache removed so it can work with glyph cache also.
         if ((brush != NULL) && needUpdatePreferredSize /*&& brush->CacheType().Get() == BitmapCache*/)
         {
            if (GetMaximumSize().GetX() >= 0)
            {
               _preferredSize.SetX(GetMaximumSize().GetX() + additionalEffectSize.GetX());
            }
            if (GetMaximumSize().GetY() >= 0)
            {
               _preferredSize.SetY(GetMaximumSize().GetY() + additionalEffectSize.GetY());
            }
         }

         _isInvalid = 0;
         _isInvalidateReq = true;
         _isInvalidateReqUpdate = true;
      }
   }
   return _preferredSize;
}


#if defined SESA_ARABIC_LAYOUT_FIX
/******************************************************************************
*  OnArrange, called from internal TextLayouter class
******************************************************************************/
void TextWidget2DV2::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   node.SetPosition(clientArea.GetPosition());
   LayouterHelper::SetArrangeActualSize(_preferredSize);
   if (!ArabicLayouterPatch::IsSceneEnabled(node))
   {
      OnClipping(node, clientArea);
   }
   if (true == _isInvalidateReq)
   {
      Invalidate();
      _isInvalidateReq = false;
   }
}


/******************************************************************************
*  OnClipping, called from internal TextLayouter class
******************************************************************************/
void TextWidget2DV2::OnClipping(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   RenderNode* renderNode = Dynamic_Cast<RenderNode*>(&node);
   if (0 != renderNode)
   {
      renderNode->SetClippingRect(clientArea);
   }
}


#else
/******************************************************************************
*  OnArrange, called from internal TextLayouter class
******************************************************************************/
void TextWidget2DV2::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   node.SetPosition(clientArea.GetPosition());
#if defined SESA_ARABIC_LAYOUT_FIX
   TextWidgetsLayouter::SetArrangeActualSize(Vector2(_preferredSize.GetX(), _preferredSize.GetY()));
#endif
}


#endif


/****************************************************************************
*  scroll test, returns true if scrolling is required else false
****************************************************************************/
bool TextWidget2DV2::ScrollRequired()
{
   return true;
}


/****************************************************************************
*  Called when Localization/Culture has changed during runtime
****************************************************************************/
void TextWidget2DV2::CultureChanged()
{
   if (0 != GetNode())
   {
      // the text has been changed. therefore we need a new layout call and because the content has been changed the OnMeassure has to be performed.
      Candera::Layouter::InvalidateLayout(GetNode());
   }

   //remember the last culture here in order to detect culture changes while the view is inactive
   _ptrCurrentCulture = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance();
   if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().PointsToNull() == false)
   {
      m_textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
   }

   _isInvalid = 2;
   triggerUpdate();
}


/****************************************************************************
*  Message object to be processed.
*  Returns true of message is consumed
****************************************************************************/
bool TextWidget2DV2::OnMessage(const Message& msg)
{
   bool bIsMsgConsumed = Base::OnMessage(msg);
   if (!bIsMsgConsumed)
   {
      return ScrollOnMessage(msg);
   }
   return bIsMsgConsumed;
}


/******************************************************************************
*  CloneFrom, Interface used by ListWidget to clone properties
******************************************************************************/
bool TextWidget2DV2::CloneFrom(const ControlTemplateCloneableWidget* originalWidget,
                               ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const TextWidget2DV2* ow = CLONEABLE_WIDGET_CAST<const TextWidget2DV2*>(originalWidget);
      if (ow == NULL)
      {
         return false;
      }

      SetMultiLineLayouting(ow->GetMultiLineLayouting());
      SetWordWrap(ow->GetWordWrap());
      SetAlpha(ow->GetAlpha());
      SetScrollBehavior(ow->GetScrollBehavior());
      SetScrollSpeed(ow->GetScrollSpeed());
      SetScrollStartMode(ow->GetScrollStartMode());
      SetLineSpacingFactor(ow->GetLineSpacingFactor());
      SetLineSpacingInPixels(ow->GetLineSpacingInPixels());
      SetMaximumNumberOfLines(ow->GetMaximumNumberOfLines());
      SetStyle(ow->GetStyle());
      SetTextColor(ow->GetTextColor());
      //   SetCulture(ow->GetCulture());
      SetHorizontalAlignment(ow->GetHorizontalAlignment());
      SetVerticalAlignment(ow->GetVerticalAlignment());
      SetTruncationMethod(ow->GetTruncationMethod());
      SetTruncationText(ow->GetTruncationText());
      SetMaximumSize(ow->GetMaximumSize());
      SetTruncationDirection(ow->GetTruncationDirection());
      SetOutlineColor(ow->GetOutlineColor());
      SetOutlineWidth(ow->GetOutlineWidth());
      SetFixedTextBoxSize(ow->GetFixedTextBoxSize());
      SetBold(ow->GetBold());
      SetItalic(ow->GetItalic());
      SetShrinkMethod(ow->GetShrinkMethod());
      SetScrollStartDelay(ow->GetScrollStartDelay());
      SetTransRefs(ow->GetTransRefs());
      SetMultiLineHorizontalAlignment(ow->GetMultiLineHorizontalAlignment());
      SetDirection(ow->GetDirection());
      SetScrollCharCount(ow->GetScrollCharCount());
      SetCultureDependentAlignment(ow->GetCultureDependentAlignment());
      if (ControlTemplateBinding::IsTextBindable(*this))
      {
         SetText(ControlTemplateBinding::GetTextValue(*this));
      }
      else
      {
         SetText(ow->GetText());
      }

      cloned = true;
   }
   return cloned;
}


/******************************************************************************
*  OnChanged
******************************************************************************/
void TextWidget2DV2::OnChanged(::FeatStd::UInt32 propertyId)
{
   switch (propertyId)
   {
      case MaximumSizePropertyId:   // todo : maximum property has to be checked in all cases !!!
         if (NULL != GetNode())
         {
            GetNode()->SetBoundingRectangle(Candera::Rectangle(GetNode()->GetBoundingRectangle().GetPosition(), GetMaximumSize()));
            //when using layouters, the Size property is the most appropriate way to set the size of a node
            Layouter::SetSize(*GetNode(), TextWidget2DBase::GetMaximumSize());
         }
         if (!bindToTextNode())
         {
            Candera::TextBrush* textBrush = getBrush();
            if (textBrush != 0)
            {
               if (textBrush->LayoutingArea().Get() != TextWidget2DBase::GetMaximumSize())
               {
                  textBrush->LayoutingArea().Set(TextWidget2DBase::GetMaximumSize());
                  //getBrush()->CacheArea().Set(TextWidget2DBase::GetMaximumSize());
               }
            }
         }

         // Truncation only allowed with a given area
         if (GetMaximumSize().GetX() <= 0.0f && GetTruncationMethod() != NoneTruncation)
         {
            //SetTruncationMethod(NoneTruncation);
            m_truncationMethodInternal = NoneTruncation;
         }
         else
         {
            m_truncationMethodInternal = GetTruncationMethod();
         }
         if (0 != GetNode())
         {
            // the MultiLineLayouting/word wrap/Alignment/Max Size has been changed. therefore we need a new layout call and because the content position has been changed the OnMeassure has to be performed.
            Candera::Layouter::InvalidateLayout(GetNode());
         }
         _isInvalid = 2;
         triggerUpdate();
         break;

      case AlphaPropertyId:
         if (NULL != GetNode())
         {
            GetNode()->SetAlphaValue(GetAlpha());
         }
         _isInvalid = 2;
         triggerUpdate();
         Invalidate();
         break;

      case TextColorPropertyId:
         if (!bindToTextNode())
         {
            Candera::TextBrush* textBrush = getBrush();
            if ((NULL != textBrush) && Color(textBrush->Color().Get()) != GetTextColor())
            {
               textBrush->Color().Set(GetTextColor().GetData());
            }
         }
         _isInvalid = 2;
         triggerUpdate();
         Invalidate();
         break;
      case StylePropertyId:
         updateScrolldistance();
      //lint -fallthrough no break
      case FixedTextBoxSizePropertyId:
      case DirectionPropertyId:
      case MultiLineLayoutingPropertyId:
      case WordWrapPropertyId:
      case HorizontalAlignmentPropertyId:
      case VerticalAlignmentPropertyId:
      {
         if (0 != GetNode())
         {
            // the MultiLineLayouting/word wrap/Alignment has been changed. therefore we need a new layout call and because the content position has been changed the OnMeassure has to be performed.
            Candera::Layouter::InvalidateLayout(GetNode());
         }
         _isInvalid = 2;
         triggerUpdate();
         break;
      }

      case BoldPropertyId:
      case ItalicPropertyId:
         decideExtensionStyleType(m_extensionStyleType);
      //lint -fallthrough no break
      case LineSpacingInPixelsPropertyId:
      case LineSpacingFactorPropertyId:
      case TruncationTextPropertyId:
      case TextPropertyId:
         // the text has been changed. therefore we need a new layout call and because the content has been changed the OnMeassure has to be performed.
         if (0 != GetNode())
         {
            // the text has been changed. therefore we need a new layout call and because the content has been changed the OnMeassure has to be performed.
            Candera::Layouter::InvalidateLayout(GetNode());
         }
      //lint -fallthrough no break

      case ScrollStartModePropertyId:
      case ScrollSpeedPropertyId:
      case ScrollBehaviorPropertyId:
         ScrollTimerStop();
         //ScrollInvokeOnAuto();
         _isInvalid = 2;
         // set the flag to perform the update of the preprocessed glyphs as part of the next OnMeasure call.
         Invalidate();
         triggerUpdate();
         break;

      case TruncationMethodPropertyId:
      {
         m_truncationMethodInternal = GetTruncationMethod();
         _isInvalid = 2;
         triggerUpdate();
         break;
      }
      case OutlineColorPropertyId:
      {
         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
         if (0 != renderNode)
         {
            Candera::GlOutlineTextBrushBlend* outlineBrush = Candera::Dynamic_Cast<Candera::GlOutlineTextBrushBlend*>(renderNode->GetEffect(0));
            if (0 != outlineBrush)
            {
               outlineBrush->GetOutlineEffect().OutlineColor().Set(GetOutlineColor());
            }
         }
      }
      _isInvalid = 2;
      triggerUpdate();
      break;
      case OutlineWidthPropertyId:
      {
         Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
         if (0 != renderNode)
         {
            Candera::GlOutlineTextBrushBlend* outlineBrush = Candera::Dynamic_Cast<Candera::GlOutlineTextBrushBlend*>(renderNode->GetEffect(0));
            if (0 != outlineBrush)
            {
               outlineBrush->GetOutlineEffect().OutlineWidth().Set(GetOutlineWidth());
            }
         }
      }
      _isInvalid = 2;
      triggerUpdate();
      break;
      case VisiblePropertyId:
      case EnabledPropertyId:
      {
         Base::OnChanged(propertyId);
      }
      _isInvalid = 2;
      triggerUpdate();
      break;
      case ScrollCharCountPropertyId:
         updateScrolldistance();
      //lint -fallthrough no break
      case TruncationDirectionPropertyId:
      case MaximumNumberOfLinesPropertyId:
      default:
         _isInvalid = 2;
         triggerUpdate();
         break;
   };
}


// #########################################################################################
//
// Scenecomposer property visibility filter
//
// #########################################################################################


bool TextWidget2DV2::ComposerPropVisibleFilterTruncation() const
{
   return (GetTruncationMethod() != NoneTruncation) && ComposerPropVisibleFilterNotTextNode2D() == true; /*&& (GetMaximumSize().GetX() > 0.0f)*/
}


bool TextWidget2DV2::ComposerPropVisibleFilterMultiline() const
{
   return GetMultiLineLayouting() == true && ComposerPropVisibleFilterNotTextNode2D() == true;
}


bool TextWidget2DV2::ComposerPropVisibleFilterScrolling() const
{
   return GetScrollBehavior() != Off && ComposerPropVisibleFilterNotTextNode2D() == true;
}


bool TextWidget2DV2::ComposerPropVisibleFilterNotTextNode2D() const
{
   if (bindToTextNode())
   {
      return false;  // bind to TextNode2D
   }
   return true;   // bind to RenderNode
}


// #########################################################################################
//
// Scrolling text section
//
// #########################################################################################

/***************************************************************************
*  ScrollInvokeOnTouch
****************************************************************************/
void TextWidget2DV2::ScrollInvokeOnTouch(bool activate)
{
   m_scrollingBehaviour = OnTouch;
   if (GetScrollBehavior() != Candera::Off && GetScrollStartMode() == Candera::OnTouch)
   {
      if (activate && (GetScrollStartDelay() > 0))
      {
         startScrollDelayTimer();
      }
      else if (activate && ScrollRequired())
      {
         ScrollTimerStart(GetScrollBehavior(), GetScrollSpeed());
      }
      else
      {
         ScrollTimerStop();
      }
   }
}


/***************************************************************************
*  ScrollInvokeOnAuto
****************************************************************************/
void TextWidget2DV2::ScrollInvokeOnAuto()
{
   if (GetScrollBehavior() != Candera::Off && GetScrollStartMode() == Candera::Auto)
   {
      if (ScrollRequired())
      {
         ScrollTimerStart(GetScrollBehavior(), GetScrollSpeed());
      }
   }
}


/***************************************************************************
*  ScrollInvokeOnExternal
****************************************************************************/
void TextWidget2DV2::ScrollInvokeOnExternal()
{
   if (GetScrollBehavior() != Candera::Off && ScrollRequired())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextWidget2DV2::startScroll() for %s", GetLegacyName()));
      ScrollTimerStart(GetScrollBehavior(), GetScrollSpeed());
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "TextWidget2DV2::startScroll() not required for %s", GetLegacyName()));
      ScrollTimerStop();
   }
}


/***************************************************************************
*  startScroll()
****************************************************************************/
void TextWidget2DV2::startScroll()
{
   m_scrollingBehaviour = OnExternal;
   if ((GetScrollBehavior() != Candera::Off) && (GetScrollStartDelay() > 0))
   {
      startScrollDelayTimer();
   }
   else
   {
      ScrollInvokeOnExternal();
   }
   if (isInvalid())
   {
      triggerUpdate();
   }
}


/***************************************************************************
*  stopScroll
****************************************************************************/
void TextWidget2DV2::stopScroll()
{
   ScrollInitLimits();
   ScrollUpdate();
   ScrollTimerStop();
   if (isInvalid())
   {
      triggerUpdate();
   }
}


/***************************************************************************
*  ScrollTimerStart
***************************************************************************/
void TextWidget2DV2::ScrollTimerStart(ScrollBehaviorType /*scrollBehavior*/, ScrollSpeedType speed)
{
   if (_scrollTimer == NULL)
   {
      _scrollTimer = CANDERA_NEW(Util::Timer);
      HMI_APP_ASSERT(_scrollTimer != NULL);
      if (_scrollTimer != NULL)
      {
         _scrollTimer->setName("TextWidget2DV2", GetLegacyName());
      }
   }

   if (_scrollTimer != NULL)
   {
      if (!_scrollTimer->stopped())
      {
         _scrollTimer->stop();
      }
      if (0 == speed)
      {
         speed = DEFAULT_SCROLLSPEED;
      }
      _scrollTimer->setTimeoutWithRepeat(speed, speed);
      //ScrollInitLimits();
      _scrollingActive = true;
      _scrollTimer->start();
      //setInvalidFlag();
      _isInvalidScroll = 1;
   }
}


/***************************************************************************
*  ScrollTimerStop
****************************************************************************/
void TextWidget2DV2::ScrollTimerStop(bool deleteInstance)
{
   _scrollingActive = false;
   if (_scrollTimer != NULL)
   {
      if (!_scrollTimer->stopped())
      {
         _scrollTimer->stop();
         if (!deleteInstance)
         {
            //setInvalidFlag();
            _isInvalidScroll = 1;
         }
      }
      if (deleteInstance)
      {
         CANDERA_DELETE(_scrollTimer);
         _isInvalidScroll = 0;
         _scrollTimer = NULL;
      }
   }
}


/***************************************************************************
*  ScrollOnMessage
****************************************************************************/
bool TextWidget2DV2::ScrollOnMessage(const Message& msg)
{
   switch (msg.GetId())
   {
      case TimerExpiredMsg::ID:
      {
         const TimerExpiredMsg* timerMsg = message_cast<const TimerExpiredMsg*>(&msg);
         if (timerMsg != 0)
         {
            ::Util::Timer* timer = timerMsg->GetTimer();
            if (timer == _scrollTimer)
            {
               ScrollUpdatePosition();
               return true;
            }
            if (timer == _scrollDelayTimer)
            {
               if (_scrollDelayTimer != NULL)
               {
                  if (!_scrollDelayTimer->stopped())
                  {
                     _scrollDelayTimer->stop();
                  }
                  CANDERA_DELETE(_scrollDelayTimer);
                  _scrollDelayTimer = NULL;
               }
               switch (m_scrollingBehaviour)
               {
                  case OnAuto:
                     ScrollInvokeOnAuto();
                     break;
                  case OnExternal:
                     ScrollInvokeOnExternal();
                     break;
                  case OnTouch:
                     ScrollInvokeOnTouch(true);
                     break;
                  case None:
                  default:
                     break;
               }
               m_scrollingBehaviour = None;
               return true;
            }
         }
         break;
      }
      default:
         break;
   }
   return false;
}


/***************************************************************************
*  setScrollVisibleEndDuration
****************************************************************************/
void TextWidget2DV2::setScrollVisibleEndDuration(ScrollSpeedType duration)
{
   _scrollVisibleEndDuration = duration;
}


/***************************************************************************
*  getBrush()
****************************************************************************/
Candera::TextBrush* TextWidget2DV2::getBrush() const
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
   if (0 != renderNode)
   {
      Candera::Effect2D* effect = renderNode->GetEffect(0);
      if (0 != effect)
      {
         Candera::TextBrush* brush = Candera::Dynamic_Cast<Candera::TextBrush*>(effect->GetBrushEffect2D());
         return brush;
      }
   }
   return 0;
}


/***************************************************************************
*  brushpropertyUpdate
****************************************************************************/
void TextWidget2DV2::brushpropertyUpdate()
{
   Candera::TextBrush* brush = getBrush();

   if ((brush != 0) && brush->IsUploaded() && !GetStyle().PointsToNull())
   {
      // if (brush->Culture().Get().GetPointerToSharedInstance() != GetCulture().GetPointerToSharedInstance())
      // {
      //    brush->Culture().Set(GetCulture());
      // }

      if (brush->VerticalAlignment() != GetVerticalAlignment())
      {
         brush->VerticalAlignment().Set(GetVerticalAlignment());
      }

      if (brush->HorizontalAlignment() != GetHorizontalAlignment())
      {
         brush->HorizontalAlignment().Set(GetHorizontalAlignment());
      }

      if (Color(brush->Color().Get()) != GetTextColor())
      {
         brush->Color().Set(GetTextColor().GetData());
      }

      if (brush->Style().Get().GetPointerToSharedInstance() != GetStyle().GetPointerToSharedInstance())
      {
         if (!m_sharedStyle.PointsToNull())			// As Style is not bindable property and m_sharedStyle is set to style
         {
            brush->Style().Set(m_sharedStyle);
         }
         else
         {
            brush->Style().Set(GetStyle());
         }
      }

      if (brush->MultiLineLayouting() != GetMultiLineLayouting())
      {
         brush->MultiLineLayouting().Set(GetMultiLineLayouting());
      }

      if (brush->WordWrap() != GetWordWrap())
      {
         brush->WordWrap().Set(GetWordWrap());
      }

      if (brush->PixelwiseLineSpacing() != (GetLineSpacingInPixels() > 0) ? true : false)
      {
         brush->PixelwiseLineSpacing().Set((GetLineSpacingInPixels() > 0) ? true : false);
         brush->LineSpacing().Set(GetLineSpacingInPixels());
      }
      if ((GetLineSpacingInPixels() <= 0) && brush->LineSpacing() != GetLineSpacingFactor())
      {
         brush->LineSpacing().Set(GetLineSpacingFactor());
      }

      if (brush->LayoutingArea().Get() != TextWidget2DBase::GetMaximumSize())
      {
         brush->LayoutingArea().Set(TextWidget2DBase::GetMaximumSize());
         //brush->CacheArea().Set(TextWidget2DBase::GetMaximumSize());
      }

      brush->Update();
   }
}


TextWidget2DV2::TextWidgetsLayouter::TextWidgetsLayouter() : _textWidget(0)
{
}


TextWidget2DV2::TextWidgetsLayouter::~TextWidgetsLayouter()
{
}


void TextWidget2DV2::TextWidgetsLayouter::Init(TextWidget2DV2* widget)
{
   _textWidget = widget;
}


#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 3)))
LayoutAlignment::LayoutDirection::Enum TextWidget2DV2::TextWidgetsLayouter::GetTextLayoutDirection(Candera::Node2D& node)
#else
Candera::LayoutDirection TextWidget2DV2::TextWidgetsLayouter::GetTextLayoutDirection(Candera::Node2D& node)
#endif
{
   return GetLanguageSensitiveDirection(node);
}


void TextWidget2DV2::TextWidgetsLayouter::Dispose()
{
   // the TextWidgetsLayouter is part of TextWidget2DV2 therefore it is not deleted.
}


Layouter* TextWidget2DV2::TextWidgetsLayouter::Clone() const
{
   // the layout has to be cloned by the TextWidget2DV2.
   return Candera::DefaultLayouter::GetInstance();
}


Candera::Vector2 TextWidget2DV2::TextWidgetsLayouter::OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea)
{
   if (0 != _textWidget)
   {
      return _textWidget->OnMeasure(node, clientArea);
   }
   return Candera::Vector2();
}


void TextWidget2DV2::TextWidgetsLayouter::OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (0 != _textWidget)
   {
      _textWidget->OnArrange(node, clientArea);
      Node2D* child = node.GetFirstChild();
      while (child != 0)
      {
         Candera::Rectangle clientAreaForChild = clientArea;
         clientAreaForChild.SetLeft(0);
         clientAreaForChild.SetTop(0);
         if (child->GetLayouter() != 0)
         {
            child->GetLayouter()->Measure(*child, Candera::Vector2(1.0f, 1.0f));
            child->GetLayouter()->Arrange(*child, clientAreaForChild);
         }
         child = child->GetNextSibling();
      }
   }
}


#if defined SESA_ARABIC_LAYOUT_FIX
void TextWidget2DV2::TextWidgetsLayouter::OnClipping(Candera::Node2D& node, const Candera::Rectangle& clientArea)
{
   if (0 != _textWidget)
   {
      _textWidget->OnClipping(node, clientArea);
   }
}


#endif
#ifdef CANDERA_2D_ENABLED
//The below function has been deprecated in CGI3.6, Will be removed in next releases.
void TextWidget2DV2::TextWidgetsLayouter::OnLimitPreferredSize(Candera::Node2D& /*node*/, Candera::Vector2& /*preferredSize*/, const Candera::Vector2& /*nodeSize*/, const Candera::Vector2& /*nodeMinSize*/, const Candera::Vector2& /*nodeMaxSize*/)
{
//               DefaultLayouter::OnLimitPreferredSize(node, preferredSize, nodeM)
}


#endif
void TextWidget2DV2::TextWidgetsLayouter::OnLimitPreferredSize(Candera::CanderaObject& /*node*/, Candera::Vector2& /*preferredSize*/, const Candera::Vector2& /*nodeSize*/, const Candera::Vector2& /*nodeMinSize*/, const Candera::Vector2& /*nodeMaxSize*/)
{
//               DefaultLayouter::OnLimitPreferredSize(node, preferredSize, nodeM)
}


void TextWidget2DV2::startScrollDelayTimer()
{
   if (_scrollDelayTimer == NULL)
   {
      _scrollDelayTimer = CANDERA_NEW(Util::Timer);
      HMI_APP_ASSERT(_scrollDelayTimer != NULL);
      if (_scrollDelayTimer != NULL)
      {
         _scrollDelayTimer->setName("TextWidget2DV2DelayTimer", GetLegacyName());
      }
   }
   if (_scrollDelayTimer != NULL)
   {
      if (!_scrollDelayTimer->stopped())
      {
         _scrollDelayTimer->stop();
      }
      _scrollDelayTimer->setTimeoutWithRepeat(GetScrollStartDelay(), 0);
      _scrollDelayTimer->start();
   }
}


void TextWidget2DV2::decideExtensionStyleType(TextWidget2DV2::ExtensionStyleType& extensionStyle)
{
   if (true == GetBold() && true == GetItalic())
   {
      extensionStyle = BoldItalic;
   }
   else if (true == GetBold())
   {
      extensionStyle = Bold;
   }
   else if (true == GetItalic())
   {
      extensionStyle = Italic;
   }
   else
   {
      extensionStyle = NoExtensionStyleType;
   }
}


bool TextWidget2DV2::getExtensionStyle(StyleId styleId, SharedStylePointer& extensionStyle, TextStyleManager::ExtensionType& textShrinkStyles, bool& isShrinkEnabled)
{
   bool getStyle = false;
   switch (m_extensionStyleType)
   {
      case Bold:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::Bold, extensionStyle, Base::GetAssetProvider());
         textShrinkStyles = (true == isShrinkEnabled) ? TextStyleManager::CondensedBold : TextStyleManager::Condensed;
         break;
      case Italic:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::Italic, extensionStyle, Base::GetAssetProvider());
         textShrinkStyles = (true == isShrinkEnabled) ? TextStyleManager::CondensedItalic : TextStyleManager::Condensed;
         break;
      case BoldItalic:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::BoldItalic, extensionStyle, Base::GetAssetProvider());
         textShrinkStyles = (true == isShrinkEnabled) ? TextStyleManager::CondensedBoldItalic : TextStyleManager::Condensed;
         break;
      case NoExtensionStyleType:
      default:
         break;
   }
   return getStyle;
}


bool TextWidget2DV2::getExtensionStyleWithShrink(StyleId styleId, SharedStylePointer& style, SharedStylePointer& extensionStyle, TextStyleManager::ExtensionType& textShrinkStyles, bool& isShrinkEnabled)
{
   bool getStyle = false, condiserNormalStyle = false;
   switch (textShrinkStyles)
   {
      case TextStyleManager::Condensed:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::Condensed, extensionStyle, Base::GetAssetProvider());
         if (getStyle && (!extensionStyle.PointsToNull()))
         {
            style = extensionStyle;
            textShrinkStyles = TextStyleManager::Shrinked;						// Assigned to value with which evaluation has to be done next
            break;
         }																		// No break : so we can check shrinked style if not mapped condensed style
      case TextStyleManager::Shrinked:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::Shrinked, extensionStyle, Base::GetAssetProvider());
         if (getStyle && (!extensionStyle.PointsToNull()))
         {
            style = extensionStyle;
            textShrinkStyles = TextStyleManager::Normal;
            break;
         }																		// No break : so we can check normal style if not mapped shrinked style
         if (TextStyleManager::Shrinked != textShrinkStyles)				    //If registered style not found at all then use main style
         {
            style = GetStyle();
         }
         isShrinkEnabled = false;											// Now no shrink will be utilized and truncation method have to be used
         //truncation = truncationWithShrink;									// move back to truncation method for calculations of text rectangle etc
         condiserNormalStyle = true;
         break;
      case TextStyleManager::CondensedBold:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::CondensedBold, extensionStyle, Base::GetAssetProvider());
         if (getStyle && (!extensionStyle.PointsToNull()))
         {
            style = extensionStyle;
            textShrinkStyles = TextStyleManager::ShrinkedBold;
            break;
         }
      //break;
      case TextStyleManager::ShrinkedBold:
         getStyle = TextStyleManager::getInstance().getExtensionStyleEntries(styleId, TextStyleManager::ShrinkedBold, extensionStyle, Base::GetAssetProvider());
         if (getStyle && (!extensionStyle.PointsToNull()))
         {
            style = extensionStyle;
            textShrinkStyles = TextStyleManager::Normal;
            break;
         }
         if (TextStyleManager::ShrinkedBold != textShrinkStyles)			//If registered style not found at all then use main style
         {
            style = GetStyle();
         }
         isShrinkEnabled = false;											// Now no shrink will be utilized and truncation method have to be used
         //truncation = truncationWithShrink;									// move back to truncation method for calculations of text rectangle etc
         condiserNormalStyle = true;
         break;
      case TextStyleManager::Normal:
         //style = GetStyle();												// No fall back to Normal text style
         isShrinkEnabled = false;											// Now no shrink will be utilized and truncation method have to be used
         //truncation = truncationWithShrink;									// move back to truncation method for calculations of text rectangle etc
         condiserNormalStyle = true;
         break;
      default:
         // No fall back to Normal text style
         //style = GetStyle();
         textShrinkStyles = TextStyleManager::None;
         break;
   }
   return condiserNormalStyle;
}


Candera::Vector2 TextWidget2DV2::getAdditionalSizeforEffects(Candera::Vector2& additionalSize)
{
   Candera::Vector2 additionalEffectSize(0.0f, 0.0f);
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(GetNode());
   if (0 != renderNode)
   {
      Candera::GlOutlineTextBrushBlend* outlineBrush = Candera::Dynamic_Cast<Candera::GlOutlineTextBrushBlend*>(renderNode->GetEffect(0));
      if (0 != outlineBrush)
      {
         //TODO: Outline effect enabling after confirmations

         //Characters are moved on x coordinate based on left or right horizontal alignment.
         //and on y coordinate based on top or bottom vertical alignment.
         //Additional effect size will be calculated in case default maximum size(-1,-1) is set, i.e twice the outlinewidth.
         Candera::Int outlineWidth = Candera::Int(outlineBrush->GetOutlineEffect().OutlineWidth());
         Candera::Int outlineHeight = outlineWidth;

         if (GetMaximumSize().GetX() >= 0)
         {
            outlineWidth = (decideCultureDepHorAlginment() == Candera::HRight) ? -outlineWidth : outlineWidth;
         }
         else
         {
            additionalEffectSize.SetX(static_cast<float>(outlineWidth * 2));
         }

         if (GetMaximumSize().GetY() >= 0)
         {
            outlineHeight = (GetVerticalAlignment() == Candera::VBottom) ? -outlineHeight : outlineHeight;
         }
         else
         {
            additionalEffectSize.SetY(static_cast<float>(outlineHeight * 2));
         }

         additionalSize.SetX(static_cast<float>(outlineWidth));
         additionalSize.SetY(static_cast<float>(outlineHeight));
      }
      //TODO: GlDropShadowTextBrushBlend effect enabling when required with adjustment of drop shadow
      /*Candera::GlDropShadowTextBrushBlend* shadowTextBrush = Candera::Dynamic_Cast<Candera::GlDropShadowTextBrushBlend*>(renderNode->GetEffect(0));
      if (0 != shadowTextBrush)
      {
         Rectangle boundingRectangle = shadowTextBrush->GetDropShadowEffect().GetBorderSize();
         Rectangle boundingRectangle1;
         shadowTextBrush->GetBoundingRectangle(boundingRectangle1);
         if (boundingRectangle.GetLeft() < 0)
         {
            additionalSize.SetX(boundingRectangle.GetWidth());
         }
         if (boundingRectangle.GetTop() < 0)
         {
            additionalSize.SetY(boundingRectangle.GetHeight());
         }
         additionalEffectSize.SetX(boundingRectangle.GetWidth());
         additionalEffectSize.SetY(boundingRectangle.GetHeight());
      }*/
   }
   return additionalEffectSize;
}


void TextWidget2DV2::AddListener(TextLayoutingListener* layoutingInfoListener)
{
   m_listenerList.push_back(layoutingInfoListener);
}


/***************************************************************************
*  decideHorizontalAlginment :: Provides the Alignement depending on  MultiLineHorizontalAlignment
****************************************************************************/
TextWidget2DBase::HorizontalAlignmentType TextWidget2DV2::decideHorizontalAlginment(void)
{
   TextWidget2DBase::HorizontalAlignmentType horAlignment = GetHorizontalAlignment();
   switch (GetMultiLineHorizontalAlignment())
   {
      case ::hmibase::widget::text::enMultiLineHorizontalAlignment::HRight:
         horAlignment = Candera::HRight;
         break;
      case ::hmibase::widget::text::enMultiLineHorizontalAlignment::HCenter:
         horAlignment = Candera::HCenter;
         break;
      case ::hmibase::widget::text::enMultiLineHorizontalAlignment::HLeft:
         horAlignment = Candera::HLeft;
         break;
      case ::hmibase::widget::text::enMultiLineHorizontalAlignment::Default:
      //lint -fallthrough no break
      default:
         break;
   }
   return horAlignment;
}


/***************************************************************************
*  decideCultureDepHorAlginment :: Provides the Alignement depending on  Culture Dpendent Alignment
****************************************************************************/
TextWidget2DBase::HorizontalAlignmentType TextWidget2DV2::decideCultureDepHorAlginment(bool maintainHorAlignment)
{
   TextWidget2DBase::HorizontalAlignmentType horAlignment = GetHorizontalAlignment();
   Candera::Globalization::TextDirection textDirection = Candera::Globalization::_undefined;
   if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().PointsToNull() == false)
   {
      textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
   }
   if (!GetCultureDependentAlignment() && (Candera::Globalization::RightToLeft == textDirection))
   {
      switch (GetHorizontalAlignment())
      {
         case Candera::HLeft:
            horAlignment = Candera::HRight;
            break;
         case Candera::HCenter:
            horAlignment = Candera::HCenter;
            break;
         case Candera::HRight:
            horAlignment = Candera::HLeft;
            break;
         default:
            break;
      }
   }
   if (maintainHorAlignment && GetCultureDependentAlignment() && (Candera::Globalization::RightToLeft == textDirection))
   {
      switch (GetHorizontalAlignment())
      {
         case Candera::HLeft:
            horAlignment = Candera::HRight;
            break;
         case Candera::HCenter:
            horAlignment = Candera::HCenter;
            break;
         case Candera::HRight:
            horAlignment = Candera::HLeft;
            break;
         default:
            break;
      }
   }
   return horAlignment;
}


/***************************************************************************
*  getBidiBaseLevel :: Provides the BidiBaseLevel depending on  DirectionType set in Text widget
****************************************************************************/
Candera::TextRendering::BidiBaseLevel::Enum TextWidget2DV2::getBidiBaseLevel(DirectionType& directionType)
{
   Candera::TextRendering::BidiBaseLevel::Enum bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::Implicit;
   switch (directionType)
   {
      case hmibase::widget::text::enDirection::Implicit:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::Implicit;
         break;
      case hmibase::widget::text::enDirection::LeftToRight:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::LeftToRight;
         break;
      case hmibase::widget::text::enDirection::RightToLeft:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::RightToLeft;
         break;
      case hmibase::widget::text::enDirection::Culture:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::Culture;
         break;
      case hmibase::widget::text::enDirection::Node:
      {
         if (0 != GetNode())
         {
            bidiBaseLevel = getBidiBaseLevelForNode(*GetNode());
         }
         break;
      }
      default:
         break;
   }
   return bidiBaseLevel;
}


/***************************************************************************
*  getBidiBaseLevelForNode :: Provides the BidiBaseLevel depending on  LayoutDirection set in node attached to Text widget
****************************************************************************/
Candera::TextRendering::BidiBaseLevel::Enum TextWidget2DV2::getBidiBaseLevelForNode(const Candera::Node2D& node)
{
   Candera::TextRendering::BidiBaseLevel::Enum bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::Culture;
   const Node2D* current = &node;
   LayoutAlignment::LayoutDirection::Enum direction = LayoutAlignment::LayoutDirection::InheritDirection;
   while (direction == LayoutAlignment::LayoutDirection::InheritDirection)
   {
      CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, current, CANDERA_LINT_REASON_NONCONST)
      if (0 != current->GetLayouter())
      {
         direction = current->GetLayouter()->GetLayoutDirection(*current);
         if (direction == LayoutAlignment::LayoutDirection::AutomaticDirection)
         {
            direction = current->GetLayouter()->GetAutomaticDirectionBehavior();
         }
         if (direction == LayoutAlignment::LayoutDirection::InheritDirection)
         {
            current = current->GetParent();
            if (0 == current)
            {
               direction = LayoutAlignment::LayoutDirection::CultureDirection;
            }
         }
      }
      else
      {
         break;
      }
   }
   switch (direction)
   {
      case LayoutAlignment::LayoutDirection::LeftToRightDirection:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::LeftToRight;
         break;
      case LayoutAlignment::LayoutDirection::RightToLeftDirection:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::RightToLeft;
         break;
      case LayoutAlignment::LayoutDirection::CultureDirection:
         bidiBaseLevel = Candera::TextRendering::BidiBaseLevel::Culture;
         break;
      default:
         break;
   }
   return bidiBaseLevel;
}


void TextWidget2DV2::updateScrolldistance()
{
   Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style;
   Candera::TextNode2D* textNode2D = Candera::Dynamic_Cast<Candera::TextNode2D*>(GetNode());
   Candera::TextBrush* brush = 0;

   if (0 != textNode2D)
   {
      style = textNode2D->GetStyle();
   }
   else
   {
      brush = getBrush();
      if (0 != brush)
      {
         style = brush->Style().Get();
      }
   }

   if ((0 != GetNode()) && (!style.PointsToNull()))
   {
      m_scrollDistance = GetScrollCharCount() * static_cast<FeatStd::UInt32>(style->GetDefaultFont().GetWidth());
   }
}


//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////


#ifdef TEXTWIDGET_SCROLL_FEATURE_TEST

static FeatStd::Int32 AbsolutePixelPosition(FeatStd::Int32 p)
{
   return (p < 0) ? -p : p;
}


static FeatStd::Int32 MaxPixelPosition(FeatStd::Int32 p1, FeatStd::Int32 p2)
{
   return (p1 < p2) ? p2 : p1;
}


void TextWidget2DV2::ScrollUpdateDistance()
{
   // we will update the scroll distance because the target and origin have been changed
   // as distance we use the larger value of either distance on x axis or y axis
   _scrollDistance = MaxPixelPosition(AbsolutePixelPosition(_scrollTarget.x - _scrollOrigin.x), AbsolutePixelPosition(_scrollTarget.y - _scrollOrigin.y));
   if (_scrollDistance <= 0)
   {
      _scrollDistance = 1;
   }
}


void TextWidget2DV2::ScrollUpdate2()
{
   if ((GetScrollBehavior() != Off) && (_textGlyphData.Size() > 0))
   {
      Candera::MemoryManagement::SharedPointer<Candera::TextRendering::SharedStyle> style;
      Candera::TextNode2D* textNode2D = Candera::Dynamic_Cast<Candera::TextNode2D*>(GetNode());
      Candera::TextBrush* brush = 0;

      const Candera::TextRendering::TextRenderContext* referenceTextRenderContext = 0;

      if (0 != textNode2D)
      {
         style = textNode2D->GetStyle();
         Candera::TextNodeRenderer* textNodeRenderer = textNode2D->GetTextNodeRenderer();
         if (0 != textNodeRenderer)
         {
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
            referenceTextRenderContext = textNodeRenderer->GetMeasureReferenceContext();
#endif
         }
      }
      else
      {
         brush = getBrush();
         if (0 != brush)
         {
            style = brush->Style().Get();
            referenceTextRenderContext = brush->GetTextRenderContext();
         }
      }

      if ((0 == GetNode()) || (!style.PointsToNull()))
      {
         return;
      }

      ScrollUpdatePosition();

      ScrollPosition scrollPosition;
      bool updateScrollToLineEnd = false;

      if (_normalizedScrollPosition >= 1.0F)
      {
         // we have reached the target position. therefore we have to go to the next mode and update origin position, target position and the distance
         _normalizedScrollPosition = 0.0F;
         scrollPosition.x = _scrollTarget.x;
         scrollPosition.y = _scrollTarget.y;
         // the new target y position is the last stored scroll limit bottom position
         // if it is the same as the current scroll origin then we have reached the bottom
         bool bottomReached = _scrollOrigin.y == _scrollLimits.m_bottom;
         _scrollOrigin = _scrollTarget;
         _normalizedScrollPosition = 0.0F;
         if (bottomReached)
         {
            // we have reached the bottom.
            // therefore, we scroll back to the top
            _scrollTarget.x = 0;
            _scrollTarget.y = 0;
            _scrollingMode = ScrollToTop;
         }
         else
         {
            switch (_scrollingMode)
            {
               case ScrollToLineEnd:
                  // we have currently performed a line scroll and have seen the content of the lines.
                  // therefore we have to go to the next page
                  _scrollTarget.x = 0;
                  _scrollTarget.y = _scrollLimits.m_bottom;
                  _scrollingMode = ScrollToNextPage;
                  break;
               case ScrollToTop:
               case ScrollToNextPage:
                  // we have either scrolled to the next page or to the top
                  // next we have to scroll the lines to their end to show the line content
                  updateScrollToLineEnd = true;
                  _scrollingMode = ScrollToLineEnd;
                  break;
            }
         }
         ScrollUpdateDistance();
      }
      else if (_normalizedScrollPosition <= 0.0F)
      {
         // just in case of an error we reset the position to the origin position
         _normalizedScrollPosition = 0.0F;
         scrollPosition.x = _scrollOrigin.x;
         scrollPosition.y = _scrollOrigin.y;
      }
      else
      {
         // the normal update between origin position and target position
         scrollPosition.x = _scrollOrigin.x + FeatStd::Int32(FeatStd::Float(_scrollTarget.x - _scrollOrigin.x) * _normalizedScrollPosition);
         scrollPosition.y = _scrollOrigin.y + FeatStd::Int32(FeatStd::Float(_scrollTarget.y - _scrollOrigin.y) * _normalizedScrollPosition);
      }

      Candera::Vector2 size = _previousLayoutClientArea;

      // now we project the current window
      // a float size that exceeds the max PixelSize will be mapped to the max PixelSize
      ScrollLimits scrollLimits = ProjectText(scrollPosition.x, scrollPosition.y,
                                              (size.GetX() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetX()),
                                              (size.GetY() >= FeatStd::Float(Candera::TextRendering::PixelSize(-1))) ? Candera::TextRendering::PixelSize(-1) : Candera::TextRendering::PixelSize(size.GetY()));

      // update the brush with the projected glyphs, update the brush and schedule the scene for rendering
      Candera::TextRendering::LayoutingOptions layoutingOptions = Candera::TextRendering::LayoutingOptions();
      TextWidgetTextRenderContext::SetPreprocessedText(_projectedTextGlyphData, layoutingOptions, textNode2D, brush);

      Invalidate();

      if (updateScrollToLineEnd)
      {
         // we have finished a page oriented scroll.
         // therefore we have to update the scroll limits with the new measured limits.
         _scrollLimits = scrollLimits;
         if (FeatStd::Float(scrollLimits.m_right) > size.GetX())
         {
            // parts of the line are not shown. therefore we have to scroll to the end of the line
            _scrollTarget.x = scrollLimits.m_right - Candera::TextRendering::PixelSize(size.GetX());
         }
         else
         {
            // no line scrolling is needed. the line scrolling will be changed to a page oriented scrolling in the next update
            _scrollTarget.x = 0;
         }
         // y should stay the same for the line scrolling
         _scrollTarget.y = _scrollOrigin.y;
         ScrollUpdateDistance();
      }
   }
}


#endif
