/* ***************************************************************************************
* FILE:          TextStyleConfigurator.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextStyleConfigurator is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef _HMIBASE_TEXT_STYLE_CONFIGURATOR_H_
#define _HMIBASE_TEXT_STYLE_CONFIGURATOR_H_

#include <string>

namespace hmibase {
namespace widget {
namespace text {

/*
TextStyleConfigurator provides support to associate extension styles (bold, italic, condensed and shrinked) to a base style.

How to use it:

* Create the extension styles in scene composer, put them in subfolders names Bold, Condensed, etc to be able to manage them easier.

* Define an enumeration type in the hmi contract for all the base styles.
enum
{
   ButtonTextStyle,
   TitleTextStyle,
};


* At start-up register the base style and associate the extension styles to it.

TextStyleConfigurator& testStyleConfigurator = TextStyleConfigurator::getInstance();
testStyleConfigurator.registerStyle(ButtonTextStyle, "TextStyles##ButtonTextStyle");
testStyleConfigurator.registerExtensionStyle(ButtonTextStyle, TextStyleConfigurator::Condensed, "TextStyles##Condensed#ButtonTextStyle");
testStyleConfigurator.registerExtensionStyle(ButtonTextStyle, TextStyleConfigurator::Shrinked, "TextStyles##Shrinked#ButtonTextStyle");
testStyleConfigurator.registerExtensionStyle(ButtonTextStyle, TextStyleConfigurator::Bold, "TextStyles##Bold#ButtonTextStyle");
testStyleConfigurator.registerExtensionStyle(ButtonTextStyle, TextStyleConfigurator::BoldCondensed, "TextStyles##BoldCondensed#ButtonTextStyle");
testStyleConfigurator.registerExtensionStyle(ButtonTextStyle, TextStyleConfigurator::BoldShrinked, "TextStyles##BoldShrinked#ButtonTextStyle");

*/
class TextStyleConfigurator
{
   public:
      enum ExtensionType
      {
         None = -1,
         Normal = 0,
         Bold = 1,
         Italic = 1 << 1,
         Condensed = 1 << 2,
         Shrinked = 1 << 3,

         BoldItalic = Bold + Italic,
         CondensedBold = Condensed + Bold,
         ShrinkedBold = Shrinked + Bold,
         CondensedItalic = Condensed + Italic,
         ShrinkedItalic = Shrinked + Italic,
         CondensedBoldItalic = Condensed + Bold + Italic
      };

      typedef const std::string StyleName;
      typedef FeatStd::Int16 StyleId;

      /*Register base style with an styleId. This styleId is uniquie for each base styles*/
      virtual bool registerStyle(StyleId styleId, StyleName styleName) = 0;

      /*Unregister base style for the styleId*/
      virtual bool unregisterStyle(StyleId styleId, StyleName styleName) = 0;

      /*Register an extension style for the base style. The styleId is same as the base style*/
      virtual bool registerExtensionStyle(StyleId styleId, ExtensionType extensionType, StyleName styleName) = 0;

      /*Unregister an extension style for the styleId*/
      virtual bool unregisterExtensionStyle(StyleId styleId, ExtensionType extensionType) = 0;

      /**/
      static TextStyleConfigurator& getInstance();
};


}
}


}

#endif //_HMIBASE_TEXT_STYLE_CONFIGURATOR_H_
