/* ***************************************************************************************
* FILE:          SpellerWidget2D.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpellerWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef HTC_SPELLERWIDGET_2D_H
#define HTC_SPELLERWIDGET_2D_H

/***************************************************************************************************************
* DESCRIPTION  : The Speller widget is a GUI element that provides an on-screen keyboard to interact with user
*                input and to feed other widgets (e.g. edit field) with information entered by the user.
*                The main purpose of Speller widget is that is it serves as an input tool for words and/or numbers.
*                This is the physical widget and plays the role of Model in MVC architecture
******************************************************************************************************************/

#include "Widgets/2D/Speller/generated/SpellerWidget2DBase.h"
#include "Widgets/2D/Speller/SpellerWidget2D_View.h"
#include "Widgets/2D/Speller/SpellerWidget2D_Ctrl.h"
#include "Widgets/2D/Speller/SpellerWidget2D_Constants.h"
#include "Widgets/2D/Speller/SpellerConfig.h"
#include "Widgets/2D/Text/TextWidget2D.h"
#include "Widgets/2D/Button/ButtonWidget2D.h"
#include "Widgets/2D/Focus/FocusGroupWidget2D.h"

/**
* SpellerWidget2D Interface class
*/
class SpellerWidget2DInterface
{
   public:
      //SpellerWidget2DInterface();
      virtual ~SpellerWidget2DInterface() {}

      //Interface(if any required to be) exposed to the
      //external world should be via this class only!
      //Add the required pure virtual functions here...
};


class SpellerWidget2D : public SpellerWidget2DBase, public SpellerWidget2DInterface, public FocusGroupBase
{
   public:

      SpellerWidget2D();
      virtual ~SpellerWidget2D();

      //Type decleration according to CGI Standards
      CGI_WIDGET_RTTI_DECLARATION(SpellerWidget2D, SpellerWidget2DBase);

      static ISpellerConfig* pGetConfig();

      /**
      * Over-ridden Framework functions
      */

      //Init Interface to initailize widget and associated resources
      virtual void InitWidget();

      //a parameter with value true signifies ParentView is rendering, false signifies not rendering
      virtual void OnParentViewRenderingEnabled(bool enable);

      //Focus Handling Interface invoked by F/W when a widget gains focus
      virtual void OnFocus();

      //Lost Focus Interface is invoked from F/W when a widget loose its focus
      virtual void OnLostFocus();

      //Update Interface invoked when Message Queue is empty
      virtual void Update();

      //Message distribution Interface
      virtual bool OnMessage(const Courier::Message& msg);

      virtual bool OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point);

      /**
      * User-defined public member functions
      */

      //Content of each Speller button
      struct SpellerBtn
      {
         Candera::Node2D* m_pNode;
         ButtonWidget2D* m_pAttachedWidget;
         TextWidget2D* m_pTextWidget;
         enSplrBtnType m_enType;
         enSubSplr m_enSubSplrExist;
      };
      SpellerBtn SpellerBtnInfo[MAX_BUTTONS_PER_LAYOUT];

      //Function to get the Speller Button Info related structure
      struct SpellerBtn* p_stGetSpellerBtn()
      {
         return SpellerBtnInfo;
      }

      //Function for the business logic on Toggle SpellerButton release
      void vToggleButtonRelease(enSplrBtnType enType);
      //Function to get the Button Type associated Control character
      Candera::UInt32 enGetButtonTypeCtrlChar(enSplrBtnType enType) const;
      //Function to handle all that is required for SubSplr activation
      void vActivateSubSpeller(Candera::UInt8 u8Index);
      //Function to handle all that is required for SubSplr de-activation
      void vDeactivateSubSpeller();
      void vHandleEnableProperty();
      bool postSpellerKeyPressedMsg(const Candera::String buttonPressedText);
      bool postSubSpellerStatusMsg(const Candera::String buttonPressedText, const hmibase::widget::speller::enSubSpllerStatus::Enum subSpellerStatus);
      const Candera::TChar* pcGetActivatedSceneName() const
      {
         return m_pcActivatedSceneName;
      }

      Candera::UInt32 u32GetActiveSpellerSeqIndex() const
      {
         return m_u32ActiveSplrSeqIdx;
      }

      Candera::UInt32 u32GetActiveLayoutIndex() const
      {
         return m_u32ActiveLayoutIdx;
      }

      Candera::UInt32 u32GetActiveSubSpellerSeqIndex() const
      {
         return m_u32ActiveSubSplrSeqIdx;
      }

      //Function to set the currently focussed button in Speller
      void vSetCurFocusedIndex(Candera::UInt32 u8Index)
      {
         m_u32CurFocusedIndex = u8Index;
      }
      Candera::UInt32 u32GetCurFocusedIndex() const
      {
         return m_u32CurFocusedIndex;
      }

      void vSetSubSpellerActivationStatus(bool bStatus)
      {
         m_bIsSubSpellerActive = bStatus;
      }
      bool bGetSubSpellerActivationStatus() const
      {
         return m_bIsSubSpellerActive;
      }

      //a parameter with value true signifies handle Speller property updates, false signifies to not handle
      void vSetSpellerParentViewRendering(bool bvalue)
      {
         m_bIsSpellerParentViewRendering = bvalue;
      }
      bool bGetSpellerParentViewRendering() const
      {
         return m_bIsSpellerParentViewRendering;
      }

      //a parameter with value true signifies focus required on default button, false signifies not required
      void vSetSpellerFocussedStatus(bool bvalue)
      {
         m_bSpellerFocussedStatus = bvalue;
      }
      bool bGetSpellerFocussedStatus() const
      {
         return m_bSpellerFocussedStatus;
      }

      //Function to determine the visibility of ValidCharacters property based on Mode property
      bool cobChkValidCharPropVisibility() const;
      //Function to determine the visibility of UserDefined[Sequence/Layout] property based on Type property
      bool cobChkCustomSequenceVisibility() const;

      bool isTouchIntersectButton(const Candera::Camera2D& camera2D, const Candera::Vector2& point);

      /**
      * @Brief Class for all sub speller one time configuration, these are project
      * specific configurations which configured only once by project
      * team at the begining of application
      */
      struct SubSpellerConfig
      {
         SubSpellerConfig(bool closeOnTappingOutside, enReaction btnReactionToOpen):
            _closeOnTappingOutside(closeOnTappingOutside), _btnReactionToOpen(btnReactionToOpen)
         {
         }

         bool getCloseOnTappingOutsideStatus()
         {
            return _closeOnTappingOutside;
         }
         enReaction getbtnReactionToOpen()
         {
            return _btnReactionToOpen;
         }
         bool _closeOnTappingOutside;
         enReaction _btnReactionToOpen;
      };

      /**
      * @Brief setter to set open and close of sub speller using this only
      * once by project team at the begining of application. Sub Speller opened
      * on any one of these enReaction - enRelease, enShortPress, enLongPress
      * and enRepeatPressSecondary
      * true  - sub speller will be closed on taping out side sub speller or
      * on selecting any button in sub speller
      * false - closed only once selecting any button
      */
      static void setSubSpellerConfig(const SubSpellerConfig& subSpellerConfig)
      {
         _subSpellerConfig = subSpellerConfig;
      }

      class SpellerConfiguration
      {
         public:
            SpellerConfiguration() {};
            ~SpellerConfiguration() {};
            SpellerConfiguration(const SubSpellerConfig& subSpellerConfig);
         private:
      };

      bool isBtnReactionToOpenSubSpeller(const enReaction& reaction)
      {
         return (reaction == _subSpellerConfig.getbtnReactionToOpen());
      }

   protected:

      //Function to handle the change of widget property values
      virtual void OnChanged(::Candera::UInt32 propertyId);

      virtual void RegisterToFocusManagerImpl(Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle);

   private:

      /**
      * User-defined private Member functions
      */
      //Method to autotogglelayout Type based on valid char set in match mode
      void performAutoToggleLayout(const Candera::String strValidCharacters);
      //Function for initializing the SpellerBtn structure
      void vInitializeSpellerBtnStruct();

      //Function for traversing & populating the SceneComposer nodes
      void vTraverseAndPopulateNodes(
         const Candera::TChar* pcLayoutType,
         const Candera::TChar* pcCharacterContent);

      //Function for populating the SceneComposer nodes
      void vPopulateNodes();

      //Function to fill the Button Child Nodes
      void vFillButtonChildNodes(SpellerBtn* pSpellerBtn) const;

      //Function to set texts on the Speller buttons
      void vExtractAndSetCharOnButton(
         SpellerBtn* pSpellerBtn,
         FeatStd::Internal::Utf8TextIterator& iter
      ) const;

      //Function to form the Sequence
      const Candera::TChar* pcFormSequence() const;

      //Function to form the Custom Sequence
      const Candera::TChar* pcFormCustomSequence() const;

      //Function to concatenate the given strings to form the complete scene path
      const Candera::TChar* pcFormSceneName(
         const Candera::TChar* pcScenePath,
         const Candera::TChar* pcLayoutTypeName) const;

      //Function to concatenate the given strings to form seq in a certain format (lan_coun_mode)
      const Candera::TChar* pcConcatenateStrings(
         const Candera::TChar* pcMode,
         const Candera::TChar* pcLanguage = NULL,
         const Candera::TChar* pcCountry  = NULL) const;

      //Function to create all Speller views at startup, this include sub-speller views also
      void vCreateAllSplrViews();
      //Function to create the active (sub-)Speller view
      void vCreateActiveSplrView();
      //Function to create all Speller views at startup, this include sub-speller views also
      void vDestroyAllSplrViews();
      //Function to activate Speller view. Ideally the view should be created also should here itself
      void vActivateView();
      //Function to delete the current Speller view. Ideally the view should be destroyed also here itself
      void vDeActivateView();
      //sets the render target of all the cameras into the speller layout view to be the same as the one used by the first camera in the speller main view
      void vConfigureSpellerView();
      //Function to handle synchronous processing of a view message triggered from the widget
      bool bProcessViewCompMsg(Courier::Message& msg);

      //Function which changes the active sequence and active layout based on the new language/country received
      void vChangeActiveSeqAndLayout();
      //Function to enable or disable the buttons based on the widget 'Enable' property whenever a new layout is populated
      //void vHandleEnableProperty();
      //Function to enable or disable the buttons based on ValidChars
      void vUpdateButtonStatus(const Candera::String strValidCharacters);

      //Function to toggle the MainSplr touchability - req due to SubSplr activation/deactivation
      void vSetMainSplrTouchableStatus(bool bStatus);
      //Function to find the Sequence index based on the property SpellerType, SpellerMode, lang & country
      void vSetSequenceIdx();
      //Function to find the Layout index based on the property KeypadFormat
      void vSetLayoutIdxOfFormat();
      //Function to set the Speller button type of the extracted character from the Speller XML
      void vSetSplrBtnType(SpellerBtn* pSpellerBtn, const Candera::TChar* pLayoutCurChar) const;
      void SpellerStatusUpdMsg();
      void vSetWidget(SpellerWidget2D* pWidget);
      //Function to get the speller mode and return relevant string
      const Candera::TChar* pcGetSpellerModeString() const;

      //Function to get the layout in a sequence
      const Candera::TChar* cosGetLayout(
         Candera::UInt32 u32SequenceIndex,
         Candera::UInt32 u32LayoutIndex) const;

      //Function to control the visibilty of the marker on the button which is used to
      //denote the presence/absence of the SubSplr character
      void vSetSubSpellerCharRendering(
         const SpellerBtn* pSpellerBtn,
         bool bIsRendered) const;

      //Function to activate the Layout as per the ActivateLayoutIndex
      void vActivateLayoutIndex(Candera::UInt32 m_ActivateLayoutIndex);

      /**
      * private Member variables
      */

      //general member variables: NOT exposed in SceneComposer
      Candera::String m_sCurrentLayout;           //Stores the content of the current active layout of a sequence
      Candera::UInt32 m_u32CurFocusedIndex;       //Stores the currently focussed button in Speller
      Candera::UInt32 m_u32LastFocusedIndexBeforeSubSplrAct;  //Stores the currently focussed button in Speller

      Candera::UInt32 m_u32ActiveSplrSeqIdx;      //Stores the Speller active  index
      Candera::UInt32 m_u32ActiveLayoutIdx;       //Stores the Speller active layout index
      Candera::UInt32 m_u32ActiveSubSplrSeqIdx;   //Stores the SubSplr active  index
      Candera::enKeypadFormat m_u32ActiveKeypadFormat;    //Stores the Active Keypad Format
      const Candera::TChar* m_pcActivatedSceneName;        //Stores the scene name which is activated - either MainSplr or could be SubSplr

      //general member variables: Exposed in SceneComposer
      //Candera::String m_strAvaliableCharSet;    //Stores the Speller available Character set: A future req if needed by any project

      //flag creation related: NOT Exposed in SceneComposer
      bool   m_bIsSubSpellerActive;      //Stores true if the SubSplr is active
      bool   m_bIsSpellerParentViewRendering;//Stores true if OnParentViewRenderingEnabled() is called with true else stores false
      bool   m_bSpellerFocussedStatus;   //Required for know if Speller is the focussed widget, true signifies focussed, false otherwise

      //instance creation related: NOT exposed in SceneComposer
      SpellerWidget2D_View m_oView;               //Speller View instance in MVC architectute
      SpellerWidget2D_Ctrl m_oCtrl;               //Speller Controller instance in MVC architectute
      SpellerWidget2D*   m_pWidget;
      static SubSpellerConfig _subSpellerConfig;
      Candera::String _mainSpellerCharReceived;
   public:

      /**
      * Cda Widget Properties
      */

      CdaWidget2DDef(SpellerWidget2D, SpellerWidget2DBase)
      CdaProperties()

      CdaProperty_FocusGroupConfigured()
      CdaProperty_FocusGroupWrapAround()
      CdaProperty_FocusGroupPreserveFocus()
      CdaProperty_FocusGroupDefaultFocusOrder()
      CdaProperty_FocusGroupLayer()

      CdaPropertiesEnd()
      CdaWidgetDefEnd()
};


#endif //HTC_SPELLERWIDGET_2D_H
