/* ***************************************************************************************
* FILE:          SpellerWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpellerWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifdef WIN32
#include <windows.h>
#endif
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h"
#include "SpellerWidget2D.h"
#include "Widgets/2D/Speller/SpellerConfig.h"
#include "Courier/Visualization/ViewHandler.h"
#include "Courier/Visualization/ViewFacade.h"

/***************************************************************************************************************
* DESCRIPTION  : The Speller widget is a GUI element that provides an on-screen keyboard to interact with user
*                input and to feed other widgets (e.g. edit field) with information entered by the user.
*                The main purpose of Speller widget is that is it serves as an input tool for words and/or numbers.
*                This is the physical widget and plays the role of Model in MVC architecture
*
******************************************************************************************************************/

using namespace Candera;
using namespace Courier;
using namespace hmibase::widget::speller;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SPELLER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpellerWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(SpellerWidget2D);

/// Pointer to the instance of this singleton config class.
static ISpellerConfig* theConfig(bool deleteInstance = false)
{
   static ISpellerConfig* pInstance = NULL;
#ifdef WIN32
   static HINSTANCE hDLL = 0;

   if (deleteInstance)
   {
      if (hDLL)
      {
         FreeLibrary(hDLL);
         hDLL = 0;
      }
      pInstance = 0;
   }
   else if (pInstance == 0)
   {
      const char* libvar = getenv("SPELLERCONFIG_PATH");
      if (libvar != NULL)
      {
         hDLL = LoadLibrary(libvar);
      }
      else
      {
         hDLL = LoadLibrary("SpellerConfig.dll");
      }
      if (hDLL != NULL)
      {
         pfktCreateSpellerConfigInstance pFkt = (pfktCreateSpellerConfigInstance)GetProcAddress(hDLL, "CreateSpellerConfigInstance");
         pInstance = pFkt();
      }
      if (pInstance == NULL)
      {
         ::MessageBox(0, "File 'SpellerConfig.dll' not found!", hmibase::trace::getAppName().c_str(), MB_ICONERROR);
         exit(1);
      }
   }
#else
   // target code
   if (deleteInstance && pInstance)
   {
      delete pInstance;
      pInstance = 0;
   }
   else if (pInstance == 0)
   {
      pInstance = CreateSpellerConfigInstance();
   }
#endif
   return pInstance;
}


ISpellerConfig* SpellerWidget2D::pGetConfig()
{
   return theConfig();
}


SpellerWidget2D::SubSpellerConfig SpellerWidget2D::_subSpellerConfig(true, enLongPress);
/****************************************************************************
* Req ID      : NA
* Function    : SpellerWidget2D()
* Called by   : HMI Framework when SpellerWidget2D object is created
* Description : Constructor of class where initializations are done
* Parameters  : None
* Return      : None
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
*****************************************************************************/
SpellerWidget2D::SpellerWidget2D():

   m_sCurrentLayout(),
   m_u32CurFocusedIndex(0),
   m_u32LastFocusedIndexBeforeSubSplrAct(0),
   m_u32ActiveSplrSeqIdx(0),
   m_u32ActiveLayoutIdx(0),
   m_u32ActiveSubSplrSeqIdx(0),

   m_pcActivatedSceneName(NULL),
   m_bIsSubSpellerActive(false),
   m_bIsSpellerParentViewRendering(false),
   m_bSpellerFocussedStatus(false),
   _mainSpellerCharReceived(String())
{
   m_oView.vSetWidget(this);
   m_oCtrl.vSetWidget(this);
   vSetWidget(this);
   SetSpellerScenesPath(SCENE_PATH);
   vInitializeSpellerBtnStruct();
}


/****************************************************************************
* Req ID      : NA
* Function    : ~SpellerWidget2D()
* Called by   : HMI Framework when SpellerWidget2D object is deleted
* Description : Destructor of class where we reset necessary variables
*               while moving out of scope
* Parameters  : None
* Return      : None
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
*****************************************************************************/
SpellerWidget2D::~SpellerWidget2D()
{
   theConfig(true);
   //Delete the pointers(if required) in the reverse order of initialization
   //& finally nullify all.
   m_pcActivatedSceneName = NULL;
   m_pWidget = NULL;
}


void SpellerWidget2D::vSetWidget(SpellerWidget2D* pWidget)
{
   m_pWidget = pWidget;
}


/****************************************************************************
* Req ID      : NA
* Function    : Init
* Called by   : HMI Framework after the SpellerWidget2D object is created
* Description : Initializes the widget so that all referred resource can be
*               resolved
* Parameters  : AssetFactory Asset provider to resolve all referred
*               resources
* Return      : void
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::InitWidget()
{
   Base::InitWidget();
   //Create SpellerData instance to read XML generated file
   theConfig();
}


/***************************************************************************/
void SpellerWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case EnabledPropertyId:
      {
         //Handle only for free mode as the in match mode this can be
         //controlled via the ValidChar property
         if (FreeMode == GetMode())
         {
            m_oCtrl.vToggleAllCharBtnEnableStatus(this->IsEnabled());
         }
         break;
      }

      case ModePropertyId:
         break;

      case TypePropertyId:
         break;

      case UserDefinedSequencePropertyId:
         break;

      case ActiveLanguagePropertyId:
      {
         if (true == m_bIsSpellerParentViewRendering)
         {
            vChangeActiveSeqAndLayout();
         }

         break;
      }

      case ActiveCountryPropertyId:
      {
         if (true == m_bIsSpellerParentViewRendering)
         {
            vChangeActiveSeqAndLayout();
         }

         break;
      }

      case ValidCharSetPropertyId:
      {
         performAutoToggleLayout(GetValidCharSet());
         m_oCtrl.vToggleAllCharBtnEnableStatus(false);
         vUpdateButtonStatus(GetValidCharSet());
         SpellerStatusUpdMsg();
      }
      break;

      case AutoLayoutTogglePropertyId:
      {
         performAutoToggleLayout(GetValidCharSet());
      }
      break;

      case KeypadFormatPropertyId:
      {
         if (GetKeypadFormat() == m_u32ActiveKeypadFormat)
         {
            return;
         }
         m_u32ActiveKeypadFormat = GetKeypadFormat();
         if (true == m_bIsSpellerParentViewRendering)
         {
            vChangeActiveSeqAndLayout();
         }
      }
      break;

      case SpellerKeyPressedPropertyId:
      {
         return; // avoid invalidation
      }
      break;                                                                                                            //lint !e527

      case AllCharLayoutDisablePropertyId:
      {
         m_oCtrl.vDisableAllCharBtnStatus(GetAllCharLayoutDisable()); // avoid invalidation
      }
      break;                                                                                                            //lint !e527
      case ActivateLayoutIndexPropertyId:
      {
         if (true == m_bIsSpellerParentViewRendering)
         {
            vActivateLayoutIndex(GetActivateLayoutIndex());
         }
      }
      break;
      default:
         break;
   }
   Invalidate();
}


/****************************************************************************
* Req ID      : NA
* Function    : OnParentViewRenderingEnabled
* Called by   : HMI Framework whenever rendering of the parent view is
*               enabled or disabled
* Description : Function for the Speller to be ready with the data required
*               to be displayed
* Parameters  : enable if the rendering is enabled or disabled
* Return      : void
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::OnParentViewRenderingEnabled(bool enable)
{
   if ((true == enable) &&
         (NULL != GetAssetProvider()))
   {
      if (NULL == GetNode())
      {
         return;
      }

      //required to handle property updates only when the Speller view is active
      m_bIsSpellerParentViewRendering = true;
      //make touchable false of the base scene where Speller widget exist
      //it should be true only when the sub-speller is active
      this->SetTouchable(false);
      //Find the Sequence index based on Speller properties
      vSetSequenceIdx();

      //Find the Layout index in the based on Speller properties if auto toggle is not availabel
      vSetLayoutIdxOfFormat();
      if (NULL != theConfig())
      {
         //Find Speller Sequence Based on Valid Char Set
         if (GetAutoLayoutToggle() && (0 != m_u32ActiveSplrSeqIdx))
         {
            m_u32ActiveLayoutIdx = theConfig()->getMatchingLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx, GetValidCharSet().GetCString());
         }
         const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
         //Start traversing and populating the SceneComposer nodes based on SeqIdx & LayoutIdx
         vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
      }
   }
   else if ((m_pcActivatedSceneName) && (false == enable))
   {
      if (true == m_bIsSubSpellerActive)
      {
         vDeactivateSubSpeller();
      }
      //required to handle property updates only when the Speller view is active
      m_bIsSpellerParentViewRendering = false;
      m_u32CurFocusedIndex = 0;
      //deactivate the previous scene
      vDeActivateView();
      //now destroy all the created speller views
      //NOTE: a MAJOR point - commenting out this function leads to memory leak
      //Known but still released as the workaround is required from immediate need (customer demo)
      //on the other hand, label with the correct expected changes will soon be released
      //just awaiting the related patch from FEAT for ViewResMsg
      //for more details, refer JIRA ticket SUZUKI-11467
      //vDestroyAllSplrViews();
      vInitializeSpellerBtnStruct();
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : vInitializeSpellerBtnStruct
* Called by   : vTraverseAndPopulateNodes
* Description : Function for initializing the SpellerBtn structure
* Parameters  : None
* Return      : void
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vInitializeSpellerBtnStruct()
{
   _TODO("The structure need not be cleared if the Layout type has not changed")

   //Can be further optimized & the button instance needs not be fetched
   //every time the layout content changes!!
   for (Candera::UInt8 i = 0 ; i < MAX_BUTTONS_PER_LAYOUT; i++)
   {
      SpellerBtnInfo[i].m_pNode           = NULL;
      SpellerBtnInfo[i].m_pAttachedWidget = NULL;
      SpellerBtnInfo[i].m_pTextWidget     = NULL;
      SpellerBtnInfo[i].m_enType          = SPLRBTNTYPE_Character;
      SpellerBtnInfo[i].m_enSubSplrExist  = SUBSPLR_NoSubSplr;
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_001
*               HTC_Speller_REQ_002
* Function    : vTraverseAndPopulateNodes
* Called by   : OnParentViewRenderingEnabled
* Description : Function for traversing & populating the SceneComposer nodes
* Parameters  : the character content to be traversed and
*               the LayoutType where this has to be populated
* Return      : void
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vTraverseAndPopulateNodes(const Candera::TChar* pcLayoutType, const Candera::TChar* pcCharacterContent)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D: LayoutType: [%s]", pcLayoutType));
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D: CharacterContent: [%s]", pcCharacterContent));

   if (!GetAssetProvider())
   {
      return;
   }

   //Initializing the SpellerBtn structure to default values
   vInitializeSpellerBtnStruct();
   m_sCurrentLayout = static_cast<Candera::String>(pcCharacterContent);
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetSpellerScenesPath());
   m_pcActivatedSceneName = pcFormSceneName(GetSpellerScenesPath().GetCString(), pcLayoutType);
   SECURE_FEATSTD_STRING_ACCESS_END();

   vCreateActiveSplrView();
   vPopulateNodes();
   //make touchable true for the main speller buttons
   vSetMainSplrTouchableStatus(true);
   //enable or disable the buttons based on the widget 'Enable' property
   vHandleEnableProperty();
   //now that the scene widgets are populated with the XML data, activate the scene
   vActivateView();
   SpellerStatusUpdMsg();

   if ((m_bSpellerFocussedStatus) && (SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget))
   {
#ifdef VARIANT_S_FTR_ENABLE_SPELLER_WIDGET_SET_FOCUS_ON_BUTTONS
      SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget->OnFocus();
#endif
   }
}


/***************************************************************************/
void SpellerWidget2D::vPopulateNodes()
{
   if ((GetParentView() == NULL) || (GetParentView()->GetViewHandler() == NULL))
   {
      return;
   }
   ViewScene2D* viewScene = dynamic_cast<ViewScene2D*>(GetParentView()->GetViewHandler()->FindView(::Courier::ViewId(m_pcActivatedSceneName)));
   if (viewScene == NULL)
   {
      return;
   }
   const ViewScene::FrameworkWidgetPtrVector& widgets = viewScene->GetFrameworkWidgetPtrVector();
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_sCurrentLayout);
   FeatStd::Internal::Utf8TextIterator iter(m_sCurrentLayout.GetCString());
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];
   for (SizeType i = 0; (i < widgets.Size()) && (pSpellerBtn != &SpellerBtnInfo[MAX_BUTTONS_PER_LAYOUT]); ++i)
   {
      Candera::WidgetBase* wb = static_cast<Candera::WidgetBase*>(widgets[i]);
      ButtonWidget2D* buttonWidget = Candera::Dynamic_Cast<ButtonWidget2D*>(wb);
      TextWidget2D* textWidget = (buttonWidget == NULL) ? Candera::Dynamic_Cast<TextWidget2D*>(wb) : NULL;

      //legacy behaviour
      if (GetUseOnlyButtonsWithText())
      {
         if (buttonWidget != NULL)
         {
            pSpellerBtn->m_pAttachedWidget = buttonWidget;
            vFillButtonChildNodes(pSpellerBtn);
         }
         else if (textWidget != NULL)
         {
            pSpellerBtn->m_pTextWidget = textWidget;
            vExtractAndSetCharOnButton(pSpellerBtn, iter);
            if (!iter.EndOfString())
            {
               iter.Advance();
            }
            ++pSpellerBtn;
         }
         else
         {
            //nothing to do
         }
      }
      //new behaviour
      else
      {
         if (buttonWidget != NULL)
         {
            //if current spellerBtn has a button widget, finish the configuration of it and move to next one
            if (pSpellerBtn->m_pAttachedWidget != NULL)
            {
               vExtractAndSetCharOnButton(pSpellerBtn, iter);
               if (!iter.EndOfString())
               {
                  iter.Advance();
               }
               ++pSpellerBtn;
               if (pSpellerBtn == &SpellerBtnInfo[MAX_BUTTONS_PER_LAYOUT])
               {
                  break;
               }
            }
            pSpellerBtn->m_pAttachedWidget = buttonWidget;
            vFillButtonChildNodes(pSpellerBtn);
         }
         else if (textWidget != NULL)
         {
            pSpellerBtn->m_pTextWidget = textWidget;
         }
         else
         {
            //nothing to do
         }
      }
   }
   if (GetUseOnlyButtonsWithText())
   {
      if ((NULL != pSpellerBtn->m_pAttachedWidget) && (NULL == pSpellerBtn->m_pTextWidget))
      {
         pSpellerBtn->m_pAttachedWidget = NULL;
      }
   }
   //finish the configuration for the last speller button
   else if (!GetUseOnlyButtonsWithText() && (pSpellerBtn != &SpellerBtnInfo[MAX_BUTTONS_PER_LAYOUT]))
   {
      vExtractAndSetCharOnButton(pSpellerBtn, iter);
   }

   SECURE_FEATSTD_STRING_ACCESS_END();
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : vFillButtonChildNodes
* Called by   : vTraverseAndPopulateNodes
* Description : Function to fill the Button Child Nodes
* Parameters  : SpellerBtn*, a pointer to the SpellerBtnInfo
* Return      : void
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vFillButtonChildNodes(SpellerBtn* pSpellerBtn) const
{
   if ((NULL == pSpellerBtn) || (NULL == (pSpellerBtn->m_pAttachedWidget)))
   {
      return;
   }

   //get the nodes required as per the button widget design
   pSpellerBtn->m_pNode   = pSpellerBtn->m_pAttachedWidget->GetNode();
   //Controll the rendering of the render node required for sub-speller indication
   vSetSubSpellerCharRendering(pSpellerBtn, false);
}


/***************************************************************************/
void SpellerWidget2D::vExtractAndSetCharOnButton(SpellerBtn* pSpellerBtn, FeatStd::Internal::Utf8TextIterator& iter) const
{
   if ((NULL == pSpellerBtn) || (iter.EndOfString()))
   {
      return ;
   }

   TChar achLayoutCurChar[MAX_CHAR_BYTES];
   //copy the character from the iterator
   SizeType copiedChar = FeatStd::Internal::TextEncoding::Copy(achLayoutCurChar, MAX_CHAR_BYTES, iter.Pos(), 1/*read only one UTF-8 character*/);

   //chcek if the copied character is the sub-speller control character
   if ((copiedChar) && (theConfig()->getSubSplrCharacter() == FeatStd::Internal::TextEncoding::Ucs4(achLayoutCurChar)))
   {
      //in layout content, sub-speller character is always before the actual character to which it is attached
      //hence take a note of this and move on to the next character in the layout content
      pSpellerBtn->m_enSubSplrExist = SUBSPLR_BothCharEnabled;
      //enable the rendering of the render node required for sub-speller indication
      vSetSubSpellerCharRendering(pSpellerBtn, true);
      iter.Advance();

      if (!iter.EndOfString())
      {
         copiedChar = FeatStd::Internal::TextEncoding::Copy(achLayoutCurChar, MAX_CHAR_BYTES, iter.Pos(), 1/*read only one UTF-8 character*/);
      }
   }
   else if ((copiedChar) && (theConfig()->getSeperatorCharacter() == FeatStd::Internal::TextEncoding::Ucs4(achLayoutCurChar)))
   {
      iter.Advance();
      if (!iter.EndOfString())
      {
         copiedChar = FeatStd::Internal::TextEncoding::Copy(achLayoutCurChar, MAX_CHAR_BYTES, iter.Pos(), 1/*read only one UTF-8 character*/);
      }
   }

   //set the type of the extracted character
   vSetSplrBtnType(pSpellerBtn, achLayoutCurChar);

   //Update the copied character only if it is not a control character
   if (0 != copiedChar)
   {
      if (pSpellerBtn->m_pAttachedWidget != NULL)
      {
         pSpellerBtn->m_pAttachedWidget->SetVisible(true);
      }
      if (SPLRBTNTYPE_Character == pSpellerBtn->m_enType)
      {
         if (pSpellerBtn->m_pTextWidget != NULL)
         {
            pSpellerBtn->m_pTextWidget->SetText(achLayoutCurChar);
         }
      }
      else if (SPLRBTNTYPE_Character_Empty == pSpellerBtn->m_enType)
      {
         if (pSpellerBtn->m_pTextWidget != NULL)
         {
            pSpellerBtn->m_pTextWidget->SetText("");
         }
      }
      else if (SPLRBTNTYPE_Control_Invisible == pSpellerBtn->m_enType)
      {
         if (pSpellerBtn->m_pAttachedWidget != NULL)
         {
            pSpellerBtn->m_pAttachedWidget->SetVisible(false);
         }
      }
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : pcFormSequence
* Called by   : vSetSequenceIdx
* Description : Function to form the Sequence
* Parameters  : None
* Return      : const Candera::TChar* pcFormedSeq
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
const Candera::TChar* SpellerWidget2D::pcFormSequence() const
{
   //Get the instance of SpellerData to read the XML generated data
   //Get the Speller mode relevant string text
   const Candera::TChar* pcMode = pcGetSpellerModeString();
   //Get the default language and the default country
   const Candera::TChar* pcDefaultLanguage = theConfig()->getDefaultLanguage();
   const Candera::TChar* pcDefaultCountry  = theConfig()->getDefaultCountry();
   //Get the language and the country based on the language/country code
   const Candera::TChar* pcLanguage = theConfig()->getLanguage(GetActiveLanguage());
   const Candera::TChar* pcCountry  = theConfig()->getCountry(GetActiveCountry());
   //Declare a variable to collect the formed sequence
   const Candera::TChar* pcFormedSeq = NULL;

   if ((pcDefaultLanguage != pcLanguage) && (pcDefaultCountry != pcCountry))
   {
      //concatenate both(language & country names) with the mode to form the sequence
      pcFormedSeq = pcConcatenateStrings(pcMode, pcLanguage, pcCountry);
   }
   else if ((pcDefaultLanguage != pcLanguage) && (pcDefaultCountry == pcCountry))
   {
      //concatenate only language name with the mode to form the sequence
      pcFormedSeq = pcConcatenateStrings(pcMode, pcLanguage);
   }
   else if ((pcDefaultLanguage == pcLanguage) && (pcDefaultCountry != pcCountry))
   {
      //concatenate only country name with the mode to form the sequence
      pcFormedSeq = pcConcatenateStrings(pcMode, NULL, pcCountry);
   }
   else if ((pcDefaultLanguage == pcLanguage) && (pcDefaultCountry == pcCountry))
   {
      //form the sequence with just the configured mode
      pcFormedSeq = pcConcatenateStrings(pcMode);
   }

   return pcFormedSeq;
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : pcFormCustomSequence
* Called by   : vSetSequenceIdx
* Description : Function to form the Custom Sequence
* Parameters  : None
* Return      : const Candera::TChar* pcFormedSeq
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
const Candera::TChar* SpellerWidget2D::pcFormCustomSequence() const
{
   Candera::String sEmptyString = "";

   if (sEmptyString != GetUserDefinedSequence())
   {
      const Candera::TChar* pcFormedSeq;
      //if user-defined custom seq is not empty, return the same
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetUserDefinedSequence());
      pcFormedSeq = GetUserDefinedSequence().GetCString();
      SECURE_FEATSTD_STRING_ACCESS_END();

      return pcFormedSeq;
   }
   else
   {
      //handle the error case where the user has not assigned any custom seq
      //return the default seq in such a case
      return SEQ_DEFAULT;
   }
}


/***************************************************************************/
const Candera::TChar* SpellerWidget2D::pcFormSceneName(
   const Candera::TChar* pcScenePath, const Candera::TChar* pcLayoutTypeName) const
{
   Candera::TChar* pcConcatenatedString = CANDERA_NEW_ARRAY(Char, CONCATENATED_STRING_SIZE);

   if (NULL == pcConcatenatedString)
   {
      return NULL;
   }

   //declare a variable to store the string length of the Concatenated string
   size_t strLength = 0;

   if ((NULL != pcScenePath) && (NULL != pcLayoutTypeName))
   {
      //copy the scene path name and then add this string length to final string length
      StringPlatform::CopyPartial(pcConcatenatedString, pcScenePath, strlen(pcScenePath));
      strLength += strlen(pcScenePath);
      StringPlatform::CopyPartial((pcConcatenatedString + strLength), pcLayoutTypeName, strlen(pcLayoutTypeName));
      strLength += strlen(pcLayoutTypeName);
   }

   //terminate the string with NULL char as CopyPartial doesn't take care of this and finally return
   pcConcatenatedString[strLength] = '\0';
   return pcConcatenatedString;
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : pcConcatenateStrings
* Called by   : pcFormSequence
* Description : Function to concatenate the given strings in a certain format
* Parameters  : None
* Return      : const Candera::TChar* pcConcatenatedString
in the form of 'language_country_mode' provided all three param
are valid, else invalid param is not added to final returned string
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
const Candera::TChar* SpellerWidget2D::pcConcatenateStrings(
   const Candera::TChar* pcMode, const Candera::TChar* pcLanguage, const Candera::TChar* pcCountry) const
{
   Candera::TChar* pcConcatenatedString = CANDERA_NEW_ARRAY(Char, CONCATENATED_STRING_SIZE);

   if (NULL == pcConcatenatedString)
   {
      return NULL;
   }

   //declare a variable to store the string length of the Concatenated string
   size_t strLength = 0;

   if (NULL != pcLanguage)
   {
      //copy the language name and then add this string length to final string length
      StringPlatform::CopyPartial(pcConcatenatedString, pcLanguage, strlen(pcLanguage));
      strLength += static_cast<Candera::UInt8>(strlen(pcLanguage));
      //copy the char '_' and then add this char length to final string length
      StringPlatform::CopyPartial((pcConcatenatedString + strLength), "_", 1);
      strLength += 1;
   }

   if (NULL != pcCountry)
   {
      //copy the country name and then add this string length to final string length
      StringPlatform::CopyPartial((pcConcatenatedString + strLength), pcCountry, strlen(pcCountry));
      strLength += static_cast<Candera::UInt8>(strlen(pcCountry));
      //copy the char '_' and then add this char length to final string length
      StringPlatform::CopyPartial((pcConcatenatedString + strLength), "_", 1);
      strLength += 1;
   }

   if (NULL != pcMode)
   {
      //copy the speller mode and then add this string length to final string length
      StringPlatform::CopyPartial((pcConcatenatedString + strLength), pcMode, strlen(pcMode));
      strLength += static_cast<Candera::UInt8>(strlen(pcMode));
   }

   //terminate the string with NULL char as CopyPartial doesn't take care of this and finally return
   pcConcatenatedString[strLength] = '\0';
   return pcConcatenatedString;
}


/***************************************************************************/
void SpellerWidget2D::vCreateAllSplrViews()
{
   //Get the instance of SpellerData to read the XML generated data
   Candera::UInt32 u32LayoutTypeCount = theConfig()->getNumberOfLayoutTypes();
   ViewReqMsg* pViewReqMsg;

   for (Candera::UInt8 u8Index = 1; u8Index < u32LayoutTypeCount; u8Index++)
   {
      const TChar* sceneName;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetSpellerScenesPath())
      sceneName = pcFormSceneName(GetSpellerScenesPath().GetCString(), theConfig()->getSplrLayoutType(u8Index));
      SECURE_FEATSTD_STRING_ACCESS_END();
      //create the scene as given in the Speller XMl
      if (sceneName == 0)
      {
         return;
      }
      pViewReqMsg = COURIER_MESSAGE_NEW(ViewReqMsg)(Courier::ViewAction::Create, ::Courier::ViewId(sceneName), true, true);

      if (NULL != pViewReqMsg)
      {
         //process the Courier message immediately
         bool bRetValue = bProcessViewCompMsg(*pViewReqMsg);

         if (bRetValue)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the creation of view [%s] succeeded", this, sceneName));
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the creation of view [%s] falied", this, sceneName));
         }
      }
   }
}


/***************************************************************************/
void SpellerWidget2D::vCreateActiveSplrView()
{
   if (NULL == m_pcActivatedSceneName)
   {
      return;
   }
   ViewReqMsg* pViewReqMsg;
   pViewReqMsg = COURIER_MESSAGE_NEW(ViewReqMsg)(Courier::ViewAction::Create, ::Courier::ViewId(m_pcActivatedSceneName), true, true);
   if (NULL != pViewReqMsg)
   {
      if (bProcessViewCompMsg(*pViewReqMsg))
      {
         ETG_TRACE_COMP_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the creation of view [%s] succeeded", this, m_pcActivatedSceneName));
         // add invalidation dependency
         if (GetParentView())
         {
            GetParentView()->AddInvalidationDependency(::Courier::ViewId(m_pcActivatedSceneName));
            Courier::View* view = GetParentView()->GetViewHandler()->FindView(::Courier::ViewId(m_pcActivatedSceneName));
            if (view)
            {
               view->AddInvalidationDependency(GetParentView()->GetId());
            }
         }
      }
      else
      {
         ETG_TRACE_SYS_MIN_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the creation of view [%s] falied", this, m_pcActivatedSceneName));
      }
   }
}


/***************************************************************************/
void SpellerWidget2D::vDestroyAllSplrViews()
{
   //Get the instance of SpellerData to read the XML generated data
   Candera::UInt32 u32LayoutTypeCount = theConfig()->getNumberOfLayoutTypes();
   ViewReqMsg* pViewReqMsg;

   for (Candera::UInt8 u8Index = 1; u8Index < u32LayoutTypeCount; u8Index++)
   {
      //destroy all the scenes as given in the Speller XMl
      //doing a check before destroying a view that whether it is created or not, is not required
      //as ViewReqMsg takes care of it
      const TChar* sceneName;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(GetSpellerScenesPath());
      sceneName = pcFormSceneName(GetSpellerScenesPath().GetCString(), theConfig()->getSplrLayoutType(u8Index));
      SECURE_FEATSTD_STRING_ACCESS_END()
      if (sceneName == 0)
      {
         return;
      }

      pViewReqMsg = COURIER_MESSAGE_NEW(ViewReqMsg)(Courier::ViewAction::Destroy, ::Courier::ViewId(sceneName), false, false);

      if (NULL != pViewReqMsg)
      {
         // remove invalidation dependency
         if (GetParentView())
         {
            GetParentView()->RemoveInvalidationDependency(::Courier::ViewId(sceneName));
         }

         //process the Courier message immediately
         if (bProcessViewCompMsg(*pViewReqMsg))
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the destruction of view [%s] succeeded", this, sceneName));
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the destruction of view [%s] falied", this, sceneName));
         }
      }
   }
}


/***************************************************************************/
void SpellerWidget2D::vActivateView()
{
   if (NULL == m_pcActivatedSceneName)
   {
      return;
   }

   //TODO("create the scene as and when required rather than creating all at startup");
   //ViewReqMsg *pViewReqMsg = COURIER_MESSAGE_NEW(ViewReqMsg) (Courier::ViewAction::Create, ::Courier::ViewId(m_pcActivatedSceneName), true);
   //if(NULL != pViewReqMsg)
   //    pViewReqMsg->Post();
   //activate the relevant scene now
   ActivationReqMsg* pActivationReqMsg = COURIER_MESSAGE_NEW(ActivationReqMsg)(::Courier::ViewId(m_pcActivatedSceneName), true, true);

   if (NULL != pActivationReqMsg)
   {
      //process the Courier message immediately
      bool bRetValue = bProcessViewCompMsg(*pActivationReqMsg);

      if (bRetValue)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the activation of view [%s] succeeded", this, m_pcActivatedSceneName));
         if (GetConfigureSpellerScenes())
         {
            vConfigureSpellerView();
         }
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the activation of view [%s] falied", this, m_pcActivatedSceneName));
      }
   }
}


/***************************************************************************/
void SpellerWidget2D::vConfigureSpellerView()
{
   RenderTarget2D* rt = NULL;
   ViewScene2D* layoutView = NULL;
   ViewScene2D* mainSpellerView = NULL;
   if (GetParentView() != NULL)
   {
      //get the RT of the first camera in the main speller view
      mainSpellerView = GetParentView()->ToViewScene2D();

      //find the speller layout view
      if ((m_pcActivatedSceneName != NULL) && (GetParentView()->GetViewHandler() != NULL))
      {
         View* view = GetParentView()->GetViewHandler()->FindView(ViewId(m_pcActivatedSceneName));
         if (view != NULL)
         {
            layoutView = view->ToViewScene2D();
         }
      }
   }
   //ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D.vConfigureSpellerView [%u]: rt %s found, layoutView %s found", this,
   //                    (rt != NULL ? "" : "NOT"),
   //                    (layoutView != NULL ? "" : "NOT")));

   //iterate all cameras and replace their RT
   if (layoutView != NULL)
   {
      if ((mainSpellerView != NULL) && (!mainSpellerView->GetCameraPtrVector().Empty()))
      {
         ViewScene2D::CameraPtrVector::Iterator itMainCamera = mainSpellerView->GetCameraPtrVector().Begin();
         for (ViewScene2D::CameraPtrVector::Iterator itLayoutCamera = layoutView->GetCameraPtrVector().Begin(); ((itLayoutCamera != layoutView->GetCameraPtrVector().End()) && (itMainCamera != mainSpellerView->GetCameraPtrVector().End())); ++itLayoutCamera, itMainCamera++)
         {
            rt = (*itMainCamera)->GetRenderTarget();
            if (rt != NULL)
            {
               (*itLayoutCamera)->SetRenderTarget(rt);
            }
         }
      }
      layoutView->Invalidate();
   }
}


/***************************************************************************/
void SpellerWidget2D::vDeActivateView()
{
   if (NULL == m_pcActivatedSceneName)
   {
      return;
   }

   ActivationReqMsg* pActivationReqMsg = COURIER_MESSAGE_NEW(ActivationReqMsg)(::Courier::ViewId(m_pcActivatedSceneName), false, false);

   if (NULL != pActivationReqMsg)
   {
      //process the Courier message immediately
      bool bRetValue = bProcessViewCompMsg(*pActivationReqMsg);

      if (bRetValue)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the de-activation of view [%s] succeeded", this, m_pcActivatedSceneName));
      }
      else
      {
         //log error trace
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: posting the de-activation of view [%s] falied", this, m_pcActivatedSceneName));
      }
   }

   //TODO("Ideally the view should also be destroyed now");
   //ViewReqMsg *pViewReqMsg = COURIER_MESSAGE_NEW(ViewReqMsg) (Courier::ViewAction::Destroy, ::Courier::ViewId(m_pcActivatedSceneName), true);
   //if(NULL != pViewReqMsg)
   //    pViewReqMsg->Post();
}


/***************************************************************************/
bool SpellerWidget2D::bProcessViewCompMsg(Courier::Message& msg)
{
   if (NULL != GetParentView())
   {
      //lint -e1788 Variable 'referer' is referenced only by its constructor or destructor
      //Apply reference counting via message referrer (manages message object lifetime from now on)
      //Explicitly taken care as the message dispatching by the framework to the subscribed component is bypassed here.
      Courier::MessageReferrer referer(&msg);
      //now start the synchrono**s processing
      //        ViewHandler* pViewHandler = dynamic_cast<::Courier::ViewHandler*>(GetParentView()->GetViewHandler());
      IViewHandler* pViewHandler = GetParentView()->GetViewHandler();

      if (NULL != pViewHandler)
      {
         Courier::ViewFacade oViewFacade;
         return oViewFacade.Init(pViewHandler) && oViewFacade.OnViewComponentMessage(msg);
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: bProcessViewCompMsg failed as pViewHandler is NULL", this));
      }

      return false;
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: bProcessViewCompMsg failed as GetParentView() is NULL", this));
   }

   return false;
}


/***************************************************************************/
void SpellerWidget2D::vChangeActiveSeqAndLayout()
{
   //Get the instance of SpellerData to read the XML generated data
   //get the sequence index after forming the sequence based on speller mode, language & country proeprties
   vSetSequenceIdx();
   //set the layout index and update the same into the member variable m_u32ActiveLayoutIdx
   vSetLayoutIdxOfFormat();
   //deactivate the previous scene
   vDeActivateView();
   const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

   if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
   {
      (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
   }

   //reset the focus to 0 (a project-specific configuration)
   m_u32CurFocusedIndex = 0;
   //Now traverse and populate the new layout
   vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
}


/***************************************************************************/
void SpellerWidget2D::vHandleEnableProperty()
{
   //In FreeMode, character buttons are by default valid, but toggle all the
   //character buttons enablity based on the Speller widget 'Enable' property value
   if (FreeMode == GetMode())
   {
      m_oCtrl.vToggleAllCharBtnEnableStatus(this->IsEnabled());
   }

   //In MatchMode, ValidChar property controls the enability
   //if ValidChar is null, it will disable all the character buttons
   //if ValidChar has the complete character set as that displayed in the active layout,
   //then it will enable all the character buttons.
   if (MatchMode == GetMode())
   {
      m_oCtrl.vToggleAllCharBtnEnableStatus(false);
      vUpdateButtonStatus(GetValidCharSet());
   }

   //Note: Speller does not modify the control buttons enablity
   //the project presentation logic has to explicity set the correct value to the
   //enable property of each control button explicity based on requirement & need
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_001
* Function    : vUpdateButtonStatus
* Called by   : vSetValidCharacters
* Description : Function to enable or disable the buttons based on ValidChars
* Parameters  : const Candera::String
* Return      : void
* History     : 2013-09-10(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vUpdateButtonStatus(const Candera::String strValidCharacters)
{
   //create a iterator for the string text content of the selected layout
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(strValidCharacters);
   FeatStd::Internal::Utf8TextIterator iter(strValidCharacters.GetCString());

   // iterate for number of items in the ValidChar string
   // but take care to enable the buttons with valid char only if the widget is enabled
   while ((!iter.EndOfString()) && (this->IsEnabled()))
   {
      TChar achLayoutCurChar[MAX_CHAR_BYTES];
      FeatStd::Internal::TextEncoding::Copy(achLayoutCurChar, MAX_CHAR_BYTES, iter.Pos(), 1/*read only one UTF-8 character*/);
      struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];
      //if(NULL == pSpellerBtn) { return; }

      // iterate for Number of times the buttons present on NumPad
      while ((pSpellerBtn) && (pSpellerBtn->m_pAttachedWidget))
      {
         _TODO("Following usecase to be handled in future:...")
         //ValidChar is more than the number of buttons and it so happens
         //that all the buttons are enabled, then the loop shall terminate somehow
         //This is a usecase when there are muliple layouts. In such a case
         //ValidChar for all the layouts will come once but only one layout will be active

         if ((pSpellerBtn->m_pAttachedWidget->IsEnabled())
            )
         {
            pSpellerBtn++;
            continue;
         }

         if (pSpellerBtn->m_pTextWidget && (SPLRBTNTYPE_Character == (pSpellerBtn->m_enType)))
         {
            Candera::Int32 i32Res;
            SECURE_FEATSTD_STRING_ACCESS_BEGIN(pSpellerBtn->m_pTextWidget->GetText());
            const Candera::TChar* cButtonChar =
               (pSpellerBtn->m_pTextWidget->GetText()).GetCString();
            i32Res = FeatStd::Internal::TextEncoding::Compare(cButtonChar, achLayoutCurChar);
            SECURE_FEATSTD_STRING_ACCESS_END();

            if (0 == i32Res)
            {
               //make button enabled
               if (pSpellerBtn->m_pAttachedWidget)
               {
                  pSpellerBtn->m_pAttachedWidget->SetEnabled(true);
               }

               break;
            }
         }
         else if (SPLRBTNTYPE_Control_Space == (pSpellerBtn->m_enType))
         {
            if (0 == FeatStd::Internal::TextEncoding::Compare(SPACE_CHAR, achLayoutCurChar))
            {
               if (pSpellerBtn->m_pAttachedWidget)
               {
                  pSpellerBtn->m_pAttachedWidget->SetEnabled(true);
               }
               break;
            }
         }
         //move to next button node
         pSpellerBtn++;
      }

      //move to next character
      iter.Advance();
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
}


void SpellerWidget2D::SpellerStatusUpdMsg()
{
   if (!m_pWidget)
   {
      return;
   }
   if (!(m_pWidget->GetParentView()))
   {
      return;
   }

   const Char* widgetName = m_pWidget->GetParentView()->GetId().CStr();

   if (m_pcActivatedSceneName == 0 || widgetName == 0 || m_pWidget->GetLegacyName() == 0)
   {
      return;
   }

   SpellerStatusChangedUpdMsg* pSpellerStatusChangedUpdMsg = COURIER_MESSAGE_NEW(SpellerStatusChangedUpdMsg)(
            ::Courier::ViewId(m_pWidget->GetParentView()->GetId().CStr()),
            ::Courier::Identifier(m_pWidget->GetLegacyName()),
            ::Courier::ViewId(m_pcActivatedSceneName));
   if (NULL != pSpellerStatusChangedUpdMsg)
   {
      pSpellerStatusChangedUpdMsg->Post();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D_Ctrl [%p]: fired the courier msg 'SpellerStatusChangedUpdMsg'", this));
   }
}


/***************************************************************************/
void SpellerWidget2D::vSetMainSplrTouchableStatus(bool bStatus)
{
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

   while ((pSpellerBtn) && (pSpellerBtn->m_pAttachedWidget))
   {
      pSpellerBtn->m_pAttachedWidget->SetTouchable(bStatus);
      pSpellerBtn++;
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : vSetSequenceIdx
* Called by   : OnParentViewRenderingEnabled
* Description : Function to find the Sequence index based on the property
SpellerType, SpellerMode, lang & country
* Parameters  : None
* Return      : void
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vSetSequenceIdx()
{
   //Get the instance of SpellerData to read the XML generated data
   //         _TODO("Hardcoded for the time-being to read only SpellerType sequence!!")
   //Sequence name to be formed based on active lang & country
   //alongwith Speller Mode & Type properties

   switch (GetType())
   {
      case Pin:
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(SEQ_PIN);
         break;

      case Phone:
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(SEQ_PHONE);
         break;

      case HouseNo:
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(SEQ_HOUSENO);
         break;

      case BTPassKey:
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(SEQ_BTPASSKEY);
         break;

      case Others:
         //check speller mode, language & country proeprties to form the seq
      {
         const TChar* formSequence = pcFormSequence();
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(formSequence);
         FEATSTD_SAFE_DELETE_ARRAY(formSequence);
      }
      break;

      case CustomSequence:
         //collect the statically configured user-defined seq
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(pcFormCustomSequence());
         break;

      case Default: //fallthrough
      default:
         m_u32ActiveSplrSeqIdx = theConfig()->getSequenceIndex(SEQ_DEFAULT);
         break;
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : vSetLayoutIdxOfFormat
* Called by   : OnParentViewRenderingEnabled
* Description : Function to find the Layout index based on the property KeypadFormat
* Parameters  : None
* Return      : void
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vSetLayoutIdxOfFormat()
{
   //Find the Layout index of the Layout which is of the passed 'KeypadFormat'
   KeypadFormatType formatTye = GetKeypadFormat();
   switch (formatTye)
   {
      case Basic_QWERTY: //upper case QWERTY
         formatTye = Basic_QWERTY;
         break;
      case Basic_ABC: //upper case ABC
         formatTye = Basic_ABC;
         break;
      case Basic_qwerty: //lower case qwerty
         formatTye = Basic_qwerty;
         break;
      case Basic_abc: //lower case abc
         formatTye = Basic_abc;
         break;
      case Numeric: //numeric layout
         formatTye = Numeric;
         break;
      case Special: //special/more characters or symbols
         formatTye = Special;
         break;
      default:
         formatTye = Basic_QWERTY;
   }
   m_u32ActiveLayoutIdx = theConfig()->getLayoutIndexOfFormat(m_u32ActiveSplrSeqIdx, static_cast<uint32_t>(formatTye));
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : vSetSplrBtnType
* Called by   : vTraverseAndPopulateNodes
* Description : Function to set the Speller button type of the extracted
*               character from the Speller XML
* Parameters  : SpellerBtn*, a pointer to the SpellerBtnInfo
*               const Candera::TChar*, the character extracted from the XML
* Return      : void
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vSetSplrBtnType(SpellerBtn* pSpellerBtn, const Candera::TChar* pLayoutCurChar) const
{
   if (!pSpellerBtn)
   {
      return;
   }

   Candera::UInt32 u32Ucs4Val = FeatStd::Internal::TextEncoding::Ucs4(pLayoutCurChar);

   pSpellerBtn->m_enType = theConfig()->getButtonBasedOnXmlInfo(u32Ucs4Val);
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : enGetButtonTypeCtrlChar
* Called by   : vToggleButtonRelease
* Description : Function to get the Button Type associated Control character
* Parameters  : enSpellerBtnType
* Return      : Candera::UInt32
* History     : 2013-08-14(kea2kor) Intermediate release
****************************************************************************/
Candera::UInt32 SpellerWidget2D::enGetButtonTypeCtrlChar(enSplrBtnType enType) const
{
   return theConfig()->getButtonTypeCtrlChar(enType);
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_002
* Function    : pcGetSpellerModeString
* Called by   : pcFormSequence
* Description : Function to get the speller mode and return relevant string
* Parameters  : None
* Return      : const Candera::TChar*
* History     : 2013-10-01(kea2kor) Intermediate release
****************************************************************************/
const Candera::TChar* SpellerWidget2D::pcGetSpellerModeString() const
{
   //Check the value of the property SpellerMode.
   //Based on this value, return relevant text
   if (MatchMode == GetMode())
   {
      return TEXT_MATCHMODE;
   }
   else
   {
      return TEXT_FREEMODE;
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_008
* Function    : cosGetLayout
* Called by   : vTraverseAndPopulateNodes
* Description : Function for getting the layout in a particular sequence
* Parameters  : Candera::UInt32 u32SequenceIndex
*               Candera::UInt32 u32LayoutIndex (valid values 0...number of layouts-1)
* Return      : Candera::String
* History     : 2013-07-30(kea2kor) Intermediate release
****************************************************************************/
const Candera::TChar* SpellerWidget2D::cosGetLayout(UInt32 u32SequenceIndex, UInt32 u32LayoutIndex) const
{
   //Get the instance of SpellerData to read the XML generated data
   //return the 'u32LayoutIndex' layout of the 'u32SequenceIndex' sequence
   return (theConfig()->getLayout(u32SequenceIndex, u32LayoutIndex));
}


/***************************************************************************/
void SpellerWidget2D::vSetSubSpellerCharRendering(const SpellerBtn* pSpellerBtn, bool bIsRendered) const
{
   //Controll the rendering of the render node required for sub-speller indication
   if ((pSpellerBtn) && (pSpellerBtn->m_pNode) && (pSpellerBtn->m_pNode->GetFirstChild()))
   {
      Candera::RenderNode* rNode = dynamic_cast<Candera::RenderNode*>(pSpellerBtn->m_pNode->GetFirstChild());

      if ((rNode) && (rNode->GetEffect(0)) && (rNode->GetEffect(0)->IsTypeOf(Candera::SolidColorBrushBlend::GetTypeId())))
      {
         pSpellerBtn->m_pNode->GetFirstChild()->SetRenderingEnabled(bIsRendered);
      }
   }
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_005
*               HTC_Speller_REQ_007
* Function    : vToggleButtonRelease
* Called by   : vOnSpellerBtnRelease
* Description : Function for the business logic on Toggle SpellerButton release
* Parameters  : enSpellerBtnType
* Return      : void
* History     : 2013-08-14(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::vToggleButtonRelease(enSplrBtnType enType)
{
   //Get the instance of SpellerData to read the XML generated data
   //get the layout attached to the received Toggle control button
   Candera::UInt32 u32NextLayout = theConfig()->getCtrlCharAttachedLayout(
                                      m_u32ActiveSplrSeqIdx,
                                      m_u32ActiveLayoutIdx,
                                      enGetButtonTypeCtrlChar(enType));
   //get the layout index and update the same into the member variable
   m_u32ActiveLayoutIdx = theConfig()->getLayoutIndex(
                             m_u32ActiveSplrSeqIdx,
                             u32NextLayout);
   //deactivate the previous scene
   vDeActivateView();
   const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

   if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
   {
      (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
   }

   //reset the focus to 0 (a project-specific configuration)
   m_u32CurFocusedIndex = 0;
   //Now traverse and populate the new layout
   vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
   Invalidate();
}


/***************************************************************************/
void SpellerWidget2D::vActivateSubSpeller(Candera::UInt8 u8Index)
{
   //change the sub-speller activation flag status
   m_bIsSubSpellerActive = true;
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

   //TODO("do boundary check for the array")
   if (pSpellerBtn[u8Index].m_pTextWidget)
   {
      _mainSpellerCharReceived = (pSpellerBtn[u8Index].m_pTextWidget->GetText());
      postSubSpellerStatusMsg(_mainSpellerCharReceived, enSubSpllerStatus::Shown);
   }

   //Get the instance of SpellerData to read the XML generated data
   const Candera::TChar* pcSubSplrSeq = theConfig()->getLanguage(GetActiveLanguage());
   m_u32ActiveSubSplrSeqIdx = theConfig()->getSubSplrSeqIndex(pcSubSplrSeq);
   const Candera::TChar* pcCollationChars;
   const Candera::TChar* pcAttachedLayoutType;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(_mainSpellerCharReceived);
   pcCollationChars = theConfig()->getSubSplrCharacters(m_u32ActiveSubSplrSeqIdx, _mainSpellerCharReceived.GetCString());
   pcAttachedLayoutType = theConfig()->getSubSplrSeqLayoutType(m_u32ActiveSubSplrSeqIdx, _mainSpellerCharReceived.GetCString());
   SECURE_FEATSTD_STRING_ACCESS_END();
   //make touchable false for the main speller buttons
   vSetMainSplrTouchableStatus(false);
   //make touchable true of the base scene where Speller widget exist as the the sub-speller is active now
   this->SetTouchable(true);

   if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
   {
      (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
   }

   m_u32LastFocusedIndexBeforeSubSplrAct = m_u32CurFocusedIndex;
   m_u32CurFocusedIndex = 0;
   //Call vTraverseAndPopulateNodes() with the fetched collation characters and populate the new layout
   vTraverseAndPopulateNodes(pcAttachedLayoutType, pcCollationChars);
}


/***************************************************************************/
void SpellerWidget2D::vDeactivateSubSpeller()
{
   struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

   if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
   {
      (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
   }

   //change the sub-speller activation flag status
   m_bIsSubSpellerActive = false;
   //deactivate the sub-speller scene
   vDeActivateView();
   //make touchable false of the base scene where Speller widget exist
   //it should be true only when the sub-speller is active
   this->SetTouchable(false);

   postSubSpellerStatusMsg(_mainSpellerCharReceived, enSubSpllerStatus::Hidden);
   //TODO("There is no need to call vTraverseAndPopulateNodes() again for main speller")
   //as the scene was never deactivated, only touchable was made false. Will be optimized later
   //Now traverse and populate the main speller character content on the layout type as before
   const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
   m_u32CurFocusedIndex = m_u32LastFocusedIndexBeforeSubSplrAct;
   //Start traversing and populating the SceneComposer nodes based on SeqIdx & LayoutIdx
   vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_001
* Function    : cobChkValidCharPropVisibility
* Called by   : HMI Framework when the SpellerMode CDA property is modified
* Description : Interface which determines the Visibility of ValidCharSet
*               property based on Mode property
* Parameters  : None
* Return      : true  - ValidCharSet property is shown
*               false - ValidCharSet property is not shown
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
bool SpellerWidget2D::cobChkValidCharPropVisibility() const
{
   //Check the value of the property SpellerMode. If it is MatchMode,
   //then make the ValidCharacters property visible, else should be invisible
   return (MatchMode == GetMode());
}


/****************************************************************************
* Req ID      : HTC_Speller_REQ_001
* Function    : cobChkCustomSequenceVisibility
* Called by   : HMI Framework when the SpellerMode CDA property is modified
* Description : Interface which determines the visibility of UserDefined[Sequence/Layout]
property based on Type property
* Parameters  : None
* Return      : true  - UserDefined[Sequence/Layout] property is shown
*               false - UserDefined[Sequence/Layout] property is not shown
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
bool SpellerWidget2D::cobChkCustomSequenceVisibility() const
{
   //Check the value of the property SpellerType. If it is of CustomSequence, then
   //make the UserDefined[Sequence/Layout] properties visible, else should be invisible
   return (CustomSequence == GetType());
}


/****************************************************************************
* Req ID      : NA
* Function    : OnFocus
* Called by   : HMI Framework when the Speller widget gains focus
* Description :
* Parameters  : None
* Return      : void
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::OnFocus()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: In OnFocus interface", this));
   m_bSpellerFocussedStatus = true;

#ifdef VARIANT_S_FTR_ENABLE_SPELLER_WIDGET_SET_FOCUS_ON_BUTTONS
   if (SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget)
   {
      SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget->OnFocus();
   }
#endif
}


/****************************************************************************
* Req ID      : NA
* Function    : OnLostFocus
* Called by   : HMI Framework when the Speller widget looses focus
* Description :
* Parameters  : None
* Return      : void
* History     : 2013-07-17(kea2kor) Initial version
*               2013-07-30(kea2kor) Intermediate release
****************************************************************************/
void SpellerWidget2D::OnLostFocus()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D [%p]: In OnLostFocus interface", this));
   m_bSpellerFocussedStatus = false;

#ifdef VARIANT_S_FTR_ENABLE_SPELLER_WIDGET_SET_FOCUS_ON_BUTTONS
   if (SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget)
   {
      SpellerBtnInfo[m_u32CurFocusedIndex].m_pAttachedWidget->OnLostFocus();
   }
#endif
}


void SpellerWidget2D::RegisterToFocusManagerImpl(Focus::FManager& focusManager, Focus::FSession& session, Focus::FWidgetConfig& handle)
{
   Base::RegisterToFocusManagerImpl(focusManager, session, handle);

   FocusGroupBase::registerGroupToFocusManager(focusManager, session, handle);
}


/****************************************************************************
*  Update is invoked during every render cycle.
****************************************************************************/
void SpellerWidget2D::Update()
{
   m_oView.Update();
}


/****************************************************************************
*  OnMessage is called when a message shall be distributed through the
*  view tree and its views and widgets
*  Returns true if the Message is consumed
****************************************************************************/
bool SpellerWidget2D::OnMessage(const Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }
   return m_oCtrl.OnMessage(msg);
}


/****************************************************************************
*  OnTouch, overloaded function to be used only if the widget should perform
*  a custom hit test on a Touch message
*  Parameters  : Camera reference and Touch coordinates
*  Return      : true  - Touch intersects the node
*                false - Touch did not intersect the node
****************************************************************************/
bool SpellerWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   if (_subSpellerConfig.getCloseOnTappingOutsideStatus() && bGetSubSpellerActivationStatus() && (!isTouchIntersectButton(camera2D, point)))
   {
      vDeactivateSubSpeller();
   }
   return GetTouchable() && m_oCtrl.OnTouch(camera2D, point);
}


#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
template<> ::FeatStd::UInt32 StringBufferAppender<Candera::enKeypadFormat>::Append(::FeatStd::StringBuffer& stringBuffer, Candera::enKeypadFormat const& object)
{
   FEATSTD_UNUSED(object);
   ::FeatStd::UInt32 tcharCount = 0;
   tcharCount += stringBuffer.Append("::Candera::enKeypadFormat {");
   tcharCount += stringBuffer.Append("todo");
   tcharCount += stringBuffer.Append(" }");
   return tcharCount;
}


} //namespace FeatStd
#endif

bool SpellerWidget2D::isTouchIntersectButton(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool isIntersected = false;
   for (UInt8 btnIndex = 0; (false == isIntersected && MAX_BUTTONS_PER_LAYOUT > btnIndex && 0 != SpellerBtnInfo[btnIndex].m_pAttachedWidget);
         ++btnIndex)
   {
      isIntersected = SpellerBtnInfo[btnIndex].m_pAttachedWidget->OnTouch(camera2D, point);
   }
   return isIntersected;
}


SpellerWidget2D::SpellerConfiguration::SpellerConfiguration(const SubSpellerConfig& subSpellerConfig)
{
   _subSpellerConfig = subSpellerConfig;
}


bool SpellerWidget2D::postSpellerKeyPressedMsg(const Candera::String buttonPressedText)
{
   bool msgPostStatus = false;

   if (NULL != GetParentView())
   {
      SpellerKeyPressed* msg = NULL;
      Identifier identifier(GetLegacyName());
      ViewId viewId(GetParentView()->GetId().CStr());

      SECURE_FEATSTD_STRING_ACCESS_BEGIN(buttonPressedText);
      msg = COURIER_MESSAGE_NEW(SpellerKeyPressed)(viewId, identifier, buttonPressedText.GetCString());
      msgPostStatus = (NULL != msg) ? (msg->Post()) : (false);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "'SpellerKeyPressed' msg fired for [%s]", buttonPressedText.GetCString()));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "'SpellerKeyPressed' msg post status = [%s]", (msgPostStatus) ? ("true") : ("false")));
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
   return msgPostStatus;
}


bool SpellerWidget2D::postSubSpellerStatusMsg(const String buttonPressedText, const enSubSpllerStatus::Enum subSpellerStatus)
{
   bool msgPostStatus = false;
   if (NULL != GetParentView())
   {
      SubSpellerStatusUpdMsg* msg = NULL;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(buttonPressedText);
      msg = COURIER_MESSAGE_NEW(SubSpellerStatusUpdMsg)(
               ViewId(GetParentView()->GetId().CStr()),
               Identifier(GetLegacyName()),
               GetUserData(),
               buttonPressedText.GetCString(),
               subSpellerStatus);
      msgPostStatus = (NULL != msg) ? (msg->Post()) : (false);
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "'SubSpellerStatusMsg' msg fired for [%s]", buttonPressedText.GetCString()));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "'SubSpellerStatusMsg' msg post status: [%5s]", (msgPostStatus) ? ("true") : ("false")));
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "'SubSpellerStatusMsg' msg fired for sub speller Visibility: [%s] status", (enSubSpllerStatus::Hidden == subSpellerStatus) ? ("Hidden") : ("Shown")));
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
   return msgPostStatus;
}


void SpellerWidget2D::performAutoToggleLayout(const Candera::String strValidCharacters)
{
   //To Handle Auto Toggle of Layout based on the Valid char set in Match Mode
   //Valid Layout is identified based on the sequence of the layouts
   if (GetAutoLayoutToggle() && (0 != m_u32ActiveSplrSeqIdx) && (NULL != theConfig()))
   {
      //Valid Layout is updated to the Active Layout based on valid character set
      Candera::UInt32 m_u32LayoutIdx = theConfig()->getMatchingLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx, strValidCharacters.GetCString());
      if (m_u32LayoutIdx != m_u32ActiveLayoutIdx)
      {
         //To updae the speller Layout
         m_u32ActiveLayoutIdx = m_u32LayoutIdx;
         //deactivate the previous scene
         vDeActivateView();
         const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
         struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

         if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
         {
            (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
         }

         //reset the focus to 0 (a project-specific configuration)
         m_u32CurFocusedIndex = 0;
         //Now traverse and populate the new layout
         vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
      }
   }
}


void SpellerWidget2D::vActivateLayoutIndex(Candera::UInt32 m_u32ActiveLayoutIdx)
{
   //Get the instance of SpellerData to read the XML generated data
   //get the sequence index after forming the sequence based on speller mode, language & country proeprties
   vSetSequenceIdx();
   //Check the if the LayoutIndex given is valid for the particular SequenceIndex
   if (theConfig()->getLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx))
   {
      //set the layout index and update the same into the member variable m_u32ActiveLayoutIdx
      vSetLayoutIdxOfFormat();
      //deactivate the previous scene
      vDeActivateView();
      const TChar* pcLayoutType = theConfig()->getSplrSeqLayoutType(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx);
      struct SpellerBtn* pSpellerBtn = &SpellerBtnInfo[0];

      if ((m_u32CurFocusedIndex < MAX_BUTTONS_PER_LAYOUT) && ((pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget))
      {
         (pSpellerBtn[m_u32CurFocusedIndex]).m_pAttachedWidget->OnLostFocus();
      }

      //reset the focus to 0 (a project-specific configuration)
      m_u32CurFocusedIndex = 0;
      //Now traverse and populate the new layout
      vTraverseAndPopulateNodes(pcLayoutType, cosGetLayout(m_u32ActiveSplrSeqIdx, m_u32ActiveLayoutIdx));
      Invalidate();
   }
   else
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerWidget2D::vActivateLayoutIndex() : No Layout found for LayoutIndex : [%d]\n", m_u32ActiveLayoutIdx));
   }
}
