/* ***************************************************************************************
* FILE:          SpellerConfig.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  TextExtensionWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*************************************************************************************** */
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdint.h>
#include <string.h>
#include <string>         // std::string
#include <cstddef>        // std::size_t
#include <map>
#include "SpellerConfig.h"

// Header file 'FileName' repeatedly included but does not have a standard include guard
//lint -efile(451, Common/SpellerConfig/Layouts.dat)

#ifndef TABSIZE
#define TABSIZE(a) (sizeof(a)/sizeof(a[1]))
#endif

int strCompare(const utf8_t* s1, const utf8_t* s2)
{
   return strcmp(s1, s2);
}


bool isSequenceCharactersAvailable(const utf8_t* pcData, const utf8_t* pcSequence)
{
   if ((NULL != pcData) && (NULL != pcSequence))
   {
      std::string strpcData(pcData);
      std::string strSequence(pcSequence);
      std::size_t found = strpcData.find_first_of(strSequence);
      return (std::string::npos != found);
   }
   return false;
}


class SPELLERWIDGETCONFIG_API SpellerConfig : public ISpellerConfig
{
   private:
      uint32_t getLayoutCount(uint32_t nSequenceIndex) const;
      const utf8_t* getNextLayout(uint32_t nSequenceIndex, uint32_t nLayoutIndex, uint32_t nCtrlChar) const;

   public:
      SpellerConfig();
      virtual ~SpellerConfig();

      uint32_t getSequenceIndex(const utf8_t* pcSequenceName) const;
      uint32_t getCtrlCharAttachedLayout(uint32_t nSequenceIndex, uint32_t nLayoutIndex, uint32_t nCtrlChar) const;
      uint32_t getLayoutIndexOfFormat(uint32_t nSequenceIndex, uint32_t nFormat) const;
      const utf8_t* getLayout(uint32_t nSequenceIndex, uint32_t nLayout) const;
      const utf8_t* getLanguage(uint32_t u32code) const;
      const utf8_t* getDefaultLanguage() const;
      const utf8_t* getCountry(uint32_t u32code) const;
      const utf8_t* getDefaultCountry() const;
      const utf8_t* getSplrSeqLayoutType(uint32_t nSequenceIndex, uint32_t nLayoutIndex) const;
      const utf8_t* getSubSplrSeqLayoutType(uint32_t u32SubSplrSeqIndex, const utf8_t* pcMainChar) const;
      const utf8_t* getSubSplrCharacters(uint32_t u32SubSplrSeqIndex, const utf8_t* pcMainChar) const;
      const utf8_t* getSplrLayoutType(unsigned char nIndex) const;
      uint32_t getSubSplrSeqIndex(const utf8_t* pcSubSplrSeqName) const;
      uint32_t getSeperatorCharacter() const;
      uint32_t getSubSplrCharacter() const;
      uint32_t getNumberOfLayoutTypes() const;
      uint32_t getLayoutIndex(uint32_t nSequenceIndex, uint32_t nLayout) const;
      uint32_t getButtonTypeCtrlChar(enSplrBtnType enType) const;
      enSplrBtnType getButtonBasedOnXmlInfo(uint32_t u32Ucs4Val) const;
      uint32_t getMatchingLayout(uint32_t nSequenceIndex, uint32_t nActiveLayout, const utf8_t* pcValidCharacterSet) const;
};


static std::map<uint32_t, const utf8_t*> m_MapOfLang;     ///< table of all languages
static std::map<uint32_t, const utf8_t*> m_MapOfCountry;  ///< table of all countries

struct stCodeInfo
{
   uint32_t  _code;
   const utf8_t*  _name;
};


/// Contains information about one Sub-speller character in a stSubSplrSequence
struct stSubSplrChar
{
   const utf8_t* _mainChar;
   const utf8_t* _collationChars;
   const utf8_t* _type;
};


/// Contains information about one sub speller sequence.
const int MAX_SUBSPELLER_CHARS_PER_SEQ = 50;
struct stSubSplrSequence
{
   const utf8_t* _name;
   stSubSplrChar _subSplrChar[MAX_SUBSPELLER_CHARS_PER_SEQ + 1];
};


/// Contains information about one speller layout control character.
struct stNextLayouts
{
   uint32_t _ctrlChar;
   uint32_t _nextContent;
};


/// Contains information about one speller layout.
const int MAX_TOGGLE_CHARS_PER_LAYOUT = 4;     ///< maximum number of toggle characters for one layout
struct stLayouts
{
   uint32_t _content;
   uint32_t _format;
   const utf8_t* _type;
   stNextLayouts   _nextlayouts[MAX_TOGGLE_CHARS_PER_LAYOUT + 1];
};


/// Contains information about one speller sequence.
///As earlier layout Max limit was 11 only, but as per new nissan requirment NCG3D-30589 , they need to switch between more than 23 layout
///That's why in order to work the above requiremnt we have extended the limit
const int MAX_LAYOUTS_PER_SEQUENCE = 100;    ///< maximum number of layouts for one layout sequence
struct stSequence
{
   const utf8_t* _name;
   stLayouts _layouts[MAX_LAYOUTS_PER_SEQUENCE + 1];
};


// Layout Enums ...
#define BEGIN_LAYOUTS()                         enum enLayoutConstants { SPELLERLAYOUT_dummy = 0,
#define LAYOUT(name, layout)                       SPELLERLAYOUT_##name,
#define END_LAYOUTS                             SPELLERLAYOUT_COUNT_END};
#include "Common/SpellerConfig/Layouts.dat"
#undef BEGIN_LAYOUTS
#undef LAYOUT
#undef END_LAYOUTS

// Layout data
#define BEGIN_LAYOUTS()                         const char* _apcLayouts[] = { "",
#define LAYOUT(name, layout)                       layout,
#define END_LAYOUTS                             };
#include "Common/SpellerConfig/Layouts.dat"
#undef BEGIN_LAYOUTS
#undef LAYOUT
#undef END_LAYOUTS


#define BEGIN_KEYPADFORMATS()                   enum enKeypadFormatConstants { FORMAT_dummy_0 = 0,
#define KEYPADFORMAT(keypad)                       FORMAT_##keypad,
#define END_KEYPADFORMATS                       FORMAT_COUNT_END};


#define BEGIN_ALL_SEQUENCES()                   const stSequence _aSequences[] =   \
		{ \
			{ 0, \
				{(uint32_t) SPELLERLAYOUT_dummy, (uint32_t)FORMAT_dummy_0, (utf8_t*)"36_CHAR_LAYOUT", \
					{ \
						{(uint32_t) 623, (uint32_t) SPELLERLAYOUT_dummy}, \
					} \
				}, \
			},

#define BEGIN_SEQUENCE(SeqName)                 { SeqName, {
#define BEGIN_ADDLAYOUT()                       {
#define ADDLAYOUT( name, keypad, type )            (uint32_t) SPELLERLAYOUT_##name, (uint32_t)FORMAT_##keypad, (utf8_t*)type, {
#define CONTROLCHAR(charVal, name)                    {(uint32_t) charVal, (uint32_t) SPELLERLAYOUT_##name },
#define END_ADDLAYOUT                           } },
#define END_SEQUENCE                               } },
#define END_ALL_SEQUENCES                       };


#define BEGIN_LANGUAGES()                       const stCodeInfo _languages[] = {
#define LANGUAGE(code, name)                       {code,name},
#define END_LANGUAGES                           {0,""}};


#define BEGIN_COUNTRIES()                       const stCodeInfo _contries[] = {
#define COUNTRY(code, name)                        {code,name},
#define END_COUNTRIES                           {0,""}};


#define BEGIN_LAYOUTTYPES()                     const utf8_t* _apcLayoutTypes[] = { "",
#define LAYOUTTYPE(type)                           type,
#define END_LAYOUTTYPES                         };


#define BEGIN_ALL_SUBSPELLER_SEQUENCES()        const stSubSplrSequence _aSubSplrSequences[] = \
		{ \
			{ 0, \
				{ \
					{(utf8_t*) 0, (utf8_t*) 0, (utf8_t*) 0}, \
				} \
			},
#define BEGIN_SUBSPELLER_SEQUENCE(lang)             { lang, {
#define ADDCHARACTER(character, subseqchars, type)    {(utf8_t*) character, (utf8_t*) subseqchars, (utf8_t*) type},
#define END_SUBSPELLER_SEQUENCE                     } },
#define END_ALL_SUBSPELLER_SEQUENCES            };


#define DEFINE_EMPTY(empty)                     const uint32_t charCode_EmptyCharacter   = (uint32_t) (empty);    ///< character code of empty (unused) character
#define DEFINE_FILL(fill)                       const uint32_t charCode_FillCharacter    = (uint32_t) (fill);     ///< character code of fill character
#define DEFINE_SEPARATOR(sep)                   const uint32_t charCode_SeperatorCharacter= (uint32_t) (sep);     ///< character code of seperator character
#define DEFINE_OK(ok)                           const uint32_t charCode_OkCharacter      = (uint32_t) (ok);       ///< character code of Ok/Enter character
#define DEFINE_INVISIBLE(inv)                   const uint32_t charCode_InvisibleCharacter= (uint32_t) (inv);
#define DEFINE_SYM(sym)                         const uint32_t charCode_SymCharacter     = (uint32_t) (sym);      ///< character code of Symbol character
#define DEFINE_SHIFT(sh)                        const uint32_t charCode_ShCharacter      = (uint32_t) (sh);       ///< character code of Shift character
#define DEFINE_SHIFT_1(sh)                      const uint32_t charCode_Shift1Character  = (uint32_t) (sh);       ///< character code of Shift_1 character
#define DEFINE_SHIFT_2(sh)                      const uint32_t charCode_Shift2Character  = (uint32_t) (sh);       ///< character code of Shift_2 character
#define DEFINE_SHIFT_3(sh)                      const uint32_t charCode_Shift3Character  = (uint32_t) (sh);       ///< character code of Shift_4 character
#define DEFINE_SHIFT_4(sh)                      const uint32_t charCode_Shift4Character  = (uint32_t) (sh);       ///< character code of Shift_4 character
#define DEFINE_SHIFT_5(sh)                      const uint32_t charCode_Shift5Character  = (uint32_t) (sh);       ///< character code of Shift_5 character
#define DEFINE_DELETE(del)                      const uint32_t charCode_DelCharacter     = (uint32_t) (del);      ///< character code of Delete character
#define DEFINE_DELETEALL(delall)                const uint32_t charCode_DelAllCharacter  = (uint32_t) (delall);   ///< character code of Delete all character
#define DEFINE_CUR_LEFTSHIFT(lsh)               const uint32_t charCode_LShCharacter     = (uint32_t) (lsh);      ///< character code of Left cursor shift character
#define DEFINE_CUR_RIGHTSHIFT(rsh)              const uint32_t charCode_RShCharacter     = (uint32_t) (rsh);      ///< character code of Right cursor shift character
#define DEFINE_SUBSPELLER(subsplr)              const uint32_t charCode_SubSplrCharacter = (uint32_t) (subsplr);  ///< character code of Sub-speller character
#define DEFINE_CUSTOMBUTTON_1(btn1)             const uint32_t charCode_CustBtn1Character= (uint32_t) (btn1);     ///< character code of CustomButton_1 character
#define DEFINE_CUSTOMBUTTON_2(btn2)             const uint32_t charCode_CustBtn2Character= (uint32_t) (btn2);     ///< character code of CustomButton_2 character
#define DEFINE_CUSTOMBUTTON_3(btn3)             const uint32_t charCode_CustBtn3Character= (uint32_t) (btn3);     ///< character code of CustomButton_3 character
#define DEFINE_CUSTOMBUTTON_4(btn4)             const uint32_t charCode_CustBtn4Character= (uint32_t) (btn4);     ///< character code of CustomButton_4 character
#define DEFINE_CUSTOMBUTTON_5(btn5)             const uint32_t charCode_CustBtn5Character= (uint32_t) (btn5);     ///< character code of CustomButton_5 character

#define DEFAULT_SEQUENCE(def)                   const utf8_t* defaultSequence = def;                                   ///< name of default sequence
#define SPACE_UCS4VAL                           (32)                                                                          ///< The UCS4 equivalent of space character used by Speller

#include "Common/SpellerConfig/Languages.dat"
#include "Common/SpellerConfig/Sequences.dat"

/**
*  Initialize the class.
*  Determines the default layout sequence, and checks data for validity.
*/
SpellerConfig::SpellerConfig()
{
   m_MapOfLang.clear();
   for (int i = 0; _languages[i]._name[0] != '\0'; i++)
   {
      m_MapOfLang[_languages[i]._code] = _languages[i]._name;
   }
   m_MapOfCountry.clear();
   for (int i = 0; _contries[i]._name[0] != '\0'; i++)
   {
      m_MapOfCountry[_contries[i]._code] = _contries[i]._name;
   }
}


SpellerConfig::~SpellerConfig()
{
}


/**
*  Find a layout index by the sequence name.
*  @param[in]  pcSequenceName  Sequence to look for
*  @return Sequence index, or 0 in case of error.
*/
uint32_t SpellerConfig::getSequenceIndex(const utf8_t* pcSequenceName) const
{
   if (pcSequenceName != NULL)
   {
      for (uint32_t nSequence = 1; nSequence < TABSIZE(_aSequences); nSequence++)
      {
         if (strCompare(_aSequences[nSequence]._name, pcSequenceName) == 0)
         {
            return nSequence;
         }
      }
   }
   return 0;
}


/**
*  Get number of layouts for a particular sequence index.
*  @param[in]  nSequenceIndex  Layout sequence index
*  @return Number of layouts in this sequence, or 0 in case of errors.
*/
uint32_t SpellerConfig::getLayoutCount(uint32_t nSequenceIndex) const
{
   uint32_t nCount = 0;

   if (nSequenceIndex > 0 && nSequenceIndex < TABSIZE(_aSequences))
   {
      while (_aSequences[nSequenceIndex]._layouts[nCount]._content != 0)
      {
         nCount++;
      }
   }

   return nCount;
}


/**
*  Get one layout content.
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nLayoutIndex    Layout index (valid values 0...number of layouts-1)
*  @return Layout string (should consist of 28 UTF-8 encoded characters),
*          or 0 in case of error (invalid parameters).
*/
const utf8_t* SpellerConfig::getLayout(uint32_t nSequenceIndex, uint32_t nLayoutIndex) const
{
   const utf8_t* pcLayout = 0;

   if (nSequenceIndex > 0 && nSequenceIndex < TABSIZE(_aSequences) && nLayoutIndex < MAX_LAYOUTS_PER_SEQUENCE)
   {
      uint32_t nLayout = _aSequences[nSequenceIndex]._layouts[nLayoutIndex]._content;
      if (nLayout != 0)
      {
         pcLayout = _apcLayouts[nLayout];
      }
   }

   return pcLayout;
}


/**
*  Get one layout content.
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nLayoutIndex    Layout index (valid values 0...number of layouts-1)
*  @param[in]  nCtrlChar       Layout toggle char index (valid values 0...number of LayoutToggleChar-1)
*  @return Layout string (should consist of 28 UTF-8 encoded characters),
*          or 0 in case of error (invalid parameters).
*/
const utf8_t* SpellerConfig::getNextLayout(uint32_t nSequenceIndex, uint32_t nLayoutIndex, uint32_t nCtrlChar) const
{
   const utf8_t* pcLayout = 0;

   uint32_t nLayout = getCtrlCharAttachedLayout(nSequenceIndex, nLayoutIndex, nCtrlChar);
   if (nLayout != 0)
   {
      pcLayout = _apcLayouts[nLayout];
   }

   return pcLayout;
}


/**
*  Get one layout.
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nLayoutIndex    Layout index (valid values 0...number of layouts-1)
*  @param[in]  nCtrlChar       Layout toggle char index (valid values 0...number of LayoutToggleChar-1)
*  @return Layout in this sequence, or 0 in case of error (invalid parameters).
*/
uint32_t SpellerConfig::getCtrlCharAttachedLayout(uint32_t nSequenceIndex, uint32_t nLayoutIndex, uint32_t nCtrlChar) const
{
   uint32_t nToggleCharindex = 0;

   if (nSequenceIndex > 0 && nSequenceIndex < TABSIZE(_aSequences) && nLayoutIndex < MAX_LAYOUTS_PER_SEQUENCE)
   {
      while ((MAX_TOGGLE_CHARS_PER_LAYOUT > nToggleCharindex) &&
             (0 != _aSequences[nSequenceIndex]._layouts[nLayoutIndex]._nextlayouts[nToggleCharindex]._ctrlChar) &&
             (nCtrlChar != _aSequences[nSequenceIndex]._layouts[nLayoutIndex]._nextlayouts[nToggleCharindex]._ctrlChar))
      {
         nToggleCharindex++;
      }
   }

// LINT !!! : Warning 661: Possible access of out-of-bounds pointer (7 beyond end of data) by operator '[' [Reference: file
   //   return ((nToggleCharindex < MAX_LAYOUTS_PER_SEQUENCE) ? (_aSequences[nSequenceIndex]._layouts[nLayoutIndex]._nextlayouts[nToggleCharindex]._nextContent) : 0);

   if (nToggleCharindex < MAX_TOGGLE_CHARS_PER_LAYOUT && nLayoutIndex < MAX_LAYOUTS_PER_SEQUENCE && nSequenceIndex < TABSIZE(_aSequences))
   {
      return _aSequences[nSequenceIndex]._layouts[nLayoutIndex]._nextlayouts[nToggleCharindex]._nextContent;
   }
   return 0;
}


/**
*  Get the layout index for given layout
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nLayout         Layout
*  @return Layout index in this sequence, or 0 in case of error (invalid parameters).
*/
uint32_t SpellerConfig::getLayoutIndex(uint32_t nSequenceIndex, uint32_t nLayout) const
{
   if (nSequenceIndex > 0 && nSequenceIndex < TABSIZE(_aSequences))
   {
      uint32_t index = 0;
      while ((index < MAX_LAYOUTS_PER_SEQUENCE) && (0 != _aSequences[nSequenceIndex]._layouts[index]._content))
      {
         if (nLayout == _aSequences[nSequenceIndex]._layouts[index]._content)
         {
            return index;
         }
         index++;
      }
   }
   return 0;
}


/**
*  Get the layout index for given Keypad Format
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nFormat         KeypadFormat
*  @return Layout index in this sequence, or 0 in case of error (invalid parameters).
*/
uint32_t SpellerConfig::getLayoutIndexOfFormat(uint32_t nSequenceIndex, uint32_t nFormat) const
{
   if (nSequenceIndex > 0 && nSequenceIndex < TABSIZE(_aSequences))
   {
      uint32_t index = 0;
      while ((index < MAX_LAYOUTS_PER_SEQUENCE) && (0 != _aSequences[nSequenceIndex]._layouts[index]._format))
      {
         if (nFormat == _aSequences[nSequenceIndex]._layouts[index]._format)
         {
            return index;
         }
         index++;
      }
   }
   return 0;
}


/**
*  Get the language name for the received language code
*  @param[in]  u32code         Language code
*  @return  Language name with this code, or "NA" in case of error (invalid parameters).
*/
const utf8_t* SpellerConfig::getLanguage(uint32_t u32code) const
{
   std::map<uint32_t, const utf8_t*>::const_iterator pos = m_MapOfLang.find(u32code);

   if (pos == m_MapOfLang.end())
   {
      //handle the error of an invalid code being passed
      pos = m_MapOfLang.begin();
      return (pos->second);
   }
   else
   {
      // the releavnt langauage name of the passed code
      return (pos->second);
   }
}


/**
*  Get the default language name. This should be the first language in the language list
*  @param[in]  none
*  @return  First Language name in the language list
*/
const utf8_t* SpellerConfig::getDefaultLanguage() const
{
   //TODO("A temproray solution since the C++ map by default stores the key in sorted order");
   //The first language in the language list should be returned as
   //this is supposed to be the default language
   std::map<uint32_t, const utf8_t*>::const_iterator pos = m_MapOfLang.begin();
   return (pos->second);
}


/**
*  Get the country name for the received country code
*  @param[in]  u32code         Country code
*  @return  Country name with this code, or "NA" in case of error (invalid parameters).
*/
const utf8_t* SpellerConfig::getCountry(uint32_t u32code) const
{
   std::map<uint32_t, const utf8_t*>::const_iterator pos = m_MapOfCountry.find(u32code);

   if (pos == m_MapOfCountry.end())
   {
      //handle the error of an invalid code being passed
      pos = m_MapOfCountry.begin();
      return (pos->second);
   }
   else
   {
      //return the releavnt langauage name of the passed code
      return (pos->second);
   }
}


/**
*  Get the default country name. This should be the first country in the country list
*  @return  First country name in the country list
*/
const utf8_t* SpellerConfig::getDefaultCountry() const
{
   //TODO("A temproray solution since the C++ map by default stores the key in sorted order");
   //The first country in the country list should be returned as
   //this is supposed to be the default country
   std::map<uint32_t, const utf8_t*>::const_iterator pos = m_MapOfCountry.begin();
   return (pos->second);
}


/**
*  Find a layout index by the sequence name.
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  nLayoutIndex    Layout index (valid values 0...number of layouts-1)
*  @return Layout type, or 0 in case of error or invalid parameter.
*/
const utf8_t* SpellerConfig::getSplrSeqLayoutType(uint32_t nSequenceIndex, uint32_t nLayoutIndex) const
{
   return _aSequences[nSequenceIndex]._layouts[nLayoutIndex]._type;
}


/**
*  Find a layout index by the sequence name.
*  @param[in]  u32SubSplrSeqIndex  Sub-speller sequence index
*  @param[in]  pcReceivedMainChar  Main character in the speller
*  @return Layout type, or 0 in case of error or invalid parameter.
*/
const utf8_t* SpellerConfig::getSubSplrSeqLayoutType(uint32_t u32SubSplrSeqIndex, const utf8_t* pcReceivedMainChar) const
{
   uint32_t nIndex = 0;

   while ((nIndex < MAX_SUBSPELLER_CHARS_PER_SEQ) && (0 != _aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[nIndex]._mainChar))
   {
      if (0 == strCompare(_aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[nIndex]._mainChar, pcReceivedMainChar))
      {
         return _aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[nIndex]._type;
      }
      nIndex++;
   }
   return NULL;
}


/**
*  Find a layout index by the sequence name.
*  @param[in]  pcSequenceName  Sequence to be looked for.
*  @return Sequence index, or 0 in case of error or invalid parameter.
*/
uint32_t SpellerConfig::getSubSplrSeqIndex(const utf8_t* pcSubSplrSeqName) const
{
   if (NULL != pcSubSplrSeqName)
   {
      uint32_t index = 1;

      while ((index < TABSIZE(_aSubSplrSequences)) && (0 != _aSubSplrSequences[index]._name))
      {
         // check for existence of received sequence
         if (0 == strCompare(_aSubSplrSequences[index]._name, pcSubSplrSeqName))
         {
            return index;
         }
         index++;
      }
   }

   return 0;
}


/**
*  Find a layout index by the sequence name.
*  @param[in]  u32SubSplrSeqIndex   The sub-speller seq index where the search is to be done
*  @param[in]  _mainChar           The main character for which the sub-speller characters are to be fetched
*  @return Sub-speller characters if main character is found, else NULL
*/
const utf8_t* SpellerConfig::getSubSplrCharacters(uint32_t u32SubSplrSeqIndex, const utf8_t* pcReceivedMainChar) const
{
   if ((0 >= u32SubSplrSeqIndex) || (NULL == pcReceivedMainChar) || (TABSIZE(_aSubSplrSequences) <= u32SubSplrSeqIndex) || (0 == _aSubSplrSequences[u32SubSplrSeqIndex]._name))
   {
      return NULL;
   }

   const utf8_t* pcSubSpellerCharacters = NULL;
   uint32_t index = 0;

   while ((index < MAX_SUBSPELLER_CHARS_PER_SEQ) &&
          (0 != _aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[index]._mainChar))
   {
      //check for existence of received sequence
      if (0 == strCompare(_aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[index]._mainChar, pcReceivedMainChar))
      {
         //pick the related collation characters of this main character
         pcSubSpellerCharacters = _aSubSplrSequences[u32SubSplrSeqIndex]._subSplrChar[index]._collationChars;
         break;
      }
      else
      {
         index++;
      }
   }

   return pcSubSpellerCharacters;
}


const utf8_t* SpellerConfig::getSplrLayoutType(unsigned char nIndex) const
{
   if (TABSIZE(_apcLayoutTypes) > nIndex)
   {
      return _apcLayoutTypes[nIndex];
   }
   return NULL;
}


uint32_t SpellerConfig::getNumberOfLayoutTypes() const
{
   //find out the number of LayoutTypes based on the total size of all types & size of one type
   return (TABSIZE(_apcLayoutTypes));
}


enSplrBtnType SpellerConfig::getButtonBasedOnXmlInfo(uint32_t u32Ucs4Val) const
{
   enSplrBtnType type = SPLRBTNTYPE_Character_Empty;

   //Set the button type based on the received character
   switch (u32Ucs4Val)
   {
      case (charCode_EmptyCharacter) :
         type = SPLRBTNTYPE_Character_Empty;
         break;

      case (charCode_FillCharacter) :
         type = SPLRBTNTYPE_Character_Fill;
         break;

      //Seperator character
      case (charCode_SeperatorCharacter) :
         type = SPLRBTNTYPE_Character_Separator;
         break;

      //Space character
      case (SPACE_UCS4VAL) :
         type = SPLRBTNTYPE_Control_Space;
         break;

      //More button
      case (charCode_SymCharacter) :
         type = SPLRBTNTYPE_Control_ToggleLayoutType;
         break;

      //Button to toggle CharCase
      case (charCode_ShCharacter) :
         type = SPLRBTNTYPE_Control_ToggleCharCase;
         break;

      case (charCode_Shift1Character) :
         type = SPLRBTNTYPE_Control_Shift_1;
         break;
      case (charCode_Shift2Character) :
         type = SPLRBTNTYPE_Control_Shift_2;
         break;
      case (charCode_Shift3Character) :
         type = SPLRBTNTYPE_Control_Shift_3;
         break;
      case (charCode_Shift4Character) :
         type = SPLRBTNTYPE_Control_Shift_4;
         break;
      case (charCode_Shift5Character) :
         type = SPLRBTNTYPE_Control_Shift_5;
         break;

      //Button to delete all characters
      case (charCode_DelAllCharacter) :
         type = SPLRBTNTYPE_Control_DeleteAll;
         break;

      //Button to delete the previous character
      case (charCode_DelCharacter) :
         type = SPLRBTNTYPE_Control_Delete;
         break;

      //Button to shift the EditField cursor to right, if possible
      case (charCode_RShCharacter) :
         type = SPLRBTNTYPE_Control_CursorRight_Shift;
         break;

      //Button to shift the EditField cursor to left, if possible
      case (charCode_LShCharacter) :
         type = SPLRBTNTYPE_Control_CursorLeft_Shift;
         break;

      case (charCode_InvisibleCharacter) :
         type = SPLRBTNTYPE_Control_Invisible;
         break;
      //Button to support firing of custom messages for project specfic needs
      case (charCode_CustBtn1Character) :
         type = SPLRBTNTYPE_Control_CustomButton_1;
         break;

      //Button to support firing of custom messages for project specfic needs
      case (charCode_CustBtn2Character) :
         type = SPLRBTNTYPE_Control_CustomButton_2;
         break;

      //Button to support firing of custom messages for project specfic needs
      case (charCode_CustBtn3Character) :
         type = SPLRBTNTYPE_Control_CustomButton_3;
         break;

      //Button to support firing of custom messages for project specfic needs
      case (charCode_CustBtn4Character) :
         type = SPLRBTNTYPE_Control_CustomButton_4;
         break;

      //Button to support firing of custom messages for project specfic needs
      case (charCode_CustBtn5Character) :
         type = SPLRBTNTYPE_Control_CustomButton_5;
         break;
      //Ok Button
      case (charCode_OkCharacter) :
         type = SPLRBTNTYPE_Control_Ok;
         break;

      default:
         type = SPLRBTNTYPE_Character;
         break;
   }
   return type;
}


uint32_t SpellerConfig::getButtonTypeCtrlChar(enSplrBtnType enType) const
{
   //Set the control character based on the received button type
   switch (enType)
   {
      case (SPLRBTNTYPE_Character_Empty) :
         return charCode_EmptyCharacter;

      case (SPLRBTNTYPE_Character_Fill) :
         return charCode_FillCharacter;

      //Seperator button
      case (SPLRBTNTYPE_Character_Separator) :
         return charCode_SeperatorCharacter;

      //OK/Enter button
      case (SPLRBTNTYPE_Control_Ok) :
         return charCode_OkCharacter;
      case (SPLRBTNTYPE_Control_Invisible) :
         return charCode_InvisibleCharacter;

      //Space character
      case (SPLRBTNTYPE_Control_Space) :
         return SPACE_UCS4VAL;

      //More button
      case (SPLRBTNTYPE_Control_ToggleLayoutType) :
         return charCode_SymCharacter;

      //Button to toggle CharCase
      case (SPLRBTNTYPE_Control_ToggleCharCase) :
         return charCode_ShCharacter;

      case (SPLRBTNTYPE_Control_Shift_1) :
         return charCode_Shift1Character;
      case (SPLRBTNTYPE_Control_Shift_2) :
         return charCode_Shift2Character;
      case (SPLRBTNTYPE_Control_Shift_3) :
         return charCode_Shift3Character;
      case (SPLRBTNTYPE_Control_Shift_4) :
         return charCode_Shift4Character;
      case (SPLRBTNTYPE_Control_Shift_5) :
         return charCode_Shift5Character;

      //Button to delete all characters
      case (SPLRBTNTYPE_Control_DeleteAll) :
         return charCode_DelAllCharacter;

      //Button to delete the previous character
      case (SPLRBTNTYPE_Control_Delete) :
         return charCode_DelCharacter;

      //Button to shift the EditField cursor to right, if possible
      case (SPLRBTNTYPE_Control_CursorRight_Shift) :
         return charCode_RShCharacter;

      //Button to shift the EditField cursor to left, if possible
      case (SPLRBTNTYPE_Control_CursorLeft_Shift) :
         return charCode_LShCharacter;

      //Button on which Speller will fire a custom msg
      case (SPLRBTNTYPE_Control_CustomButton_1) :
         return charCode_CustBtn1Character;

      //Button on which Speller will fire a custom msg
      case (SPLRBTNTYPE_Control_CustomButton_2) :
         return charCode_CustBtn2Character;

      //Button on which Speller will fire a custom msg
      case (SPLRBTNTYPE_Control_CustomButton_3) :
         return charCode_CustBtn3Character;

      //Button on which Speller will fire a custom msg
      case (SPLRBTNTYPE_Control_CustomButton_4) :
         return charCode_CustBtn4Character;

      //Button on which Speller will fire a custom msg
      case (SPLRBTNTYPE_Control_CustomButton_5) :
         return charCode_CustBtn5Character;

      default:
         break;
   }
   return 0;
}


uint32_t SpellerConfig::getSeperatorCharacter() const
{
   return charCode_SeperatorCharacter;
}


uint32_t SpellerConfig::getSubSplrCharacter() const
{
   return charCode_SubSplrCharacter;
}


/**
*  Get layout Index for a particular sequence index which matches the requested validcharacter set
*  @param[in]  nSequenceIndex  Layout sequence index
*  @param[in]  pcValidCharacterSet  Valid Charaters set
*  @return Layout Index,
*          or nActiveLayout in case of error or default data(invalid parameters).
*/
uint32_t SpellerConfig::getMatchingLayout(uint32_t nSequenceIndex, uint32_t nActiveLayout, const utf8_t* pcValidCharacterSet) const
{
   if ((pcValidCharacterSet != NULL) && (nSequenceIndex > 0) && (nSequenceIndex < TABSIZE(_aSequences)))
   {
      uint32_t layoutindex = 0;
      uint32_t layotutCount = getLayoutCount(nSequenceIndex);
      if (0 != layotutCount)
      {
         //If Current Layout has any valid characters
         if ((0 != getLayout(nSequenceIndex, nActiveLayout)) && isSequenceCharactersAvailable(pcValidCharacterSet, getLayout(nSequenceIndex, nActiveLayout)))
         {
            return nActiveLayout;
         }
         //Will be processed only when there is no active characters in current layout
         while ((layoutindex < layotutCount) && (0 != getLayout(nSequenceIndex, layoutindex)))
         {
            if (isSequenceCharactersAvailable(pcValidCharacterSet, getLayout(nSequenceIndex, layoutindex)))
            {
               return layoutindex;
            }
            layoutindex++;
         }
      }
   }
   return nActiveLayout; // If no active sequence or data by default active layout will be returned
}


// Create Object
extern "C" SPELLERWIDGETCONFIG_API ISpellerConfig* CreateSpellerConfigInstance(void)
{
   return new SpellerConfig();
}


/////// DLL interface ////////////////////////////////////////////////////////////////////////////////////

#ifdef SPELLERWIDGETCONFIG_EXPORTS
#include <windows.h>

BOOL APIENTRY DllMain(HMODULE, DWORD, LPVOID)
{
   return TRUE;
}


#endif
