/* ***************************************************************************************
* FILE:          SpellerButtonController2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SpellerButtonController2D.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "SpellerButtonController2D.h"

#include <View/CGI/CgiExtensions/AppViewHandler.h>
#include <Widgets/2D/Speller/SpellerWidget2D.h>


#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SPELLER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SpellerButtonController2D.cpp.trc.h"
#endif


using ::Candera::Char;//required for CANDERA_RTTI_DEFINITION
CANDERA_RTTI_DEFINITION(SpellerButtonController2D)


/*****************************************************************************/
SpellerButtonController2D::SpellerButtonController2D(hmibase::widget::adorner::AdornerManager& bubbleAdornerManager, const hmibase::widget::adorner::AdornerMarkerFilter& bubbleAdornerMarkerFilter) : Base(bubbleAdornerManager, bubbleAdornerMarkerFilter)
{
   SubSpellerStatusUpdMsg::Subscribe(Courier::ComponentType::View, Courier::ComponentType::Controller);
}


/*****************************************************************************/
SpellerButtonController2D::~SpellerButtonController2D()
{
}


/*****************************************************************************/
bool SpellerButtonController2D::OnMessage(DelegateWidget& widget, const Courier::Message& msg)
{
   bool consumed = Base::OnMessage(widget, msg);

   switch (msg.GetId())
   {
      case SubSpellerStatusUpdMsg::ID:
      {
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(&widget);
         const SubSpellerStatusUpdMsg* oMsg = Courier::message_cast<const SubSpellerStatusUpdMsg*>(&msg);
         if ((oMsg != NULL) && (button != NULL) && button->IsPressed())
         {
            consumed = OnSubSpellerStatusUpdMessage(widget, *oMsg) || consumed;
         }
         break;
      }

      case Courier::ActivationResMsg::ID:
      {
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(&widget);
         const Courier::ActivationResMsg* oMsg = Courier::message_cast<const Courier::ActivationResMsg*>(&msg);
         if ((oMsg != NULL) && (button != NULL) && button->IsPressed())
         {
            consumed = OnActivationResMessage(widget, *oMsg) || consumed;
         }
         break;
      }

      default:
         break;
   }

   return consumed;
}


/*****************************************************************************/
bool SpellerButtonController2D::OnSubSpellerStatusUpdMessage(DelegateWidget& widget, const SubSpellerStatusUpdMsg& msg)
{
   SpellerWidget2D* speller = Candera::Dynamic_Cast<SpellerWidget2D*>(AppViewHandler::getInstance().FindWidget(msg.GetView(), msg.GetSender()));

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::OnSubSpellerStatusUpdMessage status=%d, key=%5s, speller=%50s.%s",
                       ETG_CENUM(hmibase::widget::speller::enSubSpllerStatus::Enum, msg.GetStatus()), msg.GetKeyChar().GetCString(),
                       msg.GetView().CStr(), (speller != NULL) ? speller->GetLegacyName() : "null"));

   //when sub speller view is opened remember its name
   if ((msg.GetStatus() == hmibase::widget::speller::enSubSpllerStatus::Shown) && (speller != NULL) && speller->bGetSubSpellerActivationStatus())
   {
      _subSpellerViewName = speller->pcGetActivatedSceneName();

      UpdateSubSpellerButtons(widget, _touchCoordinates);
   }
   else
   {
      _subSpellerViewName.clear();
      _selectedSubSpellerButton.release();
   }

   return false;
}


/*****************************************************************************/
bool SpellerButtonController2D::OnActivationResMessage(DelegateWidget& widget, const Courier::ActivationResMsg& msg)
{
   if (msg.GetSuccess() && (_subSpellerViewName == msg.GetViewId().CStr()))
   {
      UpdateSubSpellerButtons(widget, _touchCoordinates);
   }

   return false;
}


/*****************************************************************************/
bool SpellerButtonController2D::onReaction(ButtonWidget2D& button, enReaction reaction)
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::onReaction reaction=%d, subspellerButton=%50s, button=%50s.%s",
                       ETG_CENUM(enReaction, reaction), _selectedSubSpellerButton.isValid() ? _selectedSubSpellerButton->GetLegacyName() : "null",
                       (button.GetParentView() != NULL) ? button.GetParentView()->GetId().CStr() : "null", button.GetLegacyName()));

   bool consumed = Base::onReaction(button, reaction);

   switch (reaction)
   {
      //on short/long press release post the reaction message for the selected sub speller button
      case enShortPressRelease:
      case enLongPressRelease:
      {
         //consume the message to prevent the reaction messages to be posted
         consumed = true;

         if (_selectedSubSpellerButton.isValid())
         {
            if ((_selectedSubSpellerButton->GetNode() != NULL) && _selectedSubSpellerButton->GetNode()->IsEffectiveRenderingEnabled())
            {
               _selectedSubSpellerButton->postReactionMessage(enRelease);
            }
            _selectedSubSpellerButton.release();
         }
         break;
      }

      default:
         break;
   }
   return consumed;
}


/*****************************************************************************/
bool SpellerButtonController2D::OnTapGesture(DelegateWidget& widget, const hmibase::input::gesture::GestureEvent& gestureData)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::OnTapGesture state=%d, touchCoordinate=[%d,%d], button=%50s.%s",
                       ETG_CENUM(hmibase::input::gesture::GestureEvent::Enum, gestureData._gestureState), gestureData._pt1.x, gestureData._pt1.y,
                       (widget.GetParentView() != NULL) ? widget.GetParentView()->GetId().CStr() : "null", widget.GetLegacyName()));

   bool consumed = Base::OnTapGesture(widget, gestureData);

   _touchCoordinates = Candera::Vector2(static_cast<FeatStd::Float>(gestureData._pt1.x), static_cast<FeatStd::Float>(gestureData._pt1.y));

   //on touch move update sub speller buttons
   if (gestureData._gestureState == hmibase::input::gesture::GestureEvent::ET_MOVE)
   {
      UpdateSubSpellerButtons(widget, _touchCoordinates);

      //consume the tap move event to prevent the button from becoming unpressed if the touch coordinate goes outside it
      consumed = true;
   }
   return consumed;
}


/*****************************************************************************/
void SpellerButtonController2D::UpdateSubSpellerButtons(DelegateWidget& widget, const Candera::Vector2& touchCoordinates)
{
   std::vector<ButtonWidget2D*> buttons;
   FindSubSpellerButtons(widget, buttons);

   ButtonWidget2D* closestButton = FindClosestSubSpellerButton(widget, buttons, touchCoordinates);

   if (closestButton != _selectedSubSpellerButton.getObjectSafely())
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::UpdateSubSpellerButtons coordinates=[%d, %d], closestButton=%s",
                          static_cast<int>(touchCoordinates.GetX()), static_cast<int>(touchCoordinates.GetY()), closestButton->GetLegacyName()));
   }

   //iterate sub speller buttons, select the closest one and unselect the others
   for (std::vector<ButtonWidget2D*>::iterator it = buttons.begin(); it != buttons.end(); ++it)
   {
      ButtonWidget2D* button = *it;
      if ((button != NULL) && (button != closestButton))
      {
         SetSubSpellerButtonSelected(*button, false);
      }
   }
   if (closestButton != NULL)
   {
      SetSubSpellerButtonSelected(*closestButton, true);
   }

   //remember the closest button in order to click it on long press release
   _selectedSubSpellerButton = ButtonWidgetAccessor(closestButton);
}


/*****************************************************************************/
ButtonWidget2D* SpellerButtonController2D::FindClosestSubSpellerButton(DelegateWidget& /*widget*/, std::vector<ButtonWidget2D*>& buttons, const Candera::Vector2& touchCoordinates)
{
   ButtonWidget2D* closestButton = NULL;
   FeatStd::Float closestButtonDistance = Candera::Math::MaxFloat();

   //iterate subspeller buttons to find the closest one
   for (std::vector<ButtonWidget2D*>::iterator it = buttons.begin(); it != buttons.end(); ++it)
   {
      ButtonWidget2D* button = *it;
      if ((button != NULL) && (button->GetNode2D() != NULL) && IsSubSpellerButtonValid(*button))
      {
         Candera::Rectangle buttonBounds;
         button->GetNode2D()->GetWorldAxisAlignedBoundingRectangle(buttonBounds);

         Candera::Vector2 buttonCenter(buttonBounds.GetLeft() + buttonBounds.GetWidth() / 2.0f, buttonBounds.GetTop() + buttonBounds.GetHeight() / 2.0f);
         FeatStd::Float buttonDistance = touchCoordinates.GetDistanceTo(buttonCenter);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::FindClosestSubSpellerButton buttonCenter=[%d,%d], distance=%d, button=%s",
                             static_cast<int>(buttonCenter.GetX()), static_cast<int>(buttonCenter.GetY()), static_cast<int>(buttonDistance), button->GetLegacyName()));

         if (buttonDistance < closestButtonDistance)
         {
            closestButtonDistance = buttonDistance;
            closestButton = button;
         }
      }
   }

   return closestButton;
}


/*****************************************************************************/
class SubSpellerButtonFinderCallback : public WidgetCheckCallback
{
   public:
      SubSpellerButtonFinderCallback(std::vector<ButtonWidget2D*>& buttons) : _buttons(buttons)
      {
      }

      virtual bool CheckWidget(Candera::Widget2D* widget)
      {
         ButtonWidget2D* button = Candera::Dynamic_Cast<ButtonWidget2D*>(widget);
         if ((button != NULL) && (button->GetNode() != NULL))
         {
            _buttons.push_back(button);
         }

         //continue searching through all the widgets to find the entire content
         return false;
      }

   private:
      FEATSTD_MAKE_CLASS_UNCOPYABLE(SubSpellerButtonFinderCallback);

      std::vector<ButtonWidget2D*>& _buttons;
};


/*****************************************************************************/
bool SpellerButtonController2D::FindSubSpellerButtons(DelegateWidget& /*widget*/, std::vector<ButtonWidget2D*>& buttons)
{
   Courier::View* subspellerView = AppViewHandler::getInstance().FindView(Courier::ViewId(_subSpellerViewName.c_str()));
   if (subspellerView != NULL)
   {
      SubSpellerButtonFinderCallback callback(buttons);
      subspellerView->DistributeMessage(WidgetCheckReqMsg(&callback));
      return true;
   }

   return false;
}


/*****************************************************************************/
bool SpellerButtonController2D::IsSubSpellerButtonValid(ButtonWidget2D& button)
{
   return button.IsEnabled() && (button.GetNode() != NULL);// && button.GetNode()->IsEffectiveRenderingEnabled();
}


/*****************************************************************************/
void SpellerButtonController2D::SetSubSpellerButtonSelected(ButtonWidget2D& button, bool selected)
{
   if (button.IsActive() != selected)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SpellerButtonController2D::SetSubSpellerButtonSelected selected=%u, button=%50s.%s",
                          selected, (button.GetParentView() != NULL) ? button.GetParentView()->GetId().CStr() : "null", button.GetLegacyName()));
   }

   button.SetActive(selected);
}
