/* ***************************************************************************************
* FILE:          SliderWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SliderWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "SliderWidget2D.h"
#include "Candera/System/GlobalizationBase/CultureManager.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "Widgets/2D/ControlTemplate/ControlTemplateMap.h"
#include "Widgets/2D/WidgetGestureConfig.h"

using namespace Courier;
using namespace Candera;
using namespace hmibase::input::gesture;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SliderWidget2D.cpp.trc.h"
#endif

#define VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT


CGI_WIDGET_RTTI_DEFINITION(SliderWidget2D);
CANDERA_RTTI_DEFINITION(SliderController2DData)
CANDERA_RTTI_DEFINITION(SliderController2D)
CANDERA_RTTI_DEFINITION(DefaultSliderController2DData)
CANDERA_RTTI_DEFINITION(DefaultSliderController2D)


/****************************************************************************
*     Function    : SliderWidget2D
*     Description : constructor
****************************************************************************/
SliderWidget2D::SliderWidget2D():
   _scaleStart(0.0f),
   _scaleEnd(1.0f),
   _markerHalfSize(1.0f),
   _updatePositionOnNextFrame(false),
   _valueInteral(0.0f),
   _bIsInTouchSession(false),
   _bMarkerBitmapChanged(false),
   _bIsSliderSelected(false),
   _stepSizeInteral(0),
   _markerMode(MarkerNormal),
   _bitmapMarker(),
   _rightToLeft(false),
   _isPartofList(false),
   _sliderTouchState(SliderNotTouched)
{
   SetTouchable(true);

#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT
   // enable gestures
   SetTap(true);
   SetPressRepeat(true);
   SetDrag(true);
#endif

   //slider should me made focusable only when it will be able to process focus input messages
   //vSetFocusableStatus(true);
}


/****************************************************************************
*     Function    : ~SliderWidget2D()
*     Description : Destructor
****************************************************************************/
SliderWidget2D::~SliderWidget2D()
{
}


/****************************************************************************
*     Function    : Init
*     Description : Initializes the widget so that all referred resource can be
*                   resolved
*     Parameters  : assetFactory Asset provider to resolve all referred resources
*     Return      : void
****************************************************************************/
void SliderWidget2D::InitWidget()
{
   Base::InitWidget();
   // get Culture direction
   Candera::Globalization::Culture::SharedPointer culture = Globalization::CultureManager::GetInstance().GetCurrentCulture();
   _rightToLeft = ((culture != 0) && culture->GetTextDirection() == ::Candera::Globalization::RightToLeft);
   dirty();
}


/****************************************************************************
*     Function    : GetGestureList
****************************************************************************/
WidgetGestureConfig SliderWidget2D::getGestureConfig() const
{
   WidgetGestureConfig gestureConfig(Base::getGestureConfig());

   //timed step movements requires drag to be disabled, tap enabled and press repeat set
   if (GetMarkerMovement() == ::Candera::TimedStepMovement)
   {
      if (!GetTap() || !gestureConfig.getTap().Enabled)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "getGestureConfig() Press(Tap) is not enabled as required by MovementType=TimedStepMovement for instance=%p name=%s", this, GetName()));
      }
      else
      {
         if (GetDrag() && gestureConfig.getDrag().Enabled)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "getGestureConfig() Drag is not disabled as required by MovementType=TimedStepMovement for instance=%p name=%s", this, GetName()));
         }

         if (!GetPressRepeat() || (gestureConfig.getTap().RepeatTime == 0))
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "getGestureConfig() PressRepeat is not enabled or RepeatTime is not set as required by MovementType=TimedStepMovement for instance=%p name=%s", this, GetName()));
         }
      }
   }

   return gestureConfig;
}


/****************************************************************************
*     Function    : CloneFrom
*     Description : required for FlexList cloning
****************************************************************************/
bool SliderWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const SliderWidget2D* original = CLONEABLE_WIDGET_CAST<const SliderWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetSliderBackGround(controlTemplateMap.ResolveNodeClone(original->GetSliderBackGround()));
      SetMarkerNode(controlTemplateMap.ResolveNodeClone(original->GetMarkerNode()));
      SetBitmapFillNode(controlTemplateMap.ResolveNodeClone(original->GetBitmapFillNode()));

      SetMinVal(original->GetMinVal());
      SetMaxVal(original->GetMaxVal());
      SetMarkerMovement(original->GetMarkerMovement());
      SetSliderOrientation(original->GetSliderOrientation());
      SetMarkerNormalBitmap(original->GetMarkerNormalBitmap());
      SetMarkerSelectedBitmap(original->GetMarkerSelectedBitmap());
      SetContinuousUpdateOnDrag(original->GetContinuousUpdateOnDrag());
      SetPaddingMinVal(original->GetPaddingMinVal());
      SetPaddingMaxVal(original->GetPaddingMaxVal());
      SetUseNumofSteps(original->GetUseNumofSteps());
      SetNumOfSteps(original->GetNumOfSteps());
      SetStepSize(original->GetStepSize());
      SetFillerPositionIsKnobCenter(original->GetFillerPositionIsKnobCenter());
      SliderWidget2DBase::SetCurrentValue(original->_valueInteral);
      // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::CloneFrome valorignal=%f ==> value=%5.1f name=%s", original->_valueInteral, _valueInteral, original->GetName()));

      cloned = true;
      _isPartofList = true;
   }
   return cloned;
}


/****************************************************************************
*     Function    : Update()
*     Description : Overridden framework method.
*                  All the widget operations like moving the slider marker,
*                  scaling the slider fill node, changing between the slider
*                  marker bitmap is done in this method.
*     Parameters  : None
*     Return      : void
****************************************************************************/
void SliderWidget2D::Update()
{
   Base::Update();

   if (initializeEffectiveMovableArea() == true)
   {
      if (_bMarkerBitmapChanged)
      {
         updateMarkerBitmap();
         _bMarkerBitmapChanged = false;
      }
      if (!_bIsInTouchSession && !GetContinuousUpdateOnDrag())
      {
         _valueInteral = nearestStepSizeValue(GetCurrentValue());
      }

      if (_updatePositionOnNextFrame)
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::UpdateBBB instance=%p val=%f name=%s", this, _valueInteral, GetName()));
         updatePosition();
         Invalidate();
      }
   }
}


/****************************************************************************
*       Function      : OnLostFocus
*       Description   : Once widget lose focus, it will be informed it via
*                            this method.
*       Parameters    : None
*       Return        : void
****************************************************************************/
void SliderWidget2D::OnLostFocus()
{
   if (GetFocusableStatus())
   {
      setMarkerMode(MarkerNormal);
   }
}


/****************************************************************************
*       Function      : OnFocus
*       Description   : Set focus of widget widget, on againing the focus.
*       Parameters    : None
*       Return        : void
****************************************************************************/
void SliderWidget2D::OnFocus()
{
   if (GetFocusableStatus())
   {
      dirty();
   }
}


/****************************************************************************
*     Function    : OnChanged
*     Description : Overridden framework method to handle updates in the widget
*                   properties.
*     Parameters  : propertyId - The ID of the property that has been updated
*     Return      : void
****************************************************************************/
void SliderWidget2D::OnChanged(::Candera::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   if (CurrentValuePropertyId == propertyId)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SliderWidget2D::OnChanged: %5.2f", GetCurrentValue()));
      _valueInteral = GetCurrentValue();
   }

   dirty();
}


/****************************************************************************
*  Function: SetCurrentValue
****************************************************************************/
void SliderWidget2D::setCurrentValue(bool writeToDataModel)
{
   if (writeToDataModel)
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::setCurrentValue: val=%f name=%s", _valueInteral, GetLegacyName()));
      SliderWidget2DBase::SetCurrentValue(_valueInteral);
   }
}


/****************************************************************************
*     Function    : initializeEffectiveMovableArea
*     Description : This method is used to initialize the effective start and end
*                   coordinates within which the slider marker will move.
*                   The following are considered for this:
*                    - Actual slider background dimensions
*                    - Marker node dimensions
*                    - Padding Min and Max values
*     Parameters  : None
*     Return      : void
****************************************************************************/
bool SliderWidget2D::initializeEffectiveMovableArea()
{
   if (GetNode() && GetSliderBackGround() && (GetPaddingMaxVal() >= 0.0f) && (GetPaddingMinVal() >= 0.0f) && (GetMaxVal() > GetMinVal()))
   {
      Candera::Float scaleStart(0.0f);
      Candera::Float scaleEnd(1.0f);
      Candera::Float markerHalfSize(0.0f);

      if (GetSliderOrientation() == ::Candera::HorizontalLeft || GetSliderOrientation() == ::Candera::HorizontalRight)
      {
         Candera::Rectangle rect;
         GetNode()->GetEffectiveBoundingRectangle(rect);
         GetSliderBackGround()->GetEffectiveBoundingRectangle(rect); // fr83hi: 11.02.2016 changed from GetNode to GetSliderBackGround see NCG3D-5342

         Candera::Float bgAreaStart = GetSliderBackGround()->GetPosition().GetX();
         scaleStart = bgAreaStart;
         scaleEnd = bgAreaStart + rect.GetWidth();
         if (GetMarkerNode())
         {
            GetMarkerNode()->GetEffectiveBoundingRectangle(rect);
            markerHalfSize = (rect.GetWidth() / 2);
         }
      }
      else if (GetSliderOrientation() == ::Candera::VerticalTop || GetSliderOrientation() == ::Candera::VerticalBottom)
      {
         Candera::Rectangle rect;
         GetNode()->GetEffectiveBoundingRectangle(rect);
         GetSliderBackGround()->GetEffectiveBoundingRectangle(rect); // fr83hi: 11.02.2016 changed from GetNode to GetSliderBackGround see NCG3D-5342
         Candera::Float bgAreaStart = GetSliderBackGround()->GetPosition().GetY();
         scaleStart = bgAreaStart;
         scaleEnd   = bgAreaStart + rect.GetHeight();
         if (GetMarkerNode())
         {
            GetMarkerNode()->GetEffectiveBoundingRectangle(rect);
            markerHalfSize = (rect.GetHeight() / 2);
         }
      }

      if (GetSliderOrientation() == ::Candera::HorizontalLeft || GetSliderOrientation() == ::Candera::VerticalTop)
      {
         if (markerHalfSize < GetPaddingMinVal())
         {
            scaleStart += GetPaddingMinVal();
         }
         else
         {
            scaleStart += markerHalfSize;
         }

         if (markerHalfSize < GetPaddingMaxVal())
         {
            scaleEnd -= GetPaddingMaxVal();
         }
         else
         {
            scaleEnd -= markerHalfSize;
         }
      }
      else if (GetSliderOrientation() == ::Candera::HorizontalRight || GetSliderOrientation() == ::Candera::VerticalBottom)
      {
         // view from bottom to top or right to left
         if (markerHalfSize < GetPaddingMaxVal())
         {
            scaleStart += GetPaddingMaxVal();
         }
         else
         {
            scaleStart += markerHalfSize;
         }

         if (markerHalfSize < GetPaddingMinVal())
         {
            scaleEnd -= GetPaddingMinVal();
         }
         else
         {
            scaleEnd -= markerHalfSize;
         }
      }

      if ((_scaleStart != scaleStart) || (_scaleEnd != scaleEnd) || (_markerHalfSize != markerHalfSize))
      {
         _scaleStart = scaleStart;
         _scaleEnd = scaleEnd;
         _markerHalfSize = markerHalfSize;
         _updatePositionOnNextFrame = true;
      }

      return checkValidStepsConfiguration();
   }
   return false;
}


/****************************************************************************
*     Function : checkValidStepsConfiguration
****************************************************************************/
bool SliderWidget2D::checkValidStepsConfiguration()
{
   Candera::Float valueRange = GetMaxVal() - GetMinVal();
   Candera::Float stepSizeInternal(1.0f);

   if (GetUseNumofSteps() && GetNumOfSteps() > 0)                 // use a step counter
   {
      // GetCurentValue()holds the current value between Min and Max
      // e.g. FM Frequenz
      // Minval :84
      // MaxVal 104
      // NumOfSteps : 5 ? (Max-Min)/(steps-1) ?  20/4  ?? stepsize=5 Mhz
      // result : step0=84 step0=89; step2=94; step3=99; step4=104;
      stepSizeInternal =  valueRange / GetNumOfSteps();
   }
   else if (GetStepSize() > 0 && GetStepSize() <= valueRange)   // step size based value range GetMinVal / GetMaxVal
   {
      // GetCurentValue() holds the current value between Min and Max
      // e.g. FM Frequenz
      // Minval :84
      // MaxVal 104
      // stepSize : 10(Mhz) ->  ? ((Max-Min)/stepsize)+1    ?? stepcount=2
      // result : step0=84 -> 0; step1=94; step2=104;
      stepSizeInternal = GetStepSize();
   }
   if (_stepSizeInteral != stepSizeInternal)
   {
      _stepSizeInteral = stepSizeInternal;
      _updatePositionOnNextFrame = true;
   }
   return true;
}


/****************************************************************************
*  Called when Localization/Culture has changed during runtime
****************************************************************************/
void SliderWidget2D::CultureChanged()
{
   // get direction
   Candera::Globalization::Culture::SharedPointer culture = Globalization::CultureManager::GetInstance().GetCurrentCulture();
   _rightToLeft = ((culture != 0) && culture->GetTextDirection() == ::Candera::Globalization::RightToLeft);
   dirty();
   triggerUpdate();
}


/****************************************************************************
*     Function    : OnMessage
*     Description : Is called when a message shall be distributed through the view
*                   tree and its views and widgets
*     Parameters  : Message object to be processed.
*     Return      : true if the message is consumed
*                   false if the message should be forwarded
****************************************************************************/
bool SliderWidget2D::OnMessage(const Courier::Message& msg)
{
   if (Base::OnMessage(msg))
   {
      return true;
   }

   if (msg.IsFocus())
   {
      switch (msg.GetId())
      {
         case Courier::TouchMsg::ID :
            return bOnTouchMessage(msg);
         default:
            break;
      }
   }
   return false;
}


/****************************************************************************
*     OnTouch
****************************************************************************/
bool SliderWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool nodeTouched = false;

   if ((GetNode() != NULL) && IsVisible() && (GetNode()->IsEffectiveRenderingEnabled()))
   {
      const Candera::Vector2 pointInWorldSpace = Math2D::TransformViewportToScene(camera2D, Math2D::TransformRenderTargetToViewport(camera2D, point));
      Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());
      matrix.Inverse();
      const Candera::Vector2 pointInNodeSpace = matrix.Multiply(pointInWorldSpace);
      Candera::Rectangle r;
      // GetSliderBackGround()->GetEffectiveBoundingRectangle(r);
      // GetNode()->GetEffectiveBoundingRectangle(rect);
      GetNode()->GetComputedBoundingRectangle(r);  // fr83hi: 11.02.2016 todo: is this correct -> GetEffectiveBoundingRectangle should be the right one
      nodeTouched = r.Contains(pointInNodeSpace);
   }
   if (!nodeTouched)
   {
      nodeTouched = Base::OnTouch(camera2D, point);
   }
   return nodeTouched;
}


/****************************************************************************
*     Function    : bOnTouchMessage
*     Description : This method is used to process touch message.
*     Parameters  : Message object to be processed.
*     Return      : true if the message is consumed
*                   false if the message should be forwarded or not consumed
****************************************************************************/
bool SliderWidget2D::bOnTouchMessage(const Courier::Message& msg)
{
#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT
   PARAM_UNUSED(msg);
   return false;
#else
   const Courier::TouchMsg* touchMsg = Courier::message_cast<const Courier::TouchMsg*>(&msg);

   if (touchMsg != 0)
   {
      // World position in node position
      Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());  // GetSliderBackGround()
      matrix.Inverse();
      const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)touchMsg->GetXPos(), (Float)touchMsg->GetYPos()));

      Float x = pointInNodeSpace.GetX();
      Float y = pointInNodeSpace.GetY();

      switch (touchMsg->GetState())
      {
         case Courier::TouchMsgState::Down:
            _bIsInTouchSession = true;
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH down"));
            _valueInteral = calculateValueOnGraphPosition(x, y);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::DOWN) val=%f name=%s", _valueInteral, GetLegacyName()));
            setMarkerMode(MarkerPressed);
            setTouchStatus(SliderTouchDown);
            setCurrentValue(GetContinuousUpdateOnDrag());
            dirty();
            return true;

         case Courier::TouchMsgState::Move:
            if (!GetSliderBackGround()->IsInsideBoundingRectangle(Candera::Vector2(x, y)))
            {
               if (_bIsInTouchSession)
               {
                  _valueInteral = calculateValueOnGraphPosition(x, y);
                  ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::MOVE) val=%f name=%s", _valueInteral, GetLegacyName()));
                  setMarkerMode(MarkerPressed);
                  setTouchStatus(SliderTouchMove);
                  setCurrentValue(GetContinuousUpdateOnDrag());
                  dirty();
                  return true;
               }
            }
            return false;

         case Courier::TouchMsgState::Up:
            if (_bIsInTouchSession)
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH up"));
               _bIsInTouchSession = false;
               _valueInteral = calculateValueOnGraphPosition(x, y);
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::UP) val=%f name=%s", _valueInteral, GetLegacyName()));
               setMarkerMode(MarkerNormal);
               setTouchStatus(SliderTouchUp);
               setCurrentValue(true);
               if (_isPartofList)
               {
                  bSendCurrentValChangedMsg();
               }
               setTouchStatus(SliderNotTouched);
               dirty();
               return true;
            }

            break;

         default:
            break;
      }
   }

   return false;
#endif
}


bool SliderWidget2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool ret = false;
#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT

   // World position in node position
   Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());  // GetSliderBackGround()
   matrix.Inverse();
   const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)gestureData._pt1.x, (Float)gestureData._pt1.y));

   Float x = pointInNodeSpace.GetX();
   Float y = pointInNodeSpace.GetY();

   Float calculatedValue = calculateValueOnGraphPosition(x, y);

   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         _bIsInTouchSession = true;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH down"));
         setMarkerMode(MarkerPressed);
         setTouchStatus(SliderTouchDown);

         if (GetMarkerMovement() == ::Candera::TimedStepMovement)
         {
            if (calculatedValue < _valueInteral)
            {
               _valueInteral -= _stepSizeInteral;
            }
            else if (calculatedValue > _valueInteral)
            {
               _valueInteral += _stepSizeInteral;
            }
            else
            {
               //nothing to do
            }
            setCurrentValue(true);
         }
         else
         {
            _valueInteral = calculatedValue;
            setCurrentValue(GetContinuousUpdateOnDrag());
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::DOWN) val=%f name=%s", _valueInteral, GetLegacyName()));
         dirty();
         ret = true;
      }
      break;

      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         _bIsInTouchSession = false;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH up"));
         setMarkerMode(MarkerNormal);
         setTouchStatus(SliderTouchUp);

         if (GetMarkerMovement() != ::Candera::TimedStepMovement)
         {
            _valueInteral = calculatedValue;
         }
         setCurrentValue(true);

         //todo: this should not be necessary due to special handling in ListWidget2DSliderWidget2D::OnChanged
         if (_isPartofList)
         {
            bSendCurrentValChangedMsg();
         }

         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::UP) val=%f name=%s", _valueInteral, GetLegacyName()));
         setTouchStatus(SliderNotTouched);
         dirty();
         ret = true;
      }
      break;

      case hmibase::input::gesture::GestureEvent::ET_HOLD:
      case hmibase::input::gesture::GestureEvent::ET_REPEAT:
      {
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: HOLD or REPEAT markerMovement=%d", ETG_CENUM(::Candera::MarkerMovement, GetMarkerMovement())));
         if (GetMarkerMovement() == ::Candera::TimedStepMovement)
         {
            if (calculatedValue < _valueInteral)
            {
               _valueInteral -= _stepSizeInteral;
            }
            else if (calculatedValue > _valueInteral)
            {
               _valueInteral += _stepSizeInteral;
            }
            else
            {
               //nothing to do
            }
            setCurrentValue(true);

            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (HOLD or REPEAT) val=%f name=%s", _valueInteral, GetLegacyName()));
            dirty();
            ret = true;
         }
      }
      break;

      default:
         break;
   }
#else
//   PARAM_UNUSED(msg);
#endif
   return ret;
}


bool SliderWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool ret = false;

#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT
   // World position in node position
   Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());  // GetSliderBackGround()
   matrix.Inverse();
   const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)gestureData._pt1.x, (Float)gestureData._pt1.y));

   Float x = pointInNodeSpace.GetX();
   Float y = pointInNodeSpace.GetY();

   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         _bIsInTouchSession = true;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH down"));
         _valueInteral = calculateValueOnGraphPosition(x, y);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::DOWN) val=%f name=%s", _valueInteral, GetLegacyName()));
         setMarkerMode(MarkerPressed);
         setTouchStatus(SliderTouchMove);	//As Touch Down will always be received first by Tap Gesture and hence set there. This is continuation of Move Operation under Drag Gesture.
         setCurrentValue(GetContinuousUpdateOnDrag());
         dirty();
         ret = true;
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         _bIsInTouchSession = false;
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D:: TOUCH up"));
         _valueInteral = calculateValueOnGraphPosition(x, y);
         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::UP) val=%f name=%s", _valueInteral, GetLegacyName()));
         setMarkerMode(MarkerNormal);
         setTouchStatus(SliderTouchUp);
         setCurrentValue(true);
         if (_isPartofList)
         {
            bSendCurrentValChangedMsg();
         }
         setTouchStatus(SliderNotTouched);
         dirty();
         ret = true;
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
      {
         if (!GetSliderBackGround()->IsInsideBoundingRectangle(Candera::Vector2(x, y)))
         {
            _valueInteral = calculateValueOnGraphPosition(x, y);
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::bOnTouchMessage (TouchMsgState::MOVE) val=%f name=%s", _valueInteral, GetLegacyName()));
            setMarkerMode(MarkerPressed);
            setTouchStatus(SliderTouchMove);
            setCurrentValue(GetContinuousUpdateOnDrag());
            dirty();
            ret = true;
         }
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_HOLD:
         break;
      case hmibase::input::gesture::GestureEvent::ET_REPEAT:
         break;
      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         break;
      default:
         break;
   }
#else
//   PARAM_UNUSED(msg);
#endif
   return ret;
}


/****************************************************************************
*     Function    : checkSliderStepsMode
*     Description : This method is used for visibility check of the widget property "NumOfSteps"
*     Parameters  : None
*     Return      :
*                 true - if slider should use number of steps
*                 false - if not
****************************************************************************/
bool SliderWidget2D::checkSliderStepsMode() const
{
   return GetUseNumofSteps();
}


/****************************************************************************
*     Function    : checkSliderSizeMode
*     Description : This method is used for visibility check of the widget property "StepSize"
*     Parameters  : None
*    Return       : true - if slider should use step size
*                   false - if not
****************************************************************************/
bool SliderWidget2D::checkSliderSizeMode() const
{
   return !checkSliderStepsMode();
}


/****************************************************************************
*     Function    : setMarkerMode
*     Description : Setter method to the set the slider marker mode
*     Parameters  : mode - the required slider marker mode (Normal/Pressed)
*     Return      : None
****************************************************************************/
void SliderWidget2D::setMarkerMode(SliderMakerMode mode)
{
   if (_markerMode != mode)
   {
      _markerMode = mode;
      _bMarkerBitmapChanged = true;
   }

   dirty();
}


/****************************************************************************
*     Function    : setTouchStatus
*     Description : Setter method to the set the slider Touch state
*     Parameters  : mode - the required slider Touch state (TouchUp/TouchDown/TouchMove)
*     Return      : None
****************************************************************************/
void SliderWidget2D::setTouchStatus(const SliderTouchState state)
{
   if (_sliderTouchState != state)
   {
      _sliderTouchState = state;
   }
}


/****************************************************************************
*     Function    : bSendCurrentValChangedMsg
*     Description : To send SliderCurrentValueChangedMsg when Slider is used inside the FlexList
*     Parameters  : None
*     Return      : Bool
****************************************************************************/
bool SliderWidget2D::bSendCurrentValChangedMsg()
{
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "PostValueChangedMsg value=%f touchState=%u [%p.%50s.%s]", GetCurrentValue(), _sliderTouchState, this, GetViewName(), GetLegacyName()));

   SliderCurrentValueChangedMsg* msg = COURIER_MESSAGE_NEW(SliderCurrentValueChangedMsg)(GetParentView()->GetId(), Courier::Identifier(GetLegacyName()), GetCurrentValue(), _sliderTouchState);
   if (0 != msg)
   {
      static_cast<void>(msg->Post());
      return true;
   }
   return false;
}


/****************************************************************************
*     Function    : updateMarkerBitmap
*     Description : This method updates the marker bitmap based on the
*                   marker mode
****************************************************************************/
void SliderWidget2D::updateMarkerBitmap()
{
   if (GetMarkerSelectedBitmap() == 0)
   {
      return;
   }

   if (GetMarkerNormalBitmap() == 0)
   {
      return;
   }

   if (GetMarkerNode() == NULL || GetAssetProvider() == NULL)
   {
      return;
   }

   if (_bitmapMarker != NULL)
   {
      _bitmapMarker.Release();
   }

   _bitmapMarker = BitmapImage2D::Create();

   if ((_bitmapMarker != NULL) /*&& (_BitmapBg != 0)*/)
   {
      Candera::MemoryManagement::SharedPointer<Candera::Bitmap> _BitmapBg;

      if (_markerMode == MarkerPressed)
      {
         _BitmapBg = GetAssetProvider()->GetBitmapById(GetMarkerSelectedBitmap());
      }
      else
      {
         _BitmapBg = GetAssetProvider()->GetBitmapById(GetMarkerNormalBitmap());
      }

      if (!_BitmapBg.PointsToNull())
      {
         _bitmapMarker->Unload();
         _bitmapMarker->SetBitmap(_BitmapBg);
         _bitmapMarker->Upload();
         _bitmapMarker->Update();
      }

      RenderNode* renderNode = Dynamic_Cast<RenderNode*>(GetMarkerNode());

      if (renderNode != 0)
      {
         if (renderNode->GetEffect(0)->IsTypeOf(BitmapBrushBlend::GetTypeId()))
         {
            renderNode->Unload();
            BitmapBrushBlend* effect = Dynamic_Cast<BitmapBrushBlend*>(renderNode->GetEffect(0));
            effect->GetBitmapBrush().Image().Set(_bitmapMarker);
            effect->Update();
            renderNode->Upload();
         }
      }
   }

   dirty();
}


/****************************************************************************
*     Function    : dirty
*     Description : handles the graphics update mechanism
****************************************************************************/
void SliderWidget2D::dirty()
{
   _updatePositionOnNextFrame = true;
   /// InvalidateLayout();
   Invalidate();
}


/****************************************************************************
*     Function    : Candera::Float calculateValueOnGraphPositionOnTouch
*     Description : This method checks for the slider alignment and forwards
*                   the touch coordinates accordingly
*     Parameters  : xPos - The reported x coordinate of the touch
*                   yPos - The reported y coordinate of the touch
*     Return      : void
****************************************************************************/
Candera::Float SliderWidget2D::calculateValueOnGraphPosition(Float xPos, Float yPos) const
{
   Candera::Float graphPosition = 0.0F;

   if (getScaleWidth() <= 0.0F)
   {
      // assert
      return _valueInteral;
   }
   switch (GetSliderOrientation())
   {
      case ::Candera::HorizontalLeft:
      case ::Candera::HorizontalRight:
         if (xPos < _scaleStart)
         {
            graphPosition = _scaleStart;
         }
         else if (xPos > _scaleEnd)
         {
            graphPosition = _scaleEnd;
         }
         else
         {
            graphPosition = xPos;
         }

         break;

      case ::Candera::VerticalBottom:
      case ::Candera::VerticalTop:
         if (yPos < _scaleStart)
         {
            graphPosition = _scaleStart;
         }
         else if (yPos > _scaleEnd)
         {
            graphPosition = _scaleEnd;
         }
         else
         {
            graphPosition = yPos;
         }

         break;

      default:
         graphPosition = 0.0F;
         break;
   }

   Candera::Float valueOffset = 0.0F;

   if (GetSliderOrientation() == ::Candera::HorizontalLeft || GetSliderOrientation() == ::Candera::VerticalTop)
   {
      valueOffset = graphPosition - _scaleStart;
   }
   else if (GetSliderOrientation() == ::Candera::HorizontalRight || GetSliderOrientation() == ::Candera::VerticalBottom)
   {
      // view from bottom to top or right to left
      valueOffset = _scaleEnd - graphPosition;
   }

   Float valueRange = GetMaxVal() - GetMinVal();
   Candera::Float value = (valueOffset * valueRange / getScaleWidth()) + GetMinVal();
   Float valueMarker = nearestStepSizeValue(value);
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::calculateValueOnGraphPosition x=%f y=%f [%f] received.", xPos, yPos, _valueInteral));
   return valueMarker;
}


/****************************************************************************
*  Function: nearestStepSizeValue
****************************************************************************/
Candera::Float SliderWidget2D::nearestStepSizeValue(Candera::Float value) const
{
   if (value < GetMinVal())
   {
      value = GetMinVal();
   }
   else if (value > GetMaxVal())
   {
      value = GetMaxVal();
   }
   if (((GetMarkerMovement() == ::Candera::StepMovement) || (GetMarkerMovement() == ::Candera::TimedStepMovement)) && (_stepSizeInteral > 0.0f))
   {
      Float v = GetMinVal();
      while (v < value)
      {
         v += _stepSizeInteral;
      }
      Float rest = v - value;
      if (rest < (_stepSizeInteral / 2))
      {
         value = v;
      }
      else
      {
         value = v - _stepSizeInteral;
      }
      /*
            Float rest = (Float)fmod(value, _stepSizeInteral);
            if (rest != 0.0f)
            {
               if (rest >= (_stepSizeInteral/2))
                  value = value + _stepSizeInteral - rest;
               else
                  value = value - rest;
            }
      */
      if (value < GetMinVal())
      {
         value = GetMinVal();
      }
      else if (value > GetMaxVal())
      {
         value = GetMaxVal();
      }
   }
   return value;
}


/****************************************************************************
*  Function: updatePosition
****************************************************************************/
void SliderWidget2D::updatePosition()
{
   Float valueRange = GetMaxVal() - GetMinVal();
   if (valueRange <= 0.0f)
   {
      return;
   }
   if (0 != GetNode())
   {
      Candera::Layouter::InvalidateLayout(GetNode());
   }
   if (_valueInteral < GetMinVal())
   {
      _valueInteral = GetMinVal();
   }
   else if (_valueInteral > GetMaxVal())
   {
      _valueInteral = GetMaxVal();
   }

   Candera::Float valuePixelPosOffs = ((_valueInteral - GetMinVal()) * getScaleWidth() / valueRange);
   Candera::Float valuePixelPos = _scaleStart;

   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** SliderWidget2D::updatePosition _valueInternal='%f, valuePixelPosOffs='%f'", _valueInteral, valuePixelPosOffs));

   if (GetSliderOrientation() == ::Candera::HorizontalRight || GetSliderOrientation() ==  ::Candera::VerticalBottom)
   {
      valuePixelPos = _scaleEnd - valuePixelPosOffs;
   }
   else // ::Candera::HorizontalLeft + ::Candera::VerticalTop
   {
      valuePixelPos = _scaleStart + valuePixelPosOffs;
   }

   switch (GetSliderOrientation())
   {
      case ::Candera::HorizontalLeft:  // (standard)
      case ::Candera::HorizontalRight: // arabic e.g.
         if (GetBitmapFillNode())
         {
            Candera::Rectangle rFiller;
            GetBitmapFillNode()->GetEffectiveBoundingRectangle(rFiller);
            if (rFiller.GetWidth() > 0)
            {
               Float xpos, scaleX = valuePixelPosOffs / rFiller.GetWidth();
               if (GetSliderOrientation() == ::Candera::HorizontalRight)
               {
                  xpos = valuePixelPos + (GetFillerPositionIsKnobCenter() ? 0 : _markerHalfSize);
               }
               else
               {
                  xpos = _scaleStart - (GetFillerPositionIsKnobCenter() ? 0 : _markerHalfSize);
               }
               GetBitmapFillNode()->SetPosition(xpos, GetBitmapFillNode()->GetPosition().GetY());
               GetBitmapFillNode()->SetScale(scaleX, 1.0F);
            }
         }
         if (GetMarkerNode())
         {
            Float xpos = (valuePixelPos - _markerHalfSize);
            GetMarkerNode()->SetPosition(xpos, GetMarkerNode()->GetPosition().GetY());
         }
         break;

      case ::Candera::VerticalBottom: // Bottom to Top (standard)
      case ::Candera::VerticalTop:  // Top to Bottom
         if (GetBitmapFillNode())
         {
            Candera::Rectangle rFiller;
            GetBitmapFillNode()->GetEffectiveBoundingRectangle(rFiller);
            if (rFiller.GetHeight() > 0)
            {
               Float ypos, scaleY = valuePixelPosOffs / rFiller.GetHeight();
               if (GetSliderOrientation() == ::Candera::VerticalBottom)
               {
                  ypos = valuePixelPos + (GetFillerPositionIsKnobCenter() ? 0 : _markerHalfSize);
               }
               else
               {
                  ypos = _scaleStart - (GetFillerPositionIsKnobCenter() ? 0 : _markerHalfSize);
               }
               GetBitmapFillNode()->SetPosition(GetBitmapFillNode()->GetPosition().GetX(), ypos);
               GetBitmapFillNode()->SetScale(1.0F, scaleY);
            }
         }
         if (GetMarkerNode())
         {
            Float ypos = (valuePixelPos - _markerHalfSize);
            GetMarkerNode()->SetPosition(GetMarkerNode()->GetPosition().GetX(), ypos);
         }
         break;

      default:
         break;
   }

   RenderNode* renderNodeFiller = Dynamic_Cast<RenderNode*>(GetBitmapFillNode());
   if (renderNodeFiller != 0)
   {
#ifndef WIN32
      renderNodeFiller->SetClippingRect();
#endif
   }

   // fr83hi: If no Layouter connected, the clipping values will not be recalculated after SetPosition
   // The Defaultlayouter work with childs; works only if the marker has a group as parent
   RenderNode* renderNodeMarker = Dynamic_Cast<RenderNode*>(GetMarkerNode());
   if (renderNodeMarker != 0)
   {
#ifndef WIN32
      renderNodeMarker->SetClippingRect();
#endif
   }
   _updatePositionOnNextFrame = false;
}


/****************************************************************************
*  Function: isInsideBoudingRectangle
****************************************************************************/
bool SliderWidget2D::isInsideBoudingRectangle(Candera::Float xPos, Candera::Float yPos)
{
   const ViewScene2D* view2D = dynamic_cast<const ViewScene2D*>(GetParentView());
   if (view2D != NULL)
   {
      const ViewScene2D::CameraPtrVector& cameras = view2D->GetCameraPtrVector();
      for (ViewScene2D::CameraPtrVector::ConstIterator it = cameras.ConstBegin(); it != cameras.ConstEnd(); ++it)
      {
         Camera2D* camera = *it;
         if ((camera != NULL) && camera->IsRenderingEnabled()
               && (GetSliderBackGround() != NULL)
               && IsPickIntersectingNode(*camera, GetSliderBackGround(), Candera::Vector2(xPos, yPos)))
         {
            return true;
         }
      }
   }

   return false;
}
