/* ***************************************************************************************
* FILE:          AngularSliderWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AngularSliderWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "Widgets/2D/ControlTemplate/ControlTemplateBinding.h"
#include "widget2D_std_if.h"
#include "AngularSliderWidget2D.h"
#include <math.h>       /* atan */
#include <cmath>
#define PI 3.14159265
#define ANIMATION_REPEAT_COUNT 1

//using namespace FeatStd;
//using namespace Courier;
using namespace Candera;

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SLIDER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AngularSliderWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(AngularSliderWidget2D)

#define VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT


/****************************************************************************
*     Function    : AngularSliderWidget2D
*     Description : constructor
****************************************************************************/
AngularSliderWidget2D::AngularSliderWidget2D() : Base(),
   _currentIndicatorRotation(0.0f),
   _previousIndicatorRotation(0.0f),
   _bIsAngularMarkerTouched(false),
   _bIsAngularBgTouched(false),
   _bIsTouchOnBgValid(false),
   _bUpdateValid(true),
   _bMoveFlag(false),
   _actualMaxAngle(0.0f),
   _actualMinAngle(0.0f),
   _prevAngle(0.0f),
   _animationHelperInterface(NULL)
{
   //Animation
   _animationHelperInterface = FEATSTD_NEW(AnimationHelperInterface);
}


/****************************************************************************
*     Function    : ~SliderWidget2D()
*     Description : Destructor
****************************************************************************/
AngularSliderWidget2D::~AngularSliderWidget2D()
{
   if (_animationHelperInterface != 0)
   {
      //_animationHelperInterface->stopAllExistingAnimations();
      FEATSTD_DELETE(_animationHelperInterface);
      _animationHelperInterface = NULL;
   }
}


/****************************************************************************
*     Function    : initializeEffectiveMovableArea
*     Description : This method is used to initialize the effective start and end
*                   angle within which the slider marker will move.
*                   The following are considered for this:
*                    - Maximum angle, Minimumangle
*                    - Marker node dimensions
*                    - Padding Min and Max angles
*     Parameters  : None
*     Return      : void
****************************************************************************/
bool AngularSliderWidget2D::initializeEffectiveMovableArea()
{
   switch (GetAngularSliderType())
   {
      case AngularSliderSemiCircle:
      {
         _actualMaxAngle = GetMaximumAngle() - GetPaddingMaximumAngle();
         _actualMinAngle = GetMinimumAngle() + GetPaddingMinimumAngle();
         break;
      }
      case AngularSliderFullCircle:
      {
         if (GetRotationType() == enClockWise)
         {
            _actualMinAngle = GetMinimumAngle() + GetPaddingMinimumAngle();
            _actualMaxAngle = GetMinimumAngle() + 360;
            _actualMaxAngle = _actualMaxAngle - GetPaddingMaximumAngle();
         }
         else if (GetRotationType() == enAntiClockWise)
         {
            _actualMinAngle = GetMinimumAngle() - GetPaddingMinimumAngle();
            _actualMaxAngle = GetMinimumAngle() + (-360);
            _actualMaxAngle = _actualMaxAngle + GetPaddingMaximumAngle();
         }
      }
      break;
      default:
         break;
   }
   return checkValidStepsConfiguration();
}


/****************************************************************************
*     Function    : CloneFrom
*     Description : required for FlexList cloning
****************************************************************************/
bool AngularSliderWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const AngularSliderWidget2D* original = CLONEABLE_WIDGET_CAST<const AngularSliderWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetSliderBackGround(controlTemplateMap.ResolveNodeClone(original->GetSliderBackGround()));
      SetMarkerNode(controlTemplateMap.ResolveNodeClone(original->GetMarkerNode()));
      SetBitmapFillNode(controlTemplateMap.ResolveNodeClone(original->GetBitmapFillNode()));

      SetMinVal(original->GetMinVal());
      SetMaxVal(original->GetMaxVal());
      SetMarkerMovement(original->GetMarkerMovement());
      SetContinuousUpdateOnDrag(original->GetContinuousUpdateOnDrag());
      SetPaddingMaximumAngle(original->GetPaddingMaximumAngle());
      SetPaddingMinimumAngle(original->GetPaddingMinimumAngle());
      SetAngularSliderType(original->GetAngularSliderType());
      SetMaximumAngle(original->GetMaximumAngle());
      SetMinimumAngle(original->GetMinimumAngle());
      SetUseNumofSteps(original->GetUseNumofSteps());
      SetNumOfSteps(original->GetNumOfSteps());
      SetStepSize(original->GetStepSize());
      SetEnableAnimation(original->GetEnableAnimation());
      SetAnimationDuration(original->GetAnimationDuration());
      SliderWidget2DBase::SetCurrentValue(original->_valueInteral);
      cloned = true;
   }
   return cloned;
}


/****************************************************************************
*     Function    : Update()
*     Description : Overridden framework method.
*                   SliderWidget2D update function is overridden to invalidate if animation is enabled and animation is playing,
*					to continuously invalidate during animation
*     Parameters  : None
*     Return      : void
****************************************************************************/
void AngularSliderWidget2D::Update()
{
   Base::Update();
   if ((_animationHelperInterface != 0) && (_animationHelperInterface->isAnimationOnAnyPlayerRunning()))
   {
      Invalidate();
   }
}


/****************************************************************************
*     OnTouch
****************************************************************************/
bool AngularSliderWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   Candera::Node2D* node, * bgNode;
   if ((GetNode() != NULL) && IsVisible() && (GetNode()->IsEffectiveRenderingEnabled()))
   {
      const Candera::Vector2 pointInWorldSpace = Math2D::TransformViewportToScene(camera2D, Math2D::TransformRenderTargetToViewport(camera2D, point));

      //To check if marker is touched
      node = GetMarkerNode();
      if (node != NULL)
      {
         Candera::Matrix3x2 matrix(node->GetWorldTransform());
         matrix.Inverse();
         const Candera::Vector2 pointInNodeSpace = matrix.Multiply(pointInWorldSpace);
         Candera::Rectangle r;
         node->GetComputedBoundingRectangle(r);
         _bIsAngularMarkerTouched = r.Contains(pointInNodeSpace);
      }
      //To check if BG is touched
      bgNode = GetSliderBackGround();
      if (bgNode != NULL)
      {
         Candera::Matrix3x2 matrix(bgNode->GetWorldTransform());
         matrix.Inverse();
         const Candera::Vector2 pointInNodeSpace = matrix.Multiply(pointInWorldSpace);
         Candera::Rectangle r;
         bgNode->GetComputedBoundingRectangle(r);
         _bIsAngularBgTouched = r.Contains(pointInNodeSpace);
      }
   }
   return (_bIsAngularBgTouched || _bIsAngularMarkerTouched);
}


/****************************************************************************
*     Function    : bOnTouchMessage
*     Description : This method is used to process touch message.
*     Parameters  : Message object to be processed.
*     Return      : true if the message is consumed
*                   false if the message should be forwarded or not consumed
****************************************************************************/
bool AngularSliderWidget2D::bOnTouchMessage(const Courier::Message& msg)
{
#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT
   PARAM_UNUSED(msg);
   return false;
#else
   return false;
#endif
}


bool AngularSliderWidget2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool ret = false;
#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT

   // World position in node position
   Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());  // AngularSliderWidgetNode
   matrix.Inverse();
   const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)gestureData._pt1.x, (Float)gestureData._pt1.y));
   Float x = pointInNodeSpace.GetX();
   Float y = pointInNodeSpace.GetY();

   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_INITIATE:
         break;
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         ret = bOnTouchDown(gestureData, x, y);
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         ret = bOnTouchUp(x, y);
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_HOLD:
         break;
      case hmibase::input::gesture::GestureEvent::ET_REPEAT:
         break;
      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         break;
      case hmibase::input::gesture::GestureEvent::ET_DT_START:
         break;
      default:
         break;
   }
#else
   //   PARAM_UNUSED(msg);
#endif
   return ret;
}


bool AngularSliderWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureData)
{
   bool ret = false;
#ifdef VARIANT_S_FTR_ENABLE_SLIDER_GESTURE_SUPPORT
   // World position in node position
   Candera::Matrix3x2 matrix(GetNode()->GetWorldTransform());   // AngularSliderWidgetNode
   matrix.Inverse();
   const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)gestureData._pt1.x, (Float)gestureData._pt1.y));
   Float x = pointInNodeSpace.GetX();
   Float y = pointInNodeSpace.GetY();

   switch (gestureData._gestureState)
   {
      case hmibase::input::gesture::GestureEvent::ET_START:
      {
         ret = bOnTouchDown(gestureData, x, y);
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_END:
      {
         ret = bOnTouchUp(x, y);
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_MOVE:
      {
         ret = bOnTouchMove(x, y);
      }
      break;
      case hmibase::input::gesture::GestureEvent::ET_HOLD:
         break;
      case hmibase::input::gesture::GestureEvent::ET_REPEAT:
         break;
      case hmibase::input::gesture::GestureEvent::ET_ABORT:
         break;
      default:
         break;
   }
#else
   //   PARAM_UNUSED(msg);
#endif
   return ret;
}


/****************************************************************************
*     Function    : updatePosition()
*     Description : SliderWidget2D update function is overridden
*     Parameters  : None
*     Return      : void
****************************************************************************/
void AngularSliderWidget2D::updatePosition()
{
   Float valueRange = GetMaxVal() - GetMinVal();
   if (valueRange <= 0.0f)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AngularSliderWidget2D: Both min and max val are equal or Min is greater than max value"));
      return;
   }

   _valueInteral = checkValueRange(_valueInteral);

   if (NULL != GetMarkerNode())
   {
      if (GetEnableAnimation())
      {
         //If animation is enabled , start animation should be called only for Touch Up recevied on Background
         //Start animation we not be called for touch on marker and drag
         //Start animation is not called on touch on Background and drag but only on Touch Up
         //Start animation will be called when there is update of the current value from the binding property with animation enabled
         if (!(_bIsAngularBgTouched) && !(_bIsAngularMarkerTouched))
         {
            _previousIndicatorRotation = _currentIndicatorRotation;
            _currentIndicatorRotation = calculateCurrentRotationVal();
            _animationHelperInterface->startAnimation(GetMarkerNode(), GetAnimationDuration(), &_previousIndicatorRotation, &_currentIndicatorRotation, enAnimateRotation, ANIMATION_REPEAT_COUNT);
         }
         else if (_bIsAngularMarkerTouched)
         {
            _currentIndicatorRotation = calculateCurrentRotationVal();
            GetMarkerNode()->SetRotation(_currentIndicatorRotation);
         }
      }
      else
      {
         _currentIndicatorRotation = calculateCurrentRotationVal();
         GetMarkerNode()->SetRotation(_currentIndicatorRotation);
      }
   }
   _updatePositionOnNextFrame = false;
}


/****************************************************************************
*     Function    : calculateCurrentRotationVal()
*     Description : Calculates the rotation angle based on current value
*     Parameters  : None
*     Return      : Candera::Float rotation angle
****************************************************************************/
Candera::Float AngularSliderWidget2D::calculateCurrentRotationVal()
{
   if (GetMaxVal() != GetMinVal())  // Max and min value shall not be equal.
   {
      _currentIndicatorRotation = _actualMinAngle + (((_actualMaxAngle - _actualMinAngle) / (GetMaxVal() - GetMinVal())) * (_valueInteral - GetMinVal()));
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "AngularSliderWidget2D: Both min and max val are equal"));
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "AngularSliderWidget2D: CurrentValue:[%f] CurrentRotation:[%f]", GetCurrentValue(), _currentIndicatorRotation));
   return _currentIndicatorRotation;
}


bool AngularSliderWidget2D::bOnTouchDown(const hmibase::input::gesture::GestureEvent& gestureData, Candera::Float x, Candera::Float y)
{
   Candera::Float currentValue;
   if (_bIsAngularMarkerTouched && GetMarkerNode() != NULL)
   {
      Candera::Node2D* markerNode = GetMarkerNode();
      Candera::Rectangle r;
      markerNode->GetEffectiveBoundingRectangle(r);
      {
         currentValue = calculateCurrentValueForAngularPosition(x, y); //(x, y) are with respect to AngularsliderWidget2D node
         if (_bUpdateValid)
         {
            _valueInteral = currentValue;
         }
         //	_bMoveFlag = true;
         setCurrentValue(GetContinuousUpdateOnDrag());
         dirty();
      }
   }
   else if (_bIsAngularBgTouched && GetSliderBackGround() != NULL)
   {
      Candera::Node2D* bGNode = GetSliderBackGround();
      Candera::Matrix3x2 matrix(bGNode->GetWorldTransform());
      matrix.Inverse();
      const Candera::Vector2 pointInNodeSpace = matrix.Multiply(Candera::Vector2((Float)gestureData._pt1.x, (Float)gestureData._pt1.y));
      Candera::Rectangle r;
      bGNode->GetEffectiveBoundingRectangle(r); //touch on BG is recognised , center of the angular slider widget is based on the width and height of the BG given
      if (r.Contains(pointInNodeSpace))
      {
         //If BG is touched, check if the touch is in within the circular region  between GetTouchableRadius() +/- GetTouchableRadiusOffset()
         if (isInside(0.0f, 0.0f, (GetTouchableRadius() + GetTouchableRadiusOffset()), (pointInNodeSpace.GetX() - r.GetWidth() / 2), (pointInNodeSpace.GetY() - r.GetHeight() / 2)))
         {
            if (!(isInside(0.0f, 0.0f, GetTouchableRadius() - GetTouchableRadiusOffset(), (pointInNodeSpace.GetX() - r.GetWidth() / 2), (pointInNodeSpace.GetY() - r.GetHeight() / 2))))
            {
               _bIsTouchOnBgValid = true; //flag to say that touch is on the small circuar area configured on BG
               currentValue = calculateCurrentValueForAngularPosition(x, y); //(x, y) are with respect to AngularsliderWidget2D node
               if (_bUpdateValid)
               {
                  _valueInteral = currentValue;
                  //	_bMoveFlag = true;
               }
               setCurrentValue(GetContinuousUpdateOnDrag());
               dirty();
            }
         }
      }
   }
   return true;
}


bool AngularSliderWidget2D::bOnTouchMove(Candera::Float x, Candera::Float y)
{
   Candera::Float currentValue;
   if (_bIsAngularMarkerTouched /*|| (_bIsAngularBgTouched) && (_bIsTouchOnBgValid)*/)
   {
      currentValue = calculateCurrentValueForAngularPosition(x, y);
      if (_bUpdateValid /*&& _bMoveFlag*/)
      {
         _valueInteral = currentValue;
         _bMoveFlag = true;
      }
      else if (_bMoveFlag)
      {
         _valueInteral = currentValue;
         _bMoveFlag = false;
      }
      else
      {
      }
      setCurrentValue(GetContinuousUpdateOnDrag());
      dirty();
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** AngularSliderWidget2D::bOnTouchMessage (TouchMsgState::MOVE) val=%f name=%s", _valueInteral, GetLegacyName()));
      return true;
   }
   return false;
}


bool AngularSliderWidget2D::bOnTouchUp(Candera::Float x, Candera::Float y)
{
   Candera::Float  currentValue;
   ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** AngularSliderWidget2D:: TOUCH up"));
   if (_bIsAngularMarkerTouched || ((_bIsAngularBgTouched) && (_bIsTouchOnBgValid)))
   {
      currentValue = calculateCurrentValueForAngularPosition(x, y);
      if (_bUpdateValid /*&& _bMoveFlag*/)
      {
         _valueInteral = currentValue;
      }
      _bIsAngularMarkerTouched = false;
      _bIsAngularBgTouched = false;
      _bIsTouchOnBgValid = false;

      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "*** AngularSliderWidget2D::bOnTouchMessage (TouchMsgState::UP) val=%f name=%s", _valueInteral, GetLegacyName()));
      setCurrentValue(true);
      dirty();
      return true;
   }
   return false;
}


/****************************************************************************
*     Function    : calculateCurrentValueForAngularPosition()
*     Description : Calculates current value based on the angle formed by the x, y co-ordinate with center
*     Parameters  : None
*     Return      : Candera::Float current value
****************************************************************************/
Candera::Float AngularSliderWidget2D::calculateCurrentValueForAngularPosition(Candera::Float x, Candera::Float y)
{
   Candera::Float fAngle = 0, fCurrentvalue = -1;

   initializeEffectiveMovableArea();
   _bUpdateValid = true;
   if (GetNode() != NULL)
   {
      //to get the angle the point makes with respect to the center.
      Candera::Rectangle rect;
      GetNode()->GetEffectiveBoundingRectangle(rect);
      Candera::Float fCentreX = rect.GetWidth() / 2;
      Candera::Float fCentreY = rect.GetHeight() / 2;

      switch (GetAngularSliderType())
      {
         case Candera::AngularSliderSemiCircle:
         {
            fAngle = HandleSemiCircle(fCentreX, fCentreY, x, y);
         }
         break;
         case Candera::AngularSliderFullCircle:
         {
            fAngle = HandleFullCircle(fCentreX, fCentreY, x, y);
         }
         break;
         default:
            break;
      }
      fCurrentvalue = GetMinVal() + (((GetMaxVal() - GetMinVal()) / (_actualMaxAngle - _actualMinAngle)) * (fAngle - _actualMinAngle));
      fCurrentvalue = checkValueRange(fCurrentvalue);
   }
   return fCurrentvalue;
}


Candera::Float AngularSliderWidget2D::checkValueRange(Candera::Float fValue)
{
   if (fValue < GetMinVal())
   {
      fValue = GetMinVal();
   }
   else if (fValue > GetMaxVal())
   {
      fValue = GetMaxVal();
   }
   return fValue;
}


bool AngularSliderWidget2D::isInside(Candera::Float circle_x, Candera::Float circle_y, Candera::Float rad, Candera::Float x, Candera::Float y)
{
   // Compare radius of circle with distance of its center from
   // given point
   if ((x - circle_x) * (x - circle_x) +
         (y - circle_y) * (y - circle_y) <= rad * rad)
   {
      return true;
   }
   else
   {
      return false;
   }
}


Candera::Float AngularSliderWidget2D::HandleSemiCircle(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   switch (GetSliderOrientation())
   {
      case HorizontalRight:
      {
         fRotationAngle = HandleSemiCircleHoriRight(centreX, centreY, x, y);
      }
      break;
      case HorizontalLeft:
      {
         fRotationAngle = HandleSemiCircleHoriLeft(centreX, centreY, x, y);
      }
      break;

      case VerticalTop:
      {
         fRotationAngle = HandleSemiCircleVerticalTop(centreX, centreY, x, y);
      }
      break;
      case VerticalBottom:
      {
         fRotationAngle = HandleSemiCircleVerticalBottom(centreX, centreY, x, y);
      }
      break;
      default:
         break;
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::HandleFullCircle(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   switch (GetRotationType())
   {
      case enClockWise:
      {
         if ((x > centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = centreY - y;
            fRotationAngle = (float)(atan2f(y , x) * 180 / (float)PI);
            fRotationAngle = -fRotationAngle;
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
         else if ((x > centreX) && (y > centreY))
         {
            x = x - centreX;
            y = y - centreY;
            fRotationAngle = (float)atan2f(y , x) * 180 / (float) PI;
            fRotationAngle = fRotationAngle;
         }
         else if ((x <= centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = centreY - y;
            fRotationAngle = atan2f(y , x) * 180 / (float) PI;
            fRotationAngle = 180 + fRotationAngle;
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
         else if ((x <= centreX) && (y > centreY))
         {
            x = centreX - x;
            y = y - centreY;
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = 180 - fRotationAngle;
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
      }
      break;
      case enAntiClockWise:
      {
         if ((x > centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = centreY - y;
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -fRotationAngle;
            //fRotationAngle = -(360 + fRotationAngle);
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
         else if ((x >= centreX) && (y > centreY))
         {
            x = x - centreX;
            y = y - centreY ;
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = fRotationAngle;
            fRotationAngle = -(360 - fRotationAngle);
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
         else if ((x <= centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = centreY - y;
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(180 - fRotationAngle);
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x ;
            y = y - centreY;
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(180 + fRotationAngle);
            fRotationAngle = ValidateAngle(fRotationAngle);
         }
      }
      break;
      default:
      {
      }
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::ValidateAngle(Candera::Float fRotationAngle)
{
   switch (GetRotationType())
   {
      case enClockWise:
      {
         if (fRotationAngle < _actualMinAngle)
         {
            if ((fRotationAngle + 360) < (_actualMaxAngle + GetPaddingMaximumAngle()))
            {
               fRotationAngle = _actualMaxAngle;
            }
            else
            {
               fRotationAngle = _actualMinAngle;
            }
            _bUpdateValid = false;		//flag to set out of given range so do not update the current value
         }
         else if (fRotationAngle > _actualMaxAngle)
         {
            fRotationAngle = _actualMaxAngle;
            _bUpdateValid = false;
         }
         else
         {
         }
      }
      break;
      case enAntiClockWise:
      {
         if (fRotationAngle > _actualMinAngle)
         {
            _bUpdateValid = false;	//flag to set out of given range so do not update the current value
            if ((fRotationAngle + (-360)) > _actualMaxAngle)
            {
               fRotationAngle = _actualMaxAngle;
            }
            else
            {
               fRotationAngle = _actualMinAngle;
            }
         }
         else if (fRotationAngle < _actualMaxAngle)
         {
            fRotationAngle = _actualMaxAngle;
            _bUpdateValid = false;
         }
         else
         {
         }
      }
      break;
      default:
         break;
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::HandleSemiCircleHoriRight(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   Candera::Float maxAngle = _actualMaxAngle;
   Candera::Float minAngle = _actualMinAngle;

   switch (GetRotationType())
   {
      case enClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -fRotationAngle;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            x = x - centreX;
            y = (y - centreY);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (y - centreY);
            //fAngle = atan(y / x) * 180 / PI;
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
      }
      break;
      case enAntiClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -fRotationAngle;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            x = x - centreX;
            y = (y - centreY);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (y - centreY);
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
      }
      break;
      default:
      {
      }
      break;
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::HandleSemiCircleHoriLeft(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   Candera::Float maxAngle = _actualMaxAngle;
   Candera::Float minAngle = _actualMinAngle;

   switch (GetRotationType())
   {
      case enClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -fRotationAngle;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (y - centreY);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
         }
      }
      break;
      case enAntiClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false; //flag to set out of given range so do not update the current value
         }
         else if ((x >= centreX) && (y > centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (y - centreY);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -fRotationAngle;
         }
      }
      break;
      default:
      {
      }
      break;
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::HandleSemiCircleVerticalTop(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   Candera::Float maxAngle = _actualMaxAngle;
   Candera::Float minAngle = _actualMinAngle;

   switch (GetRotationType())
   {
      case enClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = 90 - fRotationAngle;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(90 - fRotationAngle);
         }
         else if ((x < centreX) && (y > centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
      }
      break;
      case enAntiClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = 90 - fRotationAngle;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(90 - fRotationAngle);
         }
         else if ((x < centreX) && (y > centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
      }
      break;
      default:
      {
      }
      break;
   }
   return fRotationAngle;
}


Candera::Float AngularSliderWidget2D::HandleSemiCircleVerticalBottom(Candera::Float centreX, Candera::Float centreY, Candera::Float x, Candera::Float y)
{
   Candera::Float fRotationAngle = 0;
   Candera::Float maxAngle = _actualMaxAngle;
   Candera::Float minAngle = _actualMinAngle;
   switch (GetRotationType())
   {
      case enClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = 90 - fRotationAngle;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(90 - fRotationAngle);
         }
      }
      break;
      case enAntiClockWise:
      {
         if ((x >= centreX) && (y <= centreY))
         {
            fRotationAngle = maxAngle;
            _bUpdateValid = false;
         }
         else if ((x >= centreX) && (y > centreY))
         {
            x = x - centreX;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = 90 - fRotationAngle;
         }
         else if ((x < centreX) && (y <= centreY))
         {
            fRotationAngle = minAngle;
            _bUpdateValid = false;
         }
         else if ((x < centreX) && (y > centreY))
         {
            x = centreX - x;
            y = (centreY - y);
            fRotationAngle = atan2f(y , x) * 180 / (float)PI;
            fRotationAngle = -(90 - fRotationAngle);
         }
      }
      break;
      default:
      {
      }
      break;
   }
   return fRotationAngle;
}


/****************************************************************************
*   Function    :
*   Description :
*   Parameters  :
*   Return      :
****************************************************************************/
void AngularSliderWidget2D::OnAnimationTimeDispatcherChanged()
{
   if (_animationHelperInterface != 0)
   {
      _animationHelperInterface->setTimeDispatcher(GetAnimationTimeDispatcher());
   }
}


void AngularSliderWidget2D::OnParentViewActivate(bool enable)
{
   if (!enable)
   {
      stopExistingAnimations();
   }
   Base::OnParentViewActivate(enable);
}


void AngularSliderWidget2D::stopExistingAnimations()
{
   if (_animationHelperInterface != 0)
   {
      _animationHelperInterface->stopAllExistingAnimations();
   }
}
