/* ***************************************************************************************
* FILE:          ScrollBarWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScrollBarWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/List/Layouting/FlexListLayouter.h"
#include "ScrollBarWidget2D.h"
#include "Courier/Visualization/ViewScene2D.h"
#include "CanderaPlatform/OS/TimePlatform.h"
#include "Widgets/2D/ControlTemplate/ControlTemplate.h"
#include "Widgets/2D/List/Layouting/CommonAncestorLayouter.h"
#include "Widgets/2D/List/generated/ListEnums.h"
#include <Widgets/2D/List/ListWidget2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_SCROLLBAR
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ScrollBarWidget2D.cpp.trc.h"
#endif


CGI_WIDGET_RTTI_DEFINITION(ScrollBarWidget2D);

using namespace Candera;
using namespace Courier;

static bool IsInCompositeSceneMode(const Candera::Node2D* node)
{
   FEATSTD_UNUSED(node);
   _TODO("Check Composite2D in CGI3")
   //return ((0 != node) && (0 != Candera::Dynamic_Cast<Composite2D*>(node->GetScene())));
   return false;
}


static void SetCellSizeForHorizontalScroll(Candera::GridLayouter* grid, UInt8 column, Float size)
{
   grid->SetColumnWidth(column, size);
}


static void SetCellSizeForVerticalScroll(Candera::GridLayouter* grid, UInt8 column, Float size)
{
   grid->SetRowHeight(column, size);
}


static Candera::Float GetRelativePositionForHorizontalScroll(const TouchMsg* touchMsg, Candera::Float topPercentage, Candera::Float sliderPercentage,
      Candera::Rectangle slideRefScrollBarBoundingRectangle, const Courier::TouchInfo& slideRefTouchInfo)
{
   if ((slideRefScrollBarBoundingRectangle.GetWidth() == 0) || (sliderPercentage == 1.0F))
   {
      return 0.0F;
   }
   Float diff = Float(touchMsg->GetXPos()) - Float(slideRefTouchInfo.mX);

   Float diffPercentage = diff / slideRefScrollBarBoundingRectangle.GetWidth();

   Float relativePosition = (topPercentage + diffPercentage) / (1.0F - sliderPercentage);

   if (relativePosition < 0.0F)
   {
      relativePosition = 0.0F;
   }

   if (relativePosition > 1.0F)
   {
      relativePosition = 1.0F;
   }

   return relativePosition;
}


static Candera::Float GetRelativePositionForVerticalScroll(const TouchMsg* touchMsg, Candera::Float topPercentage, Candera::Float sliderPercentage,
      Candera::Rectangle slideRefScrollBarBoundingRectangle, const Courier::TouchInfo& slideRefTouchInfo)
{
   if ((slideRefScrollBarBoundingRectangle.GetHeight() == 0) || (sliderPercentage == 1.0F))
   {
      return 0.0F;
   }
   Float diff = Float(touchMsg->GetYPos()) - Float(slideRefTouchInfo.mY);

   Float diffPercentage = diff / slideRefScrollBarBoundingRectangle.GetHeight();

   Float relativePosition = (topPercentage + diffPercentage) / (1.0F - sliderPercentage);

   if (relativePosition < 0.0F)
   {
      relativePosition = 0.0F;
   }

   if (relativePosition > 1.0F)
   {
      relativePosition = 1.0F;
   }

   return relativePosition;
}


ScrollBarWidget2D::ScrollBarWidget2D() :
   _invalid(true),
   _lastVisible(false),
   _lastPosition(0),
   _lastMaxPosition(0),
   _flexScrollable(0),
   _state(NormalState),
   _slideRefTouchInfo(0, 0, 0, 0, 0),
   _timeStamp(0),
   _modeSpecificParameters(0, 0),
   _sliderSizePercentage(0.0F),
   _pageBackwardSizePercentage(0.0F),
   _pageBackwardSizePercentageOnDown(0.0F),
   _scrollableNode(0),
   _lockOutType(LockOutNone),
   _overscrollPercentBefore(0.0F),
   _overscrollPercentAfter(0.0F),
   _overscroll(false)
{
   SetRepeatMessage(0);
   SetSliderSize(1.0F / 3.0F);
   SetTouchable(true);
}


ScrollBarWidget2D::~ScrollBarWidget2D()
{
   _flexScrollable = 0;
   _scrollableNode = 0;
}


void ScrollBarWidget2D::OnParentViewLoad(bool activate)
{
   if (activate)
   {
      RefreshScrollable();
   }
}


void ScrollBarWidget2D::RequestNewPosition(ListChangeType listChangeType, Candera::Int32 value)
{
   if (0 != _flexScrollable)
   {
      _flexScrollable->RequestImmediatePositioning(false);
   }

   if ((0 != _flexScrollable) && (0 != _flexScrollable->GetWidget()))
   {
      ListChangeMsg listChangeMsg(0, listChangeType, value, ListChangeMsgSourceScrollBar);
      static_cast<void>(_flexScrollable->GetWidget()->OnMessage(listChangeMsg));
   }
   else
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ScrollBarWidget2D::RequestNewPosition scrollable is null"));
   }

   _timeStamp = Candera::TimePlatform::GetMilliSeconds();
}


bool ScrollBarWidget2D::IsNodeTouched(const Candera::Node2D* node, const TouchMsg& touchMsg)
{
   if (0 != node)
   {
      ViewScene2D* view2D = static_cast<ViewScene2D*>(GetParentView());
      const ViewScene2D::CameraPtrVector& cameras = view2D->GetCameraPtrVector();
      for (Int cameraIndex = 0; cameraIndex < static_cast<int>(cameras.Size()); ++cameraIndex)
      {
         Candera::Camera2D* camera2D = cameras[cameraIndex];
         if (camera2D->IsRenderingEnabled())
         {
            const Candera::Vector2 point(static_cast<Courier::Float>(touchMsg.GetXPos()), static_cast<Courier::Float>(touchMsg.GetYPos()));
            if (IsPickIntersectingNode(*camera2D, node, point))
            {
               return true;
            }
         }
      }
   }
   return false;
}


bool ScrollBarWidget2D::OnTouch(const Candera::Camera2D& camera, const Candera::Vector2& point)
{
   return IsEnabled() && (0 != GetNode()) && GetNode()->IsEffectiveRenderingEnabled() && Base::OnTouch(camera, point);
}


bool ScrollBarWidget2D::OnMessage(const Message& msg)
{
   if (msg.GetId() == hmibase::input::TouchSessionStartEvent::ID || msg.GetId() == hmibase::input::TouchSessionStopEvent::ID)
   {
      //printf("Touch EV Start,Stop ::FlexScrollBar2D:: %08x [%s]\n",this,msg.GetId() == TouchHandling::TouchSessionStartEvent::ID ? "Start":"Stop");
   }

   if (Base::OnMessage(msg) == true)
   {
      return true;
   }

   switch (msg.GetId())
   {
      case LostFocusEvent::ID:
         _state = NormalState;
         break;

      case TouchMsg::ID:
      {
         const TouchMsg* touchMsg = message_cast<const TouchMsg*>(&msg);
         return OnTouchMsg(*touchMsg);
      }

      case LockOutMsg::ID:
      {
         const LockOutMsg* touchMsg = message_cast<const LockOutMsg*>(&msg);
         return OnLockOutMsg(touchMsg);
      }

      default:
         break;
   }
   return false;
}


bool ScrollBarWidget2D::OnTouchMsg(const TouchMsg& touchMsg)
{
   bool consumed(false);

   if (touchMsg.GetState() == TouchMsgState::Up)
   {
      consumed = (_state != NormalState);
      _state = NormalState;
   }

   consumed = consumed || ProcessTouchOnSliderNode(touchMsg);
   consumed = consumed || ProcessTouchOnNode(touchMsg, GetBackwardNode(), StepBackwardState, ListChangeUp, 1);
   consumed = consumed || ProcessTouchOnNode(touchMsg, GetForwardNode(), StepForwardState, ListChangeDown, 1);
   consumed = consumed || ProcessTouchOnNode(touchMsg, GetPageBackwardNode(), PageBackwardState, ListChangePageUp, 1);
   consumed = consumed || ProcessTouchOnNode(touchMsg, GetPageForwardNode(), PageForwardState, ListChangePageDown, 1);

   consumed = consumed || (GetAdditionalPageButtons() && ProcessTouchOnNode(touchMsg, GetAdditionalPageBackwardNode(), PageBackwardState, ListChangePageUp, 1));
   consumed = consumed || (GetAdditionalPageButtons() && ProcessTouchOnNode(touchMsg, GetAdditionalPageForwardNode(), PageForwardState, ListChangePageDown, 1));

   return consumed;
}


bool ScrollBarWidget2D::ProcessTouchOnSliderNode(const TouchMsg& touchMsg)
{
   bool processed(false);

   if ((_state == SlideState) || IsNodeTouched(GetSliderNode(), touchMsg))
   {
      if (touchMsg.GetState() == TouchMsgState::Down)
      {
         _state = SlideState;

         _slideRefTouchInfo.mX = touchMsg.GetXPos();
         _slideRefTouchInfo.mY = touchMsg.GetYPos();
         _pageBackwardSizePercentageOnDown = _pageBackwardSizePercentage;
         if ((0 != GetSliderNode()) && (0 != GetSliderNode()->GetParent()))
         {
            GetSliderNode()->GetParent()->GetWorldAxisAlignedBoundingRectangle(_slideRefScrollBarBoundingRectangle);
         }
      }
      else
      {
         RefreshScrollable();
         if (0 != _flexScrollable)
         {
            Float relativePosition = 0.0F;
            if (0 != _modeSpecificParameters._relativePositionGetter)
            {
               relativePosition = _modeSpecificParameters._relativePositionGetter(&touchMsg, _pageBackwardSizePercentageOnDown, _sliderSizePercentage, _slideRefScrollBarBoundingRectangle, _slideRefTouchInfo);
            }
            Candera::UInt32 maxPosition = _flexScrollable->GetMaxPosition();
            Int32 value = Int32(relativePosition * maxPosition);

            if (GetKnobDragMode() == PointerPosition)
            {
               PositionKnob(maxPosition, static_cast<unsigned int>(value));
            }

            RequestNewPosition(ListChangeSet, value);
         }
      }

      processed = true;
   }

   return processed;
}


bool ScrollBarWidget2D::ProcessTouchOnNode(const TouchMsg& touchMsg, Candera::Node2D* node, ScrollBarWidget2D::ScrollBarState newState, ListChangeType listChangeType, Candera::Int32 value)
{
   bool processed(false);
   if (IsNodeTouched(node, touchMsg))
   {
      if (touchMsg.GetState() == TouchMsgState::Down)
      {
         _state = NormalState;
         if (GetRepeatMessage() > 0)
         {
            _state = newState;
         }
         RequestNewPosition(listChangeType, value);
      }

      processed = true;
   }

   return processed;
}


bool ScrollBarWidget2D::OnLockOutMsg(const LockOutMsg* lockOutMsg)
{
   if (0 != lockOutMsg)
   {
      ListWidget2D* listWidget = (_flexScrollable != NULL) ? Candera::Dynamic_Cast<ListWidget2D*>(_flexScrollable->GetWidget()) : NULL;
      FeatStd::UInt32 listId = (listWidget != NULL) ? listWidget->GetListId() : 0;

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "OnLockOutMsg lockOut=%u receiverId=%u listId=%u [%p.%50s.%s]",
                          lockOutMsg->GetLockOut(), lockOutMsg->GetReceiverId(), listId, this, GetViewName(), GetLegacyName()));

      if ((lockOutMsg->GetReceiverId() == 0) || (lockOutMsg->GetReceiverId() == listId))
      {
         _lockOutType = lockOutMsg->GetLockOut();
         UpdateEnabledStatus();
      }
   }

   return false;
}


void ScrollBarWidget2D::OnChanged(Courier::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   RefreshScrollable();
   _invalid = true;
}


bool ScrollBarWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ScrollBarWidget2D* original = CLONEABLE_WIDGET_CAST<const ScrollBarWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      SetScrollableNode(controlTemplateMap.ResolveNodeClone(original->GetScrollableNode()));
      SetBackwardNode(controlTemplateMap.ResolveNodeClone(original->GetBackwardNode()));
      SetPageBackwardNode(controlTemplateMap.ResolveNodeClone(original->GetPageBackwardNode()));
      SetSliderNode(controlTemplateMap.ResolveNodeClone(original->GetSliderNode()));
      SetPageForwardNode(controlTemplateMap.ResolveNodeClone(original->GetPageForwardNode()));
      SetForwardNode(controlTemplateMap.ResolveNodeClone(original->GetForwardNode()));
      SetRepeatMessage(original->GetRepeatMessage());
      SetSliderSize(original->GetSliderSize());

      cloned = true;
   }
   return cloned;
}


void ScrollBarWidget2D::Update()
{
   Base::Update();

   if ((0 != GetNode()) && (0 != GetNode()->GetParent()) && GetNode()->GetParent()->IsEffectiveRenderingEnabled())
   {
      UpdateOnRepeatedMsg();
      RefreshScrollable();

      if (_invalid)
      {
         UpdateKnob();
         UpdateEnabledStatus();
         _invalid = false;
         Invalidate();
      }
   }

   _overscroll = false;
}


void ScrollBarWidget2D::UpdateOnRepeatedMsg()
{
   if ((_state != NormalState) && (GetRepeatMessage() > 0))
   {
      const UInt32 timeStamp = Candera::TimePlatform::GetMilliSeconds();

      if (_timeStamp < timeStamp)
      {
         if (timeStamp - _timeStamp > GetRepeatMessage())
         {
            switch (_state)
            {
               case StepBackwardState:
                  RequestNewPosition(ListChangeUp, 1);
                  break;
               case StepForwardState:
                  RequestNewPosition(ListChangeDown, 1);
                  break;
               case PageBackwardState:
                  RequestNewPosition(ListChangePageUp, 1);
                  break;
               case PageForwardState:
                  RequestNewPosition(ListChangePageDown, 1);
                  break;
               case SlideState:
               case NormalState:
               default:
                  break;
            }
         }
      }
      else
      {
         // handle timer overflow => next repeat will be correct
         _timeStamp = timeStamp;
      }
   }
}


void ScrollBarWidget2D::UpdateKnob()
{
   RefreshScrollable();
   if (0 != _flexScrollable)
   {
      const bool scrollBarVisible = _flexScrollable->IsScrollBarVisible();
      const Candera::UInt32 position = _flexScrollable->GetFirstFullyVisiblePosition();
      Candera::UInt32 maxPosition = _flexScrollable->GetMaxPosition();

      _invalid = (_lastVisible != scrollBarVisible) ||
                 (_lastPosition != position) ||
                 (_lastMaxPosition != maxPosition) ||
                 _overscroll;

      const KnobDragModeType knobDragMode = GetKnobDragMode();

      bool updateKnobPosition = (knobDragMode == ListPosition) ||
                                ((knobDragMode == PointerPosition) && (_state == NormalState));

      UpdateVisibility(scrollBarVisible);

      if (_invalid && updateKnobPosition)
      {
         PositionKnob(maxPosition, position);
      }
   }
}


void ScrollBarWidget2D::OnPositionReached()
{
}


void ScrollBarWidget2D::OnContentUpdated()
{
   _invalid = true;
}


void ScrollBarWidget2D::OnPreOverscroll(FeatStd::Float percent)
{
   _overscroll = true;
   _invalid = true;
   _overscrollPercentBefore = percent;
   _overscrollPercentAfter = 0.0F;
}


void ScrollBarWidget2D::OnPostOverscroll(FeatStd::Float percent)
{
   _overscroll = true;
   _invalid = true;
   _overscrollPercentBefore = 0.0F;
   _overscrollPercentAfter = percent;
}


void ScrollBarWidget2D::UpdateEnabledStatus()
{
   View* parentView = GetParentView();
   if ((0 != _flexScrollable) && (0 != parentView) && parentView->Is2D())
   {
      ViewScene2D* viewScene2D = parentView->ToViewScene2D();

      if (0 != viewScene2D)
      {
         bool backwardLockoutEnabled((LockOutNone == _lockOutType) || (LockOutDirNext == _lockOutType));
         bool pageBackwardLockoutEnabled((LockOutNone == _lockOutType) || (LockOutPageWise == _lockOutType) || (LockOutPageWisePrev == _lockOutType));
         bool forwardLockoutEnabled((LockOutNone == _lockOutType) || (LockOutDirPrev == _lockOutType));
         bool pageForwardLockoutEnabled((LockOutNone == _lockOutType) || (LockOutPageWise == _lockOutType) || (LockOutPageWiseNext == _lockOutType));

         bool backwardEnabled(_flexScrollable->IsCircular() || (_flexScrollable->GetFirstFullyVisiblePosition() > 0));
         bool pageBackwardEnabled(backwardEnabled);
         bool forwardEnabled(_flexScrollable->IsCircular() || (_flexScrollable->GetFirstFullyVisiblePosition() < _flexScrollable->GetMaxPosition()));
         bool pageForwardEnabled(forwardEnabled);

         backwardEnabled = backwardEnabled && backwardLockoutEnabled;
         pageBackwardEnabled = pageBackwardEnabled && pageBackwardLockoutEnabled;
         forwardEnabled = forwardEnabled && forwardLockoutEnabled;
         pageForwardEnabled = pageForwardEnabled && pageForwardLockoutEnabled;

         UpdateEnabledStatus(*viewScene2D, GetBackwardNode(), backwardEnabled);
         UpdateEnabledStatus(*viewScene2D, GetPageBackwardNode(), pageBackwardEnabled);
         UpdateEnabledStatus(*viewScene2D, GetForwardNode(), forwardEnabled);
         UpdateEnabledStatus(*viewScene2D, GetPageForwardNode(), pageForwardEnabled);

         UpdateEnabledStatus(*viewScene2D, GetAdditionalPageBackwardNode(), pageBackwardEnabled);
         UpdateEnabledStatus(*viewScene2D, GetAdditionalPageForwardNode(), pageForwardEnabled);
      }
   }
}


class SubtreeWidgetEnumerator : public TEnumerator<BaseWidget2D*>::Enumerator
{
   public:
      SubtreeWidgetEnumerator(ViewScene* viewScene, Node2D* node) :
         _currentIndex(0),
         _viewScene(viewScene),
         _node(node),
         _currentNode(node)
      {
      }

      virtual ~SubtreeWidgetEnumerator()
      {
      }

      virtual bool MoveNext()
      {
         if (0 != _viewScene)
         {
            do
            {
               ++_currentIndex;
               if (_currentIndex < static_cast<FeatStd::Int>(_viewScene->GetFrameworkWidgetPtrVector().Size()))
               {
                  Candera::WidgetBase* widgetBase = static_cast<Candera::WidgetBase*>(_viewScene->GetFrameworkWidgetPtrVector()[_currentIndex]);
                  BaseWidget2D* baseWidget2D = Candera::Dynamic_Cast<BaseWidget2D*>(widgetBase);
                  if (0 != baseWidget2D)
                  {
                     Node2D* widgetNode = baseWidget2D->GetNode();
                     while (0 != widgetNode)
                     {
                        if (widgetNode == _node)
                        {
                           _current = baseWidget2D;
                           return true;
                        }
                        widgetNode = widgetNode->GetParent();
                     }
                  }
               }
               else
               {
                  _viewScene = 0;
               }
            }
            while (0 != _viewScene);
         }
         while (_enumerator.MoveNext())
         {
            BaseWidget2D* baseWidget2D = Candera::Dynamic_Cast<BaseWidget2D*>(_enumerator.Current());
            if (0 != baseWidget2D)
            {
               _current = baseWidget2D;
               return true;
            }
         }
         while (0 != _currentNode)
         {
            ControlTemplateData* controlTemplateData = 0;
            do
            {
               controlTemplateData = ControlTemplate::TryGetControlTemplateData(*_currentNode);
               if (0 != controlTemplateData)
               {
                  _enumerator = controlTemplateData->EnumerateAssociatedWidgets();
                  if (_enumerator.MoveNext())
                  {
                     BaseWidget2D* baseWidget2D = Candera::Dynamic_Cast<BaseWidget2D*>(_enumerator.Current());
                     if (0 != baseWidget2D)
                     {
                        _current = baseWidget2D;
                        return true;
                     }
                  }
               }
               if (0 == controlTemplateData)
               {
                  if (0 != _currentNode->GetFirstChild())
                  {
                     _currentNode = _currentNode->GetFirstChild();
                  }
                  else if (0 != _currentNode->GetNextSibling())
                  {
                     _currentNode = _currentNode->GetNextSibling();
                  }
                  else
                  {
                     bool moveFinished = false;
                     while ((0 != _currentNode) && (!moveFinished))
                     {
                        _currentNode = _currentNode->GetParent();
                        if (_currentNode == _node)
                        {
                           _currentNode = 0;
                        }
                        else if ((0 != _currentNode) && (0 != _currentNode->GetNextSibling()))
                        {
                           _currentNode = _currentNode->GetNextSibling();
                           moveFinished = true;
                        }
                     }
                  }
               }
            }
            while ((0 != controlTemplateData) && (0 != _currentNode));
         }
         return false;
      }

   private:
      WidgetBaseEnumerator _enumerator;
      FeatStd::Int _currentIndex;
      ViewScene* _viewScene;
      Node2D* _currentNode;
      Node2D* _node;
};


void ScrollBarWidget2D::UpdateEnabledStatus(Courier::ViewScene2D& viewScene, Candera::Node2D* node, bool enabled)
{
   if (0 != node)
   {
      SubtreeWidgetEnumerator enumerator(&viewScene, node);
      while (enumerator.MoveNext())
      {
         BaseWidget2D* w = enumerator.Current();
         if ((0 != w) && (w->IsEnabled() != enabled))
         {
            w->SetEnabled(enabled);
         }
      }
   }
}


static FeatStd::UInt32 GetNodeDepth(Node2D* node)
{
   FeatStd::UInt32 depth = 0;
   while (0 != node)
   {
      node = node->GetParent();
      ++depth;
   }
   return depth;
}


void ScrollBarWidget2D::SetLayouterToCommonAncestor()
{
   if (0 != _flexScrollable)
   {
      Candera::Widget2D* scrollableWidget = _flexScrollable->GetWidget();
      if (0 != scrollableWidget)
      {
         Node2D* scrollableNode = scrollableWidget->GetNode();
         Node2D* node = GetNode();
         // first determine the depth of both nodes
         FeatStd::UInt32 scrollableNodeDepth = GetNodeDepth(scrollableNode);
         FeatStd::UInt32 nodeDepth = GetNodeDepth(node);
         Node2D* scrollableNodeParent = scrollableNode->GetParent();
         Node2D* nodeParent = node->GetParent();
         // both nodes can only share a common ancestor if they are on the same depth level. otherwise a search is impossible
         // therefore, we iterate to the parent that has the same depth level
         while (scrollableNodeDepth > nodeDepth)
         {
            scrollableNodeParent = scrollableNodeParent->GetParent();
            --scrollableNodeDepth;
         }
         while (nodeDepth > scrollableNodeDepth)
         {
            nodeParent = nodeParent->GetParent();
            --nodeDepth;
         }
         // now we can simply compare the parents to detect the common ancestor
         while ((0 != nodeParent) && (0 != scrollableNodeParent) && (nodeParent != scrollableNodeParent))
         {
            scrollableNodeParent = scrollableNodeParent->GetParent();
            nodeParent = nodeParent->GetParent();
         }
         Node2D* commonAncestor = 0;
         // and we found the common ancestor
         if ((0 != nodeParent) && (nodeParent != scrollableNodeParent))
         {
            commonAncestor = nodeParent;
         }
         if (0 != commonAncestor)
         {
            Layouter* originalLayouter = commonAncestor->GetLayouter();

            CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, "Lifetime ownership of layouter is taken by the commonAncestor node.");
            CommonAncestorLayouter* layouter = FEATSTD_NEW(CommonAncestorLayouter)(originalLayouter);
            if (0 != layouter)
            {
               commonAncestor->SetLayouter(layouter);
            }
         }
         /*
                  if (_scrollableNode != scrollableNode)
                  {
                     _scrollableNode = scrollableNode;

                     Node2D* commonAncestor = 0;
                     while ((scrollableNode != 0) && (0 == commonAncestor))
                     {
                        while ((node != 0) && (0 == commonAncestor))
                        {
                           if (node == scrollableNode)
                           {
                              commonAncestor = node;
                           }

                           node = node->GetParent();
                        }

                        scrollableNode = scrollableNode->GetParent();

                        if (0 != commonAncestor)
                        {
                           Layouter* originalLayouter = commonAncestor->GetLayouter();

                           CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(429, "Lifetime ownership of layouter is taken by the commonAncestor node.");
                           CommonAncestorLayouter* layouter = FEATSTD_NEW(CommonAncestorLayouter)(originalLayouter);
                           if (0 != layouter)
                           {
                              commonAncestor->SetLayouter(layouter);
                           }
                        }
                     }
                  }
         */
      }
   }
}


void ScrollBarWidget2D::PositionKnob(Candera::UInt32 maxPosition, const Candera::UInt32 position)
{
   const Float p(GetOverscrollSizeLimitted());
   UpdateSliderSizePercentage();

   Node2D* sliderNode = GetSliderNode();
   Node2D* pageBackwardNode = GetPageBackwardNode();
   Node2D* pageForwardNode = GetPageForwardNode();
   if ((0 != pageBackwardNode) && (0 != pageForwardNode) && (0 != sliderNode) && !IsInCompositeSceneMode(sliderNode))
   {
      _pageBackwardSizePercentage = 0.5f;
      if ((maxPosition == 0) || (maxPosition < position))
      {
         maxPosition = position;
      }
      const Float x((maxPosition != 0) ? (Candera::Float(position) / Candera::Float(maxPosition)) : 0.0F);
      const Float prevOverscroll(p - p * _overscrollPercentBefore);
      const Float postOverscroll(p - p * _overscrollPercentAfter);
      _pageBackwardSizePercentage = x * (1.0F - prevOverscroll - postOverscroll);
      Float pageForwardSizePercentage = 1.0F - postOverscroll - prevOverscroll - _pageBackwardSizePercentage;

      if (_sliderSizePercentage < 0.0F)
      {
         _sliderSizePercentage = 0.0F;
      }
      if (_sliderSizePercentage < 1.0F)
      {
         _pageBackwardSizePercentage -= _sliderSizePercentage * _pageBackwardSizePercentage;
         pageForwardSizePercentage -= _sliderSizePercentage * pageForwardSizePercentage;
      }
      if (_pageBackwardSizePercentage < 0.0F)
      {
         pageForwardSizePercentage += _pageBackwardSizePercentage;
         _pageBackwardSizePercentage = 0.0000001F;
      }
      if (pageForwardSizePercentage < 0.0F)
      {
         _pageBackwardSizePercentage += pageForwardSizePercentage;
         pageForwardSizePercentage = 0.0000001F;
      }
      if (pageForwardSizePercentage <= 0.0F)
      {
         pageForwardSizePercentage = 0.0000001F;
      }
      if (_pageBackwardSizePercentage <= 0.0F)
      {
         _pageBackwardSizePercentage = 0.0000001F;
      }

      _pageBackwardSizePercentage += prevOverscroll;
      pageForwardSizePercentage += postOverscroll;

      pageBackwardNode->SetRenderingEnabled(_pageBackwardSizePercentage > 0.0000001F);
      pageForwardNode->SetRenderingEnabled(pageForwardSizePercentage > 0.0000001F);

      Node2D* gridNode = sliderNode->GetParent();
      if ((0 != gridNode)/* && (gridNode == pageBackwardNode->GetParent()) && (gridNode == pageForwardNode->GetParent())*/)
      {
         GridLayouter* grid = dynamic_cast<GridLayouter*>(gridNode->GetLayouter());
         if (0 != grid)
         {
            if (0 != _modeSpecificParameters._cellSizeSetter)
            {
               _modeSpecificParameters._cellSizeSetter(grid, 0, _pageBackwardSizePercentage);
               _modeSpecificParameters._cellSizeSetter(grid, 1, _sliderSizePercentage);
               _modeSpecificParameters._cellSizeSetter(grid, 2, pageForwardSizePercentage);
            }
            InvalidateLayout();
            Invalidate();
            _invalid = false;

            _lastPosition = position;
            _lastMaxPosition = maxPosition;
         }
      }
   }
}


void ScrollBarWidget2D::UpdateSliderSizePercentage()
{
   if (false == GetDynamicSliderSize() || 0 == _flexScrollable)
   {
      _sliderSizePercentage = GetSliderSize();
   }
   else
   {
      // calculation to reflect the percentage of visible items in the scrollbar knob
      if (_flexScrollable->GetCompleteSize() > 0)
      {
         _sliderSizePercentage = static_cast<Candera::Float>(_flexScrollable->GetVisibleSize());
         _sliderSizePercentage /= _flexScrollable->GetCompleteSize();
         if (_sliderSizePercentage < GetSliderSize())
         {
            _sliderSizePercentage = GetSliderSize();
         }
         if (_sliderSizePercentage < 0.05f)
         {
            _sliderSizePercentage = 0.05f;
         }
         else if (_sliderSizePercentage > 1.0f)
         {
            _sliderSizePercentage = 1.0f;
         }
      }
      else
      {
         _sliderSizePercentage = 1.0f;
      }
   }
}


void ScrollBarWidget2D::UpdateVisibility(const bool scrollBarVisible)
{
   Candera::Node2D* node = GetNode();
   if (0 != node)
   {
      node->SetRenderingEnabled(scrollBarVisible);
      if (scrollBarVisible)
      {
         Layouter::SetSize(*node, -1.0F, -1.0F);
      }
      else
      {
         Layouter::SetSize(*node, 0.0F, 0.0F);
      }

      _lastVisible = scrollBarVisible;
   }
}


void ScrollBarWidget2D::RefreshScrollable()
{
   _flexScrollable = 0;
   if ((0 != GetNode()) && (0 != GetNode()->GetParent()))
   {
      Candera::Node2D* scrollableNode = GetScrollableNode();
      if (0 != scrollableNode)
      {
         ViewScene* viewScene = static_cast<ViewScene*>(GetParentView());
         if (0 != viewScene)
         {
            WidgetBaseEnumerator enumerator = ControlTemplate::EnumerateAssociatedWidgets(*viewScene, *scrollableNode);
            while ((0 == _flexScrollable) && enumerator.MoveNext())
            {
               FEATSTD_LINT_NEXT_EXPRESSION(740, "Intended downcast.")
               _flexScrollable = dynamic_cast<FlexScrollable*>(enumerator.Current());
            }
            if (0 != _flexScrollable)
            {
               _flexScrollable->SetListener(this);

               switch (_flexScrollable->GetScrollingOrientation())
               {
                  case Horizontal:
                     _modeSpecificParameters._cellSizeSetter = SetCellSizeForHorizontalScroll;
                     _modeSpecificParameters._relativePositionGetter = GetRelativePositionForHorizontalScroll;
                     break;

                  case Vertical:
                     _modeSpecificParameters._cellSizeSetter = SetCellSizeForVerticalScroll;
                     _modeSpecificParameters._relativePositionGetter = GetRelativePositionForVerticalScroll;
                     break;

                  default:
                     _modeSpecificParameters._cellSizeSetter = SetCellSizeForVerticalScroll;
                     _modeSpecificParameters._relativePositionGetter = GetRelativePositionForVerticalScroll;
               }

               SetLayouterToCommonAncestor();
            }
         }
      }
   }
}


FeatStd::Float ScrollBarWidget2D::GetOverscrollSizeLimitted() const
{
   Float p(GetOverscrollSize());
   if ((p < 0.0F) || (p > 1.0F))
   {
      p = 0.0F;
   }

   return p;
}
