/* ***************************************************************************************
* FILE:          RichTextBaseWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichTextBaseWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h"
#include "RichTextBaseWidget2D.h"
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>

CGI_WIDGET_RTTI_DEFINITION(RichTextBaseWidget2D);

using namespace Candera;


/****************************************************************************
*    Function    : RichTextBaseWidget2D
*    Description : Constructor
*    Parameters  : void
****************************************************************************/
RichTextBaseWidget2D::RichTextBaseWidget2D() :
   Base(),
   m_bIsRelayout(false),
   m_bIsRerender(false),
   m_bIsNodeUpdate(false),
   m_bIsNodeChanged(false),
   m_pOriginalImage(0),
   m_pBitmapImage(BitmapImage2D::Create()),
   m_oRichText(),
   m_pDefaultStyleChunk(FEATSTD_NEW(RichText::StyleChunk)),
   m_pDefaultParagraphChunk(FEATSTD_NEW(RichText::ParagraphChunk)),
   m_oRenderOffset(0.0F, 0.0F),
   m_pTextArea(0)
{
   // Set non default initial values for properties
   SetNormalTextColor(Color(1.0F, 1.0F, 1.0F, 1.0F));
   SetAlphaValue(1.0F);
   // add chunks to richtext
   m_oRichText.vAddChunk(m_pDefaultParagraphChunk);
   m_oRichText.vAddChunk(m_pDefaultStyleChunk);
}


/****************************************************************************
*    Function    : ~RichTextBaseWidget2D
*    Description : Destructor
*    Parameters  : void
****************************************************************************/
RichTextBaseWidget2D::~RichTextBaseWidget2D()
{
   // set original bitmap image on assigned node
   RenderNode* pNode = Dynamic_Cast<RenderNode*>(GetNode());

   if (pNode != 0)
   {
      if (pNode->GetEffect(0) != 0)
      {
         BitmapBrush* pBrush = Dynamic_Cast<Candera::BitmapBrush*>(pNode->GetEffect(0)->GetBrushEffect2D());

         if (pBrush != 0)
         {
            pBrush->Image().Set(m_pOriginalImage);
            bool updated = pBrush->Update();
            FEATSTD_UNUSED(updated);
         }
      }
   }

   // release bitmap image of text
   m_pBitmapImage->Unload();
   m_pBitmapImage.Release();
   // Chunks will be freed by the destructor of m_oRichText
   FEATSTD_LINT_CURRENT_SCOPE(423, "Chunks will be freed by BaseClass::m_oRichText destructor")
   m_pDefaultStyleChunk = 0;
   m_pDefaultParagraphChunk = 0;
   // release textarea
   FEATSTD_DELETE(m_pTextArea);
   m_pTextArea = 0;
}


/****************************************************************************
*    Function    : Update
*    Description : Called on for each render cycle. Overridden from
*                  WidgetBase.
*    Parameters  : void
*    Return      : void
****************************************************************************/
void RichTextBaseWidget2D::Update()
{
   Base::Update();

   if (m_bIsNodeUpdate)
   {
      vUpdateNode();
   }

   if (bIsReLayoutNeeded())
   {
      vLayoutRichText();
   }

   if (bIsReRenderNeeded())
   {
      vRenderRichText();
   }
}


/****************************************************************************
*     Function    : OnChanged
*     Description : Called from the Update function for each property that
*                   has been changed. Overridden from RichTextBaseWidget2DBase.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::OnChanged(Candera::UInt32 propertyId)
{
   switch (propertyId)
   {
      //this widget has notifierPolicy=onUpdate which means that changing its specific animation related properties will trigger Update so OnChanged will be called from Update instead from the setter
      //for the other properties like Visible, Enabled => OnChanged is called from the setter
      case VisiblePropertyId:
      case EnabledPropertyId:
         Base::OnChanged(propertyId);
         break;

      case StylePropertyId:
      case TextPropertyId:
      case TextAreaSizePropertyId:
      case HorizontalAlignmentPropertyId:
      case VerticalAlignmentPropertyId:
         m_bIsRelayout = true;
         break;

      case NormalTextColorPropertyId:
         m_bIsRerender = true;
         break;

      case AlphaValuePropertyId:
         m_bIsNodeUpdate = true;
         break;

      default:
         break;
   }
   Invalidate();
}


/****************************************************************************
*     Function    : OnBeforeNodeChanged
*     Description : Called before the associated node is changed.
*                   Overridden from WidgetBase2D.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::OnBeforeNodeChanged()
{
   // restore original image on the node. Usually such operations should not
   // be performed outside of the update function but it can't be guaranteed that
   // the old node is still valid.
   RenderNode* pNode = Dynamic_Cast<RenderNode*>(GetNode());

   if (pNode != 0)
   {
      if (pNode->GetEffect(0) != 0)
      {
         BitmapBrush* pBrush = Dynamic_Cast<Candera::BitmapBrush*>(pNode->GetEffect(0)->GetBrushEffect2D());

         if (pBrush != 0)
         {
            pBrush->Image().Set(m_pOriginalImage);
            bool updated = pBrush->Update();
            FEATSTD_UNUSED(updated);
         }
      }
   }
}


/****************************************************************************
*     Function    : OnNodeChanged
*     Description : Called after the associated node is changed.
*                   Overridden from WidgetBase2D.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::OnNodeChanged()
{
   m_bIsNodeUpdate = true;
   m_bIsNodeChanged = true;
}


/****************************************************************************
*     Function    : vUpdateNode
*     Description : Update all node related properties here.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::vUpdateNode()
{
   if (m_bIsNodeChanged)
   {
      // set current text image on the node.
      RenderNode* pNode = Dynamic_Cast<RenderNode*>(GetNode());

      if (pNode != 0)
      {
         if (pNode->GetEffect(0) != 0)
         {
            BitmapBrush* pBrush = Dynamic_Cast<Candera::BitmapBrush*>(pNode->GetEffect(0)->GetBrushEffect2D());

            if (pBrush != 0)
            {
               m_pOriginalImage = Candera::MemoryManagement::SharedPointer<Image2D>(pBrush->Image().Get());
               pBrush->Image().Set(m_pBitmapImage);
               bool updated = pBrush->Update();
               FEATSTD_UNUSED(updated);
            }
         }
      }

      m_bIsNodeChanged = false;
   }

   Candera::Node2D* pNode = GetNode();

   if (pNode != 0)
   {
      pNode->SetAlphaValue(GetAlphaValue());
      pNode->SetRenderingEnabled(IsVisible());
   }

   m_bIsNodeUpdate = false;
}


/****************************************************************************
*     Function    : bIsReLayoutNeeded
*     Description : Checks if relayouting is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool RichTextBaseWidget2D::bIsReLayoutNeeded()
{
   return m_bIsRelayout;
}


/****************************************************************************
*     Function    : bIsReRenderNeeded
*     Description : Checks if rerendering is needed. Override when another
*                   task should influence the result
*     Parameters  : bool
*     Return      : void
****************************************************************************/
bool RichTextBaseWidget2D::bIsReRenderNeeded()
{
   return m_bIsRerender;
}


/****************************************************************************
*     Function    : CloneFrom
*     Description : required for FlexList cloning
****************************************************************************/
bool RichTextBaseWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const RichTextBaseWidget2D* original = CLONEABLE_WIDGET_CAST<const RichTextBaseWidget2D*>(originalWidget);
      if (original == NULL)
      {
         return false;
      }

      if (ControlTemplateBinding::IsTextBindable(*this))
      {
         SetText(ControlTemplateBinding::GetTextValue(*this));
      }
      else
      {
         SetText(original->GetText());
      }

      SetTextAreaSize(original->GetTextAreaSize());
      SetStyle(original->GetStyle());
      SetHorizontalAlignment(original->GetHorizontalAlignment());
      SetVerticalAlignment(original->GetVerticalAlignment());
      SetAlphaValue(original->GetAlphaValue());
      cloned = true;
   }
   return cloned;
}


/****************************************************************************
*     Function    : vLayoutRichText
*     Description : Will do the layouting of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after layouting.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::vLayoutRichText()
{
   // Set used text style
   if (m_pDefaultStyleChunk != 0)
   {
      m_pDefaultStyleChunk->vSetStyle(GetStyle());
   }

   // set text size
   TextAreaSizeType oTextAreaSize = GetTextAreaSize();
   m_oRichText.vSetSize(RichText::Size(Int32(oTextAreaSize.GetX() + 0.5F), Int32(oTextAreaSize.GetY() + 0.5F)));

   Candera::Globalization::TextDirection textDirection = Candera::Globalization::_undefined;
   if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().PointsToNull() == false)
   {
      //read the current culture text direction
      textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
   }
   // configure alignment
   if (m_pDefaultParagraphChunk != 0)
   {
      //Changes the text's horizontal alignment based on culture text direction
      if (textDirection == Candera::Globalization::RightToLeft)
      {
         m_pDefaultParagraphChunk->vSetHorizontalAlignment(RT_Trailing); //Example: Arabic language
      }
      else
      {
         m_pDefaultParagraphChunk->vSetHorizontalAlignment(GetHorizontalAlignment());
      }
      m_pDefaultParagraphChunk->vSetVerticalAlignment(GetVerticalAlignment());
   }

   // delete previous computed text area and calculate the new layout of the text
   FEATSTD_DELETE(m_pTextArea);
   RichText::RichTextRenderer::vLayout(&m_oRichText, &m_pTextArea);
   // mark
   m_bIsRelayout = false;
   m_bIsRerender = true;
}


/****************************************************************************
*     Function    : vRenderRichText
*     Description : Will do the rendering of the current rich text
*                   configuration. Override if a specific task needs only
*                   to be performed before or after rendering.
*     Parameters  : void
*     Return      : void
****************************************************************************/
void RichTextBaseWidget2D::vRenderRichText()
{
//        _TODO("vRenderRichText::bitmaps in rich text")

   Candera::MemoryManagement::SharedPointer<Candera::Bitmap>(pBitmap);
   pBitmap = m_pBitmapImage->GetMutableBitmap();

   // render text into the bitmap
   RichText::RichTextRenderer::vDraw(&m_oRichText,
                                     RichText::Position(Int32(m_oRenderOffset.GetX() + 0.5F), Int32(m_oRenderOffset.GetY() + 0.5F)),
                                     RichText::Size(Int32(GetTextAreaSize().GetX() + 0.5F), Int32(GetTextAreaSize().GetY() + 0.5F)),
                                     pBitmap);

   // set image
   if (m_pBitmapImage->GetBitmap() != pBitmap)
   {
      m_pBitmapImage->Unload(Candera::DeviceObject2D::Force);
      m_pBitmapImage->SetBitmap(pBitmap);
      m_pBitmapImage->Upload();
   }
   else
   {
      m_pBitmapImage->Update();
   }

   // invalidate scene
   Invalidate();
   m_bIsRerender = false;

   m_bIsNodeUpdate = true;
   m_bIsNodeChanged = true;
}
