/* ***************************************************************************************
* FILE:          RichText.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichText is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "Widgets/2D/BaseWidget2D.h"
#include "Widgets/2D/RichTextBase/RichText.h"
#include "Widgets/2D/RichTextBase/CombinedTextMeasureContext.h"
#include "Candera/System/MemoryManagement/CanderaHeap.h"
#include "Candera/EngineBase/Common/BitmapTextRenderContext.h"
#include "Candera/EngineBase/Common/Bitmap.h"
#include "FeatStd/Util/TextEncoding.h"
#include "hmibase/util/Ticker.h"

#define no_BOUNDING_RECTANGLE_WITH_TEXTRENDER
#define DEFAULT_TEXT_CHUNK_START_INDEX 1
#define INCLUDE_ZWSP 1
#define INCREMENT_BYTE_POSITION 3

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RichText.cpp.trc.h"
#endif
#define MAX_STEP_COUNT 4

#ifdef RICHTEXT_PERFORMANCE_CHECK
int rich_txt_perf_counter, rich_txt_perf_WrapText, rich_txt_perf_MeasureText;
#endif

namespace RichText {

using namespace Candera;
using namespace Candera::TextRendering;
using namespace HmiBase::TextRendering;

Candera::UInt32 RichTextChunk::m_u32renderdCharCount = 0;
Candera::UInt32 RichTextChunk::m_u32FirstLinerenderdCharCount = 0;
Candera::UInt32 RichTextChunk::m_u32LastLinerenderdCharCount = 0;
bool RichTextChunk::m_bisPageOrLineDown = true;
Candera::UInt16 RichTextChunk::m_u16NumOfLinesPerTextArea = 0;
bool RichTextChunk::_isLineUp = false;
/****************************************************************************
*     Utf8Encoding::u32Copy
****************************************************************************/
UInt32 Utf8Encoding::u32Copy(Char* pDst, UInt32 u32DstCharCount, const Char* pSrc)
{
   UInt32 u32SrcCharCount = static_cast<UInt32>(StringPlatform::Length(pSrc));

   if (u32SrcCharCount < u32DstCharCount)
   {
      StringPlatform::Copy(pDst, pSrc);
      return u32SrcCharCount;
   }
   else
   {
      return u32Copy(pDst, u32DstCharCount, pSrc, 0xFFFFFFFF);
   }
}


UInt32 Utf8Encoding::u32Copy(Char* pDst, UInt32 u32DstCharCount, const Char* pSrc, UInt32 u32CodePointCount)
{
   const Char* copSrc = pSrc;
   UInt32 u32Count = 0;

   for (;;)
   {
      if ((u32CodePointCount == 0) || (*copSrc == '\0'))
      {
         break;
      }
      UInt32 u32Length = u32CodeLength(copSrc);
      if ((u32Count + u32Length) >= u32DstCharCount)
      {
         break;
      }
      u32Count += u32Length;
      copSrc += u32Length;
      --u32CodePointCount;
   }
   if (u32Count > 0)
   {
      memcpy(pDst, pSrc, u32Count);
      pDst[u32Count] = '\0';
   }
   return u32Count;
}


/****************************************************************************
*     Utf8Encoding::u32CodeLength
****************************************************************************/
UInt32 Utf8Encoding::u32CodeLength(const Char* pCodePoint)
{
   UInt8 u8Length;
   if ((UInt8(*pCodePoint) & 0x80U) == 0U)
   {
      u8Length = 1;
   }
   else if ((UInt8(*pCodePoint) & 0xE0U) == 0xC0U)
   {
      u8Length = 2;
   }
   else if ((UInt8(*pCodePoint) & 0xF0U) == 0xE0U)
   {
      u8Length = 3;
   }
   else
   {
      u8Length = 4;
   }

   bool bValid = true;
   for (UInt8 u8Index = 1U; u8Index < u8Length; u8Index++)
   {
      if ((UInt8(pCodePoint[u8Index]) & 0xC0U) != 0x80U)
      {
         bValid = false;
         break;
      }
   }

   if (!bValid)
   {
      u8Length = 1;
   }
   return UInt32(u8Length);
}


/****************************************************************************
*     Utf8Encoding::u32Advance
****************************************************************************/
UInt32 Utf8Encoding::u32Advance(const Char*& pCodePoint)
{
   if (*pCodePoint == '\0')
   {
      return 0;
   }

   UInt32 n = u32CodeLength(pCodePoint);
   pCodePoint += n;
   return n;
}


/****************************************************************************
*     TextContext::TextContext
****************************************************************************/

Candera::BitmapTextRenderContext RichTextRenderer::TextContext::m_oTextRenderContext;

Candera::TextRendering::TextRenderer RichTextRenderer::TextContext::m_oTextRenderer;

RichTextRenderer::TextContext::TextContext() :
   m_oTextColor((Float)255, (Float)255, (Float)255, (Float)255),
   m_oStyle(0),
   m_pCurrentParagraph(0),
   m_pCurrentLine(0)
{
   m_stFontMetrics.ascender = 0;
   m_stFontMetrics.descender = 0;
   m_stFontMetrics.lineHeight = 0;
   m_stFontMetrics.maxAdvance = 0;
};


/****************************************************************************
*     RichTextRenderer::vLayout
****************************************************************************/
void RichTextRenderer::vLayout(RichText* pRichText, TextArea** ppTextArea)
{
   if (pRichText != 0)
   {
      // create context
      TextContext stContext;

      // create text area
      TextArea* pTextArea = FEATSTD_NEW(TextArea);

      if (pTextArea != 0)
      {
         // set text area size
         pTextArea->vSetSize(pRichText->cooGetSize());

         // loop through all chunks
         RichTextChunk* pChunk = pRichText->pGetFirstChunk();

         while (pChunk != 0)
         {
            // reset chunk position and size
            pChunk->vSetChunkPosition(Position(0, 0));
            pChunk->vSetChunkSize(Size(0, 0));

            // add chunk to text area
            pTextArea->vAddChunk(&stContext, pChunk);

            // get the next chunk
            pChunk = pChunk->pAdvance();
         }

         // end text area
         pTextArea->vEndTextArea(&stContext);

         // set effective size
         pRichText->vSetEffectiveSize(pTextArea->cooGetEffectiveSize());

         // return text area if requested else destroy it
         if (ppTextArea != 0)
         {
            *ppTextArea = pTextArea;
         }
         else
         {
            FEATSTD_DELETE(pTextArea);
         }
      }
   }
}


/****************************************************************************
*     RichTextRenderer::vDraw
****************************************************************************/

void RichTextRenderer::vDraw(RichText* pRichText, const Position& cooOffset, const Size& cooSize, Candera::Bitmap::SharedPointer& bitmap)
{
   int _rendrOnce = 0;
   bool isTextAreaSizeConfigured = true;
   // if size is set use it, else use the size of the text
   Size oRenderSize = pRichText->cooGetEffectiveSize();

   if (cooSize.GetWidth() > 0)
   {
      oRenderSize.SetWidth(cooSize.GetWidth());
   }

   if (cooSize.GetHeight() > 0)
   {
      oRenderSize.SetHeight(cooSize.GetHeight());
   }
   //added for textareawidget to make sure chunk size is used if textarea size is not configured
   if ((cooSize.GetWidth() == 0) && (cooSize.GetHeight() == 0))
   {
      isTextAreaSizeConfigured = false;
   }

   Candera::Bitmap::SharedPointer tempBitmap = bitmap;

   // Create bitmap if needed. This bitmap MUST be destroyed by the calling Widget/Application
   UInt32 u32TextSizeWidth = UInt32(Int32((oRenderSize.GetWidth())));
   UInt32 u32TextSizeHeight = UInt32(Int32((oRenderSize.GetHeight())));
   if ((tempBitmap == 0) || (tempBitmap->GetWidth() != u32TextSizeWidth) || (tempBitmap->GetHeight() != u32TextSizeHeight))
   {
      UInt8* pPixelData = FEATSTD_NEW_ARRAY(UInt8, u32TextSizeWidth * u32TextSizeHeight * 4);
      tempBitmap = Candera::Bitmap::Create(static_cast<Candera::UInt16>(u32TextSizeWidth), static_cast<Candera::UInt16>(u32TextSizeHeight), /*Bitmap::RgbaFormat,*/ Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, pPixelData, Candera::Bitmap::Disposer::Dispose, true);
   }

   if (tempBitmap.PointsToNull() == false)
   {
      const UInt32 bmpWidth = tempBitmap->GetWidth();
      const UInt32 bmpHeight = tempBitmap->GetHeight();
      // Clear bitmap
      Candera::Bitmap::PixelsResource pixelResource(tempBitmap->GetPixelsResourceHandle());
      memset(pixelResource.GetMutableData(), 0, bmpWidth * bmpHeight * 4);

      // Set bitmap in context
      TextContext oContext;
      TextContext::m_oTextRenderContext.SetBitmap(tempBitmap);

      // render all chunks
      RichTextChunk* pChunk = pRichText->pGetFirstChunk();
      //To get the lineheight which will be used to determine whether chunk fits inside bitmap area
      ParagraphChunk*  pParaChunk = dynamic_cast<ParagraphChunk*>(pChunk);
      Candera::Int32 lineHeight = 0;
      if (pParaChunk != NULL)
      {
         lineHeight = pParaChunk->i32GetLineHeight();
      }

      if (pChunk != 0)
      {
         pChunk->vSetRenderdCharCount(0);
      }
      Position oCursor(cooOffset.GetX(), cooOffset.GetY());
      LineList linelist;
      static Candera::UInt32 lineCharCount = 0;
      while (pChunk != 0)
      {
         /*change for pageup and extending it to LineUp*/
         if ((_rendrOnce == 0) && (pChunk->bGetIsPageOrLineDown() == false || pChunk->bGetIsLineUp() == true))
         {
            RichTextChunk*  pcolorChunk = dynamic_cast<ColorChunk*>(pChunk->pGetPreviousChunk(pChunk, pRichText));

            if (pcolorChunk != 0)
            {
               ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), " Inside VDraw= %d", pChunk->u16GetNumOfLinesPerTextArea()));
               //get chunk from which rendering needs to be started. It's decided based on  no of lines per textarea calculated
               while (_rendrOnce == 0)
               {
                  pChunk = pChunk->pAdvance();
                  if (pChunk != 0)
                  {
                     if (pChunk->bGetPageUpStartChunk() == true || pChunk->bGetLineUpStartChunk() == true)
                     {
                        _rendrOnce++;
                        if (Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection() == Candera::Globalization::RightToLeft)
                        {
                           oCursor.SetX(0);
                        }
                        //clear the flag for Lineup operation
                        if (pChunk->bGetIsLineUp() == true)
                        {
                           pChunk->vSetLineUp(false);
                        }
                     }
                  }
                  else
                  {
                     return;
                  }
               }
            }
         }

         if (pChunk != 0)
         {
            const Position& chunkPos = pChunk->cooGetChunkPosition();
            const Size& chunkSize = pChunk->cooGetChunkSize();

            oCursor.Translate(chunkPos.GetX(), chunkPos.GetY());

            Int32 cursorX = oCursor.GetX();
            Int32 cursorY = oCursor.GetY();

            if (!isTextAreaSizeConfigured)
            {
               lineHeight = chunkSize.GetHeight();
            }

            const bool isInsideBmpArea = (cursorX <= Int32(bmpWidth)) && (cursorX + chunkSize.GetWidth() >= 0) &&
                                         ((cursorY + lineHeight) <= Int32(bmpHeight)) && (cursorY + chunkSize.GetHeight() >= 0);

            TextChunk* pTextChunk = dynamic_cast<TextChunk*>(pChunk);
            const bool isTextChunk = (NULL != pTextChunk);

            if (!isTextChunk || isInsideBmpArea)
            {
               pChunk->vRender(&oContext, TextCoordinate(PixelPosition(cursorX), PixelPosition(cursorY)));
            }

            if (isInsideBmpArea)
            {
               vCalculateRenderdCharCnt(pChunk, linelist, lineCharCount);
            }

            oCursor.Translate(pChunk->cooGetChunkSize().GetWidth(), 0);

            pChunk = pChunk->pAdvance();
         }
      }

      RichTextChunk* pFirstChunk = pRichText->pGetFirstChunk();
      if (pFirstChunk && (!linelist.empty()))
      {
         pFirstChunk->vSetFirstLineRenderdCharCount(static_cast<Candera::UInt32>(linelist.front()));
         if (lineCharCount != 0)
         {
            linelist.push_back(lineCharCount);
            lineCharCount = 0;
         }
         pFirstChunk->vSetLastLineRenderdCharCount(static_cast<Candera::UInt32>(linelist.back()));
      }
   }

   bitmap = tempBitmap;
}


void RichTextRenderer::vCalculateRenderdCharCnt(RichTextChunk* pChunk, LineList& lineList, Candera::UInt32& lineCharCount)
{
   UInt32 CharCount = 0;
   TextChunk* pTextChunk = dynamic_cast<TextChunk*>(pChunk);
   TabChunk* pTabChunk = dynamic_cast<TabChunk*>(pChunk);
   ColorChunk* pColorChunk = dynamic_cast<ColorChunk*>(pChunk);
   NewLineChunk* pNewLineChunk = dynamic_cast<NewLineChunk*>(pChunk);

   if (pChunk != NULL)
   {
      if ((pTextChunk != NULL) && (pTextChunk->m_pWrapChild == NULL))
      {
         CharCount = static_cast<UInt32>(pTextChunk->cooGetText().GetCharCount());
         lineCharCount += CharCount;
      }

      else if ((pTabChunk != NULL) && (pTabChunk->u32GetTabCharCnt() > 0))
      {
         CharCount = pTabChunk->u32GetTabCharCnt();
         lineCharCount += CharCount;
      }
      else if ((pColorChunk != NULL) && (pColorChunk->u32GetColorCharCnt() > 0))
      {
         CharCount = pColorChunk->u32GetColorCharCnt();
         lineCharCount += CharCount;
      }
      else if (pNewLineChunk != NULL)
      {
         CharCount = pNewLineChunk->getNewLineCharCount();
         lineCharCount += CharCount;
         lineList.push_back(lineCharCount);
         lineCharCount = 0;
      }
      else
      {
         //nothing
      }

      pChunk->vSetRenderdCharCount(CharCount + pChunk->u32GetRenderdCharCount());
   }
}


/*RichTextChunk* RichTextRenderer::pGetStartChunkForListPageup(RichText* pRichText, Candera::UInt16 u16NumofLinesPerTextArea)
{
   Int16 _linecount = 0;
   RichTextChunk* temp = pRichText->pGetLastChunk();
   printf("nooflines______________%d", u16NumofLinesPerTextArea);
   while ((_linecount < u16NumofLinesPerTextArea) && (temp != NULL))
   {
      TextChunk* ptempChunk = NULL;
      TextChunk* ptempTextChunk = dynamic_cast<TextChunk*>(temp);
      NewLineChunk* ptempNewLineChunk = dynamic_cast<NewLineChunk*>(temp);
      RichTextChunk* tempNext = temp->pGetNextChunk();
      if (tempNext != NULL)
      {
         ptempChunk = dynamic_cast<TextChunk*>(tempNext->pGetNextChunk());
      }
      if (
         (ptempTextChunk != NULL && ptempTextChunk->m_pWrapChild == NULL) || \
         ((ptempNewLineChunk != NULL) && (dynamic_cast<NewLineChunk*>(temp->pGetPreviousChunk(temp, pRichText)) != 0)))
      {
         _linecount++;
      }
      if (ptempTextChunk != 0 && \
            (ptempTextChunk->m_pWrapChild != NULL))
      {
         temp = temp->pAdvance();
      }
      else if ((ptempTextChunk != NULL) && (ptempTextChunk->pGetNextChunk() == NULL) && \
               (ptempTextChunk->m_pWrapChild == NULL) && (ptempTextChunk->m_pWrapParent != NULL))
      {
         while (ptempTextChunk->m_pWrapParent != NULL)
         {
            ptempTextChunk = ptempTextChunk->m_pWrapParent;
         }
         temp = dynamic_cast<RichTextChunk*>(ptempTextChunk);
         if (temp != NULL)
         {
            temp = temp->pGetPreviousChunk(temp, pRichText);
         }
      }
      else if (tempNext != NULL && ptempChunk != NULL && ptempChunk->m_pWrapChild == NULL && \
               ptempChunk->m_pWrapParent == NULL)
      {
         temp = temp->pGetPreviousChunk(temp, pRichText);
      }
      else if ((tempNext != NULL) && (ptempChunk != NULL) && \
               ((ptempChunk->m_pWrapChild == NULL) || (ptempChunk->m_pWrapParent != NULL)))
      {
         temp = temp->pAdvance()->pAdvance();
      }
      else
      {
         temp = temp->pGetPreviousChunk(temp, pRichText);
      }
   }
   return temp;
}*/


/****************************************************************************
*     Line::Line
****************************************************************************/
Line::Line() :
   m_pFirstChunk(0),
   m_pLastChunk(0),
   m_i32CurrentPosition(0),
   m_oContentSize(0, 0),
   m_i32LineHeight(0),
   m_copKerningChar(0)
{
};


/****************************************************************************
*     Line::~Line
****************************************************************************/
Line::~Line()
{
   m_pFirstChunk = 0;
   m_pLastChunk = 0;
   m_copKerningChar = 0;
}


/****************************************************************************
*     Line::vAddChunk
****************************************************************************/
void Line::vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk)
{
   // store pointer to first chunk of line
   if (m_pFirstChunk == 0)
   {
      m_pFirstChunk = pChunk;
   }

   if (pChunk != 0)
   {
      // store pointer to last chunk of line
      m_pLastChunk = pChunk;

      // process the layout function of the chunk
      pChunk->vLayout(pTextContext);

      // calculate bounding box based on chunk size
      // for width the chunk position and size need to be considered separately to handle negative values
      m_i32CurrentPosition += pChunk->cooGetChunkPosition().GetX();

      if (m_oContentSize.GetWidth() < m_i32CurrentPosition)
      {
         m_oContentSize.SetWidth(m_i32CurrentPosition);
      }

      m_i32CurrentPosition += pChunk->cooGetChunkSize().GetWidth();

      if (m_oContentSize.GetWidth() < m_i32CurrentPosition)
      {
         m_oContentSize.SetWidth(m_i32CurrentPosition);
      }

      if (m_oContentSize.GetHeight() < pChunk->cooGetChunkSize().GetHeight())
      {
         m_oContentSize.SetHeight(pChunk->cooGetChunkSize().GetHeight());
      }

      // set actual line height to the biggest of all text chunks
      TextChunk* pTextChunk = dynamic_cast<TextChunk*>(pChunk);
      if ((pTextChunk != 0) && (pTextContext != 0))
      {
         if (pTextContext->m_stFontMetrics.lineHeight > m_i32LineHeight)
         {
            m_i32LineHeight = pTextContext->m_stFontMetrics.lineHeight;
         }
      }
   }
}


/****************************************************************************
*     Line::vTranslate
****************************************************************************/
void Line::vTranslate(Candera::Int32 i32X, Candera::Int32 i32Y)
{
   if (m_pFirstChunk != 0)
   {
      // get position of the first chunk and translate it
      Position oPosition = m_pFirstChunk->cooGetChunkPosition();
      oPosition.Translate(i32X, i32Y);
      m_pFirstChunk->vSetChunkPosition(oPosition);
   }
}


/****************************************************************************
*     Line::vEndLine
****************************************************************************/
void Line::vEndLine(const RichTextRenderer::TextContext* pTextContext)
{
   // if line has so size or gap use it from current set style
   if (m_i32LineHeight == 0)
   {
      m_i32LineHeight = pTextContext->m_stFontMetrics.lineHeight;
   }

   // perform alignment of chunks
   vDoAlignment();
}


/****************************************************************************
*     Line::vDoAlignment
****************************************************************************/
void Line::vDoAlignment()
{
   // OPTIONAL: align all chunks according to their base line for in line font changes
   FEATSTD_LINT_NEXT_EXPRESSION(1762, "Once implemented it won't be const")
}


/****************************************************************************
*     Paragraph::Paragraph
****************************************************************************/

Paragraph::Paragraph() :
   m_oSize(0, 0),
   m_oContentSize(0, 0),
   m_oEffectiveSize(0, 0),
   m_flLineSpacingFactor(1.0F),
   m_enHorizontalAlignment(RT_Leading),
   m_enVerticalAlignment(RT_Leading),
   m_enTextDirection(RT_LeftToRight),
   m_pTabStopList(0),
   m_i32ConstantTabWidth(0),
   m_enTextWrapMode(RT_WrapNone)
{
}


/****************************************************************************
*     Paragraph::~Paragraph
****************************************************************************/
Paragraph::~Paragraph()
{
   const SizeType uLineCount = m_oLineList.size();

   for (SizeType uIndex = 0; uIndex < uLineCount; uIndex++)
   {
      FEATSTD_DELETE(m_oLineList[uIndex]);
   }

   m_oLineList.clear();

   // tabstop list is owned by paragraph chunk
   m_pTabStopList = 0;
}


/****************************************************************************
*     Paragraph::vAddChunk
****************************************************************************/
void Paragraph::vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk)
{
   if (pTextContext != 0)
   {
      // if there is no current line and the chunk is valid create a new one, set it in the context and add it to the line list of the paragraph
      if ((pTextContext->m_pCurrentLine == 0) && (pChunk != 0))
      {
         pTextContext->m_pCurrentLine = FEATSTD_NEW(Line);

         if (pTextContext->m_pCurrentLine != 0)
         {
            m_oLineList.push_back(pTextContext->m_pCurrentLine);
         }
      }

      if (pTextContext->m_pCurrentLine != 0)
      {
         // add chunk to current line
         pTextContext->m_pCurrentLine->vAddChunk(pTextContext, pChunk);

         // if chunk is a "new line" end the current line
         if (dynamic_cast<NewLineChunk*>(pChunk) != 0)
         {
            pTextContext->m_pCurrentLine->vEndLine(pTextContext);
            pTextContext->m_pCurrentLine = 0;
         }
      }
   }
}


/****************************************************************************
*     Paragraph::vTranslate
****************************************************************************/
void Paragraph::vTranslate(Candera::Int32 i32X, Candera::Int32 i32Y)
{
   if (m_oLineList.size() > 0)
   {
      m_oLineList[0]->vTranslate(i32X, i32Y);
   }
}


/****************************************************************************
*     Paragraph::vEndParagraph
****************************************************************************/
void Paragraph::vEndParagraph(RichTextRenderer::TextContext* pTextContext)
{
   // end current line
   if (pTextContext->m_pCurrentLine != 0)
   {
      pTextContext->m_pCurrentLine->vEndLine(pTextContext);
      pTextContext->m_pCurrentLine = 0;
   }

   // perform alignment of lines
   vDoAlignment();
}


/****************************************************************************
*     Paragraph::vDoAlignment
****************************************************************************/
void Paragraph::vDoAlignment()
{
   PerformLineSpacing();

   //printf("Paragraph: line spacing finished. new paragraph size = [%d,%d]\n", m_oCurrentSize.GetWidth(), m_oCurrentSize.GetHeight());

   // calculate effective paragraph bounding box dimensions
   m_oEffectiveSize = m_oContentSize;

   if (m_oSize.GetWidth() > 0)
   {
      m_oEffectiveSize.SetWidth(m_oSize.GetWidth());
   }

   if (m_oSize.GetHeight() > 0)
   {
      m_oEffectiveSize.SetHeight(m_oSize.GetHeight());
   }

   // perform horizontal alignment for all lines of the paragraph
   PerformHorizontalAlignment();

   // perform vertical alignment for all lines of the paragraph
   PerformVerticalAlignment();
}


/******************************************************************************
*  PerformLineSpacing
******************************************************************************/
void Paragraph::PerformLineSpacing()
{
   // perform line spacing
   if (m_oLineList.size() > 0)
   {
      Int32 i32Height = 0;
      m_oContentSize = m_oLineList[0]->cooGetContentSize();

      const SizeType uLineCount = m_oLineList.size();

      ParagraphChunk*  pParaChunk = dynamic_cast<ParagraphChunk*>(m_oLineList[0]->pGetFirstChunk());
      if (pParaChunk != NULL)
      {
         Int32 i32LineHeight = m_oLineList[0]->i32GetLineHeight();
         Int32 LineSpacing = Int32(Float(i32LineHeight) * m_flLineSpacingFactor);
         pParaChunk->vSetLineHeight(LineSpacing);
      }

      for (SizeType uIndex = 1; uIndex < uLineCount; uIndex++)
      {
         Line* pLine = m_oLineList[uIndex];

         // calculate line spacing
         Int32 i32PrevLineHeight = m_oLineList[uIndex - 1]->i32GetLineHeight();
         Int32 i32LineSpacing = Int32(Float(i32PrevLineHeight) * m_flLineSpacingFactor);

         // calculate bounding box based on line size
         if (m_oContentSize.GetWidth() < pLine->cooGetContentSize().GetWidth())
         {
            m_oContentSize.SetWidth(pLine->cooGetContentSize().GetWidth());
         }

         i32Height += i32LineSpacing;

         Int32 i16NextHeight = i32Height + pLine->cooGetContentSize().GetHeight();
         if (m_oContentSize.GetHeight() < i16NextHeight)
         {
            m_oContentSize.SetHeight(i16NextHeight);
         }

         // shift current line
         Int32 iPrevLineWidth = m_oLineList[uIndex - 1]->cooGetContentSize().GetWidth();
         pLine->vTranslate(-iPrevLineWidth, i32LineSpacing);
      }
   }
   else
   {
      m_oContentSize = Size(0, 0);
   }
   // Logic to set start chunk for page up operation
   if (m_oLineList.size() > 0)
   {
      Int32 i32pageUpStartLine = 0;
      RichTextChunk* pChunk = m_oLineList[0]->pGetFirstChunk();
      if ((pChunk != 0) && (pChunk->bGetIsPageOrLineDown() == false))
      {
         i32pageUpStartLine = static_cast<Int32>(m_oLineList.size()) - pChunk->u16GetNumOfLinesPerTextArea();
         if (i32pageUpStartLine <= 0)
         {
            i32pageUpStartLine = DEFAULT_TEXT_CHUNK_START_INDEX;
         }
         RichTextChunk* pPageUpStartChunk = m_oLineList[i32pageUpStartLine]->pGetFirstChunk();
         if (pPageUpStartChunk != 0)
         {
            pPageUpStartChunk->vSetPageUpStartChunk(true);
            int start = 0;
            Candera::Globalization::TextDirection textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
            if (textDirection == Candera::Globalization::RightToLeft) //added for arabic to set position for starting chunk
            {
               start = m_oSize.GetWidth() + pPageUpStartChunk->cooGetChunkPosition().GetX();
            }
            pPageUpStartChunk->vSetChunkPosition(Position(start, 0));
         }
      }
      else if ((pChunk != 0) && (pChunk->bGetIsLineUp() == true))
      {
         Int32 i32LineUpStartLine = 0;
         i32LineUpStartLine = static_cast<Int32>(m_oLineList.size()) - pChunk->u16GetNumOfLinesPerTextArea();
         if (i32LineUpStartLine > 0)
         {
            RichTextChunk* i32LineUpStartChunk = m_oLineList[i32LineUpStartLine]->pGetFirstChunk();
            if (i32LineUpStartChunk != 0)
            {
               i32LineUpStartChunk->vSetLineUpStartChunk(true);
               int start = 0;
               Candera::Globalization::TextDirection textDirection = Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture().GetPointerToSharedInstance()->GetTextDirection();
               if (textDirection == Candera::Globalization::RightToLeft) //added for arabic to set position for starting chunk
               {
                  start = m_oSize.GetWidth() + i32LineUpStartChunk->cooGetChunkPosition().GetX();
               }
               i32LineUpStartChunk->vSetChunkPosition(Position(start, 0));
            }
         }
         else
         {
            //start from begining
            pChunk->vSetLineUp(false);
         }
      }
   }
}


/******************************************************************************
*  PerformHorizontalAlignment
******************************************************************************/
void Paragraph::PerformHorizontalAlignment()
{
   switch (m_enHorizontalAlignment)
   {
      case RT_Middle:
      {
         Int32 i32PrevLineWidth = m_oEffectiveSize.GetWidth();

         const SizeType uLineCount = m_oLineList.size();

         for (SizeType uIndex = 0; uIndex < uLineCount; uIndex++)
         {
            Position oLineTranslation(i32PrevLineWidth - m_oLineList[uIndex]->cooGetContentSize().GetWidth(), 0);
            m_oLineList[uIndex]->vTranslate(oLineTranslation.GetX() / 2, 0);
            i32PrevLineWidth = m_oLineList[uIndex]->cooGetContentSize().GetWidth();
         }
         break;
      }

      case RT_Trailing:
      {
         Int32 i32PrevLineWidth = m_oEffectiveSize.GetWidth();

         const SizeType uLineCount = m_oLineList.size();

         for (SizeType uIndex = 0; uIndex < uLineCount; uIndex++)
         {
            Position oLineTranslation(i32PrevLineWidth - m_oLineList[uIndex]->cooGetContentSize().GetWidth(), 0);
            m_oLineList[uIndex]->vTranslate(oLineTranslation.GetX(), 0);
            i32PrevLineWidth = m_oLineList[uIndex]->cooGetContentSize().GetWidth();
         }
         break;
      }

      default:
         break;
   }
}


/******************************************************************************
*  PerformVerticalAlignment
******************************************************************************/
void Paragraph::PerformVerticalAlignment()
{
   switch (m_enVerticalAlignment)
   {
      case RT_Middle:
      {
         this->vTranslate(0, (m_oEffectiveSize.GetHeight() - m_oContentSize.GetHeight()) / 2);
      }
      break;
      case RT_Trailing:
      {
         this->vTranslate(0, m_oEffectiveSize.GetHeight() - m_oContentSize.GetHeight());
      }
      break;
      default:
         break;
   }
}


/****************************************************************************
*     TextArea::TextArea
****************************************************************************/
TextArea::TextArea() :
   m_oSize(0, 0),
   m_oContentSize(0, 0),
   m_oEffectiveSize(0, 0)
{
}


/****************************************************************************
*     TextArea::~TextArea
****************************************************************************/
TextArea::~TextArea()
{
   const SizeType uLineCount = m_oParagraphList.size();

   for (SizeType uIndex = 0; uIndex < uLineCount; uIndex++)
   {
      FEATSTD_DELETE(m_oParagraphList[uIndex]);
   }

   m_oParagraphList.clear();
}


/****************************************************************************
*     TextArea::vAddChunk
****************************************************************************/
void TextArea::vAddChunk(RichTextRenderer::TextContext* pTextContext, RichTextChunk* pChunk)
{
   if (pTextContext != 0)
   {
      // if chunk is a paragraph chunk create a new paragraph and if needed close previous paragraph
      if (dynamic_cast<ParagraphChunk*>(pChunk) != 0)
      {
         if (pTextContext->m_pCurrentParagraph != 0)
         {
            pTextContext->m_pCurrentParagraph->vEndParagraph(pTextContext);
         }

         pTextContext->m_pCurrentParagraph = FEATSTD_NEW(Paragraph);

         if (pTextContext->m_pCurrentParagraph != 0)
         {
            m_oParagraphList.push_back(pTextContext->m_pCurrentParagraph);
         }
      }

      if (pTextContext->m_pCurrentParagraph != 0)
      {
         // add chunk to paragraph
         pTextContext->m_pCurrentParagraph->vAddChunk(pTextContext, pChunk);
      }
   }
}


/****************************************************************************
*     TextArea::vEndTextArea
****************************************************************************/
void TextArea::vEndTextArea(RichTextRenderer::TextContext* pTextContext)
{
   // end current paragraph
   if (pTextContext->m_pCurrentParagraph != 0)
   {
      pTextContext->m_pCurrentParagraph->vEndParagraph(pTextContext);
      pTextContext->m_pCurrentParagraph = 0;
   }

   // perform alignment of paragraphs
   vDoAlignment();

   /*
   printf("TextArea: TextArea finished and aligned new size [%d,%d]\n",
          m_oCurrentSize.GetWidth(), m_oCurrentSize.GetHeight());
          */
}


/****************************************************************************
*     TextArea::vDoAlignment
****************************************************************************/
void TextArea::vDoAlignment()
{
   // calculate text area bounding box dimensions
   if (m_oParagraphList.size() > 0)
   {
      m_oContentSize = m_oParagraphList[0]->cooGetEffectiveSize();

      const SizeType uParagraphCount = m_oParagraphList.size();

      for (SizeType uIndex = 1; uIndex < uParagraphCount; uIndex++)
      {
         Paragraph* pParagraph = m_oParagraphList[uIndex];

         // calculate new text area height
         m_oContentSize.SetHeight(m_oContentSize.GetHeight() + pParagraph->cooGetEffectiveSize().GetHeight());

         // calculate new text area width
         if (m_oContentSize.GetWidth() < pParagraph->cooGetEffectiveSize().GetWidth())
         {
            m_oContentSize.SetWidth(pParagraph->cooGetEffectiveSize().GetWidth());
         }
      }
   }
   else
   {
      m_oContentSize = Size(0, 0);
   }

   m_oEffectiveSize = m_oContentSize;

   if (m_oSize.GetWidth() > 0)
   {
      m_oEffectiveSize.SetWidth(m_oSize.GetWidth());
   }

   if (m_oSize.GetHeight() > 0)
   {
      m_oEffectiveSize.SetHeight(m_oSize.GetHeight());
   }

   // perform horizontal alignment for all center and trailing aligned paragraphs
   const SizeType uParagraphCount = m_oParagraphList.size();

   for (SizeType uIndex = 0; uIndex < uParagraphCount; uIndex++)
   {
      Paragraph* pParagraph = m_oParagraphList[uIndex];

      switch (pParagraph->coenGetHorizontalAlignment())
      {
         case RT_Middle:
         {
            pParagraph->vTranslate((m_oEffectiveSize.GetWidth() - pParagraph->cooGetEffectiveSize().GetWidth()) / 2, 0);
         }
         break;
         case RT_Trailing:
         {
            pParagraph->vTranslate(m_oEffectiveSize.GetWidth() - pParagraph->cooGetEffectiveSize().GetWidth(), 0);
         }
         break;
         default:
            break;
      }
   }

   // no vertical alignment is performed for paragraphs
}


/****************************************************************************
*     TextArea::uGetLineCount
****************************************************************************/
Candera::SizeType TextArea::uGetLineCount()
{
   SizeType uCount = 0;
   SizeType uParagraphCount = m_oParagraphList.size();

   for (SizeType uIndex = 0; uIndex < uParagraphCount; uIndex++)
   {
      uCount += m_oParagraphList[uIndex]->uGetLineCount();
   }

   return uCount;
}


/****************************************************************************
*     TextArea::pGetLine
****************************************************************************/
Line* TextArea::pGetLine(Candera::SizeType uIndex)
{
   SizeType uCount = 0;
   SizeType uParagraphCount = m_oParagraphList.size();

   for (SizeType uParaIndex = 0; uParaIndex < uParagraphCount; uParaIndex++)
   {
      Paragraph* pParagraph = m_oParagraphList[uParaIndex];
      SizeType uLineCount = pParagraph->uGetLineCount();

      if ((uIndex >= uCount) && (uIndex < uLineCount))
      {
         return m_oParagraphList[uParaIndex]->pGetLine(uIndex - uCount);
      }

      uCount += uLineCount;
   }

   return 0;
}


/****************************************************************************
*     TextArea::GetLineCoordinate
****************************************************************************/
Position TextArea::GetLineCoordinate(Candera::SizeType uIndex)
{
   //TODO store absolute paragraph positions and reuse them to faster calculate the line positions, etc.

   Position oPosition(0, 0);
   SizeType uParagraphCount = m_oParagraphList.size();
   SizeType uGlobalLineIndex = 0;

   for (SizeType uParaIndex = 0; uParaIndex < uParagraphCount; uParaIndex++)
   {
      Paragraph* pParagraph = m_oParagraphList[uParaIndex];
      SizeType uLineCount = pParagraph->uGetLineCount();

      for (SizeType uLineIndex = 0; uLineIndex < uLineCount; uLineIndex++)
      {
         Line* pLine = pParagraph->pGetLine(uLineIndex);

         RichTextChunk* pChunk = pLine->pGetFirstChunk();

         if (pChunk != 0)
         {
            oPosition.SetY(pChunk->cooGetChunkPosition().GetY());

            if (uIndex == uGlobalLineIndex + uLineIndex)
            {
               oPosition.SetX(pChunk->cooGetChunkPosition().GetY());
               return oPosition;
            }
         }
      }

      uGlobalLineIndex += uLineCount;
   }

   return Position(0, 0);
}


/****************************************************************************
*     RichText::RichText
****************************************************************************/
RichText::RichText() :
   m_pFirstChunk(0),
   m_pLastChunk(0),
   m_oSize(0, 0),
   m_oEffectiveSize(0, 0)
{
}


/****************************************************************************
*     RichText::~RichText
****************************************************************************/
RichText::~RichText()
{
   FEATSTD_DELETE(m_pFirstChunk);

   m_pFirstChunk = 0;
   m_pLastChunk = 0;
}


/****************************************************************************
*     RichText::vAddChunk
****************************************************************************/
void  RichText::vAddChunk(RichTextChunk* pChunk)
{
   if (pChunk != 0)
   {
      if (m_pLastChunk == 0)
      {
         m_pFirstChunk = pChunk;
      }
      else
      {
         m_pLastChunk->vSetNextChunk(pChunk);
      }

      m_pLastChunk = pChunk;

      FEATSTD_LINT_NEXT_EXPRESSION(613, "m_pLastChunk can never be 0 as its initial value is checked not to be 0")
      while (m_pLastChunk->pGetNextChunk() != 0)
      {
         m_pLastChunk = m_pLastChunk->pGetNextChunk();
      }
   }
}


/****************************************************************************
*     RichText::vRemoveChunk
****************************************************************************/
void  RichText::vRemoveChunk(RichTextChunk* pChunk, bool bRemoveSubsequent)
{
   if ((pChunk != 0) && (m_pFirstChunk != 0))
   {
      if (m_pFirstChunk == pChunk)
      {
         m_pFirstChunk = 0;
         m_pLastChunk = 0;
      }
      else
      {
         RichTextChunk* pCurrentChunk = m_pFirstChunk;

         FEATSTD_LINT_NEXT_EXPRESSION(613, "pCurrentChunk can never be 0 as its initial value is checked not to be 0")
         while (pCurrentChunk->pGetNextChunk() != 0)
         {
            if (pCurrentChunk->pGetNextChunk() == pChunk)
            {
               if (bRemoveSubsequent)
               {
                  pCurrentChunk->vSetNextChunk(0);
                  m_pLastChunk = pCurrentChunk;
               }
               else
               {
                  pCurrentChunk->vSetNextChunk(pChunk->pGetNextChunk());
                  pChunk->vSetNextChunk(0);

                  if (m_pLastChunk == pChunk)
                  {
                     m_pLastChunk = pCurrentChunk;
                  }
               }
               break;
            }

            pCurrentChunk = pCurrentChunk->pGetNextChunk();
         }
      }
   }
}


/****************************************************************************
*     RichTextChunk::RichTextChunk
****************************************************************************/
RichTextChunk::RichTextChunk() :
   m_oChunkPosition(0, 0),
   m_oChunkSize(0, 0),
   m_bisPageUpStartChunk(false),
   m_pNextChunk(0),
   pRichtext(0),
   m_bisLineUpStartChunk(false)
{
}


/****************************************************************************
*     RichTextChunk::~RichTextChunk
****************************************************************************/
RichTextChunk::~RichTextChunk()
{
   FEATSTD_DELETE(m_pNextChunk);
   FEATSTD_DELETE(pRichtext);

   m_pNextChunk = 0;
   pRichtext = 0;
}


/****************************************************************************
*     ParagraphChunk::ParagraphChunk
****************************************************************************/
ParagraphChunk::ParagraphChunk() :
   m_enHorizontalAlignment(RT_Leading),
   m_enVerticalAlignment(RT_Leading),
   m_enTextDirection(RT_LeftToRight),
   m_oSize(0, 0),
   m_flLineSpacingFactor(1.0F),
   m_enTextWrapMode(RT_WrapNone),
   m_i16ConstantTabWidth(10),
   m_LineHeight(0)
{
}


ParagraphChunk::ParagraphChunk(Candera::RT_TextAlignment enHAlignment, Candera::RT_TextAlignment enVAlignment, Candera::RT_TextDirection enTextDirection, const Size& cooSize, Candera::Float flLineSpacingFactor,  Candera::RT_TextWrapMode enTextWrapMode, Candera::Int16 i16ConstantTabWidth) :
   m_enHorizontalAlignment(enHAlignment),
   m_enVerticalAlignment(enVAlignment),
   m_enTextDirection(enTextDirection),
   m_oSize(cooSize),
   m_flLineSpacingFactor(flLineSpacingFactor),
   m_enTextWrapMode(enTextWrapMode),
   m_i16ConstantTabWidth(i16ConstantTabWidth),
   m_LineHeight(0)
{
}


/****************************************************************************
*     ParagraphChunk::vLayout
****************************************************************************/
void ParagraphChunk::vLayout(RichTextRenderer::TextContext* pContext)
{
   if ((pContext != 0) && (pContext->m_pCurrentParagraph != 0))
   {
      pContext->m_pCurrentParagraph->vSetHorizontalAlignment(m_enHorizontalAlignment);
      pContext->m_pCurrentParagraph->vSetVerticalAlignment(m_enVerticalAlignment);
      pContext->m_pCurrentParagraph->vSetTextDirection(m_enTextDirection);
      pContext->m_pCurrentParagraph->vSetSize(m_oSize);
      pContext->m_pCurrentParagraph->vSetLineSpacingFactor(m_flLineSpacingFactor);
      pContext->m_pCurrentParagraph->vSetTextWrapMode(m_enTextWrapMode);
      pContext->m_pCurrentParagraph->vSetTabStopList(&m_oTabStopList);
      pContext->m_pCurrentParagraph->vSetConstantTabWidth(m_i16ConstantTabWidth);
   }
}


/****************************************************************************
*     TextChunk::TextChunk
****************************************************************************/
TextChunk::TextChunk() :
   m_oText(),
   m_pWrapChild(0),
   m_pWrapParent(0)
{
}


TextChunk::TextChunk(const FeatStd::String& oText) :
   m_oText(oText),
   m_pWrapChild(0),
   m_pWrapParent(0)
{
}


/****************************************************************************
*     TextChunk::~TextChunk
****************************************************************************/
TextChunk::~TextChunk()
{
   // Note: m_pWrapParent pointer must not be freed
   FEATSTD_DELETE(m_pWrapChild);

   m_pWrapChild = 0;
   m_pWrapParent = 0;
}


/****************************************************************************
*     TextChunk::vLayout
****************************************************************************/
void TextChunk::vLayout(RichTextRenderer::TextContext* pContext)
{
   // delete text wrapping nodes
   FEATSTD_DELETE(m_pWrapChild);
   m_pWrapChild = 0;

   if (pContext != 0)
   {
      // measure chunk size
      Candera::TextRendering::TextSize size;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
      size = cooMeasureText(pContext, m_oText.GetCString(), -1).GetSize();
      SECURE_FEATSTD_STRING_ACCESS_END();
      m_oChunkSize = Size(size.GetWidth(), size.GetHeight());

      Line* pCurrentLine = pContext->m_pCurrentLine;
      Paragraph* pCurrentParagraph = pContext->m_pCurrentParagraph;

#ifdef RICHTEXT_PERFORMANCE_CHECK
      hmibase::util::Ticker _tPerf;
#endif
      if ((pCurrentLine != 0) && (pCurrentParagraph != 0))
      {
#ifdef RICHTEXT_PERFORMANCE_CHECK
         rich_txt_perf_counter++;
#endif
         if ((pCurrentParagraph->enGetTextWrapMode() != RT_WrapNone) && (pCurrentParagraph->cooGetSize().GetWidth() > 0))
         {
            // calculate remaining width
            Int32 i32RemainingWidth = pContext->m_pCurrentParagraph->cooGetSize().GetWidth() - pCurrentLine->i32GetCurrentPosition();
            if (i32RemainingWidth < 0)
            {
               i32RemainingWidth = 0;
            }

            // if wrapping is needed set chunk size to 0 and stop processing here
            if (i32RemainingWidth < m_oChunkSize.GetWidth())
            {
               vWrapText(pContext, i32RemainingWidth);
               m_oChunkSize = Size(0, 0);
               return;
            }
         }
         CheckCurrentKerningCharacter(pCurrentLine, pContext);
#ifdef RICHTEXT_PERFORMANCE_CHECK
//         ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), " <<<<<Time (%d) wrtdwrap = %ld", counter, _tPerf.diffMsec()));
#endif
         if (!m_oText.IsEmpty())
         {
            SetKerningCharacterForNextBlock(pCurrentLine);
         }
      }
   }
}


/****************************************************************************
*     TextChunk::vRender
****************************************************************************/
void TextChunk::vRender(RichTextRenderer::TextContext* pContext, const TextCoordinate& cooPosition)
{
   if ((pContext != 0) && (m_pWrapChild == 0) && !pContext->m_oStyle.PointsToNull())
   {
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
      RichTextRenderer::TextContext::m_oTextRenderer.Render(RichTextRenderer::TextContext::m_oTextRenderContext, LayoutingOptions(cooPosition), ShapingOptions(pContext->m_oStyle), TextProperties(m_oText.GetCString()));
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
}


/****************************************************************************
*     TextChunk::pAdvance
****************************************************************************/
RichTextChunk* TextChunk::pAdvance()
{
   // return chunk according to the following priorities child, next and parent.
   // Skips intermediary nodes.
   if (m_pWrapChild != 0)
   {
      TextChunk* pWrapNode = m_pWrapChild;

      while (pWrapNode->m_pWrapChild != 0)
      {
         pWrapNode = pWrapNode->m_pWrapChild;
      }
      return pWrapNode;
   }
   else
   {
      if (m_pNextChunk != 0)
      {
         return m_pNextChunk;
      }
      else
      {
         TextChunk* pWrapNode = m_pWrapParent;

         if (pWrapNode != 0)
         {
            while ((pWrapNode->m_pWrapParent != 0) && (pWrapNode->m_pNextChunk == 0))
            {
               pWrapNode = pWrapNode->m_pWrapParent;
            }

            return pWrapNode->m_pNextChunk;
         }
      }
   }

   return 0;
}


/****************************************************************************
*     TextChunk::cooMeasureText  // fr83hi ### performance usage 90%
****************************************************************************/
const TextRect TextChunk::cooMeasureText(const RichTextRenderer::TextContext* pContext, const TChar* copText, TextLength i32TextLength) const
{
   if (pContext != 0 && !pContext->m_oStyle.PointsToNull())
   {
#ifdef RICHTEXT_PERFORMANCE_CHECK
      rich_txt_perf_MeasureText++;
#endif

#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
      CombinedTextMeasureContext context(0);
      TextRenderer().Render(context, TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(pContext->m_oStyle), TextRendering::TextProperties(copText, i32TextLength));
      return context.GetTextRectangle();
#else
      return TextRendering::TextRenderer().GetBoundingRectangle(TextRendering::LayoutingOptions(), TextRendering::ShapingOptions(pContext->m_oStyle),
             TextRendering::TextProperties(copText, i32TextLength), TextRendering::MeasuringOptions(TextRendering::MeasuringOptions::DefaultFinalAdvance,
                   TextRendering::MeasuringOptions::DefaultTransversalSize));
#endif
      /* next block only for test to prevent creation of TextRenderer instance
            static TextRendering::TextRenderer tr;
            TextRect r;
            r = tr.GetBoundingRectangle(
               TextRendering::LayoutingOptions(),
               TextRendering::ShapingOptions(pContext->m_oStyle),
               TextRendering::TextProperties(copText, i32TextLength),
               TextRendering::MeasuringOptions(TextRendering::MeasuringOptions::DefaultFinalAdvance,
               TextRendering::MeasuringOptions::DefaultTransversalSize,
               TextRendering::MeasuringOptions::Cursor)
               );
            // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), " R[%d] len=%d Text= %s", r.GetWidth(), i32TextLength, copText));
            return r;
      */
   }
   else
   {
      return TextRect(0, 0, 1, 1);
   }
}


bool TextChunk::IsValidByte(Candera::UInt32 index)
{
   const TChar* l_pCharPtr = NULL;
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
   l_pCharPtr = m_oText.GetCString();

   if (l_pCharPtr != 0)
   {
      unsigned u8StartByte = (unsigned char)l_pCharPtr[index];
      if (((u8StartByte & 0xC0) == 0x80) || ((u8StartByte >= 0xC0 || u8StartByte < 0x80) == false))
      {
         // Not a starting byte */
         return false;
      }
   }
   SECURE_FEATSTD_STRING_ACCESS_END();
   return true;
}


/****************************************************************************
*     TextChunk::vWrapText
****************************************************************************/
void TextChunk::vWrapText(const RichTextRenderer::TextContext* pContext, Int32 i32RemainingWidth)  // fr83hi ### performance 90%
{
   if ((pContext != 0) && (pContext->m_pCurrentParagraph != 0) && (pContext->m_pCurrentLine != 0))
   {
      UInt32 u32FirstPartLength = 0;
      UInt32 u32SecondPartBytePosition = 0;
#ifdef RICHTEXT_PERFORMANCE_CHECK
      rich_txt_perf_WrapText++;
#endif
      RT_TextWrapMode enTextWrapMode = pContext->m_pCurrentParagraph->enGetTextWrapMode();
      if (enTextWrapMode == RT_WrapWords)
      {
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
         const TChar* copText = m_oText.GetCString();

         UInt32 u32Position = 0;
         UInt32 u32BytePosition = 0;

         while (copText[u32BytePosition] != 0)
         {
            //condition check for space
            if (copText[u32BytePosition] == ' ')
            {
//              if (u32Position > 45)
//               {
               // measure current text length
               Int16 i16MeasuredWidth = cooMeasureText(pContext, copText, Int32(u32BytePosition)).GetWidth();
               // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "WRAPTEXT LOOP search offs=%d pixelwidth=%d", www, (int)i16MeasuredWidth));

               // if it fits remember position else stop here
               if (i16MeasuredWidth <= i32RemainingWidth)
               {
                  u32FirstPartLength = u32Position;
                  u32SecondPartBytePosition = u32BytePosition + 1;
               }
               else
               {
                  // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "WRAPTEXT LOOP end steps=%d strlen=%d pixelwidth=%d", ++www, u32Position, (int)i16MeasuredWidth));
                  break;
//                }
               }
            }
            //condition check forZWSP character
            if ((UInt8(copText[u32BytePosition]) == 0xE2) && (UInt8(copText[u32BytePosition + 1]) == 0x80) && (UInt8(copText[u32BytePosition + 2]) == 0x8B))
            {
               Int16 i16MeasuredWidth = cooMeasureText(pContext, copText, Int32(u32BytePosition)).GetWidth();
               // if it fits remember position else stop here
               if (i16MeasuredWidth <= i32RemainingWidth)
               {
                  //changes done to ensure ZWSP is included in first part of the string.
                  u32FirstPartLength = u32Position + INCLUDE_ZWSP;
                  u32SecondPartBytePosition = u32BytePosition + INCREMENT_BYTE_POSITION;
               }
               else
               {
                  // ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "WRAPTEXT LOOP end steps=%d strlen=%d pixelwidth=%d", ++www, u32Position, (int)i16MeasuredWidth));
                  break;
               }
            }

            // next codepoint
            u32BytePosition += Utf8Encoding::u32CodeLength(&(copText[u32BytePosition]));
            u32Position++;
         }
         SECURE_FEATSTD_STRING_ACCESS_END();
         // if no word fits and the line is empty try character wrapping
         if ((u32FirstPartLength == 0) && (i32RemainingWidth == pContext->m_pCurrentParagraph->cooGetSize().GetWidth()))
         {
            enTextWrapMode = RT_WrapCharacters;
         }
      }
      if (enTextWrapMode == RT_WrapCharacters)
      {
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
         const TChar* copText = m_oText.GetCString();
         UInt32 u32Start = 0;
         UInt32 u32End = m_oText.GetCodePointCount();

         // IMPROVEMENT: "guess" position bases on relative size difference for faster processing of very large texts

         while ((u32End - u32Start) > 1)
         {
            // calculate next codepoint count that should be checked
            UInt32 u32Pos = (u32Start + u32End) / 2;

            // check width and shift the search boundaries accordingly
            UInt32 charCount = 0;
            for (UInt32 i = 0; i < u32Pos; i++)
            {
               charCount += Utf8Encoding::u32CodeLength(&(copText[charCount]));
            }
            // code point vs. character
            Int16 i16Width = cooMeasureText(pContext, copText, Int32(charCount)).GetWidth();

            if (i16Width > i32RemainingWidth)
            {
               u32End = u32Pos;
            }
            else
            {
               u32Start = u32Pos;
            }
         }

         u32FirstPartLength = u32Start;
         for (UInt32 u32Index = 0; u32Index < u32FirstPartLength; u32Index++)
         {
            u32SecondPartBytePosition += Utf8Encoding::u32Advance(copText);
         }
         SECURE_FEATSTD_STRING_ACCESS_END();
      }

      // if no character fits and line is empty draw at least 1
      if ((u32FirstPartLength == 0) && (i32RemainingWidth == pContext->m_pCurrentParagraph->cooGetSize().GetWidth()))
      {
         u32FirstPartLength = 1;
         SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
         u32SecondPartBytePosition = Utf8Encoding::u32CodeLength(m_oText.GetCString());
         SECURE_FEATSTD_STRING_ACCESS_END();
      }

      // create chunks based on the calculated split position
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
      const TChar* copText = m_oText.GetCString();

      if (copText != 0)
      {
         bool IsValid = false;
         Candera::UInt32 stepCount = 0;

         while (!IsValid && (stepCount < MAX_STEP_COUNT))
         {
            IsValid = IsValidByte(u32SecondPartBytePosition);
            if ((!IsValid) && (u32SecondPartBytePosition > 0))
            {
               u32SecondPartBytePosition--;
            }
            stepCount++;
         }

         FeatStd::String sText1(copText, u32FirstPartLength);
         TextChunk* pTextChunk1 = FEATSTD_NEW(TextChunk)(sText1);

         if (pTextChunk1 != 0)
         {
            NewLineChunk* pNewLineChunk;
            if (enTextWrapMode == RT_WrapCharacters)
            {
               //Adds NewLineChunk only for splitting content into lines. But that chunk won't be considered as part of rendered character count
               pNewLineChunk = FEATSTD_NEW(NewLineChunk)(0);
            }
            else
            {
               pNewLineChunk = FEATSTD_NEW(NewLineChunk);   //lint -esym(429,pNewLineChunk)
            }

            if (pNewLineChunk != 0)
            {
               pTextChunk1->vSetNextChunk(pNewLineChunk);

               FeatStd::String sText2(&(copText[u32SecondPartBytePosition]));
               if (!sText2.IsEmpty())
               {
                  TextChunk* pTextChunk2 = FEATSTD_NEW(TextChunk)(sText2);  //lint -esym(429,pTextChunk2)

                  if (pTextChunk2 != 0)
                  {
                     pNewLineChunk->vSetNextChunk(pTextChunk2);

                     m_pWrapChild = pTextChunk1;
                     pTextChunk2->m_pWrapParent = this;
                  }
                  else
                  {
                     FEATSTD_DELETE(pTextChunk1);
                  }
               }
            }
            else
            {
               FEATSTD_DELETE(pTextChunk1);
            }
         }
      }
      SECURE_FEATSTD_STRING_ACCESS_END();
   }
}


/******************************************************************************
*  CheckCurrentKerningCharacter // fr83hi ###  maybe also performance issue
******************************************************************************/
void TextChunk::CheckCurrentKerningCharacter(const Line* pCurrentLine, const RichTextRenderer::TextContext* pContext)
{
   const TChar* copKerningChar = pCurrentLine->copGetKerningCharacter();

   if ((copKerningChar != 0) && (copKerningChar[0] != '\0') && (!m_oText.IsEmpty()))
   {
      PixelPosition i16PrevCharWidth = cooMeasureText(pContext, copKerningChar, -1).GetWidth();
      PixelPosition i16NextCharWidth;
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
      TChar achNextCharBuffer[5];
      Utf8Encoding::u32Copy(achNextCharBuffer, 5, m_oText.GetCString(), 1);
      i16NextCharWidth = cooMeasureText(pContext, achNextCharBuffer, -1).GetWidth();
      SECURE_FEATSTD_STRING_ACCESS_END();
      Int i16CombinedCharWidth = i16PrevCharWidth + i16NextCharWidth;

      TChar achBuffer[11];
      UInt32 u32ByteCount = Utf8Encoding::u32Copy(achBuffer, 11, copKerningChar, 1);
      SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
      u32ByteCount += Utf8Encoding::u32Copy(&(achBuffer[u32ByteCount]), 11 - u32ByteCount, m_oText.GetCString(), 1);
      SECURE_FEATSTD_STRING_ACCESS_END();

      PixelPosition i16KernedCharWidth = cooMeasureText(pContext, achBuffer, -1).GetWidth();

      if (i16KernedCharWidth != i16CombinedCharWidth)
      {
         m_oChunkPosition.SetX(i16KernedCharWidth - i16CombinedCharWidth);
      }
   }
}


/******************************************************************************
*  SetKerningCharacterForNextBlock
******************************************************************************/
void TextChunk::SetKerningCharacterForNextBlock(Line* pCurrentLine)
{
   SECURE_FEATSTD_STRING_ACCESS_BEGIN(m_oText);
   const Char* pCodePoint = m_oText.GetCString();
   const Char* pPrevCodePoint = pCodePoint;

   while ((Utf8Encoding::u32Advance(pCodePoint) != 0) && (pCodePoint[0] != '\0'))
   {
      pPrevCodePoint = pCodePoint;
   }

   pCurrentLine->vSetKerningCharacter(pPrevCodePoint);
   SECURE_FEATSTD_STRING_ACCESS_END();
}


/****************************************************************************
*     StyleChunk::StyleChunk
****************************************************************************/
StyleChunk::StyleChunk() :
   m_oStyle(0)
{
}


StyleChunk::StyleChunk(tSharedStylePtr oStyle) :
   m_oStyle(oStyle)
{
}


/****************************************************************************
*     StyleChunk::vRender
****************************************************************************/
void StyleChunk::vRender(RichTextRenderer::TextContext* pContext, const TextCoordinate& cooPosition)
{
   FEATSTD_UNUSED(cooPosition);

   if (pContext != 0)
   {
      pContext->m_oStyle = m_oStyle;
   }
}


/****************************************************************************
*     StyleChunk::vLayout
****************************************************************************/
void StyleChunk::vLayout(RichTextRenderer::TextContext* pContext)
{
   if (pContext != 0)
   {
      pContext->m_oStyle = m_oStyle;
      if (!m_oStyle.PointsToNull())
      {
         m_oStyle->GetDefaultFont().GetMetrics(pContext->m_stFontMetrics);
      }
   }
}


/****************************************************************************
*     ColorChunk::ColorChunk
****************************************************************************/
ColorChunk::ColorChunk() :
   _color(1.0f, 1.0f, 1.0f, 1.0f),
   m_u32ColorCharCnt(0)
{
}


ColorChunk::ColorChunk(const Color& cooColor, Candera::UInt32 u32ColorCharCnt) :
   _color(cooColor),
   m_u32ColorCharCnt(u32ColorCharCnt)
{
}


/****************************************************************************
*     ColorChunk::vRender
****************************************************************************/
void ColorChunk::vRender(RichTextRenderer::TextContext* pContext, const TextCoordinate& cooPosition)
{
   FEATSTD_UNUSED(cooPosition);

   if (pContext != 0)
   {
      pContext->m_oTextColor = _color;

      RichTextRenderer::TextContext::m_oTextRenderContext.SetPenColor(_color);
   }
}


/****************************************************************************
*     TabChunk::TabChunk
****************************************************************************/
TabChunk::TabChunk() :
   m_u32TabId(USHRT_MAX),
   m_u32tabCharCnt(1)
{
}


TabChunk::TabChunk(Candera::UInt32 u32TabId, Candera::UInt32 u32TabCharCnt) :
   m_u32TabId(u32TabId),
   m_u32tabCharCnt(u32TabCharCnt)
{
}


/****************************************************************************
*     TabChunk::vLayout
****************************************************************************/
void TabChunk::vLayout(RichTextRenderer::TextContext* pContext)
{
   if (pContext != 0)
   {
      Line* pCurrentLine = pContext->m_pCurrentLine;
      Paragraph* pCurrentParagraph = pContext->m_pCurrentParagraph;

      if (pCurrentLine != 0)
      {
         // stop kerning
         pContext->m_pCurrentLine->vSetKerningCharacter(0);

         if (pCurrentParagraph != 0)
         {
            const tTabStopList* pTabStopList = pCurrentParagraph->GetTabStopList();

            if (pTabStopList != 0)
            {
               Int32 i32CurrentLinePosition = pCurrentLine->i32GetCurrentPosition();
               SizeType u32TabStopListSize = pTabStopList->size();

               // use constant width by default
               Int32 i32TabWidth = pCurrentParagraph->i32GetConstantTabWidth();

               if (m_u32TabId != USHRT_MAX)
               {
                  // if a fixed tab id is given use this if possible
                  SizeType u32Index = m_u32TabId;

                  if (u32Index < u32TabStopListSize)
                  {
                     i32TabWidth = (*pTabStopList)[u32Index] - i32CurrentLinePosition;
                  }
               }
               else
               {
                  // find next tab stop position
                  for (UInt32 u32Index = 0; u32Index < u32TabStopListSize; u32Index++)
                  {
                     if ((*pTabStopList)[u32Index] > i32CurrentLinePosition)
                     {
                        i32TabWidth = (*pTabStopList)[u32Index] - i32CurrentLinePosition;
                        break;
                     }
                  }
               }

               m_oChunkSize = Size(i32TabWidth, 0);
            }
            else
            {
               m_oChunkSize = Size(pCurrentParagraph->i32GetConstantTabWidth(), 0);
            }
         }
      }
   }
}


} // namespace RichText
