/* ***************************************************************************************
* FILE:          RtConfig.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtConfig is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "RtConfig.h"
#include <Candera/TextEngine/Internal/FullBoundCodePointIterator.h>
#include <Candera/TextEngine/Internal/GlyphRenderer.h>
#include <Candera/TextEngine/Internal/Shaper.h>
#include <Candera/TextEngine/Internal/ShaperInfo.h>
#include <Candera/TextEngine/ShapingOptions.h>
#include <Candera/TextEngine/Style.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/BitmapTextBrushCache.h>

namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera::TextRendering;

#if defined(USE_MINIMAL_TEXTRENDERER)

#define USE_SHAPER

void ProcessChunk(TextRenderContext& context, const ShapingOptions& shapingOptions, const FeatStd::TChar* text)
{
   const SharedStyle::SharedPointer& style = shapingOptions.GetStyle();
   if (style == 0)
   {
      return;
   }
   const Font& font = style->GetDefaultFont();// Internal::StyleTools::GetFontByIdentifier(style, fontIdentifier);

   // Skip rendering if font is invalid
   if (!font.IsValid())
   {
      return;
   }

   const FeatStd::UInt8 faceId = font.GetFaceId();
   const FeatStd::Int16 fontHeight = font.GetHeight();
   const FeatStd::Int16 fontWidth = font.GetWidth();

   static const TextRenderContext* cache = Candera::Internal::BitmapTextBrushCache().GetTextRenderContext();
   Candera::TextRendering::Internal::GlyphRenderInfo glyphRenderInfo;
   glyphRenderInfo.access = cache->GetGlyphCacheAccess();
   glyphRenderInfo.useCache = true;
   glyphRenderInfo.renderGlyph = true;
   glyphRenderInfo.defaultGlyph = style->GetDefaultCodepoint();

   Candera::TextRendering::Internal::GlyphRenderer glyphRenderer(glyphRenderInfo);
   TextCoordinate cursorPosition;

   // Note: -1 is needed because TextEngine Layouter::Layouter() also does this correction - for whatever reason!?
   cursorPosition.TranslateY(style->GetMetrics().ascender - 1);

#if defined(USE_SHAPER)
   Candera::TextRendering::Internal::ShaperInfo shaperInfo(shapingOptions, GlyphBitmap::LeftToRight, TextRenderContext::RenderOrder);
#if ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5))
   Candera::TextRendering::Internal::Shaper& shaper = Candera::TextRendering::Internal::Shaper::GetShaperInstance(shapingOptions.GetShaperType());
#else
   Candera::TextRendering::Internal::Shaper shaper;
#endif
   (void)shaper.Initialize(glyphRenderer, font, text, -1, shaperInfo);

   while (!shaper.IsEnd())
   {
      GlyphIndex glyph = shaper.GetGlyph();
#else
   FeatStd::Int charIndex = 0;

   // Iterate through glyphs.
   Candera::TextRendering::Internal::FullBoundCodePointIterator it(text, -1);
   while ((*it) != 0)
   {
      // Get glyph index from shaper.
      GlyphIndex glyph = glyphRenderer.GetGlyphIndex(font, (*it));
#endif

      if (glyph != 0)
      {
         GlyphBitmap glyphBitmap = GlyphBitmap();

         if (glyphRenderer.GetGlyphBitmap(glyphBitmap, font, glyph))
         {
            glyphBitmap.glyphIndex = glyph;
            glyphBitmap.faceIdentifier.m_value = faceId;
            glyphBitmap.fontHeight = fontHeight;
            glyphBitmap.fontWidth = fontWidth;
#if defined(USE_SHAPER)
            glyphBitmap.direction = shaper.GetDirection();
            glyphBitmap.characterPosition = shaper.GetCharacterPosition();
            // Correct bitmap info.
            shaper.CorrectGlyphBitmap(glyphBitmap);
#else
            glyphBitmap.direction = Candera::TextRendering::GlyphBitmap::LeftToRight;
            glyphBitmap.characterPosition = charIndex;
#endif
            // Set bounds of the glyph rect.
            TextRect glyphRect(
               TextCoordinate(cursorPosition.GetX() + glyphBitmap.left, cursorPosition.GetY() - glyphBitmap.top),
               TextSize(static_cast<FeatStd::Int16>(glyphBitmap.width), static_cast<FeatStd::Int16>(glyphBitmap.height))
            );

            context.Blit(
               glyphRect.GetLeft(),
               glyphRect.GetTop(),
               glyphBitmap);

            // Update cursor position.
            cursorPosition.Translate(glyphBitmap.xadvance, glyphBitmap.yadvance);
         }
      }
#if defined(USE_SHAPER)
      shaper.Advance();
#else
      it.Advance();
      charIndex++;
#endif
   }
}


#endif
}


}
}
