/* ***************************************************************************************
* FILE:          RtElementStyleAccessor.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtElementStyleAccessor is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtElementStyleAccessor.h"
#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <hmibase/util/Macros.h>

namespace hmibase {
namespace widget {
namespace richtext {

ElementStyleAccessor& ElementStyleAccessor::GetInstance()
{
   FEATSTD_UNSYNCED_STATIC_OBJECT(ElementStyleAccessor, s_elementStyleAccesor);
   return s_elementStyleAccesor;
}


ElementStyle::SharedPointer ElementStyleAccessor::CreateStyle(const FeatStd::Char* name) const
{
   return ElementStyle::SharedPointer(FEATSTD_NEW(ElementStyle)(name));
}


ElementStyleAccessor::ElementStyleAccessor()
{
}


bool ElementStyleAccessor::AddStyle(const ElementStyle::SharedPointer& style)
{
   return m_styles.Insert(style->GetName(), style);
}


ElementStyle::SharedPointer ElementStyleAccessor::GetStyle(const FeatStd::Char* name)
{
   ElementStyle::SharedPointer style;
   ElementStyle::SharedPointer* value = m_styles.Find(name);
   if (value != 0)
   {
      style = *value;
   }
   return style;
}


//***********************************************************************************************
//  T E S T
//_TODO("convert to a real unit test")
//***********************************************************************************************
#ifdef WIN32
class ElementStyleAccessorTest
{
   public:
      ElementStyleAccessorTest()
      {
         ElementStyleAccessor& acc = ElementStyleAccessor::GetInstance();

         ElementStyle::SharedPointer pStyle = acc.CreateStyle("p");
         if (pStyle != 0)
         {
            (void)pStyle->AddParameter(StyleParameterBase::Name(TestTypes::Int), FeatStd::Int(3));
            (void)pStyle->AddParameter(StyleParameterBase::Name(TestTypes::Vector2), Candera::Vector2(4.5F, 6.7F));
            (void)acc.AddStyle(pStyle);
         }

         const FeatStd::Char* s_test = "Test";
         ElementStyle::SharedPointer divStyle = acc.CreateStyle("div");
         if (divStyle != 0)
         {
            (void)divStyle->AddParameter(StyleParameterBase::Name(TestTypes::CharPtr), s_test);
            (void)divStyle->AddParameter(StyleParameterBase::Name(TestTypes::Int), FeatStd::Int(5));

            (void)divStyle->AddParameter(StyleParameterBase::Name(TestTypes::SizeType1), FeatStd::SizeType(1));
            (void)divStyle->AddParameter(StyleParameterBase::Name(TestTypes::SizeType2), FeatStd::SizeType(2));

            (void)acc.AddStyle(divStyle);
         }

         ElementStyle::SharedPointer style = acc.GetStyle("p");
         FEATSTD_DEBUG_ASSERT(style != 0);
         if (style != 0)
         {
            Candera::Vector2 vec2;
            bool ok = style->GetParameter(StyleParameterBase::Name(TestTypes::Vector2), vec2);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(vec2 == Candera::Vector2(4.5F, 6.7F));

            FeatStd::Int i;
            ok = style->GetParameter(StyleParameterBase::Name(TestTypes::Int), i);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(i == 3);
         }

         style = acc.GetStyle("div");
         FEATSTD_DEBUG_ASSERT(style != 0);
         if (style != 0)
         {
            const FeatStd::Char* s = 0;
            bool ok = style->GetParameter(StyleParameterBase::Name(TestTypes::CharPtr), s);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(s == s_test);

            // If MyType1 and MyType2 are not unique (both are FeatStd::SizeType), accessing MyType2 first would return
            // the instance of MyType1, because it was added first.
            FeatStd::SizeType size;
            ok = style->GetParameter(StyleParameterBase::Name(TestTypes::SizeType2), size);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(size == 2);

            ok = style->GetParameter(StyleParameterBase::Name(TestTypes::SizeType1), size);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(size == 1);

            FeatStd::Int i;
            ok = style->GetParameter(StyleParameterBase::Name(TestTypes::Int), i);
            FEATSTD_DEBUG_ASSERT(ok);
            FEATSTD_DEBUG_ASSERT(i == 5);
         }
      }

   private:
      struct TestTypes
      {
         enum Enum
         {
            Int = StyleParameterBase::_StartUserDefined_,
            CharPtr,
            SizeType1,
            SizeType2,
            Vector2,
         };
      };
};


static ElementStyleAccessorTest test;
#endif
} // namespace richtext
} // namespace widget
} // namespace hmibase
