/* ***************************************************************************************
* FILE:          ScrollableRichTextWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  ScrollableRichTextWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "ScrollableRichTextWidget2D.h"
#include "RichTextWidget2D.h"
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Widgets/2D/RichText/Engine/RtDynamicProperties.h>
#include <Widgets/2D/WidgetFinder2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/ScrollableRichTextWidget2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {


CGI_WIDGET_RTTI_DEFINITION(ScrollableRichTextWidget2D);


ScrollableRichTextWidget2D::ScrollableRichTextWidget2D() :
   m_viewportSize(),
   m_viewport(0),
   m_listener(0),
   m_offset(0.0F),
   m_docHeight(0.0F),
   m_lineHeight(0.0F)
{
}


ScrollableRichTextWidget2D::~ScrollableRichTextWidget2D()
{
   m_viewport = 0;
   m_listener = 0;
}


void ScrollableRichTextWidget2D::Update()
{
   Base::Update();

   if (m_viewport == 0)
   {
      Candera::Node2D* node = GetNode();
      if (node != 0)
      {
         Engine* engine = DynamicProperties::GetEngine(node);
         if (engine != 0)
         {
            DocAccessor::SharedPointer docAccessor = engine->GetDocAccessor();
            if (docAccessor != 0)
            {
               Document::SharedPointer doc = docAccessor->GetDocument();
               if (doc != 0)
               {
                  m_docHeight = doc->GetRect().GetHeight();
                  Candera::TextRendering::SharedStyle::SharedPointer textStyle = engine->GetData().m_textStyle;
                  if (textStyle != 0)
                  {
                     Candera::TextRendering::Metrics metrics;
                     if (textStyle->GetDefaultFont().GetMetrics(metrics))
                     {
                        m_lineHeight = metrics.lineHeight;
                     }
                  }

                  Viewport::SharedPointer viewport = engine->GetViewport();
                  m_viewport = viewport.GetPointerToSharedInstance();
                  if (m_viewport != 0)
                  {
                     m_viewportSize = m_viewport->GetSize();
                     if (m_listener != 0)
                     {
                        m_listener->OnContentUpdated();
                     }
                  }
               }
            }
         }
      }
   }

   if (m_swipe.running)
   {
      FeatStd::Float time = 0.001F * FeatStd::Float(hmibase::utils::Ticker::getTickCountMsec() - m_swipe.timeStart);
      if (time >= m_swipe.duration)
      {
         m_swipe.running = false;
      }
      else
      {
         FeatStd::Float distance = m_swipe.velocityStart * time + (0.5F * m_swipe.acceleration * time * time);
         distance *= m_swipe.direction;
         SetOffset(m_swipe.offsetStart + distance);
      }
   }
}


bool ScrollableRichTextWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const ScrollableRichTextWidget2D* original = CLONEABLE_WIDGET_CAST<const ScrollableRichTextWidget2D*>(originalWidget);
      if (original != 0)
      {
         SetScrollId(original->GetScrollId());
         SetSwipeDeceleration(original->GetSwipeDeceleration());
         SetSwipeMaxDistance(original->GetSwipeMaxDistance());
         cloned = true;
      }
   }
   return cloned;
}


void ScrollableRichTextWidget2D::Init(Candera::AssetProvider* assetProvider)
{
   Base::Init(assetProvider);
}


bool ScrollableRichTextWidget2D::OnMessage(const Courier::Message& msg)
{
   bool consumed = Base::OnMessage(msg);
   if (!consumed)
   {
      if (msg.GetId() == ListChangeMsg::ID)
      {
         const ListChangeMsg& listChangeMsg = static_cast<const ListChangeMsg&>(msg);
         consumed = OnListChangeMsg(listChangeMsg);
      }
   }

   return consumed;
}


void ScrollableRichTextWidget2D::OnBeforeNodeChanged()
{
}


void ScrollableRichTextWidget2D::OnNodeChanged()
{
   m_viewport = 0;
}


bool ScrollableRichTextWidget2D::OnTouch(const Candera::Camera2D& camera2D, const Candera::Vector2& point)
{
   bool intersects = false;
   Candera::Node2D* node = GetNode();
   if (GetTouchable() && (node != 0) && node->IsEffectiveRenderingEnabled())
   {
      const Candera::Node2D* parent = node->GetParent();
      if (parent != 0)
      {
         const Candera::Vector2& pointInWorldSpace(Candera::Math2D::TransformViewportToScene(camera2D, Candera::Math2D::TransformRenderTargetToViewport(camera2D, point)));
         Candera::Matrix3x2 parentMatrix(parent->GetWorldTransform());
         parentMatrix.Inverse();
         const Candera::Vector2 pointInParentSpace = parentMatrix.Multiply(pointInWorldSpace);

         Candera::Rectangle clippingRectangle(node->GetPosition(), m_viewportSize);

         intersects = clippingRectangle.Contains(pointInParentSpace);
      }
   }
   return intersects;
}


bool ScrollableRichTextWidget2D::OnTapGesture(const hmibase::input::gesture::GestureEvent& gestureEvent)
{
   if (gestureEvent._isPoint1Valid)
   {
      m_swipe.running = false;
   }
   return true;
}


bool ScrollableRichTextWidget2D::OnDragGesture(const hmibase::input::gesture::GestureEvent& gestureEvent)
{
   if (gestureEvent._isPoint1Valid)
   {
      FeatStd::Float offsetY = 0.0F;

      switch (gestureEvent._gestureState)
      {
         case GestureEvent::ET_START:
            m_swipe.running = false;
            m_drag.start = FeatStd::Float(gestureEvent._pt1.y);
            m_drag.offsetStart = m_offset;
            break;

         case GestureEvent::ET_MOVE:
            SetOffset((m_drag.start - FeatStd::Float(gestureEvent._pt1.y)) + m_drag.offsetStart);
            break;

         case GestureEvent::ET_END:
            m_viewport = 0;
            break;

         default:
            break;
      }
   }
   return true;
}


bool ScrollableRichTextWidget2D::OnSwipeGesture(const hmibase::input::gesture::GestureEvent& gestureEvent)
{
   if (gestureEvent._isPoint1VelocityValid)
   {
      switch (gestureEvent._gestureState)
      {
         case GestureEvent::ET_END:
         {
            m_swipe.running = true;
            m_swipe.direction = gestureEvent._velocity1.y > 0 ? -1.0F : +1.0F;
            m_swipe.velocityStart = FeatStd::Internal::Math::Absolute(FeatStd::Float(gestureEvent._velocity1.y));
            m_swipe.timeStart = hmibase::utils::Ticker::getTickCountMsec();
            m_swipe.offsetStart = m_offset;
            m_swipe.acceleration = -1.0F * FeatStd::Float(GetSwipeDeceleration());
            m_swipe.duration = m_swipe.velocityStart / abs(m_swipe.acceleration);
            FeatStd::Float distance = m_swipe.velocityStart * m_swipe.duration + (0.5F * m_swipe.acceleration * m_swipe.duration * m_swipe.duration);
            FeatStd::Float maxDistance = FeatStd::Float(GetSwipeMaxDistance());
            if ((maxDistance > 0) && (distance > maxDistance))
            {
               m_swipe.acceleration *= distance / maxDistance;
               m_swipe.duration = m_swipe.velocityStart / abs(m_swipe.acceleration);
            }
            break;
         }

         default:
            m_swipe.running = false;
            break;
      }
   }
   return true;
}


void ScrollableRichTextWidget2D::RequestImmediatePositioning(bool active)
{
}


bool ScrollableRichTextWidget2D::IsScrollBarVisible()
{
   return (m_docHeight > m_viewportSize.GetY());
}


FeatStd::UInt32 ScrollableRichTextWidget2D::GetPosition()
{
   return FeatStd::UInt32(m_offset);
}


FeatStd::UInt32 ScrollableRichTextWidget2D::GetFirstFullyVisiblePosition()
{
   return FeatStd::UInt32(m_offset);
}


FeatStd::UInt32 ScrollableRichTextWidget2D::GetVisibleSize()
{
   return FeatStd::UInt32(m_viewportSize.GetY());
}


FeatStd::UInt32 ScrollableRichTextWidget2D::GetCompleteSize()
{
   FeatStd::UInt32 lastLineOffset = 0;
   FeatStd::UInt32 visibleHeight = GetVisibleSize();

   if (m_lineHeight != 0)
   {
      lastLineOffset = visibleHeight % FeatStd::UInt32(m_lineHeight);
   }

   return (FeatStd::UInt32(m_docHeight) > visibleHeight) ? FeatStd::UInt32(m_docHeight) + lastLineOffset : visibleHeight;
}


FeatStd::UInt32 ScrollableRichTextWidget2D::GetMaxPosition()
{
   return GetCompleteSize() - FeatStd::UInt32(m_viewportSize.GetY());
}


bool ScrollableRichTextWidget2D::OnListChangeMsg(const ListChangeMsg& listChangeMsg)
{
   bool consumed = false;

   FeatStd::UInt32 listId = listChangeMsg.GetListId();
   if ((listId == 0) || (listId == GetScrollId()))
   {
      m_swipe.running = false;

      FeatStd::Int32 value = listChangeMsg.GetValue();
      switch (listChangeMsg.GetListChangeType())
      {
         case ListChangeSet:
            SetOffset(FeatStd::Float(value));
            break;

         case ListChangePageUp:
            PageMove(-value);
            break;

         case ListChangePageDown:
            PageMove(value);
            break;

         case ListChangeUp:
            LineMove(-value);
            break;

         case ListChangeDown:
            LineMove(value);
            break;

         default:
            break;
      }
      consumed = true;
      if (m_listener != 0)
      {
         m_listener->OnPositionReached();
      }
   }

   return consumed;
}


void ScrollableRichTextWidget2D::SetOffset(FeatStd::Float offset)
{
   if (m_viewport != 0)
   {
      Candera::Math::SetToMin(0.0F, offset);
      Candera::Math::SetToMax(FeatStd::Float(GetCompleteSize()) - m_viewportSize.GetY(), offset);
      m_offset = offset;
      m_viewport->SetOffset(0.0F, -m_offset);

      if (m_listener != 0)
      {
         m_listener->OnContentUpdated();
      }

      Engine* engine = DynamicProperties::GetEngine(GetNode());
      if (engine != 0)
      {
         engine->GetRenderer()->Render();
      }

      Invalidate();
   }
}


void ScrollableRichTextWidget2D::PageMove(FeatStd::Int pageOffset)
{
   SetOffset(AlignFirstLine(m_offset + (FeatStd::Float(pageOffset) * m_viewportSize.GetY())));
}


void ScrollableRichTextWidget2D::LineMove(FeatStd::Int lineOffset)
{
   SetOffset(AlignFirstLine(m_offset + (FeatStd::Float(lineOffset) * m_lineHeight)));
}


FeatStd::Float ScrollableRichTextWidget2D::AlignFirstLine(FeatStd::Float offset)
{
   return FeatStd::Float(FeatStd::Int(offset / m_lineHeight) * m_lineHeight);
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
