/* ***************************************************************************************
* FILE:          RichTextHighlightWidget2D.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RichTextHighlightWidget2D is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RichTextHighlightWidget2D.h"
#include "RichTextWidget2D.h"
#include <Widgets/2D/ControlTemplate/ControlTemplateBinding.h>
#include <Widgets/2D/RichText/Engine/RtDynamicProperties.h>
#include <Widgets/2D/WidgetFinder2D.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RichTextHighlightWidget2D.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {


CGI_WIDGET_RTTI_DEFINITION(RichTextHighlightWidget2D);


RichTextHighlightWidget2D::RichTextHighlightWidget2D() :
   m_update(false)
{
}


RichTextHighlightWidget2D::~RichTextHighlightWidget2D()
{
}


void RichTextHighlightWidget2D::Update()
{
   Base::Update();

   if (m_update)
   {
      Engine* engine = DynamicProperties::GetEngine(GetNode());
      if (engine != 0)
      {
         DocAccessor::SharedPointer docAccessor = engine->GetDocAccessor();
         if (docAccessor != 0)
         {
            Document::SharedPointer doc = docAccessor->GetDocument();
            if (doc != 0)
            {
               engine->SetHighlighter();
               Highlighter::SharedPointer highlighter = engine->GetHighlighter();
               if (highlighter != 0)
               {
                  ElementStyleAccessor& styleAccessor = ElementStyleAccessor::GetInstance();
                  ElementStyle::SharedPointer style = styleAccessor.GetStyle("Highlight");
                  if (style != 0)
                  {
                     style->RemoveAllParameters();
                  }
                  else
                  {
                     style = styleAccessor.CreateStyle("Highlight");
                  }

                  if (style != 0)
                  {
                     if (GetTextColorEnabled())
                     {
                        (void)style->AddParameter(StyleParameterBase::TextColor, GetTextColor());
                     }
                     if (GetBackgroundColorEnabled())
                     {
                        (void)style->AddParameter(StyleParameterBase::BackgroundColor, GetBackgroundColor());
                     }
                  }

                  Highlighter::Data data;
                  data.m_style = style;
                  data.m_mode = GetHighlightMode();
                  data.m_text = GetHighlightText();
                  data.m_allOccurrences = GetAllOccurrences();
                  data.m_caseSensitive = GetCaseSensitive();

                  highlighter->SetData(data);
                  highlighter->Update();
                  InvalidateLayout();
                  m_update = false;
               }
            }
         }
      }
   }
}


bool RichTextHighlightWidget2D::CloneFrom(const ControlTemplateCloneableWidget* originalWidget, ControlTemplateMap& controlTemplateMap)
{
   bool cloned(false);
   if (Base::CloneFrom(originalWidget, controlTemplateMap))
   {
      const RichTextHighlightWidget2D* original = CLONEABLE_WIDGET_CAST<const RichTextHighlightWidget2D*>(originalWidget);
      if (original != 0)
      {
         SetHighlightMode(original->GetHighlightMode());
         SetHighlightText(original->GetHighlightText());
         SetAllOccurrences(original->GetAllOccurrences());
         SetCaseSensitive(original->GetCaseSensitive());
         SetTextColorEnabled(original->GetTextColorEnabled());
         SetTextColor(original->GetTextColor());
         SetBackgroundColorEnabled(original->GetBackgroundColorEnabled());
         SetBackgroundColor(original->GetBackgroundColor());

         cloned = true;
      }
   }
   return cloned;
}


void RichTextHighlightWidget2D::OnChanged(::FeatStd::UInt32 propertyId)
{
   Base::OnChanged(propertyId);

   switch (propertyId)
   {
      case HighlightModePropertyId:
      case HighlightTextPropertyId:
      case AllOccurrencesPropertyId:
      case CaseSensitivePropertyId:
      case TextColorEnabledPropertyId:
      case TextColorPropertyId:
      case BackgroundColorEnabledPropertyId:
      case BackgroundColorPropertyId:
         m_update = true;
         break;

      default:
         break;
   }
}


void RichTextHighlightWidget2D::OnNodeChanged()
{
   Base::OnNodeChanged();
   Invalidate();
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
