/* ***************************************************************************************
* FILE:          RtTextRenderTraverser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtTextRenderTraverser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtTextRenderTraverser.h"
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/BitmapTextBrushCache.h>
#include <Widgets/2D/RichText/DocumentModel/RtSpace.h>
#include <Widgets/2D/RichText/StyleProvider/RtStyleTypes.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RtTextRenderTraverser.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera::TextRendering;
using FeatStd::Float;


TextRenderTraverser::TextRenderTraverser(Engine& engine, const Candera::Bitmap::SharedPointer& bitmap,
      FeatStd::Float offsetY, FeatStd::Float documentWidth) :
   Base(engine),
   m_renderContext(Candera::Internal::BitmapTextBrushCache().GetTextRenderContext()),
   m_rectangle(0.0F, offsetY, bitmap->GetWidth(), bitmap->GetHeight()),
   m_documentWidth(documentWidth)
{
   if (!m_renderContext.SetBitmap(bitmap))
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RenderTextTraverser: m_renderContext.SetBitmap() failed."));
   }
}


bool TextRenderTraverser::ProcessSpace(const Space& space, const Candera::Rectangle& effectiveRect)
{
   if (!IsWithinBoundaries(effectiveRect))
   {
      return false;
   }

   bool truncate = false;

   Candera::TextRendering::SharedStyle::SharedPointer style;
   if (space.GetStyleParameter(StyleParameterBase::FontStyle, style))
   {
      truncate = HandleTruncation(effectiveRect, space, style);
      if (truncate)
      {
         (void)ProcessText(space, effectiveRect);
      }
   }

   return truncate;
}


bool TextRenderTraverser::ProcessText(const Text& text, const Candera::Rectangle& effectiveRect)
{
   if (!IsWithinBoundaries(effectiveRect))
   {
      return false;
   }

   bool truncate = false;
   Candera::Color color;

   if (text.GetStyleParameter(StyleParameterBase::TextColor, color))
   {
      m_renderContext.SetColor(color);
   }

   m_renderContext.SetText(&text);

   Candera::TextRendering::SharedStyle::SharedPointer style;
   if (text.GetStyleParameter(StyleParameterBase::FontStyle, style) && (effectiveRect.GetWidth() > 0.0F))
   {
      LayoutingOptions layoutingOptions;

      TextCoordinate pos(PixelPosition(effectiveRect.GetLeft()), PixelPosition(effectiveRect.GetTop() - m_rectangle.GetTop()));
      layoutingOptions.SetOffset(pos);

      m_renderContext.SetStyle(style);

      const litehtml::el_text::Context::SharedPtr& context = text.GetContext();
      if (context != 0)
      {
         const TruncationTextContext* textContext = FeatStd::Internal::PointerToPointer<const TruncationTextContext*>(&(*text.GetContext()));
         truncate = HandleTruncation(effectiveRect, text, style);

         if (TextRenderer().Render(m_renderContext, layoutingOptions, ShapingOptions(style), TextProperties(textContext->GetIterator())))
         {
            if (m_renderContext.TruncationMissing() && text.GetTruncationData().m_hasMoreLines)
            {
               const TruncationTextContext* ellipsisContext = GetEllipsisContext(style);
               if (ellipsisContext != 0)
               {
                  m_renderContext.SetTruncation(false, Candera::Rectangle(), 0, BidiBaseLevel::Implicit);
                  pos.SetX(PixelPosition(pos.GetX() + PixelPosition(effectiveRect.GetWidth())));
                  layoutingOptions.SetOffset(pos);
                  if (!TextRenderer().Render(m_renderContext, layoutingOptions, ShapingOptions(style), TextProperties(ellipsisContext->GetIterator())))
                  {
                     ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessText: Render() for ellipsisContext failed."));
                  }
               }
            }
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessText: Render() for textContext failed."));
         }
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "ProcessText: text.GetContext() failed."));
      }
   }

   return truncate;
}


TextRenderTraverser::EllipsisContextList& TextRenderTraverser::GetEllipsisContextList()
{
   FEATSTD_SYNCED_STATIC_OBJECT(EllipsisContextList, s_ellipsisContextList);
   return s_ellipsisContextList;
}


bool TextRenderTraverser::IsWithinBoundaries(const Candera::Rectangle& effectiveRect) const
{
   Candera::Rectangle rect(m_rectangle);
   rect.Intersect(effectiveRect);
   return (rect.GetWidth() > 0.0F) && (rect.GetHeight() > 0.0F);
}


bool TextRenderTraverser::HandleTruncation(const Candera::Rectangle& effectiveRect, const Text& text,
      const Candera::TextRendering::SharedStyle::SharedPointer& style)
{
   bool truncate = false;
   Candera::Rectangle truncateRect(effectiveRect);
   const TruncationTextContext* ellipsisContext = 0;

   const Text::TruncationData& truncationData = text.GetTruncationData();
   if (truncationData.m_isTruncationLine)
   {
      const litehtml::el_text::Context::SharedPtr& context = text.GetContext();
      if (context != 0)
      {
         const TruncationTextContext* textContext = FeatStd::Internal::PointerToPointer<const TruncationTextContext*>(&(*context));
         TextOverflow textOverflow;
         if (text.GetStyleParameter(StyleParameterBase::TextOverflow, textOverflow) &&
               (textOverflow.m_value == TextOverflow::Ellipsis))
         {
            ellipsisContext = GetEllipsisContext(style);
         }

         const bool rightToLeft = truncationData.m_baseBidiLevel == BidiBaseLevel::RightToLeft;
         const Float left = text.GetRect().GetLeft();

         Float ellipsisWidth = (ellipsisContext != 0) ? Float((ellipsisContext)->GetTextWidth()) : 0.0F;
         if (truncationData.m_isLastInTruncationLine && truncationData.m_hasMoreLines)
         {
            truncate = true;
         }
         else
         {
            if (rightToLeft)
            {
               truncate = truncationData.m_truncationNeeded ? ((left - ellipsisWidth) < 0) : (left < 0);
            }
            else
            {
               const Float right = left + FeatStd::Float(textContext->GetTextWidth());
               const Float rightIncludingEllipsis = right + ellipsisWidth;

               truncate = truncationData.m_truncationNeeded ? (rightIncludingEllipsis > m_documentWidth) : (right > m_documentWidth);
               truncateRect.SetWidth(m_documentWidth - left);
            }
         }

         if (truncate)
         {
            truncateRect.Intersect(Candera::Rectangle(0.0F, truncateRect.GetTop(), m_documentWidth, truncateRect.GetHeight()));
         }
         else
         {
            if (!text.IsTypeOf(Space::GetTypeId()))
            {
               truncate = FeatStd::Float(textContext->GetTextWidth()) > effectiveRect.GetWidth();
            }
         }
      }
   }

   m_renderContext.SetTruncation(truncate, truncateRect, ellipsisContext, truncationData.m_elementBidiLevel);

   return truncate;
}


const TruncationTextContext* TextRenderTraverser::GetEllipsisContext(const Candera::TextRendering::SharedStyle::SharedPointer& style) const
{
   const TruncationTextContext* ellipsisContext = 0;

   const Candera::TextRendering::Font& font = style->GetDefaultFont();
   EllipsisContextList& list = GetEllipsisContextList();
   FontId fontId;
   fontId.faceId = font.GetFaceId();
   fontId.height = font.GetHeight();
   TruncationTextContext::SharedPointer* found = list.Find(fontId);
   if (found != 0)
   {
      TruncationTextContext::SharedPointer sharedPtr = *found;
      if (sharedPtr.IsValid())
      {
         ellipsisContext = &static_cast<TruncationTextContext&>(*sharedPtr);
      }
   }
   else
   {
      TruncationTextContext::SharedPointer context = TruncationTextContext::Create(Candera::Internal::BitmapTextBrushCache().GetTextRenderContext());
      TruncationTextContext& truncateTextContext = static_cast<TruncationTextContext&>(*context);
#if defined(USE_MINIMAL_TEXTRENDERER)
      ProcessChunk(truncateTextContext, ShapingOptions(style), "...");
#else
      TextRenderer().Render(truncateTextContext, LayoutingOptions(), ShapingOptions(style), TextProperties("..."));
#endif
      (void)list.Insert(fontId, context);
      ellipsisContext = &truncateTextContext;
   }

   return ellipsisContext;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
