/* ***************************************************************************************
* FILE:          RtRenderer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtRenderer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/Util/Rtti.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 5)))
#include <Candera/EngineBase/Layout/DefaultLayouter.h>
#else
#include <Candera/Engine2D/Layout/DefaultLayouter.h>
#endif
#include <CanderaPlatform/Device/Common/Effects/BitmapBrushBlend.h>
#include <Courier/Version.h>
#include <FeatStd/Async/AsyncRequest.h>
#include <FeatStd/Async/AsyncRequestDispatcher.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>
#include <Widgets/2D/RichText/Renderer/RtLayouter.h>
#include <Widgets/2D/RichText/Renderer/RtRenderBufferCache.h>
#include <Widgets/2D/RichText/Renderer/RtRenderBufferCacheItem.h>


namespace Candera {
class Node2D;
}


namespace FeatStd {
class AsyncRequestDispatcher;
}


namespace hmibase {
namespace widget {
namespace richtext {

class Engine;


/** Processes and control visual rendering of rich text documents. */
class RichTextRenderer : public Candera::CanderaObject
{
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(RichTextRenderer);

      struct MarqueeInfo
      {
         bool m_isValid;
         FeatStd::Float m_visibleWidth;
         FeatStd::Float m_totalWidth;

         MarqueeInfo();
      };

      static SharedPointer Create(Engine* engine);

      static void SetAlphaBlendFactor(Candera::BlendEffect& blendEffect);

      static FeatStd::AsyncRequestDispatcher& GetUpoadDispatcher();

      ~RichTextRenderer();

      void SetSize(FeatStd::UInt16 width, FeatStd::UInt16 height);
      void SetRootNode(Candera::Node2D* node);
      void Invalidate();
      void RenderAsync();
      void Render();

      const MarqueeInfo& GetMarqueeInfo() const;

      Candera::Vector2 OnMeasure(Candera::Node2D& node, const Candera::Vector2& clientArea);
      void OnArrange(Candera::Node2D& node, const Candera::Rectangle& clientArea);
      Engine& GetEngine();

      void OnUpdate();

   private:
      struct RenderState
      {
         FeatStd::UInt16 m_sliceHeight;
         FeatStd::UInt32 m_lastViewportOffset;
         FeatStd::UInt32 m_viewportSliceCount;
         RenderState() : m_sliceHeight(0), m_lastViewportOffset(0), m_viewportSliceCount(0) {}
      };
      RenderState m_renderState;

      class RenderToBufferRequest : public FeatStd::AsyncRequest<void>
      {
         public:
            RenderToBufferRequest(const RichTextRenderer::SharedPointer& renderer) : m_renderer(renderer) {}

         protected:
            virtual void Execute() override;

         private:
            RichTextRenderer::SharedPointer m_renderer;
      };

      class PrepareDocumentRequest : public FeatStd::AsyncRequest<Document::SharedPointer>
      {
         public:
            PrepareDocumentRequest(const RichTextRenderer::SharedPointer& renderer, const Candera::Vector2& clientArea) :
               m_renderer(renderer),
               m_clientArea(clientArea)
            {
            }

         protected:
            virtual void Execute() override;

         private:
            RichTextRenderer::SharedPointer m_renderer;
            Candera::Vector2 m_clientArea;
      };

      class UpdateRenderNodesRequest : public FeatStd::AsyncRequest<void>
      {
         public:
            UpdateRenderNodesRequest(const RichTextRenderer::SharedPointer& renderer) :
               m_renderer(renderer)
            {
            }

         protected:
            virtual void Execute() override;

         private:
            RichTextRenderer::SharedPointer m_renderer;
      };

      FeatStd::Internal::CriticalSection m_bitmapCs;
      Layouter m_layouter;
      RenderBufferCache m_renderBufferCache;
      Candera::Vector2 m_preferredSize;
      Candera::BitmapBrushBlend::SharedPointer m_effect;
      PrepareDocumentRequest::SharedPointer m_prepareDocumentRequest;
      MarqueeInfo m_marqueeInfo;
      Engine& m_engine;
      Candera::Node2D* m_rootNode;
      Candera::Group2D* m_layoutGroup;
      Candera::Group2D* m_textGroup;
      Candera::Group2D* m_decoGroup;
      FeatStd::Float m_documentWidth;
      FeatStd::Float m_renderedWidth;
      FeatStd::UInt16 m_width;
      FeatStd::UInt16 m_height;
      bool m_sizeChanged;
      bool m_updateDocument;
      bool m_updateBitmaps;
      bool m_rerender;

      static FeatStd::AsyncRequestDispatcher& GetAsyncRequestDispatcher();
      RichTextRenderer(Engine& engine);
      void CleanupChildNodes();
      /// Provides the maximum height in pixels of a single render partition, the provided value is never less than a defined minimum size.
      FeatStd::UInt16 GetRenderSliceHeight();
      /// Provides the number of slices to at minimum fully cover the viewport
      FeatStd::UInt32 GetViewportRenderSliceCount();
      /// Tests whether or not the given width or height exceeds the maximum texture size and clips the parameters to the maximum values if needed.
      void MitigatTextureOversize(FeatStd::UInt16& width, FeatStd::UInt16& height) const;
      /// Creates all required render buffers in m_renderBuffers incl. bitmaps for partial and full-size render modes.
      void CreateRenderBuffers();
      Candera::Group2D* AddNewGroupNode(Candera::Group2D* parent, const FeatStd::Char* newGroupName);
      /// Provides a render node that can be used for rendering a text slice. The node is attached to the scene tree structure.
      /// A node is created if no unused instance is available.
      Candera::RenderNode*  GetTextSliceNode();
      // Creates a new render node for rendering text slices with the given name.
      Candera::RenderNode* CreateTextSliceRenderNode(const FeatStd::Char* nodeName);

      void PrepareDocument(const Candera::Vector2& clientArea);
      void PrepareDocumentAsync(const Candera::Vector2& clientArea);
      void DoPrepareDocument(const Candera::Vector2& clientArea);
      void RenderDocument();
      void DoRender();
      /// Triggers text slice render node updates, either synchronously or asynchronously
      void UpdateTextSliceNodes();
      /// Performs updating the render nodes.
      void DoUpdateTextSliceNodes();
      void DoUpdateTextSliceNode(Candera::RenderNode* node, RenderBufferCacheItem::SharedPointer renderBuffer);

      FeatStd::SizeType GetCameraCount() const;
      void Activate(FeatStd::SizeType cameraIndex) const;
      void UploadImage2DInternal(Candera::BitmapBrush* brush) const;

      FeatStd::UInt32 TransparentWhitePixel();

      FEATSTD_SHARED_POINTER_DECLARATION();
};


inline
RichTextRenderer::MarqueeInfo::MarqueeInfo() :
   m_isValid(false),
   m_visibleWidth(0.0F),
   m_totalWidth(0.0F)
{
}


inline
const RichTextRenderer::MarqueeInfo& RichTextRenderer::GetMarqueeInfo() const
{
   return m_marqueeInfo;
}


inline
Engine& RichTextRenderer::GetEngine()
{
   return m_engine;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
