/* ***************************************************************************************
* FILE:          RtRenderer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtRenderer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "RtRenderer.h"
#include "RtDecorationRenderTraverser.h"
#include "RtTextRenderTraverser.h"
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/EngineBase/Common/PixelBakery.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Courier/Visualization/ViewScene2D.h>
#include <FeatStd/Async/AsyncRequestDispatcher.h>
#include <FeatStd/Util/StaticObject.h>
#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/Engine/RtDynamicProperties.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/RtRenderer.cpp.trc.h"
#endif

// Local debug and test overrides
// #define VISUAL_TRACE_ENABLED true

namespace hmibase {
namespace widget {
namespace richtext {

using namespace FeatStd;
using namespace Candera;
using namespace Candera::TextRendering;

FEATSTD_RTTI_BASECLASS_DEFINITION(RichTextRenderer);

static const FeatStd::Char* c_rtLayoutGroupName = "RT_LayoutGroup";
static const FeatStd::Char* c_rtTextGroupName = "RT_TextGroup";
static const FeatStd::Char* c_rtDecoGroupName = "RT_DecoGroup";
static const FeatStd::Char* c_rtTextRenderNodeName = "RT_TextNode";


RichTextRenderer::SharedPointer RichTextRenderer::Create(Engine* engine)
{
   return RichTextRenderer::SharedPointer(FEATSTD_NEW(RichTextRenderer)(*engine));
}


void RichTextRenderer::SetAlphaBlendFactor(BlendEffect& blendEffect)
{
   blendEffect.AlphaSourceBlendFactor().Set(RenderDevice2D::InverseDestAlpha);
   blendEffect.AlphaDestinationBlendFactor().Set(RenderDevice2D::One);
}


FeatStd::AsyncRequestDispatcher& RichTextRenderer::GetUpoadDispatcher()
{
   FEATSTD_SYNCED_STATIC_OBJECT(FeatStd::AsyncRequestDispatcher, s_asyncRequestDispatcher, false);
   return s_asyncRequestDispatcher;
}


FeatStd::AsyncRequestDispatcher& RichTextRenderer::GetAsyncRequestDispatcher()
{
   FEATSTD_SYNCED_STATIC_OBJECT(FeatStd::AsyncRequestDispatcherWorkerThread, s_asyncRequestDispatcherWorkerThread);
   static bool init = true;
   if (init)
   {
      s_asyncRequestDispatcherWorkerThread.Start();
      init = false;
   }

   return s_asyncRequestDispatcherWorkerThread.GetDispatcher();
}


RichTextRenderer::RichTextRenderer(Engine& engine) :
   m_bitmapCs(),
   m_layouter(*this),
   m_renderBufferCache(),
   m_preferredSize(0.0F, 0.0F),
   m_engine(engine),
   m_rootNode(0),
   m_layoutGroup(0),
   m_textGroup(0),
   m_decoGroup(0),
   m_documentWidth(0.0F),
   m_renderedWidth(0.0F),
   m_width(0),
   m_height(0),
   m_sizeChanged(false),
   m_updateDocument(false),
   m_updateBitmaps(false),
   m_rerender(false)
{
}


RichTextRenderer::~RichTextRenderer()
{
   Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(m_rootNode);
   if (renderNode != 0)
   {
      renderNode->SetLayouter(0);
   }
   CleanupChildNodes();
   m_layoutGroup = 0;
   m_textGroup = 0;
   m_decoGroup = 0;
   m_rootNode = 0;
}


void RichTextRenderer::SetSize(FeatStd::UInt16 width, FeatStd::UInt16 height)
{
   if ((width != m_width) || (height != m_height))
   {
      m_sizeChanged = true;
      m_updateDocument = true;
      m_width = width;
      m_height = height;
   }
}


void RichTextRenderer::SetRootNode(Candera::Node2D* node)
{
   // if root node is Group2D the own render node with its own effect is used
   // otherwise the render node with its attached effect (could also be e.g. box shadow effect) is used

   // previous node is stored in m_rootNode -> clean up
   if (m_rootNode != 0)
   {
      // remove all child nodes from managed group node
      CleanupChildNodes();
      m_rootNode->SetLayouter(DefaultLayouter::GetInstance());
   }

   // store new node to m_rootNode and set up accordingly
   m_rootNode = node;

   // prepare the new root node
   if (m_rootNode != 0)
   {
      Candera::Group2D* rootGroup = Dynamic_Cast<Group2D*>(m_rootNode);
      if (rootGroup != 0)
      {
         // node structure required by renderer:

         // m_rootNode                       associated with RT-Widget
         // │
         // ├── m_layoutGroup                group containing all nodes controlled by this renderer
         // │   │
         // │   ├── m_textGroup              group containing all nodes rendering text slices
         // │   │   │
         // │   │   ├── textSliceNode1
         // │   │   ├── textSliceNode2
         // │   │   └── ...
         // │   │
         // │   └── m_decoGroup              group containing all non-text document content, such as images or backgrounds
         // │       │
         // │       ├── decoNode1
         // │       ├── decoNode2
         // │       └── ...
         // │
         // └── ...                          other customer applied nodes (left untouched)

         // remove layout group node and all its child nodes to set up a clean scene tree
         Node2D* child = m_rootNode->GetFirstChild();
         while ((child != 0) && (child->GetName() != c_rtLayoutGroupName))
         {
            child = child->GetNextSibling();
         }
         if (child != 0)
         {
            // the found RT-root node
            m_layoutGroup = 0;
            m_textGroup = 0;
            m_decoGroup = 0;
            m_rootNode->RemoveChild(child);
         }

         // create the rich-text group node structure
         m_layoutGroup = AddNewGroupNode(rootGroup, c_rtLayoutGroupName);
         m_textGroup = AddNewGroupNode(m_layoutGroup, c_rtTextGroupName);
         m_decoGroup = AddNewGroupNode(m_layoutGroup, c_rtDecoGroupName);

         m_rootNode->SetLayouter(OverlayLayouter::GetInstance());

         if (m_layoutGroup != 0)
         {
            m_layoutGroup->SetLayouter(&m_layouter);
         }

         if (m_textGroup != 0)
         {
            // text render nodes are moved one level to the front to allow background color nodes to be one level behind but still before all other content.
            m_textGroup->SetRenderOrderRank(+1);
            m_textGroup->SetLayouter(OverlayLayouter::GetInstance());
         }

         if (m_decoGroup != 0)
         {
            m_decoGroup->SetLayouter(OverlayLayouter::GetInstance());
         }
      }
      else
      {
         m_rootNode->SetLayouter(&m_layouter);
      }
   }
}


void RichTextRenderer::Invalidate()
{
   if (m_engine.GetData().m_asynchronousRendering)
   {
      m_engine.PrepareDocument();
   }
   m_updateDocument = true;
   m_rerender = true;
}


UInt32 RichTextRenderer::TransparentWhitePixel()
{
   typedef Candera::Internal::PixelBakery::RgbaChannels Channels;
   typedef Candera::MemoryManagement::Internal::HostEndianness Endianness;
   typedef Candera::Internal::PixelBakery::ByteChannelPixel<Channels, Endianness> Pixel;
   UInt32 pixel = 0;
#ifndef VISUAL_TRACE_ENABLED
   (void)Pixel::Compose(*FeatStd::Internal::PointerToPointer<Pixel::BaseType*>(&pixel), 255, 255, 255, 0);
#else
   static UInt32 counter(0);
   UInt32 step = counter % 6;
   UInt32 col = 63 * (((counter / 6) % 2) + 1);
   UInt8 r = UInt8((step == 0 || step == 3 || step == 4) ? col : 0);
   UInt8 g = UInt8((step == 1 || step == 3 || step == 5) ? col : 0);
   UInt8 b = UInt8((step == 2 || step == 4 || step == 5) ? col : 0);
   UInt8 a = UInt(128);
   counter++;
   (void)Pixel::Compose(*FeatStd::Internal::PointerToPointer<Pixel::BaseType*>(&pixel), r, g, b, a);
#endif
   return pixel;
}


void RichTextRenderer::RenderAsync()
{
   static bool init = true;
   if (init)
   {
      UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
      FEATSTD_DEBUG_ASSERT(updateSystem != 0);
      if (updateSystem != 0)
      {
         UpdateSystem::Handle updateComponent = updateSystem->CreateComponent();
         bool success = updateSystem->AttachComponent(updateComponent, this);
         FEATSTD_DEBUG_ASSERT(success);
         if (success)
         {
            UpdateSystem::Delegate delegate = UpdateSystem::Delegate::Update<RichTextRenderer, &RichTextRenderer::OnUpdate>();
            success = updateSystem->SetComponentUpdateDelegate(updateComponent, delegate);
            FEATSTD_DEBUG_ASSERT(success);
         }
      }
      init = false;
   }

   RenderToBufferRequest::SharedPointer renderRequest = RenderToBufferRequest::SharedPointer(FEATSTD_NEW(RenderToBufferRequest)(SharedPointer(this)));
   GetAsyncRequestDispatcher().Schedule(renderRequest);
}


void RichTextRenderer::Render()
{
   if (m_engine.GetData().m_asynchronousRendering)
   {
      RenderAsync();
   }
   else
   {
      DoRender();
   }
}


void RichTextRenderer::DoRender()
{
   if (m_sizeChanged)
   {
      CreateRenderBuffers(); // fills m_renderBuffers with blank bitmaps
      m_sizeChanged = false;
   }

   DocAccessor::SharedPointer docAccessor = m_engine.GetDocAccessor();
   if (docAccessor != 0)
   {
      Document::SharedPointer doc = docAccessor->GetDocument();
      if (doc != 0)
      {
         RICHTEXT_LOG_BEGIN(stream);
         RICHTEXT_LOG(stream, *doc);
         RICHTEXT_LOG_END(stream);

         _TODO("Render the currently visible slices first (only)");

         RenderBufferCacheItem::SharedPointer cacheItem;
         FeatStd::Float offsetY = 0.0F;
         for (UInt32 idx = 0; m_updateBitmaps && m_renderBufferCache.TestAndGet(idx, cacheItem); idx++)
         {
            if (cacheItem != 0)
            {
               Candera::Bitmap::SharedPointer bufferBitmap = cacheItem->GetBitmap();

               if ((bufferBitmap != 0) && (bufferBitmap->GetPixels() != 0))
               {
                  // clear the bitmap to transparent white to get expected blend behavior with current implementation in Candera::Internal::PixelBakery::Blender::Blend
#ifndef VISUAL_TRACE_ENABLED
                  static const UInt32 backgroundPixel(TransparentWhitePixel());
#else
                  const UInt32 backgroundPixel(TransparentWhitePixel());
#endif

                  UInt32* ptr = aligned_cast<UInt32*>(Bitmap::PixelsResource(bufferBitmap->GetPixelsResourceHandle()).GetMutableData());
                  const UInt32 memorySize = bufferBitmap->GetSize();
                  const UInt32* end = ptr + (memorySize / 4);
                  while (ptr < end) //lint !e946 Relational or subtract operator applied to pointers
                  {
                     *ptr = backgroundPixel;
                     ptr++;
                  }

                  ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): rendering slice #%03i/%03i  (w: %i, h: %i)\n",
                                      this, FeatStd::Internal::Thread::GetCurrentId(), idx, m_renderBufferCache.GetCount(), bufferBitmap->GetWidth(), bufferBitmap->GetHeight()));

                  TextRenderTraverser renderTextTraverser(m_engine, bufferBitmap, offsetY, doc->GetRect().GetWidth());
                  (void)doc->Process(renderTextTraverser, doc->GetRect());
                  cacheItem->SetUploadPending();

                  // trigger updates of rendered slices
                  UpdateTextSliceNodes();
                  // end of bitmap lock scope

                  offsetY += bufferBitmap->GetHeight();
               }
            }
         }

         m_updateBitmaps = false;

         if (m_decoGroup != 0)
         {
            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): rendering deco\n", this, FeatStd::Internal::Thread::GetCurrentId()));

            DecorationRenderTraverser renderDecoTraverser(m_engine, *m_decoGroup);
            (void)doc->Process(renderDecoTraverser, doc->GetRect());
         }

         // trigger updates of rendered slices
         UpdateTextSliceNodes();
      }
   }
}


void RichTextRenderer::CleanupChildNodes()
{
   Candera::Group2D* groupNode = Candera::Dynamic_Cast<Candera::Group2D*>(m_rootNode);
   if (0 != groupNode)
   {
      Candera::Node2D* child = groupNode->GetFirstChild();
      while (0 != child)
      {
         Candera::Node2D* nextSibling = child->GetNextSibling();
         (void)groupNode->RemoveChild(child);
         child->Dispose();
         child = nextSibling;
      }
   }
}


UInt16 RichTextRenderer::GetRenderSliceHeight()
{
   UInt16 dummy = m_width;
   UInt16 sliceHeight = 0;
   const Engine::Data& data = GetEngine().GetData();

   // determine the number of sliced text nodes needed to cover the viewport completely
   if (!data.m_slicedRendering || (m_renderState.m_sliceHeight == 0) || (m_renderState.m_viewportSliceCount == 0))
   {
      sliceHeight = m_height; // full height
      m_renderState.m_sliceHeight = sliceHeight; // default - no sliced rendering
      m_renderState.m_viewportSliceCount = 1;  // default - no sliced rendering

      if (data.m_slicedRendering)
      {
         Viewport::SharedPointer viewport = m_engine.GetViewport();
         if (viewport != 0)
         {
            UInt16 vpHeight = UInt16(viewport->GetSize().GetY());
            sliceHeight = data.m_sliceHeight == 0 ? vpHeight : data.m_sliceHeight;
            m_renderState.m_sliceHeight = sliceHeight;
            m_renderState.m_viewportSliceCount = (vpHeight / sliceHeight) + (vpHeight % sliceHeight == 0 ? 0 : 1); // readjust the minimum slice count
         }
      }
   }
   else
   {
      sliceHeight = m_renderState.m_sliceHeight;
      MitigatTextureOversize(dummy, sliceHeight);
   }
   return sliceHeight;
}


UInt32 RichTextRenderer::GetViewportRenderSliceCount()
{
   if ((m_renderState.m_sliceHeight == 0) || (m_renderState.m_viewportSliceCount == 0))
   {
      (void)GetRenderSliceHeight(); // calculate the metrics
   }
   return m_renderState.m_viewportSliceCount;
}


void RichTextRenderer::MitigatTextureOversize(UInt16& width, UInt16& height) const
{
   static UInt32 maxTextureSize = 0;
   if (maxTextureSize == 0)
   {
      const DevicePackageDescriptor::RenderDevice3DCapabilities* cap;
      DevicePackageDescriptor::GetRenderDevice3DCapabilities(DevicePackageDescriptor::DefaultAdapter, DevicePackageDescriptor::TargetDevice, cap);
      FEATSTD_DEBUG_ASSERT(cap != 0);
      if (cap != 0)
      {
         maxTextureSize = cap->m_maxTextureSize;
      }
   }

   if (width > maxTextureSize)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RichTextRenderer::CreateBitmap: Maximum texture width exceeded. %d -> %d", m_width, maxTextureSize));
      width = UInt16(maxTextureSize);
   }

   if (height > maxTextureSize)
   {
      ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RichTextRenderer::CreateBitmap: Maximum texture height exceeded. %d -> %d", m_height, maxTextureSize));
      height = UInt16(maxTextureSize);
   }
}


void RichTextRenderer::CreateRenderBuffers()
{
   const UInt16 nominalSliceHeight = GetRenderSliceHeight();
   UInt32 totalSliceHeight = 0;
   m_renderBufferCache.Clear();

   while (totalSliceHeight < m_height)
   {
      // take only the remaining height if less than a slice is left over
      UInt32 remainingHeight = m_height - totalSliceHeight;
      UInt16 sliceHeight = remainingHeight > nominalSliceHeight ? nominalSliceHeight : UInt16(remainingHeight);
      const UInt32 bufferSize = FeatStd::UInt32(m_width) * sliceHeight * 4U;
      UInt8* pixelBuffer = FEATSTD_NEW_ARRAY(UInt8, bufferSize);
      Candera::Bitmap::SharedPointer sliceBitmap;

      // add empty placeholder bitmap for unavailable pixel buffers to ensure the right number of slices in the buffer
      UInt16 width = m_width;
      MitigatTextureOversize(width, sliceHeight);
      sliceBitmap = Bitmap::Create(m_width, sliceHeight,
                                   Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1,
                                   pixelBuffer, Bitmap::Disposer::Dispose, (pixelBuffer != 0 ? bufferSize : 0));
      m_renderBufferCache.Add(sliceBitmap);
      totalSliceHeight += sliceHeight;

      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): created slice buffer #%03i, h: %03i, ttl-h: %05i\n",
                          this, FeatStd::Internal::Thread::GetCurrentId(), m_renderBufferCache.GetCount(), sliceHeight, totalSliceHeight));
   }
}


Candera::Group2D* RichTextRenderer::AddNewGroupNode(Group2D* parent, const Char* newGroupName)
{
   Group2D* newGroup = 0;
   if (parent != 0)
   {
      newGroup = Group2D::Create();
      if (newGroup != 0)
      {
         newGroup->SetName(newGroupName);
         parent->AddChild(newGroup);
      }
   }
   return newGroup;
}


Candera::RenderNode* RichTextRenderer::GetTextSliceNode()
{
   // intended for encapsulating node creation and caching as needed.
   Candera::RenderNode* node = CreateTextSliceRenderNode(c_rtTextRenderNodeName);
   (void)m_textGroup->AddChild(node);
   return node;
}


Candera::RenderNode* RichTextRenderer::CreateTextSliceRenderNode(const Char* nodeName)
{
   Candera::RenderNode* renderNode = RenderNode::Create();
   if (renderNode != 0)
   {
      renderNode->SetName(nodeName);

      m_effect = BitmapBrushBlend::Create();
      if (m_effect != 0)
      {
         SetAlphaBlendFactor(m_effect->GetBlendEffect());
         if (m_effect->Upload())
         {
            if (!renderNode->AddEffect(m_effect.GetPointerToSharedInstance()))
            {
               ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "DrawBackgroundBoxes: renderNode->AddEffect() failed."));
            }
         }
         else
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "RichTextRenderer::C'tor: m_effect->Upload() failed."));
         }
      }
   }
   return renderNode;
}


void RichTextRenderer::PrepareDocument(const Candera::Vector2& clientArea)
{
   if (m_engine.GetData().m_asynchronousRendering)
   {
      PrepareDocumentAsync(clientArea);
   }
   else
   {
      DoPrepareDocument(clientArea);
   }
}


void RichTextRenderer::PrepareDocumentAsync(const Candera::Vector2& clientArea)
{
   if (m_prepareDocumentRequest == 0)
   {
      m_prepareDocumentRequest =
         PrepareDocumentRequest::SharedPointer(FEATSTD_NEW(PrepareDocumentRequest)(SharedPointer(this), clientArea));
      GetAsyncRequestDispatcher().Schedule(m_prepareDocumentRequest);
   }
   m_preferredSize = clientArea;
}


void RichTextRenderer::DoPrepareDocument(const Candera::Vector2& clientArea)
{
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): !!! START updating document ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));

   const Engine::Data& data = m_engine.GetData();
   if (data.m_source != 0)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "Parse Text='%100s'", data.m_source->ReadCompleteText().GetCString()));
   }

   Viewport::SharedPointer viewport = m_engine.GetViewport();
   if (viewport != 0)
   {
      Candera::Vector2 size(clientArea);
      if (DynamicProperties::IsMarqueeEnabled(m_rootNode))
      {
         // for marquee animation the whole text has to be rendered to the image to allow scrolling
         size.SetX(FeatStd::Internal::Limits<FeatStd::Float>::Max());
      }
      viewport->SetSize(size);
   }

   if (m_updateDocument)
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... creating document ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));
      m_engine.CreateDocument();
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... created document ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));

      m_updateBitmaps = true;

      DocAccessor::SharedPointer docAccessor;
      docAccessor = m_engine.GetDocAccessor();
      if (docAccessor != 0)
      {
         Document::SharedPointer doc = docAccessor->GetDocument();
         if (doc != 0)
         {
            m_preferredSize = doc->GetRect().GetSize();
            SetSize(static_cast<UInt16>(clientArea.GetX()), static_cast<UInt16>(m_preferredSize.GetY()));
         }
      }
   }
}


void RichTextRenderer::RenderDocument()
{
   _TODO("Cleanup nodes and bitmaps upon doc update");

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... rendering ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));
   Render();

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... rendered ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... updating slice nodes ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));
   UpdateTextSliceNodes();

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... updated slice nodes ...\n", this, FeatStd::Internal::Thread::GetCurrentId()));

   m_updateDocument = false;
   m_rerender = false;

   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): ... updated document DONE!!!!\n", this, FeatStd::Internal::Thread::GetCurrentId()));
}


void RichTextRenderer::UpdateTextSliceNodes()
{
   if (m_engine.GetData().m_asynchronousRendering)
   {
      // asynchronous update
      if (!m_renderBufferCache.TestAndSetRenderNodeUpdateRequestPending())
      {
         UpdateRenderNodesRequest::SharedPointer updateRequest = UpdateRenderNodesRequest::SharedPointer(FEATSTD_NEW(UpdateRenderNodesRequest)(SharedPointer(
                  this)));
         RichTextRenderer::GetUpoadDispatcher().Schedule(updateRequest);
         Courier::View::WakeUpAllRenderComponents();
      }
   }
   else
   {
      // synchronous updates
      DoUpdateTextSliceNodes();
   }
}


void RichTextRenderer::DoUpdateTextSliceNodes()
{
   m_renderBufferCache.ClearRenderNodeUpdateRequestPending();
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): -> start slice nodes update\n", this, FeatStd::Internal::Thread::GetCurrentId()));

   if ((m_textGroup != 0) && (m_engine.GetViewport() != 0))
   {
      // assumption: if m_textGroup node is available, children are managed by this renderer and slicing may be enabled.

      // find render buffer bitmap to display
      Float vpOffset = m_engine.GetViewport()->GetOffset().GetY();
      UInt32 offsetY = UInt32(Candera::Math::Minimum(vpOffset, 0.0f) * -1.0); // only negative offsets (scroll down)
      UInt32 totalSliceHeight = 0;
      RenderBufferCacheItem::SharedPointer renderBuffer;
      Candera::Bitmap::SharedPointer sliceBmp;

      // find first bitmap inside viewport area
      UInt32 sliceBufferIdx = 0xFFFFFFFF;
      UInt32 idx = 0;
      UInt32 advancedTotalSliceHeight = 0;
      while ((0xFFFFFFFF == sliceBufferIdx) && m_renderBufferCache.TestAndGet(idx, renderBuffer))
      {
         if (renderBuffer != 0)
         {
            UInt32 sliceHeight = UInt32(renderBuffer->GetBitmap()->GetHeight());
            if ((offsetY < totalSliceHeight) || (offsetY >= (totalSliceHeight + sliceHeight)))
            {
               // current slice is not at in visible range (+ offset)
               totalSliceHeight = advancedTotalSliceHeight;
               advancedTotalSliceHeight += sliceHeight;
            }
            else
            {
               // matching slice buffer found
               // -> start one slice above (if possible) to provide a slice for scrolling hysteresis
               sliceBufferIdx = idx - (idx > 0 ? 1 : 0);
            }
         }
         idx++;
      }

      UInt32 nodeCnt = 0;
      UInt32 reqNodeCnt = GetViewportRenderSliceCount() + 2;
      reqNodeCnt = Candera::Math::Minimum(reqNodeCnt, m_renderBufferCache.GetCount()); // clip to document slices
      Candera::Node2D* sliceNode = m_textGroup->GetFirstChild();
      while (nodeCnt < reqNodeCnt)
      {
         if (sliceNode == 0)
         {
            sliceNode = GetTextSliceNode();
         }
         if (m_renderBufferCache.TestAndGet(sliceBufferIdx, renderBuffer))
         {
            Vector2 pos(0, Float(totalSliceHeight));
            Float sliceWidth = 0.0F;
            Float sliceHeight = 0.0F;
            if (renderBuffer != 0)
            {
               RenderBufferCacheItem::LockedBitmap lockedSliceBmp = renderBuffer->GetLockedBitmap(&m_bitmapCs);
               sliceBmp = lockedSliceBmp.GetBitmap();
               if (sliceBmp != 0)
               {
                  sliceWidth = sliceBmp->GetWidth();
                  sliceHeight = sliceBmp->GetHeight();
               }
               sliceNode->SetPosition(pos);
               DoUpdateTextSliceNode(Candera::Dynamic_Cast<Candera::RenderNode*>(sliceNode), renderBuffer);
               totalSliceHeight += UInt32(sliceHeight);
            }
            sliceBufferIdx++;
            nodeCnt++;

            ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): -> updating node #%i/%03i, top %05i, w: %03i, h: %05i bitmap:%03d vp:%05d\n",
                                this, FeatStd::Internal::Thread::GetCurrentId(), nodeCnt, sliceBufferIdx, pos.GetY(), sliceWidth, sliceHeight, sliceBufferIdx, offsetY));

            if (sliceNode != 0)
            {
               sliceNode = sliceNode->GetNextSibling();
            }
            // end of LockedBitmap scope
         }
         else
         {
            nodeCnt = reqNodeCnt;
         }
      }
   }
   else
   {
      // assumption: otherwise node is a specially authored node and slicing is never applied.
      DoUpdateTextSliceNode(Candera::Dynamic_Cast<Candera::RenderNode*>(m_rootNode), m_renderBufferCache.Get(0));
   }
   ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): -> completed slice nodes update\n", this, FeatStd::Internal::Thread::GetCurrentId()));
}


void RichTextRenderer::DoUpdateTextSliceNode(Candera::RenderNode* node, RenderBufferCacheItem::SharedPointer renderBuffer)
{
   Candera::Effect2D* effect = 0;
   Candera::BitmapBrush* bitmapBrush = 0;
   Candera::BitmapImage2D::SharedPointer image;

   if (node != 0 && renderBuffer != 0 && renderBuffer->GetBitmap() != 0 /*&& renderBuffer->IsUploadPending()*/)
   {
      renderBuffer->ClearUploadPending();
      effect = node->GetEffect(0);
   }

   if (effect != 0)
   {
      bitmapBrush = Candera::Dynamic_Cast<BitmapBrush*>(effect->GetBrushEffect2D());
   }

   if (bitmapBrush != 0)
   {
      image = Candera::BitmapImage2D::Create();
   }
   if (image != 0)
   {
      RenderBufferCacheItem::LockedBitmap lockedBmp = renderBuffer->GetLockedBitmap(&m_bitmapCs);
      if (image->SetBitmap(lockedBmp.GetBitmap()))
      {
         bitmapBrush->Image().Set(image);

         ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "%p (t:#%p): -> updating node EFFECTIVE\n", this, FeatStd::Internal::Thread::GetCurrentId()));

         const bool async = m_engine.GetData().m_asynchronousRendering;
         FeatStd::SizeType cameraCount = async ? GetCameraCount() : 1;
         for (FeatStd::SizeType i = 0; i < cameraCount; i++)
         {
            if (async)
            {
               Activate(i);
            }
            UploadImage2DInternal(bitmapBrush);
         }
         Courier::View* view = m_engine.GetData().m_view;
         if (view != 0)
         {
            view->Invalidate();
         }
      }
      else
      {
         ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "UpdateImage: image->SetBitmap() failed."));
      }
      // end of bitmap lock scope
   }
}


Candera::Vector2 RichTextRenderer::OnMeasure(Node2D& node, const Candera::Vector2& clientArea)
{
   Vector2 nodeSize = Layouter::GetSize(node);
   Float clientWidth = clientArea.GetX();
   bool marqueeEnabled = DynamicProperties::IsMarqueeEnabled(m_rootNode);
   FeatStd::Float sizeCheck = marqueeEnabled ? m_marqueeInfo.m_visibleWidth : m_documentWidth;
   bool sizeChanged = !FeatStd::Internal::Math::FloatAlmostEqual(clientWidth, sizeCheck);
   if (sizeChanged || m_updateDocument)
   {
      m_updateDocument = true;
      PrepareDocument(Vector2(clientWidth, clientArea.GetY()));
      m_documentWidth = clientWidth;
   }

   return m_preferredSize;
}


void RichTextRenderer::OnArrange(Node2D& /*node*/, const Candera::Rectangle& clientArea)
{
   Candera::Vector2 size = clientArea.GetSize();
   Float clientWidth = size.GetX();
   bool sizeChanged = !FeatStd::Internal::Math::FloatAlmostEqual(clientWidth, m_documentWidth);
   bool marqueeEnabled = DynamicProperties::IsMarqueeEnabled(m_rootNode);
   if (!marqueeEnabled && sizeChanged)
   {
      PrepareDocument(size);
      m_documentWidth = clientWidth;
   }

   if (marqueeEnabled)
   {
      m_marqueeInfo.m_visibleWidth = clientWidth;
      m_marqueeInfo.m_totalWidth = m_preferredSize.GetX();
      m_marqueeInfo.m_isValid = true;
   }

   if (m_rerender || (m_renderedWidth != m_documentWidth))
   {
      RenderDocument();
      m_renderedWidth = m_documentWidth;
   }
}


void RichTextRenderer::OnUpdate()
{
   hmibase::widget::richtext::RichTextRenderer::GetUpoadDispatcher().DispatchNext(false);
}


void RichTextRenderer::RenderToBufferRequest::Execute()
{
   m_renderer->DoRender();
}


void RichTextRenderer::PrepareDocumentRequest::Execute()
{
   m_renderer->DoPrepareDocument(m_clientArea);
   m_renderer->m_prepareDocumentRequest = PrepareDocumentRequest::SharedPointer();
}


void RichTextRenderer::UpdateRenderNodesRequest::Execute()
{
   m_renderer->DoUpdateTextSliceNodes();
}


FeatStd::SizeType RichTextRenderer::GetCameraCount() const
{
   Courier::View* view = m_engine.GetData().m_view;
   if (view != 0)
   {
      Courier::ViewScene2D* viewScene2D = view->ToViewScene2D();
      if (viewScene2D != 0)
      {
         Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
         return cameras.Size();
      }
   }
   return 1U;
}


void RichTextRenderer::Activate(FeatStd::SizeType cameraIndex) const
{
   Courier::ViewScene2D* viewScene2D = 0;
   Courier::View* view = m_engine.GetData().m_view;
   // note: flattened code due to coding guidelines requirement for nesting level.
   if (view != 0)
   {
      viewScene2D = view->ToViewScene2D();
   }
   if (viewScene2D != 0)
   {
      Courier::ViewScene2D::CameraPtrVector& cameras = viewScene2D->GetCameraPtrVector();
      if (cameraIndex < cameras.Size())
      {
         Candera::RenderTarget* renderTarget = cameras[cameraIndex]->GetRenderTarget();
         if (renderTarget != 0)
         {
            static_cast<void>(renderTarget->Activate());
         }
      }
   }
}


void RichTextRenderer::UploadImage2DInternal(Candera::BitmapBrush* brush) const
{
   if (brush != 0)
   {
      if (brush->IsUploaded())
      {
         if (!brush->Update())
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Update of BitmapBrush ID %u failed", brush->GetId()));
         }
      }
      else
      {
         if (!brush->Upload())
         {
            ETG_TRACE_ERR_DCL((APP_TRACECLASS_ID(), "Upload of BitmapBrush ID %u failed", brush->GetId()));
         }
      }
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
