/* ***************************************************************************************
* FILE:          LiteHtmlParser.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LiteHtmlParser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Widgets/2D/RichText/Engine/RtParser.h>
#include <Widgets/2D/RichText/LiteHtml/DocumentContainer.h>
#include <FeatStd/Async/AsyncRequest.h>

namespace litehtml {
class html_tag;
class el_text;
class el_space;
class el_image;
}


namespace hmibase {
namespace widget {
namespace richtext {

class Engine;
class Text;

/**
 *  LiteHtmlParser implements the Parser interface.
 */
class LiteHtmlParser : public Parser
{
   public:
      LiteHtmlParser();
      virtual ~LiteHtmlParser();

      static Candera::Color GetColor(const litehtml::web_color& webColor);

      void CreateLiteHtmlDocAsync(Engine& engine);
      litehtml::document::ptr CreateLiteHtmlDoc(const Engine& engine);
      virtual Document::SharedPointer Parse(Engine& engine) override;

   private:
      class CreateLiteHtmlDocRequest : public FeatStd::AsyncRequest<litehtml::document::ptr>
      {
            typedef FeatStd::AsyncRequest<litehtml::document::ptr> Base;

         public:
            CreateLiteHtmlDocRequest(LiteHtmlParser& parser);
            virtual ~CreateLiteHtmlDocRequest();

         protected:
            virtual void Execute() override;

         private:
            LiteHtmlParser& m_parser;
      };

      class LiteHtmlBlock : public Block
      {
            typedef Block Base;

         public:
            LiteHtmlBlock() :
               Base(),
               m_hasText(false)
            {
            }

            virtual void AddChild(const ParagraphItem::SharedPointer& item) override;

            bool HasText() const
            {
               return m_hasText;
            }

         private:
            bool m_hasText;
      };

      CreateLiteHtmlDocRequest::SharedPointer m_createLiteHtmlDocRequest;
      DocumentContainer m_documentContainer;
      Chapter* m_currentChapter;
      LiteHtmlBlock* m_currentBlock;
      Candera::Rectangle m_currentParagraphRect;
      Engine* m_engine;
      bool m_isHead;
      bool m_isTitle;


#if defined (RICHTEXT_LOGGING_ENABLED)
      void LogElementTree(std::ostream& stream, const litehtml::element::ptr& element, int indentLevel);
#endif
      void CreateDocRecusive(DocElement& docElement, litehtml::element& element);
      Document::SharedPointer CreateDocument(const litehtml::element::ptr& element);

      Chapter* AddChapter(FeatStd::Int nestedLevel, DocElement& docElement, const Candera::Rectangle& rect);
      DocElement* EnsureCurrentChapter(DocElement* docElement, litehtml::element& element);
      void EnsureCurrentBlock(DocElement& docElement, litehtml::element& element);
      void HandleBackground(DocElement& docElement, litehtml::element& element) const;

      template <typename ElementType>
      bool HandleElement(void(LiteHtmlParser::*Handler)(DocElement&, ElementType&), DocElement& docElement, litehtml::element& element)
      {
         ElementType* elem = dynamic_cast<ElementType*>(&element);
         if (elem != 0)
         {
            (this->*Handler)(docElement, *elem);
            return true;
         }
         return false;
      }

      void HandleHtmlTag(DocElement& docElement, litehtml::html_tag& element);
      void HandleSpace(DocElement& docElement, litehtml::el_space& element);
      void HandleText(DocElement& docElement, litehtml::el_text& element);
      void HandleImage(DocElement& docElement, litehtml::el_image& element);

      // The first child element (text or space) of a block has to set the style that is defined for the whole block.
      // This is just to set the style once to the block instead of each single block element (text, space...).
      // EnsureStyleOnCurrentBlock() has to be called in HandleSpace() and HandleText() to simplify this process.
      void EnsureStyleOnCurrentBlock(litehtml::el_text& element);

      void SetRect(DocElement& docElement, litehtml::element& element) const;
      void SetTruncationLine(Text& text, litehtml::element& element) const;
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
