/* ***************************************************************************************
* FILE:          LiteHtmlParser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  LiteHtmlParser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "LiteHtmlParser.h"
#include "LiteHtmlContext.h"
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/TextEngine/MaximalPreprocessingContext.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <Widgets/2D/RichText/ContentProvider/RtContentProvider.h>
#include <Widgets/2D/RichText/DocumentModel/RtImage.h>
#include <Widgets/2D/RichText/DocumentModel/RtText.h>
#include <Widgets/2D/RichText/DocumentModel/RtSpace.h>
#include <Widgets/2D/RichText/Engine/RtEngine.h>
#include <Widgets/2D/RichText/StyleProvider/RtElementStyleAccessor.h>
#include <Widgets/2D/RichText/StyleProvider/RtStyleTypes.h>
#include <Widgets/2D/RichText/Utils/RtStringInStream.h>
#include <hmibase/util/Macros.h>
#include <litehtml/src/el_image.h>
#include <litehtml/src/el_space.h>
#include <litehtml/src/el_text.h>
#include <litehtml/src/style.h>

#if defined(RICHTEXT_LOGGING_ENABLED)
#include <iomanip>
#include <typeinfo>
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/LiteHtmlParser.cpp.trc.h"
#endif

namespace hmibase {
namespace widget {
namespace richtext {

using FeatStd::UInt8;
using FeatStd::UInt16;
using FeatStd::UInt32;
using FeatStd::Float;
using FeatStd::Internal::String;
using Candera::Vector2;

static Candera::Rectangle GetElementRect(litehtml::element& element)
{
   litehtml::position pos(element.get_position());
   // width and height 0 does not work for rectangle intersection. Setting them to -1 works fine!
   if (pos.width == 0)
   {
      pos.width = -1;
   }
   if (pos.height == 0)
   {
      pos.height = -1;
   }
   return Candera::Rectangle(FeatStd::Float(pos.x), FeatStd::Float(pos.y), FeatStd::Float(pos.width), FeatStd::Float(pos.height));
}


LiteHtmlParser::LiteHtmlParser() :
   m_currentChapter(0),
   m_currentBlock(0),
   m_engine(0),
   m_isHead(false),
   m_isTitle(false)
{
}


LiteHtmlParser::~LiteHtmlParser()
{
}


Candera::Color LiteHtmlParser::GetColor(const litehtml::web_color& webColor)
{
   return Candera::Color(
             FeatStd::Float(webColor.red) / 255.0F,
             FeatStd::Float(webColor.green) / 255.0F,
             FeatStd::Float(webColor.blue) / 255.0F,
             FeatStd::Float(webColor.alpha) / 255.0F);
}


#if defined(RICHTEXT_LOGGING_ENABLED)

void LiteHtmlParser::LogElementTree(std::ostream& stream, const litehtml::element::ptr& element, int indentLevel)
{
   // print this element
   if (element.IsValid())
   {
      std::string typeName(typeid(*element).name());
      // remove everything from start to last occurrence of ":"
      size_t findPos = typeName.rfind(":");
      if (findPos != std::string::npos)
      {
         typeName.erase(0, findPos + 1);
      }

      litehtml::tstring tagName(element->get_tagName());
      if (!tagName.empty())
      {
         tagName = " (" + tagName + ")";
      }

      litehtml::tstring text;
      litehtml::el_text* el_text = dynamic_cast<litehtml::el_text*>(&(*element));
      if (el_text != 0)
      {
         el_text->get_text(text);
         text = ": '" + text + "'";
      }

      litehtml::position& pos = element->get_position();
      RICHTEXT_LOG(stream, std::setw(indentLevel) << "" << typeName << tagName << " @ " << pos.left() << "/" << pos.top() << text << std::endl);
   }

   // ...and all its children
   const size_t childCount = element->get_children_count();
   for (size_t i = 0; i < childCount; i++)
   {
      LogElementTree(stream, element->get_child(i), indentLevel + 2);
   }
}


#endif

static const UInt16 c_unlimittedWidth = FeatStd::Internal::Limits<UInt16>::Max();

LiteHtmlParser::CreateLiteHtmlDocRequest::CreateLiteHtmlDocRequest(LiteHtmlParser& parser) :
   Base(),
   m_parser(parser)
{
}


LiteHtmlParser::CreateLiteHtmlDocRequest::~CreateLiteHtmlDocRequest()
{
}


void LiteHtmlParser::CreateLiteHtmlDocRequest::Execute()
{
   SetResult(m_parser.CreateLiteHtmlDoc(*m_parser.m_engine));
}


void LiteHtmlParser::CreateLiteHtmlDocAsync(Engine& engine)
{
   FEATSTD_SYNCED_STATIC_OBJECT(FeatStd::AsyncRequestDispatcherWorkerThread, s_asyncRequestDispatcherWorkerThread);
   static bool init = true;
   if (init)
   {
      s_asyncRequestDispatcherWorkerThread.Start();
      init = false;
   }

   m_engine = &engine;
   m_createLiteHtmlDocRequest = CreateLiteHtmlDocRequest::SharedPointer(FEATSTD_NEW(CreateLiteHtmlDocRequest)(*this));
   s_asyncRequestDispatcherWorkerThread.GetDispatcher().Schedule(m_createLiteHtmlDocRequest);
}


litehtml::document::ptr LiteHtmlParser::CreateLiteHtmlDoc(const Engine& engine)
{
   litehtml::document::ptr doc;

   const Engine::Data& data = engine.GetData();
   m_documentContainer.SetDefaultStyle(data.m_textStyle);

   const TextInStream::SharedPointer& source = data.m_source;
   if (source != 0)
   {
      source->Reset();

      litehtml::css userStyle;
      static litehtml::media_query_list::ptr noMedia;
      userStyle.parse_stylesheet(data.m_style.GetCString(), 0, 0, noMedia);

      if (data.m_colorEnabled)
      {
         static const FeatStd::Int c_colorStyleStringLength = 22;
         FeatStd::Char colorStyle[c_colorStyleStringLength];
         const Candera::Color& color = data.m_color;
         litehtml::web_color webColor;
         webColor.red   = UInt8(color.GetRed() * 255.0F);
         webColor.green = UInt8(color.GetGreen() * 255.0F);
         webColor.blue  = UInt8(color.GetBlue() * 255.0F);
         if (String::StringPrintf(colorStyle, c_colorStyleStringLength, "body{color:#%02X%02X%02X}", webColor.red, webColor.green, webColor.blue) > 0)
         {
            userStyle.parse_stylesheet(colorStyle, 0, 0, noMedia);
         }
      }

      if (data.m_horizontalAlignmentEnabled)
      {
         static const FeatStd::Int c_textAlignStringLength = 30;
         FeatStd::Char textAlign[c_textAlignStringLength];
         Candera::Globalization::TextDirection textDirection =
            Candera::Globalization::CultureManager::GetInstance().GetCurrentCulture()->GetTextDirection();
         m_documentContainer.SetRightToLeftCulture(textDirection == Candera::Globalization::RightToLeft);
         m_documentContainer.SetCultureDependentAlignment(data.m_cultureDependentAlignment);

         if (String::StringPrintf(textAlign, c_textAlignStringLength, "html{text-align:%s}",
                                  (data.m_horizontalAlignment == Candera::HLeft)   ? "left" :
                                  (data.m_horizontalAlignment == Candera::HCenter) ? "center" :
                                  (data.m_horizontalAlignment == Candera::HRight)  ? "right" :
                                  "justify") > 0)
         {
            userStyle.parse_stylesheet(textAlign, 0, 0, noMedia);
         }
      }

      litehtml::context* context = LiteHtmlContext::GetContext(data.m_styleSheetUrl.GetCString());
      doc = litehtml::document::createFromUTF8(source->ReadCompleteText().GetCString(), &m_documentContainer, context,
            &userStyle);
   }

   return doc;
}


Document::SharedPointer LiteHtmlParser::Parse(Engine& engine)
{
   m_engine = &engine;
   Document::SharedPointer document;

   litehtml::document::ptr doc;
   if (m_createLiteHtmlDocRequest == 0)
   {
      doc = CreateLiteHtmlDoc(engine);
   }
   else
   {
      m_createLiteHtmlDocRequest->WaitForFinished();
      doc = m_createLiteHtmlDocRequest->GetResult();
      m_createLiteHtmlDocRequest = CreateLiteHtmlDocRequest::SharedPointer();
   }

   if (doc.IsValid())
   {
      const Engine::Data& data = m_engine->GetData();
      doc->SetMaxNumberOfLines(data.m_maxNumberOfLines);
      doc->SetBaselineOffset(data.m_baselineOffset);

      Vector2 size;
      Viewport::SharedPointer viewport = engine.GetViewport();
      if (viewport != 0)
      {
         size = viewport->GetSize();
      }
      const UInt16 width = UInt16(Candera::Math::Minimum(Float(c_unlimittedWidth), size.GetX()));
      m_documentContainer.SetClientRect(litehtml::position(0, 0, width, -1));

      FeatStd::Int preferredWidth = doc->render(width);
      litehtml::element::ptr root = doc->root();
      if (root.IsValid())
      {
         litehtml::position& pos = root->get_position();
         if ((width == c_unlimittedWidth) || (preferredWidth < pos.width))
         {
            pos.width = preferredWidth;
         }
         else if (pos.width >= width)
         {
            pos.width = width;
         }
#if defined(RICHTEXT_LOGGING_ENABLED)
         RICHTEXT_LOG_BEGIN(stream);
         RICHTEXT_LOG(stream, std::endl);
         LogElementTree(stream, doc->root(), 0);
         RICHTEXT_LOG_END(stream);
#endif

         document = CreateDocument(doc->root());

         // draw call is mandatory to get background color rectangles in DocumentContainer::draw_background
         doc->draw(0, 0, 0, 0);
      }

      m_currentBlock = 0;
      m_currentChapter = 0;
   }

   return document;
}


void LiteHtmlParser::CreateDocRecusive(DocElement& docElement, litehtml::element& element)
{
   (void)(
      // because el_image is derived from html_tag, it has to be handled first
      HandleElement(&LiteHtmlParser::HandleImage, docElement, element) ||
      HandleElement(&LiteHtmlParser::HandleHtmlTag, docElement, element) ||
      // because el_space is derived from el_text, it has to be handled first
      HandleElement(&LiteHtmlParser::HandleSpace, docElement, element) ||
      HandleElement(&LiteHtmlParser::HandleText, docElement, element)
   );
}


Document::SharedPointer LiteHtmlParser::CreateDocument(const litehtml::element::ptr& element)
{
   Document* doc = LITEHTML_NEW(Document);
   if (doc != 0)
   {
      CreateDocRecusive(*doc, *element);
#if defined(RICHTEXT_LOGGING_ENABLED)
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "CreateDocument %s", doc->m_creationStream.str().c_str()));
#endif
   }

   return Document::SharedPointer(doc);
}


Chapter* LiteHtmlParser::AddChapter(FeatStd::Int nestedLevel, DocElement& docElement, const Candera::Rectangle& rect)
{
   FeatStd::Int parentLevel = m_currentChapter != 0 ? m_currentChapter->GetNestedLevel() : -1;

   while ((parentLevel >= nestedLevel) && (m_currentChapter != 0))
   {
      m_currentChapter = dynamic_cast<Chapter*>(m_currentChapter->GetParent());
      parentLevel--;
   }

   while (parentLevel < nestedLevel)
   {
      if (m_currentChapter == 0)
      {
         m_currentChapter = dynamic_cast<Chapter*>(&docElement);
      }
      Chapter* subChapter = LITEHTML_NEW(Chapter);
      if (m_currentChapter != 0)
      {
         m_currentChapter->AddChild(Chapter::SharedPointer(subChapter));
      }
      m_currentChapter = subChapter;
      parentLevel++;
   }

   m_currentBlock = 0;
   m_isTitle = true;
   m_currentParagraphRect = rect;
   return m_currentChapter;
}


DocElement* LiteHtmlParser::EnsureCurrentChapter(DocElement* docElement, litehtml::element& element)
{
   Chapter* chapter = dynamic_cast<Chapter*>(docElement);
   if (chapter != 0)
   {
      if (m_isTitle)
      {
         Paragraph* title = LITEHTML_NEW(Paragraph)(true);
         if (title != 0)
         {
            chapter->AddTitle(Paragraph::SharedPointer(title));
            title->SetRect(m_currentParagraphRect);
            docElement = title;
         }
      }
      else
      {
         Paragraph* paragraph = LITEHTML_NEW(Paragraph);
         if (paragraph != 0)
         {
            chapter->AddChild(Paragraph::SharedPointer(paragraph));
            paragraph->SetRect(m_currentParagraphRect);
            docElement = paragraph;

            HandleBackground(*paragraph, element);
         }
      }
   }
   return docElement;
}


void LiteHtmlParser::EnsureCurrentBlock(DocElement& docElement, litehtml::element& element)
{
   if (m_currentBlock == 0)
   {
      DocElement* currentObj = (m_currentChapter != 0) ? m_currentChapter : &docElement;

      currentObj = EnsureCurrentChapter(currentObj, element);

      Paragraph* paragraph = dynamic_cast<Paragraph*>(currentObj);
      if (paragraph != 0)
      {
         LiteHtmlBlock* block = LITEHTML_NEW(LiteHtmlBlock);
         if (block != 0)
         {
            paragraph->AddChild(Block::SharedPointer(block));
            //currentObj = block;

            HandleBackground(*block, element);
         }

         m_currentBlock = block;
      }
   }

   FEATSTD_DEBUG_ASSERT(m_currentBlock != 0);
}


void LiteHtmlParser::HandleBackground(DocElement& docElement, litehtml::element& element) const
{
   litehtml::html_tag* htmlTag = dynamic_cast<litehtml::html_tag*>(&element);
   litehtml::element::ptr parent = element.parent();
   while ((htmlTag == 0) && parent)
   {
      htmlTag = dynamic_cast<litehtml::html_tag*>(&(*parent));
   }
   if (htmlTag != 0)
   {
      const litehtml::background* background = htmlTag->get_background();
      if (background != 0)
      {
         (void)docElement.AddStyleParameter(StyleParameterBase::BackgroundColor, GetColor(background->m_color));

         // needed to process background color area with the help of litehtml::document::draw
         htmlTag->SetDocElement(&docElement);
      }
   }
}


void LiteHtmlParser::HandleHtmlTag(DocElement& docElement, litehtml::html_tag& element)
{
   DocElement* newObj = 0;
   Candera::Rectangle rect(GetElementRect(element));

   const litehtml::tchar_t* htmlTag = element.get_tagName();
   if (String::CompareStrings(htmlTag, "html") == 0)
   {
      Document* doc = dynamic_cast<Document*>(&docElement);
      FEATSTD_DEBUG_ASSERT(doc != 0);
      if (doc != 0)
      {
         doc->SetRect(rect);
         HandleBackground(*doc, element);
         newObj = doc;
         m_currentBlock = 0;
      }
   }
   else if (String::CompareStrings(htmlTag, "head") == 0)
   {
      m_isHead = true;
   }
   else if (element.is_body())
   {
      m_isHead = false;

      // body is needed as chapter due to position info
      Document* doc = dynamic_cast<Document*>(&docElement);
      FEATSTD_DEBUG_ASSERT(doc != 0);

      Chapter* body = LITEHTML_NEW(Chapter);
      if ((doc != 0) && (body != 0))
      {
         doc->SetBody(Chapter::SharedPointer(body));
         body->SetRect(rect);
         m_currentChapter = body;
      }
      newObj = body;
      m_currentBlock = 0;
   }
   else if (String::CompareStrings(htmlTag, "h1") == 0)
   {
      newObj = AddChapter(0, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "h2") == 0)
   {
      //RICHTEXT_LOG(std::endl << std::endl << *s_doc);
      newObj = AddChapter(1, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "h3") == 0)
   {
      newObj = AddChapter(2, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "h4") == 0)
   {
      newObj = AddChapter(3, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "h5") == 0)
   {
      newObj = AddChapter(4, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "h6") == 0)
   {
      newObj = AddChapter(5, docElement, rect);
   }
   else if (String::CompareStrings(htmlTag, "p") == 0)
   {
      if (m_currentChapter != 0)
      {
         newObj = m_currentChapter;
         m_currentParagraphRect = rect;
      }
      m_currentBlock = 0;
   }
   else if (String::CompareStrings(htmlTag, "b") == 0)
   {
      FEATSTD_DEBUG_ASSERT(m_currentBlock != 0);
      LiteHtmlBlock* block = m_currentBlock->AddChildType<LiteHtmlBlock, LiteHtmlBlock>();
      m_currentBlock = block;
      newObj = block;
   }
   else if (String::CompareStrings(htmlTag, "span") == 0)
   {
      EnsureCurrentBlock(docElement, element);
      if (m_currentBlock != 0)
      {
         LiteHtmlBlock* block = m_currentBlock->AddChildType<LiteHtmlBlock, LiteHtmlBlock>();
         if (block != 0)
         {
            HandleBackground(docElement, element);
         }
         m_currentBlock = block;
         newObj = block;
      }
   }
   else if (String::CompareStrings(htmlTag, "br") == 0)
   {
      // nothing to do?
   }
   else
   {
      RICHTEXT_LOG_BEGIN(stream);
      RICHTEXT_LOG(stream, htmlTag);
      RICHTEXT_LOG_END(stream);

      // test for e.g. <c01> - simple way to set color
      m_currentParagraphRect = rect;
      EnsureCurrentBlock(docElement, element);
      if (m_currentBlock != 0)
      {
         LiteHtmlBlock* block = m_currentBlock->AddChildType<LiteHtmlBlock, LiteHtmlBlock>();
         m_currentBlock = block;
         newObj = block;
      }
   }

   if (newObj != 0)
   {
      litehtml::text_overflow text_overflow = element.get_text_overflow();
      if (text_overflow != litehtml::text_overflow_clip)
      {
         TextOverflow::Enum value = TextOverflow::Enum(text_overflow);
         TextOverflow textOverflow(value);
         (void)newObj->AddStyleParameter(StyleParameterBase::TextOverflow, textOverflow);
      }

      const size_t childCount = element.get_children_count();

      for (size_t i = 0; i < childCount; i++)
      {
         litehtml::element::ptr child = element.get_child(i);
         if (child.IsValid())
         {
            CreateDocRecusive(*newObj, *child);
         }
      }

      if (m_currentBlock != 0)
      {
         m_currentBlock = dynamic_cast<LiteHtmlBlock*>(m_currentBlock->GetParent());
      }
   }

   m_currentParagraphRect = Candera::Rectangle(0.0F, 0.0F, -1.0F, -1.0F);
   m_isTitle = false;
}


void LiteHtmlParser::HandleSpace(DocElement& docElement, litehtml::el_space& element)
{
   // if there are only white spaces between </head> and <body>, litehtml add these elements to head -> nothing to render for head
   // if the text between </head> and <body> also holds non white space characters, these elements are added to body section
   if (!m_isHead)
   {
      EnsureCurrentBlock(docElement, element);
   }

   if (m_currentBlock != 0)
   {
      EnsureStyleOnCurrentBlock(element);

      Space* space = m_currentBlock->AddChildType<LiteHtmlBlock, Space>();
      if (space != 0)
      {
         litehtml::tstring str;
         element.get_text(str);
         space->SetText(str.c_str());      // -> not needed, text is alredy pre-rendered and stored in context
         space->SetContext(element.GetContext());
         space->SetSkip(element.skip());
         SetRect(*space, element);
         SetTruncationLine(*space, element);
      }
   }
}


void LiteHtmlParser::HandleText(DocElement& docElement, litehtml::el_text& element)
{
   EnsureCurrentBlock(docElement, element);
   if (m_currentBlock != 0)
   {
      EnsureStyleOnCurrentBlock(element);

      Text* text = m_currentBlock->AddChildType<LiteHtmlBlock, Text>();
      if (text != 0)
      {
         litehtml::tstring str;
         element.get_text(str);
         text->SetText(str.c_str());      // -> not needed, text is alredy pre-rendered and stored in context
         text->SetContext(element.GetContext());
         SetRect(*text, element);
         SetTruncationLine(*text, element);
      }
   }
}


void LiteHtmlParser::HandleImage(DocElement& docElement, litehtml::el_image& element)
{
   EnsureCurrentBlock(docElement, element);
   if (m_currentBlock != 0)
   {
      Image* image = m_currentBlock->AddChildType<LiteHtmlBlock, Image>();
      if (image != 0)
      {
         image->SetName(element.get_attr(_t("src")));
         SetRect(*image, element);
      }
   }
}


void LiteHtmlParser::SetRect(DocElement& docElement, litehtml::element& element) const
{
   Candera::Rectangle rect = GetElementRect(element);
   docElement.SetRect(rect);
}


void LiteHtmlParser::SetTruncationLine(Text& text, litehtml::element& element) const
{
   litehtml::line_box* box = element.GetBox() != 0 ? dynamic_cast<litehtml::line_box*>(&(*element.GetBox())) : 0;
   if ((box != 0) && (m_engine != 0))
   {
      Text::TruncationData truncationData;

      litehtml::WhiteSpacePropValue whiteSpace;
      const bool noWrap = (element.GetPropValue(litehtml::PropValue::WhiteSpace, true, whiteSpace) && (whiteSpace.m_value == litehtml::white_space_nowrap));
      UInt16 maxNumberOfLines = m_engine->GetData().m_maxNumberOfLines;
      const FeatStd::Int lineIndex = box->GetLineIndex();
      const bool maxNumberOfLinesReached = (maxNumberOfLines > 0) && (lineIndex == FeatStd::Int(maxNumberOfLines) - 1);
      truncationData.m_isTruncationLine = noWrap || maxNumberOfLinesReached;
      truncationData.m_truncationNeeded = box->IsTruncationNeeded();
      litehtml::LayoutDirection baseLayoutDirection = element.GetBaseLayoutDirection();
      litehtml::LayoutDirection elementLayoutDirection = element.GetLayoutDirection();
      if (baseLayoutDirection == litehtml::Neutral)
      {
         baseLayoutDirection = elementLayoutDirection;
      }
      truncationData.m_baseBidiLevel =
         baseLayoutDirection == litehtml::RightToLeft ? Candera::TextRendering::BidiBaseLevel::RightToLeft :
         Candera::TextRendering::BidiBaseLevel::LeftToRight;
      truncationData.m_elementBidiLevel =
         elementLayoutDirection == litehtml::RightToLeft ? Candera::TextRendering::BidiBaseLevel::RightToLeft :
         Candera::TextRendering::BidiBaseLevel::LeftToRight;
      if (truncationData.m_isTruncationLine)
      {
         litehtml::elements_vector elements;
         box->get_elements(elements);
         const litehtml::elements_vector::size_type size = elements.size();
         if (size > 0)
         {
            const litehtml::element* last = &(*(elements[size - 1]));
            const litehtml::element* secondLast = (size > 1) ? &(*(elements[size - 2])) : 0;

            const bool isLast = (last == &element);
            const bool isSecondLast = (secondLast == &element);
            const bool lastIsSpace = dynamic_cast<const litehtml::el_space*>(last) != 0;

            if (isLast || (isSecondLast && lastIsSpace))
            {
               truncationData.m_isLastInTruncationLine = true;
            }
         }
         truncationData.m_hasMoreLines = element.get_document()->GetLineCount() > lineIndex + 1;
      }

      text.SetTruncationData(truncationData);
   }
}


void LiteHtmlParser::EnsureStyleOnCurrentBlock(litehtml::el_text& element)
{
   if (!m_currentBlock->HasText())
   {
      // only the first Text child in current Block needs to set the style parameter to the Block as
      // those parameters must be the same for the whole block
      litehtml::web_color color = element.GetColor(litehtml::PropValue::Color, true);
      (void)m_currentBlock->AddStyleParameter(StyleParameterBase::TextColor, GetColor(color));

      Candera::TextRendering::SharedStyle::SharedPointer style = Candera::TextRendering::SharedStyle::SharedPointer(FeatStd::Internal::ScalarToPointer<Candera::TextRendering::SharedStyle*>(element.get_font()));
      if (style != 0)
      {
         (void)m_currentBlock->AddStyleParameter(StyleParameterBase::FontStyle, style);
      }
   }
}


void LiteHtmlParser::LiteHtmlBlock::AddChild(const ParagraphItem::SharedPointer& item)
{
   Base::AddChild(item);
   if (Candera::Dynamic_Cast<const Text*>(&(*item)) != 0)
   {
      m_hasText = true;
   }
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
