/* ***************************************************************************************
* FILE:          DocumentContainer.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DocumentContainer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <litehtml/include/litehtml.h>
#include <Candera/TextEngine/TextRenderer.h>

namespace hmibase {
namespace widget {
namespace richtext {

class DocumentContainer : public litehtml::document_container
{
   public:
      DocumentContainer();
      virtual ~DocumentContainer();

      virtual litehtml::uint_ptr create_font(const litehtml::tchar_t* faceName, int size, int weight, litehtml::font_style italic, unsigned int decoration,
                                             litehtml::font_metrics* fm) override;

      virtual void delete_font(litehtml::uint_ptr hFont) override;

      virtual int text_width(const litehtml::tchar_t* text, litehtml::uint_ptr hFont, litehtml::el_text::Context::SharedPtr& context, litehtml::LayoutDirection layoutDirection) override;

      virtual void draw_text(litehtml::uint_ptr hdc, const litehtml::tchar_t* text, litehtml::uint_ptr hFont, litehtml::web_color color,
                             const litehtml::position& pos) override;

      virtual int pt_to_px(int pt) override;

      virtual int get_default_font_size() const override;

      virtual const litehtml::tchar_t* get_default_font_name() const override;

      virtual void draw_list_marker(litehtml::uint_ptr hdc, const litehtml::list_marker& marker) override;

      virtual void load_image(const litehtml::tchar_t* src, const litehtml::tchar_t* baseurl, bool redraw_on_ready) override;

      virtual void get_image_size(const litehtml::tchar_t* src, const litehtml::tchar_t* baseurl, litehtml::size& sz) override;

      virtual void draw_background(litehtml::uint_ptr hdc, const litehtml::background_paint& bg) override;

      virtual void draw_borders(litehtml::uint_ptr hdc, const litehtml::borders& borders, const litehtml::position& draw_pos, bool root) override;

      virtual void set_caption(const litehtml::tchar_t* caption) override;

      virtual void set_base_url(const litehtml::tchar_t* base_url) override;

      virtual void link(const litehtml::document* doc, const litehtml::element::ptr& el) override;

      virtual void on_anchor_click(const litehtml::tchar_t* url, const litehtml::element::ptr& el) override;

      virtual void set_cursor(const litehtml::tchar_t* cursor) override;

      virtual void transform_text(litehtml::tstring& text, litehtml::text_transform tt) override;

      virtual void import_css(litehtml::tstring& text, const litehtml::tstring& url, litehtml::tstring& baseurl) override;

      virtual void set_clip(const litehtml::position& pos, const litehtml::border_radiuses& bdr_radius, bool valid_x, bool valid_y) override;

      virtual void del_clip() override;

      virtual void get_client_rect(litehtml::position& client) const override;

      virtual litehtml::element::ptr create_element(const litehtml::tchar_t* tag_name,
            const litehtml::string_map& attributes,
            const litehtml::document* doc) override;

      virtual void get_media_features(litehtml::media_features& media) const override;

      virtual void get_language(litehtml::tstring& language, litehtml::tstring& culture) const override;

#if defined(LITEHTML_USE_BIDI)
      virtual void CreateChunks(const litehtml::tchar_t* text, ChunkList& chunks, litehtml::LayoutDirection layoutDirection) const override;
#endif

      void SetClientRect(const litehtml::position& rect);
      void SetDefaultStyle(const Candera::TextRendering::SharedStyle::SharedPointer& defaultStyle);
      litehtml::LayoutDirection GetDocumentLayoutDirection() const;

      void SetCultureDependentAlignment(bool cultureDependentAlignment);
      virtual bool IsCultureDependentAlignment() const override;

      void SetRightToLeftCulture(bool rightToLeftCulture);
      virtual bool IsRightToLeftCulture() const override;

   private:
      bool m_cultureDependentAlignment;
      bool m_rightToLeftCulture;
      Candera::TextRendering::TextRenderer m_textRenderer;
      litehtml::position m_clientRect;
      Candera::TextRendering::SharedStyle::SharedPointer m_defaultStyle;

      DocumentContainer(const DocumentContainer&);
      DocumentContainer& operator = (const DocumentContainer&);

      Candera::TextRendering::SharedStyle::SharedPointer GetStyle(const litehtml::tchar_t* faceName, int size);
      bool IsDefaultStyle(const litehtml::tchar_t* faceName) const;
};


inline
void DocumentContainer::SetClientRect(const litehtml::position& rect)
{
   m_clientRect = rect;
}


inline
void DocumentContainer::SetDefaultStyle(const Candera::TextRendering::SharedStyle::SharedPointer& defaultStyle)
{
   m_defaultStyle = defaultStyle;
}


inline
void DocumentContainer::SetCultureDependentAlignment(bool cultureDependentAlignment)
{
   m_cultureDependentAlignment = cultureDependentAlignment;
}


inline
bool DocumentContainer::IsCultureDependentAlignment() const
{
   return m_cultureDependentAlignment;
}


inline
void DocumentContainer::SetRightToLeftCulture(bool rightToLeftCulture)
{
   m_rightToLeftCulture = rightToLeftCulture;
}


inline
bool DocumentContainer::IsRightToLeftCulture() const
{
   return m_rightToLeftCulture;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
