/* ***************************************************************************************
* FILE:          DocumentContainer.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DocumentContainer is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "DocumentContainer.h"
#include "LiteHtmlParser.h"
#include <Candera/TextEngine/Internal/Parser.h>
#include <Candera/TextEngine/Internal/Shaper.h>
#include <Candera/TextEngine/Style.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushCache/BitmapTextBrushCache.h>
#include <Courier/Version.h>
#include <FeatStd/MemoryManagement/Heap.h>
#include <Widgets/2D/RichText/ContentProvider/RtContentProvider.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocHelper.h>
#include <Widgets/2D/RichText/LiteHtml/TruncationTextContext.h>
#include <Widgets/2D/RichText/Utils/RtConfig.h>
#include <hmibase/util/Macros.h>
#include <iomanip>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_RICHTEXT
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/DocumentContainer.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace richtext {

using namespace Candera::TextRendering;
using Candera::Bitmap;

static bool IsWhiteSpace(litehtml::tchar_t c)
{
   return (c == ' ') || (c == '\t') || (c == '\n') || (c == '\r') || (c == '\f');
}


DocumentContainer::DocumentContainer() :
   m_cultureDependentAlignment(true),
   m_rightToLeftCulture(false),
   m_textRenderer()
{
}


DocumentContainer::~DocumentContainer()
{
}


litehtml::uint_ptr DocumentContainer::create_font(const litehtml::tchar_t* faceName, int size, int /*weight*/, litehtml::font_style style,
      unsigned int decoration, litehtml::font_metrics* fm)
{
   Candera::TextRendering::SharedStyle::SharedPointer fontStyle = GetStyle(faceName, size);

   if (fontStyle != 0)
   {
      Metrics metrics = fontStyle->GetMetrics();

      fm->ascent = metrics.ascender - 1;  //Note: 'ascender - 1' is also used in TextNode2D::GetBasePoint() to get correct baseline!!!
      fm->descent = -metrics.descender;
      fm->height = metrics.lineHeight;
      fm->x_height = metrics.ascender;
      const bool italic = (style == litehtml::fontStyleItalic);
      if (italic || decoration)
      {
         fm->draw_spaces = true;
      }
      else
      {
         fm->draw_spaces = false;
      }
   }

   return litehtml::uint_ptr(fontStyle.GetPointerToSharedInstance());
}


void DocumentContainer::delete_font(litehtml::uint_ptr /*hFont*/)
{
   //font data is cached in CustomStyleCache::Cache and freed there automatically.
}


int DocumentContainer::text_width(const litehtml::tchar_t* text, litehtml::uint_ptr hFont, litehtml::el_text::Context::SharedPtr& context, litehtml::LayoutDirection layoutDirection)
{
   static const TextRenderContext* cache = Candera::Internal::BitmapTextBrushCache().GetTextRenderContext();
   static const LayoutingOptions layoutingOptions;

   int width = 0;

   Candera::TextRendering::SharedStyle::SharedPointer style = Candera::TextRendering::SharedStyle::SharedPointer(reinterpret_cast<Candera::TextRendering::SharedStyle*>(hFont));
   if (style != 0)
   {
      ShapingOptions shapingOptions(style, ((layoutDirection == litehtml::RightToLeft) && !IsWhiteSpace(*text)) ? BidiBaseLevel::RightToLeft : BidiBaseLevel::LeftToRight);

      TruncationTextContext::SharedPointer textContext = TruncationTextContext::Create(cache);
      if (textContext.IsValid())
      {
         TruncationTextContext& truncateTextContext = static_cast<TruncationTextContext&>(*textContext);
         if (!IsWhiteSpace(*text))
         {
            truncateTextContext.SetGlyphOrder(TextRenderContext::RenderOrder);
         }
#if defined(USE_MINIMAL_TEXTRENDERER)
         ProcessChunk(truncateTextContext, shapingOptions, text);
#else
         truncateTextContext.ApplyCodePointMap(text);

         TextProperties textProperties(text);
         TextRenderer().Render(truncateTextContext, LayoutingOptions(), shapingOptions, textProperties);

#endif
         context = textContext;
         width = truncateTextContext.GetTextWidth();
      }
   }

   return width;
}


void DocumentContainer::draw_text(litehtml::uint_ptr /*hdc*/, const litehtml::tchar_t* text, litehtml::uint_ptr /*hFont*/,
                                  litehtml::web_color /*color*/,
                                  const litehtml::position& pos)
{
   RICHTEXT_LOG_BEGIN(stream);
   RICHTEXT_LOG(stream, "draw_text: '" << text << "' at " << pos.left() << "/" << pos.top() << std::endl);
   RICHTEXT_LOG_END(stream);

   FEATSTD_UNUSED(text);
   FEATSTD_UNUSED(pos);
}


int DocumentContainer::pt_to_px(int /*pt*/)
{
   return 0;
}


int DocumentContainer::get_default_font_size() const
{
   return (m_defaultStyle != 0) ? m_defaultStyle->GetMetrics().lineHeight : 0;
}


const litehtml::tchar_t* DocumentContainer::get_default_font_name() const
{
   static const litehtml::tchar_t* emptyName = "";

   const FeatStd::Char* name = (m_defaultStyle != 0) ? m_defaultStyle->GetName() : 0;

   return (name != 0) ? name : emptyName;
}


void DocumentContainer::draw_list_marker(litehtml::uint_ptr /*hdc*/, const litehtml::list_marker& /*marker*/)
{
}


void DocumentContainer::load_image(const litehtml::tchar_t* /*src*/, const litehtml::tchar_t* /*baseurl*/, bool /*redraw_on_ready*/)
{
}


void DocumentContainer::get_image_size(const litehtml::tchar_t* src, const litehtml::tchar_t* /*baseurl*/, litehtml::size& sz)
{
   Bitmap::SharedPointer bitmap = ContentProvider::GetInstance().GetBitmap(0, src);
   if (bitmap != 0)
   {
      sz.width = bitmap->GetWidth();
      sz.height = bitmap->GetHeight();
   }
}


void DocumentContainer::draw_background(litehtml::uint_ptr hdc, const litehtml::background_paint& bg)
{
   DocElement* docElement = reinterpret_cast<DocElement*>(hdc);
   if (docElement != 0)
   {
      Candera::Rectangle box;
      box = Candera::Rectangle(FeatStd::Float(bg.origin_box.x), FeatStd::Float(bg.origin_box.y), FeatStd::Float(bg.origin_box.width),
                               FeatStd::Float(bg.origin_box.height));
      docElement->AddBackgroundBox(box);
   }
}


void DocumentContainer::draw_borders(litehtml::uint_ptr /*hdc*/, const litehtml::borders& borders, const litehtml::position& draw_pos, bool /*root*/)
{
   RICHTEXT_LOG_BEGIN(stream);
   RICHTEXT_LOG(stream, "draw_borders: " << borders.left.width << "/" << borders.top.width << "/" << borders.right.width << "/" << borders.bottom.width
                <<
                " at " << draw_pos.left() << "/" << draw_pos.top() << " -> " << draw_pos.right() << "/" << draw_pos.bottom() <<
                std::endl);
   RICHTEXT_LOG_END(stream);

   FEATSTD_UNUSED(borders);
   FEATSTD_UNUSED(draw_pos);
}


void DocumentContainer::set_caption(const litehtml::tchar_t* /*caption*/)
{
}


void DocumentContainer::set_base_url(const litehtml::tchar_t* /*base_url*/)
{
}


void DocumentContainer::link(const litehtml::document* /*doc*/, const litehtml::element::ptr& /*el*/)
{
}


void DocumentContainer::on_anchor_click(const litehtml::tchar_t* /*url*/, const litehtml::element::ptr& /*el*/)
{
}


void DocumentContainer::set_cursor(const litehtml::tchar_t* /*cursor*/)
{
}


void DocumentContainer::transform_text(litehtml::tstring& /*text*/, litehtml::text_transform /*tt*/)
{
}


void DocumentContainer::import_css(litehtml::tstring& /*text*/, const litehtml::tstring& /*url*/, litehtml::tstring& /*baseurl*/)
{
}


void DocumentContainer::set_clip(const litehtml::position& /*pos*/, const litehtml::border_radiuses& /*bdr_radius*/, bool /*valid_x*/,
                                 bool /*valid_y*/)
{
}


void DocumentContainer::del_clip()
{
}


void DocumentContainer::get_client_rect(litehtml::position& client) const
{
   client = m_clientRect;
}


litehtml::element::ptr DocumentContainer::create_element(const litehtml::tchar_t* /*tag_name*/,
      const litehtml::string_map& /*attributes*/,
      const litehtml::document* /*doc*/)
{
   return litehtml::element::ptr();
}


void DocumentContainer::get_media_features(litehtml::media_features& /*media*/) const
{
}


void DocumentContainer::get_language(litehtml::tstring& /*language*/, litehtml::tstring& /*culture*/) const
{
}


#if defined(LITEHTML_USE_BIDI)

void DocumentContainer::CreateChunks(const litehtml::tchar_t* text, ChunkList& chunks, litehtml::LayoutDirection layoutDirection) const
{
   litehtml::tstring str_in(text);
   Internal::ParserData parserData;
   BidiBaseLevel::Enum bidiBaseLevel =
      (layoutDirection == litehtml::RightToLeft) ? BidiBaseLevel::RightToLeft :
      (layoutDirection == litehtml::LeftToRight) ? BidiBaseLevel::LeftToRight :
      BidiBaseLevel::Implicit;

   parserData.Initialize(text, -1, 0, true, false, bidiBaseLevel);

   Style style;
   Internal::Parser parser(style, parserData);
   parser.Initialize(false);

   const bool correctChunkOrder = false;
   const Internal::Parser::LineBreakType breakType = Internal::Parser::AllowLineBreak;

   parser.AdvanceLineBreak(breakType);
   parser.StartLineParsing(correctChunkOrder);
   RICHTEXT_LOG_BEGIN(stream);
   RICHTEXT_LOG(stream, std::endl);
   do
   {
      TextPosition start = parser.GetStart();
      TextLength length = parser.GetLength();
      FeatStd::UInt8 direction = parser.GetDirection();

      RICHTEXT_LOG(stream, "Start:" << std::setw(2) << start << " Length:" << std::setw(2) << length);
      RICHTEXT_LOG(stream, " LineStart:" << std::setw(2) << parser.GetLineStart() << " CurrentLineLength:" << std::setw(2) << parser.GetCurrentLineLength());
      RICHTEXT_LOG(stream, " Direction:" << FeatStd::Int(direction) << std::endl);
      while (length > 0)
      {
         Chunk chunk;
         chunk.m_direction = ((direction % 2) == 0) ? litehtml::LeftToRight : litehtml::RightToLeft;
         chunk.m_string = str_in.substr(start, length);

         TextLength len = 0;
         while (len < length && !IsWhiteSpace(chunk.m_string[len]))
         {
            len++;
         }
         if (len < length)
         {
            if (len > 0)
            {
               Chunk first(chunk);
               first.m_string = first.m_string.substr(0, len);
               chunks.push_back(first);
               length -= len;
               start += len;
            }

            if (len < length)
            {
               Chunk whitespace(chunk);
               whitespace.m_string = chunk.m_string.substr(len, 1);
               FEATSTD_DEBUG_ASSERT(IsWhiteSpace(whitespace.m_string[0]));
               chunks.push_back(whitespace);
               length -= 1;
               start += 1;
            }
         }
         else
         {
            chunks.push_back(chunk);
            length -= len;
            start += len;
         }
      }

      parser.AdvanceToNextChunk();

      if (parser.IsLineBreak() && !parser.IsEnd())
      {
         parser.AdvanceToNextLine();
         parser.AdvanceLineBreak(breakType);
         parser.StartLineParsing(correctChunkOrder);
      }
   }
   while (!parser.IsEnd());
   RICHTEXT_LOG_END(stream);
}


#endif

litehtml::LayoutDirection DocumentContainer::GetDocumentLayoutDirection() const
{
   // default document layout direction is Neutral (following W3C HTML standard), regardless of the HMI culture. In that case the
   // direction is taken from the first bidirectional chunk (heuristic).
   return litehtml::Neutral;
}


Candera::TextRendering::SharedStyle::SharedPointer DocumentContainer::GetStyle(const litehtml::tchar_t* faceName, int size)
{
   Candera::TextRendering::SharedStyle::SharedPointer style;

   if (IsDefaultStyle(faceName))
   {
      style = m_defaultStyle;
   }
   else if (*faceName != '\0')
   {
      // style from asset (CanderaName)
      style = ContentProvider::GetInstance().GetTextStyle("asset:path/", faceName);
      if (style == 0)
      {
         // custom style for given font and size
         CustomStyleCache* styleCache = CustomStyleCache::Get(faceName, size);
         if (styleCache != 0)
         {
            style = styleCache->GetStyle();
         }
      }
   }

   return style;
}


bool DocumentContainer::IsDefaultStyle(const litehtml::tchar_t* faceName) const
{
   bool isDefaultStyle = (faceName == 0) || (*faceName == '\0');
   if ((m_defaultStyle != 0) && (faceName != 0))
   {
      const FeatStd::Char* name = m_defaultStyle->GetName();
      if ((name != 0) && FeatStd::Internal::String::CompareStrings(name, faceName) == 0)
      {
         isDefaultStyle = true;
      }
   }

   return isDefaultStyle;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
