/* ***************************************************************************************
* FILE:          RtViewport.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtViewport is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/EngineBase/EngineBase.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * This class encapsulates presentation related attributes for one view on the
 * document. It provides basic presentation configuration, such as visibility and
 * scaling of the presentation and also converts between positions relative to the
 * top-left corner of the visible on-screen client area and the top-left corner of
 * the overall document.
 * This conversion must be considered in methods based on on-screen coordinates
 * due to scrolling or other reasons for offsets.
 * For supporting multi-view documents (documents that are concurrently presented
 * in more than one client area), multiple instances of this class can be used.
 */
class Viewport : public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Viewport);

      /** Destructor */
      virtual ~Viewport();

      /**
       * Enables or disables rendering of the document.
       *
       * @param enable True to enable rendering, false otherwise.
       */
      virtual void EnableRendering(bool enable) = 0;

      /**
       * Determines whether or not rendering of the document is enabled.
       *
       * @return True if rendering is enabled, false otherwise.
       */
      virtual bool IsRenderingEnabled() const = 0;

      /**
       * Defines the size of the on- or off-screen area into which the document content is
       * laid out and rendered.
       *
       * @param width The width of the content area in pixel.
       * @param height The height of the content area in pixel.
       */
      virtual void SetSize(FeatStd::UInt16 width, FeatStd::UInt16 height) = 0;

      /**
       * Defines the size of the on- or off-screen area into which the document content is
       * laid out and rendered.
       *
       * @param size The size of the content area in pixel.
       */
      virtual void SetSize(const Candera::Vector2& size) = 0;

      /**
       * Provides the size of the content area in which the document is presented.
       *
       * @return The size of the content area for the document.
       */
      virtual const Candera::Vector2& GetSize() const = 0;

      /**
       * Defines the scale factor of the document presentation. This attributes
       * affects the rendering of the document by scaling the underlying pixel buffer
       * instead of manipulating font sizes and scaling of single elements. This
       * approach ensures a homogeneous scaling of the presentation.

       * @param scaleFactorX The scale factor in horizontal direction.
       * @param scaleFactorY The scale factor in vertical direction.
       */
      virtual void SetScaleFactor(FeatStd::Float scaleFactorX, FeatStd::Float scaleFactorY) = 0;

      /**
       * Defines the scale factor of the document presentation. This attributes
       * affects the rendering of the document by scaling the underlying pixel buffer
       * instead of manipulating font sizes and scaling of single elements. This
       * approach ensures a homogeneous scaling of the presentation.

       * @param scale The scaleFactor factor in horizontal and vertical direction.
       */
      virtual void SetScaleFactor(const Candera::Vector2& scaleFactor) = 0;

      /**
       * Provides the scale factor of the document presentation.
       *
       * @return The scale factor.
       */
      virtual Candera::Vector2 GetScaleFactor() const = 0;

      /**
       * Converts the given on-screen position relative to the top-left corner of the
       * visible client area to the document position relative to the top-left corner
       * of the document.
       *
       * @param screenPosition The position relative to the upper left corner
       *                       of the client area.
       * @ return The corresponding position relative to the upper left corner
       *          of the document.
       */
      virtual Candera::Vector2 ToDocumentPosition(const Candera::Vector2& screenPosition) const = 0;

      /**
       * Converts the given document position relative to the top-left corner
       * of the document to the on-screen position relative to the top-left corner
       * of the visible client area.
       *
       * @param documentPosition The position relative to the upper left corner
       *                         of the document.
       * @ return The corresponding position relative to the upper left corner
       *          of the client area.
       */
      virtual Candera::Vector2 ToViewportPosition(const Candera::Vector2& documentPosition) const = 0;

      /**
       * Applies the offset from the top-left corner of the document to the top-left
       * corner of the visible client area.
       *
       * @param offsetX The horizontal offset to apply.
       * @param offsetY The vertical offset to apply.
       */
      virtual void SetOffset(FeatStd::Float offsetX, FeatStd::Float offsetY) = 0;

      /**
       * Applies the offset from the top-left corner of the document to the top-left
       * corner of the visible client area.
       *
       * @param offset The offset to apply.
       */
      virtual void SetOffset(const Candera::Vector2& offset) = 0;

      /**
       * Provides the offset from the top-left corner of the document to the top-left
       * corner of the visible client area.
       *
       * @return The offset.
       */
      virtual Candera::Vector2 GetOffset() const = 0;

   protected:

      ///Restricted constructor.
      Viewport();

   private:
};


} // namespace richtext
} // namespace widget
} // namespace hmibase
