/* ***************************************************************************************
* FILE:          RtMarker.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtMarker is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/Util/Rtti.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>
#include <Widgets/2D/RichText/Engine/RtDocRange.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * This class represents one or marked portions of the document.
 * It provides functions to manipulate the marked ranges.
 */
class Marker : public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);

   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Marker);
      // The list type for marks.
      typedef FeatStd::Internal::Vector<DocRange::SharedPointer> MarkList;

      Marker();

      virtual ~Marker();

      /**
       * Adds the given document range to the list of marks. This is useful for
       * temporarily removing and reattaching a mark.
       *
       * @param mark The portion of the document to mark.
       */
      void Add(DocRange::SharedPointer mark);

      /**
       * Removes the given mark.
       *
       * @param mark The mark to remove.
       */
      void Remove(DocRange::SharedPointer mark);

      /**
       * Removes and returns the mark at the given index.
       *
       * @param index The index of the mark to remove.
       * @return The mark at the given index or a null-object, if no such index exists.
       */
      DocRange::SharedPointer Remove(FeatStd::UInt32 index);

      /**
       * Provides the number of marks associated with the document.
       *
       * @return The total number of marks.
       */
      FeatStd::UInt32 GetCount() const;

      /**
       * Provides the list of all marks associated with the document.
       *
       * @return The list of all marks or an empty list, if the document has no marks.
       */
      const MarkList& Get() const;

      /**
       * Provides the mark associated with the document at the given index.
       *
       * @param index The index of the mark to retrieve.
       * @return The mark associated with the document at the given index or a null-
       * object if no such mark exists.
       */
      DocRange::SharedPointer Get(FeatStd::UInt32 index);

      /**
       * Removes all marks.
       */
      void ClearAll();

   private:
      // The marked ranges within the document this marker is attached to.
      MarkList m_markList;
};


}
}


}
