/* ***************************************************************************************
* FILE:          RtHighlighter.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtHighlighter is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <FeatStd/MemoryManagement/SharedPointer.h>

#include <Widgets/2D/RichText/Engine/RtEngineModuleBase.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocElement.h>
#include <Widgets/2D/RichText/StyleProvider/RtElementStyle.h>
#include <Candera/System/MetaInfo/EnumDataTypeInternal.h>

namespace hmibase {
namespace widget {
namespace richtext {

/**
 * This class supports highlighting of multiple parts of the document
 * using index-based or text-mask based text search. It creates markers
 * for the addressed document parts and applies the defined style for
 * presenting the highlighting. The marked parts can be accessed using
 * Marker instance provided by the Engine.
 *
 * @note: This class is not intended for interactive selection of document portions.
 */
class Highlighter: public RichTextEngineModuleBase
{
      FEATSTD_TYPEDEF_BASE(RichTextEngineModuleBase);
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Highlighter);

      enum HighlightMode
      {
         Mask,
         Range
      };

      /** Destructor */
      virtual ~Highlighter();

      /**
       * Clears all marks regardless of the 'mark all' pass.
       */
      virtual void ClearAll() = 0;

      /** All relevant data for highlighting. */
      struct Data
      {
         HighlightMode m_mode;
         FeatStd::String m_text;
         bool m_allOccurrences;
         bool m_caseSensitive;
         ElementStyle::SharedPointer m_style;

         Data();
      };

      /** Pass the highlight data to the Highlighter. This has to be done whenever any highlight parameter changes. */
      void SetData(const Data& data);

      /** Has to be called either after SetData() or if the engine detects an update of the rich text document to
          highlight the new document accordingly. */
      virtual void Update() = 0;

   protected:
      Data m_data;

      ///Restricted constructor.
      Highlighter();

      /**
       * Adds markers for all occurrences of the text parts identified by 'searchCriteria'.
       * Previous markers are not cleared beforehand, however, subsequent search will overwrite
       * conflicting marks.
       *
       * @param searchCriteria The criteria for marking parts of document text. The criteria (plural)
       * are provided as string considering the following syntax described in EBNF:
       * <ul>
       * <li>criteria = criterion {, ",", criterion} ;</li>
       * <li>criterion = index | index-range | text-mask ;</li>
       * <li>index = number ;</li>
       * <li>index-range = number, "-", number ;</li>
       * <li>text-mask = "'", char, "'" ;</li>
       * <li>number = digit {, digit} ;</li>
       * <li>digit = "0"| ... |"9" ;</li>
       * <li>char = ? any character including digits, letters, special characters. Any white-space character is treated as 'space' ? ;</li>
       * </ul>
       * Spaces and undefined characters within the searchCriterion string are ignored.
       * Incomplete index ranges or text-masks are automatically closed at the end of the string.
       * Index and index-range addresses only characters by their index within the overall document
       * Text-masks addresses any complete occurrence of exactly the text equal to the text-mask.
       *
       * @param highlightStyle The style for presentation of the highlight.
       */
      virtual void MarkAll(const FeatStd::String& searchCriteria, ElementStyle::SharedPointer highlightStyle) = 0;

      /**
       * Adds markers for all addressed parts or part-ranges identified by 'range'.
       * Previous markers are not cleared beforehand, however, subsequent search will overwrite
       * conflicting marks.
       *
       * @param range The criteria for marking parts of document text addressed by indexes.
       * This method uses the same EBNF as method 'MarkAll', but does not support terminal 'text-mask'.
       *
       * @param highlightStyle The style for presentation of the highlight.
       */
      virtual void MarkRange(const FeatStd::String& range, ElementStyle::SharedPointer highlightStyle) = 0;

      /**
       * Adds markers for all occurrences of 'searchText' up to the number of maxOccurencies.
       * Previous markers are not cleared beforehand, however, subsequent search will overwrite
       * conflicting marks.
       *
       * @param searchText       The criteria for marking parts of document text matching the this parameter exactly.
       * @param maxOccurencies   The maximum number of matches to highlight. A value less than 1 to marks all occurrences.
       * @param caseSensitibe    true to search case sensitive, false otherwise.
       * @param highlightStyle   The style for presentation of the highlight.
       */
      virtual void MarkText(const FeatStd::String& searchText, FeatStd::UInt32 maxOccurencies, bool caseSensitive, ElementStyle::SharedPointer highlightStyle) = 0;

   private:
};


inline
void Highlighter::SetData(const Data& data)
{
   m_data = data;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase

CANDERA_INTERNAL_ENUM_METAINFO_BEGIN(::hmibase::widget::richtext::Highlighter::HighlightMode, _hmibase_widget_richtext_Highlighter_HighlightMode)
CANDERA_INTERNAL_ENUM_METAINFO_ITEM(Mask)
CANDERA_INTERNAL_ENUM_METAINFO_ITEM(Range)
CANDERA_INTERNAL_ENUM_METAINFO_END()

#if defined(FEATSTD_STRINGBUFFER_APPENDER_ENABLED)
namespace FeatStd {
template<> ::FeatStd::UInt32 StringBufferAppender< ::hmibase::widget::richtext::Highlighter::HighlightMode >::Append(::FeatStd::StringBuffer& stringBuffer, ::hmibase::widget::richtext::Highlighter::HighlightMode const& object);
} // namespace FeatStd
#endif
