/* ***************************************************************************************
* FILE:          RtEngine.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  RtEngine is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2018 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#pragma once

#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <Candera/Candera.h>

#include <Widgets/2D/RichText/ContentProvider/RtUrlResolver.h>
#include <Widgets/2D/RichText/DocumentFactory/RtFormattedDocFab.h>
#include <Widgets/2D/RichText/Engine/RtMarkups.h>
#include <Widgets/2D/RichText/Engine/RtDocAccessor.h>
#include <Widgets/2D/RichText/Engine/RtCaretBasedDocManipulator.h>
#include <Widgets/2D/RichText/Engine/RtViewport.h>
#include <Widgets/2D/RichText/Engine/RtHighlighter.h>
#include <Widgets/2D/RichText/DocumentModel/RtDocument.h>
#include <Widgets/2D/RichText/Renderer/RtRenderer.h>

namespace Courier {
class View;
}


namespace hmibase {
namespace widget {
namespace richtext {


/**
    RichTextEngine provides the API to presentation object (widgets, behaviors) for the overall rich text related functionality.

    This class directly defines only the most basic methods for creating rich text documents and uses separated class supporting a certain functional entity.
    This class is instantiated for each rich text document instance, as it hold references and inner states of the document and its presentation. Associated
    functional classes are lacy instantiated, which enables a lightweight memory model for small and rather simple documents while the powerful feature-set is
    always available but created only on demand.
    As a second aspect this approach allows the omission of certain state flags for the document, such as read-only. Read-only presentation of documents simply
    are not accessing document manipulation, especially caret based manipulation. Programmatic document manipulation is still supported due to separation of
    functionality. For switching the document into a read-write mode, it is sufficient to retrieve the caret based document manipulator and to activate the
    caret presentation.
    The same is valid for other functional blocks, such as scrolling (not need for single-line documents), animations, styling, markers, etc.

    @note: Functional modules provided by this class are instantiated on demand (late instantiation approach) to
           avoid instantiation of the complete functionality for use cases that uses reduced feature-sets.
    */
class Engine
{
   public:
      FEATSTD_RTTI_DECLARATION();
      FEATSTD_TYPEDEF_SHARED_POINTER(Engine);

      ///
      struct Data
      {
         /// Defines either to use Color value or not.
         bool m_colorEnabled : 1;
         /// Defines either to use Horizontal Alignment value or not.
         bool m_horizontalAlignmentEnabled : 1;
         /// Enables asynchronous parsing and rendering.
         bool m_asynchronousRendering : 1;
         /// Enables rendering into separate bitmap slices.
         bool m_slicedRendering : 1;
         ///
         bool m_cultureDependentAlignment : 1;
         /// If set to a value greater than zero, text truncation happens at last specified line.
         FeatStd::UInt16 m_maxNumberOfLines;
         /// The pixel height for all slices. Value 0 defines a slice height that is automatically set to the viewport height.
         FeatStd::UInt16 m_sliceHeight;
         /// The vertical offset between node position and baseline of first rendered text line.
         FeatStd::Int32 m_baselineOffset;
         /// The markup language of the source document.
         Markup::Enum m_markup;
         /// The horizontal alignment for the text, overrules alignment defined in style.
         Candera::HorizontalAlignment m_horizontalAlignment;
         /// The View, need for asynchronous rendering.
         Courier::View* m_view;
         /// The content of the source document.
         TextInStream::SharedPointer m_source;
         /// The default text style to be used if not define in style sheet.
         Candera::TextRendering::SharedStyle::SharedPointer m_textStyle;
         /// The application- or system-wide 'master' stylesheet.
         FeatStd::String m_styleSheetUrl;
         /// The local style that overrules the style defined by styleSheetUrl.
         FeatStd::String m_style;
         /// The Color to be set on the text, overrules color defined in style.
         Candera::Color m_color;

         Data();
      };

      /// Destructor
      virtual ~Engine();

      /**
       * Creates and provides the instance of RichTextEngine for processing and manipulating the addressed source document.
       * Clients of rich text presentation and manipulation (e.g. widgets) are required to maintain that instance throughout
       * the lifecycle of the underlying document. This instance provides accessor, manipulation, and rendering functionality
       * operating on the underlying document to clients. It is intended being shared between multiple client classes
       * (e.g. widgets for different purposes) that operate on the same document.
       *
       * @return The shared instance of the rich text engine that encapsulates the source document.
       */
      static SharedPointer Create();

      /**
       * Sets the source data for document creation. This method invalidates the renderer to trigger re-rendering.
       */
      void SetData(const Data& data);

      /** Retrieves the source data. */
      const Data& GetData() const;

      void SetDocFab(const FormattedDocFab::SharedPointer& docFab)
      {
         m_docFab = docFab;
      }

      const FormattedDocFab::SharedPointer& GetDocFab() const
      {
         return m_docFab;
      }

      virtual void PrepareDocument() = 0;
      virtual void CreateDocument() = 0;

      /**
       * Provides the functional module for accessing the underlying document.
       *
       * @return The DocAccessor instance associated with this engine instance.
       */
      virtual DocAccessor::SharedPointer GetDocAccessor() = 0;

      /**
       * Provides the functional module for manipulating the underlying document using a caret.
       *
       * @return The CaretBasedDocManipulator instance associated with this engine instance.
       */
      virtual CaretBasedDocManipulator::SharedPointer GetCaretBasedDocManipulator() = 0;

      /**
       * Provides the functional module for defining and manipulating the viewport of the rendered document.
       *
       * @return The Viewport instance associated with this engine instance.
       */
      virtual Viewport::SharedPointer GetViewport() = 0;

      /**
       * Provides the renderer for the document.
       *
       * @return The associated rich text renderer.
       */
      virtual RichTextRenderer::SharedPointer GetRenderer() = 0;

      /**
       * Provides the highlighter for the document.
       *
       * @return The associated rich text renderer.
       */
      virtual Highlighter::SharedPointer GetHighlighter() = 0;

      virtual void SetHighlighter() = 0;

      /**
       * Renders the document to the given node.
       *
       * @return The bitmap containing the rendered document.
       */
      virtual void Render() = 0;

   protected:
      /// Restricted constructor
      Engine();

   private:
      Data m_data;
      FormattedDocFab::SharedPointer m_docFab;

      // Support smart references
      FEATSTD_SHARED_POINTER_DECLARATION();
};


inline
const Engine::Data& Engine::GetData() const
{
   return m_data;
}


} // namespace richtext
} // namespace widget
} // namespace hmibase
